/*
 * CmdTZ.c --
 *
 * Commands with names beginning with the letters T through Z.
 *
 *     ********************************************************************* 
 *     * Copyright (C) 1985, 1990 Regents of the University of California. * 
 *     * Permission to use, copy, modify, and distribute this              * 
 *     * software and its documentation for any purpose and without        * 
 *     * fee is hereby granted, provided that the above copyright          * 
 *     * notice appear in all copies.  The University of California        * 
 *     * makes no representations about the suitability of this            * 
 *     * software for any purpose.  It is provided "as is" without         * 
 *     * express or implied warranty.  Export of this software outside     * 
 *     * of the United States of America may require an export license.    * 
 *     *********************************************************************
 */

#ifndef lint
static char rcsid[] = "$Header: /ufs/repository/magic/commands/CmdTZ.c,v 1.7 2001/03/29 19:50:05 tim Exp $";
#endif  /* not lint */

#include <stdio.h>
#include <ctype.h>
#include <math.h>

#include "tcltk/tclmagic.h"
#include "misc/magic.h"
#include "utils/geometry.h"
#include "utils/utils.h"
#include "tiles/tile.h"
#include "utils/hash.h"
#include "database/database.h"
#include "windows/windows.h"
#include "dbwind/dbwind.h"
#include "main/main.h"
#include "commands/commands.h"
#include "textio/textio.h"
#include "textio/txcommands.h"
#include "signals/signals.h"
#include "undo/undo.h"
#include "select/select.h"
#include "misc/styles.h"
#include "wiring/wiring.h"
#include "netlist/netlist.h"
#include "netmenu/netmenu.h"
#include "tech/tech.h"

#ifdef	LLNL
#include "yacr.h"
#endif	/* LLNL */

extern void DisplayWindow();

/*
 * ----------------------------------------------------------------------------
 *
 * CmdTech --
 *
 * 	Implement the "tech" command.  This largely replaces the old wizard
 *	"*showtech" command, and is meant to facilitate technology file
 *	writing and debugging, as well as providing information about the
 *	current technology.  The "tech layers" command replaces the "layers"
 *	command.
 *
 * Usage:
 *	tech [name|filename|version|lambda|load|help|planes|layers]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Output.
 *
 * ----------------------------------------------------------------------------
 */

#define TECH_LOAD	0
#define TECH_HELP	1
#define TECH_NAME	2
#define TECH_FILE	3
#define TECH_VERSION	4
#define TECH_LAMBDA	5
#define TECH_PLANES	6
#define TECH_LAYERS	7

	/* ARGSUSED */
Void
CmdTech(w, cmd)
    MagWindow *w;		/* Window in which command was invoked. */
    TxCommand *cmd;		/* Info about command options. */
{
    int	option, action, i, locargc;
    char **msg;
#ifdef MAGIC_WRAPPER
    Tcl_Obj *lobj;
#endif
    bool noprompt = FALSE;
    int saveNumPlanes;
    int changePlanesFunc();	/* forward declaration */

    static char *actionNames[] =
	{ "no", "yes", 0 };

    static char *cmdTechOption[] =
    {	
	"load filename [-noprompt][-nooverride]\n\
				Load a new technology",
	"help			Display techinfo command options",
	"name			Show current technology name",
	"filename		Show current technology filename",
	"version		Show current technology version",
	"lambda			Show internal units per lambda", 
	"planes			Show defined planes",
	"layers			Show defined layers",
	NULL
    };

    if (cmd->tx_argc == 1)
	option = TECH_HELP;
    else
	option = Lookup(cmd->tx_argv[1], cmdTechOption);

    if (option == -1)
    {
	TxError("Ambiguous techinfo option: \"%s\"\n", cmd->tx_argv[1]);
	goto usage2;
    }
    if (option < 0)
	goto usage;

    switch (option)
    {

#ifdef MAGIC_WRAPPER
	case TECH_NAME:
	    Tcl_SetResult(magicinterp, DBTechName, NULL);
	    break;
	case TECH_FILE:
	    Tcl_SetResult(magicinterp, TechFileName, NULL);
	    break;
	case TECH_VERSION:
	    Tcl_SetResult(magicinterp, DBTechVersion, NULL);
	    Tcl_AppendElement(magicinterp, DBTechDescription);
	    break;
	case TECH_LAMBDA:
	    if ((cmd->tx_argc > 2) && (StrIsInt(cmd->tx_argv[2])))
	    {
		DBLambda[1] = atoi(cmd->tx_argv[2]);
		if (cmd->tx_argc == 3)
		    DBLambda[0] = 1;
		else if ((cmd->tx_argc > 3) && (StrIsInt(cmd->tx_argv[3])))
		DBLambda[0] = atoi(cmd->tx_argv[3]);
		ReduceFraction(&DBLambda[0], &DBLambda[1]);
	    }

	    lobj = Tcl_NewListObj(0, NULL);
 	    Tcl_ListObjAppendElement(magicinterp, lobj,
			Tcl_NewIntObj(DBLambda[0]));
 	    Tcl_ListObjAppendElement(magicinterp, lobj,
			Tcl_NewIntObj(DBLambda[1]));
 	    Tcl_SetObjResult(magicinterp, lobj);
	    break;
	case TECH_PLANES:
	    for (i = 0; i < DBNumPlanes; i++)
		Tcl_AppendElement(magicinterp, DBPlaneLongName(i));
	    break;
#else
	case TECH_NAME:
	    TxPrintf("Technology name is \"%s\"\n", DBTechName);
	    break;
	case TECH_FILE:
	    TxPrintf("Technology filename is \"%s\"\n", TechFileName);
	    break;
	case TECH_VERSION:
	    TxPrintf("Technology version is: \"%s\"\n", DBTechVersion);
	    TxPrintf("Description: %s\n", DBTechDescription);
	    break;
	case TECH_LAMBDA:
	    if ((cmd->tx_argc > 2) && (StrIsInt(cmd->tx_argv[2])))
	    {
		DBLambda[1] = atoi(cmd->tx_argv[2]);
		if (cmd->tx_argc == 3)
		    DBLambda[0] = 1;
		else if ((cmd->tx_argc > 3) && (StrIsInt(cmd->tx_argv[3])))
		DBLambda[0] = atoi(cmd->tx_argv[3]);
		ReduceFraction(&DBLambda[0], &DBLambda[1]);
	    }

	    TxPrintf("Internal units per Lambda = %d / %d\n",
		DBLambda[0], DBLambda[1]);
	    break;
	case TECH_PLANES:
	    TxPrintf("Defined planes (%d) are:\n", DBNumPlanes);
	    for (i = 0; i < DBNumPlanes; i++)
		TxPrintf("   %s\n", DBPlaneLongName(i));
	    break;
#endif

	case TECH_LAYERS:
	    if (cmd->tx_argc == 3)
	    {
		if (!strcmp(cmd->tx_argv[2], "*"))
		    DBTechPrintTypes(TRUE);
		else
		    DBTechPrintCanonicalType(cmd->tx_argv[2]);
	    }
	    else if (cmd->tx_argc == 2)
		DBTechPrintTypes(FALSE);
	    else
		goto wrongNumArgs;
	    break;

	case TECH_LOAD:
	    locargc = cmd->tx_argc;
	    while ((*cmd->tx_argv[locargc - 1] == '-') && (locargc > 3))
	    {
		if (!strcmp(cmd->tx_argv[locargc - 1], "-nooverride"))
		{
		    /* Disallow the "tech load" command from overriding */
		    /* a technology specified on the command line.	*/
		    /* This is used when "tech load" sets up a specific	*/
		    /* project technology from a .magic startup script.	*/

		    if (TechOverridesDefault)
			return;
		}
		else if (!strcmp(cmd->tx_argv[locargc - 1], "-noprompt"))
		    noprompt = TRUE;
		else
		    break;
		locargc--;
	    }

	    if (locargc != 3)
	    {
		TxError("Usage: tech load <filename> [-noprompt] [-nooverride]\n");
		break;
	    }

	    /* Here:  We need to do a test to see if any structures	*/
	    /* exist in memory and delete them, or else we need to have	*/
	    /* all tech stuff set up such that multiple technologies	*/
	    /* can be present at the same time.				*/

	    /* For now: warn without checking */

	    if (!noprompt)
	    {
		action = TxDialog("Technology file (re)loading may invalidate"
			" any existing layout.  Continue? ", actionNames, 0);
		if (action == 0) return;
	    }
	
	    /* Changing number of planes requires handling on every	*/
	    /* celldef.  So we need to save the original number of	*/
	    /* planes to see if it shrinks or expands.			*/

	    saveNumPlanes = DBNumPlanes;

	    /* CIF istyle, CIF ostyle, and extract sections need calls	*/
	    /* to the init functions which clean up memory devoted to	*/
	    /* remembering all the styles.				*/

#ifdef CIF_MODULE
	    CIFTechInit();
	    CIFReadTechInit();
#endif
	    ExtTechInit();

	    if (!TechLoad(cmd->tx_argv[2], 0))
	    {
#ifdef MAGIC_WRAPPER
		Tcl_SetResult(magicinterp, "Error in loading technology file\n", NULL);
#else
		TxError("Error in loading technology file\n");
#endif
		break;
	    }

	    /* If internal scalefactor is not the default 1:1, then we	*/
	    /* need to scale the techfile numbers accordingly.		*/

	    if ((DBLambda[0] != 1) || (DBLambda[1] != 1))
	    {
		int d = DBLambda[0];
		int n = DBLambda[1];

		CIFTechInputScale(d, n);
		CIFTechOutputScale(d, n);
		DRCTechScale(d, n);
		ExtTechScale(d, n);
		WireTechScale(d, n);
#ifdef ROUTE_MODULE
		RtrTechScale(d, n);
#endif
		TxPrintf("Scaled tech values by %d / %d to"
			" match internal grid scaling\n", n, d);
	    }

	    if (saveNumPlanes != DBNumPlanes)
	    {
		TxError("Warning:  Number of planes has changed.  Any"
			" existing layout may be invalid.\n");
		(void) DBCellSrDefs(0, changePlanesFunc, (ClientData) &saveNumPlanes);
	    }
	    break;

	case TECH_HELP:
	    TxPrintf("Tech commands have the form \"tech option\",\n");
	    TxPrintf("where option is one of:\n");
	    for (msg = &(cmdTechOption[0]); *msg != NULL; msg++)
	    {
		TxPrintf("    %s\n", *msg);
	    }
	    break;
    }
    return;

wrongNumArgs:
    TxPrintf("wrong number of arguments to command \"%s\"\n", cmd->tx_argv[0]);
    goto usage2;

usage:
    TxError("\"%s\" isn't a valid %s option.", cmd->tx_argv[1], cmd->tx_argv[0]);

usage2:
    TxError("  Type \":%s help\" for help.\n", cmd->tx_argv[0]);
    return;
}

/*
 * ----------------------------------------------------------------------------
 *
 * This function hacks the existing layout database in case a tech file
 * is loaded which contains more or fewer planes than the exisiting
 * technology.  This is doing nothing fancy; it is simply making sure
 * that all memory allocation is accounted for.
 *
 * As a note for future implementation, it would be helpful to keep the
 * old plane name definitions around and try to match up the old and new
 * planes, so that it is possible to load a technology file which matches
 * the existing technology except for the addition or subtraction of one
 * or more planes (e.g., extra metal layer option) without completely
 * invalidating an existing layout.
 *
 * As written, this function is inherently dangerous.  It is intended for
 * use when loading a new tech file when there is no layout, just empty
 * tile planes.
 * ----------------------------------------------------------------------------
 */

int
changePlanesFunc(cellDef, arg)
    CellDef *cellDef;
    int *arg;
{
    int oldnumplanes = *arg;
    int pNum;

    if (oldnumplanes < DBNumPlanes)
    {
	/* New planes to be added */
	for (pNum = oldnumplanes; pNum < DBNumPlanes; pNum++)
	{
	    cellDef->cd_planes[pNum] = DBNewPlane((ClientData) TT_SPACE);
	}
    }
    else
    {
	/* Old planes to be subtracted */
	for (pNum = DBNumPlanes; pNum < oldnumplanes; pNum++)
	{
	    DBFreePaintPlane(cellDef->cd_planes[pNum]);
	    TiFreePlane(cellDef->cd_planes[pNum]);
	    cellDef->cd_planes[pNum] = (Plane *) NULL;
	}
    }
    return 0;
}


/*
 * ----------------------------------------------------------------------------
 *
 * CmdTool --
 *
 * 	Implement the "tool" command.
 *
 * Usage:
 *	tool [name|info]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current tool that's active in layout windows may be changed.
 *
 * ----------------------------------------------------------------------------
 */

	/* ARGSUSED */
Void
CmdTool(w, cmd)
    MagWindow *w;		/* Window in which command was invoked. */
    TxCommand *cmd;		/* Info about command options. */
{
    if (cmd->tx_argc == 1)
    {
	(void) DBWChangeButtonHandler((char *) NULL);
	return;
    }

    if (cmd->tx_argc > 2)
    {
	TxError("Usage: %s [name|info]\n", cmd->tx_argv[0]);
	return;
    }

    if (strcmp(cmd->tx_argv[1], "info") == 0)
	DBWPrintButtonDoc();
    else (void) DBWChangeButtonHandler(cmd->tx_argv[1]);
}

/*
 * ----------------------------------------------------------------------------
 *
 * CmdUnexpand --
 *
 * Implement the "unexpand" command.
 *
 * Usage:
 *	unexpand
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Unexpands all cells under the box that don't completely
 *	contain the box.
 *
 * ----------------------------------------------------------------------------
 */

CmdUnexpand(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    int windowMask, boxMask;
    Rect rootRect;
    int cmdUnexpandFunc();		/* Forward reference. */

    if (cmd->tx_argc != 1)
    {
	TxError("Usage: %s\n", cmd->tx_argv[0]);
	return;
    }
    
    windCheckOnlyWindow(&w, DBWclientID);
    if (w == (MagWindow *) NULL)
    {
	TxError("Point to a window first.\n");
	return;
    }
    windowMask = ((DBWclientRec *) w->w_clientData)->dbw_bitmask;

    (void) ToolGetBoxWindow(&rootRect, &boxMask);
    if ((boxMask & windowMask) != windowMask)
    {
	TxError("The box isn't in the same window as the cursor.\n");
	return;
    }
    DBExpandAll(((CellUse *) w->w_surfaceID), &rootRect, windowMask,
	    FALSE, cmdUnexpandFunc, (ClientData) windowMask);
}

/* This function is called for each cell whose expansion status changed.
 * It forces the cells area to be redisplayed, then returns 0 to keep
 * looking for more cells to unexpand.
 */

int
cmdUnexpandFunc(use, windowMask)
    CellUse *use;		/* Use that was just unexpanded. */
    int windowMask;		/* Window where it was unexpanded. */
{
    if (use->cu_parent == NULL) return 0;
    DBWAreaChanged(use->cu_parent, &use->cu_bbox, windowMask,
	    (TileTypeBitMask *) NULL);
    return 0;
}


/*
 * ----------------------------------------------------------------------------
 *
 * CmdUpsidedown --
 *
 * Implement the "upsidedown" command.
 *
 * Usage:
 *	upsidedown
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The box and verything in the selection are flipped upside down
 *	using the point as the axis around which to flip.
 *
 * ----------------------------------------------------------------------------
 */
    
    /* ARGSUSED */

CmdUpsidedown(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    Transform trans;
    Rect rootBox, bbox;
    CellDef *rootDef;

    if (cmd->tx_argc != 1)
    {
	TxError("Usage: %s\n", cmd->tx_argv[0]);
	return;
    }

    
    /* To flip the selection upside down, first flip it around the
     * x-axis, then move it back so its lower-left corner is in
     * the same place that it used to be.
     */
    
    GeoTransRect(&GeoUpsideDownTransform, &SelectDef->cd_bbox, &bbox);
    GeoTranslateTrans(&GeoUpsideDownTransform,
	SelectDef->cd_bbox.r_xbot - bbox.r_xbot,
	SelectDef->cd_bbox.r_ybot - bbox.r_ybot, &trans);

    SelectTransform(&trans);

    /* Flip the box, if it exists and is in the same window as the
     * selection.
     */
    
    if (ToolGetBox(&rootDef, &rootBox) && (rootDef == SelectRootDef))
    {
	Rect newBox;

	GeoTransRect(&trans, &rootBox, &newBox);
	DBWSetBox(rootDef, &newBox);
    }

    return;
}

/*
 * ----------------------------------------------------------------------------
 *
 * CmdWhat --
 *
 * 	Print out information about what's selected.
 *
 * Usage:
 *	what
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets printed to identify the kinds of paint, plus
 *	labels and subcells, that are selected.
 *
 * ----------------------------------------------------------------------------
 */
typedef struct labelstore
{
    TileType lab_type;
    char *lab_text;
    char *cell_name;
} LabelStore;

static int moreLabelEntries, labelEntryCount;
static LabelStore *labelBlockTop, *labelEntry;

    /* ARGSUSED */

CmdWhat(w, cmd)
    MagWindow *w;		/* Window in which command was invoked. */
    TxCommand *cmd;		/* Information about the command. */
{
    bool foundAny;
    TileTypeBitMask layers;
    extern int cmdWhatPaintFunc(), cmdWhatLabelFunc(), cmdWhatCellFunc();
    extern int cmdWhatLabelPreFunc(), orderLabelFunc();
    if (cmd->tx_argc > 1)
    {
	TxError("Usage: what\n");
	return;
    }

    /* Find all the selected paint and print out the layer names. */

    TTMaskZero(&layers);
    (void) SelEnumPaint(&DBAllButSpaceAndDRCBits, FALSE, (bool *) NULL,
	    cmdWhatPaintFunc, (ClientData) &layers);
    if (!TTMaskIsZero(&layers))
    {
	int i;

	TxPrintf("Selected mask layers:\n");
	for (i = TT_SELECTBASE; i < DBNumUserLayers; i++)
	{
	    if (TTMaskHasType(&layers, i))
		TxPrintf("    %s\n", DBTypeLongName(i));
	}
    }

    /* Enumerate all of the selected labels. */

    moreLabelEntries = 0;
    labelEntryCount = 0;
    labelBlockTop = NULL; 
    (void) SelEnumLabels(&DBAllTypeBits, FALSE, (bool *) NULL,
	    cmdWhatLabelPreFunc, (ClientData) &foundAny);
    foundAny = FALSE;
    if (labelBlockTop)
    {
	int i;

	qsort(labelBlockTop, labelEntryCount, sizeof(LabelStore), orderLabelFunc);
    		/* now print them out */
	for (labelEntry = labelBlockTop; labelEntryCount-- > 0; labelEntry++)
            i = cmdWhatLabelFunc(labelEntry, &foundAny);
	if (i > 1)
		TxPrintf(" (%i instances)", i);
        TxPrintf("\n");
	freeMagic(labelBlockTop);
    }
    

    /* Enumerate all of the selected subcells. */

    foundAny = FALSE;
    (void) SelEnumCells(FALSE, (bool *) NULL, (SearchContext *) NULL,
	    cmdWhatCellFunc, (ClientData) &foundAny);
}

/* Search function invoked for each paint tile in the selection:
 * just set a bit in a tile type mask.
 */

    /*ARGSUSED*/
int
cmdWhatPaintFunc(rect, type, mask)
    Rect *rect;			/* Not used. */
    TileType type;		/* Type of this piece of paint. */
    TileTypeBitMask *mask;	/* Place to OR in type's bit. */
{
#ifdef NONMANHATTAN 
    if (type & TT_DIAGONAL)
	type = (type & TT_SIDE) ? (type & TT_RIGHTMASK) >> 14 :
		(type & TT_LEFTMASK);
#endif
    TTMaskSetType(mask, type);
    return 0;
}

/* Search function invoked for each label in the selection:  print
 * out information about the label.
 */

    /*ARGSUSED*/
int
cmdWhatLabelPreFunc(label, cellDef, transform, foundAny)
    Label *label;		/* Label that's selected. */
    CellDef *cellDef;		/* Cell definition containing label. */
    Transform *transform;	/* Not used. */
    bool *foundAny;		/* Use to print extra stuff for the first
				 * label found.
				 */
{
    LabelStore	*newPtr;

    if (moreLabelEntries == 0)
    {
	newPtr = (LabelStore *)mallocMagic((labelEntryCount + 100)
		* sizeof(LabelStore));
	if (!newPtr)
		return 1;	/* no space stop the search */
	if (labelBlockTop)
	{
	   memcpy(newPtr, labelBlockTop, labelEntryCount * sizeof(LabelStore));
	   freeMagic(labelBlockTop);
	}
	labelBlockTop = newPtr;
        labelEntry = &labelBlockTop[labelEntryCount];
	moreLabelEntries = 100;
    }
	/* store the pointers for sorting later */	
    labelEntry->lab_type = label->lab_type;
    labelEntry->lab_text = label->lab_text;
    labelEntry->cell_name = cellDef->cd_name;
    labelEntry++;
    moreLabelEntries--;
    labelEntryCount++;
    return 0;
}
	

int
cmdWhatLabelFunc(entry, foundAny)
    LabelStore *entry;		/* stored pointers to label info*/
    bool *foundAny;		/* Use to print extra stuff for the first
				 * label found.
				 */
{
    static TileType last_type;
    static char *last_name, *last_cell;
    static int counts;

    if (!*foundAny)
    {
	TxPrintf("Selected label(s):");
	*foundAny = TRUE;
        last_name = NULL;
	counts = 0;
    }

    if ((last_name && (strcmp(entry->lab_text, last_name) !=0 ||
                       strcmp(entry->cell_name, last_cell) != 0)) ||
       (entry->lab_type != last_type) || (last_name == NULL))
    {
	if (counts > 1)
		TxPrintf(" (%i instances)", counts);
        TxPrintf("\n    \"%s\" is attached to %s in cell %s", entry->lab_text,
	    DBTypeLongName(entry->lab_type), entry->cell_name);
        last_type = entry->lab_type;
        last_cell = entry->cell_name;
        last_name = entry->lab_text;
	counts = 1;
    } else 
	counts++;
    return counts;
}

/* orderLabelFunc is called by qsort to compare the labels */
/* they are sorted by label name, then cell name, then attached material */
/* that way all of identical names are grouped together */
int
orderLabelFunc(one, two)
    LabelStore *one;		/* one of the labels being compared */
    LabelStore *two;		/* the other label to compare with */
{
    int i;

    if ((i = strcmp(one->lab_text, two->lab_text)) != 0)
        return i;
    if ((i = strcmp(one->cell_name, two->cell_name)) != 0)
        return i;
    if (one->lab_type != two->lab_type)
	return (one->lab_type < two->lab_type) ? 1 : -1;
    return 0;
}

/* Search function invoked for each selected subcell.  Just print out
 * its name and use id.
 */

    /*ARGSUSED*/
int
cmdWhatCellFunc(selUse, realUse, transform, foundAny)
    CellUse *selUse;		/* Not used. */
    CellUse *realUse;		/* Selected cell use. */
    Transform *transform;	/* Not used. */
    bool *foundAny;		/* Used to print extra stuff for the first
				 * use found.
				 */
{
    if (!*foundAny)
    {
	TxPrintf("Selected subcell(s):\n");
	*foundAny = TRUE;
    }
    TxPrintf("    Instance \"%s\" of cell \"%s\"\n", realUse->cu_id,
	    realUse->cu_def->cd_name);
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * CmdWire --
 *
 * Implement the "wire" command, which provides a wiring-style interface
 * for painting.
 *
 * Usage:
 *	wire option args
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The edit cell is modified to contain additional paint.
 *
 * ----------------------------------------------------------------------------
 */

#define HELP		0
#define HORIZONTAL	1
#define LEG		2
#define SWITCH		3
#define TYPE		4
#define VERTICAL	5

	/* ARGSUSED */
CmdWire(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    int option;
    char **msg;
    TileType type;
    int width;

    static char *cmdWireOption[] =
    {	
	"help                   print this help information",
	"horizontal             add a new horizontal wire leg",
	"leg                    add a new horizontal or vertical leg",
	"switch [layer width]   place contact and switch layers",
	"type [layer width]     select the type and size of wires",
	"vertical               add a new vertical wire leg",
	NULL
    };

    if (cmd->tx_argc < 2)
    {
	option = HELP;
	cmd->tx_argc = 2;
    }
    else
    {
	option = Lookup(cmd->tx_argv[1], cmdWireOption);
	if (option < 0)
	{
	    TxError("\"%s\" isn't a valid wire option.\n", cmd->tx_argv[1]);
	    option = HELP;
	    cmd->tx_argc = 2;
	}
    }

    switch (option)
    {
	case HELP:
	    TxPrintf("Wiring commands have the form \":wire option\",");
	    TxPrintf(" where option is one of:\n");
	    for (msg = &(cmdWireOption[0]); *msg != NULL; msg++)
	    {
		TxPrintf("    %s\n", *msg);
	    }
	    return;
	
	case HORIZONTAL:
	    WireAddLeg((Rect *) NULL, (Point *) NULL, WIRE_HORIZONTAL);
	    return;
	
	case LEG:
	    WireAddLeg((Rect *) NULL, (Point *) NULL, WIRE_CHOOSE);
	    return;

	case SWITCH:
	    if (cmd->tx_argc == 2)
		WireAddContact(-1, 0);
	    else if (cmd->tx_argc != 4)
		goto badargs;
	    else
	    {
		type = DBTechNameType(cmd->tx_argv[2]);
		if (type == -2)
		{
		    TxError("Layer name \"%s\" doesn't exist.\n",
			    cmd->tx_argv[2]);
		    return;
		}
		else if (type == -1)
		{
		    TxError("Layer name \"%s\" is ambiguous.\n",
			    cmd->tx_argv[2]);
		    return;
		}
		width = cmdParseCoord(w, cmd->tx_argv[3], TRUE, TRUE);
		WireAddContact(type, width);
		return;
	    }
	    break;

	case TYPE:
	    if (cmd->tx_argc == 2)
		WirePickType(-1, 0);
	    else if (cmd->tx_argc != 4)
	    {
		badargs:
		TxError("Wrong arguments.  The correct syntax is\n");
		TxError("    \"wire %s\"\n", cmdWireOption[option]);
		return;
	    }
	    else
	    {
		type = DBTechNameType(cmd->tx_argv[2]);
		if (type == -2)
		{
		    TxError("Layer name \"%s\" doesn't exist.\n",
			    cmd->tx_argv[2]);
		    return;
		}
		else if (type == -1)
		{
		    TxError("Layer name \"%s\" is ambiguous.\n",
			    cmd->tx_argv[2]);
		    return;
		}
		width = cmdParseCoord(w, cmd->tx_argv[3], TRUE, TRUE);
		WirePickType(type, width);
		return;
	    }
	    break;

	case VERTICAL:
	    WireAddLeg((Rect *) NULL, (Point *) NULL, WIRE_VERTICAL);
	    return;
    }
}

/*
 * ----------------------------------------------------------------------------
 *
 * CmdWriteall --
 *
 * Implement the "writeall" command.
 * Write out all modified cells to disk.
 *
 * Usage:
 *	writeall [force]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	For each cell that has been modified since it was last written,
 *	the user is asked whether he wants to write it, flush it,
 *	skip it, or abort the "writeall" command.  If the decision
 *	is made to write, the cell is written out to disk and the
 *	modified bit in its definition's flags is cleared.  If the
 *	decision is made to flush, all paint and subcell uses are
 *	removed from the cell, and it is re-read from disk.
 *
 * ----------------------------------------------------------------------------
 */

static bool cmdWriteAllForce;  	/* If true, the user specified 'force' */

    /* ARGSUSED */

CmdWriteall(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    int cmdWriteallFunc();
    bool autoWrite;
    static char *force[] = { "force", 0 };

    if ((cmd->tx_argc > 2) || 
	((cmd->tx_argc == 2) && (Lookup(cmd->tx_argv[1], force) < 0)))
    {
	TxError("Usage: %s [force]\n", cmd->tx_argv[0]);
	return;
    }

    cmdWriteAllForce = autoWrite = (cmd->tx_argc == 2);

    DBUpdateStamps();
    (void) DBCellSrDefs(CDMODIFIED|CDBOXESCHANGED|CDSTAMPSCHANGED,
	cmdWriteallFunc, (ClientData) &autoWrite);
}

/*
 * Filter function used by CmdWriteall() above.
 * This function is called for each known CellDef whose modified bit
 * is set.
 */

    /*ARGSUSED*/
int
cmdWriteallFunc(def, autoWrite)
    CellDef *def;	/* Pointer to CellDef to be saved.  This def might
			 * be an internal buffer; if so, we ignore it.
			 */
    bool *autoWrite;	/* Client data passed to DBCellSrDefs; pointer to
			 * flag:  if flag true, then write without asking.
			 * Otherwise, ask user whether or not to write
			 * dirty cells.
			 */
{
    char *prompt;
    int action, cidx = 0;
    static char *actionNames[] =
        { "write", "flush", "skip", "abort", "autowrite", 0 };
    static char *explain[] =
	{ "", "(bboxes)", "(timestamps)", "(bboxes/timestamps)", 0 };

    if (def->cd_flags & CDINTERNAL) return 0;
    if (SigInterruptPending) return 1;

    if (*autoWrite)
    {
	action = 4;
    }
    else
    {
	if (!(def->cd_flags & CDMODIFIED))
	{
	    if (!(def->cd_flags & CDSTAMPSCHANGED))
		cidx = 1;
	    else if (!(def->cd_flags & CDBOXESCHANGED))
		cidx = 2;
	    else cidx = 3;
	}
	prompt = TxPrintString("%s %s: write, autowrite, flush, skip, "
		"or abort command? ", def->cd_name, explain[cidx]);
	action = TxDialog(prompt, actionNames, 0);
    }

    switch (action)
    {
	case 0:		/* Write */
	    cmdSaveCell(def, (char *) NULL, cmdWriteAllForce, TRUE);
	    break;
	case 1:		/* Flush */
	    cmdFlushCell(def);
	    break;
	case 2:		/* Skip */
	    break;
	case 3:		/* Abort command */
	    return 1;
	case 4:		/* Automatically write everything */
	    *autoWrite = TRUE;
	    TxPrintf("Writing '%s'\n", def->cd_name);
	    cmdSaveCell(def, (char *) NULL, cmdWriteAllForce, TRUE);
	    break;
    }
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * CmdXload --
 *
 * Implement the "xload" command.
 *
 * Usage:
 *	xload [name]
 *
 * If name is supplied, then the window containing the point tool is
 * remapped so as to edit the cell with the given name.
 *
 * If no name is supplied, then a new cell with the name "(UNNAMED)"
 * is created in the selected window.  If there is already a cell by
 * that name in existence (eg, in another window), that cell gets loaded
 * rather than a new cell being created.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets EditCellUse.
 *
 * ----------------------------------------------------------------------------
 */

Void
CmdXload(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    windCheckOnlyWindow(&w, DBWclientID);
    if (w == (MagWindow *) NULL)
    {
	TxError("Point to a window first.\n");
	return;
    }

    if (cmd->tx_argc > 2)
    {
	TxError("Usage: %s [name]\n", cmd->tx_argv[0]);
	return;
    }

    if (cmd->tx_argc == 2)
    {
	if (CmdIllegalChars(cmd->tx_argv[1], "[],", "Cell name"))
	    return;
	DBWxloadWindow(w, cmd->tx_argv[1]);
    }
    else DBWxloadWindow(w, (char *) NULL);
}
