/*
 * lef.c --      
 *
 * This module incorporates the LEF/DEF format for standard-cell place and
 * route.
 *
 * Version 0.1 (May 1, 2003):  LEF output for cells, to include pointer to
 * GDS, automatic generation of GDS if not already made, bounding box export,
 * port export, export of irouter "fence", "magnet", and "rotate" layers    
 * for defining router hints, and generating areas for obstructions and
 * pin layers.
 *
 */

#ifndef lint
static char rcsid[] = "$Header: /ufs/repository/magic/lef/lef.c,v 1.0 2003/05/01 10:37:00 tim Exp $";            
#endif  /* not lint */

#include <stdio.h>
#include <stdlib.h>
#ifdef SYSV
#include <string.h>
#endif
#include <errno.h>
#include <stdarg.h>

#include "magic/tclmagic.h"
#include "misc/magic.h"
#include "utils/geometry.h"
#include "tiles/tile.h"
#include "utils/hash.h"
#include "database/database.h"
#include "windows/windows.h"
#include "dbwind/dbwind.h"
#include "utils/malloc.h"
#include "lef/lefInt.h"

/* ---------------------------------------------------------------------

/* Current line number for reading */
int lefCurrentLine;

/* Cell reading hash tables */
HashTable LefCellTable;
HashTable lefDefInitHash;

/* Some constants for LEF files */

#define LEF_LINE_MAX 2048  /* Maximum length fixed by LEF specifications */
#define LEF_MAX_ERRORS 100 /* Max # errors to report; limits output if */
			   /* something is really wrong about the file */

/*
 *------------------------------------------------------------
 *
 * LefNextLine --
 *
 *	Find the next line to process in a LEF file
 *
 * Results:
 *	Pointer to next line to parse
 *
 * Side Effects:
 *	May read a new line from the specified file.
 *
 *------------------------------------------------------------
 */

char *
LefNextLine(f)
    FILE *f;
{
    static char *nexttoken = NULL;
    static char line[LEF_LINE_MAX];
    char *lineptr;
    char *lineend;

    if (nexttoken == NULL)
    {
	/* Read another line from the file */

	while (nexttoken == NULL)
	{
	    if (fgets(line, LEF_LINE_MAX, f) == NULL)
		return NULL; 
	    lefCurrentLine++;
	    nexttoken = line;
	    while (isspace(*nexttoken) || (*nexttoken == ';'))
		nexttoken++;
	    if (*nexttoken == '\0')
		nexttoken = NULL;
	}
	lineptr = nexttoken;

	/* Remove any newline character from the string */
	if ((lineend = strchr(lineptr, '\n')) != NULL) *lineend = '\0';
    }
    else
	lineptr = nexttoken;

    /* Find the next complete LEF statement and set "nexttoken" to */
    /* point to it.  If we encounter EOL, set nexttoken to NULL.   */

    nexttoken = strchr(nexttoken, ';');
    if (nexttoken != NULL)
    {
	*nexttoken++ = '\0';
	while (isspace(*nexttoken) || (*nexttoken == ';'))
	    nexttoken++;
	if (*nexttoken == '\0')
	    nexttoken = NULL;
    }
    return lineptr;
}

/*
 *------------------------------------------------------------
 *
 * LefError --
 *
 *	Print an error message (via TxError) giving the line
 *	number of the input file on which the error occurred.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Prints to the output (stderr).
 *
 *------------------------------------------------------------
 */

void
LefError(char *fmt, ...)
{  
    static int errors = 0;
    va_list args;

    if (fmt == NULL)
    {
	if (errors)
	    TxPrintf("LEF Read: encountered %d error%s total.\n", errors,
			(errors == 1) ? "" : "s");
	return;
    }

    if (errors < LEF_MAX_ERRORS)
    {
	TxError("LEF Read, Line %d: ", lefCurrentLine);
	va_start(args, fmt);
	(void) GrVfprintf(stderr, fmt, args);
	va_end(args);
    }
    else if (errors == LEF_MAX_ERRORS)
	TxError("LEF Read:  Further errors will not be reported.\n");

    errors++;
}

/*
 *------------------------------------------------------------
 *
 * LefParseEndStatement --
 *
 *	Check if the string passed in "lineptr" contains the
 *	appropriate matching keyword.  Sections in LEF files
 *	should end with "END (keyword)" or "END".  To check
 *	against the latter case, "match" should be NULL.
 *
 * Results:
 *	TRUE if the line matches the expected end statement,
 *	FALSE if not. 
 *
 * Side effects:
 *	None.
 *
 *------------------------------------------------------------
 */

bool
LefParseEndStatement(lineptr, match)
    char *lineptr;
    char *match;
{
    char token[256];
    int keyword, words;
    char *match_name[2];

    match_name[0] = match;
    match_name[1] = NULL;

    words = sscanf(lineptr, "%*s %255s", token);

    if ((words <= 0) && (match == NULL)) return TRUE;
    else if (words != 1) return FALSE;

    keyword = Lookup(token, match_name);
    if (keyword == 0)
	return TRUE;
    else
	return FALSE;
}

/*
 *------------------------------------------------------------
 *
 * LefSkipSection --
 *
 *	Skip to the "END" record of a LEF input section
 *	String "section" must follow the "END" statement in
 *	the file to be considered a match;  however, if
 *	section = NULL, then "END" must have no strings
 *	following.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Reads input from the specified file.  Prints an
 *	error message if the expected END record cannot
 *	be found.
 *
 *------------------------------------------------------------
 */

void
LefSkipSection(f, section)
    FILE *f;
    char *section;
{
    char *lineptr;
    char token[256];
    static char *end_section[] = {
	"END",
	NULL
    };

    while ((lineptr = LefNextLine(f)) != NULL)
	if (sscanf(lineptr, "%255s", token) == 1)
	    if (Lookup(token, end_section) == 0)
		if (LefParseEndStatement(lineptr, section))
		    return;

    LefError("Section %s has no END record!\n", section);
    return;
}

/*
 *------------------------------------------------------------
 *
 * lefFindCell --
 *
 *------------------------------------------------------------
 */

CellDef *
lefFindCell(name)
    char *name;		/* Name of the cell to search for */
{
    HashEntry *h;
    CellDef *def;

    h = HashFind(&LefCellTable, name);
    if (HashGetValue(h) == 0)
    {
	def = DBCellLookDef(name);
	if (def == NULL)
	{
	    def = DBCellNewDef(name, (char *)NULL);
	    DBReComputeBbox(def);
	}
	HashSetValue(h, def);
    }
    return (CellDef *) HashGetValue(h);
}

/*
 *------------------------------------------------------------
 *
 * LefSkipToken --
 *
 *	Skip past the current token in the specified line.
 *	Write an end-of-string in front of all whitespace
 *	at the end of the line.
 *
 * Results:
 *	Pointer to the start of the next keyword (token).
 *
 * Side effects:
 *	None.
 *
 *------------------------------------------------------------
 */

char *
LefSkipToken(lineptr)
    char *lineptr;
{
    char *tokptr = lineptr;
    char *endptr;

    while (isspace(*tokptr)) tokptr++;  /* skip whitespace */
    while (!isspace(*tokptr)) tokptr++;	/* skip keyword */
    while (isspace(*tokptr)) tokptr++;  /* skip whitespace */

    endptr = tokptr;
    while (*endptr != '\0') endptr++;	/* go to end of line */
    while (isspace(*(endptr - 1)))
	endptr--;			/* skip whitespace */
    *endptr = '\0';			/* truncate line */

    return tokptr;
}

/*
 *------------------------------------------------------------
 * LefReadGeometry --
 *
 *	Read Geometry information from a LEF file.
 *	Used for PORT records and OBS statements.
 *	pinNum should be -1 for reading obstruction
 *	geometry.
 *
 * Results:
 *	Returns a linked list of all areas and types
 *	painted.  However, if "do_list" is FALSE, then
 *	LefReadGeometry always returns NULL.
 *
 * Side Effects:
 *	Reads input from file f;
 *	Paints into the CellDef lefMacro.
 *
 *------------------------------------------------------------
 */

typedef struct _linkedRect {
   Rect	     area;
   TileType  type;
   struct _linkedRect *rect_next;
} linkedRect; 

enum lef_geometry_keys {LEF_LAYER = 0, LEF_WIDTH, LEF_PATH,
	LEF_RECT, LEF_POLYGON, LEF_VIA, LEF_GEOMETRY_END};

linkedRect *
LefReadGeometry(lefMacro, f, do_list)
    CellDef *lefMacro;
    FILE *f;
    bool do_list;
{
    TileType curlayer;

    char token[256], *tptr;
    char *lineptr;
    int keyword;
    float llx, lly, urx, ury;
    float oscale;
    Rect paintrect;
    linkedRect *newRect, *rectList;

    static char *geometry_keys[] = {
	"LAYER",
	"WIDTH",
	"PATH",
	"RECT",
	"POLYGON",
	"VIA",
	"END",
	NULL
    };

    oscale = CIFGetOutputScale(1000);
    rectList = NULL;

    while ((lineptr = LefNextLine(f)) != NULL)
    {
	if (sscanf(lineptr, "%s", token) == 0)
	    continue;	/* Blank line or null statement */

	keyword = Lookup(token, geometry_keys);
	if (keyword < 0)
	{
	    LefError("Unknown keyword \"%s\" in LEF file; ignoring.\n", token);
	    continue;
	}
	switch (keyword)
	{
	    case LEF_LAYER:
		if (sscanf(lineptr, "%*s %s", token) != 1)
		    LefError("Bad Layer statement\n");
		else
		{
		    /* NOTE:  magic-to-LEF layer mappings should be	*/
		    /* declared in the technology file.  This ad-hoc	*/
		    /* mapping is temporary!				*/

		    for (tptr = token; *tptr != '\0'; *tptr++)
			*tptr = tolower(*tptr);
	
		    if (!strncmp(token, "metal", 5))
		    {
			curlayer = DBTechNameType(token);
		    }
		    else if (!strncmp(token, "via", 3))
		    {
			curlayer = DBTechNameType(token);
		    }
		    else if (!strncmp(token, "poly", 4))
		    {
			curlayer = DBTechNameType(token);
		    }
		    if (curlayer < 0)
		    {
			LefError("Don't know how to parse layer \"%s\"\n", token);
			LefError("Try adding this name to the technology styles\n");
		    }
		}
		break;
	    case LEF_WIDTH:
		break;
	    case LEF_PATH:
		break;
	    case LEF_RECT:
		tptr = LefSkipToken(lineptr);
		if (sscanf(tptr, "%f %f %f %f", &llx, &lly, &urx, &ury) != 4)
		    LefError("Bad port geometry: RECT requires 4 values.\n");
		if (curlayer < 0)
		    LefError("No layer defined for RECT.\n");
		else
		{
		    /* Scale coordinates (microns to magic internal units) */
		    /* Need to scale grid if necessary!			   */
		
		    paintrect.r_xbot = (int)(llx / oscale);
		    paintrect.r_ybot = (int)(lly / oscale);
		    paintrect.r_xtop = (int)(urx / oscale);
		    paintrect.r_ytop = (int)(ury / oscale);

		    /* Paint the area */
		    DBPaint(lefMacro, &paintrect, curlayer);

		    /* Remember the area */
		    if (do_list)
		    {
			newRect = (linkedRect *)mallocMagic(sizeof(linkedRect));
			newRect->type = curlayer;
			newRect->area = paintrect;
			newRect->rect_next = rectList;
			rectList = newRect;
		    }

		    TxPrintf("   Painting %s at (%d, %d), (%d, %d) in cell %s.\n",
			DBTypeLongNameTbl[curlayer],
			paintrect.r_xbot, paintrect.r_ybot,
			paintrect.r_xtop, paintrect.r_ytop,
			lefMacro->cd_name);
		}
		break;
	    case LEF_POLYGON:
		break;
	    case LEF_VIA:
		break;
	    case LEF_GEOMETRY_END:
		if (!LefParseEndStatement(lineptr, NULL))
		{
		    LefError("Geometry (PORT or OBS) END statement missing.\n");
		    keyword = -1;
		}
		break;
	}
	if (keyword == LEF_GEOMETRY_END) break;
    }
    return rectList;
}

/*
 *------------------------------------------------------------
 * LefReadPort --
 *
 *	A wrapper for LefReadGeometry, which adds a label
 *	to the last rectangle generated by the geometry
 *	parsing.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Reads input from file f;
 *	Paints into the CellDef lefMacro.
 *
 *------------------------------------------------------------
 */

void
LefReadPort(lefMacro, f, pinName, pinNum, pinDir, pinUse)
    CellDef *lefMacro;
    FILE *f;
    char *pinName;
    int pinNum, pinDir, pinUse;
{
    Label *newlab;
    linkedRect *rectList;

    rectList = LefReadGeometry(lefMacro, f);

    while (rectList != NULL)
    {
	if (pinNum >= 0)
	{
	    /* Label this area */
	    DBPutLabel(lefMacro, &rectList->area, -1, pinName, rectList->type, 0);

	    /* Set this label to be a port */

	    if (lefMacro->cd_labels == NULL)
		LefError("Internal error: No labels in cell!\n");
	    else
	    {
		for (newlab = lefMacro->cd_labels; newlab->lab_next != NULL;
				newlab = newlab->lab_next);  /* Nothing */

		if (strcmp(newlab->lab_text, pinName))
		    LefError("Internal error:  Can't find the label!\n");
		else /* Make this a port */
		    newlab->lab_flags = pinNum | pinUse | pinDir | PORT_DIR_MASK;
	    }
	    DBAdjustLabels(lefMacro, &rectList->area);
	}

	freeMagic((char *)rectList);
	rectList = rectList->rect_next;
    }
}

/*
 *------------------------------------------------------------
 * LefReadPin --
 *
 *	Read a PIN statement from a LEF file.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Reads input from file f;
 *	Paints into the CellDef lefMacro.
 *
 *------------------------------------------------------------
 */

enum lef_pin_keys {LEF_DIRECTION = 0, LEF_USE, LEF_PORT, LEF_PIN_END};

void
LefReadPin(lefMacro, f, pinname, pinNum)
   CellDef *lefMacro;
   FILE *f;
   char *pinname;
   int pinNum;
{
    char token[256], *tptr;
    char *lineptr;
    int keyword, subkey;
    int pinDir = PORT_CLASS_DEFAULT;
    int pinUse = PORT_USE_DEFAULT;

    static char *pin_keys[] = {
	"DIRECTION",
	"USE",
	"PORT",
	"END",
	NULL
    };

    static char *pin_classes[] = {
	"DEFAULT",
	"INPUT",
	"OUTPUT TRISTATE",
	"OUTPUT",
	"INOUT",
	"FEEDTHRU",
	NULL
    };

    static int lef_class_to_bitmask[] = {
	PORT_CLASS_DEFAULT,
	PORT_CLASS_INPUT,
	PORT_CLASS_TRISTATE,
	PORT_CLASS_OUTPUT,
	PORT_CLASS_BIDIRECTIONAL,
	PORT_CLASS_FEEDTHROUGH
    };

    static char *pin_uses[] = {
	"DEFAULT",
	"SIGNAL",
	"ANALOG",
	"POWER",
	"GROUND",
	"CLOCK",
	NULL
    };

    static int lef_use_to_bitmask[] = {
	PORT_USE_DEFAULT,
	PORT_USE_SIGNAL,
	PORT_USE_ANALOG,
	PORT_USE_POWER,
	PORT_USE_GROUND,
	PORT_USE_CLOCK
    };

    while ((lineptr = LefNextLine(f)) != NULL)
    {
	if (sscanf(lineptr, "%s", token) == 0)
	    continue;	/* Blank line or null statement */

	keyword = Lookup(token, pin_keys);
	if (keyword < 0)
	{
	    LefError("Unknown keyword \"%s\" in LEF file; ignoring.\n", token);
	    continue;
	}
	switch (keyword)
	{
	    case LEF_DIRECTION:
		tptr = LefSkipToken(lineptr);
		sscanf(lineptr, "%*s %255s", token);
		subkey = Lookup(tptr, pin_classes);
		if (subkey < 0)
		    LefError("Improper DIRECTION statement\n");
		else
		    pinDir = lef_class_to_bitmask[subkey];
		break;
	    case LEF_USE:
		tptr = LefSkipToken(lineptr);
		subkey = Lookup(tptr, pin_uses);
		if (subkey < 0)
		    LefError("Improper USE statement\n");
		else
		    pinUse = lef_use_to_bitmask[subkey];
		break;
	    case LEF_PORT:
		LefReadPort(lefMacro, f, pinname, pinNum, pinDir, pinUse);
		break;
	    case LEF_PIN_END:
		if (!LefParseEndStatement(lineptr, pinname))
		{
		    LefError("Pin END statement missing.\n");
		    keyword = -1;
		}
		break;
	}
	if (keyword == LEF_PIN_END) break;
    }

    /* Add use and direction information to the pin. */
}

/*
 *------------------------------------------------------------
 *
 * LefReadMacro --
 *
 *	Read in a MACRO section from a LEF file.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Creates a new cell definition in the magic database.
 *
 *------------------------------------------------------------
 */

enum lef_macro_keys {LEF_CLASS = 0, LEF_SIZE, LEF_ORIGIN,
	LEF_SYMMETRY, LEF_SOURCE, LEF_SITE, LEF_PIN, LEF_OBS,
	LEF_TIMING, LEF_MACRO_END};

void
LefReadMacro(f, mname)
    FILE *f;			/* LEF file being read	*/
    char *mname;		/* name of the macro 	*/
{
    CellDef *lefMacro;
    HashEntry *he;

    char token[256], *tptr;
    char *lineptr;
    int keyword, pinNum;
    float x, y, oscale;
    bool has_size;
    Rect lefBBox;

    static char *macro_keys[] = {
	"CLASS",
	"SIZE",
	"ORIGIN",
	"SYMMETRY",
	"SOURCE",
	"SITE",
	"PIN",
	"OBS",
	"TIMING",
	"END",
	NULL
    };

    /* Start by creating a new celldef */

    he = HashFind(&lefDefInitHash, mname);
    if (HashGetValue(he))
    {
	int suffix;
	char newname[256];

	for (suffix = 1; HashGetValue(he) != NULL; suffix++)
	{
	    sprintf(newname, "%250s_%d", mname, suffix);
	    he = HashFind(&lefDefInitHash, newname);
	}
	LefError("Cell \"%s\" was already defined in this file.  "
		"Renaming this cell \"%s\"\n", mname, newname);
	lefMacro = lefFindCell(newname);
    }
    else
	lefMacro = lefFindCell(mname);

    DBCellClearDef(lefMacro);
    DBCellSetAvail(lefMacro);
    HashSetValue(he, lefMacro);

    /* Initial values */
    pinNum = 1;
    has_size = FALSE;
    oscale = CIFGetOutputScale(1000);
    lefBBox.r_xbot = 0;
    lefBBox.r_ybot = 0;

    while ((lineptr = LefNextLine(f)) != NULL)
    {
	if (sscanf(lineptr, "%s", token) == 0)
	    continue;	/* Blank line or null statement */

	keyword = Lookup(token, macro_keys);
	if (keyword < 0)
	{
	    LefError("Unknown keyword \"%s\" in LEF file; ignoring.\n", token);
	    continue;
	}
	switch (keyword)
	{
	    case LEF_CLASS:
		tptr = LefSkipToken(lineptr);
		if (*tptr != '\0')
		    DBPropPut(lefMacro, "LEFclass", tptr);
		break;
	    case LEF_SIZE:
		tptr = LefSkipToken(lineptr);
		if (sscanf(tptr, "%f %*s %f", &x, &y) != 2)
		    LefError("Bad macro SIZE; requires values X BY Y.\n");
		else
		{
		    lefBBox.r_xtop = (int)(x / oscale) + lefBBox.r_xbot;
		    lefBBox.r_ytop = (int)(y / oscale) + lefBBox.r_ybot;
		    has_size = TRUE;
		}
		break;
	    case LEF_ORIGIN:
		tptr = LefSkipToken(lineptr);
		if (sscanf(tptr, "%f %f", &x, &y) != 2)
		    LefError("Bad macro ORIGIN; requires 2 values.\n");
		else
		{
		    lefBBox.r_xbot = (int)(x / oscale);
		    lefBBox.r_ybot = (int)(y / oscale);
		    if (has_size)
		    {
			lefBBox.r_xtop += lefBBox.r_xbot;
			lefBBox.r_ytop += lefBBox.r_ybot;
		    }
		}
		break;
	    case LEF_SYMMETRY:
		tptr = LefSkipToken(lineptr);
		if (*tptr != '\0')
		    DBPropPut(lefMacro, "LEFsymmetry", tptr);
		break;
	    case LEF_SOURCE:
		tptr = LefSkipToken(lineptr);
		if (*tptr != '\0')
		    DBPropPut(lefMacro, "LEFsource", tptr);
		break;
	    case LEF_SITE:
		tptr = LefSkipToken(lineptr);
		if (*tptr != '\0')
		    DBPropPut(lefMacro, "LEFsite", tptr);
		break;
		break;
	    case LEF_PIN:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("   Macro defines pin %s\n", token);
		LefReadPin(lefMacro, f, token, pinNum++);
		break;
	    case LEF_OBS:
		TxPrintf("   Macro defines obstruction\n");
		LefReadGeometry(lefMacro, f, NULL);
		LefSkipSection(f, NULL);
		break;
	    case LEF_TIMING:
		LefSkipSection(f, macro_keys[LEF_TIMING]);
		break;
	    case LEF_MACRO_END:
		if (!LefParseEndStatement(lineptr, mname))
		{
		    LefError("Macro END statement missing.\n");
		    keyword = -1;
		}
		break;
	}
	if (keyword == LEF_MACRO_END) break;
    }

    /* Finish up creating the cell */

    DBAdjustLabelsNew(lefMacro, &TiPlaneRect, 1);

    if (!has_size)
    {
	LefError("   Macro does not define size:  computing from geometry\n");
	DBReComputeBbox(lefMacro);
    }
    else
	lefMacro->cd_bbox = lefBBox;

    /* Fix the bounding box and do not allow edits */
    lefMacro->cd_flags |= CDNOEDIT | CDFIXEDBBOX;

    DRCCheckThis(lefMacro, TT_CHECKPAINT, &lefMacro->cd_bbox);
    DBWAreaChanged(lefMacro, &lefMacro->cd_bbox, DBW_ALLWINDOWS,
		&DBAllButSpaceBits);
    /* DBCellSetModified(lefMacro, TRUE); */
}

/*
 *------------------------------------------------------------
 *
 * LefRead --
 *
 *	Read a .lef file into a magic layout.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Many.  Cell definitions and uses are created and added to
 *	the database.
 *
 *------------------------------------------------------------
 */

enum lef_sections {LEF_VERSION = 0, LEF_NAMESCASESENSITIVE,
	LEF_PROPERTYDEFS, LEF_UNITS, LEF_SECTION_LAYER,
	LEF_SECTION_VIA, LEF_SECTION_VIARULE,
	LEF_SECTION_SPACING, LEF_SECTION_SITE, LEF_PROPERTY,
	LEF_NOISETABLE, LEF_CORRECTIONTABLE, LEF_IRDROP,
	LEF_MACRO, LEF_END};
enum lef_units_keys {LEF_DATABASE = 0};
enum lef_library_keys {LEF_LIBRARY = 0};

void
LefRead(inName)
    char *inName;
{
    FILE *f;
    char *filename;
    char token[256];
    char *lineptr;
    int keyword;

    static char *sections[] = {
	"VERSION",
	"NAMESCASESENSITIVE",
	"PROPERTYDEFINITIONS",
	"UNITS",
	"LAYER",
	"VIA",
	"VIARULE",
	"SPACING",
	"SITE",
	"PROPERTY",
	"NOISETABLE",
	"CORRECTIONTABLE",
	"IRDROP",
	"MACRO",
	"END",
	NULL
    };
    static char *units_keys[] = {
	"DATABASE",
	NULL
    };

    f = lefFileOpen(NULL, inName, "r", &filename);

    if (f == NULL)
    {
#ifdef MAGIC_WRAPPER
	TxError("Cannot open input file %s (%s).\n", filename,
		strerror(errno));
#else
	TxError("Cannot open input file: ");
	perror(filename);
#endif
	return;
    }

    TxPrintf("Reading LEF data from file %s.\n", filename);
    TxPrintf("This action is undoable.\n");
    UndoDisable();

    /* Initialize */
    HashInit(&LefCellTable, 32, HT_STRINGKEYS);
    HashInit(&lefDefInitHash, 32, HT_STRINGKEYS);

    while ((lineptr = LefNextLine(f)) != NULL)
    {
	if (sscanf(lineptr, "%s", token) == 0)
	    continue;	/* Blank line or null statement */

	keyword = Lookup(token, sections);
	if (keyword < 0)
	{
	    LefError("Unknown keyword \"%s\" in LEF file; ignoring.\n", token);
	    continue;
	}
	switch (keyword)
	{
	    case LEF_VERSION:
		break;
	    case LEF_NAMESCASESENSITIVE:
		break;
	    case LEF_PROPERTYDEFS:
		LefSkipSection(f, sections[LEF_PROPERTYDEFS]);
		break;
	    case LEF_UNITS:
		LefSkipSection(f, sections[LEF_UNITS]);
		break;
	    case LEF_SECTION_LAYER:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("LEF file:  Defines layer %s\n", token);
		LefSkipSection(f, token);
		break;
	    case LEF_SECTION_VIA:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("LEF file:  Defines via %s\n", token);
		LefSkipSection(f, token);
		break;
	    case LEF_SECTION_VIARULE:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("LEF file:  Defines via rule %s\n", token);
		LefSkipSection(f, token);
		break;
	    case LEF_SECTION_SPACING:
		LefSkipSection(f, sections[LEF_SECTION_SPACING]);
		break;
	    case LEF_SECTION_SITE:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("LEF file:  Defines site %s\n", token);
		LefSkipSection(f, token);
		break;
	    case LEF_PROPERTY:
		LefSkipSection(f, NULL);
		break;
	    case LEF_NOISETABLE:
		LefSkipSection(f, sections[LEF_NOISETABLE]);
		break;
	    case LEF_CORRECTIONTABLE:
		LefSkipSection(f, sections[LEF_CORRECTIONTABLE]);
		break;
	    case LEF_IRDROP:
		LefSkipSection(f, sections[LEF_IRDROP]);
		break;
	    case LEF_MACRO:
		sscanf(lineptr, "%*s %255s", token);
		TxPrintf("LEF file:  Defines new cell %s\n", token);
		LefReadMacro(f, token);
		break;
	    case LEF_END:
		if (!LefParseEndStatement(lineptr, "LIBRARY"))
		{
		    LefError("END statement out of context.\n");
		    keyword = -1;
		}
		break;
	}
	if (keyword == LEF_END) break;
    }
    TxPrintf("LEF read:  Done reading file\n");
    LefError(NULL);	/* print statement of errors, if any */

    /* Cleanup */
    HashKill(&LefCellTable);
    HashKill(&lefDefInitHash);
    UndoDisable();
    if (f != NULL) fclose(f);
}
