#-------------------------------------------------------
# Useful tools for the Tcl-based version of magic
#-------------------------------------------------------
# This file is included by wrapper.tcl if it is found
# in the magic install directory.
#-------------------------------------------------------

# Suspend and resume drawing in windows
# Modified 8/17/04 so that calls to suspendall and resumeall
# may nest.

proc magic::suspendall {} {
   global Winopts
   foreach window [magic::windownames] {
      set framename [winfo parent $window]
      if {$framename == "."} {
	 set framename $window
      }
      if {[catch {incr Winopts(${framename},suspend)}]} {
	 set Winopts(${framename},suspend) 1
	 $window update suspend
      }
   }
}

proc magic::resumeall {} {
   global Winopts
   foreach window [magic::windownames] {
      set framename [winfo parent $window]
      if {$framename == "."} {
	 set framename $window
      }
      if {[catch {incr Winopts($framename,suspend) -1}]} {
	 error "resume called without suspend"
      } else {
	 if { $Winopts(${framename},suspend) <= 0 } {
	    unset Winopts(${framename},suspend)
	    $window update resume
	 }
      }
   }
}

# Push and Pop---Treat the edit hierarchy like a stack.

proc magic::pushstack {{name ""}} {
   global editstack  
   if {$name == ""} {
      # no cell selected, so see if we can select one
      set selected [what -list]
      if {[llength [lindex $selected 2]] == 0} {
	 pushbox
	 select cell
	 popbox
      }
      set name [cellname list self]
      if {$name == ""} {
	 error "No cell to push!"
      } elseif {[llength $name] > 1} {
         error "Too many cells selected!"
      }
   }
   if {[catch {lindex $editstack end}]} {
      set editstack {}
   }
   lappend editstack [cellname list window]
   load $name
   return
}

proc magic::popstack {} {
   global editstack
   load [lindex $editstack end]             
   set editstack [lrange $editstack 0 end-1]
   return
}

# More stacking stuff---stacked box values

#---------------------------------------------------------------------
# pushbox --
#       Remember the current box values
#
#---------------------------------------------------------------------

proc magic::pushbox {{values {}}} {
   global boxstack
   if {[catch {set boxstack}]} {
      set boxstack {}
   }
   if {$values == {}} {
      lappend boxstack [box values]
   } else {
      lappend boxstack $values
   }
   return
}

#---------------------------------------------------------------------
# popbox --
#       Recall the last pushed box position
#
# Option "type" may be empty, or "size" or "position" to pop a specific
# box size or position without affecting the other box parameters.
#---------------------------------------------------------------------

proc magic::popbox {{type values}} {
   global boxstack
   if {[catch {set boxstack}]} {
      error "No stack"
   } elseif {$boxstack == {}} {
      error "Empty stack"
   }
   set b [lindex $boxstack end]
   switch -exact $type {
      values {
        box values [lindex $b 0] [lindex $b 1] [lindex $b 2] [lindex $b 3]
      }
      size {
        box size [expr {[lindex $b 2] - [lindex $b 0]}] \
                  [expr {[lindex $b 3] - [lindex $b 1]}]
      }
      position {
        box position [lindex $b 0] [lindex $b 1]
      }
   }
   set boxstack [lrange $boxstack 0 end-1]
   return $b
}

#---------------------------------------------------------------------
# peekbox --
#       Shell procedure that calls popbox but follows by pushing the
#       popped value back onto the stack, resulting in a "peek" mode.
#
# Options are the same as for "popbox" (see above).
#---------------------------------------------------------------------

proc magic::peekbox {{type values}} {
   global bidx
   if {![catch {set b [magic::popbox $type]}]} {
      magic::pushbox $b
   } else {
      error "No stack"
   }
   return $b
}

#---------------------------------------------------------------------
# Because this file is read prior to setting the magic command
# names in Tcl, we cannot run the magic commands here.  Create
# a procedure to enable the commands, then run that procedure
# from the system .magic script.
#---------------------------------------------------------------------

proc magic::enable_tools {} {
   global Opts

   # Set keystrokes for push and pop
   magic::macro XK_greater {magic::pushstack [cellname list self]}
   magic::macro XK_less {magic::popstack}
 
   # Set keystrokes for the "changetool" command.
   magic::macro space		{magic::changetool}
   magic::macro Shift_space	{magic::changetool box}

   set Opts(tool) box
}

#---------------------------------------------------------------------
# changetool --- A scripted replacement for the "tool"
# command, as handling of button events has been modified
# to act like the handling of key events, so the "tool"
# command just swaps macros for the buttons.
#
# Added By NP 10/27/2004
#---------------------------------------------------------------------

proc magic::changetool {{type next}} {
   global Opts
   #puts stdout {TOOLS.TCL : changing tool....}
   # Not wire! "wiring"	
   if {$type == "next"} {
      puts stdout "value of type is $type"
      puts stdout "value of Opts(tool) is $Opts(tool)"
      switch $Opts(tool) {
	 box { set type wiring }
	 wiring { set type netlist }
	 netlist { set type box }
      }
      puts stdout "After switch:"
      puts stdout "value of type is $type"
      puts stdout "value of Opts(tool) is $Opts(tool)"

   }
   switch $type {
      box {
	 puts stdout {Swtching to BOX tool.}
	 set Opts(tool) box
	 tool box	;# sets the cursor
	 macro  Button1          "box move bl cursor"
	 macro  Shift_Button1    "box corner bl cursor"
	 macro  Button2          "paint cursor"
	 macro  Shift_Button2    "erase cursor"
	 macro  Button3          "box corner ur cursor"
	 macro  Shift_Button3    "box move ur cursor"
      }
      wiring {
	 puts stdout {Swtching to WIRING tool.}
	 set Opts(tool) wiring
	 tool wiring	;# sets the cursor
	 macro  Button1          "wire type"
	 macro  Button2          "wire switch"
	 macro  Button3          "wire leg"
	 # Remove button binding
         macro  Shift_Button1    ""
	 macro  Shift_Button2    ""
	 macro  Shift_Button3    ""
	
      }
      netlist {
	 puts stdout {Swtching to NETLIST tool.}
	 set Opts(tool) netlist
	 tool netlist	;# sets the cursor
         # Remove button binding (to be corrected. . .)
         macro  Button1          ""
	 macro  Button2          ""
	 macro  Button3          ""
         macro  Shift_Button1    ""
	 macro  Shift_Button2    ""
	 macro  Shift_Button3    ""
      }
   }
}
