/*
 * lefCmd.c --
 *
 * Commands for the LEF module only.
 *
 */

#ifndef lint
static char rcsid[] = "$Header: /ufs/repository/magic/lef/lefCmd.c,v 1.0 2003/05/01 10:54:00 tim Exp $";
#endif  /* not lint */

#include <stdio.h>
#include <stdlib.h>
#ifdef SYSV
#include <string.h>
#endif

#include "magic/tclmagic.h"
#include "misc/magic.h"
#include "utils/geometry.h"
#include "tiles/tile.h"
#include "utils/hash.h"
#include "database/database.h"
#include "windows/windows.h"
#include "dbwind/dbwind.h"
#include "main/main.h"
#include "textio/txcommands.h"
#include "commands/commands.h"


/*
 * ----------------------------------------------------------------------------
 *
 * CmdLef --
 *
 * Implement the "lef" and "def" commands:  generate LEF-format output
 * for a layout, or read LEF- or DEF-format files.
 *
 * Usage:
 *	lef [options]
 *	def [options]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Generates LEF-format output on disk somewhere, or reads
 *	LEF- or DEF-format files, generating cell definitions and
 *	uses in the process.
 *
 * ----------------------------------------------------------------------------
 */

/* These definitions must correspond to the ordering in cmdLefOption[] below */

#define LEF_READ		0
#define LEF_WRITE		1
#define LEF_WRITEALL		2
#define LEF_HELP		3

Void
CmdLef(w, cmd)
    MagWindow *w;
    TxCommand *cmd;
{
    int option;
    char **msg, *namep;
    CellUse *selectedUse;
    CellDef *selectedDef;
    bool is_lef;

    static char *cmdLefOption[] =
    {	
	"read [filename]	read a file filename[.lef|.def]",
	"write [cell]		write current or indicated cell",
	"writeall		write all cells from the top-level cell",
	"help                   print this help information",
	NULL
    };

    /* Determine whether this function has been called via the "lef" or
     * the "def" command.
     */
    is_lef = (cmd->tx_argv[0][0] == 'd') ? FALSE : TRUE;

    if (cmd->tx_argc < 2)
	option = LEF_HELP;
    else
    {
	option = Lookup(cmd->tx_argv[1], cmdLefOption);
	if (option < 0)
	{
	    TxError("\"%s\" isn't a valid %s option.\n", cmd->tx_argv[1],
			cmd->tx_argv[0]);
	    option = LEF_HELP;
	}
    }

    if (option != LEF_HELP)
    {
        windCheckOnlyWindow(&w, DBWclientID);
        if (w == (MagWindow *) NULL)
        {
            if (ToolGetBox(&selectedDef,NULL) == FALSE)
            {
                TxError("Point to a window first\n");
                return;
            }
            selectedUse = selectedDef->cd_parents;
        }
        else
        {
            selectedUse = (CellUse *)w->w_surfaceID;
        }
    }

    switch (option)
    {
	case LEF_READ:
            if (cmd->tx_argc != 3) goto wrongNumArgs;
            namep = cmd->tx_argv[2];
	    if (is_lef)
		LefRead(namep);
	    else
		DefRead(namep);
	    break;
	case LEF_WRITEALL:
	    if (!is_lef)
		TxError("Sorry, can't write DEF files at this time.\n");
	    else
		LefWriteAll(selectedUse);
	    break;
	case LEF_WRITE:
	    if (!is_lef)
	    {
		TxError("Sorry, can't write DEF files at this time.\n");
		return;
	    }
            if (cmd->tx_argc != 3) goto wrongNumArgs;
            namep = cmd->tx_argv[2];
            selectedUse = CmdGetSelectedCell((Transform *) NULL);
            if (selectedUse == NULL)
            {
                TxError("No cell selected\n");
                return;
            }
            LefWriteCell(selectedUse->cu_def, namep, selectedUse->cu_def
		== EditRootDef);
	    break;
	case LEF_HELP:
wrongNumArgs:
	    TxPrintf("The \"%s\" options are:\n", cmd->tx_argv[0]);
	    for (msg = &(cmdLefOption[0]); *msg != NULL; msg++)
	    {
		TxPrintf("    %s %s\n", cmd->tx_argv[0], *msg);
	    }
	    break;
    }
}
