# This is the "Magic wrapper".
# It's main purpose is to redefine the "openwindow" command in magic so that
# opening a new window creates a window wrapped by a GUI interface.
#
# Written by Tim Edwards, August 23, 2002.

# revision A: proof-of-concept.  Add whatever you want to this basic wrapper.
# revision B: Adds Tk scrollbars and caption
# revision C: Adds a layer manager toolbar on the left side
# revision D: Adds a menubar on top with cell and tech manager tools

global MagicBLT
global windowsopen
global Glyph
global Opts
global Winopts

set MagicBLT 0
if {! [catch {package require BLT}]} {set MagicBLT 1}

# Simple console commands (like TkCon, but much simpler)

if {[lsearch [namespace children] ::tkshell] < 0} {
   catch {source ${CAD_HOME}/lib/magic/tcl/tkshell.tcl}
}

# Button images

set Glyph(up) [image create bitmap \
	-file ${CAD_HOME}/lib/magic/tcl/bitmaps/up.xbm \
	-background gray -foreground steelblue]
set Glyph(down) [image create bitmap \
	-file ${CAD_HOME}/lib/magic/tcl/bitmaps/down.xbm \
	-background gray -foreground steelblue]
set Glyph(left) [image create bitmap \
	-file ${CAD_HOME}/lib/magic/tcl/bitmaps/left.xbm \
	-background gray -foreground steelblue]
set Glyph(right) [image create bitmap \
	-file ${CAD_HOME}/lib/magic/tcl/bitmaps/right.xbm \
	-background gray -foreground steelblue]
set Glyph(zoom) [image create bitmap \
	-file ${CAD_HOME}/lib/magic/tcl/bitmaps/zoom.xbm \
	-background gray -foreground steelblue]

# Menu button callback functions

proc magic::promptload {type} {
   switch $type {
      cif { set Layoutfilename [ tk_getOpenFile -filetypes \
		{{CIF {.cif {.cif}}} {"All files" {*}}}]
	    if {$Layoutfilename != ""} {
		magic::load [file root $Layoutfilename]
		magic::cif read [file root $Layoutfilename]
	    }
	  }
      gds { set Layoutfilename [ tk_getOpenFile -filetypes \
		{{GDS {.gds .strm .cal {.gds .strm .cal}}} {"All files" {*}}}]
	    if {$Layoutfilename != ""} {
		magic::load [file root $Layoutfilename]
		magic::gds read [file root $Layoutfilename]
	    }
	  }
      magic { set Layoutfilename [ tk_getOpenFile -filetypes \
		{{Magic {.mag {.mag}}} {"All files" {*}}}]
	    if {$Layoutfilename != ""} {
		magic::load [file root $Layoutfilename]

		# Append path to cell search path if it's not there already

		if {[string index $Layoutfilename 0] != "/"} {
		   set $Layoutfilename "./$Layoutfilename"
		}
		set sidx [string last "/" $Layoutfilename]
		if {$sidx > 0} {
		   set cellpath [string range $Layoutfilename 0 $sidx]
		   magic::path cell +$cellpath
		}
	    }
	  }
   }
}

proc magic::promptsave {type} {
   switch $type {
      cif { set Layoutfilename [ tk_getSaveFile -filetypes \
		{{CIF {.cif {.cif}}} {"All files" {*}}}]
	    if {$Layoutfilename != ""} {
	       magic::cif write $Layoutfilename
	    }
	  }
      gds { set Layoutfilename [ tk_getSaveFile -filetypes \
		{{GDS {.gds .strm .cal {.gds .strm .cal}}} {"All files" {*}}}]
	    if {$Layoutfilename != ""} {
		magic::gds write $Layoutfilename
	    }
	  }
      magic {
	    set CellList [ magic::cellname list window ]
	    if {[lsearch $CellList "(UNNAMED)"] >= 0} {
	       set Layoutfilename [ tk_getSaveFile -filetypes \
		   {{Magic {.mag {.mag}}} {"All files" {*}}}]
	       if {$Layoutfilename != ""} {
		   set cellpath [file dirname $Layoutfilename]
		   if {$cellpath == [pwd]} {
		      set Layoutfilename [file tail $Layoutfilename]
		   } else {
		      magic::path cell +$cellpath
		   }
		   magic::save $Layoutfilename
	       }
	    }
	    magic::writeall
	  }
   }
}

# Callback to the cell manager

proc magic::instcallback {command} {
   global Opts

   set rpath [join [.cellmgr.box.view get -full anchor]]
   set rootdef [lindex $rpath 0]
   set cellpath [lrange $rpath 1 end]

   if { $Opts(target) == "default" } {
      set winlist [magic::windownames layout]
      set winname [lindex $winlist 0]
   } else {
      set winname $Opts(target)
   }

   if { $cellpath == {} } {
      switch $command {
	 load {$winname load $rootdef}
	 default {
	    magic::select top cell
	    switch $command {
	       edit {$winname expand; $winname edit}
	       expand {$winname expand}
	       zoom {$winname view}
	    }
	 }
      }
   } else {
      set celluse [join $cellpath "/"]
      set celldef [magic::instance list celldef $celluse]

      set curpath [$winname windowcaption]
      set curname [lindex $curpath 2]
      set curroot [lindex $curpath 0]

      switch $command {
         load {$winname load $celldef}
         default {
	    # Here: need to check first for the select cell belonging to the
	    # current loaded root cell (get the first use).
	    set defpath [list $rootdef]
	    foreach i $cellpath {
	       lappend defpath [magic::instance list celldef $i]
	    }
	    set rootpos [lsearch $defpath $curroot]
	    if {$rootpos < 0} {
	       $winname load $rootdef
	       set rootpos 0
	    }
	    set usepath [join [lrange $cellpath $rootpos end] "/"]
	    $winname select cell ${usepath}

	    switch $command {
	       edit {$winname expand; $winname edit}
	       expand {$winname expand toggle}
	       zoom {$winname findbox zoom}
	    }
	 }
      }
   }
}

# The cell manager

proc magic::makecellmanager { mgrpath } {
   global MagicBLT

   if {$MagicBLT} {
      toplevel ${mgrpath}
      wm withdraw ${mgrpath}
      frame ${mgrpath}.actionbar
      frame ${mgrpath}.box
      frame ${mgrpath}.target
      blt::hiertable ${mgrpath}.box.view -bg white -allowduplicates false -hideroot true \
	   -yscrollcommand "${mgrpath}.box.vert set" \
	   -xscrollcommand "${mgrpath}.box.vert set"
      scrollbar ${mgrpath}.box.vert -orient vertical -command "${mgrpath}.box.view yview"

      pack ${mgrpath}.actionbar -side top -fill x
      pack ${mgrpath}.box.view -side left -fill both -expand true
      pack ${mgrpath}.box.vert -side right -fill y
      pack ${mgrpath}.box -side top -fill both -expand true
      pack ${mgrpath}.target -side top -fill x

      button ${mgrpath}.actionbar.done -text "Zoom" -command {magic::instcallback zoom}
      button ${mgrpath}.actionbar.edit -text "Edit" -command {magic::instcallback edit}
      button ${mgrpath}.actionbar.load -text "Load" -command {magic::instcallback load}
      button ${mgrpath}.actionbar.expand -text "Expand" -command \
		{magic::instcallback expand}

      pack ${mgrpath}.actionbar.load -side left
      pack ${mgrpath}.actionbar.edit -side left
      pack ${mgrpath}.actionbar.expand -side left
      pack ${mgrpath}.actionbar.done -side right

      label ${mgrpath}.target.name -text "Target window:"
      menubutton ${mgrpath}.target.list -text "default" \
		-menu ${mgrpath}.target.list.winmenu
 
      pack ${mgrpath}.target.name -side left -padx 2
      pack ${mgrpath}.target.list -side left

      menu ${mgrpath}.target.list.winmenu -tearoff 0	  ;# No entries by default
   }
}

proc magic::addlistinstance {instin} {
   set hierinst [join [lrange $instin 1 end] "/"]
   set defname [magic::instance list celldef $hierinst]
   set instnum [lindex [split [lindex $instin end] _ ] end]
   .cellmgr.box.view insert end $instin -label "${defname} (${instnum})"

   set sublist [magic::instance list children $hierinst]
   foreach i $sublist {
      set instout [concat $instin $i]
      magic::addlistinstance $instout
   }
}

proc magic::cellmanager {{option "update"}} {
   global MagicBLT
   if {$MagicBLT} {

      # Check for existence of the manager widget
      if {[catch {wm state .cellmgr}]} {
	 magic::makecellmanager .cellmgr
	 magic::cellmanager update
      }

      if {$option == "update"} {

         # determine the full cell heirarchy
         set tl [magic::cellname list topcells]
         # remove the previous entry and create a new one
         .cellmgr.box.view delete all
         foreach i $tl {
	    if { [file extension $i] == ".mag" } {
	       set nameroot [file rootname $i]
	    } else {
	       set nameroot $i
	    }
	    set nameroot [file tail $nameroot]
	    .cellmgr.box.view insert end $i -foreground red -label $nameroot

	    set cd [magic::cellname list childinst $i]
	    foreach j $cd {
	       set ci [concat $i $j]
	       magic::addlistinstance $ci
	    }
	 }
      }
   }
}

# Technology manager callback functions

proc magic::techparseunits {} {
   set techlambda [magic::tech lambda]
   set tech1 [lindex $techlambda 1]
   set tech0 [lindex $techlambda 0]

   set target0 [.techmgr.lambda1.lval0 get]
   set target1 [.techmgr.lambda1.lval1 get]

   set newval0 [expr {$target0 * $tech0}]
   set newval1 [expr {$target1 * $tech1}]

   magic::scalegrid $newval1 $newval0
   magic::techmanager update
}

# The technology manager

proc magic::maketechmanager { mgrpath } {
   toplevel $mgrpath
   wm withdraw $mgrpath

   frame ${mgrpath}.title
   label ${mgrpath}.title.tlab -text "Technology: "
   menubutton ${mgrpath}.title.tname -text "(none)" -foreground red3 \
	-menu ${mgrpath}.title.tname.menu
   label ${mgrpath}.title.tvers -text "" -foreground blue
   label ${mgrpath}.subtitle -text "" -foreground sienna4

   frame ${mgrpath}.lambda0
   label ${mgrpath}.lambda0.llab -text "Microns per lambda (CIF): "
   label ${mgrpath}.lambda0.lval -text "1" -foreground blue

   frame ${mgrpath}.lambda1
   label ${mgrpath}.lambda1.llab -text "Internal units per lambda: "
   entry ${mgrpath}.lambda1.lval0 -foreground red3 -background white -width 3
   label ${mgrpath}.lambda1.ldiv -text " / "
   entry ${mgrpath}.lambda1.lval1 -foreground red3 -background white -width 3

   frame ${mgrpath}.cif0
   label ${mgrpath}.cif0.llab -text "CIF input style: "
   menubutton ${mgrpath}.cif0.lstyle -text "" -foreground blue \
	-menu ${mgrpath}.cif0.lstyle.menu
   label ${mgrpath}.cif0.llab2 -text " Microns/lambda="
   label ${mgrpath}.cif0.llambda -text "" -foreground red3

   frame ${mgrpath}.cif1
   label ${mgrpath}.cif1.llab -text "CIF output style: "
   menubutton ${mgrpath}.cif1.lstyle -text "" -foreground blue \
	-menu ${mgrpath}.cif1.lstyle.menu
   label ${mgrpath}.cif1.llab2 -text " Microns/lambda="
   label ${mgrpath}.cif1.llambda -text "" -foreground red3

   frame ${mgrpath}.extract
   label ${mgrpath}.extract.llab -text "Extract style: "
   menubutton ${mgrpath}.extract.lstyle -text "" -foreground blue \
	-menu ${mgrpath}.extract.lstyle.menu

   pack ${mgrpath}.title.tlab -side left
   pack ${mgrpath}.title.tname -side left
   pack ${mgrpath}.title.tvers -side left
   pack ${mgrpath}.lambda0.llab -side left
   pack ${mgrpath}.lambda0.lval -side left
   pack ${mgrpath}.lambda1.llab -side left
   pack ${mgrpath}.lambda1.lval0 -side left
   pack ${mgrpath}.lambda1.ldiv -side left
   pack ${mgrpath}.lambda1.lval1 -side left
   pack ${mgrpath}.cif0.llab -side left
   pack ${mgrpath}.cif0.lstyle -side left
   pack ${mgrpath}.cif0.llab2 -side left
   pack ${mgrpath}.cif0.llambda -side left
   pack ${mgrpath}.cif1.llab -side left
   pack ${mgrpath}.cif1.lstyle -side left
   pack ${mgrpath}.cif1.llab2 -side left
   pack ${mgrpath}.cif1.llambda -side left
   pack ${mgrpath}.extract.llab -side left
   pack ${mgrpath}.extract.lstyle -side left

   pack ${mgrpath}.title -side top -fill x
   pack ${mgrpath}.subtitle -side top -fill x
   pack ${mgrpath}.lambda0 -side top -fill x
   pack ${mgrpath}.lambda1 -side top -fill x
   pack ${mgrpath}.cif0 -side top -fill x
   pack ${mgrpath}.cif1 -side top -fill x
   pack ${mgrpath}.extract -side top -fill x

   bind ${mgrpath}.lambda1.lval0 <Return> magic::techparseunits
   bind ${mgrpath}.lambda1.lval1 <Return> magic::techparseunits
}

# Create or redisplay the technology manager

proc magic::techmanager {{option "update"}} {
   global tcl_precision
   global CAD_HOME

   if {[catch {wm state .techmgr}]} {
      magic::maketechmanager .techmgr
      magic::techmanager initall
   }

   if {$option == "initall"} {
      set m [menu .techmgr.title.tname.menu -tearoff 0]
      set dirlist [subst [magic::path sys]]
      set tlist {}
      foreach i $dirlist {
	 lappend tlist [glob -nocomplain ${i}/*.tech]
	 lappend tlist [glob -nocomplain ${i}/*.tech27]
      }
      foreach i [join $tlist] {
	 set j [file tail [file rootname ${i}]]
	 $m add command -label $j -command \
		"magic::tech load $j ; magic::techmanager update"
      }

      menu .techmgr.cif0.lstyle.menu -tearoff 0
      menu .techmgr.cif1.lstyle.menu -tearoff 0
      menu .techmgr.extract.lstyle.menu -tearoff 0
    
   }

   if {$option == "init"} {
	.techmgr.cif0.lstyle.menu delete 0 end
	.techmgr.cif1.lstyle.menu delete 0 end
	.techmgr.extract.lstyle.menu delete 0 end
   }

   if {$option == "init" || $option == "initall"} {
      set tlist [magic::cif listall istyle]
      foreach i $tlist {
	 .techmgr.cif0.lstyle.menu add command -label $i -command \
		"magic::cif istyle $i ; \
		 magic::techmanager update"
      }

      set tlist [magic::cif listall ostyle]
      foreach i $tlist {
	 .techmgr.cif1.lstyle.menu add command -label $i -command \
		"magic::cif ostyle $i ; \
		 magic::techmanager update"
      }

      set tlist [magic::extract listall style]
      foreach i $tlist {
	 .techmgr.extract.lstyle.menu add command -label $i -command \
		"magic::extract style $i ; \
		 magic::techmanager update"
      }
   }

   if {$option != "check"} {
      set psave $tcl_precision
      set tcl_precision 3

      set techlambda [magic::tech lambda]
      set tech1 [lindex $techlambda 1]
      set tech0 [lindex $techlambda 0]
      set tscale [expr {$tech1 / $tech0}]

      .techmgr.title.tname configure -text [magic::tech name]
      set techstuff [magic::tech version]
      .techmgr.title.tvers configure -text "(version [lindex $techstuff 0])"
      .techmgr.subtitle configure -text [lindex $techstuff 1]
      .techmgr.lambda0.lval configure -text [expr {[magic::cif lambda output] * $tscale}]
      .techmgr.cif0.lstyle configure -text [magic::cif list istyle]
      .techmgr.cif0.llambda configure -text [expr {[magic::cif lambda input] * $tscale}]
      .techmgr.cif1.lstyle configure -text [magic::cif list ostyle]
      .techmgr.cif1.llambda configure -text [expr {[magic::cif lambda output] * $tscale}]
      .techmgr.extract.lstyle configure -text [magic::extract list style]

      .techmgr.lambda1.lval0 delete 0 end
      .techmgr.lambda1.lval1 delete 0 end
      .techmgr.lambda1.lval0 insert end $tech1
      .techmgr.lambda1.lval1 insert end $tech0

      set tcl_precision $psave
   }
}

proc magic::caption {winpath} {
   set framename [winfo parent $winpath]
   set caption [$winpath windowcaption]
   set subcaption1 [lindex $caption 0]
   if {[lindex $caption 1] == "EDITING"} {
      set subcaption2 [lindex $caption 2]
   } else {
      set subcaption2 [join [lrange $caption 1 end]]
   }
   ${framename}.titlebar.caption configure -text \
	"Loaded: ${subcaption1} Editing: ${subcaption2}"
}

# Generate the cell and tech manager windows
magic::makecellmanager .cellmgr
magic::maketechmanager .techmgr

# Allow captioning in the title window by tagging the "load" and "edit" commands
# Note that the "box" tag doesn't apply to mouse-button events, so this function
# is duplicated by Tk binding of mouse events in the layout window.

magic::tag load "magic::cellmanager; magic::caption %W"
magic::tag getcell "magic::cellmanager"
magic::tag edit "magic::caption %W"
magic::tag save "magic::caption %W"
magic::tag box "magic::boxview %W"
magic::tag move "magic::boxview %W"
magic::tag scroll "magic::scrollupdate %W"
magic::tag view "magic::scrollupdate %W"
magic::tag zoom "magic::scrollupdate %W"
magic::tag findbox "magic::scrollupdate %W"
magic::tag see "magic::toolupdate %W %1 %2"
magic::tag tech "magic::techrebuild %W %1"
if {$MagicBLT} {
   magic::tag cellname "magic::mgrupdate %W %1"
   magic::tag select "magic::mgrselect %R"
   magic::tag cif "magic::mgrupdate %W %1"
   magic::tag gds "magic::mgrupdate %W %1"
}

# This should be a list. . . do be done later
set windowsopen 0

set Opts(cellmgr) 0
set Opts(techmgr) 0
set Opts(target)  default

# Procedures for the layout scrollbars, which are made from canvas
# objects to avoid the problems associated with Tk's stupid scrollbar
# implementation.

# Repainting function for scrollbars, title, etc., to match the magic
# Change the colormap (most useful in 8-bit PseudoColor)

proc magic::repaintwrapper { win } {
   set bgcolor [magic::magiccolor -]
   ${win}.xscroll configure -background $bgcolor
   ${win}.xscroll configure -highlightbackground $bgcolor
   ${win}.xscroll configure -highlightcolor [magic::magiccolor K]

   ${win}.yscroll configure -background $bgcolor
   ${win}.yscroll configure -highlightbackground $bgcolor
   ${win}.yscroll configure -highlightcolor [magic::magiccolor K]

   ${win}.titlebar.caption configure -background [magic::magiccolor w]
   ${win}.titlebar.caption configure -foreground [magic::magiccolor c]

   ${win}.titlebar.message configure -background [magic::magiccolor w]
   ${win}.titlebar.message configure -foreground [magic::magiccolor c]

   ${win}.titlebar.pos configure -background [magic::magiccolor w]
   ${win}.titlebar.pos configure -foreground [magic::magiccolor c]

}

# Coordinate display callback function
# Because "box" calls "box", use the "info level" command to avoid
# infinite recursion.

proc magic::boxview {win} {
   if {[info level] <= 1} {

      # For NULL window, find all layout windows and apply update to each.
      if {$win == {}} {
         set winlist [magic::windownames layout]
         foreach lwin $winlist {
	    magic::boxview $lwin
         }
         return
      }

      set framename [winfo parent $win]
      set lambda [${win} tech lambda]
      set lr [expr {(0.0 + [lindex $lambda 0]) / [lindex $lambda 1]}]
      set bval [${win} box values]
      set bllx [expr {[lindex $bval 0] * $lr }]
      set blly [expr {[lindex $bval 1] * $lr }]
      set burx [expr {[lindex $bval 2] * $lr }]
      set bury [expr {[lindex $bval 3] * $lr }]
      if {[expr {$bllx == int($bllx)}]} {set bllx [expr {int($bllx)}]}
      if {[expr {$blly == int($blly)}]} {set blly [expr {int($blly)}]}
      if {[expr {$burx == int($burx)}]} {set burx [expr {int($burx)}]}
      if {[expr {$bury == int($bury)}]} {set bury [expr {int($bury)}]}
      ${framename}.titlebar.pos configure -text \
		"box ($bllx, $blly) to ($burx, $bury) lambda"
   }
}

proc magic::cursorview {win} {
   if {$win == {}} {
      return
   }
   set framename [winfo parent $win]
   set lambda [${win} tech lambda]
   set lr [expr {(0.0 + [lindex $lambda 0]) / [lindex $lambda 1]}]
   set olst [${win} cursor]
   set olstx [expr {[lindex $olst 0] * $lr }]
   set olsty [expr {[lindex $olst 1] * $lr }]
   if {[expr {$olstx == int($olstx)}]} {set olstx [expr {int($olstx)}]}
   if {[expr {$olsty == int($olsty)}]} {set olsty [expr {int($olsty)}]}

   if {[${win} box exists]} {
      set dlst [${win} box position]
      set dx [expr {([lindex $olst 0] - [lindex $dlst 0]) * $lr }]
      set dy [expr {([lindex $olst 1] - [lindex $dlst 1]) * $lr }]
      if {[expr {$dx == int($dx)}]} {set dx [expr {int($dx)}]}
      if {[expr {$dy == int($dy)}]} {set dy [expr {int($dy)}]}
      if {$dx >= 0} {set dx "+$dx"}
      if {$dy >= 0} {set dy "+$dy"}
      ${framename}.titlebar.pos configure -text "($olstx $olsty) $dx $dy lambda"
   } else {
      ${framename}.titlebar.pos configure -text "($olstx $olsty) lambda"
   }
}

proc magic::toolupdate {win {yesno "yes"} {layerlist "none"}} {
   global Winopts

   # For NULL window, get current window
   if {$win == {}} {
      set win [magic::windownames]
   }

   # Wind3d has a "see" function, so make sure this is not a 3d window
   if {$win == [magic::windownames wind3d]} {
      return
   }

   set framename [winfo parent $win]

   # Don't do anything if toolbar is not present
   if { $Winopts(${framename},toolbar) == 0 } { return }

   if {$layerlist == "none"} {
	set layerlist $yesno
	set yesno "yes"
   }
   if {$layerlist == "*"} {
      set layerlist [magic::tech layer "*"]
   }
   foreach layer $layerlist {
      switch $layer {
	 none {}
	 errors {set canon $layer}
	 subcell {set canon $layer}
	 labels {set canon $layer}
	 default {set canon [magic::tech layer $layer]}
      }
      if {$layer != "none"} {
         if {$yesno == "yes"} {
	    ${framename}.toolbar.b$canon configure -image img_$canon 
         } else {
	    ${framename}.toolbar.b$canon configure -image img_space
	 }
      }
   }
}

# Redirect and reformat Tcl output of "select" command

if {$MagicBLT} {
   proc magic::mgrselect {{sstr ""}} {
      if {$sstr == ""} return
      set savetag [magic::tag select]
      magic::tag select ""
      .cellmgr.box.view selection clearall
      if {[llength $sstr] == 5} {
	 # sstr is "Topmost cell in the window"
	 set clist [magic::cellname list window]
	 set isrch [magic::cellname list instances $clist]
	 foreach i $isrch {
	    if {[llength $i] != 5} {
	       set gsrch $i
	       break
	    } else {
	       set gsrch $clist
	    }
         }
         set idx [lindex [.cellmgr.box.view find -glob -full *${gsrch}*] 0]
	 set cdef $clist
      } else {
	 regsub -all {\[.*\]} $sstr {[^a-z]+} newstr
         set gsrch [string map {"/" " "} ".*${newstr}\$"]
	 set cdef [magic::instance list celldef $sstr]
         set idx [.cellmgr.box.view find -regexp -full "$gsrch"]
      }
      if {$idx != "" } {
         .cellmgr.box.view open $idx
         if {[wm state .cellmgr] == "normal"} { .cellmgr.box.view see $idx }
         .cellmgr.box.view selection anchor $idx
         .cellmgr.box.view selection set $idx
         if {$sstr != ""} {
	    puts stdout "Selected cell is $cdef ($sstr)"
	 }
      }
      magic::tag select $savetag
   }

   # Update cell and tech managers in response to a cif or gds read command

   proc magic::mgrupdate {win {cmdstr ""}} {
      if {${cmdstr} == "read"} {
	 magic::cellmanager
	 magic::caption $win
	 magic::techmanager update
      } elseif {${cmdstr} == "delete" || ${cmdstr} == "rename"} {
	 magic::cellmanager
	 magic::caption $win
      }
   }
}

# Generate the toolbar for the wrapper

proc magic::maketoolbar { framename } {

   # Destroy any existing toolbar before starting
   set alltools [winfo children ${framename}.toolbar]
   foreach i $alltools {
      destroy $i
   }

   # All toolbar commands will be passed to the appropriate window
   set win ${framename}.magic

   # Make sure that window has been created so we will get the correct
   # height value.

   update idletasks
   set winheight [expr {[winfo height ${framename}] - \
		[winfo height ${framename}.titlebar]}]

   # Generate a layer image for "space" that will be used when layers are
   # invisible.

   image create layer img_space -name none

   # Generate layer images and buttons for toolbar

   set all_layers [concat {errors labels subcell} [magic::tech layer "*"]]
   foreach layername $all_layers {
      image create layer img_$layername -name $layername
      button ${framename}.toolbar.b$layername -image img_$layername -command \
		"$win see $layername"

      # Bindings:  Entering the button puts the canonical layer name in the 
      # message window.
      bind ${framename}.toolbar.b$layername <Enter> \
		[subst {focus %W ; ${framename}.titlebar.message configure \
		 -text "$layername"}]
      bind ${framename}.toolbar.b$layername <Leave> \
		[subst {${framename}.titlebar.message configure -text ""}]

      # 3rd mouse button makes layer invisible; 1st mouse button restores it.
      # 2nd mouse button paints the layer color.  Key "p" also does paint, esp.
      # for users with 2-button mice.  Key "e" erases, as does Shift-Button-2.

      bind ${framename}.toolbar.b$layername <ButtonPress-2> \
		"$win paint $layername"
      bind ${framename}.toolbar.b$layername <KeyPress-p> \
		"$win paint $layername"
      bind ${framename}.toolbar.b$layername <Shift-ButtonPress-2> \
		"$win erase $layername"
      bind ${framename}.toolbar.b$layername <KeyPress-e> \
		"$win erase $layername"
      bind ${framename}.toolbar.b$layername <ButtonPress-3> \
		"$win see no $layername"
      bind ${framename}.toolbar.b$layername <KeyPress-s> \
		"$win select more area $layername"
      bind ${framename}.toolbar.b$layername <KeyPress-S> \
		"$win select less area $layername"
   }

   # Figure out how many columns we need to fit all the layer buttons inside
   # the toolbar without going outside the window area.

   set ncols 1
   while {1} {
      incr ncols
      set i 0
      set j 0
      foreach layername $all_layers {
         grid ${framename}.toolbar.b$layername -row $i -column $j -sticky news
         incr j
         if {$j == $ncols} {
	    set j 0
	    incr i
         }
      }
      # tkwait visibility ${framename}.toolbar
      update idletasks
      set toolheight [lindex [grid bbox ${framename}.toolbar] 3]
      if {$toolheight <= $winheight} {break}
   }
}

# Delete and rebuild the toolbar buttons in response to a "tech load"
# command.

proc magic::techrebuild {winpath {cmdstr ""}} {

   # For NULL window, find all layout windows and apply update to each.
   if {$winpath == {}} {
      set winlist [magic::windownames layout]
      foreach lwin $winlist {
	 magic::techrebuild $lwin $cmdstr
      }
      return
   }

   set framename [winfo parent $winpath]
   if {${cmdstr} == "load"} {
      maketoolbar ${framename}
      magic::techmanager init
   }
}

# Scrollbar callback procs

# Procedure to return the effective X and Y scrollbar bounds for the
# current view in magic (in pixels)

proc magic::setscrollvalues {win} {
   set svalues [${win} view get]
   set bvalues [${win} view bbox]

   set bwidth [expr {[lindex $bvalues 2] - [lindex $bvalues 0]}]
   set bheight [expr {[lindex $bvalues 3] - [lindex $bvalues 1]}]

   set framename [winfo parent ${win}]
   set wwidth [winfo width ${framename}.xscroll.bar]  ;# horizontal scrollbar
   set wheight [winfo height ${framename}.yscroll.bar]  ;# vertical scrollbar

   # Note that adding 0.0 here forces floating-point

   set xscale [expr {(0.0 + $wwidth) / $bwidth}]
   set yscale [expr {(0.0 + $wheight) / $bheight}]

   set xa [expr {$xscale * ([lindex $svalues 0] - [lindex $bvalues 0]) }]
   set xb [expr {$xscale * ([lindex $svalues 2] - [lindex $bvalues 0]) }]
   set ya [expr {$yscale * ([lindex $svalues 1] - [lindex $bvalues 1]) }]
   set yb [expr {$yscale * ([lindex $svalues 3] - [lindex $bvalues 1]) }]

   # Magic's Y axis is inverted with respect to X11 window coordinates
   set ya [expr { $wheight - $ya }]
   set yb [expr { $wheight - $yb }]

   ${framename}.xscroll.bar coords slider $xa 2 $xb 15
   ${framename}.yscroll.bar coords slider 2 $ya 15 $yb

   set xb [expr { 1 + ($xa + $xb) / 2 }]
   set xa [expr { $xb - 2 }]
   ${framename}.xscroll.bar coords centre $xa 4 $xb 13

   set yb [expr { 1 + ($ya + $yb) / 2 }]
   set ya [expr { $yb - 2 }]
   ${framename}.yscroll.bar coords centre 4 $ya 13 $yb
}

# Procedure to update scrollbars in response to an internal command
# "view" calls "view", so avoid infinite recursion.

proc magic::scrollupdate {win} {

   if {[info level] <= 1} {

      # For NULL window, find current window
      if {$win == {}} {
	 set win [magic::windownames]
      }

      # Make sure we're not a 3D window, which doesn't take window commands
      # This is only necessary because the 3D window declares a "view"
      # command, too.

      if {$win != [magic::windownames wind3d]} {
 	 magic::setscrollvalues $win
      }
   }
}

# scrollview:  update the magic display to match the
# scrollbar positions.

proc magic::scrollview { w win orient } {
   global scale
   set v1 $scale($orient,origin)
   set v2 $scale($orient,update)
   set delta [expr {$v2 - $v1}]

   set bvalues [${win} view bbox]
   set wvalues [${win} windowpositions]

   # Note that adding 0.000 in expression forces floating-point

   if {"$orient" == "x"} {

      set bwidth [expr {[lindex $bvalues 2] - [lindex $bvalues 0]}]
      set wwidth [expr {0.000 + [lindex $wvalues 2] - [lindex $wvalues 0]}]
      set xscale [expr {$bwidth / $wwidth}]
      ${win} scroll e [expr {$delta * $xscale}]

   } else {

      set bheight [expr {[lindex $bvalues 3] - [lindex $bvalues 1]}]
      set wheight [expr {0.000 + [lindex $wvalues 3] - [lindex $wvalues 1]}]
      set yscale [expr {$bheight / $wheight}]
      ${win} scroll s [expr {$delta * $yscale}]
   }
}

# setscroll: get the current cursor position and save it as a
# reference point.

proc magic::setscroll { w v orient } {
   global scale
   set scale($orient,origin) $v
   set scale($orient,update) $v
}

proc magic::dragscroll { w v orient } {
   global scale
   set v1 $scale($orient,update)
   set scale($orient,update) $v
   set delta [expr {$v - $v1}]

   if { "$orient" == "x" } {
      $w move slider $delta 0
      $w move centre $delta 0
   } else {
      $w move slider 0 $delta
      $w move centre 0 $delta
   }
}

# Scrollbar generator for the wrapper window

proc magic::makescrollbar { fname orient win } {
   global scale
   global Glyph

   set scale($orient,update) 0
   set scale($orient,origin) 0

   # To be done:  add glyphs for the arrows

   if { "$orient" == "x" } {
      canvas ${fname}.bar -height 13 -relief sunken -borderwidth 1
      button ${fname}.lb -image $Glyph(left) -borderwidth 1 \
		-command "${win} scroll left .1 w"
      button ${fname}.ub -image $Glyph(right) -borderwidth 1 \
		-command "${win} scroll right .1 w"
      pack ${fname}.lb -side left
      pack ${fname}.bar -fill $orient -expand true -side left
      pack ${fname}.ub -side right
   } else {
      canvas ${fname}.bar -width 13 -relief sunken -borderwidth 1
      button ${fname}.lb -image $Glyph(down) -borderwidth 1 \
		-command "${win} scroll down .1 w"
      button ${fname}.ub -image $Glyph(up) -borderwidth 1 \
		-command "${win} scroll up .1 w"
      pack ${fname}.ub
      pack ${fname}.bar -fill $orient -expand true
      pack ${fname}.lb
   }

   # Create the bar which controls the scrolling and bind actions to it
   ${fname}.bar create rect 2 2 15 15 -fill steelblue -width 0 -tag slider
   ${fname}.bar bind slider <Button-1> "magic::setscroll %W %$orient $orient"
   ${fname}.bar bind slider <ButtonRelease-1> "magic::scrollview %W $win $orient"
   ${fname}.bar bind slider <B1-Motion> "magic::dragscroll %W %$orient $orient"

   # Create a small mark in the center of the scrolling rectangle which aids
   # in determining how much the window is being scrolled when the full
   # scrollbar extends past the window edges.
   ${fname}.bar create rect 4 4 13 13 -fill black -width 0 -tag centre
   ${fname}.bar bind centre <Button-1> "magic::setscroll %W %$orient $orient"
   ${fname}.bar bind centre <ButtonRelease-1> "magic::scrollview %W $win $orient"
   ${fname}.bar bind centre <B1-Motion> "magic::dragscroll %W %$orient $orient"
}

# Create the wrapper and open up a layout window in it.

proc magic::openwrapper { {cell ""} } {
   global windowsopen
   global MagicBLT
   global Glyph
   global Opts
   global Winopts
   
   # Disallow scrollbars and title caption on windows---we'll do these ourselves

   if {$windowsopen == 0} {
      windowcaption off
      windowscrollbars off
      windowborder off
   }

   incr windowsopen 
   set framename .layout${windowsopen}
   set winname ${framename}.magic
   
   toplevel $framename
   tkwait visibility $framename

   frame ${framename}.xscroll -height 13
   frame ${framename}.yscroll -width 13

   magic::makescrollbar ${framename}.xscroll x ${winname}
   magic::makescrollbar ${framename}.yscroll y ${winname}
   button ${framename}.zb -image $Glyph(zoom) -borderwidth 1 -command "${winname} zoom 2"

   # Add bindings for mouse buttons 2 and 3 to the zoom button
   bind ${framename}.zb <Button-3> "${winname} zoom 0.5"
   bind ${framename}.zb <Button-2> "${winname} view"

   frame ${framename}.titlebar
   label ${framename}.titlebar.caption -text "Loaded: none Editing: none" \
	-foreground white -background sienna4 -anchor w -padx 15
   label ${framename}.titlebar.message -text "" -foreground white \
	-background sienna4 -anchor w -padx 5
   label ${framename}.titlebar.pos -text "" -foreground white \
	-background sienna4 -anchor w -padx 5

   # Menu buttons

   menubutton ${framename}.titlebar.filebutton -text File -relief raised \
		-menu ${framename}.titlebar.filebutton.filemenu -borderwidth 2
   menubutton ${framename}.titlebar.optsbutton -text Options -relief raised \
		-menu ${framename}.titlebar.optsbutton.optsmenu -borderwidth 2

   magic::openwindow $cell $winname

   # Create toolbar frame.  Make sure it has the same visual and depth as
   # the layout window, so there will be no problem using the GCs from the
   # layout window to paint into the toolbar.
   frame ${framename}.toolbar \
	-visual "[winfo visual ${winname}] [winfo depth ${winname}]"

   # Repaint to magic colors
   magic::repaintwrapper ${framename}

   grid ${framename}.titlebar -row 0 -column 0 -columnspan 3 -sticky news
   grid ${framename}.yscroll -row 1 -column 0 -sticky ns
   grid $winname -row 1 -column 1 -sticky news
   grid ${framename}.zb -row 2 -column 0
   grid ${framename}.xscroll -row 2 -column 1 -sticky ew
   # The toolbar is not attached by default

   grid rowconfigure ${framename} 1 -weight 1
   grid columnconfigure ${framename} 1 -weight 1

   grid ${framename}.titlebar.filebutton -row 0 -column 0 -sticky news
   grid ${framename}.titlebar.optsbutton -row 0 -column 1 -sticky news
   grid ${framename}.titlebar.caption -row 0 -column 2 -sticky news
   grid ${framename}.titlebar.message -row 0 -column 3 -sticky news
   grid ${framename}.titlebar.pos -row 0 -column 4 -sticky news
   grid columnconfigure ${framename}.titlebar 2 -weight 1

   bind $winname <Enter> {focus %W}

   # Button press events bypass the normal command mechanism, so any tag features
   # required on button presses/releases need to be duplicated.

   bind ${winname} <Motion> "::magic::cursorview ${winname}"

   # Resize the window to be 3/4 of the screen size.
   set winwidth [expr 3 * [winfo screenwidth $framename] / 4]
   set winheight [expr 3 * [winfo screenheight $framename] / 4]
   wm geometry $framename ${winwidth}x${winheight}+100+100

   set Winopts(${framename},toolbar) 0

   # Generate the file menu

   set m [menu ${framename}.titlebar.filebutton.filemenu -tearoff 0]
   $m add command -label "New window" -command "magic::openwrapper \
		[${winname} cellname list window]"
   $m add command -label "Close" -command "magic::closewrapper ${framename}"
   $m add separator
   $m add command -label "Load layout" -command {magic::promptload magic}
   $m add command -label "Read CIF" -command {magic::promptload cif}
   $m add command -label "Read GDS" -command {magic::promptload gds}
   $m add separator
   $m add command -label "Save layout" -command {magic::promptsave magic}
   $m add command -label "Write CIF" -command {magic::promptsave cif}
   $m add command -label "Write GDS" -command {magic::promptsave gds}
   $m add separator
   $m add command -label "Flush current" -command {magic::flush}
   $m add command -label "Delete current" -command {magic::cellname delete \
		[magic::cellname list window]}
   $m add separator
   $m add command -label "Quit" -command {magic::quit}

   # Generate the options menu

   set m [menu ${framename}.titlebar.optsbutton.optsmenu -tearoff 0]
   $m add check -label "Toolbar" -variable Winopts(${framename},toolbar) \
	-command [subst {if { \$Winopts(${framename},toolbar) } { \
		magic::maketoolbar ${framename} ; \
		grid ${framename}.toolbar -row 1 -column 2 -rowspan 2 -sticky new ; \
		} else { \
		grid forget ${framename}.toolbar } }]

   # Create or update the cell manager
   magic::cellmanager


   if {$MagicBLT} {
      $m add check -label "Cell Manager" -variable Opts(cellmgr) \
		-command [subst { magic::cellmanager check; \
		if { \$Opts(cellmgr) } { \
		wm deiconify .cellmgr ; raise .cellmgr \
		} else { \
		wm withdraw .cellmgr } }]

      .cellmgr.target.list.winmenu add radio -label ${framename} \
		-variable Opts(target) -value ${winname} \
		-command ".cellmgr.target.list configure \
		-text ${framename}"
   }

   $m add check -label "Tech Manager" -variable Opts(techmgr) \
	-command [subst { magic::techmanager check; \
		if { \$Opts(techmgr) } { \
		wm deiconify .techmgr ; raise .techmgr \
		} else { \
		wm withdraw .techmgr } }]

   # Set the default view

   update idletasks
   ${winname} magic::view

   magic::caption ${winname}

   # Rename "open" macro by rebinding
   magic::macro o "magic::openwrapper"
   magic::macro O "magic::closewrapper"

   return ${winname}
}

# Delete the wrapper and the layout window in it.

proc magic::closewrapper { framename } {
   global MagicBLT
   global Opts

   # Remove this window from the target list in cellmanager
   if {$MagicBLT} {
      if { $Opts(target) == "${framename}.magic" } {
	 set Opts(target) "default"
	 .cellmgr.target.list configure -text "default"
      } 
      set idx [.cellmgr.target.list.winmenu index $framename ]
      .cellmgr.target.list.winmenu delete $idx
   }

   ${framename}.magic magic::closewindow
   destroy $framename
}

# This procedure adds a command-line entry window to the bottom of
# a wrapper window (rudimentary functionality---incomplete)

proc magic::addcommandentry { framename } {
   global CAD_HOME
   if {![winfo exists ${framename}.eval]} {
      tkshell::YScrolled_Text ${framename}.eval -height 5
      tkshell::MakeEvaluator ${framename}.eval.text \
		"${framename}> " "${framename}.magic "
      tkshell::MainInit
   }
   set rc [grid size ${framename}]  
   set cols [lindex $rc 0]
   grid ${framename}.eval -row 3 -column 0 -columnspan $cols -sticky ew
   bind ${framename}.eval.text <Enter> {focus %W}
}

# Remove the command entry window from the bottom of a frame.

proc magic::deletecommandentry { framename } {
   grid forget ${framename}.eval
}

namespace import magic::openwrapper
puts "Use openwrapper to create a new GUI-based layout window"
namespace import magic::closewrapper
puts "Use closewrapper to remove a new GUI-based layout window"

# Create a simple wrapper for the 3D window. . . this can be
# greatly expanded upon.

proc magic::render {{cell ""}} {
   toplevel .render
   tkwait visibility .render
   magic::specialopen wind3d $cell .render.magic
   pack .render.magic
   bind .render.magic <Enter> {focus %W}
}

