/* CIFgen.c -
 *
 *	This file provides routines that generate CIF from Magic
 *	tile information, using one of the styles read from the
 *	technology file.
 *
 *     ********************************************************************* 
 *     * Copyright (C) 1985, 1990 Regents of the University of California. * 
 *     * Permission to use, copy, modify, and distribute this              * 
 *     * software and its documentation for any purpose and without        * 
 *     * fee is hereby granted, provided that the above copyright          * 
 *     * notice appear in all copies.  The University of California        * 
 *     * makes no representations about the suitability of this            * 
 *     * software for any purpose.  It is provided "as is" without         * 
 *     * express or implied warranty.  Export of this software outside     * 
 *     * of the United States of America may require an export license.    * 
 *     *********************************************************************
 */

#ifndef lint
static char rcsid[] __attribute__ ((unused)) = "$Header: /usr/cvsroot/magic-8.0/cif/CIFgen.c,v 1.1.1.1 2008/02/03 20:43:50 tim Exp $";
#endif  /* not lint */

#include <stdio.h>
#include <stdlib.h>             /* for abs() */
#include <math.h>		/* for ceil() and sqrt() */

#include "utils/magic.h"
#include "utils/geometry.h"
#include "tiles/tile.h"
#include "utils/hash.h"
#include "database/database.h"
#include "cif/CIFint.h"
#include "calma/calma.h"	/* for CalmaContactArrays */
#include "utils/stack.h"

/* The following global arrays hold pointers to the CIF planes
 * generated by the procedures in this module.  There are two
 * kinds of planes:  real CIF, which will ostensibly be output,
 * and temporary layers used to build up more complex geometric
 * functions.
 */

global Plane *CIFPlanes[MAXCIFLAYERS];

/* The following are paint tables used by the routines that implement
 * the geometric operations on mask layers, such as AND, OR, GROW,
 * etc.  There are really only two tables.  The first will paint
 * CIF_SOLIDTYPE over anything else, and the second will paint
 * space over anything else.  These tables are used on planes with
 * only two tile types, TT_SPACE and CIF_SOLIDTYPE, so they only
 * have two entries each.
 */

PaintResultType CIFPaintTable[] = {CIF_SOLIDTYPE, CIF_SOLIDTYPE};
PaintResultType CIFEraseTable[] = {TT_SPACE, TT_SPACE};

/* The following local variables are used as a convenience to pass
 * information between CIFGen and the various search functions.
 */

static int growDistance;	/* Distance to grow stuff. */
static Plane *cifPlane;		/* Plane acted on by search functions. */
static int cifScale;		/* Scale factor to use on tiles. */

extern void cifClipPlane();
extern void cifGenClip();

/*
 * ----------------------------------------------------------------------------
 *
 * cifPaintFunc --
 *
 * 	This search function is called during CIF_AND and CIF_OR
 *	and CIF_ANDNOT operations for each relevant tile.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	For the area of the tile, either paints or erases in
 *	cifNewPlane, depending on the paint table passed as parameter.
 *	The tile's area is scaled by cifScale first.
 * ----------------------------------------------------------------------------
 */

int
cifPaintFunc(tile, table)
    Tile *tile;
    PaintResultType *table;		/* Used for painting. */
{
    Rect area;

    TiToRect(tile, &area);
    area.r_xbot *= cifScale;
    area.r_xtop *= cifScale;
    area.r_ybot *= cifScale;
    area.r_ytop *= cifScale;

#ifdef NONMANHATTAN
    DBNMPaintPlane(cifPlane, TiGetTypeExact(tile), &area, table,
		(PaintUndoInfo *) NULL);
#else
    DBPaintPlane(cifPlane, &area, table, (PaintUndoInfo *) NULL);
#endif
    CIFTileOps += 1;
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifGrowGridFunc --
 *
 * 	Called for each relevant tile during grow-grid operations.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Scales the tile by cifScale, then expands its area by the
 *	remainder of the distance to the nearest grid point, as
 *	defined by the grid distance (growDistance) in the current
 *	CIFOp, then paints this area into cifNewPlane using the table
 *	passed as parameter.
 * ----------------------------------------------------------------------------
 */

int
cifGrowGridFunc(tile, table)
    Tile *tile;
    PaintResultType *table;		/* Table to be used for painting. */
{
    Rect area;
    int remainder;

    /* To be done---handle non-Manhattan geometry */
    TileType oldType = TiGetType(tile);

    TiToRect(tile, &area);

    /* In scaling the tile, watch out for infinities!!  If something
     * is already infinity, don't change it. */

    if (area.r_xbot > TiPlaneRect.r_xbot) area.r_xbot *= cifScale;
    if (area.r_ybot > TiPlaneRect.r_ybot) area.r_ybot *= cifScale;
    if (area.r_xtop < TiPlaneRect.r_xtop) area.r_xtop *= cifScale;
    if (area.r_ytop < TiPlaneRect.r_ytop) area.r_ytop *= cifScale;

    /* In scaling the tile, watch out for infinities!!  If something
     * is already infinity, don't change it. */

    if (area.r_xbot > TiPlaneRect.r_xbot)
	area.r_xbot -= (abs(area.r_xbot) % growDistance);
    if (area.r_ybot > TiPlaneRect.r_ybot)
	area.r_ybot -= (abs(area.r_ybot) % growDistance);
    if (area.r_xtop < TiPlaneRect.r_xtop)
	area.r_xtop += (abs(area.r_xtop) % growDistance);
    if (area.r_ytop < TiPlaneRect.r_ytop)
	area.r_ytop += (abs(area.r_ytop) % growDistance);

    DBPaintPlane(cifPlane, &area, table, (PaintUndoInfo *) NULL);

    CIFTileOps += 1;
    return 0;
}

#ifdef NONMANHATTAN
/*
 * ----------------------------------------------------------------------------
 *
 * cifGrowEuclideanFunc --
 *
 * 	Called for each relevant tile during grow or shrink operations.
 *	For growing, these are solid tiles.  For shrinking, these are
 *	space tiles.   This routine differs from cifGrowFunc in that it
 *	grows the minimum distance on non-Manhattan edges necessary to
 *	create a euclidean distance to the edge of growDistance while
 *	keeping corner points on-grid.  This requires a substantially
 *	different algorithm from cifGrowFunc.
 *	
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Scales the tile by cifScale, then expands its area by the
 *	distance in the current CIFOp, then paints this area into
 *	cifNewPlane using the table passed as parameter.
 * ----------------------------------------------------------------------------
 */

#define GROW_NORTH 0x1
#define GROW_SOUTH 0x2
#define GROW_EAST  0x4
#define GROW_WEST  0x8

#define STOP_NW 0x1	/*	   WN     EN	*/
#define STOP_SW 0x2	/*	NW +-------+ NE	*/
#define STOP_NE 0x4	/*	   |       |	*/
#define STOP_SE 0x8	/*	   |       |	*/
#define STOP_WN 0x10	/*      SW +-------+ SE	*/
#define STOP_WS 0x20	/*         WS     ES	*/
#define STOP_EN 0x40
#define STOP_ES 0x80

int
cifGrowEuclideanFunc(tile, table)
    Tile *tile;
    PaintResultType *table;		/* Table to be used for painting. */
{
    Tile *tp;
    Rect area, rtmp;
    TileType oldType = TiGetTypeExact(tile);
    unsigned char growDirs = GROW_NORTH | GROW_SOUTH | GROW_EAST | GROW_WEST;
    unsigned char cornerDirs = 0;

    TiToRect(tile, &area);

    /* In scaling the tile, watch out for infinities!!  If something
     * is already infinity, don't change it. */

    if (area.r_xbot > TiPlaneRect.r_xbot)
	area.r_xbot *= cifScale;
    else
	growDirs &= ~GROW_WEST;
    if (area.r_ybot > TiPlaneRect.r_ybot)
	area.r_ybot *= cifScale;
    else
	growDirs &= ~GROW_SOUTH;
    if (area.r_xtop < TiPlaneRect.r_xtop)
	area.r_xtop *= cifScale;
    else
	growDirs &= ~GROW_EAST;
    if (area.r_ytop < TiPlaneRect.r_ytop)
	area.r_ytop *= cifScale;
    else
	growDirs &= ~GROW_NORTH;

    /* Grow on diagonal tiles:  grow rectangular tiles around the	*/
    /* straight edges of the right-triangle, then copy the diagonal	*/
    /* tile (at its original size) in the direction of the diagonal.	*/
    /* Note:  A diagonal tile, by definition, can't have infinities.	*/

    if (oldType & TT_DIAGONAL)
    {
	int growDistanceX, growDistanceY;
	int height, width;
	double hyp;

	if (oldType & TT_SIDE)
	    growDirs &= ~GROW_WEST;
	else
	    growDirs &= ~GROW_EAST;

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION))
	    growDirs &= ~GROW_SOUTH;
	else
	    growDirs &= ~GROW_NORTH;

	/* Grow non-Manhattan edges to (the closest integer value	*/
	/* to) growDistance along the normal to the edge.  This	*/
	/* will overestimate the distance only to the minimum 	*/
	/* amount necessary to ensure on-grid endpoints.		*/

	width = area.r_xtop - area.r_xbot;
	height = area.r_ytop - area.r_ybot;
	hyp = sqrt((double)(width * width + height * height));
	growDistanceY = ceil((growDistance * (hyp - height)) / width);
	growDistanceX = ceil((growDistance * (hyp - width)) / height);

	/* Draw vertical tile to distance X */

	rtmp = area;
	if (!(growDirs & GROW_EAST))  rtmp.r_xtop = rtmp.r_xbot + growDistanceX;
	if (!(growDirs & GROW_WEST))  rtmp.r_xbot = rtmp.r_xtop - growDistanceX;
	if (!(growDirs & GROW_SOUTH)) rtmp.r_ybot -=growDistance;
	if (!(growDirs & GROW_NORTH)) rtmp.r_ytop += growDistance;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);

	/* Draw horizontal tile to distance Y */

	rtmp = area;
	if (!(growDirs & GROW_EAST))  rtmp.r_xtop += growDistance;
	if (!(growDirs & GROW_WEST))  rtmp.r_xbot -= growDistance;
	if (!(growDirs & GROW_SOUTH)) rtmp.r_ybot = rtmp.r_ytop - growDistanceY;
	if (!(growDirs & GROW_NORTH)) rtmp.r_ytop = rtmp.r_ybot + growDistanceY;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);

	/* Finally:  translate, resize, and paint the diagonal tile */

	rtmp = area;

	if (!(growDirs & GROW_NORTH))
	    rtmp.r_ytop += growDistance;
	else
	    rtmp.r_ytop -= growDistanceY;
	if (!(growDirs & GROW_SOUTH))
	    rtmp.r_ybot -= growDistance;
	else
	    rtmp.r_ybot += growDistanceY;
	if (!(growDirs & GROW_EAST))
	    rtmp.r_xtop += growDistance;
	else
	    rtmp.r_xtop -= growDistanceX;
	if (!(growDirs & GROW_WEST))
	    rtmp.r_xbot -= growDistance;
	else
	    rtmp.r_xbot += growDistanceX;

	DBNMPaintPlane(cifPlane, oldType, &rtmp, table, (PaintUndoInfo *) NULL);
	oldType = (growDirs & GROW_EAST) ? TiGetRightType(tile) : TiGetLeftType(tile);
    }
    else
	DBPaintPlane(cifPlane, &area, table, (PaintUndoInfo *) NULL);

    /* Check tile corner areas for paint */

    tp = TR(tile);
    if (TiGetLeftType(tp) == oldType) cornerDirs |= STOP_NE;
    for (; TOP(LB(tp)) > BOTTOM(tile); tp = LB(tp));
    if (TiGetLeftType(tp) == oldType) cornerDirs |= STOP_SE;

    tp = RT(tile);
    if (TiGetBottomType(tp) == oldType) cornerDirs |= STOP_EN;
    for (; RIGHT(BL(tp)) > LEFT(tile); tp = BL(tp));
    if (TiGetBottomType(tp) == oldType) cornerDirs |= STOP_WN;

    tp = BL(tile);
    if (TiGetRightType(tp) == oldType) cornerDirs |= STOP_SW;
    for (; BOTTOM(RT(tp)) < TOP(tile); tp = RT(tp));
    if (TiGetRightType(tp) == oldType) cornerDirs |= STOP_NW;

    tp = LB(tile);
    if (TiGetTopType(tp) == oldType) cornerDirs |= STOP_WS;
    for (; LEFT(TR(tp)) < RIGHT(tile); tp = TR(tp));
    if (TiGetTopType(tp) == oldType) cornerDirs |= STOP_ES;

    if (growDirs & GROW_NORTH)
    {
	rtmp = area;
	rtmp.r_ybot = area.r_ytop;
	rtmp.r_ytop = area.r_ytop + growDistance;
	if ((cornerDirs & (STOP_EN | STOP_NE)) == 0) rtmp.r_xtop += growDistance;
	if ((cornerDirs & (STOP_WN | STOP_NW)) == 0) rtmp.r_xbot -= growDistance;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);
    }

    if (growDirs & GROW_EAST)
    {
	rtmp = area;
	rtmp.r_xbot = area.r_xtop;
	rtmp.r_xtop = area.r_xtop + growDistance;
	if ((cornerDirs & (STOP_EN | STOP_NE)) == 0) rtmp.r_ytop += growDistance;
	if ((cornerDirs & (STOP_SE | STOP_ES)) == 0) rtmp.r_ybot -= growDistance;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);
    }

    if (growDirs & GROW_SOUTH)
    {
	rtmp = area;
	rtmp.r_ytop = area.r_ybot;
	rtmp.r_ybot = area.r_ybot - growDistance;
	if ((cornerDirs & (STOP_SE | STOP_ES)) == 0) rtmp.r_xtop += growDistance;
	if ((cornerDirs & (STOP_SW | STOP_WS)) == 0) rtmp.r_xbot -= growDistance;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);
    }

    if (growDirs & GROW_WEST)
    {
	rtmp = area;
	rtmp.r_xtop = area.r_xbot;
	rtmp.r_xbot = area.r_xbot - growDistance;
	if ((cornerDirs & (STOP_SW | STOP_WS)) == 0) rtmp.r_ybot -= growDistance;
	if ((cornerDirs & (STOP_NW | STOP_WN)) == 0) rtmp.r_ytop += growDistance;
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);
    }

    CIFTileOps++;
    return 0;
}
#endif	 /* NONMANHATTAN */

/*
 * ----------------------------------------------------------------------------
 *
 * cifGrowFunc --
 *
 * 	Called for each relevant tile during grow or shrink operations.
 *	For growing, these are solid tiles.  For shrinking, these are
 *	space tiles.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Scales the tile by cifScale, then expands its area by the
 *	distance in the current CIFOp, then paints this area into
 *	cifNewPlane using the table passed as parameter.
 * ----------------------------------------------------------------------------
 */

int
cifGrowFunc(tile, table)
    Tile *tile;
    PaintResultType *table;		/* Table to be used for painting. */
{
    Rect area;
    TileType oldType = TiGetTypeExact(tile);

    TiToRect(tile, &area);

    /* In scaling the tile, watch out for infinities!!  If something
     * is already infinity, don't change it. */

    if (area.r_xbot > TiPlaneRect.r_xbot) area.r_xbot *= cifScale;
    if (area.r_ybot > TiPlaneRect.r_ybot) area.r_ybot *= cifScale;
    if (area.r_xtop < TiPlaneRect.r_xtop) area.r_xtop *= cifScale;
    if (area.r_ytop < TiPlaneRect.r_ytop) area.r_ytop *= cifScale;

#ifdef NONMANHATTAN
    /* Grow on diagonal tiles:  grow rectangular tiles around the	*/
    /* straight edges of the right-triangle, then copy the diagonal	*/
    /* tile (at its original size) in the direction of the diagonal.	*/
    /* Note:  A diagonal tile, by definition, can't have infinities.	*/

    if (oldType & TT_DIAGONAL)
    {
	Rect rtmp;

	/* Grow top and bottom */

	rtmp.r_ybot = area.r_ybot - growDistance;
	rtmp.r_ytop = area.r_ytop + growDistance;

	/* Grow around left or right edge */

	if (oldType & TT_SIDE)
	{
	    rtmp.r_xbot = area.r_xtop - growDistance;
	    rtmp.r_xtop = area.r_xtop + growDistance;
	}
	else
	{
	    rtmp.r_xbot = area.r_xbot - growDistance;
	    rtmp.r_xtop = area.r_xbot + growDistance;
	}
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);

	/* Now do the same for the other edge. */

	rtmp.r_xbot = area.r_xbot - growDistance;
	rtmp.r_xtop = area.r_xtop + growDistance;

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION)) /* top */
	{
	    rtmp.r_ybot = area.r_ytop - growDistance;
	    rtmp.r_ytop = area.r_ytop + growDistance;
	}
	else /* bottom */
	{
	    rtmp.r_ybot = area.r_ybot - growDistance;
	    rtmp.r_ytop = area.r_ybot + growDistance;
	}
	DBPaintPlane(cifPlane, &rtmp, table, (PaintUndoInfo *) NULL);

	/* Finally, Move and replace the diagonal tile */

	if (oldType & TT_SIDE)
	{
	    rtmp.r_xtop = area.r_xtop - growDistance;
	    rtmp.r_xbot = area.r_xbot - growDistance;
	}
	else
	{
	    rtmp.r_xtop = area.r_xtop + growDistance;
	    rtmp.r_xbot = area.r_xbot + growDistance;
	}

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION)) /* top */
	{
	    rtmp.r_ytop = area.r_ytop - growDistance;
	    rtmp.r_ybot = area.r_ybot - growDistance;
	}
	else	/* bottom */
	{
	    rtmp.r_ytop = area.r_ytop + growDistance;
	    rtmp.r_ybot = area.r_ybot + growDistance;
	}
	DBNMPaintPlane(cifPlane, oldType, &rtmp, table, (PaintUndoInfo *) NULL);
    }
    else
    {
#endif

    /* In scaling the tile, watch out for infinities!!  If something
     * is already infinity, don't change it. */

    if (area.r_xbot > TiPlaneRect.r_xbot) area.r_xbot -= growDistance;
    if (area.r_ybot > TiPlaneRect.r_ybot) area.r_ybot -= growDistance;
    if (area.r_xtop < TiPlaneRect.r_xtop) area.r_xtop += growDistance;
    if (area.r_ytop < TiPlaneRect.r_ytop) area.r_ytop += growDistance;

    DBPaintPlane(cifPlane, &area, table, (PaintUndoInfo *) NULL);

#ifdef NONMANHATTAN
    }
#endif

    CIFTileOps += 1;
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifBloatFunc --
 *
 * 	Called once for each tile to be selectively bloated.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Uses the bloat table in the current CIFOp to expand the
 *	tile depending on which tiles it abuts, then paints the
 *	expanded area into the new CIF plane.  The bloat table
 *	contains one entry for each tile type.  When that tile
 *	type is seen next to the current tile, the area of the current
 *	tile is bloated by the table value in that location.  The
 *	only exception to this rule is that internal edges (between
 *	two tiles of the same type) cause no bloating.
 *	Note:  the original tile is scaled into CIF coordinates.
 * ----------------------------------------------------------------------------
 */

int
cifBloatFunc(tile, clientData)
    Tile *tile;
    ClientData clientData;
{
    Rect tileArea, cifArea, bloat;
    TileType oldType, type, topLeftType, bottomRightType;
    Tile *t;
    int tilestart, tilestop, cifstart, cifstop;
    BloatData *bloats = (BloatData *)clientData;
    int *bloatTable = (int *)bloats->bl_distance;

    oldType = TiGetTypeExact(tile);
    TiToRect(tile, &tileArea);

    /* Output the original area of the tile. */

    cifArea = tileArea;
    cifArea.r_xbot *= cifScale;
    cifArea.r_xtop *= cifScale;
    cifArea.r_ybot *= cifScale;
    cifArea.r_ytop *= cifScale;

#ifdef NONMANHATTAN
    DBNMPaintPlane(cifPlane, oldType, &cifArea,
		CIFPaintTable, (PaintUndoInfo *) NULL);
#else
    DBPaintPlane(cifPlane, &cifArea, CIFPaintTable, (PaintUndoInfo *) NULL);
#endif

    /* Go around the tile, scanning the neighbors along each side.
     * Start with the left side, and output the bloats along that
     * side, if any.
     */
    
    tilestop = tileArea.r_ytop;
    cifstop = cifArea.r_ytop;
    type = oldType;

#ifdef NONMANHATTAN
    /* If the tile type doesn't exist on the left side, skip */
    if (oldType & TT_DIAGONAL)
    {
	type = TiGetLeftType(tile);
	if (oldType & TT_SIDE)
	{
	    if (oldType & TT_DIRECTION)
	    {
		topLeftType = type;
		goto dotop;
	    }
	    else
	    {
		tilestop = tileArea.r_ybot;
		cifstop = cifArea.r_ybot;
		type = TiGetBottomType(tile);
	    }
	}
    }
#endif

    bloat.r_ybot = cifArea.r_ybot - bloatTable[TiGetRightType(LB(tile))];
    bloat.r_xtop = cifArea.r_xbot;
    for (t = BL(tile); BOTTOM(t) < TOP(tile); t = RT(t))
    {
	topLeftType = TiGetRightType(t);
	bloat.r_xbot = bloat.r_xtop - bloatTable[topLeftType];
	if (TOP(t) > tilestop)
	    bloat.r_ytop = cifstop;
	else bloat.r_ytop = cifScale * TOP(t);
	if ((bloatTable[topLeftType] != 0) && (topLeftType != type))
	    DBPaintPlane(cifPlane, &bloat, CIFPaintTable,
		(PaintUndoInfo *) NULL);
	bloat.r_ybot = bloat.r_ytop;
    }

    /* Now do the top side.  Use the type of the top-left tile to
     * side-extend the left end of the top bloat in order to match
     * things up at the corner.
     */

dotop:
    cifstart = cifArea.r_xtop;
#ifdef NONMANHATTAN
    /* If the tile type doesn't exist on the top side, skip */
    if (oldType & TT_DIAGONAL)
    {
	type = TiGetTopType(tile);
	if (((oldType & TT_SIDE) >> 1) != (oldType & TT_DIRECTION))
	{
	    if (oldType & TT_SIDE)
		goto doright;
	    else
	    {
		cifstart = cifArea.r_xbot;
		type = TiGetLeftType(tile);
	    }
	}
    }
#endif

    bloat.r_ybot = cifArea.r_ytop;
    bloat.r_xtop = cifstart;
    for (t = RT(tile); RIGHT(t) > LEFT(tile); t = BL(t))
    {
	TileType otherType = TiGetBottomType(t);
	bloat.r_ytop = bloat.r_ybot + bloatTable[otherType];
	if (LEFT(t) <= tileArea.r_xbot)
	    bloat.r_xbot = cifArea.r_xbot - bloatTable[topLeftType];
	else bloat.r_xbot = cifScale * LEFT(t);
	if ((bloatTable[otherType] != 0) && (otherType != type))
	    DBPaintPlane(cifPlane, &bloat, CIFPaintTable,
		(PaintUndoInfo *) NULL);
	bloat.r_xtop = bloat.r_xbot;
    }

    /* Now do the right side. */

doright:
    tilestop = tileArea.r_ybot;
    cifstop = cifArea.r_ybot;
#ifdef NONMANHATTAN
    /* If the tile type doesn't exist on the right side, skip */
    if (oldType & TT_DIAGONAL)
    {
	type = TiGetRightType(tile);
	if (!(oldType & TT_SIDE))
	{
	    if (oldType & TT_DIRECTION)
	    {
		bottomRightType = type;
		goto dobottom;
	    }
	    else
	    {
		tilestop = tileArea.r_ytop;
		cifstop = cifArea.r_ytop;
		type = TiGetTopType(tile);
	    }
	}
    }
#endif

    bloat.r_ytop = cifArea.r_ytop + bloatTable[TiGetLeftType(RT(tile))];
    bloat.r_xbot = cifArea.r_xtop;
    for (t = TR(tile); TOP(t) > BOTTOM(tile); t = LB(t))
    {
	bottomRightType = TiGetLeftType(t);
	bloat.r_xtop = bloat.r_xbot + bloatTable[bottomRightType];
	if (BOTTOM(t) < tilestop)
	    bloat.r_ybot = cifstop;
	else bloat.r_ybot = cifScale * BOTTOM(t);
	if ((bloatTable[bottomRightType] != 0) && (bottomRightType != type))
	    DBPaintPlane(cifPlane, &bloat, CIFPaintTable,
		(PaintUndoInfo *) NULL);
	bloat.r_ytop = bloat.r_ybot;
    }

    /* Now do the bottom side.  Use the type of the bottom-right tile
     * to side-extend the right end of the bottom bloat in order to match
     * things up at the corner.
     */

dobottom:
    cifstart = cifArea.r_xbot;
#ifdef NONMANHATTAN
    /* If the tile type doesn't exist on the bottom side, skip */
    if (oldType & TT_DIAGONAL)
    {
	type = TiGetBottomType(tile);
	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION))
	{
	    if (!(oldType & TT_DIRECTION))
		goto dodiagonal;
	    else
	    {
		cifstart = cifArea.r_xtop;
		type = TiGetRightType(tile);
	    }
	}
    }
#endif

    bloat.r_ytop = cifArea.r_ybot;
    bloat.r_xbot = cifstart;
    for (t = LB(tile); LEFT(t) < RIGHT(tile); t = TR(t))
    {
	TileType otherType = TiGetTopType(t);
	bloat.r_ybot = bloat.r_ytop - bloatTable[otherType];
	if (RIGHT(t) >= tileArea.r_xtop)
	    bloat.r_xtop = cifArea.r_xtop + bloatTable[bottomRightType];
	else bloat.r_xtop = cifScale * RIGHT(t);
	if ((bloatTable[otherType] != 0) && (otherType != type))
	    DBPaintPlane(cifPlane, &bloat, CIFPaintTable,
		(PaintUndoInfo *) NULL);
	bloat.r_xbot = bloat.r_xtop;
    }

#ifdef NONMANHATTAN
dodiagonal:

    /* This is a modified version of the nonmanhattan grow function.	*/
    /* We grow only in the direction of the diagonal.			*/			
    /* This will not work in all situations!  Corner extensions are not	*/
    /* considered (but should be, for completeness).			*/

    if (oldType & TT_DIAGONAL)
    {
	TileType otherType = (oldType & TT_SIDE) ?
		TiGetLeftType(tile) : TiGetRightType(tile);
	int dist = bloatTable[otherType];

	/* Grow top and bottom */

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION)) /* top */
	{
	    bloat.r_ybot = cifArea.r_ytop - dist;
	    bloat.r_ytop = cifArea.r_ytop;
	}
	else /* bottom */
	{
	    bloat.r_ybot = cifArea.r_ybot;
	    bloat.r_ytop = cifArea.r_ybot + dist;
	}

	/* Grow around left or right edge */

	if (oldType & TT_SIDE)
	{
	    bloat.r_xbot = cifArea.r_xbot - dist;
	    bloat.r_xtop = cifArea.r_xtop;
	}
	else
	{
	    bloat.r_xbot = cifArea.r_xbot;
	    bloat.r_xtop = cifArea.r_xtop + dist;
	}
	DBPaintPlane(cifPlane, &bloat, CIFPaintTable, (PaintUndoInfo *) NULL);

	/* Now do the same for the left or right edge. */

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION)) /* top */
	{
	    bloat.r_ybot = cifArea.r_ybot - dist;
	    bloat.r_ytop = cifArea.r_ytop;
	}
	else /* bottom */
	{
	    bloat.r_ybot = cifArea.r_ybot;
	    bloat.r_ytop = cifArea.r_ytop + dist;
	}

	if (oldType & TT_SIDE)
	{
	    bloat.r_xbot = cifArea.r_xtop - dist;
	    bloat.r_xtop = cifArea.r_xtop;
	}
	else
	{
	    bloat.r_xbot = cifArea.r_xbot;
	    bloat.r_xtop = cifArea.r_xbot + dist;
	}
	DBPaintPlane(cifPlane, &bloat, CIFPaintTable, (PaintUndoInfo *) NULL);

	/* Finally, Move and replace the diagonal tile */

	if (oldType & TT_SIDE)
	{
	    bloat.r_xtop = cifArea.r_xtop - dist;
	    bloat.r_xbot = cifArea.r_xbot - dist;
	}
	else
	{
	    bloat.r_xtop = cifArea.r_xtop + dist;
	    bloat.r_xbot = cifArea.r_xbot + dist;
	}

	if (((oldType & TT_SIDE) >> 1) == (oldType & TT_DIRECTION)) /* top */
	{
	    bloat.r_ytop = cifArea.r_ytop - dist;
	    bloat.r_ybot = cifArea.r_ybot - dist;
	}
	else	/* bottom */
	{
	    bloat.r_ytop = cifArea.r_ytop + dist;
	    bloat.r_ybot = cifArea.r_ybot + dist;
	}
	DBNMPaintPlane(cifPlane, oldType, &bloat, CIFPaintTable,
			(PaintUndoInfo *) NULL);
    }
#endif

    CIFTileOps += 1;
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifBloatAllFunc --
 *
 * 	Called once for each tile to be selectively bloated
 *	using the CIFOP_BLOATALL operation.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Uses the bloat table in the current CIFOp to expand the tile,
 *	depending on which tiles it abuts.  All bordering material that
 *	has bl_distance = 1 is painted into the result plane.
 *
 * ----------------------------------------------------------------------------
 */

#define CIF_PENDING     0
#define CIF_UNPROCESSED CLIENTDEFAULT
#define CIF_PROCESSED   1

#define PUSHTILE(tp) \
    if ((tp)->ti_client == (ClientData) CIF_UNPROCESSED) { \
	(tp)->ti_client = (ClientData) CIF_PENDING; \
	STACKPUSH((ClientData) (tp), BloatStack); \
    }

int
cifBloatAllFunc(tile, op)
    Tile *tile;			/* The tile to be processed. */
    CIFOp *op;			/* Describes the operation to be performed */
{
    Rect area;
    TileTypeBitMask connect;
    Tile *t, *tp;
    TileType type;
    BloatData *bloats = (BloatData *)op->co_client;
    int i;
    static Stack *BloatStack = (Stack *)NULL;

    /* Create a mask of all connecting types (these must be in a single
     * plane), then call a search function to find all connecting material
     * of these types.
     */

    TTMaskZero(&connect);
    for (i = 0; i < TT_MAXTYPES; i++)
	if (bloats->bl_distance[i] != 0)
	    TTMaskSetType(&connect, i);

    /* This search function is based on drcCheckArea */

    if (BloatStack == (Stack *)NULL)
	BloatStack = StackNew(64);

    PUSHTILE(tile);
    while (!StackEmpty(BloatStack))
    {
	t = (Tile *) STACKPOP(BloatStack);
	if (t->ti_client != (ClientData)CIF_PENDING) continue;
        t->ti_client = (ClientData)CIF_PROCESSED;
	
	/* Get the tile into CIF coordinates. */

	TiToRect(t, &area);
	area.r_xbot *= cifScale;
	area.r_ybot *= cifScale;
	area.r_xtop *= cifScale;
	area.r_ytop *= cifScale;

#ifdef NONMANHATTAN
	DBNMPaintPlane(cifPlane, TiGetTypeExact(t), &area,
		CIFPaintTable, (PaintUndoInfo *) NULL);
#else
	DBPaintPlane(cifPlane, &area, CIFPaintTable, (PaintUndoInfo *) NULL);
#endif

	/* Top */
	for (tp = RT(t); RIGHT(tp) > LEFT(t); tp = BL(tp))
	    if (TTMaskHasType(&connect, TiGetBottomType(tp)))
		PUSHTILE(tp);

	/* Left */
	for (tp = BL(t); BOTTOM(tp) < TOP(t); tp = RT(tp))
	    if (TTMaskHasType(&connect, TiGetRightType(tp)))
		PUSHTILE(tp);

	/* Bottom */
	for (tp = LB(t); LEFT(tp) < RIGHT(t); tp = TR(tp))
	    if (TTMaskHasType(&connect, TiGetTopType(tp)))
		PUSHTILE(tp);

	/* Right */
	for (tp = TR(t); TOP(tp) > BOTTOM(t); tp = LB(tp))
	    if (TTMaskHasType(&connect, TiGetLeftType(tp)))
		PUSHTILE(tp);
    }

    /* Clear the tiles that were processed */

    tile->ti_client = (ClientData)CIF_UNPROCESSED;
    STACKPUSH(tile, BloatStack);
    while (!StackEmpty(BloatStack))
    {
	t = (Tile *) STACKPOP(BloatStack);

	/* Top */
	for (tp = RT(t); RIGHT(tp) > LEFT(t); tp = BL(tp))
	    if (tp->ti_client != (ClientData)CIF_UNPROCESSED)
	    {
		tp->ti_client = (ClientData)CIF_UNPROCESSED;
		STACKPUSH(tp, BloatStack);
	    }

	/* Left */
	for (tp = BL(t); BOTTOM(tp) < TOP(t); tp = RT(tp))
	    if (tp->ti_client != (ClientData)CIF_UNPROCESSED)
	    {
		tp->ti_client = (ClientData)CIF_UNPROCESSED;
		STACKPUSH(tp, BloatStack);
	    }

	/* Bottom */
	for (tp = LB(t); LEFT(tp) < RIGHT(t); tp = TR(tp))
	    if (tp->ti_client != (ClientData)CIF_UNPROCESSED)
	    {
		tp->ti_client = (ClientData)CIF_UNPROCESSED;
		STACKPUSH(tp, BloatStack);
	    }

	/* Right */
	for (tp = TR(t); TOP(tp) > BOTTOM(t); tp = LB(tp))
	    if (tp->ti_client != (ClientData)CIF_UNPROCESSED)
	    {
		tp->ti_client = (ClientData)CIF_UNPROCESSED;
		STACKPUSH(tp, BloatStack);
	    }
    }
    return 0;	/* Keep the search alive. . . */
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifBloatMaxFunc --
 *
 * 	Called once for each tile to be selectively bloated or
 *	shrunk using the CIFOP_BLOATMAX or CIFOP_BLOATMIN operation.
 *
 * Results:
 *	Always returns 0 to keep the search alive.
 *
 * Side effects:
 *	Uses the bloat table in the current CIFOp to expand or shrink
 *	the tile, depending on which tiles it abuts.  The rectangular
 *	area of the tile is modified by moving each side in or out
 *	by the maximum bloat distance for any of its neighbors on
 *	that side.  The result is a new rectangle, which is OR'ed
 *	into the result plane.  Note:  any edge between two tiles of
 *	same type is given a zero bloat factor.
 *
 * ----------------------------------------------------------------------------
 */

int
cifBloatMaxFunc(tile, op)
    Tile *tile;			/* The tile to be processed. */
    CIFOp *op;			/* Describes the operation to be performed
				 * (all we care about is the opcode and
				 * bloat table).
				 */
{
    Rect area;
    int bloat, tmp;
    Tile *t;
    TileType type, otherType;
    BloatData *bloats = (BloatData *)op->co_client;

    /* Get the tile into CIF coordinates. */

    type = TiGetType(tile);
    TiToRect(tile, &area);
    area.r_xbot *= cifScale;
    area.r_ybot *= cifScale;
    area.r_xtop *= cifScale;
    area.r_ytop *= cifScale;

    /* See how much to adjust the left side of the tile. */

    if (op->co_opcode == CIFOP_BLOATMAX) bloat = -10000000;
    else bloat = 10000000;
    for (t = BL(tile); BOTTOM(t) < TOP(tile); t = RT(t))
    {    
	otherType = TiGetType(t);
	if (otherType == type) continue;
	tmp = bloats->bl_distance[otherType];
	if (op->co_opcode == CIFOP_BLOATMAX)
	{
	    if (tmp > bloat) bloat = tmp;
	}
	else if (tmp < bloat) bloat = tmp;
    }
    if ((bloat < 10000000) && (bloat > -10000000))
	area.r_xbot -= bloat;

    /* Now figure out how much to adjust the top side of the tile. */

    if (op->co_opcode == CIFOP_BLOATMAX) bloat = -10000000;
    else bloat = 10000000;
    for (t = RT(tile); RIGHT(t) > LEFT(tile); t = BL(t))
    {    
	otherType = TiGetType(t);
	if (otherType == type) continue;
	tmp = bloats->bl_distance[otherType];
	if (op->co_opcode == CIFOP_BLOATMAX)
	{
	    if (tmp > bloat) bloat = tmp;
	}
	else if (tmp < bloat) bloat = tmp;
    }
    if ((bloat < 10000000) && (bloat > -10000000))
	area.r_ytop += bloat;

    /* Now the right side. */
 
    if (op->co_opcode == CIFOP_BLOATMAX) bloat = -10000000;
    else bloat = 10000000;
    for (t = TR(tile); TOP(t) > BOTTOM(tile); t = LB(t))
    {    
	otherType = TiGetType(t);
	if (otherType == type) continue;
	tmp = bloats->bl_distance[otherType];
	if (op->co_opcode == CIFOP_BLOATMAX)
	{
	    if (tmp > bloat) bloat = tmp;
	}
	else if (tmp < bloat) bloat = tmp;
    }
    if ((bloat < 10000000) && (bloat > -10000000))
	area.r_xtop += bloat;

    /* Lastly, do the bottom side. */

    if (op->co_opcode == CIFOP_BLOATMAX) bloat = -10000000;
    else bloat = 10000000;
    for (t = LB(tile); LEFT(t) < RIGHT(tile); t = TR(t))
    {    
	otherType = TiGetType(t);
	if (otherType == type) continue;
	tmp = bloats->bl_distance[otherType];
	if (op->co_opcode == CIFOP_BLOATMAX)
	{
	    if (tmp > bloat) bloat = tmp;
	}
	else if (tmp < bloat) bloat = tmp;
    }
    if ((bloat < 10000000) && (bloat > -10000000))
	area.r_ybot -= bloat;

    /* Make sure the tile didn't shrink into negativity.  If it's
     * ok, paint it into the new plane being built.
     */

    if ((area.r_xbot > area.r_xtop) || (area.r_ybot > area.r_ytop))
    {
	TiToRect(tile, &area);
	area.r_xbot *= cifScale;
	area.r_xtop *= cifScale;
	area.r_ybot *= cifScale;
	area.r_ytop *= cifScale;
	CIFError(&area, "tile inverted by shrink");
    }
    else
#ifdef NONMANHATTAN
	DBNMPaintPlane(cifPlane, TiGetTypeExact(tile), &area,
		CIFPaintTable, (PaintUndoInfo *) NULL);
#else
	DBPaintPlane(cifPlane, &area, CIFPaintTable, (PaintUndoInfo *) NULL);
#endif

    CIFTileOps += 1;
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * inside_triangle --
 *
 *	Test if the specified rectangle is completely inside the tile.
 *	Tile is presumed to be a split tile.
 *
 * Results:
 *	TRUE if inside, FALSE if outside or overlapping.
 *
 * Side Effects:
 *	None.
 *
 * Notes:
 *	This is an optimized version of the code in DBtiles.c.  It
 *	assumes that the square is never infinite.
 *
 * ----------------------------------------------------------------------------
 */

bool
inside_triangle(rect, tile)
    Rect *rect;
    Tile *tile;
{
    int theight, twidth;
    dlong f1, f2, f3, f4;

    theight = TOP(tile) - BOTTOM(tile);
    twidth = RIGHT(tile) - LEFT(tile);

    f1 = (dlong)(TOP(tile) - rect->r_ybot) * twidth;
    f2 = (dlong)(rect->r_ytop - BOTTOM(tile)) * twidth;

    if (SplitLeftType(tile) != TT_SPACE)
    {
        /* Inside-of-triangle check */
        f4 = (dlong)(rect->r_xbot - LEFT(tile)) * theight;
        if (SplitDirection(tile) ? (f1 > f4) : (f2 > f4))
	    return TRUE;
    }
    else
    {
        /* Inside-of-triangle check */
        f3 = (dlong)(RIGHT(tile) - rect->r_xtop) * theight;
        if (SplitDirection(tile) ? (f2 > f3) : (f1 > f3))
	    return TRUE;
    }
    return FALSE;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifContactFunc --
 *
 *	Called for each relevant tile when chopping up contacts into
 *	square cuts using the procedure of using a pre-defined cell
 *	definition of a single contact and arraying the cell.
 *	Technically, this is not a CIF function because CIF does not
 *	have an array method for cells, so there is no advantage to
 *	drawing lots of subcells in place of drawing lots of boxes.
 *	In GDS, however, the array method is much more compact than
 *	drawing individual cuts when the array size is larger than 1.
 *
 * Results:
 *	Normally returns 0 to keep the search going.  Return 1 if
 *	the contact type cannot be found, which halts the search.
 *
 * Side effects:
 *	Stuff is painted into cifPlane.
 *
 * ----------------------------------------------------------------------------
 */

int
cifContactFunc(tile, csi)
    Tile *tile;			/* Tile to be diced up. */
    CIFSquaresInfo *csi; 	/* Describes how to generate squares. */
{
    Rect area;
    int i, nAcross, j, nUp, left, bottom, pitch, halfsize;
    bool result;
    SquaresData *squares = csi->csi_squares;

#ifdef NONMANHATTAN
    /* For now, don't allow squares on non-manhattan tiles */
    if (IsSplit(tile))
	return 0;
#endif

    TiToRect(tile, &area);
    pitch = squares->sq_size + squares->sq_sep;

    nAcross = (area.r_xtop - area.r_xbot + squares->sq_sep
	    - (2 * squares->sq_border)) / pitch;
    if (nAcross == 0)
    {
	left = (area.r_xbot + area.r_xtop - squares->sq_size) / 2;
	if (left >= area.r_xbot) nAcross = 1;
    }
    else
	left = (area.r_xbot + area.r_xtop + squares->sq_sep
		- (nAcross * pitch)) / 2;

    nUp = (area.r_ytop - area.r_ybot + squares->sq_sep
	    - (2 * squares->sq_border)) / pitch;
    if (nUp == 0)
    {
	bottom = (area.r_ybot + area.r_ytop - squares->sq_size) / 2;
	if (bottom >= area.r_ybot) nUp = 1;
    }
    else
	bottom = (area.r_ybot + area.r_ytop + squares->sq_sep
		- (nUp * pitch)) / 2;

    /* Lower-left coordinate should be centered on the contact cut, as
     * contact definitions are centered on the origin.
     */
    halfsize = (squares->sq_size / 2);
    left += halfsize;
    bottom += halfsize;

    result = CalmaGenerateArray((FILE *)csi->csi_client, csi->csi_type,
		left, bottom, pitch, nAcross, nUp);

    return (result == TRUE) ? 0 : 1;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifSlotFunc --
 *
 * 	Called for each relevant tile when chopping areas up into
 *	slots (rectangles).  Chops each tile up into slots and paints
 *	into cifPlane.
 *
 * Results:
 *	Always return 0
 *
 * Side effects:
 *	Stuff is painted into cifPlane.
 *
 * ----------------------------------------------------------------------------
 */

int
cifSlotFunc(tile, op)
    Tile *tile;			/* Tile to be diced up. */
    CIFOp *op;			/* Describes how to generate squares. */
{
    Rect area, slot;
    int i, nAcross, j, nUp, width, height, xpitch, ypitch, left;
    int *axtop, *axbot, *aytop, *aybot;
    int *sxtop, *sxbot, *sytop, *sybot;
    TileType type;
    SlotsData *slots = (SlotsData *)op->co_client;

#ifdef NONMANHATTAN
    /* For now, don't allow squares on non-manhattan tiles */
    if (IsSplit(tile))
	return 0;
#endif

    TiToRect(tile, &area);
    width = area.r_xtop - area.r_xbot;
    height = area.r_ytop - area.r_ybot;

    /* Orient to the short/long orientation of the tile */

    /* Assume a vertical tile;  if not, reorient area and remember */
    if (width > height)
    {
	axbot = &area.r_ybot;
	aybot = &area.r_xbot;
	axtop = &area.r_ytop;
	aytop = &area.r_xtop;
	sxbot = &slot.r_ybot;
	sybot = &slot.r_xbot;
	sxtop = &slot.r_ytop;
	sytop = &slot.r_xtop;
	width = *axtop - *axbot;
	height = *aytop - *aybot;
    }
    else
    {
	axbot = &area.r_xbot;
	aybot = &area.r_ybot;
	axtop = &area.r_xtop;
	aytop = &area.r_ytop;
	sxbot = &slot.r_xbot;
	sybot = &slot.r_ybot;
	sxtop = &slot.r_xtop;
	sytop = &slot.r_ytop;
    }

    xpitch = slots->sl_ssize + slots->sl_ssep;

calcX:
    nAcross = (width + slots->sl_ssep - (2 * slots->sl_sborder)) / xpitch;
    if (nAcross == 0)
    {
	left = (*axbot + *axtop - slots->sl_ssize) / 2;
	if (left >= *axbot) nAcross = 1;
    }
    else
	left = (*axbot + *axtop + slots->sl_ssep - (nAcross * xpitch)) / 2;

    /* Check that we are not violating any gridlimit */

    if (CIFCurStyle && (CIFCurStyle->cs_gridLimit > 0))
    {
	int delta = abs(left) % CIFCurStyle->cs_gridLimit;
	if (delta > 0)
	{
	    *axtop -= 2 * delta;
	    goto calcX;
	}
    }


    if (slots->sl_lsize > 0)
    {
	ypitch = slots->sl_lsize + slots->sl_lsep;
calcY:
	nUp = (*aytop - *aybot + slots->sl_lsep - (2 * slots->sl_lborder)) / ypitch;
	if (nUp == 0)
	{
	    *sybot = (*aybot + *aytop - slots->sl_lsize) / 2;
	    if (*sybot >= *aybot) nUp = 1;
	}
	else
	    *sybot = (*aybot + *aytop + slots->sl_lsep - (nUp * ypitch)) / 2;
	*sytop = *sybot + slots->sl_lsize;

	/* Check that we are not violating any gridlimit */

	if (CIFCurStyle && (CIFCurStyle->cs_gridLimit > 0))
	{
	    int delta = abs(*sybot) % CIFCurStyle->cs_gridLimit;
	    if (delta > 0)
	    {
		*aytop -= 2 * delta;
		goto calcY;
	    }
	}
    }
    else
    {
	nUp = 1;
	*sybot = *aybot + slots->sl_lborder;
	*sytop = *aytop - slots->sl_lborder;
	/* There's no space to fit a slot */
	if (*sytop - *sybot <= 0)
	    return 0;
    }

    /* To be done---slot offsets */

    for (i = 0; i < nUp; i++)
    {
	*sxbot = left;
	for (j = 0; j < nAcross; j++)
	{
	    *sxtop = *sxbot + slots->sl_ssize;

	    DBPaintPlane(cifPlane, &slot, CIFPaintTable,
			(PaintUndoInfo *) NULL);
	    CIFTileOps += 1;
	    *sxbot += xpitch;
	}
	*sybot += ypitch;
	*sytop += ypitch;
    }
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifSquareFunc --
 *
 * 	Called for each relevant tile when chopping areas up into squares.
 *	Chops each tile up into squares and paints into cifPlane.
 *
 * Results:
 *	Always return 0
 *
 * Side effects:
 *	Stuff is painted into cifPlane.
 *
 * ----------------------------------------------------------------------------
 */

int
cifSquareFunc(tile, op)
    Tile *tile;			/* Tile to be diced up. */
    CIFOp *op;			/* Describes how to generate squares. */
{
    Rect area, square;
    int i, nAcross, j, nUp, left, pitch;
    TileType type;
    SquaresData *squares = (SquaresData *)op->co_client;

    TiToRect(tile, &area);
    pitch = squares->sq_size + squares->sq_sep;

    /* Compute the real border to leave around the sides.  If only
     * one square will fit in a particular direction, center it
     * regardless of the requested border size.  If more than one
     * square will fit, then fit it in extras only if at least the
     * requested border size can be left.  Also center things in the
     * rectangle, so that the box's exact size doesn't matter. This
     * trickiness is done so that (a) coincident contacts from overlapping
     * cells always have their squares line up, regardless of the
     * orientation of the cells, and (b) we can generate contact vias
     * for non-square contact areas, for example poly-metal contacts
     * in the MOSIS CMOS process.
     */

    nAcross = (area.r_xtop - area.r_xbot + squares->sq_sep
	    - (2 * squares->sq_border)) / pitch;
    if (nAcross == 0)
    {
	left = (area.r_xbot + area.r_xtop - squares->sq_size) / 2;
	if (left >= area.r_xbot) nAcross = 1;
    }
    else
    {
	left = (area.r_xbot + area.r_xtop + squares->sq_sep
		- (nAcross * pitch)) / 2;
    }

    nUp = (area.r_ytop - area.r_ybot + squares->sq_sep
	    - (2 * squares->sq_border)) / pitch;
    if (nUp == 0)
    {
	square.r_ybot = (area.r_ybot + area.r_ytop - squares->sq_size) / 2;
	if (square.r_ybot >= area.r_ybot) nUp = 1;
    }
    else
    {
	square.r_ybot = (area.r_ybot + area.r_ytop + squares->sq_sep
		- (nUp * pitch)) / 2;
    }

    for (i = 0; i < nUp; i += 1)
    {
	square.r_ytop = square.r_ybot + squares->sq_size;
	square.r_xbot = left;
	for (j = 0; j < nAcross; j += 1)
	{
	    square.r_xtop = square.r_xbot + squares->sq_size;

	    /* Check if the entire cut falls within the triangle area;	*/
	    /* only output cuts that satisfy this condition.		*/
	    /* To-do:  Need to make sure that cut is at least sq_border	*/
	    /* distance to the diagonal edge of the triangle.		*/

	    if (IsSplit(tile))
		if (!inside_triangle(&square, tile))
		    goto aftersq;

	    DBPaintPlane(cifPlane, &square, CIFPaintTable,
			(PaintUndoInfo *) NULL);
	    CIFTileOps += 1;
aftersq:
	    square.r_xbot += pitch;
	}
	square.r_ybot += pitch;
    }
    return 0;
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifSquareGridFunc --
 *
 * 	Called for each relevant tile when chopping areas up into squares.
 *	Chops each tile up into squares and paints into cifPlane. The
 *      generated squares are on the grid (this code comes from UNC).
 *
 * Results:
 *	Always return 0
 *
 * Side effects:
 *	Stuff is painted into cifPlane.
 *
 * ----------------------------------------------------------------------------
 */

int
cifSquareGridFunc(tile, op)
    Tile *tile;			/* Tile to be diced up. */
    CIFOp *op;			/* Describes how to generate squares. */
{
    Rect area, square;
    int i, nAcross, j, nUp, left, bottom, right, top, pitch;
    int gridx, gridy, margin;
    SquaresData *squares = (SquaresData *)op->co_client;

#ifdef NONMANHATTAN
    /* For now, don't allow squares on non-manhattan tiles */
    if (IsSplit(tile))
	return 0;
#endif

    TiToRect(tile, &area);

    /*
     * It may be necessary to generate contacts on a specific grid; e.g.,
     * to avoid placing contacts at half-lambda.  If this is the case,
     * then the DRC section of the techfile should specify "no overlap"
     * for these types.
     *
     * This routine has been simplified.  It flags a warning when there
     * is not enough space for a contact cut, rather than attempting to
     * draw a cut without enough border area.  The routine has also been
     * extended to allow different x and y grids to be specified.  This
     * allows certain tricks, such as generating offset contact grids on
     * a pad, as required by some processes.
     */

    pitch = squares->sq_size + squares->sq_sep;
    gridx = squares->sq_gridx;
    gridy = squares->sq_gridy;

    area.r_xtop -= squares->sq_border;
    area.r_ytop -= squares->sq_border;
    area.r_xbot += squares->sq_border;
    area.r_ybot += squares->sq_border;

    left = area.r_xbot / gridx;
    left *= gridx;
    if (left < area.r_xbot) left += gridx;

    bottom = area.r_ybot / gridy;
    bottom *= gridy;
    if (bottom < area.r_ybot) bottom += gridy;

    nAcross = (area.r_xtop - left + squares->sq_sep) / pitch;
    if (nAcross == 0)
    {
	CIFError(&area, "contact area too small to make cuts!");
	return 0;
    }
    
    nUp = (area.r_ytop - bottom + squares->sq_sep) / pitch;
    if (nUp == 0)
    {
	CIFError(&area, "contact area too small to make cuts!");
	return 0;
    }

    /* Center the contacts while remaining on-grid */
    right = left + nAcross * squares->sq_size + (nAcross - 1) * squares->sq_sep;
    top = bottom + nUp * squares->sq_size + (nUp - 1) * squares->sq_sep;
    margin = ((area.r_xtop - right) - (left - area.r_xbot)) / (gridx * 2);
    area.r_xbot = left + margin * gridx;
    margin = ((area.r_ytop - top) - (bottom - area.r_ybot)) / (gridy * 2);
    area.r_ybot = bottom + margin * gridy;

    square.r_ybot = area.r_ybot;
    for (i = 0; i < nUp; i++)
    {
	square.r_ytop = square.r_ybot + squares->sq_size;
	square.r_xbot = area.r_xbot;
	for (j = 0; j < nAcross; j++)
	{
	    square.r_xtop = square.r_xbot + squares->sq_size;
	    DBPaintPlane(cifPlane, &square, CIFPaintTable,
			(PaintUndoInfo *) NULL);
	    CIFTileOps++;
	    square.r_xbot += pitch;
	}
	square.r_ybot += pitch;
    }
    return 0;
}


/*
 * ----------------------------------------------------------------------------
 *
 * cifSrTiles --
 *
 * 	This is a utility procedure that just calls DBSrPaintArea
 *	one or more times for the planes being used in processing
 *	one CIFOp.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	This procedure itself has no side effects.  For each of the
 *	paint or temporary planes indicated in cifOp, we call
 *	DBSrPaintArea to find the desired tiles in the desired
 *	area for the operation.  DBSrPaintArea is given func as a
 *	search function, and cdArg as ClientData.
 *
 * ----------------------------------------------------------------------------
 */

void
cifSrTiles(cifOp, area, cellDef, temps, func, cdArg)
    CIFOp *cifOp;		/* Geometric operation being processed. */
    Rect *area;			/* Area of Magic paint to consider. */
    CellDef *cellDef;		/* CellDef to search for paint. */
    Plane *temps[];		/* Planes to use for temporaries. */
    int (*func)();		/* Search function to pass to DBSrPaintArea. */
    ClientData cdArg;		/* Client data for func. */
{
    TileTypeBitMask maskBits;
    TileType t;
    int i;
    BloatData *bloats;

    /* When reading data from a cell, it must first be scaled to
     * CIF units.  Check for CIFCurStyle, as we don't want to
     * crash while reading CIF/GDS just becuase the techfile
     * "cifoutput" section was blank.
     */

    cifScale = (CIFCurStyle) ? CIFCurStyle->cs_scaleFactor : 1;

    /* Bloat operations have to be in a single plane */

    switch (cifOp->co_opcode) {
	case CIFOP_BLOAT:
	case CIFOP_BLOATMIN:
	case CIFOP_BLOATMAX:
	    bloats = (BloatData *)cifOp->co_client;
	    i = bloats->bl_plane;
	    maskBits = DBPlaneTypes[i];
	    TTMaskAndMask(&maskBits, &cifOp->co_paintMask);
	    if (!TTMaskEqual(&maskBits, &DBZeroTypeBits))
		DBSrPaintArea((Tile *) NULL, cellDef->cd_planes[i],
				area, &cifOp->co_paintMask, func, cdArg);
	    break;

	default:
	    for (i = PL_DRC_CHECK; i < DBNumPlanes; i++)
	    {
		maskBits = DBPlaneTypes[i];
		TTMaskAndMask(&maskBits, &cifOp->co_paintMask);
		if (!TTMaskEqual(&maskBits, &DBZeroTypeBits))
		    (void) DBSrPaintArea((Tile *) NULL, cellDef->cd_planes[i],
			area, &cifOp->co_paintMask, func, cdArg);
	    }
	    break;
    }

    /* When processing CIF data, use everything in the plane. */

    cifScale = 1;
    for (t = 0; t < TT_MAXTYPES; t++, temps++)
	if (TTMaskHasType(&cifOp->co_cifMask, t))
	    (void) DBSrPaintArea((Tile *) NULL, *temps, &TiPlaneRect,
		&CIFSolidBits, func, (ClientData) cdArg);
}

/*
 * ----------------------------------------------------------------------------
 *
 * CIFGenLayer --
 *
 *	This routine will generate one CIF layer.
 *	It provides the core of the CIF generator.
 *
 * Results:
 *	Returns a malloc'ed plane with tiles of type CIF_SOLIDTYPE
 *	marking the area of this CIF layer as built up by op.
 *
 * Side effects:
 *	None, except to create a new plane holding the CIF for the layer.
 *	The CIF that's generated may fall outside of area... it's what
 *	results from considering everything in area.  In most cases the
 *	caller will clip the results down to the desired area.
 *
 * ----------------------------------------------------------------------------
 */

Plane *
CIFGenLayer(op, area, cellDef, temps, clientdata)
    CIFOp *op;			/* List of CIFOps telling how to make layer. */
    Rect *area;			/* Area to consider when generating CIF.  Only
				 * material in this area will be considered, so
				 * the caller should usually expand his desired
				 * area by one CIF radius.
				 */
    CellDef *cellDef;		/* CellDef to search when paint layers are
				 * needed for operation.
				 */
    Plane *temps[];		/* Temporary layers to be used when needed
				 * for operation.
				 */
    ClientData clientdata;	/*
				 * Data that may be passed to the CIF operation
				 * function.
				 */
{
    Plane *temp;
    static Plane *nextPlane, *curPlane;
    Rect bbox;
    CIFOp *tempOp;
    CIFSquaresInfo csi;
    int (*cifGrowFuncPtr)() = (CIFCurStyle->cs_flags & CWF_GROW_EUCLIDEAN) ?
		cifGrowEuclideanFunc : cifGrowFunc;

    /* Set up temporary planes used during computation.  One of these
     * will be returned as the result (whichever is curPlane at the
     * end of the computation).  The other is saved for later use.
     */

    if (nextPlane == NULL)
	nextPlane = DBNewPlane((ClientData) TT_SPACE);
    curPlane = DBNewPlane((ClientData) TT_SPACE);

    /* Go through the geometric operations and process them one
     * at a time.
     */

    for ( ; op != NULL; op = op->co_next)
    {
	switch (op->co_opcode)
	{
	    /* For AND, first collect all the stuff to be anded with
	     * plane in a temporary plane.  Then find all the places
	     * where there isn't any stuff, and erase from the
	     * current plane.
	     */

	    case CIFOP_AND:
		DBClearPaintPlane(nextPlane);
		cifPlane = nextPlane;
		cifSrTiles(op, area, cellDef, temps, cifPaintFunc,
		    (ClientData) CIFPaintTable);
		cifPlane = curPlane;
		cifScale = 1;
		(void) DBSrPaintArea((Tile *) NULL, nextPlane, &TiPlaneRect,
		    &DBSpaceBits, cifPaintFunc,
		    (ClientData) CIFEraseTable);
		break;
	    
	    /* For OR, just use cifPaintFunc to OR the areas of all
	     * relevant tiles into plane.  HOWEVER, if the co_client
	     * record is non-NULL and CalmaContactArrays is TRUE,
	     * then for each contact type, we do the paint function
	     * separately, then call the contact array generation
	     * procedure.
	     */

	    case CIFOP_OR:
		cifPlane = curPlane;
		cifScale = (CIFCurStyle) ? CIFCurStyle->cs_scaleFactor : 1;

		if ((op->co_client != (ClientData)NULL)
				&& (CalmaContactArrays == TRUE))
		{
		    TileTypeBitMask paintMask, errMask;
		    TileType i, j;

		    TTMaskZero(&errMask);
		    TTMaskZero(&paintMask);
 		    TTMaskSetMask(&paintMask, &op->co_paintMask);
		    for (i = TT_TECHDEPBASE; i < DBNumUserLayers; i++)
		    {
			if (TTMaskHasType(&paintMask, i))
			{
			    TTMaskSetOnlyType(&op->co_paintMask, i);
			    for (j = DBNumUserLayers; j < DBNumTypes; j++)
				if (TTMaskHasType(DBResidueMask(j), i))
				    TTMaskSetType(&op->co_paintMask, j);

			    cifSrTiles(op, area, cellDef, temps, cifPaintFunc,
			    		(ClientData) CIFPaintTable);

			    csi.csi_squares = (SquaresData *)op->co_client;
			    csi.csi_type = i;
			    csi.csi_client = clientdata;

			    if (DBSrPaintArea((Tile *) NULL, curPlane,
					&TiPlaneRect, &CIFSolidBits,
					cifContactFunc,	(ClientData) &csi))
			    {
				/* Failure of DBSrPaintArea() (returns 1)
				 * indicates that a contact cell type
				 * could not be found for magic layer i.
				 * Record the error for subsequent handling.
				 */
				TTMaskSetType(&errMask, i);
			    }
			    DBClearPaintPlane(curPlane);
			}
		    }
		    if (!TTMaskIsZero(&errMask))
		    {
			/*
			 * Handle layers for which a contact cell could
			 * not be found in the default manner.
			 */
			TTMaskZero(&op->co_paintMask);
 			TTMaskSetMask(&op->co_paintMask, &errMask);
			cifSrTiles(op, area, cellDef, temps, cifPaintFunc,
		    		(ClientData) CIFPaintTable);
		    }

		    /* Recover the original magic layer mask for the cifop */

		    TTMaskZero(&op->co_paintMask);
 		    TTMaskSetMask(&op->co_paintMask, &paintMask);
		}
		else
		{
		    cifSrTiles(op, area, cellDef, temps, cifPaintFunc,
		    	(ClientData) CIFPaintTable);
		}
		break;
	    
	    /* For ANDNOT, do exactly the same thing as OR, except erase
	     * instead of paint.
	     */

	    case CIFOP_ANDNOT:
		cifPlane = curPlane;
		cifSrTiles(op, area, cellDef, temps, cifPaintFunc,
		    (ClientData) CIFEraseTable);
		break;
	    
	    /* For GROW, just find all solid tiles in the current plane,
	     * and paint a larger version into a new plane.  The switch
	     * the current and new planes.
	     */
	
	    case CIFOP_GROW:
		growDistance = op->co_distance;
		DBClearPaintPlane(nextPlane);
		cifPlane = nextPlane;
		cifScale = 1;
		(void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
		    &CIFSolidBits, *cifGrowFuncPtr, (ClientData) CIFPaintTable);
		temp = curPlane;
		curPlane = nextPlane;
		nextPlane = temp;
		break;

	    /* GROW_G grows non-uniformly to the indicated grid. */

	    case CIFOP_GROW_G:
		growDistance = op->co_distance;
		DBClearPaintPlane(nextPlane);
		cifPlane = nextPlane;
		cifScale = 1;
		(void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
		    &CIFSolidBits, cifGrowGridFunc, (ClientData) CIFPaintTable);
		temp = curPlane;
		curPlane = nextPlane;
		nextPlane = temp;
		break;
	    
	    /* SHRINK is just like grow except work from the space tiles. */

	    case CIFOP_SHRINK:
		growDistance = op->co_distance;
		DBClearPaintPlane(nextPlane);
		DBPaintPlane(nextPlane, &TiPlaneRect, CIFPaintTable,
		    (PaintUndoInfo *) NULL);
		cifPlane = nextPlane;
		cifScale = 1;
		(void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
		    &DBSpaceBits, *cifGrowFuncPtr, (ClientData) CIFEraseTable);
		temp = curPlane;
		curPlane = nextPlane;
		nextPlane = temp;
		break;
	    
	    case CIFOP_BLOAT:
		cifPlane = curPlane;
		cifSrTiles(op, area, cellDef, temps,
		    cifBloatFunc, op->co_client);
		break;
	    
	    case CIFOP_BLOATMAX:
	    case CIFOP_BLOATMIN:
		cifPlane = curPlane;
		cifSrTiles(op, area, cellDef, temps,
		    cifBloatMaxFunc, (ClientData) op);
		break;

	    case CIFOP_BLOATALL:
		cifPlane = curPlane;
		cifSrTiles(op, area, cellDef, temps,
		    cifBloatAllFunc, (ClientData) op);
		break;
	    
	    case CIFOP_SQUARES:
		if (CalmaContactArrays == FALSE)
		{
		    DBClearPaintPlane(nextPlane);
		    cifPlane = nextPlane;
		    (void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
			&CIFSolidBits, cifSquareFunc, (ClientData) op);
		    temp = curPlane;
		    curPlane = nextPlane;
		    nextPlane = temp;
		}
		break;

	    case CIFOP_SQUARES_G:
		DBClearPaintPlane(nextPlane);
		cifPlane = nextPlane;
		(void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
		    &CIFSolidBits, cifSquareGridFunc, (ClientData) op);
		temp = curPlane;
		curPlane = nextPlane;
		nextPlane = temp;
		break;

	    case CIFOP_SLOTS:
		DBClearPaintPlane(nextPlane);
		cifPlane = nextPlane;
		(void) DBSrPaintArea((Tile *) NULL, curPlane, &TiPlaneRect,
		    &CIFSolidBits, cifSlotFunc, (ClientData) op);
		temp = curPlane;
		curPlane = nextPlane;
		nextPlane = temp;
		break;

	    case CIFOP_BBOX:
		if (CIFErrorDef == NULL) break;

		/* co_client contains the flag (1) for top-level only */
		if ((int)op->co_client == 1)
		{
		    /* Only generate output for the top-level cell */
		    int found = 0;
		    CellUse *celluse;
		    for (celluse = CIFErrorDef->cd_parents;
				celluse != (CellUse *) NULL;
				celluse = celluse->cu_nextuse)
		    {
			if (celluse->cu_parent != (CellDef *) NULL)
			    if ((celluse->cu_parent->cd_flags & CDINTERNAL)
					!= CDINTERNAL)
			    {
			 	found = 1;
				break;
			    }
		    }
		    if (found != 0) break;
		}
		cifPlane = curPlane;
		bbox = CIFErrorDef->cd_bbox;
		cifScale = (CIFCurStyle) ? CIFCurStyle->cs_scaleFactor : 1;
		bbox.r_xbot *= cifScale;
		bbox.r_xtop *= cifScale;
		bbox.r_ybot *= cifScale;
		bbox.r_ytop *= cifScale;
#ifdef NONMANHATTAN
		DBNMPaintPlane(curPlane, CIF_SOLIDTYPE, &bbox,
			CIFPaintTable, (PaintUndoInfo *)NULL); 
#else
		DBPaintPlane(curPlane, &bbox, CIFPaintTable,
			(PaintUndoInfo *)NULL); 
#endif
		break;

	    default:
		continue;
	}
    }

    return curPlane;
}

/*
 * ----------------------------------------------------------------------------
 *
 * CIFGen --
 *
 * 	This procedure generates a complete set of CIF layers for
 *	a particular area of a particular cell.  NOTE: if the argument
 *	genAllPlanes is FALSE, only planes for those layers having
 *	a bit set in 'layers' are generated; the others are set
 *	to NULL.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The parameters realPlanes and tempPlanes are modified
 *	to hold the CIF and temporary layers for area of cellDef,
 *	as determined by the current CIF generation rules.
 *
 * ----------------------------------------------------------------------------
 */

void
CIFGen(cellDef, area, planes, layers, replace, genAllPlanes, clientdata)
    CellDef *cellDef;		/* Cell for which CIF is to be generated. */
    Rect *area;			/* Any CIF overlapping this area (in coords
				 * of cellDef) will be generated.  The CIF
				 * will be clipped to this area.
				 */
    Plane **planes;		/* Pointer to array of pointers to planes
				 * to hold "real" CIF layers that are
				 * generated.  Pointers may initially be
				 * NULL.
				 */
    TileTypeBitMask *layers;	/* CIF layers to generate. */
    bool replace;		/* TRUE means that the new CIF is to replace
				 * anything that was previously in planes.
				 * FALSE means that the new CIF is to be
				 * OR'ed in with the current contents of
				 * planes.
				 */
    bool genAllPlanes;		/* If TRUE, generate a tile plane even for
				 * those layers not specified as being
				 * generated in the 'layers' mask above.
				 */
    ClientData clientdata;	/* Data that may be passed along to the
				 * CIF operation functions.
				 */
{
    int i;
    Plane *new[MAXCIFLAYERS];
    Rect expanded, clip;

    /*
     * Generate the area in magic coordinates to search, and the area in
     * cif coordinates to use in clipping the results of CIFGenLayer().
     */
    cifGenClip(area, &expanded, &clip);

    /*
     * Generate all of the new layers in a temporary place.
     * If a layer isn't being generated, leave new[i] set to
     * NULL to indicate this fact.
     */
    for (i = 0; i < CIFCurStyle->cs_nLayers; i++)
    {
	if (TTMaskHasType(layers,i))
	{
	    CIFErrorLayer = i;
	    new[i] = CIFGenLayer(CIFCurStyle->cs_layers[i]->cl_ops,
		    &expanded, cellDef, new, clientdata);
	}
	else if (genAllPlanes) new[i] = DBNewPlane((ClientData) TT_SPACE);
	else new[i] = (Plane *) NULL;
    }

    /*
     * Now mask off all the unwanted material in the new layers, and
     * either OR them into the existing layers or replace the existing
     * material with them.
     */
    for (i = 0; i < CIFCurStyle->cs_nLayers; i += 1)
    {
	if (new[i])
	    cifClipPlane(new[i], &clip);

	if (replace)
	{
	    if (planes[i])
	    {
		DBFreePaintPlane(planes[i]);
		TiFreePlane(planes[i]);
	    }
	    planes[i] = new[i];
	    continue;
	}

	if (planes[i])
	{
	    if (new[i])
	    {
		cifPlane = planes[i];
		cifScale = 1;
		(void) DBSrPaintArea((Tile *) NULL, new[i], &TiPlaneRect,
			&CIFSolidBits, cifPaintFunc,
			(ClientData) CIFPaintTable);
		DBFreePaintPlane(new[i]);
		TiFreePlane(new[i]);
	    }
	}
	else planes[i] = new[i];
    }
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifClipPlane --
 *
 * Erase the portions of the plane 'plane' that lie outside of 'clip'.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	See above.
 *
 * ----------------------------------------------------------------------------
 */

void
cifClipPlane(plane, clip)
    Plane *plane;
    Rect *clip;
{
    Rect r;

    if (clip->r_xtop < TiPlaneRect.r_xtop)
    {
	r = TiPlaneRect;
	r.r_xbot = clip->r_xtop;
	DBPaintPlane(plane, &r, CIFEraseTable, (PaintUndoInfo *) NULL);
    }
    if (clip->r_ytop < TiPlaneRect.r_ytop)
    {
	r = TiPlaneRect;
	r.r_ybot = clip->r_ytop;
	DBPaintPlane(plane, &r, CIFEraseTable, (PaintUndoInfo *) NULL);
    }
    if (clip->r_xbot > TiPlaneRect.r_xbot)
    {
	r = TiPlaneRect;
	r.r_xtop = clip->r_xbot;
	DBPaintPlane(plane, &r, CIFEraseTable, (PaintUndoInfo *) NULL);
    }
    if (clip->r_ybot > TiPlaneRect.r_ybot)
    {
	r = TiPlaneRect;
	r.r_ytop = clip->r_ybot;
	DBPaintPlane(plane, &r, CIFEraseTable, (PaintUndoInfo *) NULL);
    }
}

/*
 * ----------------------------------------------------------------------------
 *
 * cifGenClip --
 *
 * Compute two new areas from the original area:  one ('expanded')
 * is expanded by a CIF halo and is used to determine how much of
 * the database to search to find what's relevant for CIF generation;
 * the other ('clip') is the CIF equivalent of area and is used to
 * clip the resulting CIF.  This code is tricky because area may run
 * off to infinity, and we have to be careful not to expand past infinity.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets *expanded and *clip.
 *
 * ----------------------------------------------------------------------------
 */

void
cifGenClip(area, expanded, clip)
    Rect *area;			/* Any CIF overlapping this area (in coords
				 * of cellDef) will be generated.  The CIF
				 * will be clipped to this area.
				 */
    Rect *expanded, *clip;
{
    if (area->r_xbot > TiPlaneRect.r_xbot)
    {
	clip->r_xbot = area->r_xbot * CIFCurStyle->cs_scaleFactor;
	expanded->r_xbot = area->r_xbot - CIFCurStyle->cs_radius;
    }
    else clip->r_xbot = expanded->r_xbot = area->r_xbot;
    if (area->r_ybot > TiPlaneRect.r_ybot)
    {
	clip->r_ybot = area->r_ybot * CIFCurStyle->cs_scaleFactor;
	expanded->r_ybot = area->r_ybot - CIFCurStyle->cs_radius;
    }
    else clip->r_ybot = expanded->r_ybot = area->r_ybot;
    if (area->r_xtop < TiPlaneRect.r_xtop)
    {
	clip->r_xtop = area->r_xtop * CIFCurStyle->cs_scaleFactor;
	expanded->r_xtop = area->r_xtop + CIFCurStyle->cs_radius;
    }
    else clip->r_xtop = expanded->r_xtop = area->r_xtop;
    if (area->r_ytop < TiPlaneRect.r_ytop)
    {
	clip->r_ytop = area->r_ytop * CIFCurStyle->cs_scaleFactor;
	expanded->r_ytop = area->r_ytop + CIFCurStyle->cs_radius;
    }
    else clip->r_ytop = expanded->r_ytop = area->r_ytop;
}

/*
 * ----------------------------------------------------------------------------
 *
 * CIFClearPlanes --
 *
 * 	This procedure clears out a collection of CIF planes.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Each of the planes in "planes" is re-initialized to point to
 *	an empty paint plane.
 *
 * ----------------------------------------------------------------------------
 */

void
CIFClearPlanes(planes)
    Plane **planes;		/* Pointer to an array of MAXCIFLAYERS
				 * planes.
				 */
{
    int i;

    for (i = 0; i < MAXCIFLAYERS; i++)
    {
	if (planes[i] == NULL)
	{
	    planes[i] = DBNewPlane((ClientData) TT_SPACE);
	}
	else
	{
	    DBClearPaintPlane(planes[i]);
	}
    }
}
