/*

  Copyright (c) 1992, 1993
        Regents of the University of California
  All rights reserved.

  Use and copying of this software and preparation of derivative works
  based upon this software are permitted.  However, any distribution of
  this software or derivative works must include the above copyright 
  notice.

  This software is made available AS IS, and neither the Electronics
  Research Laboratory or the Universify of California make any
  warranty about the software, its performance or its conformity to
  any specification.

  Author: Szu-Tsung Cheng, stcheng@ic.Berkeley.EDU
          10/92
          10/93

  $Header: /projects/development/hsv/CVSRepository/vl2mv/src/parser/vl_copy.c,v 1.3 2003/11/28 17:06:59 fabio Exp $


*/


#include "util.h"
#include "st.h"
#include "list.h"
#include "array.h"
#include "set.h"
#include "stack.h"
#include "vl_types.h"
#include "vl_defs.h"
#include "vlr_int.h"
#include "vl_create.h"
#include "vl_copy.h"
#include "vl_traverse.h"
#include "dataflow.h"
#include "verilog.h"

extern vl_desc *mod_list;	
static int    CpyTrace=0;      
static vl_module *creatingModule;  


vl_module *vl_copy_rename_module(vl_module *mod, char *new_name)
{
    vl_module *retval;
    vl_id_range *new_id;

    if (mod == NULL) return NIL(vl_module);

    ASSERT(mod->type==ModDecl, "Expecting ModDecl");

    CpyTRACE("copying module\n");
    new_id = vl_copy_id_range(mod->name);
    vl_chk_free(new_id->name);
    new_id->name = vlStrdup(new_name);
    retval = vl_create_module(new_id, vl_copy_ports(mod->ports), (lsList)0);
    creatingModule = retval;
    retval->mod_items = vl_copy_mod_item_list(mod->mod_items);
    retval->type = mod->type;

    return retval;
}

vl_module *vl_copy_module(vl_module *mod)
{
    vl_module *retval;

    if (mod == NULL) return NIL(vl_module);

    ASSERT(mod->type==ModDecl, "Expecting ModDecl");

    CpyTRACE("copying module\n");
    retval = vl_create_module(vl_copy_id_range(mod->name),
			      vl_copy_ports(mod->ports), (lsList)0);
    creatingModule = retval;
    retval->mod_items = vl_copy_mod_item_list(mod->mod_items);
    retval->type = mod->type;

    return retval;
}


vl_primitive *vl_copy_primitive(vl_primitive *prim)
{
    vl_primitive *retval;
    
    if (prim == NULL) return NIL(vl_primitive);

    ASSERT((prim->type==CombPrimDecl || prim->type==SeqPrimDecl), 
	   "Expecting PrimDecl");

    CpyTRACE("copying primitive\n");
    retval = vl_create_primitive(vl_copy_id_range(prim->name),
				 vl_copy_ports(prim->ports),
				 vl_copy_mod_item_list(prim->decls),
				 vl_copy_prim_table(prim->type,prim->entries));
    retval->type = prim->type;

    return retval;
}


lsList vl_copy_prim_table(short type, lsList entries)
{
    vl_prim_entry *e, *entry;
    int i;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    retval = lsCreate();

    if (entries == NULL) return retval;

    CpyTRACE("copying primitive table\n");
    for(gen = lsStart(entries); lsNext(gen, (lsGeneric*)&e, &handle) 
	                            != LS_NOMORE; ) {
	entry = (vl_prim_entry*)chk_malloc(sizeof(vl_prim_entry));
	for (i=0; i<10; i++) {
	    if (type) type = type; 
	    entry->inputs[i] = e->inputs[i];
	}
	entry->state = e->state;
	entry->next_state = e->next_state;
	lsNewEnd(retval, (lsGeneric)entry, 0);
    }
    (void) lsFinish(gen);

    return retval;
}


vl_decl *vl_copy_basicdecl(vl_decl *decl)
{
    vl_decl *retval;

    CpyTRACE("copying Basic decl\n");

    retval = (vl_decl*)chk_malloc(sizeof(vl_decl));
    retval->type = decl->type;
    switch(decl->type) {
    case RealDecl:
	retval->ids = vl_copy_id_list(decl->ids);
	break;
    case EventDecl:
	retval->ids = vl_copy_id_list(decl->ids);
	break;
    case IntDecl:
	retval->ids = vl_copy_id_range_list(decl->ids);
	break;
    case TimeDecl:
	retval->ids = vl_copy_id_range_list(decl->ids);
	break;
    default:
	internal_error("Unexpected Basic Decl type");	break;
    }

    return retval;
}


vl_rangedecl *vl_copy_rangedecl(vl_rangedecl *decl)
{
    vl_rangedecl *retval;
    vl_module *orig_module;

    CpyTRACE("copying Range decl\n");

    switch(decl->type) {
    case InputDecl:  break;
    case OutputDecl: break;
    case InoutDecl:  break;
    case RegDecl:    break;
    default: internal_error("Unexpected Range Decl");	break;
    }

    
    orig_module = vl_currentModule;
    vl_currentModule = creatingModule;
    retval = vl_create_rangedecl(NIL(vl_type), decl->type, 
				 vl_copy_range(decl->range),
				 vl_copy_id_list(decl->ids), 0);
    vl_currentModule = orig_module;
    
    
    
    return retval;
}

vl_paramdecl *vl_copy_paramdecl(vl_paramdecl *decl)
{
    vl_module *orig_module;
    vl_paramdecl *retval, *old_paramdecl;

    CpyTRACE("Printing Parameter decl\n");
    
    retval = (vl_paramdecl*)chk_malloc(sizeof(vl_paramdecl));
    retval->type = decl->type;
    switch(decl->type) {
    case ParamDecl:    break;
    case DefparamDecl: break;
    default: internal_error("Unexpected Param Decl");	break;
    }
    
    {
	vl_bassign_stmt *assign;
	lsHandle handle;
	lsGen gen;

	retval->assigns = lsCreate();
	gen = lsStart(decl->assigns);
	if (lsNext(gen, (lsGeneric*)&assign, &handle) == LS_OK) {
	    lsNewEnd(retval->assigns, 
		     (lsGeneric)vl_copy_bassign_stmt(assign), 0);
	    while(lsNext(gen, (lsGeneric*)&assign, &handle) != LS_NOMORE) {
		lsNewEnd(retval->assigns, 
			 (lsGeneric)vl_copy_bassign_stmt(assign), 0);
	    }
	}
    }

    old_paramdecl = retval;
    orig_module = vl_currentModule;
    vl_currentModule = creatingModule;
    retval = vl_create_paramdecl(retval->type, retval->assigns);
    vl_currentModule = orig_module;
    vl_chk_free((char*)old_paramdecl);

    return retval;
}

vl_netdecl *vl_copy_netdecl(vl_netdecl *decl)
{
    vl_netdecl *retval;

    CpyTRACE("copying Net decl\n");
    retval = (vl_netdecl*)chk_malloc(sizeof(vl_netdecl));
    retval->type = decl->type;
    switch(decl->type) {
    case WireDecl: break;
    case TriDecl:  break;
    case Tri0Decl: break;
    case Tri1Decl: break;
    case Supply0Decl: break;
    case Supply1Decl: break;
    case WandDecl: break;
    case TriandDecl:  break;
    case WorDecl:  break;
    case TriorDecl:   break;
    case TriregDecl:  break;
    default: internal_error("Unexpected Net Decl");	break;
    }
    retval->strength = vl_copy_strength(decl->strength);
    retval->range = vl_copy_range(decl->range);
    retval->delay = vl_copy_delay(decl->delay);
    retval->ids = vl_copy_net_list(decl->ids); 

    return retval;
}

vl_task *vl_copy_task(vl_task *task)
{
    vl_task *retval;

    CpyTRACE("copying Task\n");
    retval = (vl_task*)chk_malloc(sizeof(vl_task));
    ASSERT(task->type == TaskDecl, "Unexpected Task Type");

    retval->type = task->type;
    retval->name  = vl_copy_id_range(task->name);
    retval->decls = vl_copy_decl_list(task->decls);
    retval->stmts = vl_copy_stmt_list(task->stmts);

    return retval;
}

vl_function *vl_copy_function(vl_function *func)
{
    vl_function *retval;

    CpyTRACE("Printing Function\n");
    retval = (vl_function*)chk_malloc(sizeof(vl_function));
    retval->type = func->type;
    switch(func->type) {
    case IntFuncDecl:
	
	break;
    case RealFuncDecl:
	
 	break;
    case RangeFuncDecl:
	
	break;
    default:
	internal_error("Unexpected Function Type");
    }
    retval->name = vl_copy_id_range(func->name);
    retval->decls = vl_copy_decl_list(func->decls);
    retval->stmts = vl_copy_stmt_list(func->stmts);

    return retval;
}

vl_gate_inst_list *vl_copy_gate_inst_list(vl_gate_inst_list *gatelist)
{
    vl_gate_inst_list *retval;

    CpyTRACE("copying Gate Instance List\n");
    retval = (vl_gate_inst_list*)chk_malloc(sizeof(vl_gate_inst_list));
    retval->type = gatelist->type;

    switch(gatelist->type) {
    case AndGate:   break;
    case NandGate:  break;
    case OrGate:    break;
    case NorGate:   break;
    case XorGate:   break;
    case XnorGate:  break;
    case BufGate:   break;
    case Bufif0Gate:break;
    case Bufif1Gate:break;
    case NotGate:   break;
    case Notif0Gate:break;
    case Notif1Gate:break;
    case PulldownGate: break;
    case PullupGate:   break;
    case NmosGate:     break;
    case RnmosGate:    break;
    case PmosGate:     break;
    case RpmosGate:    break;
    case CmosGate:     break;
    case RcmosGate:    break;
    case TranGate:     break;
    case RtranGate:    break;
    case Tranif0Gate:  break;
    case Rtranif0Gate: break;
    case Tranif1Gate:  break;
    case Rtranif1Gate: break;
    default: internal_error("Unexpected Gate Type");	break;
    }

    retval->strength = vl_copy_strength(gatelist->strength);
    retval->delays = vl_copy_delay(gatelist->delays);
    retval->gates = vl_copy_gates(gatelist->gates);
    
    return retval;
}

lsList vl_copy_gates(lsList gates)
{
    vl_gate_inst *gate;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    CpyTRACE("copying Gate Instances\n");

    retval = lsCreate();
    gen = lsStart(gates);
    if (lsNext(gen, (lsGeneric*)&gate, &handle) == LS_OK) {
	lsNewEnd(retval, (lsGeneric)vl_copy_gate(gate), 0);    
	while(lsNext(gen, (lsGeneric*)&gate, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_gate(gate), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}

vl_mod_prim_inst_list *vl_copy_prim_inst_list(vl_mod_prim_inst_list *primlist)
{
    vl_mod_prim_inst_list *retval;

    CpyTRACE("copying Primitive Instance List\n");

    retval = (vl_mod_prim_inst_list*)chk_malloc(sizeof(vl_mod_prim_inst_list));
    retval->type = primlist->type;

    retval->name = vl_copy_id_range(primlist->name);
    retval->strength = vl_copy_strength(primlist->strength);
    retval->delays = vl_copy_delay(primlist->delays);
    retval->mps = vl_copy_prims(primlist->mps);

    return retval;
}

lsList vl_copy_prims(lsList prims)
{
    vl_mod_prim_inst *prim;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    CpyTRACE("copying Primitive Instances\n");
    retval = lsCreate();
    gen = lsStart(prims);
    if (lsNext(gen, (lsGeneric*)&prim, &handle) == LS_OK) {
	lsNewEnd(retval, (lsGeneric)vl_copy_prim(prim), 0);    
	while(lsNext(gen, (lsGeneric*)&prim, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_prim(prim), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}

vl_mod_prim_inst_list *vl_copy_mod_inst_list(vl_mod_prim_inst_list *modinstlist)
{
    vl_mod_prim_inst_list *retval;

    CpyTRACE("copying Module Instance List\n");
    retval = (vl_mod_prim_inst_list*)chk_malloc(sizeof(vl_mod_prim_inst_list));
    retval->type = modinstlist->type;
    retval->name = vl_copy_id_range(modinstlist->name);
    retval->delays = 
	(vl_delay*)vl_copy_param_vals((lsList)modinstlist->delays);
    retval->mps = vl_copy_mod_insts(modinstlist->mps);

    return retval;
}

lsList vl_copy_mod_insts(lsList modinsts)
{
    vl_mod_prim_inst *modinst;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    CpyTRACE("copying Module Instances\n");
    retval = lsCreate();
    gen = lsStart(modinsts);
    if (lsNext(gen, (lsGeneric*)&modinst, &handle) == LS_OK) {
	lsNewEnd(retval, (lsGeneric)vl_copy_modinst(modinst), 0);    
	while(lsNext(gen, (lsGeneric*)&modinst, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_modinst(modinst), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}


vl_procstmt *vl_copy_procstmt(vl_procstmt *pstmt)
{
    vl_procstmt *retval;

    CpyTRACE("copying Process stmt\n");
    
    retval = (vl_procstmt*)chk_malloc(sizeof(vl_procstmt));
    retval->type = pstmt->type;

    switch(pstmt->type) {
    case AlwaysStmt:  break;
    case InitialStmt: break;
    default: internal_error("Unexpected Process Statement Type");	break;
    }

    retval->stmt = vl_copy_stmt(pstmt->stmt);
    retval->pc = vl_create_symbolic_var(vlStrdup(new_pc()));

    return retval;
}

vl_begin_end_stmt *vl_copy_begin_end_stmt(vl_begin_end_stmt *bestmt)
{
    vl_begin_end_stmt *retval;
    
    CpyTRACE("copying beginend\n");
    ASSERT(bestmt->type == BeginEndStmt, "Unexpected BeginEndStmt Type");

    retval = (vl_begin_end_stmt*)chk_malloc(sizeof(vl_begin_end_stmt));
    retval->type = bestmt->type;

    retval->decls = vl_copy_decl_list(bestmt->decls);
    retval->stmts = vl_copy_stmt_list(bestmt->stmts);

    return retval;
}

vl_if_else_stmt *vl_copy_if_else_stmt(vl_if_else_stmt *stmt)
{
    vl_if_else_stmt *retval;

    CpyTRACE("copying if_else_stmt\n");
    ASSERT(stmt->type == IfElseStmt, "Unexpected IfElseStmt Type");

    retval = (vl_if_else_stmt*)chk_malloc(sizeof(vl_if_else_stmt));
    retval->type = stmt->type;

    retval->cond = vl_copy_expr(stmt->cond);
    retval->if_stmt = vl_copy_stmt(stmt->if_stmt);

    if (stmt->else_stmt) {
	retval->else_stmt = vl_copy_stmt(stmt->else_stmt);
    } else {
	retval->else_stmt = NIL(void);
    }

    return retval;
}

vl_case_stmt *vl_copy_case_stmt(vl_case_stmt *stmt)
{
    vl_case_stmt *retval;
    
    CpyTRACE("copying case_stmt\n");

    retval = (vl_case_stmt*)chk_malloc(sizeof(vl_case_stmt));
    retval->type = stmt->type;
    switch(stmt->type) {
    case CaseStmt:  break;
    case CasexStmt: break;
    case CasezStmt: break;
    default: internal_error("Unexpected Case Type");	break;
    }
    retval->cond = vl_copy_expr(stmt->cond);
    
    {
	vl_case_item *item, *case_item;
	lsHandle handle;
	lsGen gen;

	retval->case_items = lsCreate();
	gen = lsStart(stmt->case_items);
	while(lsNext(gen, (lsGeneric*)&item, &handle) != LS_NOMORE) {
	    case_item = vl_copy_case_item(item);
	    case_item->type = item->type;
	    lsNewEnd(retval->case_items, (lsGeneric)case_item, 0);
	}
    }

    return retval;
}

vl_forever_stmt *vl_copy_forever_stmt(vl_forever_stmt *stmt)
{
    vl_forever_stmt *retval;

    CpyTRACE("Printing forever_stmt\n");
    ASSERT(stmt->type == ForeverStmt, "Unexpected ForeverStmt Type");

    retval = (vl_forever_stmt*)chk_malloc(sizeof(vl_forever_stmt));
    retval->type = stmt->type;
    retval->stmt = vl_copy_stmt(stmt->stmt);

    return retval;
}

vl_repeat_stmt *vl_copy_repeat_stmt(vl_repeat_stmt *stmt)
{
    vl_repeat_stmt *retval;
    
    CpyTRACE("copying repeat_stmt\n");
    ASSERT(stmt->type == RepeatStmt, "Unexpected RepeatStmt Type");

    retval = (vl_repeat_stmt*)chk_malloc(sizeof(vl_repeat_stmt));
    retval->type = stmt->type;
    retval->count = vl_copy_expr(stmt->count);
    retval->stmt = vl_copy_stmt(stmt->stmt);

    return retval;
}

vl_while_stmt *vl_copy_while_stmt(vl_while_stmt *stmt)
{
    vl_while_stmt *retval;
    
    CpyTRACE("Printing while_stmt\n");
    ASSERT(stmt->type == WhileStmt, "Unexpected WhileStmt Type");

    retval = (vl_while_stmt*)chk_malloc(sizeof(vl_while_stmt));
    retval->type = stmt->type;
    retval->cond = vl_copy_expr(stmt->cond);
    retval->stmt = vl_copy_stmt(stmt->stmt);
    
    return retval;
}

vl_for_stmt *vl_copy_for_stmt(vl_for_stmt *stmt)
{
    vl_for_stmt *retval;

    CpyTRACE("copying for_stmt\n");
    ASSERT(stmt->type == ForStmt, "Unexpected ForStmt Type");

    retval = (vl_for_stmt*)chk_malloc(sizeof(vl_for_stmt));
    retval->type = stmt->type;
    retval->init = vl_copy_bassign_stmt(stmt->init);
    retval->cond = vl_copy_expr(stmt->cond);
    retval->end = vl_copy_bassign_stmt(stmt->end);
    retval->stmt = vl_copy_stmt(stmt->stmt);

    return retval;
}

vl_delay_control_stmt *vl_copy_delay_control_stmt(vl_delay_control_stmt *stmt)
{
    vl_delay_control_stmt *retval;
    
    CpyTRACE("copying delay_control_stmt\n");
    ASSERT(stmt->type == DelayControlStmt, "Unexpected DelayControlStmt Type");

    retval = (vl_delay_control_stmt*)chk_malloc(sizeof(vl_delay_control_stmt));
    retval->type = stmt->type;
    retval->delay = vl_copy_delay(stmt->delay);
    retval->stmt = vl_copy_stmt(stmt->stmt);
    
    return retval;
}

vl_event_control_stmt *vl_copy_event_control_stmt(vl_event_control_stmt *stmt)
{
    vl_event_control_stmt *retval;

    CpyTRACE("copying event_control_stmt\n");
    ASSERT(stmt->type == EventControlStmt, "Unexpected EventControlStmt Type");

    retval = (vl_event_control_stmt*)chk_malloc(sizeof(vl_event_control_stmt));
    retval->type = stmt->type;
    retval->event = vl_copy_event_expr(stmt->event);
    retval->stmt = vl_copy_stmt(stmt->stmt);

    return retval;
}


vl_bassign_stmt *vl_copy_bassign_stmt(vl_bassign_stmt *bassign)
{
    vl_bassign_stmt *retval;

    CpyTRACE("Printing bassign_stmt\n");
    
    if (bassign->type == AssignStmt) {
	vl_fprintf_stmt(stdout, "assign "); /* Fabio's temp. fix */
    }

    retval = (vl_bassign_stmt*)chk_malloc(sizeof(vl_bassign_stmt));
    retval->type = bassign->type;

    retval->lhs = vl_copy_lval(bassign->lhs);

    bassign->control = NIL(void);
    switch(bassign->type) {
    case AssignStmt:
    case BassignStmt:
	break;
    case NbassignStmt:
	break;
    case DelayBassignStmt:
	retval->control = vl_copy_delay((vl_delay *)bassign->control);
	break;
    case DelayNbassignStmt:
	retval->control = vl_copy_delay((vl_delay *)bassign->control);
	break;
    case EventBassignStmt:
	retval->control = vl_copy_event_expr((vl_event_expr*)bassign->control);
	break;
    case EventNbassignStmt:
	retval->control = vl_copy_event_expr((vl_event_expr*)bassign->control);
	break;
    default: internal_error("Unexpected Assign Type");	break;
    }
    retval->rhs = vl_copy_expr(bassign->rhs);

    return retval;
}

vl_wait_stmt *vl_copy_wait_stmt(vl_wait_stmt *stmt)
{
    vl_wait_stmt *retval;

    CpyTRACE("copying wait_stmt\n");
    ASSERT(stmt->type == WaitStmt, "Unexpected WaitStmt Type");

    retval = (vl_wait_stmt*)chk_malloc(sizeof(vl_wait_stmt));
    retval->type = stmt->type;
    retval->cond = vl_copy_expr(stmt->cond);
    retval->stmt = vl_copy_stmt(stmt->stmt);
    
    return retval;
}

vl_fork_join_stmt *vl_copy_fork_join_stmt(vl_fork_join_stmt *stmt)
{
    vl_fork_join_stmt *retval;

    CpyTRACE("copying fork_join_stmt\n");
    ASSERT(stmt->type == ForkJoinStmt, "Unexpected ForkJoinStmt Type");

    retval = (vl_fork_join_stmt*)chk_malloc(sizeof(vl_fork_join_stmt));
    retval->type = stmt->type;
    retval->decls = vl_copy_decl_list(stmt->decls);
    retval->stmts = vl_copy_stmt_list(stmt->stmts);

    return retval;
}


vl_task_enable_stmt *vl_copy_task_enable_stmt(vl_task_enable_stmt *stmt)
{
    vl_task_enable_stmt *retval;

    CpyTRACE("Printing task_enable_stmt\n");
    ASSERT(((stmt->type == TaskEnableStmt) || 
	    (stmt->type == SysTaskEnableStmt)), 
	   "Unexpected TaskEnableStmt Type");

    retval = (vl_task_enable_stmt*)chk_malloc(sizeof(vl_task_enable_stmt));
    retval->type = stmt->type;
    retval->name = vl_copy_id_range(stmt->name);
    if (stmt->args) {		
	
	retval->args = vl_copy_expr_list(stmt->args); 
    } else {
	retval->args = (lsList)0;
    }

    return retval;
}

vl_disable_stmt *vl_copy_disable_stmt(vl_disable_stmt *stmt)
{
    vl_disable_stmt *retval;

    CpyTRACE("copying disable_stmt\n");
    ASSERT(stmt->type == DisableStmt, "Unexpected DisableStmt Type");

    retval = (vl_disable_stmt*)chk_malloc(sizeof(vl_disable_stmt));
    retval->type = stmt->type;
    retval->name = vl_copy_id_range(stmt->name);

    return retval;
}

vl_deassign_stmt *vl_copy_deassign_stmt(vl_deassign_stmt *stmt)
{
    vl_deassign_stmt *retval;

    CpyTRACE("Printing deassign_stmt\n");
    ASSERT(stmt->type == DeassignStmt, "Unexpected DeassignStmt Type");

    retval = (vl_deassign_stmt*)chk_malloc(sizeof(vl_deassign_stmt));
    retval->type = stmt->type;
    retval->lhs = vl_copy_lval(stmt->lhs);

    return retval;
}

vl_case_item *vl_copy_case_item(vl_case_item *item)
{
    vl_case_item *retval;

    CpyTRACE("Printing case_item\n");
    retval = (vl_case_item*)chk_malloc(sizeof(vl_case_item));
    retval->type = item->type;
    switch(item->type) {
    case CaseItem: retval->exprs = vl_copy_expr_list(item->exprs); break;
    case DefaultItem: break;
    default: internal_error("Unexpected CaseItem Type"); break;
    }
    retval->stmt = vl_copy_stmt(item->stmt);
    
    return retval;
}

vl_event_expr *vl_copy_event_expr(vl_event_expr *event)
{
    vl_event_expr *retval;

    CpyTRACE("Printing event_expr\n");
    if (event == NIL(vl_event_expr)) return NIL(vl_event_expr);

    retval = (vl_event_expr*)chk_malloc(sizeof(vl_event_expr));
    retval->type = event->type;
    retval->list = (lsList)0;
    switch(event->type) {
    case OrEventExpr:  {
	vl_event_expr *e;
	lsHandle handle;
	lsGen gen;

	retval->list = lsCreate();
	gen = lsStart(event->list);
	if (lsNext(gen, (lsGeneric*)&e, &handle) == LS_OK) {
	    lsNewEnd(retval->list, (lsGeneric)vl_copy_event_expr(e), 0);    
	    while(lsNext(gen, (lsGeneric*)&e, &handle) != LS_NOMORE) {
		lsNewEnd(retval->list, (lsGeneric)vl_copy_event_expr(e), 0);
	    }
	}
	(void) lsFinish(gen);

	return retval;
    }
    case NegedgeEventExpr: retval->expr = vl_copy_expr(event->expr); break;
    case PosedgeEventExpr: retval->expr = vl_copy_expr(event->expr); break;
    case EdgeEventExpr: retval->expr = vl_copy_expr(event->expr); break;
    case EventExpr:     retval->expr = vl_copy_expr(event->expr); break;
    default: internal_error("Unexpected EventExpr Type"); break;
    }

    return retval;
}

vl_lval *vl_copy_lval(vl_lval *lval)
{
    vl_lval *retval;

    CpyTRACE("Printing lval\n");

    retval = (vl_lval*)chk_malloc(sizeof(vl_lval));
    retval->type = lval->type;
    switch(lval->type) {
    case IDExpr: retval->name = vl_copy_id_range(lval->name); break;
    case BitSelExpr: 
    case PartSelExpr:
	retval->name = vl_copy_id_range(lval->name);
	retval->range = vl_copy_range(lval->range);
	break;
    case ConcatExpr: {
	vl_expr *e;
	lsHandle handle;
	lsGen gen;

	retval->concat = lsCreate();
	gen = lsStart(lval->concat);
	if (lsNext(gen, (lsGeneric*)&e, &handle) == LS_OK) {
	    lsNewEnd(lval->concat, (lsGeneric)vl_copy_expr(e), 0);    
	    while(lsNext(gen, (lsGeneric*)&e, &handle) != LS_NOMORE) {
		lsNewEnd(lval->concat, (lsGeneric)vl_copy_expr(e), 0);
	    }
	}
	(void) lsFinish(gen);
	
	break;
    }

    default: internal_error("Unexpected Lval Type"); break;
    }

    return retval;
}

vl_expr *vl_copy_expr(vl_expr *expr)
{
    vl_expr *retval;

    if (!expr) return NIL(vl_expr);
    CpyTRACE("Printing expr\n");

    retval = (vl_expr*)chk_malloc(sizeof(vl_expr));
    retval->type = expr->type;
    switch (expr->type) {
    case BitExpr:
	retval->u.exprs.e1 = expr->u.exprs.e1;
	retval->u.exprs.e2 = expr->u.exprs.e2;
	retval->u.exprs.e3 = (vl_expr*)vlStrdup((char*)expr->u.exprs.e3);
	break;
    case IntExpr:
	retval->u.intval = expr->u.intval;
	break;
    case RealExpr:
	retval->u.realval = expr->u.realval;
	break;
    case IDExpr:
	retval->u.name = vl_copy_id_range(expr->u.name);
	break;
    case BitSelExpr:    case PartSelExpr:
	retval->u.idrng = vl_copy_id_range_rec(expr->u.idrng);
	break;
    case ConcatExpr: {
	
	vl_expr *e;
	lsHandle handle;
	lsGen gen;

	retval->u.exprs.e1 = (vl_expr*)lsCreate();
	gen = lsStart((lsList)(expr->u.exprs.e1));
	if (lsNext(gen, (lsGeneric*)&e, &handle) == LS_OK) {
	    lsNewEnd((lsList)retval->u.exprs.e1, 
		     (lsGeneric)vl_copy_expr(e), 0);    
	    while(lsNext(gen, (lsGeneric*)&e, &handle) != LS_NOMORE) {
		lsNewEnd((lsList)retval->u.exprs.e1, 
			 (lsGeneric)vl_copy_expr(e), 0);
	    }
	}
	(void) lsFinish(gen);
	break;			
    }
    case MinTypMaxExpr: {
	
	retval->u.exprs.e1 = vl_copy_expr(expr->u.exprs.e1);
	if (expr->u.exprs.e2) {
	    retval->u.exprs.e2 = vl_copy_expr(expr->u.exprs.e2);
	}
	if (expr->u.exprs.e3) {
	    retval->u.exprs.e3 = vl_copy_expr(expr->u.exprs.e3);
	}
	break;			
    }
    case StringExpr: retval->u.string = vlStrdup(expr->u.string); break;
    case FuncExpr:
	retval->u.func_call.name = vl_copy_id_range(expr->u.func_call.name);
	if (expr->u.func_call.args) {
	    retval->u.func_call.args = 
		vl_copy_expr_list(expr->u.func_call.args);
	}
	break;
    case UplusExpr:    case UminusExpr:
    case UnotExpr:    case UcomplExpr:
    
    case UandExpr:    case UnandExpr:
    case UorExpr:    case UnorExpr:
    case UxorExpr:    case UxnorExpr:
	retval->u.exprs.e1 = vl_copy_expr(expr->u.exprs.e1);
	break;
    
    case BplusExpr:    case BminusExpr:
    case BtimesExpr:    case BdivExpr:
    case BremExpr:    case Beq2Expr:
    case Bneq2Expr:    case Beq3Expr:
    case Bneq3Expr:    case BlandExpr:
    case BlorExpr:    case BltExpr:
    case BleExpr:   case BgtExpr:
    case BgeExpr:    case BandExpr:
    case BorExpr:    case BxorExpr:
    case BxnorExpr:    case BlshiftExpr:
    case BrshiftExpr:
	retval->u.exprs.e1 = vl_copy_expr(expr->u.exprs.e1);
	retval->u.exprs.e2 = vl_copy_expr(expr->u.exprs.e2);
	break;
    
    case TcondExpr:
	retval->u.exprs.e1 = vl_copy_expr(expr->u.exprs.e1);
	retval->u.exprs.e2 = vl_copy_expr(expr->u.exprs.e2);
	retval->u.exprs.e3 = vl_copy_expr(expr->u.exprs.e3);
	break;
    case NondExpr:
        retval->u.expr_list = vl_copy_expr_list(expr->u.expr_list);
        break;
    }

    return retval;
}


lsList vl_copy_expr_list(lsList exprs)
{
    void *expr;
    lsHandle handle;
    lsGen gen;
    lsList retval;
    
    CpyTRACE("copying Expression List\n");
    if (lsLength(exprs) <= 0) return (lsList)0;
    retval = lsCreate();
    gen = lsStart(exprs);
    if (lsNext(gen, (lsGeneric*)&expr, &handle) == LS_OK) {
	lsNewEnd(retval, (lsGeneric)vl_copy_expr(expr), 0);    
	while(lsNext(gen, (lsGeneric*)&expr, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_expr(expr), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}

vl_range *vl_copy_range(vl_range *range)
{
    vl_range *retval;

    CpyTRACE("copying range\n");

    if (range == NIL(vl_range) || 
	(range->left == NIL(vl_expr) && range->right == NIL(vl_expr))) 
	return NIL(vl_range);

    retval = (vl_range*)chk_malloc(sizeof(vl_range));
    retval->left = vl_copy_expr(range->left);
    if (range->right) {
	retval->right = vl_copy_expr(range->right);
    }

    return retval;
}


vl_delay *vl_copy_delay(vl_delay *delay)
{
    vl_delay *retval;

    CpyTRACE("copying delay\n");
    if (delay == NIL(vl_delay)) return NIL(vl_delay);
    
    retval = (vl_delay*)chk_malloc(sizeof(vl_delay));
    retval->delay1 = vl_copy_expr(delay->delay1);
    if (delay->delay2) {
	delay->delay2 = vl_copy_expr(delay->delay2);
	if (delay->delay3) {
	    delay->delay3 = vl_copy_expr(delay->delay3);
	}
    }

    return retval;
}


lsList vl_copy_id_list(lsList ids)
{
    vl_id_range *id;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    retval = lsCreate();
    gen = lsStart(ids);
    if (lsNext(gen, (lsGeneric*)&id, &handle) == LS_OK) {
        lsNewEnd(retval, (lsGeneric)vl_copy_id_range(id), 0);
	while(lsNext(gen, (lsGeneric*)&id, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_id_range(id), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}


lsList vl_copy_id_range_list(lsList ids)
{
    vl_id_range *id;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    if (ids == (lsList)0) return (lsList)0;
    retval = lsCreate();
    gen = lsStart(ids);
    if (lsNext(gen, (lsGeneric*)&id, &handle) == LS_OK) {
	lsNewEnd(retval, (lsGeneric)vl_copy_id_range(id), 0);    
	while(lsNext(gen, (lsGeneric*)&id, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_id_range(id), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}


lsList vl_copy_net_list(lsList nets)
{
    void *net;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    retval = lsCreate();
    gen = lsStart(nets);
    if (lsNext(gen, (lsGeneric*)&net, &handle) == LS_OK) {
	lsNewEnd(retval, vl_copy_net(net), 0);
	while(lsNext(gen, (lsGeneric*)&net, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, vl_copy_net(net), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}


void *vl_copy_net(void *net)
{    
    void *retval=NIL(void);

    switch(((vl_expr *)net)->type) {
    case BassignStmt: retval = (void*)vl_copy_bassign_stmt(net); break;
    case IDExpr: retval = (void*)vl_copy_expr(net); break;
    default: retval = (void*)vl_copy_id_range((vl_id_range*)net); break;
    }

    return retval;
}


lsList vl_copy_ports (lsList ports)
{
    void *item;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    if (ports != (lsList)0) {
	retval = lsCreate();
	gen = lsStart(ports);
	if (lsNext(gen, (lsGeneric*)&item, &handle) == LS_OK) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_port(item), 0);
	    while (lsNext(gen, (lsGeneric*)&item, &handle) != LS_NOMORE) {
		lsNewEnd(retval, (lsGeneric)vl_copy_port(item), 0);
	    }
	}
	(void) lsFinish(gen);
	return retval;
    }
    
    return (lsList)0;
}

vl_port *vl_copy_port(vl_portPtr port)
{
    vl_port *retval;

    retval = (vl_port*)chk_malloc(sizeof(vl_port));
    retval->type = port->type;
    if (port->type == NamedPort) {
	retval->id = vl_copy_id_range(port->id);
    }

    retval->port_exp = vl_copy_id_range_list(port->port_exp);
    
    return retval;
}


lsList vl_copy_connects (lsList connects)
{
    void *item;
    lsHandle handle;
    lsGen gen;
    lsList retval;

    
    if (connects != (lsList)0) {
	retval = lsCreate();
	gen = lsStart(connects);
	if (lsNext(gen, (lsGeneric*)&item, &handle) == LS_OK) {
	    lsNewEnd(retval, (lsGeneric)vl_copy_port_connect(item), 0);
	    while (lsNext(gen, (lsGeneric*)&item, &handle) != LS_NOMORE) {
		lsNewEnd(retval, (lsGeneric)vl_copy_port_connect(item), 0);
	    }
	}
	(void) lsFinish(gen);
	return retval;
    }

    return (lsList)0;
}

vl_port_connect *vl_copy_port_connect(vl_port_connectPtr connect)
{
    vl_port_connect *retval=NIL(vl_port_connect);

    retval = (vl_port_connect*)chk_malloc(sizeof(vl_port_connect));
    retval->type = connect->type;
    if (connect->type == NamedConnect) {
	retval->id = vl_copy_id_range(connect->id);
    }

    retval->expr = vl_copy_expr(connect->expr);

    return retval;
}


lsList vl_copy_mod_item_list (lsList mitems)
{
    void *item;
    lsHandle handle;
    lsGen gen;
    lsList retval;
    
    CpyTRACE("copying Module Items\n");
    if (mitems != (lsList)0) {
	retval = lsCreate();
	gen = lsStart(mitems);
	if (lsNext(gen, (lsGeneric*)&item, &handle) == LS_OK) {
	    lsNewEnd(retval, vl_copy_mod_item(item), 0);    
	    while(lsNext(gen, (lsGeneric*)&item, &handle) != LS_NOMORE) {
		lsNewEnd(retval, vl_copy_mod_item(item), 0);
	    }
	}
	(void) lsFinish(gen);

	return retval;
    }

    return (lsList)0;
}


void *vl_copy_mod_item(void *item)
{
    void *retval=NIL(void);
    
    switch(((vl_decl *)item)->type) {

    case RealDecl:  	case EventDecl:
	retval = (void*)vl_copy_basicdecl((vl_decl *)item);
	break;

    case IntDecl:   	case TimeDecl:
	retval = (void*)vl_copy_basicdecl((vl_decl *)item);
	break;

    case InputDecl: 	case OutputDecl:
    case InoutDecl: 	case RegDecl:
	retval = (void*)vl_copy_rangedecl((vl_rangedecl *)item);
	break;	    

    case ParamDecl: 	case DefparamDecl:
	retval = (void*)vl_copy_paramdecl((vl_paramdecl *)item);
	break;
	
    case WireDecl:  	case TriDecl:
    case Tri0Decl:  	case Tri1Decl:
    case Supply0Decl:	case Supply1Decl:
    case WandDecl:  	case TriandDecl:
    case WorDecl:   	case TriorDecl:
    case TriregDecl:
	retval = (void*)vl_copy_netdecl((vl_netdecl *)item);
	break;
	
    case ContAssign:
	retval = (void*)vl_copy_cont_assign((vl_cont_assign *)item);
	break;

    case TaskDecl:
	retval = (void*)vl_copy_task((vl_task *)item);
	break;
	    
    case IntFuncDecl:	case RealFuncDecl:
    case RangeFuncDecl:
	retval = (void*)vl_copy_function((vl_function *)item);
	break;
	    
    case AndGate:   	case NandGate:
    case OrGate:    	case NorGate:
    case XorGate:   	case XnorGate:
    case BufGate:   	case Bufif0Gate:
    case Bufif1Gate:	case NotGate:
    case Notif0Gate:	case Notif1Gate:
    case PulldownGate:	case PullupGate:
    case NmosGate:  	case RnmosGate:
    case PmosGate:  	case RpmosGate:
    case CmosGate:  	case RcmosGate:
    case TranGate:  	case RtranGate:
    case Tranif0Gate: 	case Rtranif0Gate:
    case Tranif1Gate:	case Rtranif1Gate:
	retval = (void*)vl_copy_gate_inst_list((vl_gate_inst_list *)item);
	break;

    case ModInst: {
	char *mp;

	if (st_lookup(vl_description->mp_st,
		       ((vl_mod_prim_inst_list *)item)->name->name,
		       &mp)) { 
	    if (((typestructPtr)mp)->type == CombPrimDecl ||
		((typestructPtr)mp)->type == SeqPrimDecl)
		goto copy_prim_inst;
	}
        }

	retval = (void*)vl_copy_mod_inst_list((vl_mod_prim_inst_list *)item);
	break;

copy_prim_inst:
	    
    case PrimInst:
	retval = (void*)vl_copy_prim_inst_list((vl_mod_prim_inst_list *)item);
	break;

    case AlwaysStmt:
    case InitialStmt:
	retval = (void*)vl_copy_procstmt((vl_procstmt *)item);
	break;
	    
    default: internal_error("Unexpected Module Item"); break;
    }

    return retval;
}


lsList vl_copy_decl_list (void *decls)
{
    void *decl;
    lsHandle handle;
    lsGen gen;
    lsList retval;
    
    CpyTRACE("copying Decls\n");
    if (decls == NIL(void)) return (lsList)0;
    retval = lsCreate();
    gen = lsStart(decls);
    if (lsNext(gen, (lsGeneric*)&decl, &handle) == LS_OK) {
	lsNewEnd(retval, vl_copy_decl(decl), 0);    
	while(lsNext(gen, (lsGeneric*)&decl, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, vl_copy_decl(decl), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}

lsList vl_copy_stmt_list (void *stmts)
{
    void *stmt;
    lsHandle handle;
    lsGen gen;
    lsList retval;
    
    CpyTRACE("copying Statements\n");
    retval = lsCreate();
    gen = lsStart(stmts);
    if (lsNext(gen, (lsGeneric*)&stmt, &handle) == LS_OK) {
	lsNewEnd(retval, vl_copy_stmt(stmt), 0);    
	while(lsNext(gen, (lsGeneric*)&stmt, &handle) != LS_NOMORE) {
	    lsNewEnd(retval, vl_copy_stmt(stmt), 0);
	}
    }
    (void) lsFinish(gen);

    return retval;
}

int vl_copy_strength (int s)
{
    return s;
}


vl_gate_inst *vl_copy_gate (vl_gate_inst *gate)
{
    vl_gate_inst *retval;

    retval = (vl_gate_inst*)chk_malloc(sizeof(vl_gate_inst));
    retval->name = vl_copy_id_range(gate->name);
    retval->terms = vl_copy_expr_list(gate->terms); 
    
    return retval;
}

vl_mod_prim_inst *vl_copy_prim (vl_mod_prim_inst *prim)
{
    vl_mod_prim_inst *retval;
    
    retval = (vl_mod_prim_inst*)chk_malloc(sizeof(vl_mod_prim_inst));
    retval->name = vl_copy_id_range(prim->name);
    retval->ports = vl_copy_expr_list(prim->ports);

    return retval;
}

lsList vl_copy_param_vals (lsList param_vals)
{
    if (param_vals == (lsList)0) return (lsList)0;

    return vl_copy_expr_list(param_vals);
}

vl_mod_prim_inst *vl_copy_modinst (vl_mod_prim_inst *mod)
{
    vl_mod_prim_inst *retval;

    retval = (vl_mod_prim_inst*)chk_malloc(sizeof(vl_mod_prim_inst));
    retval->name = vl_copy_id_range(mod->name);
    retval->ports = vl_copy_connects(mod->ports);
    
    return retval;
}
    
void *vl_copy_stmt (void *stmt)
{
    void *retval=NIL(void);
    
    if (stmt == NIL(void)) {
	return NIL(void);
    }

    switch(((vl_bassign_stmt *)stmt)->type) {
    case BeginEndStmt:
	retval = (void*)vl_copy_begin_end_stmt((vl_begin_end_stmt *)stmt);
	break;
    case IfElseStmt:
	retval = (void*)vl_copy_if_else_stmt((vl_if_else_stmt *)stmt);
	break;
    case CaseStmt:
    case CasexStmt:
    case CasezStmt:
	retval = (void*)vl_copy_case_stmt((vl_case_stmt *)stmt);
	break;
    case CaseItem:
	retval = (void*)vl_copy_stmt(((vl_case_item *)stmt)->stmt);
	break;
    case DefaultItem:
	retval = (void*)vl_copy_stmt(((vl_case_item *)stmt)->stmt);
	break;
    case ForeverStmt:
	retval = (void*)vl_copy_forever_stmt((vl_forever_stmt *)stmt);
	break;
    case RepeatStmt:
	retval = (void*)vl_copy_repeat_stmt((vl_repeat_stmt *)stmt);
	break;
    case WhileStmt:
	retval = (void*)vl_copy_while_stmt((vl_while_stmt *)stmt);
	break;
    case ForStmt:
	retval = (void*)vl_copy_for_stmt((vl_for_stmt *)stmt);
	break;
    case DelayControlStmt:
	retval = (void*)vl_copy_delay_control_stmt((vl_delay_control_stmt *)stmt);
	break;
    case EventControlStmt:
	retval = (void*)vl_copy_event_control_stmt((vl_event_control_stmt *)stmt);
	break;
    case AssignStmt:
    case BassignStmt:
    case NbassignStmt:
    case DelayBassignStmt:
    case DelayNbassignStmt:
    case EventBassignStmt:
    case EventNbassignStmt:
	retval = (void*)vl_copy_bassign_stmt((vl_bassign_stmt *)stmt);
	break;
    case WaitStmt:
	retval = (void*)vl_copy_wait_stmt((vl_wait_stmt *)stmt);
	break;
    case ForkJoinStmt:
	retval = (void*)vl_copy_fork_join_stmt((vl_fork_join_stmt *)stmt);
	break;
    case TaskEnableStmt:
    case SysTaskEnableStmt:
	retval = (void*)vl_copy_task_enable_stmt((vl_task_enable_stmt *)stmt);
	break;
    case DisableStmt:
	retval = (void*)vl_copy_disable_stmt((vl_disable_stmt *)stmt);
	break;
    case DeassignStmt:
	retval = (void*)vl_copy_deassign_stmt((vl_deassign_stmt *)stmt);
	break;
    case SendEventStmt:
	retval = (void*)vl_copy_id_range(((vl_send_event_stmt *)stmt)->name);
	break;
    default: {
	char msg[MAXSTRLEN];
	sprintf(msg, "vl_copy_stmt:Unexpected Statement Type %d", 
		((vl_bassign_stmt *)stmt)->type);
	internal_error(msg);
    }
    }

    return retval;
}

void *vl_copy_decl (void *decl)
{
    void *retval=NIL(void);

    switch(((vl_decl *)decl)->type) {
    case RealDecl:    case EventDecl:    
    case IntDecl:     case TimeDecl:
	retval = (void*)vl_copy_basicdecl((vl_decl *)decl);
	break;

    case InputDecl:    case OutputDecl:
    case InoutDecl:    case RegDecl:
	retval = (void*)vl_copy_rangedecl((vl_rangedecl *)decl);
	break;
    case ParamDecl:    case DefparamDecl:
	retval = (void*)vl_copy_paramdecl((vl_paramdecl *)decl);
	break;
    case WireDecl:   	case TriDecl:
    case Tri0Decl:    	case Tri1Decl:
    case Supply0Decl:   case Supply1Decl:
    case WandDecl:	case TriandDecl:
    case WorDecl:	case TriorDecl:
    case TriregDecl:
	retval = (void*)vl_copy_netdecl((vl_netdecl *)decl);
	break;
    default: 
	internal_error("Unexprected Declaration Type");
	break;
    }
    
    return retval;
}

vl_cont_assign *vl_copy_cont_assign (vl_cont_assign *assign)
{
    vl_cont_assign *retval;

    retval = (vl_cont_assign*)chk_malloc(sizeof(vl_cont_assign));
    retval->type = ContAssign;
    retval->strength = vl_copy_strength(assign->strength);
    retval->delay = vl_copy_delay(assign->delay);
    retval->assigns = vl_copy_net_list(assign->assigns);

    return retval;
}

vl_id_range *vl_copy_id_range_rec(vl_id_range *id_range)
{
    vl_id_range *retval;

    CpyTRACE("Copying id_range\n");
    retval = (vl_id_range *)chk_malloc(sizeof(vl_id_range));
    retval->name = vlStrdup(id_range->name);
    retval->flags = id_range->flags;
    retval->range = vl_copy_range(id_range->range);
    retval->initial = lsCreate();
    retval->syndrome_expr_list = id_range->syndrome_expr_list;
    retval->rst_syndrome_expr_list = id_range->rst_syndrome_expr_list;
    retval->mpg_master_exp = id_range->mpg_master_exp;
    retval->id_type = id_range->id_type;
    return(retval);
}

