/*

  Copyright (c) 1992, 1993
        Regents of the University of California
  All rights reserved.

  Use and copying of this software and preparation of derivative works
  based upon this software are permitted.  However, any distribution of
  this software or derivative works must include the above copyright 
  notice.

  This software is made available AS IS, and neither the Electronics
  Research Laboratory or the Universify of California make any
  warranty about the software, its performance or its conformity to
  any specification.

  Author: Szu-Tsung Cheng, stcheng@ic.Berkeley.EDU
          10/92
          10/93

  $Header: /projects/development/hsv/CVSRepository/vl2mv/src/parser/vl_create.c,v 1.2 2001/07/09 23:48:39 rbloem Exp $


*/


#include "util.h"
#include "list.h"
#include "st.h"
#include "set.h"
#include "array.h"
#include "vl_types.h"
#include "vl_defs.h"
#include "vlr_int.h"
#include "vl_create.h"
#include "vl_write.h"
#include "vl_write_util.h"
#include "verilog.h"

extern int yylineno;
extern int task_mode;
extern int aliasSP;
extern st_table *aliasStack[];
extern int use_libraryGate;

vl_descPtr mod_list;		
static int CreTrace = 0;

static int bstr_to_i ARGS((char *str));

vl_desc *vl_create_desc(char *filename)
{
    vl_descPtr desc;

    CreTRACE("Creating Description\n");
    desc = (vl_descPtr)chk_malloc(sizeof(vl_desc));
    desc->filename = filename;
    desc->type_st = st_init_table(strcmp, st_strhash);
    desc->modules = lsCreate();
    desc->mp_st = st_init_table(strcmp, st_strhash);
    desc->decl_st = st_init_table(ptrcmp, ptrhash);
    desc->lib_st = st_init_table(strcmp, st_strhash);
    desc->mp_undefined = set_empty();
    desc->fg_vars_st = st_init_table(ptrcmp, ptrhash);
    desc->handle = (lsHandle)0;
    desc->mod_holes = st_init_table(strcmp, st_strhash);
    desc->synclock_st = st_init_table(strcmp, st_strhash);
    return desc;
}

vl_type *vl_create_typedecl(char *type_name, struct vl_type_specifier *type_specifier)
{
    vl_type *retval;

    retval = (vl_type*)chk_malloc(sizeof(vl_type));
    retval->name = type_name;
    retval->specifier = type_specifier;
    st_insert(mod_list->type_st, type_name, (char*)retval);

    return retval;
}

vl_type_specifier *vl_create_type_specifier(short typeform, void *type_spc)
{
    vl_type_specifier *retval;

    retval = (vl_type_specifier*)chk_malloc(sizeof(vl_type_specifier));
    retval->typeform = typeform;
    switch(typeform) {
    case EnumType: retval->u.enum_type = (struct vl_enum_type*)type_spc; break;
    default: break;
    }

    return retval;
}

vl_enum_type *vl_create_enum_type(char *name, lsList elts_list)
{
    vl_enum_type *retval;
    vl_enumerator *enum_elt;
    lsGen gen;
    lsHandle handle;

    retval = (vl_enum_type*)chk_malloc(sizeof(vl_enum_type));
    retval->name = name;
    retval->domain_list = elts_list;
    retval->domain_st = st_init_table(strcmp, st_strhash);
    for (gen=lsStart(elts_list); 
	 lsNext(gen, (lsGeneric*)&enum_elt, &handle)!=LS_NOMORE;) {
	st_insert(retval->domain_st, enum_elt->name, (char*)enum_elt);
    }
    lsFinish(gen);

    return retval;
}

vl_enumerator *vl_create_enumerator(char *enum_name, int enum_val)
{
    vl_enumerator *retval;

    retval = (vl_enumerator*)chk_malloc(sizeof(vl_enumerator));
    retval->name = enum_name;
    retval->val = enum_val;

    return retval;
}

vl_module *vl_create_module(vl_id_range *mod_name, lsList ports, lsList mod_items)
{
    vl_module *mod;
    vl_module *clash_module;	

    CreTRACE("Creating module\n");
    if (st_lookup(mod_list->mp_st, mod_name->name, (char**)&clash_module)) {
	clash_module->type = ModDecl;
	clash_module->flags = 0;
	clash_module->ports = ports;
	clash_module->mod_items = mod_items;
	if (clash_module->inst_st) {
	    clash_module->inst_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->param_st) {
	    clash_module->param_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->sig_st) {
	    clash_module->sig_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->latch_st) {
	    clash_module->latch_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->quasi_st) {
	    clash_module->quasi_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->func_st) {
	    clash_module->func_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->uninit_set) {
	    clash_module->uninit_set = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->combVar_st) {
	    clash_module->combVar_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->seqVar_st) {
	    clash_module->seqVar_st = st_init_table(strcmp, st_strhash);
	}
	if (clash_module->task_st) {
	    clash_module->task_st = st_init_table(strcmp, st_strhash);
	}
	return clash_module;
    }
    mod = (vl_module *)chk_malloc(MAX(sizeof(vl_module),sizeof(vl_primitive)));
    st_insert(mod_list->mp_st, mod_name->name, (char *)mod);
    lsNewEnd(mod_list->modules, (char *)mod, LS_NH);
    mod->type = ModDecl;
    mod->flags = 0;
    mod->name = mod_name;
    mod->ports = ports;
    mod->sig_st = st_init_table(strcmp, st_strhash);
    mod->param_st = st_init_table(strcmp, st_strhash);
    mod->param_list = lsCreate();
    mod->inst_st = st_init_table(strcmp, st_strhash);
    mod->latch_st = st_init_table(strcmp, st_strhash);
    mod->quasi_st = st_init_table(strcmp, st_strhash);
    mod->func_st = st_init_table(strcmp, st_strhash);
    mod->uninit_set = st_init_table(strcmp, st_strhash);
    mod->combVar_st = st_init_table(strcmp, st_strhash);
    mod->seqVar_st = st_init_table(strcmp, st_strhash);
    mod->mod_items = mod_items;
    mod->clk = NIL(vl_event_expr);
    mod->syndrome_width = 0;
    mod->rst_syndrome_width = 0;
    mod->flow_graphs = lsCreate();
    mod->task_st = st_init_table(strcmp, st_strhash);
    mod->dup_count = 0;
    mod->visited = 0;
    return(mod);
}


vl_primitive *vl_create_primitive(vl_id_range *prim_name, lsList ports,lsList  prim_decls, lsList prim_entries)
{
    vl_primitive *prim;
    vl_primitive *clash_prim;	

    CreTRACE("Creating primitive\n");
    if (st_lookup(mod_list->mp_st, prim_name->name, (char**)&clash_prim)) {
	clash_prim->type = CombPrimDecl;
	clash_prim->flags = 0;
	clash_prim->ports = ports;
	clash_prim->sig_st = st_init_table(strcmp, st_strhash);
	clash_prim->uninit_set = st_init_table(strcmp, st_strhash);
	clash_prim->decls = prim_decls;
	clash_prim->entries = prim_entries;
	clash_prim->handle = (lsHandle)0;
	return clash_prim;
    }
    prim = (vl_primitive *)chk_malloc(sizeof(vl_primitive));
	st_insert(mod_list->mp_st, prim_name->name, (char *)prim);
    prim->type = CombPrimDecl;
    prim->flags = 0;
    prim->name = prim_name;
    prim->ports = ports;
    prim->sig_st = st_init_table(strcmp, st_strhash);
    prim->uninit_set = st_init_table(strcmp, st_strhash);
    prim->decls = prim_decls;
    prim->entries = prim_entries;
    prim->handle = (lsHandle)0;
    return(prim);
}

vl_port *vl_create_port(short type, vl_id_range *id, lsList exprs)
{
    vl_port *port;

    CreTRACE("Creating Port\n");
    port = (vl_port *)chk_malloc(sizeof(vl_port));
    port->type = type;
    port->id = id;
    port->port_exp = exprs;
    return port;
}

vl_port_connect *vl_create_port_connect(short type, vl_id_range *id, vl_expr *expr)
{
    vl_port_connect *port_connect;

    CreTRACE("Creating Port Connect\n");
    port_connect = (vl_port_connect *)chk_malloc(sizeof(vl_port_connect));
    port_connect->type = type;
    port_connect->id = id;
    port_connect->expr = expr;
    return port_connect;
}

vl_prim_entry *vl_create_prim_entry(lsList inputs, unsigned char state,
				    unsigned char output) 
{
    vl_prim_entry *entry;
    lsHandle handle;
    lsGen gen;
    unsigned int insym;
    int i = 0;

    CreTRACE("Creating primitive entry");
    entry = (vl_prim_entry *)chk_malloc(sizeof(vl_prim_entry));
    entry->state = state;
    entry->next_state = output;
    for (gen = lsStart(inputs);
	 i<MAXINLEN && lsNext(gen,(lsGeneric*)&insym,&handle)!=LS_NOMORE; ) {
	    entry->inputs[i++] = insym;
    }
    return entry;
}


vl_decl *vl_create_basicdecl(short type, lsList ids)
{
    vl_decl *decl;

    CreTRACE("Creating Basic decl\n");
    decl = (vl_decl *)chk_malloc(sizeof(vl_decl));
    decl->type = type;
    decl->flags = 0;
    decl->ids = ids;
    decl->handle = (lsHandle)0;
    if (type == EventDecl) vl_create_rangedecl(NIL(vl_type), type,
					       NIL(vl_range), ids, 0);

    return(decl);
}

vl_rangedecl *vl_create_rangedecl(vl_type *var_type, short type, vl_range
				  *range, lsList ids, int mv)
{
    vl_rangedecl *decl;
    lsGen gen;
    lsHandle handle;
    char *id, *dummy;
    char *idname;

    CreTRACE("Creating Range decl\n");
    decl = (vl_rangedecl *)chk_malloc(sizeof(vl_rangedecl));
    decl->type = type;
    decl->flags = (type==InputDecl)?InPort:(type==OutputDecl)?OutPort:
                  (type==InoutDecl)?(InPort|OutPort):RegVar;
    decl->flags |= mv;
    decl->range = range;
    decl->ids = ids;
    decl->handle = (lsHandle)0;

    if (var_type) {
	switch(var_type->specifier->typeform) {
	case EnumType: decl->flags |= MVar; break;
	default: semantic_error("can not handle types other than enum");
	}
    }
    
    
    if (task_mode && 
	(type==InputDecl || type==OutputDecl || type==InoutDecl))
	return decl;

    
    gen = lsStart(ids);
    while (lsNext(gen, &id, &handle) != LS_NOMORE) {
	if (var_type) ((vl_id_range *)id)->id_type = var_type;
	((vl_id_range *)id)->mpg_master_exp = decl;
	((vl_id_range *)id)->flags |= (mv | decl->flags);
	idname = ((vl_id_range *)id)->name;

	if (type==EventDecl) ((vl_id_range*)id)->flags |= EventVar;
	
	if (vl_currentModule != NIL(vl_module)) {
	    if (!st_lookup(SIG_ST(vl_currentModule), idname, &dummy)) {
		st_insert(SIG_ST(vl_currentModule), idname, id);
		dummy = id;
	    } else {
		((vl_id_range*)dummy)->flags |= mv;
		((vl_id_range*)dummy)->flags |= decl->flags;
		if (!(((vl_id_range*)dummy)->range))
		    ((vl_id_range*)dummy)->range = ((vl_id_range*)id)->range;
		if (!((vl_id_range*)dummy)->id_type)
		    ((vl_id_range*)dummy)->id_type = 
			((vl_id_range*)id)->id_type;
		if (!get_decl_range(((vl_id_range*)dummy)->mpg_master_exp)) {
		    void *new_decl;
		    new_decl = chng_decl_range(((vl_id_range*)dummy)->mpg_master_exp,
					       range);
		    if (new_decl && range) {
			lsNewEnd(((vl_netdecl*)new_decl)->ids, dummy, 0);
			((vl_id_range*)dummy)->mpg_master_exp = new_decl;
		    }
		}
	    }
	} else if (vl_currentPrimitive != NIL(vl_primitive)) {
	    if (!st_lookup(vl_currentPrimitive->sig_st, idname, &dummy)) {
		st_insert(vl_currentPrimitive->sig_st, idname, id);
		dummy = id;
	    } else {
		((vl_id_range*)dummy)->flags |= mv;
		((vl_id_range*)dummy)->flags |= decl->flags;
		if (!(((vl_id_range*)dummy)->range))
		    ((vl_id_range*)dummy)->range = ((vl_id_range*)id)->range;
		if (!((vl_id_range*)dummy)->id_type)
		    ((vl_id_range*)dummy)->id_type = 
			((vl_id_range*)id)->id_type;
	    }
	} else
	    compile_error("declaring signal not within module/primitive");

	
        
	if (vl_currentFunction) {
	    if (type==OutputDecl) {
		char buf[MAXSTRLEN];
		sprintf(buf, "function %s can not have output port %s",
			vl_currentFunction->name->name, idname);
		yylineno = -1;
		compile_error(buf);
	    }

	    if (type==InputDecl) {
		lsNewEnd(vl_currentFunction->ports, (lsGeneric)dummy, 0);
	    }
	}
    }
    lsFinish(gen);

    return(decl);
}

vl_paramdecl *vl_create_paramdecl(short type, lsList assigns)
{
    vl_paramdecl *decl;
    lsGen gen;
    lsHandle handle;
    vl_bassign_stmt *assign;
    char *dummy;
    int val;
 
    CreTRACE("Creating Parameter decl\n");
    decl = (vl_paramdecl *)chk_malloc(sizeof(vl_paramdecl));
    decl->type = type;
    decl->flags = 0;
    decl->assigns = assigns;
    decl->handle = (lsHandle)0;
    
    for (gen=lsStart(assigns); 
	 lsNext(gen, (lsGeneric*)&assign, &handle) != LS_NOMORE; ) {
	val = vl_eval_expr(assign->rhs);
	assign->lhs->name->mpg_master_exp = (void*)val;
	if (st_lookup(vl_currentModule->param_st,
		      assign->lhs->name->name, &dummy)) {
	    char buf[MAXSTRLEN];
	    sprintf(buf, "%s:duplicated parameter declaration - %s",
		    vl_currentModule->name->name, assign->lhs->name->name);
	    yylineno = -1;
	    compile_error(buf);
	}
	lsNewEnd(vl_currentModule->param_list, assign->lhs->name->name, 0);
	st_insert(vl_currentModule->param_st, 
		  assign->lhs->name->name, (char*)assign->lhs->name);
    }
    lsFinish(gen);

    return(decl);
}

vl_netdecl *vl_create_netdecl(vl_type *var_type, short type, int strength, vl_range *range, vl_delay *delay, lsList ids)
{
    vl_netdecl *decl;
    lsGen gen;
    lsHandle handle;
    vl_id_range *id;
    vl_id_range *id_sym;

    CreTRACE("Creating Net decl\n");
    decl = (vl_netdecl *)chk_malloc(sizeof(vl_netdecl));
    decl->type = type;
    decl->flags = 0;
    decl->strength = strength;
    decl->range = range;
    decl->delay = delay;
    decl->ids = ids;
    decl->handle = (lsHandle)0;

    if (var_type) {
	switch(var_type->specifier->typeform) {
	case EnumType: decl->flags |= MVar; break;
	default: semantic_error("can not handle types other than enum");
	}
    }

    
    gen = lsStart(ids);
    while (lsNext(gen, (lsGeneric*)&id, &handle) != LS_NOMORE) {
	id->mpg_master_exp = decl;
	id->flags |= decl->flags;
	if (var_type) id->id_type = var_type;
	
	if (vl_currentModule != NIL(vl_module)) {
	    if (!st_lookup(SIG_ST(vl_currentModule), id->name,(char**)&id_sym))
		st_insert(SIG_ST(vl_currentModule), id->name, (char*)id);
	    else {
		id_sym->flags |= decl->flags;
		if (!id_sym->range)
		    id_sym->range = ((vl_id_range*)id)->range;
		if (!id_sym->id_type)
		    id_sym->id_type = ((vl_id_range*)id)->id_type;
		if (!get_decl_range(id_sym->mpg_master_exp)) {
		    void *new_decl;
		    new_decl = chng_decl_range(id_sym->mpg_master_exp, range);
		    if (new_decl && range) {
			lsNewEnd(((vl_netdecl*)new_decl)->ids, 
				 (lsGeneric)id_sym, 0);
			id_sym->mpg_master_exp = new_decl;
		    }
		}
	    }
	} else if (vl_currentPrimitive != NIL(vl_primitive)) {
	    if (!st_lookup(vl_currentPrimitive->sig_st, 
			   id->name, (char**)&id_sym))
		st_insert(vl_currentPrimitive->sig_st, id->name, (char*)id);
	    else {
		id_sym->flags |= decl->flags;
		if (!id_sym->range)
		    id_sym->range = ((vl_id_range*)id)->range;
		if (!id_sym->id_type)
		    id_sym->id_type = ((vl_id_range*)id)->id_type;
	    }
	} else
	    compile_error("declaring signal not within module/primitive");
    }
    lsFinish(gen);

    return(decl);
}

vl_netdecl *vl_create_netdecl_assigned(vl_type *var_type, short type, 
				       int strength, vl_range *range, vl_delay *delay, lsList assignments)
{
    vl_netdecl *decl;
    lsGen gen;
    lsHandle handle;
    vl_id_range *id;
    vl_id_range *id_sym;
    lsList ids;

    CreTRACE("Creating Net decl\n");
    decl = (vl_netdecl *)chk_malloc(sizeof(vl_netdecl));
    decl->type = type;
    decl->flags = 0;
    decl->strength = strength;
    decl->range = range;
    decl->delay = delay;
    decl->ids = assignments;
    decl->handle = (lsHandle)0;
    ids = get_lhs_ids(assignments);
    if (var_type) {
	switch(var_type->specifier->typeform) {
	case EnumType: decl->flags |= MVar; break;
	default: semantic_error("can not handle types other than enum");
	}
    }

    
    gen = lsStart(ids);
    while (lsNext(gen, (lsGeneric*)&id, &handle) != LS_NOMORE) {
	id->mpg_master_exp = decl;
	id->flags |= decl->flags;
	if (var_type) id->id_type = var_type;
	
	if (vl_currentModule != NIL(vl_module)) {
	    if (!st_lookup(SIG_ST(vl_currentModule), id->name,(char**)&id_sym))
		st_insert(SIG_ST(vl_currentModule), id->name, (char*)id);
	    else {
		id_sym->flags |= decl->flags;
		if (!id_sym->range)
		    id_sym->range = ((vl_id_range*)id)->range;
		if (!id_sym->id_type)
		    id_sym->id_type = ((vl_id_range*)id)->id_type;
	    }
	} else if (vl_currentPrimitive != NIL(vl_primitive)) {
	    if (!st_lookup(vl_currentPrimitive->sig_st, 
			   id->name, (char**)&id_sym))
		st_insert(vl_currentPrimitive->sig_st, id->name, (char*)id);
	    else {
		id_sym->flags |= decl->flags;
		if (!id_sym->range)
		    id_sym->range = ((vl_id_range*)id)->range;
		if (!id_sym->id_type)
		    id_sym->id_type = ((vl_id_range*)id)->id_type;
	    }
	} else
	    compile_error("declaring signal not within module/primitive");
    }
    lsFinish(gen);

    return(decl);
}


vl_task *vl_create_task(vl_id_range *name, lsList decls, lsList stmts)
{
    vl_task *task;
    lsGen gen, id_gen;
    lsHandle handle, id_handle;
    vl_rangedecl *decl;
    vl_id_range *id;

    CreTRACE("Creating Task\n");
    task = (vl_task *)chk_malloc(sizeof(vl_task));
    task->type = TaskDecl;
    task->flags = 0;
    task->name = name;
    task->decls = decls;
    task->stmts = stmts;
    task->handle = (lsHandle)0;
    task->io_lst = lsCreate();
    task->sig_st = st_init_table(strcmp, st_strhash);
    for (gen=lsStart(decls); 
	 lsNext(gen, (lsGeneric*)&decl, &handle) != LS_NOMORE; ) {
        if (decl->ids) {
	    for (id_gen=lsStart(decl->ids);
		 lsNext(id_gen, (lsGeneric*)&id, &id_handle) != LS_NOMORE; ) {
	        
	        st_insert(task->sig_st, id->name, id->name);
		if (decl->type == InputDecl || decl->type == OutputDecl)
		    lsNewEnd(task->io_lst, id->name, 0);
	    }
	    lsFinish(id_gen);
	}
    }
    lsFinish(gen);
    st_insert(vl_currentModule->task_st, name->name, (char*)task);
    return(task);
}

vl_function *vl_create_function(short type, vl_range *range, vl_id_range *name, lsList decls, lsList stmts)
{
    vl_function *func;

    CreTRACE("Creating Function\n");
    func = (vl_function *)chk_malloc(sizeof(vl_function));
    func->type = type;
    func->flags = 0;
    func->range = range;
    func->name = name;
    func->decls = decls;
    func->stmts = stmts;
    func->ports = lsCreate();
    func->sig_st = st_init_table(strcmp, st_strhash);
    func->uninit_set = st_init_table(strcmp, st_strhash);
    func->handle = (lsHandle)0;
    func->param_st = st_init_table(strcmp, st_strhash);
    return(func);
}

vl_gate_inst_list *vl_create_gate_inst_list(short type, int strength, vl_delay *delays, lsList gates)
{
    vl_gate_inst_list *gatelist;

    CreTRACE("Creating Gate Instance List\n");
    gatelist = (vl_gate_inst_list *)chk_malloc(sizeof(vl_gate_inst_list));
    gatelist->type = type;
    gatelist->flags = 0;
    gatelist->strength = strength;
    gatelist->delays = delays;
    gatelist->gates = gates;
    gatelist->handle = (lsHandle)0;
    return(gatelist);
}

vl_gate_inst *vl_create_gate_inst(vl_id_range *name, lsList terms)
{
    vl_gate_inst *gate;

    use_libraryGate = 1;

    CreTRACE("Creating Gate Instance\n");
    gate = (vl_gate_inst *)chk_malloc(sizeof(vl_gate_inst));
    gate->name = name;
    gate->terms = terms;
    gate->handle = (lsHandle)0;
    return(gate);
}

typestruct *vl_add_find_mod_prim_instances(vl_id_range *mp_name, vl_delay *delay, int strength, lsList instances)
{
    char *mod_pri;
    typestructPtr retval=NULL;
    int still_unknown=0;

    if (!st_lookup(vl_description->mp_st, mp_name->name, &mod_pri)) {
	still_unknown = 1;
    } else if (mod_pri == NIL(char)) {
	still_unknown = 1;
    }
    
    if (still_unknown) {
	
	
	
	mod_pri = (char*)vl_create_module(mp_name, LS_NIL, LS_NIL);
	st_insert(mod_list->mp_st, mp_name->name, mod_pri);
	retval = (typestructPtr)vl_create_mod_prim_inst_list(mp_name, 
				    strength, delay, instances);
	set_add(mp_name->name, mod_list->mp_undefined);
	return retval;
    } else {
	if (((typestructPtr)mod_pri)->type == ModDecl) {
	    if (strength!=0)
		compile_error("redundant strength for module instance");
	    retval = (typestructPtr)vl_create_mod_prim_inst_list(mp_name, 
				        strength, delay, instances);
	    ((vl_mod_prim_inst_listPtr)retval)->type = ModInst;
	} else if (((typestructPtr)mod_pri)->type == CombPrimDecl ||
		   ((typestructPtr)mod_pri)->type == SeqPrimDecl) {
	    retval = (typestructPtr)vl_create_mod_prim_inst_list(mp_name, 
				        strength, delay, instances);
	    ((vl_mod_prim_inst_listPtr)retval)->type = PrimInst;
	} else {
	    compile_error("unknown module/primitive");
	}
    }

    return retval;
}


vl_mod_prim_inst_list *vl_create_mod_prim_inst_list(vl_id_range *name, int strength, vl_delay *delays, lsList mps)
{
    vl_mod_prim_inst_list *mplist;

    CreTRACE("Creating Module/Primitive Instance List\n");
    mplist = (vl_mod_prim_inst_list *)chk_malloc(sizeof(vl_mod_prim_inst_list));
    mplist->type = ModInst;
    mplist->flags = 0;
    mplist->name = name;
    mplist->strength = strength;
    mplist->delays = delays;
    mplist->mps = mps;
    mplist->handle = (lsHandle)0;
    return(mplist);
}

vl_mod_prim_inst *vl_create_mod_prim_inst(vl_id_range *name,
					  lsList ports)
{
    vl_mod_prim_inst *mp;

    CreTRACE("Creating Primitive Instance\n");
    mp = (vl_mod_prim_inst *)chk_malloc(sizeof(vl_mod_prim_inst));
    mp->name = name;
    mp->ports = ports;
    mp->handle = (lsHandle)0;
    return(mp);
}


vl_procstmt *vl_create_procstmt(short type, void *stmt)
{
    vl_procstmt *pstmt;

    CreTRACE("Creating Process stmt\n");
    pstmt = (vl_procstmt *)chk_malloc(sizeof(vl_procstmt));
    pstmt->type = type;
    pstmt->flags = 0;
    pstmt->stmt = stmt;
    pstmt->pc = vl_create_symbolic_var(vlStrdup(new_pc()));
    pstmt->fg_info = NIL(char);
    return(pstmt);
}

vl_begin_end_stmt *vl_create_begin_end_stmt(vl_id_range *name, void *decls, void *stmts)
{
    vl_begin_end_stmt *bestmt;

    CreTRACE("Creating beginend\n");
    bestmt = (vl_begin_end_stmt *)chk_malloc(sizeof(vl_begin_end_stmt));
    bestmt->type = BeginEndStmt;
    bestmt->flags = 0;
    bestmt->name = name;
    bestmt->decls = decls;
    bestmt->stmts = stmts;
    bestmt->handle = (lsHandle)0;
    return(bestmt);
}

vl_if_else_stmt *vl_create_if_else_stmt(vl_expr *cond, void *if_stmt, void *else_stmt)
{
    vl_if_else_stmt *stmt;

    CreTRACE("Creating if_else_stmt\n");
    stmt = (vl_if_else_stmt *)chk_malloc(sizeof(vl_if_else_stmt));
    stmt->type = IfElseStmt;
    stmt->flags = 0;
    stmt->cond = cond;
    stmt->if_stmt = if_stmt;
    stmt->else_stmt = else_stmt;
    stmt->handle = (lsHandle)0;
    return(stmt);
}

vl_case_stmt *vl_create_case_stmt(short type, vl_expr *cond, lsList case_items)
{
    vl_case_stmt *stmt;

    CreTRACE("Creating case_stmt\n");
    stmt = (vl_case_stmt *)chk_malloc(sizeof(vl_case_stmt));
    stmt->type = type;
    stmt->flags = 0;
    stmt->cond = cond;
    stmt->case_items = case_items;
    stmt->handle = (lsHandle)0;
    return(stmt);
}

vl_forever_stmt *vl_create_forever_stmt(void *stmt)
{
    vl_forever_stmt *fstmt;

    CreTRACE("Creating forever_stmt\n");
    fstmt = (vl_forever_stmt *)chk_malloc(sizeof(vl_forever_stmt));
    fstmt->type = ForeverStmt;
    fstmt->flags = 0;
    fstmt->stmt = stmt;
    fstmt->handle = (lsHandle)0;
    return(fstmt);
}

vl_repeat_stmt *vl_create_repeat_stmt( vl_expr *count, void *stmt)
{
    vl_repeat_stmt *rstmt;

    CreTRACE("Creating repeat_stmt\n");
    rstmt = (vl_repeat_stmt *)chk_malloc(sizeof(vl_repeat_stmt));
    rstmt->type = RepeatStmt;
    rstmt->flags = 0;
    rstmt->count = count;
    rstmt->stmt = stmt;
    rstmt->handle = (lsHandle)0;
    return(rstmt);
}

vl_while_stmt *vl_create_while_stmt(vl_expr *cond, void *stmt)
{
    vl_while_stmt *wstmt;

    CreTRACE("Creating while_stmt\n");
    wstmt = (vl_while_stmt *)chk_malloc(sizeof(vl_while_stmt));
    wstmt->type = WhileStmt;
    wstmt->flags = 0;
    wstmt->cond = cond;
    wstmt->stmt = stmt;
    wstmt->handle = (lsHandle)0;
    return(wstmt);
}

vl_for_stmt *vl_create_for_stmt(vl_bassign_stmt *init, vl_expr *cond, vl_bassign_stmt *end, void *stmt)
{
    vl_for_stmt *fstmt;

    CreTRACE("Creating for_stmt\n");
    fstmt = (vl_for_stmt *)chk_malloc(sizeof(vl_for_stmt));
    fstmt->type = ForStmt;
    fstmt->flags = 0;
    fstmt->init = init;
    fstmt->cond = cond;
    fstmt->end = end;
    fstmt->stmt = stmt;
    fstmt->handle = (lsHandle)0;
    return(fstmt);
}

vl_delay_control_stmt *vl_create_delay_control_stmt(vl_delay *delay, void *stmt)
{
    vl_delay_control_stmt *dcstmt;

    CreTRACE("Creating delay_control_stmt\n");
    dcstmt = (vl_delay_control_stmt *)chk_malloc(sizeof(vl_delay_control_stmt));
    dcstmt->type = DelayControlStmt;
    dcstmt->flags = 0;
    dcstmt->delay = delay;
    dcstmt->stmt = stmt;
    dcstmt->fg_info = NIL(char);
    return(dcstmt);
}

vl_event_control_stmt *vl_create_event_control_stmt(vl_event_expr *event, void *stmt)
{
    vl_event_control_stmt *ecstmt;

    CreTRACE("Creating event_control_stmt\n");
    ecstmt = (vl_event_control_stmt *)chk_malloc(sizeof(vl_event_control_stmt));
    ecstmt->type = EventControlStmt;
    ecstmt->flags = 0;
    ecstmt->event = event;
    ecstmt->stmt = stmt;
    ecstmt->fg_info = NIL(char);
    return(ecstmt);
}


vl_bassign_stmt *vl_create_bassign_stmt(short type, vl_lval *lhs, void *control, vl_expr *rhs)
{
    vl_bassign_stmt *bassign;

    CreTRACE("Creating bassign_stmt\n");
    bassign = (vl_bassign_stmt *)chk_malloc(sizeof(vl_bassign_stmt));
    bassign->type = type;
    bassign->flags = 0;
    bassign->lhs = lhs;
    bassign->control = control;
    bassign->rhs = rhs;
    bassign->lineno = yylineno;
    bassign->fg_info = NIL(char);
    return(bassign);
}

vl_cont_assign *vl_create_cont_assign_stmt(int strength, vl_delay *delay, lsList assigns)
{
    vl_cont_assign *cassign;

    CreTRACE("Creating cont_assign");
    cassign = (vl_cont_assign *)chk_malloc(sizeof(vl_cont_assign));
    cassign->type = ContAssign;
    cassign->flags = 0;
    cassign->strength = strength;
    cassign->delay = delay;
    cassign->assigns = assigns;
    return cassign;
}

vl_wait_stmt *vl_create_wait_stmt(vl_expr *cond, void *stmt)
{
    vl_wait_stmt *wstmt;

    CreTRACE("Creating wait_stmt\n");
    wstmt = (vl_wait_stmt *)chk_malloc(sizeof(vl_wait_stmt));
    wstmt->type = WaitStmt;
    wstmt->flags = 0;
    wstmt->cond = cond;
    wstmt->stmt = stmt;		
    wstmt->handle = (lsHandle)0;
    return(wstmt);
}

vl_send_event_stmt *vl_create_send_event_stmt(vl_id_range *name)
{
    vl_send_event_stmt *send_event;

    CreTRACE("creating send_event_stmt\n");
    send_event = (vl_send_event_stmt*)chk_malloc(sizeof(vl_send_event_stmt));
    send_event->type = SendEventStmt;
    send_event->name = name;
    send_event->fg_info = NIL(char);
    return send_event;
}

vl_fork_join_stmt *vl_create_fork_join_stmt(vl_id_range *name, void *decls, void *stmts)
{
    vl_fork_join_stmt *fjstmt;

    CreTRACE("Creating fork_join_stmt\n");
    fjstmt = (vl_fork_join_stmt *)chk_malloc(sizeof(vl_fork_join_stmt));
    fjstmt->type = ForkJoinStmt;
    fjstmt->flags = 0;
    fjstmt->name = name;
    fjstmt->decls = decls;
    fjstmt->stmts = stmts;
    fjstmt->handle = (lsHandle)0;
    return(fjstmt);
}


vl_task_enable_stmt *vl_create_task_enable_stmt(short type, vl_id_range *name, lsList args)
{
    vl_task_enable_stmt *testmt;

    CreTRACE("Creating task_enable_stmt\n");
    testmt = (vl_task_enable_stmt *)chk_malloc(sizeof(vl_task_enable_stmt));
    testmt->type = type;
    testmt->flags = 0;
    testmt->name = name;
    testmt->args = args;
    testmt->handle = (lsHandle)0;
    return(testmt);
}

vl_disable_stmt *vl_create_disable_stmt(vl_id_range *name)
{
    vl_disable_stmt *dstmt;

    CreTRACE("Creating disable_stmt\n");
    dstmt = (vl_disable_stmt *)chk_malloc(sizeof(vl_disable_stmt));
    dstmt->type = DisableStmt;
    dstmt->flags = 0;
    dstmt->name = name;
    dstmt->handle = (lsHandle)0;
    return(dstmt);
}

vl_deassign_stmt *vl_create_deassign_stmt(vl_lval *lhs)
{
    vl_deassign_stmt *dstmt;

    CreTRACE("Creating deassign_stmt\n");
    dstmt = (vl_deassign_stmt *)chk_malloc(sizeof(vl_deassign_stmt));
    dstmt->type = DeassignStmt;
    dstmt->flags = 0;
    dstmt->lhs = lhs;
    dstmt->handle = (lsHandle)0;
    return(dstmt);
}

vl_case_item *vl_create_case_item(short type, lsList expr, void *stmt)
{
    vl_case_item *item;

    CreTRACE("Creating case_item\n");
    item = (vl_case_item *)chk_malloc(sizeof(vl_case_item));
    item->type = type;
    item->flags = 0;
    item->exprs = expr;
    item->stmt = stmt;
    item->handle = (lsHandle)0;
    item->fg_info = NIL(char);
    return(item);
}

vl_event_expr *vl_create_event_expr(short type, struct vl_expr *expr)
{
    vl_event_expr *event;

    CreTRACE("Creating event_expr\n");
    event = (vl_event_expr *)chk_malloc(sizeof(vl_event_expr));
    event->type = type;
    event->flags = 0;
    event->expr = expr;
    event->list = (lsHandle)0;
    return(event);
}

vl_lval *vl_create_lval(short type, vl_id_range *name, vl_range *range,
			lsList concat) 
{
    vl_lval *lval;

    CreTRACE("Creating lval\n");
    lval = (vl_lval *)chk_malloc(sizeof(vl_lval));
    lval->type = type;
    lval->flags = 0;
    lval->name = name;
    lval->range = range;
    lval->concat = concat;
    return(lval);
}

vl_expr *vl_create_expr(short type, int intval, double realval, void *p1, void
			*p2, void *p3) 
{
    vl_expr *expr;

    CreTRACE("Creating expr\n");
    expr = (vl_expr *)chk_malloc(sizeof(vl_expr));
    expr->type = type;
    expr->flags = 0;
    expr->fg_info1 = expr->fg_info2 = NIL(char);
    expr->fg_aux1 = expr->fg_aux2 = NIL(char);
    switch (type) {
    case BitExpr:
	expr->u.bexpr.part1 = (int)p1;
	expr->u.bexpr.part0 = (int)p2;
	expr->u.exprs.e3 = (vl_expr*)p3;
	expr->u.intval = bstr_to_i(p3);
	break;
    case IntExpr:
	expr->u.intval = intval; 
	break;
    case RealExpr:
	expr->u.realval = realval; 
	break;
    case IDExpr:
	expr->u.name = (vl_id_range *)p1; 
	break;
    case BitSelExpr:    case PartSelExpr:
	expr->u.idrng = (vl_id_range *)p1; 
	break;
    case ConcatExpr:
	expr->u.exprs.e1 = (vl_expr*)p1;
	expr->u.exprs.e2 = (vl_expr*)p2;
	expr->u.exprs.e3 = NIL(vl_expr);
	break;			
    case MinTypMaxExpr:
	expr->u.exprs.e1 = (vl_expr*)p1;
	expr->u.exprs.e2 = (vl_expr*)p2;
	expr->u.exprs.e3 = (vl_expr*)p3;
	break;			
    case StringExpr:
	expr->u.string = (char *)p1;
	break;
    case FuncExpr:
	expr->u.func_call.name = (vl_id_range *)p1;
	expr->u.func_call.args = (lsList)p2;
	break;
    case UplusExpr:    case UminusExpr:
    case UnotExpr:    case UcomplExpr:
    
    case UandExpr:    case UnandExpr:
    case UorExpr:    case UnorExpr:
    case UxorExpr:    case UxnorExpr:
	expr->u.exprs.e1 = (vl_expr *)p1;
	expr->u.exprs.e2 = NIL(vl_expr);
	expr->u.exprs.e3 = NIL(vl_expr);
	break;
    
    case BplusExpr:    case BminusExpr:
    case BtimesExpr:    case BdivExpr:
    case BremExpr:    case Beq2Expr:
    case Bneq2Expr:    case Beq3Expr:
    case Bneq3Expr:    case BlandExpr:
    case BlorExpr:    case BltExpr:
    case BleExpr:   case BgtExpr:
    case BgeExpr:    case BandExpr:
    case BorExpr:    case BxorExpr:
    case BxnorExpr:    case BlshiftExpr:
    case BrshiftExpr:
	expr->u.exprs.e1 = (vl_expr *)p1;
	expr->u.exprs.e2 = (vl_expr *)p2;
	expr->u.exprs.e3 = NIL(vl_expr);
	break;
	
    case TcondExpr:
	expr->u.exprs.e1 = (vl_expr *)p1;
	expr->u.exprs.e2 = (vl_expr *)p2;
	expr->u.exprs.e3 = (vl_expr *)p3;
	break;
    case NondExpr: {
	int i;
	lsGen gen;
	lsHandle handle;
	vl_expr *sexp;
	
	for (gen=lsStart(p1), i=0; 
	     lsNext(gen, (lsGeneric*)&sexp, &handle) != LS_NOMORE; i++) {

	}
	(void)lsFinish(gen);
	
	expr->type = type;
	expr->u.expr_list = (lsList)p1;
	
	break;
    }
    }
    return(expr);
}


vl_range *vl_create_range(vl_expr *left, vl_expr *right)
{
    vl_range *range;

    CreTRACE("Creating range\n");
    range = (vl_range *)chk_malloc(sizeof(vl_range));
    range->left = left;
    range->right = right;
    return(range);
}

vl_delay *vl_create_delay(vl_expr *delay1, vl_expr *delay2, vl_expr *delay3)
{
    vl_delay *delay;

    CreTRACE("Creating delay\n");
    delay = (vl_delay *)chk_malloc(sizeof(vl_delay));
    delay->delay1 = delay1;
    delay->delay2 = delay2;
    delay->delay3 = delay3;
    return(delay);
}



vl_range_or_typePtr vl_create_range_or_type(short type, vl_rangePtr range)
{
    vl_range_or_typePtr rng_typ;

    rng_typ = (vl_range_or_typePtr)chk_malloc(sizeof(vl_range_or_type));
    rng_typ->type = type;
    rng_typ->range = range;
    return rng_typ;
}


vl_id_range *vl_create_id_range(char *name, vl_range *range)
{
    vl_id_range *id_range;

    CreTRACE("Creating id_range\n");
    id_range = (vl_id_range *)chk_malloc(sizeof(vl_id_range));
    id_range->dummy_type = None;
    id_range->name = name;
    id_range->range = range;
    id_range->flags = 0;
    id_range->initial = lsCreate();
    id_range->syndrome_expr_list = lsCreate();
    id_range->rst_syndrome_expr_list = lsCreate();
    id_range->symbolic_values = st_init_table(strcmp, st_strhash);
    id_range->mpg_master_exp = NIL(void);
    id_range->id_type = NIL(vl_type);
    id_range->edge_trigger = NIL(vl_term);
    id_range->sensitiveList = NIL(st_table);
    id_range->unintType = NIL(char);
    return(id_range);
}

vl_term *vl_create_term(vl_id_range *name, int lo, int hi)
{
    vl_term *term;

    CreTRACE("Creating term\n");
    term = (vl_term *)chk_malloc(sizeof(vl_term));
    term->name = name;
    term->flag = 0;
    term->lo = lo;
    term->hi = hi;
    term->term_type = name->id_type;
    return term;
}


vl_id_range *vl_copy_id_range(vl_id_range *id_range)
{
    vl_id_range *retval;
    int i;
    char *alias, *dummy;

    CreTRACE("Copying id_range\n");
    retval = (vl_id_range *)chk_malloc(sizeof(vl_id_range));
    
    alias=id_range->name;
    for (i=aliasSP-1; i>=0; i--) {
	if (st_lookup(aliasStack[i], id_range->name, &dummy)) {
	    alias = dummy;
	    break;
	}
    }
    retval->name = vlStrdup(alias);
    retval->flags = id_range->flags;
    retval->range = id_range->range;
    retval->initial = lsCreate();
    retval->syndrome_expr_list = id_range->syndrome_expr_list;
    retval->rst_syndrome_expr_list = id_range->rst_syndrome_expr_list;
    retval->mpg_master_exp = id_range->mpg_master_exp;
    retval->id_type = id_range->id_type;
    retval->unintType = id_range->unintType;
    return(retval);
}

vl_term *vl_copy_term(vl_term *term)
{
    vl_term *retval;

    CreTRACE("Copying term\n");
    retval = (vl_term*)chk_malloc(sizeof(vl_term));
    retval->name = vl_copy_id_range(term->name);
    retval->flag = term->flag;
    retval->lo = term->lo;
    retval->hi = term->hi;
    retval->term_type = term->term_type;
    return retval;
}

var_info *copy_var_info(var_info *var)
{
    var_info *retval;

    retval = (var_info*)chk_malloc(sizeof(var_info));
    retval->id = vl_copy_id_range(var->id);
    retval->current_terminal = vl_copy_term(var->current_terminal);
    retval->cond_list = lsCopy(var->cond_list,0);
    retval->extra = NIL(void);
    return retval;
}


vl_term *create_rename_term(vl_id_range *id_range, char *newname, int lo, int hi)

{
    vl_id_range *new_id;
    vl_term *new_term;

    new_id = vl_copy_id_range(id_range);

    new_id->name = vlStrdup(newname);
    new_term = vl_create_term(new_id, lo, hi);
    new_term->flag = get_decl_flags(id_range->mpg_master_exp);
    return new_term;
}

var_info *create_var_info(vl_id_range *id, vl_term *term)
{
    var_info *retval;

    retval = (var_info*)chk_malloc(sizeof(var_info));
    retval->id = id;
    retval->current_terminal = term;
    retval->cond_list = lsCreate();
    return retval;
}



void vl_free_id(vl_id_range * id)
{
/* #ifndef __linux__ */
    if (id->name) vl_chk_free(id->name);
/* #endif */
    vl_chk_free((char*)id);
}

void vl_free_term(vl_term * term)
{
    if (term->name) vl_free_id(term->name);
    vl_chk_free((char*)term);
}

void free_var_info(var_info *vinfo)
{
    if (vinfo->id) vl_free_id(vinfo->id);
    if (vinfo->current_terminal) vl_free_term(vinfo->current_terminal);
    vl_chk_free((char*)vinfo);
}

blif_latch *create_latch(vl_id_range *name, vl_term *input, vl_term *output)
{
    blif_latch *retval;

    retval = (blif_latch*)chk_malloc(sizeof(blif_latch));
    retval->name = name;
    retval->inport = input;
    retval->outport = output;
    return retval;
}

syndrome_expr *create_syndrome_expr(char *synd, vl_term *expr, st_table *pre_cond, int fg_id)
{
    syndrome_expr *retval;

    retval = (syndrome_expr*)chk_malloc(sizeof(syndrome_expr));
    retval->syndrome = synd;
    retval->expr = expr;
    retval->pre_cond = pre_cond;
    retval->fg_id = fg_id;
    return retval;
}

const_term *create_const_term(vl_expr *const_expr, vl_term *term)
{
    const_term *retval;

    retval = (const_term*)chk_malloc(sizeof(const_term));
    retval->const_expr = const_expr;
    retval->term = term;
    return retval;
}

vl_term *typed_new_term(vl_type *type, vl_range *range, int lo, int hi)
{
    vl_term *retval;

    retval = (vl_term*)chk_malloc(sizeof(vl_term));
    retval = vl_create_term(vl_create_id_range(vlStrdup(new_termname()),range),
			    lo, hi);
    retval->name->id_type = type;
    retval->term_type = type;
    return retval;
}

vl_term *new_term(vl_range *range, int lo, int hi)
{
    vl_term *retval;

    retval = (vl_term*)chk_malloc(sizeof(vl_term));
    retval = vl_create_term(vl_create_id_range(vlStrdup(new_termname()),range),
			    lo, hi);
    return retval;
}

symbolic_var *vl_create_symbolic_var(char *var_name)
{
    symbolic_var *retval;
    
    retval = (symbolic_var*)chk_malloc(sizeof(symbolic_var));
    retval->name = var_name;
    retval->values = set_empty();

    return retval;
}

char *new_pc()
{
    static int pc_number = 0;
    static char buf[MAXSTRLEN];
    
    sprintf(buf, "%s%d", PC, pc_number++);
    return buf;
}

lsList get_lhs_ids(lsList assignments)
{
    lsGen gen;
    lsHandle handle;
    lsList retval;
    vl_bassign_stmt *assignment;

    retval = lsCreate();

    gen = lsStart(assignments);
    while (lsNext(gen, (lsGeneric*)&assignment, &handle) != LS_NOMORE) {
	lsNewEnd(retval, (lsGeneric)assignment->lhs->name, 0);
    }
    lsFinish(gen);

    return retval;
}

static int bstr_to_i(char *str)
{
    int i, l;
    int retval = 0;

    l = strlen(str)-1;
    for (i=0; i<=MIN(l,31); i++)
	retval = (retval << 1) | ((str[i]=='1') || (str[i]=='?'));

    return retval;
}

multi_concat *vl_create_mconcat(vl_expr *rep, lsList concat)
{
    multi_concat *retval;

    retval = (multi_concat*)chk_malloc(sizeof(multi_concat));
    retval->rep = rep;
    retval->concat = concat;

    return retval;
}
