/*-----------------------------------------------------------------------*/
/* files.c --- file handling routines for xcircuit			 */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University       	 */
/*-----------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <time.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*------------------------------------------------------------------------*/
/* Useful (local) defines						  */
/*------------------------------------------------------------------------*/

#define OUTPUTWIDTH 80	/* maximum text width of output */

#define xmat(a)	(psscale == 0 ? (a) : (a) - localdata->lowerleft.x \
		+ (int)(xmargin / psscale))
#define ymat(a) (psscale == 0 ? (a) : (a) - localdata->lowerleft.y \
		+ (int)(ymargin / psscale))
#define S_OBLIQUE 13	/* position of Symbol-Oblique in font array */

/* type checks */

#define IS_POLYGON(a) 	((*a)->type == POLYGON)
#define IS_LABEL(a)	((*a)->type == LABEL)
#define IS_OBJINST(a)	((*a)->type == OBJECT)
#define IS_ARC(a)	((*a)->type == ARC)
#define IS_SPLINE(a)	((*a)->type == SPLINE)
#define IS_PATH(a)	((*a)->type == PATH)


/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern char _STR2[250], _STR[150];
extern Clientdata areastruct;
extern char **fonts;
extern short fontcount;
extern Display *dpy;
extern short eventmode;
extern short beeper;
extern short pushes;
extern int *appcolors;
extern int number_colors;
extern colorindex *colorlist;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c            */
/*------------------------------------------------------------------------*/

void savefile(), startloadfile(), importfile(), loadfile(), printobjects();
void loadlibrary(), savelibrary();
Boolean objectread();
float getpsscale();
float version;

extern void calcbbox(), setgridtype();
extern XtCallbackProc zoomview();

/*---------------------------------------------------------*/
/* Reset an object structure by freeing all alloc'd memory */
/*---------------------------------------------------------*/

void reset(localdata, mode)
  objectptr	localdata;
  short		mode;
{
   labelptr *resetlabel;
   polyptr  *resetpoly;

   if (localdata->parts > 0) {
      genericptr *genobj;
      if (mode != SAVE) {
	 for (genobj = localdata->plist; genobj < localdata->plist
	        + localdata->parts; genobj++) {
	    if (IS_POLYGON(genobj)) free(((polyptr)(*genobj))->points);
	    else if (IS_LABEL(genobj)) free(((labelptr)(*genobj))->string);
	    else if (IS_PATH(genobj)) free(((pathptr)(*genobj))->plist);
	    free(*genobj);
	 }
      }
      free(localdata->plist);
      if (mode != DESTROY)
	 localdata->plist = (genericptr *)malloc(sizeof(genericptr));
      localdata->parts = 0;
      localdata->hidden = False;
   }
}

void topreset()
{
   short rpage;
   objectptr resetpage;

   /* free alloc'd filename but */
   /* don't free it if another page points to it */

   if (areastruct.filename[areastruct.page] != NULL) {
      for (rpage = 0; rpage < areastruct.pages; rpage++) {
         resetpage = *(areastruct.pagelist + rpage);
         if (resetpage != NULL) {
            if ((rpage != areastruct.page) && areastruct.filename[rpage] ==
	        areastruct.filename[areastruct.page]) break;
         }
      }
      if (rpage == areastruct.pages) free(areastruct.filename[areastruct.page]);
   }

   rpage = areastruct.page;

   if (areastruct.selects > 0) {
      free(areastruct.selectlist);
      areastruct.selects = 0;
   }
   areastruct.textscale = 1.0;
   areastruct.wirewidth[rpage] = 2.0;
   areastruct.orient[rpage] = 0;
   areastruct.pmode[rpage] = 0;
   areastruct.filename[rpage] = (char *)malloc((1 + strlen((*(areastruct.pagelist
	+ rpage))->name)) * sizeof(char));
   strcpy(areastruct.filename[rpage], (*(areastruct.pagelist + rpage))->name);
   areastruct.viewscale[rpage] = 0.5;
   areastruct.drawingscale[rpage].x = areastruct.drawingscale[rpage].y = 1;
   areastruct.pcorner[rpage].x = -areastruct.width;
   areastruct.pcorner[rpage].y = -areastruct.height;
   areastruct.outscale[rpage] = 1.0;

   /* use the coordstyle of page 0 as the default for new pages */

   if (areastruct.coordstyle[0] == CM) {
      areastruct.coordstyle[rpage] = CM;
      areastruct.pagesize[rpage].x = 595;	 /* A4 */
      areastruct.pagesize[rpage].y = 842;
   }
   else {
      areastruct.coordstyle[rpage] = FRAC_INCH;
      areastruct.pagesize[rpage].x = 612;	 /* letter */
      areastruct.pagesize[rpage].y = 792;
   }

}

void initmem(localdata)
  objectptr localdata;
{
   localdata->parts = 0;
   localdata->plist = (genericptr *)malloc(sizeof(genericptr));
}

/*--------------------------------------------------------------*/
/* Exhaustively compare the contents of two objects and return  */
/* true if equivalent, false if not.				*/
/*--------------------------------------------------------------*/

Boolean elemcompare(compgen, gchk)
   genericptr *compgen, *gchk;
{
   Boolean bres;
   switch((*compgen)->type) {
      case(ARC):
	 bres = (TOARC(compgen)->position.x == TOARC(gchk)->position.x &&
            TOARC(compgen)->position.y == TOARC(gchk)->position.y &&
	    TOARC(compgen)->style == TOARC(gchk)->style &&
	    TOARC(compgen)->width == TOARC(gchk)->width &&
	    abs(TOARC(compgen)->radius) == abs(TOARC(gchk)->radius) &&
	    TOARC(compgen)->yaxis  == TOARC(gchk)->yaxis &&
	    TOARC(compgen)->angle1 == TOARC(gchk)->angle1 &&
	    TOARC(compgen)->angle2 == TOARC(gchk)->angle2);
	 break;
      case(SPLINE):
	 bres = (TOSPLINE(compgen)->style == TOSPLINE(gchk)->style &&
	    TOSPLINE(compgen)->width == TOSPLINE(gchk)->width &&
	    TOSPLINE(compgen)->ctrl[0].x == TOSPLINE(gchk)->ctrl[0].x &&
	    TOSPLINE(compgen)->ctrl[0].y == TOSPLINE(gchk)->ctrl[0].y &&
	    TOSPLINE(compgen)->ctrl[1].x == TOSPLINE(gchk)->ctrl[1].x &&
	    TOSPLINE(compgen)->ctrl[1].y == TOSPLINE(gchk)->ctrl[1].y &&
	    TOSPLINE(compgen)->ctrl[2].x == TOSPLINE(gchk)->ctrl[2].x &&
	    TOSPLINE(compgen)->ctrl[2].y == TOSPLINE(gchk)->ctrl[2].y &&
	    TOSPLINE(compgen)->ctrl[3].x == TOSPLINE(gchk)->ctrl[3].x &&
	    TOSPLINE(compgen)->ctrl[3].y == TOSPLINE(gchk)->ctrl[3].y);
	 break;
      case(POLYGON): {
	 int i;
	 if (TOPOLY(compgen)->style == TOPOLY(gchk)->style &&
	       TOPOLY(compgen)->width == TOPOLY(gchk)->width &&
	       TOPOLY(compgen)->number == TOPOLY(gchk)->number) {
	    for (i = 0; i < TOPOLY(compgen)->number; i++) {
	       if (TOPOLY(compgen)->points[i].x != TOPOLY(gchk)->points[i].x
		     || TOPOLY(compgen)->points[i].y != TOPOLY(gchk)->points[i].y)
		  break;
	    }
	    bres = (i == TOPOLY(compgen)->number);
	 }
	 }break;
   }
   return bres;
}

/*--------------------------------------------------------------------*/

short objcompare(obja, objb)
   objectptr obja, objb;
{
   genericptr *compgen, *glist, *gchk, *remg;
   short      csize;
   Boolean    bres;

   /* quick check on equivalence of number of objects */

   if (obja->parts != objb->parts) return False;

   /* For the exhaustive check we must match component for component. */
   /* Best not to assume that elements are in same order for both.    */

   csize = obja->parts;

   glist = (genericptr *)malloc(csize * sizeof(genericptr));
   for (compgen = objb->plist; compgen < objb->plist + csize; compgen++)
      (*(glist + (int)(compgen - objb->plist))) = *compgen;
   for (compgen = obja->plist; compgen < obja->plist + obja->parts;
	 compgen++) {
      bres = False;
      for (gchk = glist; gchk < glist + csize; gchk++) {
	 if ((*gchk)->type == (*compgen)->type && (*compgen)->color == (*gchk)->color) {
	    switch((*compgen)->type) {
	       case(OBJECT):
	 	  bres = (TOOBJINST(compgen)->position.x == TOOBJINST(gchk)->position.x &&
			TOOBJINST(compgen)->color == TOOBJINST(gchk)->color &&
	    		TOOBJINST(compgen)->position.y == TOOBJINST(gchk)->position.y &&
	     		TOOBJINST(compgen)->rotation == TOOBJINST(gchk)->rotation &&
	     		TOOBJINST(compgen)->scale == TOOBJINST(gchk)->scale &&
	     		TOOBJINST(compgen)->thisobject == TOOBJINST(gchk)->thisobject);
		  break;
	       case(LABEL):
	 	  bres = (TOLABEL(compgen)->position.x == TOLABEL(gchk)->position.x &&
			TOLABEL(compgen)->color == TOLABEL(gchk)->color &&
	     		TOLABEL(compgen)->position.y == TOLABEL(gchk)->position.y &&
	     		TOLABEL(compgen)->rotation == TOLABEL(gchk)->rotation &&
	     		TOLABEL(compgen)->scale == TOLABEL(gchk)->scale &&
	     		TOLABEL(compgen)->justify == TOLABEL(gchk)->justify &&
	     		!strcmp(TOLABEL(compgen)->string, TOLABEL(gchk)->string));
		  break;
	       case(PATH): /* elements *must* be in same order for a path */
		  bres = (TOPATH(compgen)->parts == TOPATH(gchk)->parts &&
			TOPATH(compgen)->color == TOPATH(gchk)->color &&
			TOPATH(compgen)->style == TOPATH(gchk)->style &&
			TOPATH(compgen)->width == TOPATH(gchk)->width);
		  if (bres) {
		     genericptr *pathchk, *gpath;
		     for (pathchk = TOPATH(compgen)->plist, gpath =
			   TOPATH(gchk)->plist; pathchk < TOPATH(compgen)->plist
			   + TOPATH(compgen)->parts; pathchk++, gpath++) {
		        if (!elemcompare(pathchk, gpath)) bres = False;
		     }
		  }
		  break;
	       case(ARC): case(SPLINE): case(POLYGON):
		  bres = elemcompare(compgen, gchk);
		  break;
	    }
	 }
	 if (bres) {
	   csize--;
	   for (remg = gchk; remg < glist + csize; remg++)
               *remg = *(remg + 1);
           break;
	 }
      }
   }
   free(glist);
   if (csize != 0) return False;

   return(True);
}

/*------------------------*/
/* scale renormalization  */
/*------------------------*/

float getpsscale(value, page)
  float value;
  short page;
{
   if (areastruct.coordstyle[page] == FRAC_INCH ||
         areastruct.coordstyle[page] == DEC_INCH)
      return (value * INCHSCALE);
   else
      return (value * CMSCALE);
}

/*------------------------*/
/* Load a PostScript file */
/*------------------------*/

void getfile(button, mode, nulldata)
  Widget        button;
  void		*mode;
  caddr_t       nulldata;
{
   buttonsave *savebutton;

   if (pushes != 0) {
      Wprintf("Can only read file into top-level page!");
      return;
   }
   savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getfile, mode);
   if ((int)mode == 0)
      popupprompt(button, "Enter filename to load:", "\0", startloadfile, savebutton,
		True);
   else
      popupprompt(button, "Enter filename to import:", "\0", importfile, savebutton,
		True);
}

Boolean nextfilename()	/* extract next filename from _STR2 into _STR */
{
   char *cptr, *slptr;

   sprintf(_STR, "%.149s", _STR2);
   if ((cptr = strrchr(_STR2, ',')) != NULL) {
      slptr = strrchr(_STR, '/');
      if (slptr == NULL || ((slptr - _STR) > (cptr - _STR2))) slptr = _STR - 1;
      sprintf(slptr + 1, "%s", cptr + 1); 
      *cptr = '\0';
      return True;
   }
   else return False;
}

void loadblib()
{
   while (nextfilename()) loadlibrary(LIBRARY);
   loadlibrary(LIBRARY);
   composelib(LIBRARY);
}

void loadulib()
{
   while (nextfilename()) loadlibrary(USERLIB);
   loadlibrary(USERLIB);
   composelib(USERLIB);
}

void getlib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getlib, NULL);
   popupprompt(button, "Enter library to load:", "\0", loadblib, savebutton,
	True);
}

void getuserlib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getuserlib, NULL);
   popupprompt(button, "Enter library to load:", "\0", loadulib, savebutton,
	True);
}

void loadlibrary(mode)
  short mode;
{
   FILE *ps;
   char inname[150], temp[150], keyword[20], percentc;
   objectptr	*libobj;

   sscanf(_STR, "%s", inname);

   ps = fopen(inname, "r");
   if (ps == NULL) {

      /* try adding a .lps */
      if (strchr(inname, '.') == NULL) {
	 sprintf(inname, "%s.lps", _STR);
         ps = fopen(inname, "r");
      }
      if (ps == NULL) {
         /* also look in global directory if not found in cwd */
      	 sprintf(inname, "%s/%s", BUILTINS_DIR, _STR);
      	 ps = fopen(inname, "r");
      	 if (ps == NULL) {
	    sprintf(inname, "%s/%s.lps", BUILTINS_DIR, _STR);
	    ps = fopen(inname, "r");
	    if (ps == NULL) {
               Wprintf("No library file found.");
               return;
	    }
	 }
      }
   }

   for(;;) {
      if (fgets(temp, 149, ps) == NULL) {
         Wprintf("Error in library.");
         return;
      }
      sscanf(temp, "%c %.19s", &percentc, keyword);
	    if(percentc == '%' && !strcmp(keyword, "XCircuitLib")) break;
   }
   objectread(ps, objectdata, 0, 0, mode, temp, DEFAULTCOLOR);

   sprintf(_STR, "Loaded library %s", inname);
   Wprintf(_STR);
}

void startloadfile()
{

   while (nextfilename()) {
      loadfile(0);

      /* find next undefined page */

      while(areastruct.page < areastruct.pages && *(areastruct.pagelist
	   + areastruct.page) != NULL) areastruct.page++;
      newpage(areastruct.page);
   }
   loadfile(0);
}

void importfile()
{
   while (nextfilename()) loadfile(1);
   loadfile(1);
}

void loadfile(mode)
  short mode;
{
   FILE *ps;
   char inname[250], temp[150], keyword[20], percentc, *pdchar;
   char teststr[20], teststr2[20], pagestr[100];
   short offx, offy, multipage, page, temppmode = 0, oldpage;
   objectptr	*libobj;
   float tmpfl;
   XPoint pagesize;

   /* First, if we're in catalog mode, return with error */

   if (eventmode == CATALOG_MODE) {
      Wprintf("Cannot load file from library window");
      return;
   }

   sscanf(_STR, "%s", inname);
   ps = fopen(inname, "r");
   if (ps == NULL) {
      sprintf(inname, "%s.ps", _STR);
      ps = fopen(inname, "r");
      if (ps == NULL) {

#ifdef LGF
	 sprintf(inname, "%s.lgf", _STR);
	 if ((ps = fopen(inname, "r")) != NULL) {
	    fclose(ps);
	    loadlgf(NULL);
	    return;
	 }
	 else {
	    sprintf(inname, "%s.lfo", _STR);
	    if ((ps = fopen(inname, "r")) != NULL) {
	       fclose(ps);
	       loadlgf(NULL);
	       return;
	    }
	 }
#endif

         /* If we're on an empty page, go ahead and rename it.  Otherwise, don't */
         /* do it.                                                               */

         if (objectdata->parts == 0 && (mode == 0)) {

	    /* If the filename has a path component, use only the root */
	    if ((pdchar = strrchr(inname, '/')) != NULL) {
               sprintf(objectdata->name, "%s", pdchar + 1);
	       
	    }
            else
	       sprintf(objectdata->name, "%s", inname);
            if ((pdchar = strchr(objectdata->name, '.')) != NULL)
		if (!strcmp(pdchar + 1, "ps")) *pdchar = '\0';
	    renamepage(areastruct.page);
	    printname(objectdata);
            Wprintf("Starting new drawing");
         }
         else {
	    sprintf(_STR, "Can't open PS file %s", inname);
            Wprintf(_STR);
         }
	 return;
      }
   }

   multipage = 1;
   pagesize.x = 612;
   pagesize.y = 792;
   for(;;) {
      if (fgets(temp, 149, ps) == NULL) {
	 Wprintf("Error: EOF in or before prolog.");
	 return;
      }
      sscanf(temp, "%c%.19s", &percentc, keyword);
      for (pdchar = keyword; isspace(*pdchar); pdchar++);
      if (percentc == '%') {
	 if (!strcmp(pdchar, "XCircuit")) break;
	 if (!strcmp(pdchar, "%Page:")) break;
	 if (strstr(pdchar, "PS-Adobe") != NULL)
	    temppmode = (strstr(temp, "EPSF") != NULL) ? 0 : 1;
         else if (!strcmp(pdchar, "Version:"))
	    sscanf(temp, "%*c%*s %f", &version);
	 else if (!strcmp(pdchar, "%Pages:")) {
	    pdchar = temp;
	    for (; !isspace(*pdchar); pdchar++);
	    for (; isspace(*pdchar); pdchar++);
	    multipage = atoi(pdchar);
	 }
	 else if (!strcmp(pdchar, "%BoundingBox:")) {
	    sscanf(temp, "%*s %hd %hd %hd %hd", &offx, &offy,
		&(pagesize.x), &(pagesize.y));
	    pagesize.x += offx;
	    pagesize.y += offy;
	 }
      }
#ifdef LGF
      else if (percentc == '-' && !strcmp(keyword, "5")) {
	 fclose(ps);
	 loadlgf(NULL);
	 return;
      }
#endif
   }

   /* Look for old-style files (no %%Page; maximum one page in file) */

   if (!strcmp(pdchar, "XCircuit")) {
      int pch;

      do {
	 pch = getc(ps);
      } while (pch == '\n');
      ungetc(pch, ps);
      if (pch == '%') fgets(temp, 149, ps);
   }

   for (page = 0; page < multipage; page++) {
      sprintf(pagestr, "%d", page + 1);

      /* read out-of-page library definitions */

      if (strstr(temp, "%%Page:") == NULL && strstr(temp, "offsets") == NULL)
	 objectread(ps, objectdata, 0, 0, USERLIB, temp, DEFAULTCOLOR);

      if (strstr(temp, "%%Page:") != NULL)
	 sscanf(temp + 8, "%s", pagestr);

      /* go to new page if necessary */

      if (page > 0) {

	 /* find next undefined page */

	 while(areastruct.page < areastruct.pages && *(areastruct.pagelist
	       + areastruct.page) != NULL) areastruct.page++;
	 newpage(areastruct.page);
      }

      while (strstr(temp, "offsets") == NULL) {
         if (fgets(temp, 149, ps) == NULL) {
	    Wprintf("Error: No offsets found for page.");
	    return;
	 }
      }
      sscanf(temp, "%c %hd %hd %*s", &percentc, &offx, &offy);
      if(percentc != '%') {
         Wprintf("Something wrong in offsets line.");
         offx = offy = 0;
      }

      /* skip over everything up to the "scale" line */

      while (strstr(temp, "scale") == NULL && strstr(temp, "rotate") == NULL) {
         if(fgets(temp, 149, ps) == NULL) {
            Wprintf("Error: No scale was set for this page.");
            return;
         }
      }

      /* good so far;  let's clear out the old data structure */
      /* and set the filename and pagename */

      if (mode == 0) {
	 char tpstr[6], *rootptr;

         reset(objectdata, NORMAL);
         topreset();
	 areastruct.pmode[areastruct.page] = temppmode;
	 if (temppmode == 1) {
	    areastruct.pagesize[areastruct.page].x = pagesize.x;
	    areastruct.pagesize[areastruct.page].y = pagesize.y;
	 }

	 /* set filename and page title.				     */
         /* for multiple-page files, filename is not linked to the page name */
	 /* but should point to a single string containing the file name and */
	 /* malloc'd from the first page of the set.			     */
	 /* for single-page files, filename is normally linked to the page   */
	 /* name unless the page label has been made different.		     */

         /* (delete the filename suffix if it is ".ps") */

	 if (page == 0) {
	    if (multipage > 1) oldpage = areastruct.page;
            areastruct.filename[areastruct.page] = (char *)realloc(
		 areastruct.filename[areastruct.page], (strlen(inname) + 1)
		 * sizeof(char));
            sprintf(areastruct.filename[areastruct.page], "%s", inname);
            if ((pdchar = strchr(areastruct.filename[areastruct.page], '.')) != NULL) 
	       if (!strcmp(pdchar + 1, "ps")) *pdchar = '\0';
	 }
	 else {
	    areastruct.filename[areastruct.page] = areastruct.filename[oldpage];
	 }

	 rootptr = strrchr(areastruct.filename[areastruct.page], '/');
	 if (rootptr == NULL) rootptr = areastruct.filename[areastruct.page];
	 else rootptr++;

	 sprintf(tpstr, "%d", page + 1);
	 if (!strcmp(pagestr, tpstr)) {
	    if (multipage > 1)
	       sprintf(objectdata->name, "%.72s:%s", rootptr, tpstr);
	    else
	       sprintf(objectdata->name, "%.79s", rootptr);
	 }
	 else
	    sprintf(objectdata->name, "%.79s", pagestr);

         renamepage(areastruct.page);
      }

      /* rotation (landscape mode) is optional; parse accordingly */

      sscanf(temp, "%f %s", &tmpfl, teststr);
      if (strstr(teststr, "scale") != NULL) {
	 setgridtype(teststr);
         areastruct.outscale[areastruct.page] = tmpfl;
      }
      else if (!strcmp(teststr, "rotate")) {
         areastruct.orient[areastruct.page] = (short)tmpfl;
         fgets(temp, 149, ps);
         sscanf(temp, "%f %s", &tmpfl, teststr2);
	 if (strstr(teststr2, "scale") != NULL) {
	    setgridtype(teststr2);
	    areastruct.outscale[areastruct.page] = tmpfl;
	 }
	 else {
	    sscanf(temp, "%*f %*f %s", teststr2);
	    if (!strcmp(teststr2, "scale"))
	       areastruct.outscale[areastruct.page] = tmpfl /
		    getpsscale(1.0, areastruct.page);
	    else {
	       Wprintf("Error in scale/rotate constructs.");
	       return;
	    }
	 }
      }
      else {     /* old style scale? */
         sscanf(temp, "%*f %*s %s", teststr2);
	 if ((teststr2 != NULL) && (!strcmp(teststr2, "scale")))
            areastruct.outscale[areastruct.page] = tmpfl /
		  getpsscale(1.0, areastruct.page);
	 else {
            Wprintf("Error in scale/rotate constructs.");
            return;
	 }
      }
   
      while (strstr(temp, "setlinewidth") == NULL) {
         if (fgets(temp, 149, ps) == NULL) {
            Wprintf("Error: no \"setlinewidth\" found for this page.");
            return;
	 }
      }

      sscanf(temp, "%f %*s", &areastruct.wirewidth[areastruct.page]);
      areastruct.wirewidth[areastruct.page] /= 1.3;

      objectread(ps, objectdata, offx, offy, NORMAL, temp, DEFAULTCOLOR);

      /* skip to next page boundary or file trailer */

      if (strstr(temp, "showpage") != NULL && multipage != 1) {
	 int pch;
         do {
	    pch = getc(ps);
         } while (pch == '\n');
	 ungetc(pch, ps);
         if (pch == '%') fgets(temp, 149, ps);
      }

      /* zoom to fit to window separately for each page */
      zoomview(NULL, Number(1), NULL);
   }

   sprintf(_STR, "Loaded file: %s (%d page%s)", inname, multipage,
	(multipage > 1 ? "s" : ""));
   Wprintf(_STR);

   printname(objectdata);
   composelib(USERLIB);

   sscanf(VERSION, "%f", &version);
}

Boolean objectread(ps, localdata, offx, offy, mode, retstr, ccolor)
  FILE *ps;
  objectptr	localdata;
  short		offx, offy, mode;
  char		*retstr;
  int		ccolor;
{
   char *temp, *buffer, keyword[80]; 
   short tmpfont = 0;
   int bufsize = 256;
   float tmpscale = 0.0;
   objectptr	*libobj;
   int curcolor = ccolor;

   /* path-handling variables */
   pathptr *newpath;
   XPoint startpoint;

   buffer = (char *)malloc(bufsize * sizeof(char));
   temp = buffer;

   for(;;) {
      char *lineptr, *keyptr;

      if (fgets(temp, 255, ps) == NULL) {
	 if (strcmp(keyword, "restore")) {
            Wprintf("Error: end of file.");
	    *retstr = '\0';
	    free(buffer);
            return True;
	 }
	 else break;
      }
      temp = buffer;

      /* because PostScript is a stack language, we will scan from the end */
      for (lineptr = buffer; (*lineptr != '\n') && (*lineptr != '\0'); lineptr++);
      if (lineptr != buffer) {  /* ignore any blank lines */
         for (keyptr = lineptr - 1; isspace(*keyptr) && keyptr != buffer; keyptr--);
         for (; !isspace(*keyptr) && keyptr != buffer; keyptr--);
         sscanf(keyptr, "%.79s", keyword);

         if (!strcmp(keyword, "showpage")) {
            strncpy(retstr, buffer, 150);
            retstr[149] = '\0';
	    free(buffer);
	    return False;  /* end of page */
	 }

	 /* make a color change, adding the color if necessary */

	 else if (!strcmp(keyword, "scb")) {
	    float red, green, blue;
	    sscanf(buffer, "%f %f %f", &red, &green, &blue);
	    curcolor = rgb_alloccolor((int)(red * 65535), (int)(green * 65535),
		(int)(blue * 65535)); 
	    addnewcolorentry(curcolor);
	 }

	 /* end the color change, returning to default */

	 else if (!strcmp(keyword, "sce")) curcolor = ccolor;

	 /* begin a path constructor */

	 else if (!strcmp(keyword, "beginpath")) {
	    sscanf(buffer, "%hd %hd", &startpoint.x, &startpoint.y);
	    startpoint.x -= offx; startpoint.y -= offy;
	    NEW_PATH(newpath, localdata);
	    (*newpath)->plist = (genericptr *)malloc(sizeof(genericptr));
	    (*newpath)->parts = 0;
	    (*newpath)->color = curcolor;
	 }

	 /* end the path constructor */

	 else if (!strcmp(keyword, "endpath")) {
	    sscanf(buffer, "%hd %f", &(*newpath)->style, &(*newpath)->width);
	    if ((*newpath)->parts > 0) {
	       localdata->parts++;
	    }
	    else {	/* in case of an empty path */
	       free((*newpath)->plist);
	       free(*newpath);
	    }
	    newpath = NULL;
	 }

	 /* read path parts */

	 else if (!strcmp(keyword, "polyc")) {
	    polyptr *newpoly;
	    pointlist newpoints;
	    short tmpnum;

	    NEW_POLY(newpoly, (*newpath));
	    (*newpath)->parts++;

	    for (--keyptr; *keyptr == ' '; keyptr--);
	    for (; *keyptr != ' '; keyptr--);
	    sscanf(keyptr, "%hd", &tmpnum);
	    (*newpoly)->number = tmpnum + 1;
	    (*newpoly)->width = 1.0;
	    (*newpoly)->style = UNCLOSED;
	    (*newpoly)->color = curcolor;

            (*newpoly)->points = (pointlist) malloc((*newpoly)->number * 
		   sizeof(XPoint));

	    lineptr = buffer;
            for (newpoints = (*newpoly)->points + (*newpoly)->number - 1;
		   newpoints > (*newpoly)->points; newpoints--) {
	       sscanf(lineptr, "%hd", &newpoints->x); 
	       newpoints->x -= offx;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	       sscanf(lineptr, "%hd", &newpoints->y);
	       newpoints->y -= offy;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	    }
	    newpoints->x = startpoint.x;
	    newpoints->y = startpoint.y;
	    startpoint.x = (newpoints + (*newpoly)->number - 1)->x;
	    startpoint.y = (newpoints + (*newpoly)->number - 1)->y;
	 }

	 else if (!strcmp(keyword, "arc") || !strcmp(keyword, "arcn")) {
	    arcptr *newarc;
	    NEW_ARC(newarc, (*newpath));
	    (*newpath)->parts++;
	    sscanf(buffer, "%hd %hd %hd %f %f", &(*newarc)->position.x,
		  &(*newarc)->position.y, &(*newarc)->radius,
		  &(*newarc)->angle1, &(*newarc)->angle2);
	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->yaxis = (*newarc)->radius;
	    (*newarc)->width = 1.0;
	    (*newarc)->style = UNCLOSED;
	    (*newarc)->color = curcolor;
	    if (!strcmp(keyword, "arcn")) {
	       float tmpang = (*newarc)->angle1;
	       (*newarc)->radius = -((*newarc)->radius);
	       (*newarc)->angle1 = (*newarc)->angle2;
	       (*newarc)->angle2 = tmpang;
	    }
		
	    calcarc(*newarc);
	    startpoint.x = (short)(*newarc)->points[(*newarc)->number - 1].x;
	    startpoint.y = (short)(*newarc)->points[(*newarc)->number - 1].y;
	 }

	 else if (!strcmp(keyword, "pellip") || !strcmp(keyword, "nellip")) {
	    arcptr *newarc;
	    NEW_ARC(newarc, (*newpath));
	    (*newpath)->parts++;
	    sscanf(buffer, "%hd %hd %hd %hd %f %f", &(*newarc)->position.x,
		 &(*newarc)->position.y, &(*newarc)->radius, &(*newarc)->yaxis,
		 &(*newarc)->angle1, &(*newarc)->angle2);
	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->width = 1.0;
	    (*newarc)->style = UNCLOSED;
	    (*newarc)->color = curcolor;
	    if (!strcmp(keyword, "nellip")) {
	       float tmpang = (*newarc)->angle1;
	       (*newarc)->radius = -((*newarc)->radius);
	       (*newarc)->angle1 = (*newarc)->angle2;
	       (*newarc)->angle2 = tmpang;
		
	    }
	    calcarc(*newarc);
	    startpoint.x = (short)(*newarc)->points[(*newarc)->number - 1].x;
	    startpoint.y = (short)(*newarc)->points[(*newarc)->number - 1].y;
	 }

	 else if (!strcmp(keyword, "curveto")) {
	    splineptr *newspline;
	    NEW_SPLINE(newspline, (*newpath));
	    (*newpath)->parts++;
            sscanf(buffer, "%hd %hd %hd %hd %hd %hd", 
		  &(*newspline)->ctrl[1].x, &(*newspline)->ctrl[1].y,
		  &(*newspline)->ctrl[2].x, &(*newspline)->ctrl[2].y,
		  &(*newspline)->ctrl[3].x, &(*newspline)->ctrl[3].y);
	    (*newspline)->ctrl[0].x = startpoint.x;
	    (*newspline)->ctrl[0].y = startpoint.y;
	    (*newspline)->ctrl[1].x -= offx; (*newspline)->ctrl[2].x -= offx;
	    (*newspline)->ctrl[3].x -= offx;
	    (*newspline)->ctrl[1].y -= offy; (*newspline)->ctrl[2].y -= offy;
	    (*newspline)->ctrl[3].y -= offy;
	    (*newspline)->width = 1.0;
	    (*newspline)->style = UNCLOSED;
	    (*newspline)->color = curcolor;
	    calcspline(*newspline);
	    startpoint.x = (*newspline)->ctrl[3].x;
	    startpoint.y = (*newspline)->ctrl[3].y;
	 }

         /* read arcs */

         else if (!strcmp(keyword, "xcarc")) {
	    arcptr *newarc;
	 
	    NEW_ARC(newarc, localdata);

	    /* backward compatibility */
	    if (version < 1.5) 
	       sscanf(buffer, "%hd %hd %hd %f %f %f %hd", &(*newarc)->position.x,
	          &(*newarc)->position.y, &(*newarc)->radius, &(*newarc)->angle1,
	          &(*newarc)->angle2, &(*newarc)->width, &(*newarc)->style);
	    else
	       sscanf(buffer, "%hd %f %hd %hd %hd %f %f", &(*newarc)->style,
		  &(*newarc)->width, &(*newarc)->position.x, &(*newarc)->position.y,
		  &(*newarc)->radius, &(*newarc)->angle1, &(*newarc)->angle2);

	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->yaxis = (*newarc)->radius;
	    (*newarc)->color = curcolor;
	    calcarc(*newarc);
	    localdata->parts++;
         }

	 /* read ellipses */

         else if (!strcmp(keyword, "ellipse")) {
	    arcptr *newarc;
	 
	    NEW_ARC(newarc, localdata);

	    sscanf(buffer, "%hd %f %hd %hd %hd %hd %f %f", &(*newarc)->style,
		 &(*newarc)->width, &(*newarc)->position.x, &(*newarc)->position.y,
		 &(*newarc)->radius, &(*newarc)->yaxis, &(*newarc)->angle1,
		 &(*newarc)->angle2);

	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->color = curcolor;
	    calcarc(*newarc);
	    localdata->parts++;
         }

         /* read polygons */
	 /* (and wires---backward compatibility for v1.5 and earlier) */

         else if (!strcmp(keyword, "polygon") || !strcmp(keyword, "wire")) {
	    polyptr *newpoly;
	    pointlist newpoints;

	    NEW_POLY(newpoly, localdata);
	    lineptr = buffer;

	    if (!strcmp(keyword, "wire")) {
	       (*newpoly)->number = 2;
	       (*newpoly)->width = 1.0;
	       (*newpoly)->style = UNCLOSED;
	    }
	    else {
	       /* backward compatibility */
	       if (version < 1.5) {
	          for (--keyptr; *keyptr == ' '; keyptr--);
	          for (; *keyptr != ' '; keyptr--);
	          sscanf(keyptr, "%hd", &(*newpoly)->style);
	          for (--keyptr; *keyptr == ' '; keyptr--);
	          for (; *keyptr != ' '; keyptr--);
	          sscanf(keyptr, "%f", &(*newpoly)->width);
	       }
	       for (--keyptr; *keyptr == ' '; keyptr--);
	       for (; *keyptr != ' '; keyptr--);
	       sscanf(keyptr, "%hd", &(*newpoly)->number);

	       if (version >= 1.5) {
	          sscanf(lineptr, "%hd", &(*newpoly)->style);
	          for (; *lineptr == ' '; lineptr++);
	          for (; *lineptr != ' '; lineptr++); 
	          sscanf(lineptr, "%f", &(*newpoly)->width);
	          for (; *lineptr == ' '; lineptr++);
	          for (; *lineptr != ' '; lineptr++); 
	       }
	    }

	    (*newpoly)->color = curcolor;
            (*newpoly)->points = (pointlist) malloc((*newpoly)->number * 
		   sizeof(XPoint));

            for (newpoints = (*newpoly)->points; newpoints < (*newpoly)->points
		+ (*newpoly)->number; newpoints++) {
	       sscanf(lineptr, "%hd", &newpoints->x); 
	       newpoints->x -= offx;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	       sscanf(lineptr, "%hd", &newpoints->y);
	       newpoints->y -= offy;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	    }
	    localdata->parts++;
         }

	 /* read spline curves */

         else if (!strcmp(keyword, "spline")) {
            splineptr *newspline;
	    short i;

	    NEW_SPLINE(newspline, localdata);

	    /* backward compatibility */
	    if (version < 1.5) {
               sscanf(buffer, "%f %hd %hd %hd %hd %hd %hd %hd %hd %hd", 
	          &(*newspline)->width, &(*newspline)->ctrl[1].x,
	          &(*newspline)->ctrl[1].y, &(*newspline)->ctrl[2].x,
	          &(*newspline)->ctrl[2].y, &(*newspline)->ctrl[3].x,
	          &(*newspline)->ctrl[3].y, &(*newspline)->ctrl[0].x,
	          &(*newspline)->ctrl[0].y, &(*newspline)->style);
	    }
	    else {
               sscanf(buffer, "%hd %f %hd %hd %hd %hd %hd %hd %hd %hd", 
		  &(*newspline)->style, &(*newspline)->width,
		  &(*newspline)->ctrl[1].x, &(*newspline)->ctrl[1].y,
		  &(*newspline)->ctrl[2].x, &(*newspline)->ctrl[2].y,
		  &(*newspline)->ctrl[3].x, &(*newspline)->ctrl[3].y,
		  &(*newspline)->ctrl[0].x, &(*newspline)->ctrl[0].y);
	    }

	    (*newspline)->ctrl[1].x -= offx; (*newspline)->ctrl[2].x -= offx;
	    (*newspline)->ctrl[0].x -= offx;
	    (*newspline)->ctrl[3].x -= offx;
	    (*newspline)->ctrl[1].y -= offy; (*newspline)->ctrl[2].y -= offy;
	    (*newspline)->ctrl[3].y -= offy;
	    (*newspline)->ctrl[0].y -= offy;
	    (*newspline)->color = curcolor;
            localdata->parts++;
	    calcspline(*newspline);
         }

         /* read labels */

         else if (!strcmp(keyword, "fontset")) { 	/* old style */
            char tmpstring[100];
            int i;
            sscanf(buffer, "%f %*c%s", &tmpscale, tmpstring);
            for (i = 0; i < fontcount; i++)
               if (!strcmp(tmpstring, fonts[i])) {
		  tmpfont = i;
		  break;
	       }
	    if (i == fontcount) i = 0;	/* Why bother with anything fancy? */
         }

         else if (!strcmp(keyword, "label")) {
	    labelptr *newlabel;
	    short segs, i, j;
	    uchar tmpstring[150], *tmpptr = tmpstring;
	    char *sptr;

	    NEW_LABEL(newlabel, localdata);

	    /* scan backwards to get the number of substrings */
	    lineptr = keyptr - 1;
	    for (i = 0; i < 5; i++) {
	       for (; *lineptr == ' '; lineptr--);
	       for (; *lineptr != ' '; lineptr--);
	    }
	    sscanf(lineptr, "%hd %hd %hd %hd %hd", &segs, &(*newlabel)->justify,
		&(*newlabel)->rotation, &(*newlabel)->position.x,
		&(*newlabel)->position.y);
	    (*newlabel)->position.x -= offx; (*newlabel)->position.y -= offy;
	    if ((*newlabel)->rotation > 0)
	       (*newlabel)->rotation = 360 - (*newlabel)->rotation;
	    (*newlabel)->rotation /= ROT_UNIT;
	    (*newlabel)->rotation++;
	    (*newlabel)->scale = tmpscale;
	    (*newlabel)->color = curcolor;

	    for (; *lineptr == ' '; lineptr--);

	    if (*lineptr == ')') {

	       /* This is for old-style labels only */

	       *tmpptr++ = TEXT_ESC;
	       *tmpptr++ = tmpfont + FONT_START;
	       *lineptr-- = '\0';
	       for (; *lineptr != '(' || (*lineptr == '(' && *(lineptr - 1) == 
		     '\\'); lineptr--)
	          if (lineptr == buffer + 1) {
		     lineptr--;
		     break;
	          }
	       for (sptr = lineptr + 1; *sptr != '\0'; sptr++) {
	          if (*sptr == '\\') {
		     sptr++;
		     if (*sptr >= '0' && *sptr < '8') {
		        int tmpdig;
		        sscanf(sptr, "%3o", &tmpdig);
		        *tmpptr++ = (uchar)tmpdig;
		        sptr += 2;
		     }
		     else *tmpptr++ = (uchar) *sptr;
	          }
	          else *tmpptr++ = (uchar) *sptr;
	       }
	       segs--;	/* for compatibility with new style */
	    }
	    else lineptr++;

	    for (i = 0; i < segs; i++) {
	       Boolean fline = False;
	       for (lineptr -= 1; *lineptr == ' '; lineptr--);
	       *lineptr-- = '\0';
	       for (; *lineptr != '{'; lineptr--);
	       if (!strcmp(lineptr + 1, "Ss")) {
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = SUPERSCRIPT;
	       }
	       else if (!strcmp(lineptr + 1, "ss")) {
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = SUBSCRIPT;
	       }
	       else if (!strcmp(lineptr + 1, "ns")) {
		   *tmpptr++ = TEXT_ESC;
		   *tmpptr++ = NORMALSCRIPT;
	       }
	       else if (!strcmp(lineptr + 1, "qS")) {
		   *tmpptr++ = TEXT_ESC;
		   *tmpptr++ = QTRSPACE;
	       }
	       else if (!strcmp(lineptr + 1, "hS")) {
		   *tmpptr++ = TEXT_ESC;
		   *tmpptr++ = HALFSPACE;
	       }
	       else if (*(lineptr + 1) == '(')   /* "bs" for backspace assumed */
		  for (sptr = lineptr + 2; *sptr != ')'; sptr++) {
		     *tmpptr++ = TEXT_ESC;
		     *tmpptr++ = BACKSPACE;
	          }
	       else if (!strcmp(lineptr + 1, "ol")) {
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = OVERLINE;
		  fline = True;
	       }
	       else if (!strcmp(lineptr + 1, "ul")) {
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = UNDERLINE;
		  fline = True;
	       }
	       else if (*(lineptr + 1) == '\0');
	       else {      /* "cf" for changefont assumed */
		  char *newptr = lineptr + 2;
		  for (; *newptr != ' '; newptr++);  *newptr = '\0'; /* end-of-word */

		  for (j = 0; j < fontcount; j++)
		     if (!strcmp(lineptr + 2, fonts[j])) {
			 tmpfont = j; 
			 break;
		     }
		  if (j == fontcount) {		/* this is a non-loaded font */
		     fonts = (char **) realloc (fonts, (fontcount + 1) * sizeof(char *));
		     fonts[fontcount] = (char *) malloc (sizeof(char));
		     strcpy(_STR, _STR2);
		     strcpy(_STR2, lineptr + 2);
		     setnewfont(NULL, fonts[fontcount]);
		     strcpy(_STR2, _STR);
		  }
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = (char)(j + FONT_START);

		  for (++newptr; *newptr == ' '; newptr++);
		  if (isdigit(*newptr)) {
		     /* second form of "cf" command---includes scale */
		     sscanf(newptr, "%f", &(*newlabel)->scale);
		  }
	       }
	       for (lineptr -= 1; *lineptr == ' '; lineptr--);
	       *lineptr-- = '\0';
	       for (; *lineptr != '(' || (*lineptr == '(' && *(lineptr - 1) ==
		    '\\'); lineptr--)
		  if (lineptr == buffer + 1) {
		     lineptr--;
		     break;
		  }
               for (sptr = lineptr + 1; *sptr != '\0'; sptr++) {
                  if (*sptr == '\\') {
		     sptr++;
		     if (*sptr >= '0' && *sptr < '8') {
			int tmpdig;
			sscanf(sptr, "%3o", &tmpdig);
			*tmpptr++ = (uchar)tmpdig;
			sptr += 2;
		     }
                     else *tmpptr++ = (uchar) *sptr;
		  }
                  else *tmpptr++ = (uchar) *sptr;
	       }
	       if (fline == True) {
		  *tmpptr++ = TEXT_ESC;
		  *tmpptr++ = NOLINE;
		  fline = False;
	       }
	    }
	    *tmpptr = '\0';

	    (*newlabel)->string = (uchar *) malloc((strlen(tmpstring) + 1) *
               sizeof(uchar));
	    strcpy((*newlabel)->string, tmpstring);

	    localdata->parts++;
         }

         /* read bounding box */

         else if (!strcmp(keyword, "bbox")) {
	    for (lineptr = buffer; *lineptr == ' '; lineptr++);
            if (*lineptr != '%') {
	       Wprintf("Illegal bbox.");
	       free(buffer);
               *retstr = '\0';
	       return True;
	    }
	    sscanf(++lineptr, "%hd %hd %hd %hd", &localdata->lowerleft.x, 
	       &localdata->lowerleft.y, &localdata->width, 
	       &localdata->height);
         }

	 /* read "hidden" attribute */

	 else if (!strcmp(keyword, "hidden")) {
	    localdata->hidden = True;
	 }

         /* read objects */

         else if (!strcmp(keyword, "{")) {  /* This is an object definition */
	    Boolean redef = False;
	    objectptr *newobject;
	    objectptr *curlib = (mode == LIBRARY) ? areastruct.library :
		areastruct.userlib;
	    short *libobjects = (mode == LIBRARY) ? &areastruct.builtins :
		&areastruct.userobjs;

	    for (lineptr = buffer; *lineptr == ' '; lineptr++);
	    if (*lineptr++ != '/') {
	       Wprintf("Bad object definition.");
	       free(buffer);
               *retstr = '\0';
	       return True;
	    }
	    sscanf(lineptr, "%.79s", keyword);

	    curlib = (objectptr *) realloc(curlib, (*libobjects + 1)
		      * sizeof(objectptr));
	    if (mode == LIBRARY) areastruct.library = curlib;
	    else areastruct.userlib = curlib;
	    /* initial 1-pointer allocations */
            newobject = curlib + (*libobjects);
	    *newobject = (objectptr) malloc(sizeof(object));
	    initmem(*newobject);

	    /* check that this object is not already in list of objects */

	    for (libobj = areastruct.library; libobj != areastruct.userlib +
	         areastruct.userobjs; libobj++) {
	       if (libobj == areastruct.library + areastruct.builtins) {
		  libobj = areastruct.userlib;
		  if (areastruct.userobjs == 0) break;
	       }
	       if (!strcmp(keyword, (*libobj)->name)) {  /* Is a redefinition */
		  redef = True;
		  break;
	       }
	    }

            (*libobjects)++;
	    sprintf((*newobject)->name, "%s", keyword);
	    if (objectread(ps, *newobject, 0, 0, mode, retstr, curcolor) == True) {
               strncpy(retstr, buffer, 150);
               retstr[149] = '\0';
	       free(buffer);
	       return True;
            }

	    /* do an exhaustive comparison between the objects. */
	    /* if they are the same, destroy the duplicate.  If */
	    /* different, rename the original one.		*/

	    else if (redef == True) {
	       if (objcompare(*newobject, *libobj) == True) {
		   reset(*newobject, DESTROY);
		   (*libobjects)--;
	       }
	       else {
		   strcpy(_STR, (*libobj)->name);
		   checkname(*libobj);
	       }
	    }
         }
         else if (!strcmp(keyword, "def")) {
            strncpy(retstr, buffer, 150);
            retstr[149] = '\0';
	    free (buffer);
	    return False; /* end of object def or end of object library */
	 }

	 else if (!strcmp(keyword, "loadlibrary")) {
	    for (lineptr = buffer; *lineptr != '%'; lineptr++);
	    sscanf (lineptr + 1, "%s", _STR);
	    loadlibrary(mode);
	 }
         else if (!strcmp(keyword, "EndLib")) break;
	 else if (!strcmp(keyword, "restore"));    /* handled at top */
	 else if (!strcmp(keyword, "grestore"));   /* ignore */
         else if (!strcmp(keyword, "begingate"));  /* also ignore */
         else if (!strcmp(keyword, "endgate"));    /* also ignore */
	 else if (!strcmp(keyword, "xyarray"));	   /* ignore for now */
         else {
	    objinstptr *newinst;
	    char *tmpptr;

	    /* first, make sure this is not a general comment line */
	    /* and return if we have a page boundary	   	   */

	    for (tmpptr = buffer; isspace(*tmpptr); tmpptr++);
	    if (*tmpptr == '%') {
	       if (strstr(buffer, "%%Page:") == tmpptr) {
                  strncpy(retstr, buffer, 150);
                  retstr[149] = '\0';
		  free (buffer);
		  return True;
	       }
	       continue;
	    }

	    /* (Assume that this line calls an object instance)	     */
	    /* Tricky double loop through builtin and user libraries */

	    for (libobj = areastruct.library; libobj != areastruct.userlib + 
	         areastruct.userobjs; libobj++) {
	       if (libobj == areastruct.library + areastruct.builtins) {
		  libobj = areastruct.userlib;
		  if (areastruct.userobjs == 0) break;
	       }
	       if (!strcmp(keyword, (*libobj)->name)) {
		  NEW_OBJINST(newinst, localdata);
      	          localdata->parts++;
		  localdata->hidden = False;

	          sscanf(buffer, "%f %hd %hd %hd", &(*newinst)->scale,
		     &(*newinst)->rotation, &(*newinst)->position.x,
		     &(*newinst)->position.y); 
		  (*newinst)->position.x -= offx;
		  (*newinst)->position.y -= offy;
                  if ((*newinst)->rotation > 0)
                     (*newinst)->rotation = (360 - (*newinst)->rotation) /
                        ROT_UNIT + 1;
                  else if ((*newinst)->rotation == 0) (*newinst)->rotation++;
                  else
                     (*newinst)->rotation = ((*newinst)->rotation + 1) /
                         ROT_UNIT - 1;
		  (*newinst)->thisobject = *libobj;
		  (*newinst)->color = curcolor;
	          break;
	       }
	    }
	    if (libobj == areastruct.userlib + areastruct.userobjs) {

	       /* will assume that we have a continuation line */

               for (lineptr = buffer; (*lineptr != '\n') && (*lineptr != '\0');
		     lineptr++);
	       if (*lineptr == '\n') *lineptr = ' ';

	       bufsize = (int)(lineptr - buffer) + 256;
	       buffer = (char *)realloc(buffer, bufsize * sizeof(char));
	       temp = buffer + (bufsize - 256);
 	    }
         }
      }
   }
   strncpy(retstr, buffer, 150);
   retstr[149] = '\0';
   free(buffer);
}

/*------------------------*/
/* Save a PostScript file */
/*------------------------*/

void setfile(button, fnamewidget, clientdata) 
  Widget	button, fnamewidget;	/* Widget containing filename info */
  caddr_t	clientdata;
{
   /* see if name has been changed in the buffer */

   strcpy(_STR2, (char *)XwTextCopyBuffer(fnamewidget));
   if (strcmp(areastruct.filename[areastruct.page], _STR2)) {
      Wprintf("Changing name of edit file.");
      areastruct.filename[areastruct.page] = (char *) realloc (
	   areastruct.filename[areastruct.page], (strlen(_STR2) + 1) *
	   sizeof(char));
      strcpy(areastruct.filename[areastruct.page], _STR2);
   }
   if (strstr(areastruct.filename[areastruct.page], "Page ") != NULL) {
      Wprintf("Warning: Enter a new name.");
      if (beeper) XBell(dpy, 100);
   }
   else {

      Arg wargs[1];
      Widget db, di;

      strcpy(_STR2, areastruct.filename[areastruct.page]);
      savefile(objectdata); 

      /* Change "close" button to read "done" */

      di = XtParent(button);
      db = XtNameToWidget(di, "Close");
      XtSetArg(wargs[0], XtNlabel, "  Done  ");
      XtSetValues(db, wargs, 1);
      if (beeper) XBell(dpy, 100);
   }
}

void savelib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   if (areastruct.userobjs == 0) {
      Wprintf("No user objects to save.");
      return;
   }
   getgeneric(savebutton, button, savelib, objectdata);
   popupprompt(button, "Enter name for library:", "\0", savelibrary,
	savebutton, False);
}

void savelibrary()
{
   FILE *ps;
   char outname[150];
   objectptr *wroteobjs, *libptr;
   short written = 0;

   sscanf(_STR2, "%s", outname);
   if (strstr(outname, ".") == NULL) sprintf(outname, "%s.lps", _STR2);

   ps = fopen(outname, "w");
   if (ps == NULL) {
      Wprintf("Can't open PS file.");
      return;
   }

   fprintf(ps, "%%! PostScript set of library objects for XCircuit\n");
   fprintf(ps, "%%  Library name is: %s\n", _STR2);
   fprintf(ps, "%%\n%%\n\n%% XCircuitLib library built-in objects\n");

   /* list of library objects already written */

   wroteobjs = (objectptr *) malloc (areastruct.builtins * sizeof(objectptr));

   /* write all of the object definitions used, bottom up */

   for (libptr = areastruct.userlib; libptr < areastruct.userlib + 
        areastruct.userobjs; libptr++)
      printobjects(ps, *libptr, 0.0, 0.0, 0.0, 0, 0, wroteobjs, &written, FOREGROUND);
   free(wroteobjs);

   /* and the postlog */

   fprintf(ps, "\n%% EndLib\n");
   fclose(ps);
   sprintf(_STR, "Library %s saved.", outname);
   Wprintf(_STR);
}

void savefile(localdata) 
   objectptr	localdata;
{
   FILE *ps, *pro;
   char outname[150], temp[150], prologue[150], *fname;
   uchar *chp;
   short written = 0, fontsused[256], i, page, multipage, savepage, stcount;
   objectptr *wroteobjs, writepage, *mpages;
   int *mpageno;
   genericptr *dfp;
   float psscale, psnorm;
   float xmargin = 72.0, ymargin = 72.0;
   int bboxx, bboxy;
   time_t tdate;

   fname = areastruct.filename[areastruct.page];
   sscanf(_STR2, "%s", fname);

   if (strstr(fname, ".") == NULL)
      sprintf(outname, "%s.ps", fname);
   else sprintf(outname, "%s", fname);

   ps = fopen(outname, "w");
   if (ps == NULL) {
      Wprintf("Can't open PS file.");
      return;
   }

   /* calculate the bounding box of the drawing */

   calcbbox(areastruct.topobject);

   /* find margins */

   psscale = getpsscale(areastruct.outscale[areastruct.page], areastruct.page);
   if (areastruct.pmode[areastruct.page] == 1) {
      if (areastruct.orient[areastruct.page] == 90) {
	 xmargin = (areastruct.pagesize[areastruct.page].x -
		((float)localdata->height * psscale)) / 2;
	 ymargin = (areastruct.pagesize[areastruct.page].y -
		((float)localdata->width * psscale)) / 2;
      }
      else {
         xmargin = (areastruct.pagesize[areastruct.page].x -
		((float)localdata->width * psscale)) / 2;
         ymargin = (areastruct.pagesize[areastruct.page].y -
		((float)localdata->height * psscale)) / 2;
      }
   }

   /* check for multiple-page output: get the number of pages */

   multipage = 0;
   mpages = (objectptr *) malloc (sizeof(objectptr));
   mpageno = (int *) malloc (sizeof(int));
   savepage = areastruct.page;

   for (page = 0; page < areastruct.pages; page++) {
      writepage = *(areastruct.pagelist + page);
      if (writepage != NULL)
	 if (areastruct.filename[page] == areastruct.filename[areastruct.page]) {
	    multipage++;
	    mpages = (objectptr *)realloc(mpages, multipage * sizeof(objectptr));
	    mpageno = (int *)realloc(mpageno, multipage * sizeof(int));
	    mpages[multipage - 1] = writepage;
	    mpageno[multipage - 1] = page;
	 }
   }
   if (multipage == 0) {
      Wprintf("Panic:  could not find this page in page list!");
      free (mpages);
      free (mpageno);
      return;
   }

   /* Print the PostScript DSC Document Header */

   fprintf(ps, "%%!PS-Adobe-3.0");
   if (multipage == 1 && areastruct.pmode[areastruct.page] == 0)
      fprintf(ps, " EPSF-3.0\n");
   else
      fprintf(ps, "\n");
   fprintf(ps, "%%%%Title: %s\n", areastruct.filename[areastruct.page]);
   fprintf(ps, "%%%%Creator: Xcircuit v%s\n", VERSION);
   tdate = time(NULL);
   fprintf(ps, "%%%%CreationDate: %s", asctime(localtime(&tdate)));
   fprintf(ps, "%%%%Pages: %d\n", multipage);
   if (areastruct.orient[areastruct.page] == 90) {
      bboxx = (int)((float)localdata->height * psscale) + (int)xmargin + 4;
      bboxy = (int)((float)localdata->width * psscale) + (int)ymargin + 4;
   }
   else {
      bboxx = (int)((float)localdata->width * psscale) + (int)xmargin + 4;
      bboxy = (int)((float)localdata->height * psscale) + (int)ymargin + 4;
   }
   /* in order to reconstruct the page size, it is necessary that the   */
   /* bounding box extent + margin = page size.  This of course assumes */
   /* that the picture is centered on the page so margins are equal.    */

   if (areastruct.pmode[areastruct.page] == 1) 
      fprintf(ps, "%%%%BoundingBox: %d %d %d %d\n",
	   areastruct.pagesize[areastruct.page].x - bboxx,
	   areastruct.pagesize[areastruct.page].y - bboxy, bboxx, bboxy);
   else
      fprintf(ps, "%%%%BoundingBox: %d %d %d %d\n", (int)xmargin - 4,
	   (int)ymargin - 4, bboxx, bboxy);
   for(i = 0; i <= fontcount; i++) fontsused[i] = 0;
   fprintf(ps, "%%%%DocumentNeededResources: ");
   stcount = 27;

   /* find all of the fonts used in this document */

   for (page = 0; page < multipage; page++) {
      writepage = mpages[page];
      for (dfp = writepage->plist; dfp < writepage->plist + writepage->parts; dfp++) {
	 if (IS_LABEL(dfp)) {
            for (chp = TOLABEL(dfp)->string; *chp != '\0'; chp++) {
	       int findex;
	       if ((short)(*chp) == TEXT_ESC && (findex = (short)(*(chp + 1)) -
		      FONT_START) >= 0) {
	          if (fontsused[findex] == 0) {
	             if (findex != S_OBLIQUE) {
		        stcount += strlen(fonts[findex]) + 6;
	                if (stcount > OUTPUTWIDTH) {
			   stcount = strlen(fonts[findex]) + 6;
		           fprintf(ps, "\n%%%%+ ");
	                }
	                fprintf(ps, "font %s ", fonts[findex]);
	             }
	             fontsused[findex] = 1;
		  }
	       }
	    }
	 }
      }
   }
   fprintf(ps, "\n%%%%EndComments\n");

   sprintf(prologue, "%s/%s", PROLOGUE_DIR, PROLOGUE_FILE);
   pro = fopen(prologue, "r");
   if (pro == NULL) {
      sprintf(prologue, "%s", PROLOGUE_FILE);
      pro = fopen(prologue, "r");
      if (pro == NULL) {
         Wprintf("Can't open prolog.");
	 free(mpages);
	 free(mpageno);
         return;
      }
   }

   /* write the prolog to the output */

   for(;;) {
      if (fgets(temp, 149, pro) == NULL) break;
      fputs(temp, ps);
   }
   fclose(pro);

   if (fontsused[S_OBLIQUE] == 1)
      fprintf(ps, "/Symbol-Oblique /Symbol .167 fontslant\n\n");

   fprintf(ps, "%% XCircuit output starts here.\n\n");

   /* list of objects already written---maximum possible = # of builtins */

   wroteobjs = (objectptr *) malloc (areastruct.builtins * sizeof(objectptr));

   for (page = 0; page < multipage; page++) {
      writepage = mpages[page];

      psnorm = areastruct.outscale[mpageno[page]];
      psscale = getpsscale(psnorm, mpageno[page]);
      if (areastruct.pmode[mpageno[page]] == 1) {
         if (areastruct.orient[mpageno[page]] == 90) {
	    xmargin = (areastruct.pagesize[areastruct.page].x -
		((float)writepage->height * psscale)) / 2;
	    ymargin = (areastruct.pagesize[areastruct.page].y -
		((float)writepage->width * psscale)) / 2;
         }
         else {
            xmargin = (areastruct.pagesize[areastruct.page].x -
		((float)writepage->width * psscale)) / 2;
            ymargin = (areastruct.pagesize[areastruct.page].y -
		((float)writepage->height * psscale)) / 2;
         }
      }

      /* write all of the object definitions used, bottom up */

      printobjects(ps, writepage, psnorm, xmargin, ymargin, page + 1,
	    mpageno[page], wroteobjs, &written, FOREGROUND);
      fprintf(ps, "pgsave restore showpage\n\n");
   }
   free (wroteobjs);
   free (mpages);

   /* Done! */

   fprintf(ps, "%%%%Trailer\n");
   fprintf(ps, "XCIRCsave restore\n");
   fprintf(ps, "%%%%EOF\n");
   fclose(ps);

   sprintf(_STR, "File %s saved (%d page%s).", areastruct.filename[areastruct.page],
	multipage, (multipage > 1 ? "s" : ""));
   Wprintf(_STR);
}

void dostcount(ps, count, addlength)
  FILE *ps;
  short *count, addlength;
{
   *count += addlength;
   if (*count > OUTPUTWIDTH) {
      *count = addlength;
      fprintf(ps, "\n");
   }
}

/* recursive routine to print out the library objects used in this drawing, */
/* starting at the bottom of the object hierarchy so that each object is    */
/* defined before it is called.  A list of objects already written is 	    */
/* maintained so that no object is written twice. 			    */

void printobjects(ps, localdata, psnorm, xmargin, ymargin, page, mpage,
	wrotelist, written, ccolor)
  FILE *ps;
  objectptr	localdata, *wrotelist;
  float		psnorm, xmargin, ymargin;
  short		page, mpage, *written;
  int		ccolor;
{
   genericptr *gptr, *savegen;
   objectptr *optr;
   pointlist savept;
   Boolean already, cchangeflag = False;
   short stcount;
   int curcolor = ccolor;
   float psscale = getpsscale(psnorm, mpage);

   /* first, get a total count of all objects and give warning if large */

   if (localdata->parts > 255) {
      sprintf(_STR, "Warning: \"%s\" may exceed printer's PS limit for definitions",
	    localdata->name);
      Wprintf(_STR);
   }
	   
   /* search for all object definitions needed */

   for (gptr = localdata->plist; gptr < localdata->plist + localdata->parts; gptr++)
      if (IS_OBJINST(gptr))
         printobjects(ps, TOOBJINST(gptr)->thisobject, 0.0, 0.0, 0.0, 0, 0,
		wrotelist, written, curcolor);

   if (psscale != 0) {  /* on topmost object of page */
      char *rootptr = strrchr(areastruct.filename[mpage], '/');
      if (rootptr == NULL) rootptr = areastruct.filename[mpage];
      else rootptr++;

      /* If the page label is just the root name of the file, or has been left   */
      /* as "Page n" or "Page_n", just do the normal page numbering.  Otherwise, */
      /* write out the page label explicitly.					 */

      if ((!strcmp(rootptr, localdata->name)) || (strchr(localdata->name, ' ')
		!= NULL) || (strstr(localdata->name, "Page_") != NULL))
         fprintf (ps, "%%%%Page: %d %d\n", page, page);
      else
         fprintf (ps, "%%%%Page: %s %d\n", localdata->name, page);
     
      fprintf (ps, "/pgsave save def bop\n");
      fprintf(ps, "%% %hd %hd offsets\n", (int)(xmargin / psscale) - localdata->
	    lowerleft.x, (int)(ymargin / psscale) - localdata->lowerleft.y);

      if (areastruct.orient[mpage] == 90)
         fprintf(ps, "90 rotate %d %d translate\n", (int)(ymargin - xmargin),
	     -((int)((float)localdata->height * psscale) + 
	     (int)(xmargin + ymargin)));
      fprintf(ps, "%5.4f ", psnorm);
      switch(areastruct.coordstyle[mpage]) {
	 case FRAC_INCH: case DEC_INCH:
	    fprintf(ps, "inchscale\n");
	    break;
	 case CM:
	    fprintf(ps, "cmscale\n");
	    break;
      };
      fprintf(ps, "%5.4f setlinewidth\n\n", 1.3 *
      	     areastruct.wirewidth[mpage]);
   }

   /* search among the list of objects already written to the output */

   already = False;
   for (optr = wrotelist; optr < wrotelist + *written; optr++)
      if (*optr == localdata) {
	  already = True;
	  break;
      }

   if (!already) {

      /* update the list of objects already written to the output */
      if (psscale == 0) {
         *(wrotelist + *written) = localdata;
         (*written)++;
      }

      if (psscale == 0) {
	 fprintf(ps, "/%s {\n", localdata->name);
	 fprintf(ps, "%% %hd %hd %hd %hd bbox\n", localdata->lowerleft.x,
	    localdata->lowerleft.y, localdata->width, localdata->height);
	 if (localdata->hidden == True) fprintf(ps, "%% hidden\n");
	 fprintf(ps, "begingate\n");
      }

      /* write all the elements in order */

      for (savegen = localdata->plist; savegen < localdata->plist +
	   localdata->parts; savegen++) {

	 /* change current color if different */

	 if ((*savegen)->color != curcolor) {
	    short i;

	    if (cchangeflag == True) {
	       fprintf(ps, "sce\n");
	       cchangeflag = False;
	       curcolor = ccolor;
	    }
	    if ((*savegen)->color != DEFAULTCOLOR) {
	       for (i = 0; i < number_colors; i++)
	          if (colorlist[i].color.pixel == (*savegen)->color) {
	             fprintf(ps, "%4.3f %4.3f %4.3f scb\n",
			   (float)colorlist[i].color.red   / 65535,
			   (float)colorlist[i].color.green / 65535,
			   (float)colorlist[i].color.blue  / 65535);
		     break;
	          }
	       curcolor = (*savegen)->color;
	       cchangeflag = True;
	    }
	 }

	 switch((*savegen)->type) {

	    case(POLYGON):
               sprintf(_STR, "%hd %5.2f ", TOPOLY(savegen)->style,
		     TOPOLY(savegen)->width);
	       stcount = strlen(_STR);
	       fputs(_STR, ps);
               for (savept = TOPOLY(savegen)->points; savept < TOPOLY(savegen)->
		      points + TOPOLY(savegen)->number; savept++) {
	          sprintf(_STR, "%hd %hd ", xmat(savept->x), ymat(savept->y));
	          dostcount (ps, &stcount, strlen(_STR));
	          fputs(_STR, ps);
	       }
	       sprintf(_STR, "%hd polygon\n", TOPOLY(savegen)->number);
	       dostcount (ps, &stcount, strlen(_STR));
	       fputs(_STR, ps);
	       break;

	    case(PATH): {
	       XPoint *startpoint;
	       genericptr *pgen;
	       pgen = TOPATH(savegen)->plist;
	       switch((*pgen)->type) {
		  case ARC:
	             fprintf(ps, "%hd %hd beginpath\n",
				xmat((short)(TOARC(pgen)->points[0].x)),
		     		ymat((short)(TOARC(pgen)->points[0].y)));
		     break;
		  case POLYGON:
		     startpoint = &(TOPOLY(pgen)->points[0]);
	             fprintf(ps, "%hd %hd beginpath\n", xmat(startpoint->x),
			 ymat(startpoint->y));
		     break;
		  case SPLINE:
		     startpoint = &(TOSPLINE(pgen)->ctrl[0]);
	             fprintf(ps, "%hd %hd beginpath\n",  xmat(startpoint->x),
			 ymat(startpoint->y));
		     break;
	       }
	       for (pgen = TOPATH(savegen)->plist; pgen < TOPATH(savegen)->plist
			+ TOPATH(savegen)->parts; pgen++) {
	          switch((*pgen)->type) {
		     case ARC:
			if (abs(TOARC(pgen)->radius) == TOARC(pgen)->yaxis) {
			   fprintf(ps, "%hd %hd %hd ",
			      xmat(TOARC(pgen)->position.x),
			      ymat(TOARC(pgen)->position.y),
			      abs(TOARC(pgen)->radius));
			   if (TOARC(pgen)->radius < 0)
			      fprintf(ps, "%3.2f %3.2f arcn\n",
			         TOARC(pgen)->angle2, TOARC(pgen)->angle1);
			   else
			      fprintf(ps, "%3.2f %3.2f arc\n",
			         TOARC(pgen)->angle1, TOARC(pgen)->angle2);
			}
		        else {
			   fprintf(ps, "%hd %hd %hd %hd ",
			      xmat(TOARC(pgen)->position.x),
			      ymat(TOARC(pgen)->position.y),
			      abs(TOARC(pgen)->radius), TOARC(pgen)->yaxis);
			   if (TOARC(pgen)->radius < 0)
			      fprintf(ps, "%3.2f %3.2f nellip\n",
			         TOARC(pgen)->angle2, TOARC(pgen)->angle1);
			   else
			      fprintf(ps, "%3.2f %3.2f pellip\n",
			         TOARC(pgen)->angle1, TOARC(pgen)->angle2);
			}
			break;
		     case POLYGON:
			stcount = 0;
               		for (savept = TOPOLY(pgen)->points + TOPOLY(pgen)->number
				- 1; savept > TOPOLY(pgen)->points; savept--) {
	          	   sprintf(_STR, "%hd %hd ", xmat(savept->x), ymat(savept->y));
	          	   dostcount (ps, &stcount, strlen(_STR));
	          	   fputs(_STR, ps);
	       		}
	       		sprintf(_STR, "%hd polyc\n", TOPOLY(pgen)->number - 1);
	       		dostcount (ps, &stcount, strlen(_STR));
	       		fputs(_STR, ps);
	       		break;
		     case SPLINE:
			fprintf(ps, "%hd %hd %hd %hd %hd %hd curveto\n",
			   xmat(TOSPLINE(pgen)->ctrl[1].x),
			   ymat(TOSPLINE(pgen)->ctrl[1].y),
			   xmat(TOSPLINE(pgen)->ctrl[2].x),
			   ymat(TOSPLINE(pgen)->ctrl[2].y),
			   xmat(TOSPLINE(pgen)->ctrl[3].x),
			   ymat(TOSPLINE(pgen)->ctrl[3].y));
			break;
		  }
	       }
	       fprintf(ps, "%hd %5.2f endpath\n", TOPATH(savegen)->style,
			TOPATH(savegen)->width);
	       } break;
	    case(SPLINE):
	       fprintf(ps, "%hd %5.2f %hd %hd %hd %hd %hd %hd %hd %hd spline\n",
	          TOSPLINE(savegen)->style, TOSPLINE(savegen)->width,
	          xmat(TOSPLINE(savegen)->ctrl[1].x), ymat(TOSPLINE(savegen)->ctrl[1].y),
	          xmat(TOSPLINE(savegen)->ctrl[2].x), ymat(TOSPLINE(savegen)->ctrl[2].y),
	          xmat(TOSPLINE(savegen)->ctrl[3].x), ymat(TOSPLINE(savegen)->ctrl[3].y),
	          xmat(TOSPLINE(savegen)->ctrl[0].x), ymat(TOSPLINE(savegen)->ctrl[0].y));
	       break;

	    case(ARC):
	       if (abs(TOARC(savegen)->radius) == TOARC(savegen)->yaxis)
                  fprintf(ps, "%hd %5.2f %hd %hd %hd %3.2f %3.2f xcarc\n",
			TOARC(savegen)->style, TOARC(savegen)->width,
			xmat(TOARC(savegen)->position.x),
			ymat(TOARC(savegen)->position.y), abs(TOARC(savegen)->radius),
			TOARC(savegen)->angle1, TOARC(savegen)->angle2);
	       else
                  fprintf(ps, "%hd %5.2f %hd %hd %hd %hd %3.2f %3.2f ellipse\n",
			TOARC(savegen)->style, TOARC(savegen)->width,
			xmat(TOARC(savegen)->position.x),
			ymat(TOARC(savegen)->position.y),
			abs(TOARC(savegen)->radius), TOARC(savegen)->yaxis,
			TOARC(savegen)->angle1, TOARC(savegen)->angle2);
	       break;

	    case(OBJECT):
	       {short rotation = TOOBJINST(savegen)->rotation;
	       if (rotation < 0) rotation = (rotation + 1) * ROT_UNIT - 1;
	       else rotation = 360 - ((rotation - 1) * ROT_UNIT);
	       if (rotation == 360) rotation = 0;
               fprintf(ps, "%3.2f %hd %hd %hd %s\n", TOOBJINST(savegen)->scale, 
	           rotation, xmat(TOOBJINST(savegen)->position.x), 
		   ymat(TOOBJINST(savegen)->position.y),
		   TOOBJINST(savegen)->thisobject->name);
	       }break;
            
	    case(LABEL):
	       {uchar tmpstring[250];
	       uchar *chrptr, *tmpptr = tmpstring;
	       short segs = 0, rotation;
	       stcount = 0;   /* approx. length of output line, for formatting */

               /* sprintf but with modifications for special characters */

	       chrptr = TOLABEL(savegen)->string;
	       *tmpptr++ = *chrptr;
	       for (++chrptr; *chrptr != '\0'; chrptr++) {

	          /* extended character set */

		  if (*(chrptr - 1) != TEXT_ESC) {
	             if (*chrptr == ')' || *chrptr == '(' || *chrptr == '\\' ||
		      	    *chrptr > (char)126) *tmpptr++ = '\\';
	             if (*chrptr > (char)126) {
		         sprintf(tmpptr, "%3o", (int)(*chrptr));
	 	         tmpptr += 3; 
	             }
	             else *tmpptr++ = *chrptr;
		  }
		  else *tmpptr++ = *chrptr;
	       }
	       *tmpptr = '\0';

	       /* Search label string for instances of redundant font changes */
	       /* and delete them.  Also look for instances of no text.	*/

	       for (chrptr = tmpstring; chrptr < tmpstring + strlen(tmpstring);
		      chrptr++) {
	          if ((short)(*chrptr) == TEXT_ESC && (short)(*(chrptr + 1))
			  >= FONT_START) {
	             uchar *chtmp;
	             for (chtmp = chrptr + 2; (short)(*chtmp) == TEXT_ESC &&
			      *chtmp != '\0'; chtmp += 2) {
		        if ((short)(*chtmp) == TEXT_ESC && (short)(*(chtmp + 1))
			      >= FONT_START) {
		           uchar *chclose;
		           *(chrptr + 1) = *(chtmp + 1);
		           for (chclose = chtmp; chclose < tmpstring +
			        strlen(tmpstring); chclose++)
			      *chclose = *(chclose + 2);
		           chtmp -= 2;
		        }
	             }
	             /* no text after font change */
	             if (*chtmp == '\0') {
		        *chrptr = '\0'; 
		        break;
	             }
	             else chrptr = chtmp;
	          }
	       }

	       /* New style label always starts with a font and scale */

	       for (chrptr = tmpstring + strlen(tmpstring) - 1; chrptr > tmpstring;
		      chrptr--) {
	          if (*(chrptr - 1) == TEXT_ESC) {
	             short control = (short)*chrptr;
	             segs++;
	             if (*(chrptr + 1) == TEXT_ESC) { 
		        if (control != NOLINE) {
		           sprintf(_STR, "() ");
		           dostcount(ps, &stcount, 3);
		        }
	             }
	             else {
	                sprintf(_STR, "(%s) ", chrptr + 1);
		        dostcount(ps, &stcount, 3 + strlen(chrptr + 1));
	             }
	             fputs(_STR, ps);  /* write the string output */
	       
	             if (control == SUBSCRIPT) {
		        sprintf(_STR, "{ss} ");
		        dostcount(ps, &stcount, 5);
	             }
	             else if (control == SUPERSCRIPT) {
		        sprintf(_STR, "{Ss} ");
		        dostcount(ps, &stcount, 5);
	             }
	             else if (control == NORMALSCRIPT) {
		        sprintf(_STR, "{ns} ");
		        dostcount(ps, &stcount, 5);
	             }
	             else if (control == UNDERLINE) {
		        sprintf(_STR, "{ul} ");
		        dostcount (ps, &stcount, 5);
	             }
	             else if (control == OVERLINE) {
		        sprintf(_STR, "{ol} ");
		        dostcount (ps, &stcount, 5);
	             }
		     else if (control == HALFSPACE) {
			sprintf(_STR, "{hS} ");
			dostcount (ps, &stcount, 5);
		     }
		     else if (control == QTRSPACE) {
			sprintf(_STR, "{qS} ");
			dostcount (ps, &stcount, 5);
		     }
	             else if (control == BACKSPACE) {
		        short bsctr = 1;
		
		        for (chrptr -= 2; (short)(*chrptr) == BACKSPACE &&
			      (short)(*(chrptr - 1)) == TEXT_ESC; chrptr-=2) bsctr++;
		        *(++chrptr) = '\0';
		        sprintf(_STR, "{(%s) bs} ", chrptr - bsctr);
		        dostcount (ps, &stcount, 6 + strlen(chrptr - bsctr));
		        chrptr++;
	             }
	             else if (control == NOLINE) {
		        if (*(chrptr + 1) != TEXT_ESC) {
		           sprintf(_STR, "{} ");
		           dostcount(ps, &stcount, 3);
		        }
		        else segs--;
	             }
	             else if (control >= FONT_START) {
		        char *chtmp, *lastfont = fonts[control - FONT_START];
		        for (chtmp = chrptr - 2; chtmp > (char *)tmpstring; chtmp--)
		           if ((short)(*chtmp - 1) == TEXT_ESC && (short)(*chtmp)
			      >= FONT_START) break;
		        if (chtmp <= (char *)tmpstring + 1) {
		           sprintf(_STR, "{/%s %5.3f cf} ", lastfont, 
				TOLABEL(savegen)->scale);
		           dostcount(ps, &stcount, 13 + strlen(lastfont));
		        }
		        else {
		           sprintf(_STR, "{/%s cf} ", lastfont);
		           dostcount(ps, &stcount, 7 + strlen(lastfont));
		        }
	             }
	             *(--chrptr) = '\0';
	             fputs(_STR, ps);
	          }
	       }
	       rotation = 360 - (TOLABEL(savegen)->rotation - 1) * ROT_UNIT;
	       if (rotation == 360) rotation = 0; 
	       if (segs > 0) {
                  sprintf(_STR, "%hd %hd %hd %hd %hd label\n", segs,
		      TOLABEL(savegen)->justify, rotation, xmat(TOLABEL(savegen)
		      ->position.x), ymat(TOLABEL(savegen)->position.y));
	          stcount += strlen(_STR);
	          if (stcount > OUTPUTWIDTH) fprintf(ps, "\n");
	          fputs(_STR, ps);
	       }
	       }break;
         }
      }

      /* if the color has been changed, end the current color context */

      if (cchangeflag == True) fprintf(ps, "sce\n");

      /* if this is not the top level, end the object definition */

      if (psscale == 0) fprintf(ps, "endgate\n} def\n\n");
   }
}
