/*-------------------------------------------------------------------------*/
/* libraries.c --- xcircuit routines for the builtin and user libraries    */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University       	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

float   savewidth; 	/* saved value of wirewidth on a page */

extern short   eventmode;	/* keep track of the mode the screen is in */
extern Display	*dpy;   /* Works well to make this globally accessible */
extern Window win;
extern int *appcolors;
extern Cursor	appcursors[NUM_CURSORS];
extern Clientdata areastruct;
extern char _STR[150];
extern objectptr *pushlist;
extern short pushes;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to events.c	 	  */
/*------------------------------------------------------------------------*/

extern void drawarea(), drag(), newpage(), setvscale();
extern short *allocselect();
void composelib(), changecat();

/*-------------------------------------------*/
/* Return to drawing window from the library */
/*-------------------------------------------*/

void catreturn()
{
   /* restore the graphics state of areastruct */

   areastruct.wirewidth[areastruct.page] = savewidth;

   /* pop the object being edited from the push stack */

   popobject(NULL, Number(1), NULL);
}

/*-----------------------------------------*/
/* Draw the catalog of predefined elements */
/*-----------------------------------------*/

void composelib(mode)
   short mode;
{
   short visobjects, libobjects = (mode == LIBRARY) ? areastruct.builtins :
	areastruct.userobjs;
   objinstptr *drawinst;
   labelptr *drawname;
   objectptr *curlib, *libobj, libinst = areastruct.libtop[mode], saveobj;
   int xpos = 0, ypos = 0;
   int nypos = 220, nxpos;

   reset(libinst, NORMAL);
   if (libobjects == 0) return;

   /* Create a pointer to the current library */

   curlib = (mode == LIBRARY) ? areastruct.library : areastruct.userlib;

   /* Look for all "hidden" objects and subtract them from the total number */

   visobjects = libobjects;
   for(libobj = curlib; libobj < curlib + libobjects; libobj++)
      if ((*libobj)->hidden == True) visobjects--;

   /* generate the list of object instances and their labels */

   libinst->plist = (genericptr *) realloc(libinst->plist,
	(visobjects * 2) * sizeof(genericptr));
   libinst->parts = 0;

   for(libobj = curlib; libobj < curlib + libobjects; libobj++){
      
      /* "Hidden" objects are not drawn */

      if ((*libobj)->hidden == True) continue;

      /* Otherwise, determine the area needed on the page to draw the object */

      nxpos = xpos + (((*libobj)->width > 170) ? (*libobj)->width + 30 : 200);
      if (nxpos > (areastruct.width << 1)  && (xpos > 0)) {
	 nxpos -= xpos; 
	 xpos = 0;
	 ypos -= nypos;
	 nypos = 200;
      }
      /* extra space of 20 is to leave room for the label */

      if ((*libobj)->height > (nypos - 50)) nypos = (*libobj)->height + 50;

      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->scale = 1.0;
      (*drawinst)->thisobject = (*libobj);

      (*drawinst)->position.x = xpos - (*libobj)->lowerleft.x;
      if ((*libobj)->width <= 170) (*drawinst)->position.x += 
	   ((170 - (*libobj)->width) >> 1);

      (*drawinst)->position.y = ypos - ((*libobj)->height + (*libobj)->lowerleft.y);
      if ((*libobj)->height <= 170) (*drawinst)->position.y -=
	   ((170 - (*libobj)->height) >> 1);

      drawname = (labelptr *)libinst->plist + (libinst->parts++) + visobjects;
      *drawname = (labelptr) malloc(sizeof(label));      
      (*drawname)->type = LABEL;
      (*drawname)->color = DEFAULTCOLOR;
      (*drawname)->rotation = 1;
      (*drawname)->scale = 0.75;
      (*drawname)->string = (uchar *) malloc((strlen((*libobj)->name) + 3) *
	   sizeof(uchar));
      sprintf((*drawname)->string, "%c%c%s", (char)TEXT_ESC, (char)FONT_START,
	   (*libobj)->name);
      (*drawname)->justify = TOP | NOTBOTTOM | NOTLEFT;

      if ((*libobj)->width > 170)
         (*drawname)->position.x = xpos + ((*libobj)->width >> 1);
      else
         (*drawname)->position.x = xpos + 85;

      if ((*libobj)->height > 170)
         (*drawname)->position.y = (*drawinst)->position.y + (*libobj)->lowerleft.y - 10;
      else
         (*drawname)->position.y = ypos - 180;

      xpos = nxpos;
   }
   libinst->parts = (visobjects * 2);
   saveobj = objectdata;
   objectdata = libinst;
   calcbbox(areastruct.topobject);
   objectdata = saveobj;
}

/*----------------------------------------------------------------*/
/* Find any dependencies on an object.				  */
/*   Return values:  0 = no dependency, 1 = dependency on page,	  */
/*	2 = dependency in another library object.		  */
/*   Object/Page with dependency (if any) returned in "compobjp". */
/*----------------------------------------------------------------*/

short finddepend(libobj, compobjp)
  objinstptr libobj;
  objectptr **compobjp;
{
   genericptr *testobj;
   objectptr *compobj;

   for (compobj = areastruct.library; compobj != areastruct.userlib +
          areastruct.userobjs; compobj++) {
      if (compobj == areastruct.library + areastruct.builtins) {
         compobj = areastruct.userlib;
         if (areastruct.userobjs == 0) break;
      }
      *compobjp = compobj;
		     
      for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	       + (*compobj)->parts; testobj++) {
	 if ((*testobj)->type == OBJECT) {
	    if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 2;
	 }
      }
   }

   /* also look in the pagelist */

   for (compobj = areastruct.pagelist; compobj != areastruct.pagelist +
             areastruct.pages; compobj++) {
      if (*compobj == NULL) continue;
      *compobjp = compobj;
      for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	       + (*compobj)->parts; testobj++) {
	 if ((*testobj)->type == OBJECT) {
	    if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 1;
	 }
      }
   }
   return 0;
}

/*----------------------------------------------------------------*/
/* "Hide" an object (must have a dependency or else it disappears)*/
/*----------------------------------------------------------------*/

cathide()
{
   short *newselect;
   objectptr *libpage, *compobj;
   objinstptr libobj;
   short *libpobjs;

   if (areastruct.selects == 0) return;

   if (objectdata == areastruct.libtop[LIBRARY]) {
      libpage = areastruct.library;
      libpobjs = &areastruct.builtins;
   }
   else {
      libpage = areastruct.userlib;
      libpobjs = &areastruct.userobjs;
   }

   /* Can only hide objects which are instances in other objects; */
   /* Otherwise, object would be "lost".			  */

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      if (finddepend(libobj, &compobj) == 0) {
	 sprintf(_STR, "Cannot hide: no dependencies");
	 Wprintf(_STR);
      }
      else { 		/* All's clear to hide. */
	 libobj->thisobject->hidden = True;
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if (objectdata == areastruct.libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*----------------------------------------------------------------*/
/* Delete an object from the library if there are no dependencies */
/*----------------------------------------------------------------*/

catdelete()
{
   short *newselect;
   genericptr *testobj, *tobj;
   objinstptr libobj;
   objectptr *libpage, *compobj, *tlib, *slib;
   short *libpobjs;

   if (areastruct.selects == 0) return;

   if (objectdata == areastruct.libtop[LIBRARY]) {
      libpage = areastruct.library;
      libpobjs = &areastruct.builtins;
   }
   else {
      libpage = areastruct.userlib;
      libpobjs = &areastruct.userobjs;
   }

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      /* Cannot delete an object if another object uses an instance of it, */
      /* or if the object is used on a page.				   */

      if (finddepend(libobj, &compobj)) {
	 sprintf(_STR, "Cannot delete: dependency in \"%s\"", (*compobj)->name);
	 Wprintf(_STR);
      }
      else { 		/* All's clear to delete.		     	   */

         /* First remove any instances of this object in the delete buffer */

         for (compobj = areastruct.delbuffer; compobj != areastruct.delbuffer +
                areastruct.deletes; compobj++) {
	    for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	          + (*compobj)->parts; testobj++) {
	       if ((*testobj)->type == OBJECT) {
	          if (TOOBJINST(testobj)->thisobject == libobj->thisobject) {
		     free(TOOBJINST(testobj));
		     for (tobj = testobj; tobj < (*compobj)->plist
			   + (*compobj)->parts - 1; tobj++) {
		        (*tobj) = (*(tobj + 1));
		     }
		     (*compobj)->parts--;
		  }
	       }
	    }
         }

	 for (tlib = libpage; tlib < libpage + *libpobjs; tlib++)
	    if ((*tlib) == libobj->thisobject) {
	       for (slib = tlib; slib < libpage + *libpobjs - 1; slib++)
		  (*slib) = (*(slib + 1));
	       (*libpobjs)--;
	       break;
	    }
	 
	 reset(libobj->thisobject, DESTROY);
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if (objectdata == areastruct.libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Rearrange objects in the library			  */
/*--------------------------------------------------------*/

void catmove()
{
   short *newselect;
   objectptr lobj, exchobj;
   objectptr *libpage, iobj, *sobj, *testobj, *tobj2;
   objinstptr *libobj, *saveobj;
   short *libpobjs, begflag;
   short ocentx, ocenty, rangey;

   if (areastruct.selects == 0) return;
   else if (areastruct.selects > 2) {
      Wprintf("Select maximum of two objects.");
      return;
   }

   if (objectdata == areastruct.libtop[LIBRARY]) {
      libpage = areastruct.library;
      libpobjs = &areastruct.builtins;
   }
   else {
      libpage = areastruct.userlib;
      libpobjs = &areastruct.userobjs;
   }

   /* If two objects are selected, then exchange their order */

   if (areastruct.selects == 2) {
      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      newselect = areastruct.selectlist + 1;
      lobj = SELTOOBJINST(newselect)->thisobject;

      for (testobj = libpage; testobj < libpage + *libpobjs; testobj++)
	 if (*testobj == exchobj) break;

      for (tobj2 = libpage; tobj2 < libpage + *libpobjs; tobj2++)
	 if (*tobj2 == lobj) break;

      iobj = *testobj;
      *testobj = *tobj2;
      *tobj2 = iobj;
   }
   else {	/* one object selected; find place to put from cursor position */

      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      saveobj = NULL;
      begflag = 1;

      for (libobj = (objinstptr *)ENDPART - 1; libobj >=
		(objinstptr *)objectdata->plist; libobj--) {

	 if ((*libobj)->type == OBJECT) {
 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->lowerleft.x
	        + ((*libobj)->thisobject->width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->lowerleft.y
	        + ((*libobj)->thisobject->height >> 1);
	    rangey = ((*libobj)->thisobject->height > 200) ? 
	        ((*libobj)->thisobject->height >> 1) : 100;

	    if (areastruct.save.y < ocenty + rangey && areastruct.save.y 
	         > ocenty - rangey) {
	       if (areastruct.save.x > ocentx) break;
	       saveobj = libobj;
	    }
	 }
      }

      if (libobj < (objinstptr *)objectdata->plist) {
	 /* insert at beginning or relocate to end */
	 if (areastruct.save.y > ocenty - rangey) {
            begflag = 0;
	    libobj++;
	 }
	 else if (saveobj != NULL) {
	    libobj = saveobj - 1;
	 }
	 else {
	    libobj = (objinstptr *)ENDPART;
	    while ((*(--libobj))->type != OBJECT);
	 }
      }

      for (testobj = libpage; testobj < libpage + *libpobjs; testobj++)
	 if (*testobj == exchobj) break;
      for (tobj2 = libpage; tobj2 < libpage + *libpobjs; tobj2++)
	 if (*tobj2 == (*libobj)->thisobject) break;

      iobj = *testobj;

      /* shift library list */

      if (tobj2 > testobj) {
	 for (sobj = testobj; sobj < tobj2; sobj++)
	    *sobj = *(sobj + 1);
         *(tobj2) = iobj;
      }
      else if (tobj2 != testobj) {
	 for (sobj = testobj; sobj > tobj2 + begflag; sobj--)
	    *sobj = *(sobj - 1);
         *(tobj2 + begflag) = iobj;
      }
   }

   objectdeselect();
   if (objectdata == areastruct.libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Make a duplicate of an object, put in the User Library */
/*--------------------------------------------------------*/

void copycat()
{
   short *newselect;
   objectptr *newobj, *curlib, oldobj;
   objinstptr libobj;
   int i;

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {

      libobj = SELTOOBJINST(newselect);
      oldobj = libobj->thisobject;

      /* generate new entry in user library */

      curlib = (objectptr *) realloc(areastruct.userlib,
	 (areastruct.userobjs + 1) * sizeof(objectptr));
      areastruct.userlib = curlib;
      newobj = areastruct.userlib + areastruct.userobjs;
      *newobj = (objectptr) malloc(sizeof(object));
      areastruct.userobjs++;

      /* give the new object a unique name */

      sprintf(_STR, "_%s", oldobj->name);
      checkname(*newobj);

      /* copy other object properties */

      (*newobj)->width = oldobj->width;
      (*newobj)->height = oldobj->height;
      (*newobj)->lowerleft.x = oldobj->lowerleft.x;
      (*newobj)->lowerleft.y = oldobj->lowerleft.y;
      (*newobj)->hidden = False;

      /* copy over all the elements of the original object */

      (*newobj)->parts = 0;
      (*newobj)->plist = (genericptr *)malloc(sizeof(genericptr));

      for (i = 0; i < oldobj->parts; i++) {
	 switch((*(oldobj->plist + i))->type) {
	    case(PATH): {
	       register pathptr *npath, cpath = TOPATH(oldobj->plist + i);
	       register genericptr *gpart;
	       NEW_PATH(npath, (*newobj));
	       (*npath)->style = cpath->style;
	       (*npath)->width = cpath->width;
	       (*npath)->parts = 0;
	       (*npath)->plist = (genericptr *)malloc(cpath->parts *
			sizeof(genericptr));
	       for (gpart = cpath->plist; gpart < cpath->plist + cpath->parts;
			gpart++) {
		  switch((*gpart)->type){
		     case ARC: {
			arcptr copyarc = TOARC(gpart);
			arcptr *newarc;
			NEW_ARC(newarc, (*npath));
			arccopy(*newarc, copyarc);
			(*npath)->parts++;
			} break;
		     case POLYGON: {
                        polyptr copypoly = TOPOLY(gpart);
                        polyptr *newpoly;
                        NEW_POLY(newpoly, (*npath));
                        polycopy(*newpoly, copypoly);
                        (*npath)->parts++;
                        } break;
                     case SPLINE: {
                        splineptr copyspline = TOSPLINE(gpart);
                        splineptr *newspline;
                        NEW_SPLINE(newspline, (*npath));
                        splinecopy(*newspline, copyspline);
                        (*npath)->parts++;
                        } break;
		  }
	       }	
	       } break;
	    case(ARC): {
	       register arcptr *narc, carc = TOARC(oldobj->plist + i);
	       NEW_ARC(narc, (*newobj));
	       arccopy(*narc, carc);
               } break;
      
	    case(POLYGON): {
	       register polyptr *npoly, cpoly;
	       register int j;

	       cpoly = TOPOLY(oldobj->plist + i);
	       NEW_POLY(npoly, (*newobj));
	       polycopy(*npoly, cpoly);
               } break;

	    case(SPLINE): {
	       register splineptr *nspl, cspl;
	       register int j;

	       cspl = TOSPLINE(oldobj->plist + i);
	       NEW_SPLINE(nspl, (*newobj));
	       splinecopy(*nspl, cspl);
      	       } break;

	    case(LABEL): {
	       register labelptr *nlabel, clabel;

	       clabel = TOLABEL(oldobj->plist + i);
	       NEW_LABEL(nlabel, (*newobj));
	       (*nlabel)->rotation = clabel->rotation;
	       (*nlabel)->color = clabel->color;
	       (*nlabel)->position.x = clabel->position.x;
	       (*nlabel)->position.y = clabel->position.y;
	       (*nlabel)->scale = clabel->scale;
	       (*nlabel)->justify = clabel->justify;
	       (*nlabel)->string = (char *)malloc((strlen(clabel->string) + 1) *
		      sizeof(char));
	       strcpy((*nlabel)->string, clabel->string);
               } break;
      
	    case(OBJECT): {
	       register objinstptr *ninst, cinst;

	       cinst = TOOBJINST(oldobj->plist + i);
	       NEW_OBJINST(ninst, (*newobj));
	       (*ninst)->rotation = cinst->rotation;
	       (*ninst)->color = cinst->color;
	       (*ninst)->position.x = cinst->position.x;
	       (*ninst)->position.y = cinst->position.y;
	       (*ninst)->scale = cinst->scale;
	       (*ninst)->thisobject = cinst->thisobject;
               } break;
         }
	 (*newobj)->parts++;
      }
   }

   composelib(USERLIB);
   objectdeselect();

   if (objectdata == areastruct.libtop[USERLIB])
      drawarea(NULL, objectdata, NULL);
   else changecat();
}

/*--------------------------------------------------------*/
/* ButtonPress handler during normal catalog viewing mode */
/*--------------------------------------------------------*/

XtEventHandler catbutton(w, mode, event)
  Widget w;
  caddr_t mode;
  XButtonEvent *event;
{
   short *newselect;
   objinstptr *newobject, *libobj;
   objectptr libpage = objectdata;
   short ocentx, ocenty, rangex, rangey, xdiff, ydiff, flag = 0;
   XPoint oldpos;

   if (event->button == Button1 || event->button == Button2) {

      window_to_user(event->x, event->y, &areastruct.save);
	
      for (libobj = (objinstptr *)objectdata->plist; libobj <
		(objinstptr *)objectdata->plist + objectdata->parts; libobj++) {
	 if ((*libobj)->type == OBJECT) {

 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->lowerleft.x
	        + ((*libobj)->thisobject->width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->lowerleft.y
	        + ((*libobj)->thisobject->height >> 1);

	    rangex = ((*libobj)->thisobject->width > 200) ? 
	        ((*libobj)->thisobject->width >> 1) : 100;
	    rangey = ((*libobj)->thisobject->height > 200) ? 
	        ((*libobj)->thisobject->height >> 1) : 100;

	    if (areastruct.save.x > ocentx - rangex && areastruct.save.x <
		   ocentx + rangex && areastruct.save.y < ocenty + rangey
		   && areastruct.save.y > ocenty - rangey) {

	       /* setup to move object around and draw the selected object */

               if (event->button == Button1) {

	          /* revert to old page */

	          objectdata = (pushes == 0) ? *(areastruct.pagelist + areastruct.page)
			   : *(pushlist + pushes - 1);

	          /* retrieve drawing window state and set position of object to
	             be correct in reference to that window */

	          snap(event->x, event->y, &oldpos);

   	          setvscale();
	    
	          snap(event->x, event->y, &areastruct.save);
	          xdiff = areastruct.save.x - oldpos.x;
	          ydiff = areastruct.save.y - oldpos.y;

                  /* collect all of the selected items */

	          for (newselect = areastruct.selectlist; newselect <
		     areastruct.selectlist + areastruct.selects; newselect++) {
		     NEW_OBJINST(newobject, objectdata);
		     objectdata->parts++;
		     (*newobject)->color = areastruct.color;
		     (*newobject)->rotation = TOOBJINST(libpage->plist +
			    *newselect)->rotation;
	             (*newobject)->scale = TOOBJINST(libpage->plist +
		            *newselect)->scale;
	             (*newobject)->thisobject = TOOBJINST(libpage->plist +
		            *newselect)->thisobject;
		     (*newobject)->position.x = TOOBJINST(libpage->plist +
		            *newselect)->position.x + xdiff;
		     (*newobject)->position.y = TOOBJINST(libpage->plist +
		            *newselect)->position.y + ydiff;
	             u2u_snap(&((*newobject)->position));
		     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);
		     if ((*newobject)->thisobject == (*libobj)->thisobject)
		        flag = 1;
	          }

                  /* add final object to the list of object instances */

	          if (!flag) {
		     NEW_OBJINST(newobject, objectdata);
                     objectdata->parts++;
	             (*newobject)->rotation = (*libobj)->rotation;
		     (*newobject)->color = areastruct.color;
	             (*newobject)->scale = (*libobj)->scale;
	             (*newobject)->thisobject = (*libobj)->thisobject;
                     (*newobject)->position.x = areastruct.save.x;
	             (*newobject)->position.y = areastruct.save.y; 

	             /* add this object to the list of selected items */

	             newselect = allocselect();
                     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);

	          }
	          if ((void *)mode == (void *)(1)) {

		     /* key "c" pressed for "copy" */

                     XDefineCursor(dpy, win, COPYCURSOR);
                     eventmode = COPY2_MODE;
                     XtAddEventHandler(areastruct.area, PointerMotionMask, False, drag,
                          NULL);
	          }
	          else {
                     eventmode = PRESS_MODE;
                     XtAddEventHandler(areastruct.area, Button1MotionMask, False,
		          drag, NULL);
	          }
                  catreturn();
               }

               /* If button2 was pressed, select and stay in the catalog. */
	       else {
	          short newinst = (short)(libobj - (objinstptr *)objectdata->plist);

	          /* (but ignore this object if it is already in the list of selects) */

	          for (newselect = areastruct.selectlist; newselect <
		        areastruct.selectlist + areastruct.selects; newselect++)
		     if (*newselect == newinst) break;
	          if (newselect == areastruct.selectlist + areastruct.selects) {

	             /* add this object to the list of selected items */

	             newselect = allocselect();
                     *newselect = newinst;

	             XcSetFunction(GXcopy);
	             XcSetForeground(SELECTCOLOR);
	             UDrawObject(*libobj, (*libobj)->thisobject, SINGLE, SELECTCOLOR);
	          }
	       }
	       break;
	    }
	 }
      }
   }
   /* If button3 was pressed, return without a selection. */

   else {
      if (areastruct.selects > 0) {
	 free(areastruct.selectlist);
         areastruct.selects = 0;
      }
      eventmode = NORMAL_MODE;
      catreturn();
   }
}

/*--------------------------------------*/
/* Change from one catalog to the other */
/*--------------------------------------*/

void changecat()
{
   if (objectdata == areastruct.libtop[LIBRARY]) {
      if ((areastruct.libtop[USERLIB])->parts > 0) {
	 objectdata = areastruct.libtop[USERLIB];
      }
      else {
	Wprintf("User library is empty");
	return;
      }
   }
   else if (objectdata == areastruct.libtop[USERLIB]) {
      if ((areastruct.libtop[LIBRARY])->parts > 0) {
	 objectdata = areastruct.libtop[LIBRARY];
      }
      else {
	Wprintf("Built-in library is empty");
	return;
      }
   }
   else return;

   (*areastruct.vscale = 0.5);
   areastruct.lowerleft->x = objectdata->lowerleft.x - 5;
   areastruct.lowerleft->y = objectdata->lowerleft.y + objectdata->height
	- (areastruct.height << 1) + 5;

   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawarea(NULL, objectdata, NULL);
   drawhbar(areastruct.scrollbarh, NULL, Number(1));
   drawvbar(areastruct.scrollbarv, NULL, Number(1));

   printname(objectdata);
}

/*--------------------------------------*/
/* Begin catalog viewing mode		*/
/*--------------------------------------*/

XtCallbackProc startcatalog(w, libmod, nulldata)
  Widget w;
  void *libmod;
  caddr_t nulldata;
{
   /* Can't look at a library with no objects in it. . . */
   /* But first try to switch to a library which contains something useful. */

   if (areastruct.libtop[USERLIB]->parts == 0) libmod = (void *)LIBRARY;
   if (areastruct.libtop[LIBRARY]->parts == 0) libmod = (void *)USERLIB;
   if (areastruct.libtop[(int)libmod]->parts == 0) {
      Wprintf("Both libraries are empty.");
      return;
   }

   /* If we're already in catalog mode. . . */

   if (eventmode == CATALOG_MODE) {
      if (objectdata == areastruct.libtop[(int)libmod]) {
	 if (libmod == USERLIB)
	    Wprintf("Already in User Library!");
	 else
	    Wprintf("Already in Library!");
      }
      else changecat();
      return;
   }
   else if (eventmode != NORMAL_MODE) return;

   eventmode = CATALOG_MODE;

   /* push previously edited object on push stack */

   if (pushes == 0)
      pushlist = (objectptr *) malloc(sizeof(objectptr));
   else
      pushlist = (objectptr *) realloc(pushlist,
         (pushes + 1) * sizeof(objectptr));
   *(pushlist + pushes) = objectdata;
   pushes++;

   /* set library as new object */

   objectdata = areastruct.libtop[(int)libmod];

   /* save the graphics state of areastruct and create a new one */

   savewidth = areastruct.wirewidth[areastruct.page];

   setvscale();
   (*areastruct.vscale) = 0.5;
   areastruct.lowerleft->x = objectdata->lowerleft.x - 5;
   areastruct.lowerleft->y = objectdata->lowerleft.y + objectdata->height
	- (areastruct.height << 1) + 5;
   areastruct.wirewidth[areastruct.page] = 2.0;

   /* unselect all current selects */

   if (areastruct.selects > 0) {
      free(areastruct.selectlist);
      areastruct.selects = 0;
   }

   /* draw the new screen */

   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawarea(w, objectdata, nulldata);
   drawhbar(areastruct.scrollbarh, NULL, Number(1));
   drawvbar(areastruct.scrollbarv, NULL, Number(1));

   printname(objectdata);
}


/*-------------------------------------------------------------------------*/
