/*-----------------------------------------------------------------------*/
/* text.c --- text processing routines for xcircuit		 	 */
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	 */
/*-----------------------------------------------------------------------*/

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Display *dpy;
extern Clientdata areastruct;
extern Globaldata xobjs;
extern short textpos, textend;
extern short fontcount;
extern fontinfo *fonts;
extern short eventmode;
extern int *appcolors;
extern colorindex *colorlist;
extern char _STR[150];
extern objectpair *pushlist;

/* Global value of distance between characters in the font catalog */
short del;

/*----------------------------------------------------------------------*/
/* Determine if a label contains a parameter.				*/
/*----------------------------------------------------------------------*/

Boolean hasparameter(labelptr curlabel)
{
   stringpart *chrptr;

   for (chrptr = curlabel->string; chrptr != NULL; chrptr = chrptr->nextpart)
      if (chrptr->type == PARAM_START)
         return True;

   return False;
}

/*----------------------------------------------------------------------*/
/* Insert a new segment into a string					*/
/*----------------------------------------------------------------------*/

stringpart *makesegment(stringpart **strhead, stringpart *before)
{
   stringpart *newptr, *lastptr, *nextptr;

   newptr = (stringpart *)malloc(sizeof(stringpart));
   newptr->data = NULL;

   if (before == *strhead) {	/* insert at beginning */
      newptr->nextpart = *strhead;
      *strhead = newptr;
   }
   else {				/* otherwise */
      for(lastptr = *strhead; lastptr != NULL;) {
	 nextptr = nextstringpart(lastptr, NORMINST);
	 if (nextptr == before) {
	    if (lastptr->type == PARAM_START) {
	       oparamptr obs = NULL;
	       int paramno = lastptr->data.paramno;
	       if (NORMINST->params != NULL)
	          obs = NORMINST->params[paramno];
	       if (obs == NULL)
		  obs = NORMINST->thisobject->params[paramno];
	       if (obs == NULL) {
		  Wprintf("Error:  Bad parameter");
		  return NULL;
	       }
	       obs->parameter.string = newptr;
	    }
	    else {
	       lastptr->nextpart = newptr;
	    }
	    newptr->nextpart = nextptr;
	    break;
         }
	 else if (lastptr->nextpart == before && lastptr->type == PARAM_START) {
	    lastptr->nextpart = newptr;
	    newptr->nextpart = before;
	    break;
	 }
	 lastptr = nextptr;
      }
   }
   return newptr;
}

/*----------------------------------------------------------------------*/
/* Split a string across text segments					*/
/*----------------------------------------------------------------------*/

stringpart *splitstring(int tpos, stringpart **strtop, objinstptr localdata)
{
   int locpos, slen;
   stringpart *newpart, *ipart;

   ipart = findstringpart(tpos, &locpos, *strtop, localdata);
   if (locpos != 0) {        /* split the string */
      newpart = makesegment(strtop, ipart);
      newpart->type = TEXT_STRING;
      newpart->data.string = ipart->data.string;
      slen = strlen(newpart->data.string) - locpos;
      ipart->data.string = (u_char *)malloc(slen + 1);
      strncpy(ipart->data.string, newpart->data.string + locpos, slen  + 1);
      *(newpart->data.string + locpos) = '\0';
   }
   else newpart = ipart;

   return newpart;
}

/*----------------------------------------------------------------------*/
/* Get the next string part, linking to a paramter if necessary		*/
/*----------------------------------------------------------------------*/

stringpart *nextstringpart(stringpart *strptr, objinstptr thisinst)
{
   stringpart *nextptr = strptr->nextpart;

   if (strptr->type == PARAM_START)
      nextptr = linkstring(thisinst, strptr);
   else if (strptr->type == PARAM_END)
      strptr->nextpart = NULL;

   return nextptr;
}


/*----------------------------------------------------------------------*/
/* Remove a string part from the string					*/
/*----------------------------------------------------------------------*/

stringpart *deletestring(stringpart *dstr, stringpart **strtop, objinstptr thisinst)
{
   stringpart *strptr, *nextptr;
   int paramno;

   if (dstr == *strtop)
      *strtop = dstr->nextpart;
   else {
      strptr = *strtop;
      while (strptr != NULL) {
	 nextptr = nextstringpart(strptr, thisinst);
	 if (nextptr == dstr) break;
	 strptr = nextptr;
      }
      if (strptr == NULL)
	 return NULL;

      /* If this is the begining of a parameter, then we have to figure */
      /* out if it's an instance or a default, and change the pointer	*/
      /* to the parameter in the parameter list, accordingly.		*/

      else if ((strptr->type == PARAM_START) && (thisinst != NULL)) {
	 paramno = strptr->data.paramno;
	 if (thisinst->params == NULL || thisinst->params[paramno] == NULL)
	    thisinst->thisobject->params[paramno]->parameter.string =
			dstr->nextpart;
	 else
	    thisinst->params[paramno]->parameter.string = dstr->nextpart;
      }
      /* If this is the end of a parameter, we have to link the		*/
      /* PARAM_START, not the PARAM_END, which has already been nulled.	*/ 
      else if (strptr->type == PARAM_END) {
	 for (strptr = *strtop; strptr != NULL; strptr = strptr->nextpart) {
	    if (strptr->nextpart == dstr) {
	       strptr->nextpart = dstr->nextpart;
	       break;
	    }
         }
      }
      else
	 strptr->nextpart = dstr->nextpart;
   }
   if (dstr->type == TEXT_STRING) free(dstr->data.string);
   free(dstr);
}

/*----------------------------------------------------------------------*/
/* Merge string parts at boundary, if parts can be legally merged	*/
/* If the indicated string part is text and the part following the	*/
/* indicated string part is also text, merge the two.  The indicated	*/
/* string part is returned, and the following part is freed.		*/
/*----------------------------------------------------------------------*/

stringpart *mergestring(stringpart *firststr)
{
   stringpart *nextstr = firststr->nextpart;
   if (nextstr != NULL) {
      if (firststr->type == TEXT_STRING && nextstr->type == TEXT_STRING) {
         firststr->nextpart = nextstr->nextpart;
         firststr->data.string = (char *)realloc(firststr->data.string,
		1 + strlen(firststr->data.string) + strlen(nextstr->data.string));
         strcat(firststr->data.string, nextstr->data.string);
         free(nextstr->data.string);
         free(nextstr);
         return firststr;
      }
   }
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Link a parameter to a string						*/
/*----------------------------------------------------------------------*/

stringpart *linkstring(objinstptr localdata, stringpart *strstart)
{
   int paramno;
   stringpart *nextptr = NULL, *tmpptr;

   if (strstart->type != PARAM_START) return NULL;

   paramno = strstart->data.paramno;

   /* In case of no calling instance, always get the default from the	*/
   /* current page object.						*/

   if (localdata == NULL) {
      if (objectdata->num_params > paramno)
         nextptr = objectdata->params[paramno]->parameter.string;
      else
	 return NULL;  /* There is no parameter here. */
   }
   else {
      if (localdata->params != NULL)
	 if (localdata->params[paramno] != NULL)
            nextptr = localdata->params[paramno]->parameter.string;

      /* If there's no instance, substitute the default parameter */

      if (nextptr == NULL)
         nextptr = localdata->thisobject->params[paramno]->parameter.string;
   }

   /* If the parameter exists, link the end of the parameter back to	*/
   /* the calling string.						*/

   if (nextptr != NULL) {
      tmpptr = nextptr;
      while (tmpptr->type != PARAM_END)
	 if ((tmpptr = tmpptr->nextpart) == NULL)
	    return NULL;
      tmpptr->nextpart = strstart->nextpart;
      return nextptr;
   }
}

/*----------------------------------------------------------------------*/
/* Find the last font used prior to the indicated text position		*/
/*----------------------------------------------------------------------*/

int findcurfont(int tpos, stringpart *strtop, objinstptr thisinst)
{
   stringpart *curpos;
   int cfont = -1;
   stringpart *strptr;

   curpos = findstringpart(tpos, NULL, strtop, thisinst);
   for (strptr = strtop; strptr != curpos; strptr = nextstringpart(strptr, thisinst))
      if (strptr->type == FONT_NAME)
         cfont = strptr->data.font;

   return cfont;
}

/*----------------------------------------------------------------------*/
/* Return a local position and stringpart for "tpos" positions into	*/
/* the indicated string.  Position and stringpart are for the character */
/* or command immediately preceding "tpos"				*/
/*----------------------------------------------------------------------*/

stringpart *findstringpart(int tpos, int *locpos, stringpart *strtop,
	objinstptr localdata)
{
   stringpart *strptr = strtop;
   int testpos = 0, tmplen;

   for (strptr = strtop; strptr != NULL; strptr = nextstringpart(strptr, localdata)) {
      if (strptr->type == TEXT_STRING) {
	 tmplen = strlen(strptr->data.string);
	 if (testpos + tmplen > tpos) {
	    if (locpos != NULL) *locpos = (tpos - testpos);
	    return strptr;
	 }
	 else testpos += tmplen - 1;
      }
      if (locpos != NULL) *locpos = -1;
      if (testpos >= tpos) return strptr;

      testpos++;
   }
   return NULL;
}

/*----------------------------------------------------------------------*/
/* The following must be in an order matching the "Text string part	*/
/* types" defined in xcircuit.h. 					*/
/*----------------------------------------------------------------------*/

static char *nonprint[] = {
	"Text", "Subscript", "Superscript", "Normalscript",
	"Underline", "Overline", "Noline",
	"Tab_Stop", "Tab_Forward", "Tab_Backward",
	"Halfspace", "Quarterspace", "<Return>",
	"Font", "Scale", "Color", "Kern", 
        "Parameter", ">", "Error"};

/*----------------------------------------------------------------------*/
/* charprint():  							*/
/* Write a printable version of the character or command at the 	*/
/* indicated string part and position.					*/
/*----------------------------------------------------------------------*/

void charprint(char *sout, stringpart *strptr, int locpos)
{
   char sc;
 
   switch (strptr->type) {
      case TEXT_STRING:
         if (locpos > strlen(strptr->data.string)) {
	    strcpy(sout, "<ERROR>");
	 }
         else sc = *(strptr->data.string + locpos);
         if (isprint(sc))
	    sprintf(sout, "%c", sc);
         else
	    sprintf(sout, "/%03o", (u_char)sc);
	 break;
      case FONT_NAME:
	 sprintf(sout, "Font=%s", fonts[strptr->data.font].psname);
	 break;
      case FONT_SCALE:
	 sprintf(sout, "Scale=%3.2f", strptr->data.scale);
	 break;
      case KERN:
	 sprintf(sout, "Kern=(%d,%d)", strptr->data.kern[0], strptr->data.kern[1]);
	 break;
      case PARAM_START:
	 sprintf(sout, "Parameter(%d)<", strptr->data.paramno + 1);
	 break;
      default:
         strcpy(sout, nonprint[strptr->type]);
	 break;
   }
}

/*----------------------------------------------------------------------*/
/* Print a string (allocates memory for the string; must be freed by	*/
/* the calling routine).						*/
/*----------------------------------------------------------------------*/

char *xcstringtostring(stringpart *strtop, objinstptr localdata, Boolean textonly)
{
   stringpart *strptr;
   int pos = 0, locpos;
   char *sout;

   sout = (char *)malloc(1);
   sout[0] = '\0';

   while ((strptr = findstringpart(pos++, &locpos, strtop, localdata)) != NULL) {
      if (!textonly || strptr->type == TEXT_STRING) {
         charprint(_STR, strptr, locpos);
         sout = (char *)realloc(sout, strlen(sout) + strlen(_STR) + 1);
         strcat(sout, _STR);
      }
      /* Overbar on schematic names is translated to logical-NOT ("!") */
      else if (textonly && strptr->type == OVERLINE) {
         sout = (char *)realloc(sout, strlen(sout) + 2);
         strcat(sout, "!");
      }
   }
   return sout;
}

/*----------------------------------------------------------------------*/
/* Wrappers for xcstringtostring():					*/
/*  stringprint() includes information on text controls appropriate	*/
/*  for printing in the message window (charreport())			*/
/*----------------------------------------------------------------------*/

char *stringprint(stringpart *strtop, objinstptr localdata)
{
   xcstringtostring(strtop, localdata, False);
}

/*----------------------------------------------------------------------*/
/*  textprint() excludes text controls, resulting in a string 		*/
/*  appropriate for netlist information.				*/
/*----------------------------------------------------------------------*/

char *textprint(stringpart *strtop, objinstptr localdata)
{
   xcstringtostring(strtop, localdata, True);
}

/*----------------------------------------------------------------------*/
/* Test equivalence of the text string parts of a label with the	*/
/* indicated char * string.						*/
/*    If "exact" is True, requires an exact match, otherwise requires	*/
/*    that the label only match text to the length of text.		*/
/*----------------------------------------------------------------------*/

int textcompx(stringpart *string, char *text, Boolean exact, objinstptr localdata)
{
   stringpart *strptr;
   char *tptr = text;
   char *sptr;
   int rval, llen = strlen(text), slen;

   for (strptr = string; strptr != NULL; strptr = nextstringpart(strptr, localdata)) {
      if (strptr->type == TEXT_STRING) {
	 sptr = strptr->data.string;
	 slen = min(strlen(sptr), llen);
	 llen -= slen;
	 if (!exact && (rval = strncmp(sptr, tptr, slen)))
	    return rval;
	 else if (exact && (rval = strcmp(sptr, tptr)))
	    return rval;
	 else if (!exact && (llen == 0))
	    return 0;
	 else
	    tptr += slen;
      }
   }
   return 0;
}

/*----------------------------------------------------------------------*/
/* Wrappers for textcompx(), equivalent to strcmp() and strncmp().	*/
/*----------------------------------------------------------------------*/

int textcomp(stringpart *string, char *text, objinstptr localdata)
{
   textcompx(string, text, True, localdata);
}

/*----------------------------------------------------------------------*/

int textncomp(stringpart *string, char *text, objinstptr localdata)
{
   textcompx(string, text, False, localdata);
}

/*----------------------------------------------------------------------*/
/* Test equivalence of two label strings				*/
/*----------------------------------------------------------------------*/

int stringcomp(stringpart *string1, stringpart *string2)
{
   stringpart *strptr1, *strptr2;

   for (strptr1 = string1, strptr2 = string2; strptr1 != NULL && strptr2 != NULL;
		strptr1 = strptr1->nextpart, strptr2 = strptr2->nextpart) {
      if (strptr1->type != strptr2->type)
	 return 1;
      else {
         switch (strptr1->type) {
	    case TEXT_STRING:
	       if (strcmp(strptr1->data.string, strptr2->data.string))
		  return 1;
	       break;
	    case FONT_SCALE:
	       if (strptr1->data.scale != strptr2->data.scale) return 1;
	       break;
	    case FONT_COLOR:
	       if (strptr1->data.color != strptr2->data.color) return 1;
	       break;
	    case FONT_NAME:
	       if (strptr1->data.font != strptr2->data.font) return 1;
	       break;
	    case KERN:
	       if (strptr1->data.kern[0] != strptr2->data.kern[0] ||
		   strptr1->data.kern[1] != strptr2->data.kern[1]) return 1;
	       break;
	 }
      }
   }
   return 0;
}

/*----------------------------------------------------------------------*/
/* Find the number of parts in a string	(excluding parameter contents)	*/
/*----------------------------------------------------------------------*/

int stringparts(stringpart *string)
{
   stringpart *strptr;
   int ptotal = 0;

   for (strptr = string; strptr != NULL; strptr = strptr->nextpart)
      ptotal++;

   return ptotal;
}

/*----------------------------------------------------------------------*/
/* Compute the total character length of a string 			*/
/*    If "doparam" is True, include parameter contents.			*/
/*----------------------------------------------------------------------*/

int stringlength(stringpart *string, Boolean doparam, objinstptr thisinst)
{
   stringpart *strptr;
   int ctotal = 0;

   for (strptr = string; strptr != NULL; strptr = (doparam) ?
		nextstringpart(strptr, thisinst) : strptr->nextpart) {
      if (strptr->type == TEXT_STRING)
	 ctotal += strlen(strptr->data.string);
      else
	 ctotal++;
   }

   return ctotal;
}

/*----------------------------------------------------------------------*/
/* Copy the contents of a string (excluding parameter contents)		*/
/*----------------------------------------------------------------------*/

stringpart *stringcopy(stringpart *string)
{
   stringpart *strptr, *newpart, *newtop = NULL;

   for (strptr = string; strptr != NULL; strptr = strptr->nextpart) {
      newpart = makesegment(&newtop, NULL);
      newpart->type = strptr->type;
      if (strptr->type == TEXT_STRING) {
	 newpart->data.string = (char *)malloc(1 + strlen(strptr->data.string));
	 strcpy(newpart->data.string, strptr->data.string);
      }
      else
         newpart->data = strptr->data;
   }
   return newtop;
}

/*----------------------------------------------------------------------*/
/* Copy the contents of a string, embedding parameter contents		*/
/*----------------------------------------------------------------------*/

stringpart *stringcopyall(stringpart *string, objinstptr thisinst)
{
   stringpart *strptr, *newpart, *newtop, *topend;

   for (strptr = string; strptr != NULL;
		strptr = nextstringpart(strptr, thisinst)) {
      newpart = (stringpart *)malloc(sizeof(stringpart));
      newpart->type = strptr->type;
      newpart->nextpart = NULL;
      if (strptr == string) newtop = newpart;
      else topend->nextpart = newpart;
      topend = newpart;
      if (strptr->type == TEXT_STRING) {
	 newpart->data.string = (char *)malloc(1 + strlen(strptr->data.string));
	 strcpy(newpart->data.string, strptr->data.string);
      }
      else
         newpart->data = strptr->data;
   }
   return newtop;
}

/*----------------------------------------------------------------------*/
/* Copy the contents of a saved string with embedded parameter contents	*/
/* back to the string and the instance parameters.			*/
/*----------------------------------------------------------------------*/

stringpart *stringcopyback(stringpart *string, objinstptr thisinst)
{
   stringpart *strptr, *newpart, *curend, *rettop, *curtop, *savend;
   int paramno = -1;
   oparamptr pparam;

   for (strptr = string; strptr != NULL; strptr = strptr->nextpart) {

      newpart = (stringpart *)malloc(sizeof(stringpart));
      newpart->type = strptr->type;
      newpart->nextpart = NULL;

      if (strptr == string) rettop = newpart;	/* initial segment */
      else curend->nextpart = newpart;  	/* append segment to label */

      if (curend->type == PARAM_START) {
	 paramno = curend->data.paramno;
	 curtop = newpart;
	 savend = curend;
      }
      else if (curend->type == PARAM_END) {
	 curend->nextpart = NULL;
	 savend->nextpart = newpart;
      }
      curend = newpart;

      if (strptr->type == TEXT_STRING) {
	 newpart->data.string = (char *)malloc(1 + strlen(strptr->data.string));
	 strcpy(newpart->data.string, strptr->data.string);
      }
      else if (strptr->type == PARAM_END) {
	 if (paramno >= 0) {
	    if ((thisinst->params) != NULL && (thisinst->params[paramno] != NULL))
	       pparam = thisinst->params[paramno];
	    else 
	       pparam = thisinst->thisobject->params[paramno];

	    freelabel(pparam->parameter.string);
	    pparam->parameter.string = curtop;
	    paramno = -1;
	 }
         else {
	    fprintf(stderr, "Error:  Bad parameter in stringcopyback()\n");
	 }
      }
      else
         newpart->data = strptr->data;
   }

   /* tie up loose ends, if necessary */
   if ((curend != NULL) && (curend->type == PARAM_END))
      savend->nextpart = NULL;

   return rettop;
}

/*---------------------------------------------------------------------*/
/* draw a single character at 0, 0 using current transformation matrix */
/*---------------------------------------------------------------------*/

short UDrawChar(localdata, code, styles, ffont, groupheight, passcolor)
  objinstptr 	localdata;
  u_char	code;
  short 	styles, ffont;
  int		groupheight;
  int		passcolor;
{
   objectptr drawchar;
   XPoint alphapts[2];
   short  localwidth;
   objinst charinst;

   alphapts[0].x = 0;
   alphapts[0].y = 0;
   charinst.type = OBJECT;
   charinst.color = DEFAULTCOLOR;
   charinst.rotation = 0;
   charinst.scale = fonts[ffont].scale;
   charinst.position = alphapts[0];
   
   /* get proper font and character */

   drawchar = fonts[ffont].encoding[(u_char)code];
   charinst.thisobject = drawchar;

   localwidth = (drawchar->lowerleft.x + drawchar->width) * fonts[ffont].scale;

   if ((fonts[ffont].flags & 0x22) == 0x22) { /* font is derived and italic */
      USlantCTM(DCTM, 0.25);  		/* premultiply by slanting function */
   }

   if (!(styles & 64)) {
      
      UDrawObject(&charinst, drawchar, SINGLE, passcolor);

      /* under- and overlines */
      if (styles & 8)
         alphapts[0].y = alphapts[1].y = -6;
      else if (styles & 16)
         alphapts[0].y = alphapts[1].y = groupheight + 4;
      if (styles & 24) {
         alphapts[0].x = 0; alphapts[1].x = localwidth;
         UDrawSimpleLine(localdata, &alphapts[0], &alphapts[1]);
      }
   }
   return localwidth;
}

/*----------------------------------------------------------------------*/
/* Draw and entire string, including parameter substitutions		*/
/*----------------------------------------------------------------------*/

void UDrawString(objinstptr localdata, labelptr drawlabel, int passcolor)
{
   stringpart *strptr, *tmpptr;
   char *textptr;
   short  fstyle, ffont, tmpjust, baseline;
   int    paramno, pos, group = 0;
   int    defaultcolor, curcolor, scolor;
   short  oldx, oldy, oldfont, oldstyle;
   float  tmpscale = 1.0;
   float  tmpthick = xobjs.pagelist[areastruct.page]->wirewidth;
   XPoint newpoint, bboxin[2], bboxout[2];
   Boolean xm, ym;
   TextExtents tmpext;
   short *tabstops = NULL;
   short tabno, numtabs = 0;

   /* Instance from which we get the parameter list */
   objinstptr pinst = (localdata == areastruct.topobject) ? NORMINST : localdata;

   if (fontcount == 0) return;

   if (passcolor == DOSUBSTRING)
      defaultcolor = curcolor = drawlabel->color;
   else
      defaultcolor = curcolor = passcolor;

   if (defaultcolor != DOFORALL) {
      if (drawlabel->color != DEFAULTCOLOR)
	 curcolor = drawlabel->color;
      else
	 curcolor = defaultcolor;
      XTopSetForeground(curcolor);
   }

   /* calculate the transformation matrix for this object */
   /* in natural units of the alphabet vectors		  */
   /* (conversion to window units)			  */

   UPushCTM();
   UPreMultCTM(DCTM, drawlabel->position, drawlabel->scale, drawlabel->rotation);

   /* check for flip invariance; recompute CTM and justification if necessary */

   tmpjust = flipadjust(drawlabel->justify);

   /* "natural" (unscaled) length */
   tmpext = ULength(drawlabel->string, pinst, 0.0, 0, NULL);

   newpoint.x = (tmpjust & NOTLEFT ?
       (tmpjust & RIGHT ? -tmpext.width : -tmpext.width >> 1) : 0);
   newpoint.y = (tmpjust & NOTBOTTOM ?
       (tmpjust & TOP ? -tmpext.ascent : -(tmpext.ascent + tmpext.base) >> 1)
		: -tmpext.base);

#ifdef SCHEMA
   /* Pinlabels have an additional offset spacing to pad */
   /* them from the circuit point to which they attach.  */

   if (drawlabel->pin) {
      if (!areastruct.schemon)
	 goto enddraw;	/* Don't draw pins when schematic capture is off */
      else {
	 pinadjust(tmpjust, &(newpoint.x), &(newpoint.y), 1);
         UDrawX(drawlabel);
      }
   }
#endif

   oldx = newpoint.x;
   oldy = newpoint.y;
   baseline = newpoint.y;

   /* do quick calculation on bounding box; don't draw if off-screen */

   bboxin[0].x = newpoint.x;
   bboxin[0].y = newpoint.y + tmpext.descent;
   bboxin[1].x = newpoint.x + tmpext.width;
   bboxin[1].y = newpoint.y + tmpext.ascent;
   UTransformbyCTM(DCTM, bboxin, bboxout, 2);
   xm = (bboxout[0].x < bboxout[1].x) ? 0 : 1;
   ym = (bboxout[0].y < bboxout[1].y) ? 0 : 1;

   if (bboxout[xm].x < areastruct.width && bboxout[ym].y < areastruct.height &&
       bboxout[!xm].x > 0 && bboxout[!ym].y > 0) {

       pos = 0;
       for (strptr = drawlabel->string; strptr != NULL;
		strptr = nextstringpart(strptr, pinst)) {

	  /* All segments other than text cancel any	*/
	  /* existing overline/underline in effect.	*/

	  if (strptr->type != TEXT_STRING)
	     fstyle &= 0xfc7;

          /* deal with each text segment type */

	  switch(strptr->type) {
	     case FONT_NAME:
		ffont = strptr->data.font;
		fstyle = 0;		   /* style reset by font change */
	        if (oldy == newpoint.y) {  /* set top-level font and style */
	           oldfont = ffont;
	           oldstyle = fstyle;
	        }
		
		/* simple boldface technique for derived fonts */

		xobjs.pagelist[areastruct.page]->wirewidth =
   		   ((fonts[ffont].flags & 0x21) == 0x21) ?  4.0 : 2.0;
		break;

	     case FONT_SCALE:
		tmpscale = strptr->data.scale;
		break;

	     case KERN:
	        newpoint.x += strptr->data.kern[0];
	        newpoint.y += strptr->data.kern[1];
		break;
		
	     case FONT_COLOR:
		if (defaultcolor != DOFORALL) {
		   if (strptr->data.color != DEFAULTCOLOR)
		      curcolor = colorlist[strptr->data.color].color.pixel;
		   else {
		      if (curcolor != DEFAULTCOLOR) {
			 XTopSetForeground(defaultcolor);
		      }
		      curcolor = DEFAULTCOLOR;
		   }
		}
		break;

	     case TABBACKWARD:	/* find first tab value with x < xtotal */
	        for (tabno = numtabs - 1; tabno >= 0; tabno--) {
	           if (tabstops[tabno] < newpoint.x) {
		      newpoint.x = tabstops[tabno];
		      break;
	           }
	        }
	        break;

	     case TABFORWARD:	/* find first tab value with x > xtotal */
	        for (tabno = 0; tabno < numtabs; tabno++) {
	           if (tabstops[tabno] > newpoint.x) {
		      newpoint.x = tabstops[tabno];
		      break;
	           }
	        }
	        break;

	     case TABSTOP:
	        numtabs++;
	        if (tabstops == NULL) tabstops = (short *)malloc(sizeof(short));
	        else tabstops = (short *)realloc(tabstops, numtabs * sizeof(short));
	        tabstops[numtabs - 1] = newpoint.x;
		break;

	     case RETURN:
		baseline -= BASELINE * tmpscale;
	        newpoint.y = baseline;
		newpoint.x = oldx;
		break;
	
	     case SUBSCRIPT:
	        tmpscale *= SUBSCALE; 
	        newpoint.y -= (short)((TEXTHEIGHT >> 1) * tmpscale);
		break;

	     case SUPERSCRIPT:
	        tmpscale *= SUBSCALE;
	        newpoint.y += (short)(TEXTHEIGHT * tmpscale);
		break;

	     case NORMALSCRIPT:
	        tmpscale = 1.0;
	        ffont = oldfont;	/* revert to top-level font and style */
	        fstyle = oldstyle;
	        newpoint.y = oldy;
		break;

	     case UNDERLINE:
	        fstyle |= 8;
		break;

	     case OVERLINE:
		if (strptr->nextpart != NULL && strptr->nextpart->type == TEXT_STRING) {
		   objectptr charptr;
		   int tmpheight;

		   group = 0;
		   for (textptr = strptr->nextpart->data.string; *textptr != '\0';
				textptr++) {
		      charptr = fonts[ffont].encoding[*textptr];
		      tmpheight = (int)((float)charptr->height * fonts[ffont].scale);
		      if (group < tmpheight) group = tmpheight;
		   }
	           fstyle |= 16;
		}
		break;

	     case NOLINE:
		break;

	     case HALFSPACE: case QTRSPACE: {
		short addx = UDrawChar(localdata, (u_char)32, fstyle, ffont, group,
			curcolor);
		newpoint.x += addx >> ((strptr->type == HALFSPACE) ? 1 : 2);
		} break;

	     case TEXT_STRING:
		for (textptr = strptr->data.string; *textptr != '\0'; textptr++) {
		   pos++;
		   UPushCTM();
		   UPreMultCTM(DCTM, newpoint, tmpscale, 0);

	           /* Special case of selection:  only substring is	*/
		   /* drawn in the selection color.			*/

		   scolor = curcolor;
	           if (passcolor == DOSUBSTRING) {
	              if (pos <= textpos && pos > textend)
		         scolor = SELECTCOLOR;
		      else if (pos > textpos) {
      		         XTopSetForeground(curcolor);
		      }
	           }
	           newpoint.x += UDrawChar(localdata, *textptr, fstyle, ffont,
			group, scolor) * tmpscale;

		   UPopCTM();
		}
		pos--;
		break;
	  }
	  pos++;
       }
   }

enddraw:

   if (tabstops != NULL) free(tabstops);

   /* Pop the string transformation matrix */

   UPopCTM();

   if ((defaultcolor != DOFORALL) && (passcolor != curcolor))
      XTopSetForeground(passcolor);

   xobjs.pagelist[areastruct.page]->wirewidth = tmpthick;
}

/*----------------------------------------------------------------------*/
/* Compute the actual length of a string or portion thereof.		*/
/*----------------------------------------------------------------------*/

TextExtents ULength(stringpart *string, objinstptr localdata, float newscale,
	short dostop, XPoint *tbreak)
{
   float oldscale, strscale, locscale = 1.0, xtotal = 0.5;
   float lasttotal = xtotal;
   stringpart *strptr;
   u_char *textptr;
   objectptr *somebet = NULL, chptr;
   short locpos = 0, lastpos = 0;
   float ykern = 0.0;
   TextExtents retext;
   short *tabstops = NULL;
   short tabno, numtabs = 0;

   retext.width = retext.ascent = retext.descent = retext.base = 0;
   if (fontcount == 0) return retext;

   if (newscale > 0.0) strscale = UTopScale() * newscale;
   else strscale = 1.0;
     
   oldscale = strscale;

   for (strptr = string; strptr != NULL;
		strptr = nextstringpart(strptr, localdata)) {
      switch (strptr->type) {
	 case SUPERSCRIPT:
	    strscale *= SUBSCALE;
	    ykern += TEXTHEIGHT * strscale / 2.0;
	    break;
	 case SUBSCRIPT:
	    strscale *= SUBSCALE;
	    ykern -= TEXTHEIGHT * strscale / 2.0;
	    break;
	 case NORMALSCRIPT:
	    strscale = oldscale;
	    ykern = 0.0;
	    break;
	 case RETURN:
	    retext.base -= BASELINE * strscale;
	    if (!dostop)
	       retext.width = max(retext.width, xtotal);
	    xtotal = 0.5;
	    break;
	 case HALFSPACE: 
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale / 2;
	    break;
	 case QTRSPACE:
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x) 
			* locscale * strscale / 4;
	    break;
	 case TABBACKWARD:	/* find first tab value with x < xtotal */
	    for (tabno = numtabs - 1; tabno >= 0; tabno--) {
	       if (tabstops[tabno] < xtotal) {
		  xtotal = tabstops[tabno];
		  break;
	       }
	    }
	    break;
	 case TABFORWARD:	/* find first tab value with x > xtotal */
	    for (tabno = 0; tabno < numtabs; tabno++) {
	       if (tabstops[tabno] > xtotal) {
		  xtotal = tabstops[tabno];
		  break;
	       }
	    }
	    break;
	 case TABSTOP:
	    numtabs++;
	    if (tabstops == NULL) tabstops = (short *)malloc(sizeof(short));
	    else tabstops = (short *)realloc(tabstops, numtabs * sizeof(short));
	    tabstops[numtabs - 1] = xtotal;
	    break;
	 case FONT_SCALE:
	    strscale = strptr->data.scale;
   	    if (newscale > 0.0)
	       strscale *= UTopScale();
	    break;
	 case KERN:
	    xtotal += strptr->data.kern[0];
	    ykern += strptr->data.kern[1];
	    break;
	 case FONT_NAME:
	    somebet = fonts[strptr->data.font].encoding;
	    locscale = fonts[strptr->data.font].scale;
	    break;
	 case TEXT_STRING:
	    for (textptr = strptr->data.string; *textptr != '\0'; textptr++) {
               if (dostop && (locpos >= dostop)) break;
	       locpos++;

	       chptr = (*(somebet + *textptr));
	       xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale;
	       retext.ascent = max(retext.ascent, (short)(retext.base + ykern +
			(float)((chptr->height + chptr->lowerleft.y)
			* locscale * strscale)));
	       retext.descent = min(retext.descent, (short)(retext.base + ykern +
			(float)(chptr->lowerleft.y * locscale * strscale)));

               if (tbreak != NULL) 
	          if ((xtotal > tbreak->x) && (retext.base <= tbreak->y)) break;
               lasttotal = xtotal;
	       lastpos = locpos;
	    }
	    break;
      }
      if (strptr->type != TEXT_STRING) locpos++;
      if (dostop && (locpos >= dostop)) break;
   }
   if (tabstops != NULL) free(tabstops);

   /* special case: return character position in retext.width */
   if (tbreak != NULL) {
      int slen = stringlength(string, True, localdata);
      if ((tbreak->x - lasttotal) < (xtotal - tbreak->x))
	 locpos = lastpos + 1;
      if (locpos < 1) locpos = 1;
      else if (locpos > slen) locpos = slen;
      retext.width = locpos;
      return retext;
   }
   retext.width = max(retext.width, xtotal);
   return retext;
}

/*----------------------------------------------------------------------*/
/* low-level routines for drawing and erasing labels			*/
/*----------------------------------------------------------------------*/

void undrawtextsimple(labelptr settext)
{
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XTopSetForeground(BACKGROUND);
   UDrawString(areastruct.topobject, settext, DOFORALL);
}

/*----------------------------------------------------------------------*/

void redrawtextsimple(labelptr settext)
{
   UDrawString(NORMINST, settext, settext->color);
}

/*----------------------------------------------------------------------*/
/* Redraw all labels in the current object which contain the same	*/
/* parameter(s) as the indicated label.					*/
/* (It's easier not to bother to check for the same parameter, as there	*/
/* are typically so few parameters in an object that the extra compute	*/
/* and draw time is negligible.)					*/
/*									*/
/* Function pointer (undrawtextsimple or redrawtextsimple) indicates	*/
/* which function to call on this text label.				*/
/*----------------------------------------------------------------------*/

void drawtextandupdate(labelptr curlabel, void (*function)(labelptr))
{
   genericptr *pgen;
   labelptr slab;

   for (pgen = objectdata->plist; pgen < objectdata->plist + objectdata->parts;
		pgen++) {
      if ((*pgen)->type == LABEL) {
	 slab = TOLABEL(pgen);
	 if (slab == curlabel) continue;
	 if (hasparameter(slab))
	    function(slab);
      }
   }
}

/*----------------------------------------------------------------------*/
/* Wrapper functions for drawtextandupdate()				*/
/*----------------------------------------------------------------------*/

void undrawtext(labelptr curlabel)
{
   undrawtextsimple(curlabel);

   if (hasparameter(curlabel))
      drawtextandupdate(curlabel, undrawtextsimple);
}

/*----------------------------------------------------------------------*/

void redrawtext(labelptr curlabel)
{
   redrawtextsimple(curlabel);

   if (hasparameter(curlabel))
      drawtextandupdate(curlabel, redrawtextsimple);
}

/*----------------------------------------------------------------------*/
/* Draw the catalog of font characters 					*/
/*----------------------------------------------------------------------*/

void composefontlib(short cfont)
{
   objinstptr *drawinst;
   objectptr *curlib, libobj, saveobj, nullobj;
   objectptr libinst = xobjs.libtop[FONTLIB];
   short visobjects, i, qdel;
   polyptr *drawbox;
   pointlist pointptr;

   reset(libinst, NORMAL);

   /* Create a pointer to the font library */

   curlib = xobjs.fontlib.library;

   /* Find the number of non-null characters.  Do this by assuming */
   /* that all fonts encode nullchar at position zero.		   */

   visobjects = 0;
   nullobj = fonts[cfont].encoding[0];
   for(i = 1; i < 256; i++)
      if (fonts[cfont].encoding[i] != nullobj) visobjects++;

   /* add the box and gridlines */

   visobjects += 34;

   /* generate the list of object instances */

   libinst->plist = (genericptr *) realloc(libinst->plist, visobjects
		* sizeof(genericptr));
   libinst->parts = 0;

   /* 0.5 is the default vscale;  16 is no. characters per line */
   del = min(areastruct.width, areastruct.height) / (0.5 * 16);
   qdel = del >> 2;

   for (i = 0; i < 256; i++) {

      if ((libobj = fonts[cfont].encoding[i]) == nullobj) continue;
      
      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->scale = 1.0;
      (*drawinst)->thisobject = libobj;
      (*drawinst)->params = NULL;
      (*drawinst)->position.x = (i % 16) * del + qdel;
      (*drawinst)->position.y = -(i / 16) * del + qdel;

      libinst->parts++;
   }

   /* separate characters with gridlines (17 vert., 17 horiz.) */

   for (i = 0; i < 34; i++) {
      drawbox = (polyptr *)libinst->plist + libinst->parts;
      *drawbox = (polyptr) malloc(sizeof(polygon));
      (*drawbox)->type = POLYGON;
      (*drawbox)->color = SNAPCOLOR;   /* default red */
      (*drawbox)->style = UNCLOSED;
      (*drawbox)->width = 1.0;
      (*drawbox)->number = 2;
      (*drawbox)->points = (pointlist) malloc(2 * sizeof(XPoint));
      if (i < 17) {
         pointptr = (*drawbox)->points;
         pointptr->x = i * del;
         pointptr->y = 0;
         pointptr = (*drawbox)->points + 1;
         pointptr->x = i * del;
         pointptr->y = -16 * del;
      }
      else {
         pointptr = (*drawbox)->points;
         pointptr->x = 0;
         pointptr->y = (17 - i) * del;
         pointptr = (*drawbox)->points + 1;
         pointptr->x = 16 * del;
         pointptr->y = (17 - i) * del;
      }
      libinst->parts++;
   }

   /* calculate a bounding box for this display */

   saveobj = objectdata;
   objectdata = libinst;
   calcbbox(objectdata);
   objectdata = saveobj;
}

/*------------------------------------------------------*/
/* ButtonPress handler during font catalog viewing mode */
/*------------------------------------------------------*/

void fontcatbutton(XButtonEvent *event)
{
   short chx, chy;
   u_long rch = 0;

   if (event->button == Button1 || event->button == Button2) {

      window_to_user(event->x, event->y, &areastruct.save);
 
      chy = -areastruct.save.y / del + 1;
      chx = areastruct.save.x / del;

      chx = min(15, chx);
      chy = min(15, chy);
    
      rch = (u_long)(chy * 16 + chx);
   }

   catreturn();

   if (rch != 0)
      labeltext(rch, NULL);
}

/*-------------------------------------------------------------------------*/
