/*-------------------------------------------------------------------------*/
/* filelist.c --- Xcircuit routines for the filelist widget		   */ 
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(DARWIN)
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#include <sys/types.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#include <unistd.h>
#define direct dirent
#else
#include <sys/dir.h>
#endif

#include <sys/stat.h>
#include <errno.h>
#include <limits.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include "Xw/Xw.h"
#include "Xw/WorkSpace.h"
#include "Xw/TextEdit.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*-------------------------------------------------------------------------*/
/* Local definitions							   */
/*-------------------------------------------------------------------------*/

#define INITDIRS 10

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

extern Display    *dpy;
extern Window     win;
extern Clientdata areastruct;
extern ApplicationData appdata;
extern int 	  *appcolors;
extern short	  popups;     /* total number of popup windows */
extern char	  _STR2[250];
extern char	  _STR[150];
extern Globaldata xobjs;

Pixmap   flistpix = (Pixmap)NULL;    /* For file-selection widget */
short    flstart, flfiles, flcurrent;
int	 flcurwidth;

GC	 hgc = NULL, sgc = NULL;
char     **filenames = NULL;
char	 *cwdname = NULL;

/*-------------------------------------------------------------------------*/
/* Compare two filenames (for use by qsort())				   */
/*-------------------------------------------------------------------------*/

int fcompare(const void *a, const void *b)
{
   return(strcmp(*((char **)a), *((char **)b)));
}

/*-------------------------------------------------------------------------*/
/* Routines for drawing a box around the currently selected file	   */
/*-------------------------------------------------------------------------*/

void dragfilebox(Widget w, caddr_t clientdata, XMotionEvent *event)
{
   short filenum;
   int twidth;
   Window lwin = XtWindow(w);

   filenum = (event->y - 10 + FILECHARHEIGHT) / FILECHARHEIGHT + flstart - 1;
   if (filenum < 0) filenum = 0;
   else if (filenum >= flfiles) filenum = flfiles - 1;
  
   if (filenum == flcurrent) return;

   if (flcurrent >= 0) 		/* erase previous box */
      XDrawRectangle(dpy, lwin, areastruct.gc, 5,
	   10 + FILECHARHEIGHT * (flcurrent
	   - flstart), flcurwidth + 10, FILECHARHEIGHT);

   twidth = XTextWidth(appdata.filefont, filenames[filenum],
	    strlen(filenames[filenum]));
   XDrawRectangle(dpy, lwin, areastruct.gc, 5,
	   10 + FILECHARHEIGHT * (filenum
	   - flstart), twidth + 10, FILECHARHEIGHT);

   flcurrent = filenum;
   flcurwidth = twidth;
}

/*-------------------------------------------------------------------------*/
/* Begin tracking the cursor position relative to the files in the list    */
/*-------------------------------------------------------------------------*/

void startfiletrack(Widget w, caddr_t clientdata, XCrossingEvent *event)
{
   XtAddEventHandler(w, PointerMotionMask, False, (XtEventHandler)dragfilebox, NULL);

   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR);
   XSetLineAttributes(dpy, areastruct.gc, 1, LineSolid, CapRound, JoinMiter);

   /* draw initial box */

   flcurrent = -1;
   dragfilebox(w, NULL, (XMotionEvent *)event);

   XSetFunction(dpy, areastruct.gc, GXxor);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR ^ BACKGROUND);
}

/*-------------------------------------------------------------------------*/
/* Stop tracking the cursor and return to default state			   */
/*-------------------------------------------------------------------------*/

void endfiletrack(Widget w, caddr_t clientdata, XCrossingEvent *event)
{
   Window lwin = XtWindow(w);

   XDrawRectangle(dpy, lwin, areastruct.gc, 5,
	   10 + FILECHARHEIGHT * (flcurrent
	   - flstart), flcurwidth + 10, FILECHARHEIGHT);

   XtRemoveEventHandler(w, Button1MotionMask | Button2MotionMask,
	False, (XtEventHandler)dragfilebox, NULL);

   /* Restore graphics state values */
   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
}

/*----------------------------------------------------------------------*/
/* Read a crash file to find the name of the original file.		*/
/*----------------------------------------------------------------------*/

char *getcrashfilename()
{
   FILE *fi;
   char temp[256];
   char *retstr = NULL, *tpos, *spos;
   int slen;

   if ((fi = fopen(_STR2, "r")) != NULL) {
      while (fgets(temp, 255, fi) != NULL) {
	 if ((tpos = strstr(temp, "Title:")) != NULL) {
	    ridnewline(temp);
	    tpos += 7;
	    if ((spos = strrchr(temp, '/')) != NULL)
	       tpos = spos + 1;
	    retstr = (char *)malloc(1 + strlen(tpos));
	    strcpy(retstr, tpos);
	 }
	 else if ((tpos = strstr(temp, "CreationDate:")) != NULL) {
	    ridnewline(temp);
	    tpos += 14;
	    slen = strlen(retstr);
	    retstr = (char *)realloc(retstr, 4 + slen + strlen(tpos));
	    sprintf(retstr + slen, " (%s)", tpos);
	    break;
	 }
      }
   }
   return retstr;
}

/*----------------------------------------------------------------------*/
/* Crash recovery:  load the file, and link the tempfile name to it.    */
/*----------------------------------------------------------------------*/

void crashrecover()
{
   if (xobjs.tempfile != NULL) {
      unlink(xobjs.tempfile);
      free(xobjs.tempfile);
   }
   xobjs.tempfile = strdup(_STR2);

   startloadfile();
}

/*----------------------------------------------------------------------*/
/* Look for any files left over from a crash.                           */
/*----------------------------------------------------------------------*/

void findcrashfiles()
{
   DIR *cwd;
   struct direct *dp;
   struct stat sbuf;
   uid_t userid = getuid();
   time_t recent = 0;

   cwd = opendir(xobjs.tempdir);
   if (cwd == NULL) return;	/* No tmp directory, no tmp files! */

   while ((dp = readdir(cwd)) != NULL) {
      sprintf(_STR, "%s/%s", xobjs.tempdir, dp->d_name);
      if (!strncmp(_STR + 5, "XC", 2)) {
         stat(_STR, &sbuf);
         if (sbuf.st_uid == userid) {
	    if ((recent == 0) || (sbuf.st_ctime > recent)) {
	       recent = sbuf.st_ctime;
	       strcpy(_STR2, _STR);
	    }
	 }
      }
   }
   closedir(cwd);
   
   if (recent > 0) {	/* There exists at least one temporary file */
			/* belonging to this user.  Ask to recover  */
			/* the most recent one.			    */

      /* Warn user of existing tempfile, and ask user if file	*/
      /* should be recovered immediately.			*/

      getfile(NULL, Number(RECOVER), NULL);   /* Crash recovery mode */
   }
}  

/*-------------------------------------------------------------------------*/
/* Make a list of the files in the list widget window			   */
/*-------------------------------------------------------------------------*/

void listfiles(Widget w, caddr_t clientdata, caddr_t calldata)
{
   XGCValues	values;
   Arg wargs[2];
   DIR *cwd;
   Window lwin = XtWindow(w);
   short allocd = INITDIRS;
   short n = 0;
   struct direct *dp;
   struct stat statbuf;
   int pixheight;
   Dimension textwidth, textheight;

   if (sgc == NULL) {
      values.foreground = FOREGROUND;
      values.font = appdata.filefont->fid;
      values.function = GXcopy;
      sgc = XCreateGC(dpy, lwin, GCForeground | GCFont | GCFunction, &values);
   }

   XtnSetArg(XtNwidth, &textwidth);
   XtnSetArg(XtNheight, &textheight);
   XtGetValues(w, wargs, n);

   /* Generate a new flistpix pixmap if currently nonexistent */

   if (!flistpix) {

      /* get list of files in the current directory (cwd) */

      if (filenames == NULL) 
         filenames = (char **) malloc (INITDIRS * sizeof(char *));
      flfiles = 0;
      if (cwdname == NULL) {
	 cwdname = (char *) malloc (sizeof(char));
	 cwdname[0] = '\0';
      }
      if (cwdname[0] == '\0')
         cwd = opendir(".");
      else
         cwd = opendir(cwdname);

      /* If current directory cannot be accessed for some reason, */
      /* print "Invalid Directory" to the file list window.	  */

      if (cwd == NULL) {
         XSetForeground(dpy, sgc, BACKGROUND);
         XFillRectangle(dpy, lwin, sgc, 0, 0, textwidth, textheight);
         XSetForeground(dpy, sgc, AUXCOLOR);
         XDrawString(dpy, lwin, sgc, 10, textheight / 2,
	    "(Invalid Directory)", 19);
	 return;
      }
      else {

	 /* write the contents of the current directory into the   */
	 /* array "filenames[]" (except for current directory ".") */

         while ((dp = readdir(cwd)) != NULL) {
	    /* don't put current directory in list */
	    if (!strcmp(dp->d_name, ".")) continue;
	    filenames[flfiles] = (char *) malloc ((strlen(dp->d_name) + 2) *
		 sizeof(char));
	    strcpy(filenames[flfiles++], dp->d_name);
	    if (flfiles == allocd) {
	       allocd += INITDIRS;
	       filenames = (char **) realloc (filenames, allocd * sizeof(char *));
	    }
         }
      }
      closedir(cwd);

      /* sort the filenames[] array into alphabetical order (like "ls") */

      qsort((void *)filenames, (size_t)flfiles, sizeof(char *), fcompare);

      pixheight = flfiles * FILECHARHEIGHT + 20;

      flistpix = XCreatePixmap(dpy, win, textwidth, pixheight,
	   DefaultDepthOfScreen(XtScreen(w)));

      /* Write the filenames onto the pixmap */

      XSetForeground(dpy, sgc, BACKGROUND);
      XFillRectangle(dpy, flistpix, sgc, 0, 0, textwidth, pixheight);
      XSetForeground(dpy, sgc, FOREGROUND);
      for (n = 0; n < flfiles; n++) {
	 sprintf(_STR2, "%s%s", cwdname, filenames[n]); 
	 stat(_STR2, &statbuf);
	 if ((statbuf.st_mode & S_IFDIR) != 0) {  /* is a directory */
	    XSetForeground(dpy, sgc, SELECTCOLOR);
	    strcat(filenames[n], "/");
	 }
         XDrawString(dpy, flistpix, sgc, 10, 10 + FILECHARASCENT + n * FILECHARHEIGHT,
	    filenames[n], strlen(filenames[n]));
	 if ((statbuf.st_mode & S_IFDIR) != 0)
	    XSetForeground(dpy, sgc, FOREGROUND);
      }
   }

   /* Copy the pixmap of filenames into the file list window */

   XSetForeground(dpy, sgc, BACKGROUND);
   XFillRectangle(dpy, lwin, sgc, 0, 0, textwidth, textheight);
   XCopyArea(dpy, flistpix, lwin, sgc, 0, flstart * FILECHARHEIGHT,
	textwidth, textheight, 0, 0);
}

/*-------------------------------------------------------------------------*/
/* Generate a new pixmap for writing the filelist and set the scrollbar    */
/* size accordingly.							   */
/*-------------------------------------------------------------------------*/

void newfilelist(Widget w, Widget textw)
{
   short n;

   for (n = 0; n < flfiles; n++)
      free(filenames[n]);
   free(filenames);
   if (flistpix != (Pixmap)NULL) XFreePixmap(dpy, flistpix);
   filenames = NULL;
   flistpix = (Pixmap)NULL;
   flstart = 0;
   listfiles(w, NULL, NULL);
   showlscroll(XtNameToWidget(XtParent(w), "LScroll"), NULL, NULL);
   XwTextClearBuffer(textw);
   XwTextInsert(textw, cwdname);
   XwTextResize(textw);
}

/*-------------------------------------------------------------------------*/
/* Button press handler for file list window				   */
/*-------------------------------------------------------------------------*/

void fileselect(Widget w, Widget textw, XButtonEvent *event)
{
   Arg wargs[2];
   Window lwin = XtWindow(w);
   Dimension textwidth, textheight;
   short filenum, n = 0;
   char *tbuf;

   flcurrent = -1;

   XtnSetArg(XtNwidth, &textwidth);
   XtnSetArg(XtNheight, &textheight);
   XtGetValues(w, wargs, n);

   /* third mouse button cancels selection and reverts buffer to cwd name */

   if (event->button == Button3) {
      newfilelist(w, textw);
      return;
   }

   filenum = (event->y - 10 + FILECHARHEIGHT) / FILECHARHEIGHT + flstart - 1;
   if (filenum < 0) filenum = 0;
   else if (filenum >= flfiles) filenum = flfiles - 1;

   /* Attempt to enter invalid directory. . . treat like button 3 */

   if (filenum < 0) {
      newfilelist(w, textw);
      return;
   }
   
   /* check if this file is a directory or not */

   if (strchr(filenames[filenum], '/') == NULL)	{

      /* highlight the entry. . . */

      XSetForeground(dpy, sgc, AUXCOLOR);
      XDrawString(dpy, flistpix, sgc, 10, 10 + FILECHARASCENT + filenum * FILECHARHEIGHT,
   	   filenames[filenum], strlen(filenames[filenum]));
      XCopyArea(dpy, flistpix, lwin, sgc, 0, flstart * FILECHARHEIGHT,
	   textwidth, textheight, 0, 0);

      /* . . .and append it to the text field */

      tbuf = (char *)malloc((XwTextGetLastPos(textw)
	     + strlen(filenames[filenum]) + 5) * sizeof(char));
      strcpy(tbuf, (char *)XwTextCopyBuffer(textw)); 

      /* add a comma if there is already text in the destination buffer */

      if (tbuf[0] != '\0') {
         if (tbuf[strlen(tbuf) - 1] != '/') strcat(tbuf, ",");
      }
      else {
	 if (cwdname != NULL) {
	    if (cwdname[0] != '\0') {
	       tbuf = (char *)realloc(tbuf, (strlen(cwdname) +
			strlen(filenames[filenum]) + 5) * sizeof(char));
	       strcpy(tbuf, cwdname);
	    }
 	 }
      }
      strcat(tbuf, filenames[filenum]);
      XwTextClearBuffer(textw);
      XwTextInsert(textw, tbuf);
      XwTextResize(textw);
      free(tbuf);
   }
   else {  /* move to new directory */

      if (!strcmp(filenames[filenum], "../")) {
         char *cptr, *sptr = cwdname;
	 if (!strcmp(cwdname, "/")) return;	/* no way up from root dir. */
	 while(strstr(sptr, "../") != NULL) sptr += 3;
	 if ((cptr = strrchr(sptr, '/')) != NULL) {
	    *cptr = '\0';
	    if ((cptr = strrchr(sptr, '/')) != NULL) *(cptr + 1) = '\0';
	    else *sptr = '\0';
         }
	 else {
      	    cwdname = (char *)realloc(cwdname, (strlen(cwdname) +
	        4) * sizeof(char));
            strcat(cwdname, "../");
	 }
      }
      else {
	 cwdname = (char *)realloc(cwdname, (strlen(cwdname) +
	        strlen(filenames[filenum]) + 1) * sizeof(char));
         strcat(cwdname, filenames[filenum]);
      }
      newfilelist(w, textw);
   }
}

/*-------------------------------------------------------------------------*/
/* Scrollbar handler for file list widget				   */
/*-------------------------------------------------------------------------*/

void showlscroll(Widget w, caddr_t clientdata, caddr_t calldata)
{
   Arg wargs[2];
   Window swin = XtWindow(w);
   Dimension swidth, sheight;
   int pstart, pheight, finscr;
   short n = 0;


   XtnSetArg(XtNwidth, &swidth);
   XtnSetArg(XtNheight, &sheight);
   XtGetValues(w, wargs, n);

   XClearWindow(dpy, swin);

   if (flfiles > 0) {	/* no files, no bar */

      finscr = LISTHEIGHT / FILECHARHEIGHT;
      if (finscr > flfiles) finscr = flfiles;

      pstart = (flstart * sheight) / flfiles;
      pheight = (finscr * sheight) / flfiles;

      XSetForeground(dpy, sgc, BARCOLOR);
      XFillRectangle(dpy, swin, sgc, 0, pstart, swidth, pheight);
   }
   flcurrent = -1;
}

/*-------------------------------------------------------------------------*/
/* Button Motion handler for moving the scrollbar up and down		   */
/*-------------------------------------------------------------------------*/

void draglscroll(Widget w, Widget filew, XButtonEvent *event)
{
   Arg wargs[2];
   Dimension swidth, sheight;
   int phheight, finscr, flsave = flstart;
   short n = 0;

   XtnSetArg(XtNwidth, &swidth);
   XtnSetArg(XtNheight, &sheight);
   XtGetValues(w, wargs, n);

   finscr = LISTHEIGHT / FILECHARHEIGHT;
   if (finscr > flfiles) finscr = flfiles;

   /* center scrollbar on pointer vertical position */   

   phheight = (finscr * sheight) / (flfiles * 2);
   flstart = (event->y > phheight) ? ((event->y - phheight) * flfiles) / sheight : 0;
   if (flstart >= flfiles) flstart = flfiles - 1;

   if (flstart != flsave) {
      showlscroll(w, NULL, NULL);
      listfiles(filew, NULL, NULL); 
   }
}

/*-------------------------------------------------------------------------*/
/* Generate the file list window					   */
/*-------------------------------------------------------------------------*/

void genfilelist(Widget parent, Widget entertext, Dimension width)
{
   Arg		wargs[8];
   Widget 	listarea, lscroll;
   short 	n = 0;

   XtnSetArg(XtNx, 20);
   XtnSetArg(XtNy, FILECHARHEIGHT - 10);
   XtnSetArg(XtNwidth, width - SBARSIZE - 40);
   XtnSetArg(XtNheight, LISTHEIGHT - FILECHARHEIGHT);
   XtnSetArg(XtNfont, appdata.filefont);

   listarea = XtCreateManagedWidget("Filelist", XwworkSpaceWidgetClass,
	  parent, wargs, n); n = 0;
   XtAddCallback(listarea, XtNexpose, (XtCallbackProc)listfiles, NULL);
   XtAddEventHandler(listarea, ButtonPressMask, False,
	  (XtEventHandler)fileselect, entertext);
   XtAddEventHandler(listarea, EnterWindowMask, False,
	  (XtEventHandler)startfiletrack, NULL);
   XtAddEventHandler(listarea, LeaveWindowMask, False,
	  (XtEventHandler)endfiletrack, NULL);
   flstart = 0;

   XtnSetArg(XtNx, width - SBARSIZE - 20);
   XtnSetArg(XtNy, FILECHARHEIGHT - 10);
   XtnSetArg(XtNwidth, SBARSIZE);
   XtnSetArg(XtNheight, LISTHEIGHT - FILECHARHEIGHT);
   XtnSetArg(XtNfont, appdata.xcfont);

   lscroll = XtCreateManagedWidget("LScroll", XwworkSpaceWidgetClass,
	     parent, wargs, n); n = 0;

   XtAddCallback(lscroll, XtNexpose, (XtCallbackProc)showlscroll, NULL);
   XtAddEventHandler(lscroll, Button1MotionMask | Button2MotionMask,
	     False, (XtEventHandler)draglscroll, listarea);
}

/*-------------------------------------------------------------------------*/
/* Look for a directory name in a string and update cwdname accordingly	   */
/*-------------------------------------------------------------------------*/

int lookdirectory(char *lstring)
{
   int slen;

   tilde_expand(lstring);
   slen = strlen(lstring);

   if (lstring[slen - 1] == '/' || (opendir(lstring) != NULL)) {
      if (lstring[slen - 1] != '/') strcat(lstring, "/");
      cwdname = (char *)realloc(cwdname, (slen + 2) * sizeof(char));
      strcpy(cwdname, lstring);
      return(1);
   }
   return(0);
}

/*-------------------------------------------------------------------------*/
