/*-------------------------------------------------------------------------*/
/* events.c --- xcircuit routines handling Xevents and Callbacks	   */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#define  XK_MISCELLANY
#define  XK_LATIN1
#include <X11/keysymdef.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

char	  *labelbuf;
short	  saverot; 
short     eventmode;	  /* keep track of the mode the screen is in */
short     textpos;	  /* keep track of the cursor position in text */
unsigned  char curbutton = 0;	/* current working button */
short     refselect;
short	  attachto = 0;
short	  pushes;
objectptr *pushlist;

extern Display	*dpy;
extern Window win;
extern int *appcolors;
extern Cursor	appcursors[NUM_CURSORS];
extern float *cosine[RSTEPS + 1], *sine[RSTEPS + 1];
extern Clientdata areastruct;
extern ApplicationData appdata;
extern short popups, help_up;
extern Widget message2, top;
extern char  _STR[150], _STR2[250];
extern short beeper;
extern double saveratio;

#define RADFAC 0.0174532925199   /* pi / 180 */
/* INVRFAC = 57.295779 = (180 / pi)  is defined in xcircuit.h */

/* double buffer */
#ifdef DOUBLEBUFFER
Pixmap dbuf = (Pixmap)NULL;
#endif

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to events.c	 	  */
/*------------------------------------------------------------------------*/

extern void snapobject(), objectrotate(), edit(), objectvflip(),
	objectflip(), panbutton(), deletebutton(), exchange();
extern void printname(), selectsave(), copybutton(),
	undrawtext(), redrawtext(), setsnap(), starthelp(), releasebutton();
extern void charreport(), togglegrid(), setvscale(), pushobject();

extern XtEventHandler trackpoly(), trackspline(), trackbox();
extern XtEventHandler catbutton(), trackselarea(), trackarc();
extern XtTimerCallbackProc startselect();
 
extern short checkbounds();
extern float findsplinemin();
extern short *objectselect();

short undelete();
void objectdelete();
void pathedit(genericptr *, short);
void pathbutton(genericptr *editpart, XButtonEvent *);

XtEventHandler drag();
XtCallbackProc drawarea(), drawhbar(), drawvbar();

/*----------------------------------------------------------------------------*/
/* Edit Object pushing and popping.					      */
/*----------------------------------------------------------------------------*/

Boolean recursefind(parent, suspect)
  objectptr parent, suspect;
{
   genericptr *shell;
   objectptr child;

   if (parent == suspect) return True;

   for (shell = parent->plist; shell < parent->plist + parent->parts; shell++)
      if ((*shell)->type == OBJECT)
         if (recursefind(TOOBJINST(shell)->thisobject, suspect)) return True;

   return False;
}

/*------------------------------------------------------------------*/
/* Transfer objects in the select list to the current object	    */
/* (assumes they have been deleted from the previous object)	    */
/*------------------------------------------------------------------*/
/* Disallow infinitely recursive loops!				    */
/*------------------------------------------------------------------*/

void transferselects()
{
   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE) {
      short ps = objectdata->parts;
      short *newsel;
      areastruct.selects = undelete(NULL, Number(NORMAL), NULL);
      areastruct.selectlist = (short *) malloc(areastruct.selects *
	   sizeof(short));
  
      /* replace the select list */
      newsel = areastruct.selectlist;
      for (; ps < objectdata->parts; ps++) (*newsel++) = ps;
	 
      /* check to make sure this object is not the current object	   */
      /* or one of its direct ancestors, else an infinite loop results. */

      for (ps = 0; ps < objectdata->parts; ps++)
	 if ((*(objectdata->plist + ps))->type == OBJECT) {
	    if (recursefind(TOOBJINST(objectdata->plist + ps)->thisobject,
		    objectdata)) {
	       Wprintf("Attempt to place object inside of itself");
	       objectdelete(NORMAL);
	       break;
	    }
	 }
   }
}

/*-------------------------------------------------------*/
/* set the viewscale variable to the proper address	 */
/*-------------------------------------------------------*/

void setvscale()
{
   if (objectdata == *(areastruct.pagelist + areastruct.page)) {
      areastruct.vscale = &areastruct.viewscale[areastruct.page];
      areastruct.lowerleft = &areastruct.pcorner[areastruct.page];
   }
   else if (objectdata == areastruct.libtop[LIBRARY] ||
          objectdata == areastruct.libtop[USERLIB]) {
      areastruct.vscale = &areastruct.libvscale;
      areastruct.lowerleft = &areastruct.libpcorner;
   }
   else {
      areastruct.vscale = &areastruct.tempvscale;
      areastruct.lowerleft = &areastruct.temppcorner;
   }
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);
}

/*-------------------------------------------------------*/
/* switch to a new page					 */
/*-------------------------------------------------------*/

void newpage(pagenumber)
  short pagenumber;
{
   objectptr *pageobj;

   /* to add to existing number of top level pages. . . */

   if (pagenumber == 255) {
      if (areastruct.pages == 255) {
	 Wprintf("Out of available pages!");
	 return;
      }
      else pagenumber = areastruct.pages;
   }

   if (pagenumber >= areastruct.pages) {
      
      areastruct.filename = (char **)realloc(areastruct.filename,
	  (pagenumber + 1) * sizeof(char *));
      areastruct.filename[pagenumber] = NULL;
      areastruct.pagelist = (objectptr *)realloc(areastruct.pagelist,
	  (pagenumber + 1) * sizeof(objectptr));
      areastruct.wirewidth = (float *)realloc(areastruct.wirewidth,
	  (pagenumber + 1) * sizeof(float));
      areastruct.viewscale = (float *)realloc(areastruct.viewscale,
	  (pagenumber + 1) * sizeof(float));
      areastruct.outscale = (float *)realloc(areastruct.outscale,
	  (pagenumber + 1) * sizeof(float));
      areastruct.orient = (short *)realloc(areastruct.orient,
	  (pagenumber + 1) * sizeof(short));
      areastruct.pmode = (short *)realloc(areastruct.pmode,
	  (pagenumber + 1) * sizeof(short));
      areastruct.gridspace = (float *)realloc(areastruct.gridspace,
	  (pagenumber + 1) * sizeof(float));
      areastruct.snapspace = (float *)realloc(areastruct.snapspace,
	  (pagenumber + 1) * sizeof(float));
      areastruct.coordstyle = (short *)realloc(areastruct.coordstyle,
	  (pagenumber + 1) * sizeof(short));
      areastruct.drawingscale = (XPoint *)realloc(areastruct.drawingscale,
	  (pagenumber + 1) * sizeof(XPoint));
      areastruct.pagesize = (XPoint *)realloc(areastruct.pagesize,
	  (pagenumber + 1) * sizeof(XPoint));
      areastruct.pcorner = (XPoint *)realloc(areastruct.pcorner,
	  (pagenumber + 1) * sizeof(XPoint));
      for (pageobj = areastruct.pagelist + areastruct.pages; pageobj <=
		areastruct.pagelist + pagenumber; pageobj++)
	 *pageobj = NULL;
      areastruct.pages = pagenumber + 1;
      makepagebutton();
   }

   pageobj = areastruct.pagelist + pagenumber;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);
   else
      objectdeselect();
   areastruct.page = pagenumber;

   if (pushes != 0) {
      pushes = 0;
      free(pushlist);
   }
   if (*pageobj == NULL) {
      /* initialize a new page */
      *pageobj = (objectptr) malloc (sizeof(object));
      objectdata = *pageobj;
      initmem(objectdata);
      sprintf(objectdata->name, "Page %d", pagenumber + 1);
      areastruct.filename[areastruct.page] = NULL;
      topreset();
      objectdata->width = 0;
      objectdata->height = 0;
   }
   else
      objectdata = *pageobj;

   setvscale();
   transferselects();
   togglegrid((unsigned short)areastruct.coordstyle[areastruct.page]);
   printname(objectdata);

   if ((*pageobj)->parts == 0) {
      drawarea(NULL, objectdata, NULL);
      drawhbar(areastruct.scrollbarh, NULL, Number(0));
      drawvbar(areastruct.scrollbarv, NULL, Number(1));
   }
   else
      zoomview(NULL, Number(0), NULL);
}


/*--------------------------------*/
/* Set snap spacing from keyboard */
/*--------------------------------*/

void setsnap(direction)
  short direction;
{
   float oldsnap = areastruct.snapspace[areastruct.page];
   char buffer[50];

   if (direction > 0) areastruct.snapspace[areastruct.page] *= 2;
   else {
      if (oldsnap >= 2.0)
         areastruct.snapspace[areastruct.page] /= 2;
      else {
	 measurestr(areastruct.snapspace[areastruct.page], buffer);
	 sprintf(_STR, "Snap space at minimum value of %s", buffer);
	 Wprintf(_STR);
      }
   }
   if (areastruct.snapspace[areastruct.page] != oldsnap) {
      measurestr(areastruct.snapspace[areastruct.page], buffer);
      sprintf(_STR, "Snap spacing set to %s", buffer);
      Wprintf(_STR);
      drawarea(NULL, objectdata, NULL);
   }
}

/*-----------------------------*/
/* Menu push hierarchy command */
/*-----------------------------*/

void startpush(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   XEvent devent;
   XButtonEvent *bevent = (XButtonEvent *)(&devent);

   if (eventmode == NORMAL_MODE) {
      if (areastruct.selects > 1) objectdeselect();

      if (areastruct.selects == 1) {
	 bevent->x = bevent->y = 0;
	 pushobject(w, clientdata, bevent);
      }
      else {
         eventmode = PUSH_MODE;
         Wprintf("Click on object to push");
      }
   }
}

/*------------------------------------------*/
/* Push object onto hierarchy stack to edit */
/*------------------------------------------*/

void pushobject(button, clientdata, event)
  Widget        button;
  caddr_t       clientdata;
  XButtonEvent *event;
{
   short *selectobj;
   objinstptr pushinst;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);

   window_to_user(event->x, event->y, &areastruct.save);
   selectobj = areastruct.selectlist;
   if (areastruct.selects == 0) selectobj = objectselect(OBJECT);
   if (areastruct.selects == 0) {
      Wprintf("No objects selected.");
      if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
	 transferselects();
      return;
   }
   else if (areastruct.selects > 1) {
      Wprintf("Choose only one object.");
      return;
   }
   else if (SELECTTYPE(selectobj) != OBJECT) {
      Wprintf("Element to push must be an object.");
      return;
   }
   else if (event->button == Button3) {
      objectdeselect();
      return;
   }
   else pushinst = SELTOOBJINST(selectobj);

   /* save the address of the current object to the push stack */

   if (pushes == 0)
      pushlist = (objectptr *) malloc(sizeof(objectptr));
   else
      pushlist = (objectptr *) realloc(pushlist,
         (pushes + 1) * sizeof(objectptr));
   *(pushlist + pushes) = objectdata;
   pushes++;
   
   objectdata = pushinst->thisobject;

   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;

   /* move selected items to the new object */

   setvscale();
   transferselects();

   zoomview(button, Number(1), NULL);
   printname(objectdata);

}

/*--------------------------*/
/* Pop edit hierarchy stack */
/*--------------------------*/

void popobject(w, mode, clientdata)
  Widget w;
  void *mode;
  caddr_t clientdata;
{
   if (pushes == 0 || (eventmode != NORMAL_MODE && eventmode != PRESS_MODE &&
	eventmode != COPY2_MODE && eventmode != FONTCAT_MODE &&
	eventmode != FONTCAT2_MODE)) return;

   /* don't allow pop into library with copied or held objects */

   if ((eventmode == PRESS_MODE || eventmode == COPY2_MODE) &&
      (*(pushlist + pushes - 1) == areastruct.libtop[LIBRARY] ||
       *(pushlist + pushes - 1) == areastruct.libtop[USERLIB])) return;

   /* remove any selected items from the current object */

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);
   else
      objectdeselect();

   pushes--;
   calcbbox(objectdata);
   objectdata = *(pushlist + pushes);
   if (pushes == 0) free(pushlist);

   /* move selected items to the new object */

   if (objectdata == areastruct.libtop[LIBRARY] || objectdata ==
	 areastruct.libtop[USERLIB]) eventmode = CATALOG_MODE;

   setvscale();
   transferselects();

   if ((int)mode == 0)
      zoomview(w, Number(2), NULL);
   else {
      drawarea(NULL, objectdata, NULL);
      drawhbar(areastruct.scrollbarh, NULL, Number(0));
      drawvbar(areastruct.scrollbarv, NULL, Number(1));
   }
   printname(objectdata);
}

/*-------------------------------------------------------------------------*/
/* Destructive reset of entire object		 			   */
/*-------------------------------------------------------------------------*/

void resetbutton(button, clientdata, calldata)
  Widget        button;
  caddr_t       calldata, clientdata;
{
   objectptr *pageobj;
   short page;
   if (eventmode != NORMAL_MODE) return;

   for (page = 0; page < areastruct.pages; page++) 
      if (objectdata == *(areastruct.pagelist + page)) {
	 pageobj = areastruct.pagelist + page;
         sprintf((*pageobj)->name, "Page %d", page + 1);
	 areastruct.filename[page] = (char *)realloc(areastruct.filename[page],
		(strlen((*pageobj)->name) + 1) * sizeof(char));
	 strcpy(areastruct.filename[page], (*pageobj)->name);
      }

   reset(objectdata, NORMAL);
   drawarea(button, clientdata, calldata);

   printname(*pageobj);
   renamepage(areastruct.page);
   Wprintf("Page cleared.");
}

/*------------------------------------------------------*/
/* Special horizontal and vertical scrollbars		*/
/*------------------------------------------------------*/

/*------------------------------------------------------*/
/* Redraw the horizontal scrollbar			*/
/*------------------------------------------------------*/

XtCallbackProc drawhbar(bar, clientdata, calldata)
  Widget	bar;
  caddr_t	clientdata, calldata;
{
   Window bwin = XtWindow(bar);
   float frac;
   long rleft, rright, rmid;

   if (!XtIsRealized(bar)) return;

   if ((unsigned int) calldata != 1)
      calcbbox(objectdata);

   frac = (objectdata->width == 0) ? 1 :
	(float) areastruct.width / (float) objectdata->width;
   
   rleft = (long)(frac * (float)(areastruct.lowerleft->x - objectdata->lowerleft.x));
   rright = rleft + (long)(frac * (float)areastruct.width / (*areastruct.vscale));
   rmid = (rright + rleft) >> 1;

   if (rleft < 0) rleft = 0;
   if (rright > areastruct.width) rright = areastruct.width;

   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BARCOLOR);
   if (rmid > 0)
      XClearArea(dpy, bwin, 0, 0, (int)rleft, SBARSIZE, FALSE);
   XFillRectangle(dpy, bwin, areastruct.gc, (int)rleft + 1, 1,
	  (int)(rright - rleft), SBARSIZE - 1);
   if (rright > rmid)
      XClearArea(dpy, bwin, (int)rright + 1, 0, areastruct.width
	  - (int)rright, SBARSIZE, FALSE);
   XClearArea(dpy, bwin, (int)rmid - 1, 1, 3, SBARSIZE, FALSE);

   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
}

/*------------------------------------------------------*/
/* Redraw the vertical scrollbar			*/
/*------------------------------------------------------*/

XtCallbackProc drawvbar(bar, clientdata, calldata)
  Widget	bar;
  caddr_t	clientdata, calldata;
{
   Window bwin = XtWindow(bar);
   float frac;
   long rtop, rbot, rmid;

   if (!XtIsRealized(bar)) return;

   if ((unsigned int) calldata != 1)
      calcbbox(objectdata);

   frac = (objectdata->height == 0) ? 1 : (float)areastruct.height /
	(float)objectdata->height;

   rbot = (long)(frac * (float)(objectdata->lowerleft.y - areastruct.lowerleft->y
	  + objectdata->height));
   rtop = rbot - (long)(frac * (float)areastruct.height / (*areastruct.vscale));
   rmid = (rtop + rbot) >> 1;

   if (rtop < 0) rtop = 0;
   if (rbot > areastruct.height) rbot = areastruct.height;

   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BARCOLOR);
   if (rmid > 0)
      XClearArea(dpy, bwin, 0, 0, SBARSIZE, (int)rtop, FALSE);
   XFillRectangle(dpy, bwin, areastruct.gc, 1, (int)rtop + 1, SBARSIZE - 1,
	     (int)(rbot - rtop));
   if (rbot > rmid)
      XClearArea(dpy, bwin, 0, (int)rbot + 1, SBARSIZE, areastruct.height
		- (int)rbot, FALSE);
   XClearArea(dpy, bwin, 0, (int)rmid - 1, SBARSIZE, 3, FALSE);

   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
}

/*------------------------------------------------------*/
/* Simultaneously scroll the screen and horizontal	*/
/* bar when dragging the mouse in the scrollbar area 	*/
/*------------------------------------------------------*/

XtEventHandler panhbar(bar, clientdata, event)
  Widget        bar;
  caddr_t	clientdata;
  XButtonEvent *event;
{
   long  newx, newpx;
   short savex = areastruct.lowerleft->x;

   if (eventmode == SELAREA_MODE) return;

   newx = (long)(event->x * ((float)objectdata->width /
	areastruct.width) + objectdata->lowerleft.x - 0.5 * 
	((float)areastruct.width / (*areastruct.vscale)));
   areastruct.lowerleft->x = (short)newx;
   drawhbar(bar, NULL, Number(1));
   areastruct.lowerleft->x = savex;

#ifdef DOUBLEBUFFER
   newpx = (long)(newx - savex) * (*areastruct.vscale);
   XSetFunction(dpy, areastruct.gc, GXcopy);
   if (newpx > 0) {
      XCopyArea(dpy, dbuf, win, areastruct.gc, newpx, 0,
	     areastruct.width - newpx, areastruct.height, 0, 0);
      XClearArea(dpy, win, areastruct.width - newpx, 0, newpx,
	     areastruct.height, FALSE);
   }
   else {
      XCopyArea(dpy, dbuf, win, areastruct.gc, 0, 0,
	     areastruct.width + newpx, areastruct.height, -newpx, 0);
      XClearArea(dpy, win, 0, 0, -newpx, areastruct.height, FALSE);
   }
#endif
}

/*------------------------------------------------------*/
/* End the horizontal scroll and refresh entire screen	*/
/*------------------------------------------------------*/

XtEventHandler endhbar(bar, clientdata, event)
  Widget        bar;
  caddr_t	clientdata;
  XButtonEvent *event;
{
   long  newx;
   short savex = areastruct.lowerleft->x;

   newx = (long)(event->x * ((float)objectdata->width /
	areastruct.width) + objectdata->lowerleft.x - 0.5 * 
	((float)areastruct.width / (*areastruct.vscale)));

   areastruct.lowerleft->x = (short)newx;

   if ((newx << 1) != (long)((short)(newx << 1)) || checkbounds() == -1) {
      areastruct.lowerleft->x = savex;
      Wprintf("Reached boundary:  cannot pan further");
   }
   else
      Wprintf(" ");

   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawhbar(bar, NULL, Number(1));
   drawarea(bar, objectdata, NULL);
}

/*------------------------------------------------------*/
/* Simultaneously scroll the screen and vertical	*/
/* bar when dragging the mouse in the scrollbar area 	*/
/*------------------------------------------------------*/

XtEventHandler panvbar(bar, clientdata, event)
  Widget        bar;
  caddr_t	clientdata;
  XButtonEvent *event;
{
   long  newy, newpy;
   short savey = areastruct.lowerleft->y;

   if (eventmode == SELAREA_MODE) return;

   newy = (int)((areastruct.height - event->y) *
	((float)objectdata->height / areastruct.height) +
	objectdata->lowerleft.y - 0.5 * ((float)areastruct.height /
	     (*areastruct.vscale)));
   areastruct.lowerleft->y = (short)newy;
   drawvbar(bar, NULL, Number(1));
   areastruct.lowerleft->y = savey;

#ifdef DOUBLEBUFFER
   newpy = (long)(newy - savey) * (*areastruct.vscale);
   XSetFunction(dpy, areastruct.gc, GXcopy);
   if (newpy > 0) {
      XCopyArea(dpy, dbuf, win, areastruct.gc, 0, 0,
	     areastruct.width, areastruct.height - newpy, 0, newpy);
      XClearArea(dpy, win, 0, 0, areastruct.width, newpy, FALSE);
   }
   else {
      XCopyArea(dpy, dbuf, win, areastruct.gc, 0, -newpy,
	     areastruct.width, areastruct.height + newpy, 0, 0);
      XClearArea(dpy, win, 0, areastruct.height + newpy,
	     areastruct.width, -newpy, FALSE);
   }
#endif

}

/*------------------------------------------------------*/
/* End the vertical scroll and refresh entire screen	*/
/*------------------------------------------------------*/

XtEventHandler endvbar(bar, clientdata, event)
  Widget        bar;
  caddr_t	clientdata;
  XButtonEvent *event;
{
   long  newy;
   short savey = areastruct.lowerleft->y;

   newy = (int)((areastruct.height - event->y) *
	((float)objectdata->height / areastruct.height) +
	objectdata->lowerleft.y - 0.5 * ((float)areastruct.height /
	     (*areastruct.vscale)));

   areastruct.lowerleft->y = (short)newy;

   if ((newy << 1) != (long)((short)(newy << 1)) || checkbounds() == -1) {
      areastruct.lowerleft->y = savey;
      Wprintf("Reached boundary:  cannot pan further");
   }
   else
      Wprintf(" ");

   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawvbar(bar, NULL, Number(1));
   drawarea(bar, objectdata, NULL);
}

/*--------------------------------------------------------------------*/
/* Zoom functions-- zoom box, zoom in, zoom out, and pan.	      */
/*--------------------------------------------------------------------*/

void zoombox(button, clientdata, calldata)
  Widget        button;
  caddr_t       calldata, clientdata;
{
   if (eventmode == NORMAL_MODE) {
      eventmode = SELAREA2_MODE;
      Wprintf("Click and make box to zoom.");
   }
}

/*--------------------------------------------------------------------*/

void zoomin(button, mode, event)
  Widget        button;
  caddr_t	mode;
  XButtonEvent *event;
{
   int  minzoom = 32768 / max(areastruct.width, areastruct.height);
   float savescale;
   XPoint ucenter, ncenter, savell;

   savescale = *areastruct.vscale;
   savell.x = areastruct.lowerleft->x;
   savell.y = areastruct.lowerleft->y;

   /* zoom-box function: corners are in areastruct.save and areastruct.origin */

   if (eventmode == SELAREA_MODE) {
      float delxscale, delyscale;
      /* select box has lower-left corner in .origin, upper-right in .save */

      /* ignore if zoom box is size zero */

      if (areastruct.save.x == areastruct.origin.x || areastruct.save.y ==
	  areastruct.origin.y) {
	 Wprintf("Zoom box of size zero: Ignoring.");
         XtRemoveEventHandler(areastruct.area, ButtonMotionMask, False,
	     (XtEventHandler)trackselarea, NULL);
         if ((unsigned int) mode != 0)
	    eventmode = NORMAL_MODE;
	 else
            eventmode = DESEL_MODE;
	 return;
      }

      /* determine whether x or y is limiting factor in zoom */
      delxscale = (areastruct.width / (*areastruct.vscale)) /
	   abs(areastruct.save.x - areastruct.origin.x);
      delyscale = (areastruct.height / (*areastruct.vscale)) /
	   abs(areastruct.save.y - areastruct.origin.y);
      (*areastruct.vscale) *= min(delxscale, delyscale);

      areastruct.lowerleft->x = min(areastruct.origin.x, areastruct.save.x) -
	 (areastruct.width / (*areastruct.vscale) - 
	 abs(areastruct.save.x - areastruct.origin.x)) / 2;
      areastruct.lowerleft->y = min(areastruct.origin.y, areastruct.save.y) -
	 (areastruct.height / (*areastruct.vscale) - 
	 abs(areastruct.save.y - areastruct.origin.y)) / 2;
      XtRemoveEventHandler(areastruct.area, ButtonMotionMask, False,
	  (XtEventHandler)trackselarea, NULL);
      if ((unsigned int) mode != 0)
	 eventmode = NORMAL_MODE;
      else
         eventmode = DESEL_MODE;
   }
   else {
      window_to_user(areastruct.width / 2, areastruct.height / 2, &ucenter);
      (*areastruct.vscale) *= SCALEFAC;
      window_to_user(areastruct.width / 2, areastruct.height / 2, &ncenter);
      areastruct.lowerleft->x += (ucenter.x - ncenter.x);
      areastruct.lowerleft->y += (ucenter.y - ncenter.y);
   }

   /* check for minimum scale */

   if (checkbounds() == -1) {
      areastruct.lowerleft->x = savell.x;
      areastruct.lowerleft->y = savell.y;
      (*areastruct.vscale) = savescale;
      Wprintf("At minimum scale: cannot scale further");

      /* this is a rare case where an object gets out-of-bounds */

      if (checkbounds() == -1) {
	 if (beeper) XBell(dpy, 100);
	 Wprintf("Unable to scale: Delete out-of-bounds object!");
      }
      return;
   }
   else if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      drag(areastruct.area, NULL, event);

   Wprintf(" ");
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawarea(button, objectdata, NULL);
   drawhbar(areastruct.scrollbarh, NULL, Number(0));
   drawvbar(areastruct.scrollbarv, NULL, Number(1));
}

/*--------------------------------------------------------------------*/

void zoomout(button, mode, event)
  Widget        button;
  caddr_t	mode;
  XButtonEvent *event;
{
   float maxzoom = (float)max(areastruct.width, areastruct.height) / 32768;
   float savescale;
   XPoint ucenter, ncenter, savell;
   XlPoint newll;

   savescale = (*areastruct.vscale);
   savell.x = areastruct.lowerleft->x;
   savell.y = areastruct.lowerleft->y;

   /* zoom-box function, analogous to that for zoom-in */

   if (eventmode == SELAREA_MODE) {
      float delxscale, delyscale, scalefac;

      /* ignore if zoom box is size zero */

      if (areastruct.save.x == areastruct.origin.x || areastruct.save.y ==
	  areastruct.origin.y) {
	 Wprintf("Zoom box of size zero: Ignoring.");
         XtRemoveEventHandler(areastruct.area, ButtonMotionMask, False,
	     (XtEventHandler)trackselarea, NULL);
         if ((unsigned int) mode != 0)
	    eventmode = NORMAL_MODE;
	 else
            eventmode = DESEL_MODE;
	 return;
      }

      /* determine whether x or y is limiting factor in zoom */
      delxscale = abs(areastruct.save.x - areastruct.origin.x) /
	   (areastruct.width / (*areastruct.vscale));
      delyscale = abs(areastruct.save.y - areastruct.origin.y) /
	   (areastruct.height / (*areastruct.vscale));
      scalefac = min(delxscale, delyscale);
      (*areastruct.vscale) *= scalefac;

      /* compute lower-left corner of (reshaped) select box */
      if (delxscale < delyscale) {
         newll.y = min(areastruct.save.y, areastruct.origin.y);
	 newll.x = (areastruct.save.x + areastruct.origin.x
		- (abs(areastruct.save.y - areastruct.origin.y) *
		areastruct.width / areastruct.height)) / 2;
      }
      else {
         newll.x = min(areastruct.save.x, areastruct.origin.x);
	 newll.y = (areastruct.save.y + areastruct.origin.y
		- (abs(areastruct.save.x - areastruct.origin.x) *
		areastruct.height / areastruct.width)) / 2;
      }

      /* extrapolate to find new lower-left corner of screen */
      newll.x = areastruct.lowerleft->x - (int)((float)(newll.x -
		areastruct.lowerleft->x) / scalefac);
      newll.y = areastruct.lowerleft->y - (int)((float)(newll.y -
		areastruct.lowerleft->y) / scalefac);

      XtRemoveEventHandler(areastruct.area, ButtonMotionMask, False,
	  (XtEventHandler)trackselarea, NULL);
      if ((unsigned int) mode != 0)
	 eventmode = NORMAL_MODE;
      else
         eventmode = DESEL_MODE;
   }
   else {
      window_to_user(areastruct.width / 2, areastruct.height / 2, &ucenter); 
      (*areastruct.vscale) /= SCALEFAC;
      window_to_user(areastruct.width / 2, areastruct.height / 2, &ncenter); 
      newll.x = (long)areastruct.lowerleft->x + (long)(ucenter.x - ncenter.x);
      newll.y = (long)areastruct.lowerleft->y + (long)(ucenter.y - ncenter.y);
   }
   areastruct.lowerleft->x = (short)newll.x;
   areastruct.lowerleft->y = (short)newll.y;

   if ((newll.x << 1) != (long)(areastruct.lowerleft->x << 1) || (newll.y << 1)
	 != (long)(areastruct.lowerleft->y << 1) || checkbounds() == -1) {
      (*areastruct.vscale) = savescale; 
      areastruct.lowerleft->x = savell.x;
      areastruct.lowerleft->y = savell.y;
      Wprintf("At maximum scale: cannot scale further.");
      return;
   }
   else if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      drag(areastruct.area, NULL, event);

   Wprintf(" ");
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawarea(button, objectdata, NULL);
   drawhbar(areastruct.scrollbarh, NULL, Number(0));
   drawvbar(areastruct.scrollbarv, NULL, Number(1));
}

/*--------------------------------------------------------------*/
/* Pan screen so that cursor position moves to center of screen */
/*--------------------------------------------------------------*/

void centerpan(button, clientdata, calldata)
  Widget	button;
  caddr_t	clientdata, calldata;
{
   Wprintf("Click button1 to pan.");
   XDefineCursor (dpy, win, CIRCLE); 
   if (eventmode == CATALOG_MODE) eventmode = CATPAN_MODE;
   else eventmode = PAN_MODE;
}

/*----------------------------------------------------------------*/
/* Check for out-of-bounds before warping pointer, and pan window */
/* if necessary.                                                  */
/*----------------------------------------------------------------*/

checkwarp(userpt)
  XPoint *userpt;
{
  XPoint wpoint;
  XButtonEvent event;

  user_to_window(*userpt, &wpoint);

  if (wpoint.x < 0 || wpoint.y < 0 || wpoint.x > areastruct.width ||
        wpoint.y > areastruct.height) {
     event.x = wpoint.x;
     event.y = wpoint.y;
     panbutton(areastruct.area, Number(5), &event); 
     wpoint.x = areastruct.width >> 1;
     wpoint.y = areastruct.height >> 1;
     snap(wpoint.x, wpoint.y, &userpt);
  }
  XWarpPointer(dpy, None, win, 0, 0, 0, 0, wpoint.x, wpoint.y);
}

/*--------------------------------------------------------------*/
/* Change to next edit point on a polygon			*/
/*--------------------------------------------------------------*/

nextpolycycle(nextpoly, dir)
  polyptr nextpoly;
  short dir;
{
   XPoint *polypt;

   areastruct.editcycle += dir;
   if (areastruct.editcycle < 0) areastruct.editcycle += nextpoly->number;
   areastruct.editcycle %= nextpoly->number;
   finddir(nextpoly);
   Wprintf("Move point, x=break d=delete i=insert");

   polypt = nextpoly->points + areastruct.editcycle;
   checkwarp(polypt);
}

/*--------------------------------------------------------------*/
/* Change to next edit cycle on a spline			*/
/*--------------------------------------------------------------*/

nextsplinecycle(nextspline, dir)
  splineptr nextspline;
  short dir;
{
   areastruct.editcycle += dir;
   if (areastruct.editcycle < 0) areastruct.editcycle += 4;
   areastruct.editcycle %= 4;

   if (areastruct.editcycle == 1 || areastruct.editcycle == 2)
      Wprintf("Adjust control point");
   else
      Wprintf("Adjust endpoint position");

   checkwarp(&nextspline->ctrl[areastruct.editcycle]);
}

/*--------------------------------------------------------------*/
/* Change to next edit cycle on an arc				*/
/*--------------------------------------------------------------*/

nextarccycle(nextarc, dir)
  arcptr nextarc;
  short dir;
{
   XPoint curang;
   double rad;

   areastruct.editcycle += dir;
   if (areastruct.editcycle < 0) areastruct.editcycle += 4;
   areastruct.editcycle %= 4;

   switch(areastruct.editcycle) {
      case 0:
	 curang.x = nextarc->position.x + abs(nextarc->radius);
	 curang.y = nextarc->position.y;
	 if (abs(nextarc->radius) != nextarc->yaxis)
	    Wprintf("Adjust ellipse size");
	 else
	    Wprintf("Adjust arc radius");
	 break;
      case 3:
	 curang.x = nextarc->position.x;
	 curang.y = nextarc->position.y + nextarc->yaxis;
	 Wprintf("Adjust ellipse minor axis");
	 break;
      case 1:
         rad = (double)(nextarc->angle1 * RADFAC);
         curang.x = nextarc->position.x + abs(nextarc->radius) * cos(rad);
         curang.y = nextarc->position.y + nextarc->yaxis * sin(rad);
	 Wprintf("Adjust arc endpoint");
  	 break;
      case 2:
         rad = (double)(nextarc->angle2 * RADFAC);
         curang.x = nextarc->position.x + abs(nextarc->radius) * cos(rad);
         curang.y = nextarc->position.y + nextarc->yaxis * sin(rad);
	 Wprintf("Adjust arc endpoint");
	 break;
      }
   checkwarp(&curang);
}

/*------------------------*/
/* Handle keyboard inputs */
/*------------------------*/

void keyhandler(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XKeyEvent *event;
{
   short userx, usery;
   int keystate = (event->state & 4) ? 1 : 0;
   KeySym keypressed;
   if (popups > 0 && help_up == 0) return;

   /* Key events during a button press are like regular key events */
   /* (No press activity defined for Button3)			   */
   /* Mod2Mask seems to equate to NumLock on my SGI machine. . .   */

   if (event->state & (Button1Mask | Button2Mask | Mod2Mask))
	event->state &= ~(Button1Mask | Button2Mask | Mod2Mask);
/*  keypressed = XKeycodeToKeysym(dpy, event->keycode, keystate ? 1 : event->state); */

   XLookupString(event, _STR, 255, &keypressed, NULL);

   ((XButtonEvent *)event)->button = Button1;

   /* Translation of some function keys */
   if (keypressed == XK_Redo) keypressed = XK_space;
   if (keypressed == XK_Undo) keypressed = XK_u;

   /* A few event-specific things */

   if ((eventmode == COPY2_MODE || eventmode == PRESS_MODE ||
	  eventmode == EBOX_MODE) && keypressed == XK_A) {

      /* Conditions: One element is selected, key "A" is pressed.	*/
      /* Then there must exist a spline, polygon, or arc to attach to.	*/

      if (areastruct.selects <= 1) {
	 short *refsel;

         if (attachto == 1) {
	    attachto = 0;	/* toggle value */
	    Wprintf("Unconstrained moving");
	 }
	 else {
	    attachto = 1;
	    if ((refsel = objectselect(SPLINE|ARC|POLYGON)) != NULL) {
	
	       /* transfer refsel over to (global) refselect */

	       refselect = *refsel;
	       areastruct.selects--;
	       XSetFunction(dpy, areastruct.gc, GXcopy);
	       XTopSetForeground(SELTOCOLOR(refsel));
	       easydraw(refselect, DEFAULTCOLOR);

	       /* restore graphics state */
	       XSetFunction(dpy, areastruct.gc, areastruct.gctype);
	       XSetForeground(dpy, areastruct.gc, areastruct.gccolor);

	       Wprintf("Constrained attach");
	    }
	    else {
	       attachto = 0;
	       Wprintf("Nothing found to attach to");
	    }
         }
      }
   }
   else if (eventmode == COPY2_MODE || eventmode == NORMAL_MODE || 
	eventmode == PRESS_MODE) {
      if (keypressed >= XK_0 && keypressed <= XK_9) {
         short pageno = (short)(keypressed - XK_1);
         if (pageno < 0) pageno = 9;
         newpage(pageno);
	 return;
      }
      else if ((keypressed >= min(XK_KP_1, XK_KP_9)) && (keypressed <= 
	    max(XK_KP_1, XK_KP_9))) {
         rejustify((short)(keypressed - min(XK_KP_1, XK_KP_9)));
      }
   }
   else if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      if ((keypressed >= min(XK_KP_1, XK_KP_9)) && (keypressed <= 
	    max(XK_KP_1, XK_KP_9))) {
         rejustify((short)(keypressed - min(XK_KP_1, XK_KP_9)));
      }
      else if (keypressed == XK_KP_Add) {
	 keypressed = SUPERSCRIPT;
	 keystate = 2;
      }
      else if (keypressed == XK_KP_Subtract) {
	 keypressed = SUBSCRIPT;
	 keystate = 2;
      }
      else if (keypressed == XK_KP_Enter) {
	 keypressed = NORMALSCRIPT;
	 keystate = 2;
      }
      labeltext(keypressed, keystate);
      return;
   }
   else if (eventmode == CATTEXT_MODE) {
      if ((keypressed > 31 && keypressed < 128) || keypressed == XK_Delete ||
	     keypressed == XK_BackSpace || keypressed == XK_Return ||
	     keypressed == XK_Right || keypressed == XK_Left)
         labeltext(keypressed, keystate);
      return;
   }

   /* Ways of changing functionality while in edit mode */
   /* (this needs to be substantially revised)		*/

   else if (eventmode == EBOX_MODE || eventmode == EARC_MODE ||
	eventmode == ESPLINE_MODE || eventmode == EPATH_MODE) {
      genericptr *keygen = EDITPART;
      if ((*keygen)->type == PATH)
	 keygen = (*((pathptr *)EDITPART))->plist + areastruct.editsubpart;

      switch((*keygen)->type) {
	 case POLYGON: {
   	    polyptr lwire = TOPOLY(keygen);
            XPoint *lpoint;

            /* Break the polygon at the point, if the point isn't an endpoint */
            if (keypressed == XK_x) {
	       polyptr *newpoly;
	       XPoint *npoint;

	       if (areastruct.editcycle == lwire->number - 1 ||
			areastruct.editcycle == 0)
	          return;
	       UDrawPolygon(areastruct.topobject, lwire);
	       lwire->style |= UNCLOSED;
	       NEW_POLY(newpoly, objectdata);
	       objectdata->parts++;
	       (*newpoly)->style = lwire->style;
	       (*newpoly)->color = lwire->color;
	       (*newpoly)->width = lwire->width;
	       (*newpoly)->number = lwire->number - areastruct.editcycle;
	       (*newpoly)->points = (XPoint *)malloc((*newpoly)->number *
		      sizeof(XPoint));
	       lpoint = lwire->points + areastruct.editcycle;
	       for (npoint = (*newpoly)->points; npoint < (*newpoly)->points +
	             (*newpoly)->number; npoint++) {
	          npoint->x = lpoint->x;
	          npoint->y = (lpoint++)->y;
	       }
	       lwire->number = areastruct.editcycle + 1;
               areastruct.editcycle = 0;
	       reset(areastruct.editstack, NORMAL);

	       UDrawPolygon(areastruct.topobject, lwire);
	       lwire = (*newpoly);
	       UDrawPolygon(areastruct.topobject, lwire);
            }

            /* Remove a point from the polygon */
            else if (keypressed == XK_Delete || keypressed == XK_d) {
	       if (lwire->number < 3) return;
	       UDrawPolygon(areastruct.topobject, lwire);
	       if (lwire->number == 3 && !(lwire->style & UNCLOSED)) 
	          lwire->style |= UNCLOSED;
	       lwire->number--;
	       for (lpoint = lwire->points + areastruct.editcycle; lpoint <
			lwire->points + lwire->number; lpoint++)
	          *lpoint = *(lpoint + 1);
	       UDrawPolygon(areastruct.topobject, lwire);
	       nextpolycycle(lwire, -1);
            }
            /* Add a point to the polygon */
            else if (keypressed == XK_Insert || keypressed == XK_i) {
	       UDrawPolygon(areastruct.topobject, lwire);
	       lwire->number++;
	       lwire->points = (XPoint *)realloc(lwire->points, lwire->number
		      * sizeof(XPoint));
	       for (lpoint = lwire->points + lwire->number - 1; lpoint > lwire->
		         points + areastruct.editcycle; lpoint--)
	          *lpoint = *(lpoint - 1);
	       UDrawPolygon(areastruct.topobject, lwire);
            }
            else if (keypressed == XK_e) {
               nextpolycycle(lwire, 1);
            }
            polyeditpush(lwire);
         } break;
	 case SPLINE:
	    if (keypressed == XK_e) {
   	       splineptr keyspline = TOSPLINE(keygen);
               nextsplinecycle(keyspline, -1);
               splineeditpush(keyspline);
            }
	    break;
	 case ARC:
	    if (keypressed == XK_e) {
   	       arcptr keyarc = TOARC(keygen);
               nextarccycle(keyarc, 1);
               arceditpush(keyarc);
	    }
      }
   }
   else if (eventmode == CATALOG_MODE) {
      switch(keypressed) {
	 case XK_h:
		starthelp(w, objectdata, NULL);
		break;
	 case XK_l: case XK_L:
		changecat();
		break;
	 case XK_c:
   	   	((XButtonEvent *)event)->button = Button1;
	   	catbutton(w, (void *)(1), (XButtonEvent *)event);
		break;
	 case XK_E:
           	window_to_user(event->x, event->y, &areastruct.save);
	   	objectdeselect();
	   	objectselect(LABEL);
	   	if (areastruct.selects == 1) 
	      	   edit(w, objectdata, event);  
		break;
	 case XK_D:
   	   	((XButtonEvent *)event)->button = Button2;
	   	catbutton(w, NULL, (XButtonEvent *)event);
	   	catdelete();
		break;
	 case XK_C:
   	   	((XButtonEvent *)event)->button = Button2;
	   	catbutton(w, NULL, (XButtonEvent *)event);
	   	copycat();
		break;
	 case XK_M:
           	window_to_user(event->x, event->y, &areastruct.save);
		catmove();
		break;
	 case XK_H:
		((XButtonEvent *)event)->button = Button2;
		catbutton(w, NULL, (XButtonEvent *)event);
		cathide();
		break;
	 case XK_space:
		if (objectdata == areastruct.libtop[LIBRARY])
		   composelib(LIBRARY);
		else
		   composelib(USERLIB);
		break;
         case XK_greater:
		eventmode = NORMAL_MODE;
		break;
      }
   }

   /* The following events are allowed in any mode */

   switch (keypressed) {
      case XK_Home: 
      case XK_v:     zoomview(w, Number(1), NULL);   break;
      case XK_space: drawarea(w, objectdata, NULL);  break;
      case XK_Z:     zoomin(w, Number(0), event);     break;
      case XK_z:     zoomout(w, Number(0), event);    break;
      case XK_p:     panbutton(w, Number(0), event); break;
      case XK_plus:  setsnap(1);		     break;
      case XK_minus: setsnap(-1);		     break;
      case XK_Left:  panbutton(w, Number(1), event); break;
      case XK_Right: panbutton(w, Number(2), event); break;
      case XK_Up:    panbutton(w, Number(3), event); break;
      case XK_Down:  panbutton(w, Number(4), event); break;
      case XK_W:     dooutput(NULL, NULL, NULL);     break;
   }

   /* Some events can occur in both NORMAL, PRESS, and COPY2 modes */

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE) {
      snap(event->x, event->y, &areastruct.save);
      switch (keypressed) {
	 case XK_O: objectrotate(-1);                   break;
	 case XK_R: objectrotate(-3);                   break;
	 case XK_o: objectrotate(1);                    break;
	 case XK_r: objectrotate(3);                    break;
	 case XK_f: objectflip();                       break;
	 case XK_F: objectvflip();                      break;
	 case XK_S: snapobject();                       break;
      	 case XK_less: popobject(w, Number(0), NULL);	break;
	 case XK_greater: pushobject(w, objectdata, event); break;
#ifdef SCHEMA
	 case XK_slash: swapschem(w, NULL, NULL);	break;
#endif
      }
   }

   /* The following events are restricted to normal mode */

   if (eventmode == NORMAL_MODE) {
      window_to_user(event->x, event->y, &areastruct.save);
      switch (keypressed) {
         case XK_F19: objectselect(ANY);		break;
         case XK_Delete:  deletebutton(w, objectdata, event); break;
         case XK_greater: pushobject(w, objectdata, event);  break;
      	 case XK_less: popobject(w, Number(0), NULL);	break;
#ifdef SCHEMA
	 case XK_slash: swapschem(w, NULL, NULL);	break;
	 case XK_T: eventmode = TEXT2_MODE;
	    textbutton(w, True, event);			break;
	 case XK_g: gennet("sim",     "sim");		break;
	 case XK_N: gennet("spice",   "spc");		break;
	 case XK_n: gennet("pcb",     "pcb");		break;
#endif
	 case XK_O: objectrotate(-1);			break;
         case XK_R: objectrotate(-3);    		break;
         case XK_b: boxbutton(w, objectdata, event);    break;
	 case XK_a: arcbutton(w, objectdata, event);    break;
	 case XK_t: eventmode = TEXT2_MODE;
	     textbutton(w, False, event);  		break;
         case XK_S: snapobject();             	   	break;
	 case XK_o: objectrotate(1);			break;
         case XK_r: objectrotate(3);     		break;
         case XK_f: objectflip();          	     	break;
	 case XK_F: objectvflip();			break;
	 case XK_X: exchange();				break;
         case XK_d: deletebutton(w, objectdata, event); break;
	 case XK_c: copybutton(w, objectdata, event);   break;
         case XK_l: startcatalog(w, Number(LIBRARY), NULL);   break;
	 case XK_L: startcatalog(w, Number(USERLIB), NULL);   break;
	 case XK_j: join();				break;
	 case XK_J: unjoin();				break;
	 case XK_s: splinebutton(w, objectdata, event); break;
	 case XK_e: edit(w, objectdata, event);	   	break;
	 case XK_u: undelete(w, Number(DRAW), event);	break;
	 case XK_m:
	 case XK_M: selectsave(w, objectdata, NULL);	break;
	 case XK_x: objectselect(-ANY);			break;
	 case XK_question:
	 case XK_h: starthelp(w, objectdata, NULL);	break;
	 case XK_bar: getline(NULL, Number(DASHED), NULL); break;
	 case XK_colon: getline(NULL, Number(DOTTED), NULL); break;
	 case XK_underscore: getline(NULL, Number(0), NULL); break;
	 case XK_period: snap(event->x, event->y, &areastruct.save);
		    drawdot(areastruct.save.x, areastruct.save.y);
		    drawarea(w, objectdata, NULL);	break;
      }
   }

}

/*-----------------------------------------*/
/* Reposition an object onto the snap grid */
/*-----------------------------------------*/

void snapobject()
{
   short *selectobj;

   if (!checkselect(ANY)) return;
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BACKGROUND);
   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
	+ areastruct.selects; selectobj++) {
      easydraw(*selectobj, DOFORALL);
      switch(SELECTTYPE(selectobj)) {
         case OBJECT: {
	    objinstptr snapobj = SELTOOBJINST(selectobj);

            u2u_snap(&snapobj->position);
	    } break;
         case LABEL: {
	    labelptr snaplabel = SELTOLABEL(selectobj);

	    u2u_snap(&snaplabel->position);
	    } break;
         case POLYGON: {
	    polyptr snappoly = SELTOPOLY(selectobj);
	    pointlist snappoint;

	    for (snappoint = snappoly->points; snappoint < snappoly->points +
	         snappoly->number; snappoint++)
	       u2u_snap(snappoint);
	    } break;
         case ARC: {
	    arcptr snaparc = SELTOARC(selectobj);

	    u2u_snap(&snaparc->position);
	    if (areastruct.snapto) {
	       snaparc->radius = (snaparc->radius /
		    areastruct.snapspace[areastruct.page]) *
		    areastruct.snapspace[areastruct.page];
	       snaparc->yaxis = (snaparc->yaxis /
		    areastruct.snapspace[areastruct.page]) *
	            areastruct.snapspace[areastruct.page];
	    }
	    calcarc(snaparc);
	    } break;
	 case SPLINE: {
	    splineptr snapspline = SELTOSPLINE(selectobj);
	    short i;

	    for (i = 0; i < 4; i++)
	       u2u_snap(&snapspline->ctrl[i]);
	    calcspline(snapspline);
	    } break;
      }
      if (eventmode != NORMAL_MODE) {
         XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
	 easydraw(*selectobj, DOFORALL);
      }
   }
   if (eventmode == NORMAL_MODE) objectdeselect();
}

/*-------------------------------------------*/
/* Routines to print the cursor position     */
/*-------------------------------------------*/

/* fast integer power-of-10 routine */

int ipow10(a)
   int a;
{
   int i;
   char istr[12];
   
   switch (a) {
      case 0: return 1;     break;
      case 1: return 10;    break;
      case 2: return 100;   break;
      case 3: return 1000;  break;
      default:
         istr[0] = '1';
	 for (i = 1; i < a + 1; i++) istr[i] = '0';
	 istr[i] = '\0';
	 return atoi(istr);
	 break;
   }
}

/* find greatest common factor between to integers */

int calcgcf(a, b)
   int a, b;
{
   register int mod;

   if ((mod = a % b) == 0) return (b);
   else return (calcgcf(b, mod));
}

/* generate a fraction from a float, if possible */

fraccalc(xyval, fstr)
  float xyval;
  char *fstr;
{
   short i, j, k, t, rept;
   int ip, mant, divisor, denom, numer, rpart;
   double fp;
   char num[10], *nptr = &num[2], *sptr;
   
   ip = (int)xyval;
   fp = fabs(xyval - ip);

   /* write fractional part and grab mantissa as integer */

   sprintf(num, "%1.7f", fp);
   num[8] = '\0';		/* no rounding up! */
   sscanf(nptr, "%d", &mant);

   if (mant != 0) {    /* search for repeating substrings */
      for (i = 1; i <= 3; i++) {
         rept = 1;
	 nptr = &num[8] - i;
	 while ((sptr = nptr - rept * i) >= &num[2]) {
            for (t = 0; t < i; t++)
	       if (*(sptr + t) != *(nptr + t)) break;
	    if (t != i) break;
	    else rept++;
	 }
	 if (rept > 1) break;
      }
      nptr = &num[8] - i;
      sscanf(nptr, "%d", &rpart);  /* rpart is repeating part of mantissa */
      if (i > 3 || rpart == 0) { /* no repeat */ 
	 divisor = calcgcf(1000000, mant);
	 denom = 1000000 / divisor;
      }
      else { /* repeat */
	 int z, p, fd;

	 *nptr = '\0';
	 sscanf(&num[2], "%d", &z);
	 p = ipow10(i) - 1;
	 mant = z * p + rpart;
	 fd = ipow10(nptr - &num[2]) * p;

	 divisor = calcgcf(fd, mant);
	 denom = fd / divisor;
      }
      numer = mant / divisor;
      if (denom > 1024)
	 sprintf(fstr, "%5.3f", xyval); 
      else if (ip == 0)
         sprintf(fstr, "%hd/%hd", (xyval > 0) ? numer : -numer, denom);
      else
         sprintf(fstr, "%hd %hd/%hd", ip, numer, denom);
   }
   else sprintf(fstr, "%hd", ip);
}

printpos(xval, yval)
  short xval, yval;
{
   float f1, f2, f3;
   float oscale, iscale = (float)areastruct.drawingscale[areastruct.page].y /
	(float)areastruct.drawingscale[areastruct.page].x;
   int llen;
   unsigned char wlflag = False;
   XPoint *tpoint;

   if (eventmode == EBOX_MODE || eventmode == WIRE_MODE) {  /* print wire length */
      polyptr lwire = TOPOLY(EDITPART);
      tpoint = lwire->points + lwire->number - 1;
      llen = wirelength(tpoint - 1, tpoint);
      wlflag = True;
   }

   switch (areastruct.coordstyle[areastruct.page]) {
      case DEC_INCH:
         oscale = areastruct.outscale[areastruct.page] * INCHSCALE;
         f1 = ((float)(xval) * iscale * oscale) / 72.0;
         f2 = ((float)(yval) * iscale * oscale) / 72.0;
	 if (wlflag == True) {
            f3 = ((float)(llen) * iscale * oscale) / 72.0;
   	    sprintf(_STR, "%5.3f, %5.3f in  (length %5.3f in)", f1, f2, f3);
	 }
	 else sprintf(_STR, "%5.3f, %5.3f in", f1, f2);
	 break;
      case FRAC_INCH: {
	 char fstr1[30], fstr2[30], fstr3[30];
	 
         oscale = areastruct.outscale[areastruct.page] * INCHSCALE;
	 fraccalc((((float)(xval) * iscale * oscale) / 72.0), fstr1); 
         fraccalc((((float)(yval) * iscale * oscale) / 72.0), fstr2);
	 if (wlflag == True) {
	    fraccalc((((float)(llen) * iscale * oscale) / 72.0), fstr3);
	    sprintf(_STR, "%s, %s in  (length %s in)", fstr1, fstr2, fstr3);
	 }
   	 else sprintf(_STR, "%s, %s in", fstr1, fstr2); }
	 break;
      case CM:
         oscale = areastruct.outscale[areastruct.page] * CMSCALE;
         f1 = ((float)(xval) * iscale * oscale) / IN_CM_CONVERT;
         f2 = ((float)(yval) * iscale * oscale) / IN_CM_CONVERT;
	 if (wlflag == True) {
            f3 = ((float)(llen) * iscale * oscale) / IN_CM_CONVERT;
   	    sprintf(_STR, "%5.3f, %5.3f cm  (length %5.3f cm)", f1, f2, f3);
	 }
   	 else sprintf(_STR, "%5.3f, %5.3f cm", f1, f2);
	 break;
   }
   W1printf(_STR);
}

/*---------------------------------------------------*/
/* Find nearest point of intersection of the cursor  */
/* position to a wire and move there.		     */
/*---------------------------------------------------*/

findwirex(endpt1, endpt2, userpt, newpos, rot)
  XPoint *endpt1, *endpt2, *userpt, *newpos;
  int *rot;
{
   long xsq, ysq, zsq;
   float frac;

   xsq = sqwirelen(endpt1, endpt2);
   ysq = sqwirelen(endpt1, userpt);
   zsq = sqwirelen(endpt2, userpt);
   frac = 0.5 + (float)(ysq - zsq) / (float)(xsq << 1);
   if (frac > 1) frac = 1;
   else if (frac < 0) frac = 0;
   newpos->x = endpt1->x + (int)((endpt2->x - endpt1->x) * frac);
   newpos->y = endpt1->y + (int)((endpt2->y - endpt1->y) * frac);

   *rot = 180 + (int)(INVRFAC * atan2((double)(endpt1->x -
	  endpt2->x), (double)(endpt1->y - endpt2->y)));

   /* make adjustment for nearest-integer calculation 	 */
   /* ((*rot)++, (*rot)-- works if (360 / RSTEPS >= 2))  */

   if (*rot > 0)
      (*rot)++;
   else if (*rot < 0)
      (*rot)--;
}

/*----------------------------------------------------------------*/
/* Find the closest point of attachment from the pointer position */
/* to the "refselect" element.					  */
/*----------------------------------------------------------------*/

findattach(newpos, rot, userpt)
   XPoint *newpos, *userpt;
   int *rot;
{
   XPoint *endpt1, *endpt2;
   float frac;
   double tmpang;

   /* find point of intersection and slope */

   if (SELECTTYPE(&refselect) == ARC) {
      arcptr aarc = SELTOARC(&refselect);
      float tmpdeg;
      tmpang = atan2((double)(userpt->y - aarc->position.y) * (double)
		(abs(aarc->radius)), (double)(userpt->x - aarc->position.x) *
		(double)aarc->yaxis);

      /* don't follow the arc beyond its endpoints */

      tmpdeg = (float)(tmpang * INVRFAC);
      if (tmpdeg < 0) tmpdeg += 360;
      if (((aarc->angle2 > 360) && (tmpdeg > aarc->angle2 - 360) &&
		(tmpdeg < aarc->angle1)) ||
	  	((aarc->angle1 < 0) && (tmpdeg > aarc->angle2) &&
		(tmpdeg < aarc->angle1 + 360)) ||
	  	((aarc->angle1 >= 0) && (aarc->angle2 <= 360) && ((tmpdeg
		> aarc->angle2) || (tmpdeg < aarc->angle1)))) {
	 float testd1 = aarc->angle1 - tmpdeg;
	 float testd2 = tmpdeg - aarc->angle2;
	 if (testd1 < 0) testd1 += 360;
	 if (testd2 < 0) testd2 += 360;

	 /* if arc is closed, attach to the line between the endpoints */

	 if (!(aarc->style & UNCLOSED)) {
	    XPoint end1, end2;
	    tmpang = (double) aarc->angle1 / INVRFAC;
	    end1.x = aarc->position.x + abs(aarc->radius) * cos(tmpang);
	    end1.y = aarc->position.y + aarc->yaxis  * sin(tmpang);
	    tmpang = (double) aarc->angle2 / INVRFAC;
	    end2.x = aarc->position.x + abs(aarc->radius) * cos(tmpang);
	    end2.y = aarc->position.y + aarc->yaxis  * sin(tmpang);
            findwirex(&end1, &end2, userpt, newpos, rot);
	    return;
	 }
	 else
	    tmpang = (double)((testd1 < testd2) ? aarc->angle1 : aarc->angle2)
			/ INVRFAC;
      }

      /* get position in user coordinates nearest to the intersect pt */

      newpos->x = aarc->position.x + abs(aarc->radius) * cos(tmpang);
      newpos->y = aarc->position.y + aarc->yaxis  * sin(tmpang);

      /* rotation of object is normal to the curve of the ellipse */

      *rot = 90 - (int)(INVRFAC * tmpang);
      if (*rot < 0) *rot += 360;
   }
   else if (SELECTTYPE(&refselect) == SPLINE) {
       splineptr aspline = SELTOSPLINE(&refselect);
       frac = findsplinemin(aspline, userpt);
       findsplinepos(aspline, frac, newpos, rot);
   }
   else if (SELECTTYPE(&refselect) == POLYGON) {
       polyptr apoly = SELTOPOLY(&refselect);
       XPoint *testpt, *minpt, *nxtpt;
       long mindist = 1000000, testdist;
       for (testpt = apoly->points; testpt < apoly->points +
		apoly->number - 1; testpt++) {
	 testdist =  finddist(testpt, testpt + 1, userpt);
	 if (testdist < mindist) {
	    mindist = testdist;
	    minpt = testpt;
	    nxtpt = testpt + 1;
	 }
      }
      if (!(apoly->style & UNCLOSED)) {
	 testdist = finddist(testpt, apoly->points, userpt);
	 if (testdist < mindist) {
	    mindist = testdist;
	    minpt = testpt;
	    nxtpt = apoly->points;
	 }
      }
      endpt1 = minpt;
      endpt2 = nxtpt;
      findwirex(endpt1, endpt2, userpt, newpos, rot);
   }
}

/*--------------------------------------------*/
/* Find closest point in a path to the cursor */
/*--------------------------------------------*/

XPoint *pathclosepoint(dragpath, newpos)
  pathptr dragpath;
  XPoint  *newpos;
{
   XPoint *rpoint;
   genericptr *cpoint;
   short mpoint;
   int mdist = 1000000, tdist, t2dist;

   for (cpoint = dragpath->plist; cpoint < dragpath->plist + dragpath->parts;
	   cpoint++) {
      switch((*cpoint)->type) {
	 case ARC:
	   tdist = wirelength(&(TOARC(cpoint)->position), newpos);
	    if (tdist < mdist) {
	       mdist = tdist;
	       rpoint = &(TOARC(cpoint)->position);
	    }
	    break;
	 case POLYGON:
	    mpoint = closepoint(TOPOLY(cpoint), newpos);
	    tdist = wirelength(TOPOLY(cpoint)->points + mpoint, newpos);
	    if (tdist < mdist) {
	       mdist = tdist;
	       rpoint = TOPOLY(cpoint)->points + mpoint;
	    }
	    break;
	 case SPLINE:
	    tdist = wirelength(&(TOSPLINE(cpoint)->ctrl[0]), newpos);
	    if (tdist < mdist) {
	       mdist = tdist;
	       rpoint = &(TOSPLINE(cpoint)->ctrl[0]);
	    }
	    tdist = wirelength(&(TOSPLINE(cpoint)->ctrl[3]), newpos);
	    if (tdist < mdist) {
	       mdist = tdist;
	       rpoint = &(TOSPLINE(cpoint)->ctrl[3]);
	    }
	    break;
      }
   }
   return rpoint;
}

/*-------------------------------------------*/
/* Drag a selected element around the screen */
/*-------------------------------------------*/

XtEventHandler drag(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XButtonEvent *event;
{
   XEvent again;
   Boolean eventcheck = False;
   XPoint userpt, newpos;
   short deltax, deltay, closest;
   short *dragselect;
   int rot;

   /* flush out multiple pointermotion events from the event queue */
   /* use only the last motion event */
   while (XCheckWindowEvent(dpy, win, PointerMotionMask |
	Button1MotionMask, &again) == True) eventcheck = True;
   if (eventcheck) event = (XButtonEvent *)&again;

   snap(event->x, event->y, &userpt);
   deltax = userpt.x - areastruct.save.x;
   deltay = userpt.y - areastruct.save.y;
   if (deltax == 0 && deltay == 0) return;
   printpos(userpt.x, userpt.y);
   areastruct.save.x = userpt.x;
   areastruct.save.y = userpt.y;

   /* set up the graphics state for moving a selected object */

   XSetXORFg(SELECTCOLOR, BACKGROUND);
   XSetFunction(dpy, areastruct.gc, GXxor);

   /* under attachto condition, keep element attached to */
   /* the refselect element.				 */

   if (attachto) findattach(&newpos, &rot, &userpt);

   for (dragselect = areastruct.selectlist; dragselect < areastruct.selectlist
      + areastruct.selects; dragselect++) {

      switch(SELECTTYPE(dragselect)) {
         case OBJECT: {
	    objinstptr dragobject = SELTOOBJINST(dragselect);
	    UDrawObject(dragobject, dragobject->thisobject, SINGLE, DOFORALL);
	    if (attachto) {
	       dragobject->position.x = newpos.x;
	       dragobject->position.y = newpos.y;
	       if (dragobject->rotation > 0) {
	          if (abs(rot - (((dragobject->rotation - 1) * 360) / RSTEPS))
			> 90) rot += 180;
	          if (rot >= 360) rot -= 360;
	          dragobject->rotation = ((rot * RSTEPS) / 360) + 1;
	       }
	       else {
	          if (abs(rot - (((dragobject->rotation + 1) * 360) / RSTEPS))
			> 90) rot = 180 - rot;
	          if (rot > 0) rot -= 360;
	          dragobject->rotation = ((rot * RSTEPS) / 360) - 1;
	       }
	    }
	    else {
	       dragobject->position.x += deltax;
	       dragobject->position.y += deltay;
	    }
	    UDrawObject(dragobject, dragobject->thisobject, SINGLE, DOFORALL);
	 } break;
	 case LABEL: {
	    labelptr draglabel = SELTOLABEL(dragselect);
	    UDrawString(areastruct.topobject, draglabel);
#ifdef SCHEMA
	    if (draglabel->pin == False)
#endif
	    UDrawX(draglabel);
	    if (attachto) {
	       draglabel->position.x = newpos.x;
	       draglabel->position.y = newpos.y;
	       draglabel->rotation = ((rot * RSTEPS) / 360) + 1;
	    }
	    else {
	       draglabel->position.x += deltax;
	       draglabel->position.y += deltay;
	    }
	    UDrawString(areastruct.topobject, draglabel);
#ifdef SCHEMA
	    if (draglabel->pin == False)
#endif
	    UDrawX(draglabel);
	 } break;
	 case PATH: {
	    pathptr dragpath = SELTOPATH(dragselect);
	    genericptr *pathlist;
	    
	    UDrawPath(areastruct.topobject, dragpath);
	    if (attachto) {
	       XPoint *pdelta = pathclosepoint(dragpath, &newpos);
	       deltax = newpos.x - pdelta->x;
	       deltay = newpos.y - pdelta->y;
	    }
	    for (pathlist = dragpath->plist;  pathlist < dragpath->plist
		  + dragpath->parts; pathlist++) {
	       movepoints(pathlist, deltax, deltay);
	    }
	    UDrawPath(areastruct.topobject, dragpath);
	 } break;
	 case POLYGON: {
	    polyptr dragpoly = SELTOPOLY(dragselect);
	    pointlist dragpoints;

	    UDrawPolygon(areastruct.topobject, dragpoly);
	    if (attachto) {
	       closest = closepoint(dragpoly, &newpos);
	       deltax = newpos.x - dragpoly->points[closest].x;
	       deltay = newpos.y - dragpoly->points[closest].y;
	    }
	    for (dragpoints = dragpoly->points; dragpoints < dragpoly->points
	           + dragpoly->number; dragpoints++) {
	       dragpoints->x += deltax;
	       dragpoints->y += deltay;
	    }
	    UDrawPolygon(areastruct.topobject, dragpoly);
	 } break;   
	 case SPLINE: {
	    splineptr dragspline = SELTOSPLINE(dragselect);
	    short j;
	    fpointlist dragpoints;

	    UDrawSpline(areastruct.topobject, dragspline);
	    if (attachto) {
	       closest = (wirelength(&dragspline->ctrl[0], &newpos)
		  > wirelength(&dragspline->ctrl[3], &newpos)) ? 3 : 0;
	       deltax = newpos.x - dragspline->ctrl[closest].x;
	       deltay = newpos.y - dragspline->ctrl[closest].y;
	    }
	    for (dragpoints = dragspline->points; dragpoints < dragspline->
		   points + INTSEGS; dragpoints++) {
	       dragpoints->x += deltax;
	       dragpoints->y += deltay;
	    }
	    for (j = 0; j < 4; j++) {
	       dragspline->ctrl[j].x += deltax;
	       dragspline->ctrl[j].y += deltay;
	    }
	    UDrawSpline(areastruct.topobject, dragspline);
	 } break;
	 case ARC: {
	    arcptr dragarc = SELTOARC(dragselect);
	    fpointlist dragpoints;

	    UDrawArc(areastruct.topobject, dragarc);
	    if (attachto) {
	       deltax = newpos.x - dragarc->position.x;
	       deltay = newpos.y - dragarc->position.y;
	    }
	    dragarc->position.x += deltax;
	    dragarc->position.y += deltay;
	    for (dragpoints = dragarc->points; dragpoints < dragarc->
		 points + dragarc->number; dragpoints++) {
	       dragpoints->x += deltax;
	       dragpoints->y += deltay;
	    }
	    UDrawArc(areastruct.topobject, dragarc);
         } break;
      }
   }

   /* restore graphics state */

   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
}

/*-------------------------------------------------*/
/* Wrapper for object rotate routine               */
/*-------------------------------------------------*/
   
void rotatebutton(w, clientdata, event)
  Widget w;
  objectptr     clientdata;
  XButtonEvent *event;
{     
   window_to_user(event->x, event->y, &areastruct.save);
         
   if (event->button == Button1) {
      if (!checkselect(ANY)) return;
      if (saverot == 64) objectflip();
      else if (saverot == 65) objectvflip();
      else objectrotate(saverot);
      objectdeselect();
   }     
   else if (event->button == Button2)
      objectselect(ANY);
   if (event->button == Button3) {
      objectdeselect();
      eventmode = NORMAL_MODE;
      XDefineCursor (dpy, win, CROSS);
   }
}

/*-------------------------------------------------*/
/* Rotate an object or label			   */
/*-------------------------------------------------*/

void objectrotate(direction)
  short	direction;
{
   short      *selectobj;
   Boolean    flipflag = False;

   if (!checkselect(ANY)) return;
   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
	+ areastruct.selects; selectobj++) { 
      short ldirec = direction;

      /* erase the object */
      XSetFunction(dpy, areastruct.gc, GXcopy);
      XSetForeground(dpy, areastruct.gc, BACKGROUND);
      easydraw(*selectobj, DOFORALL);

      switch(SELECTTYPE(selectobj)) {

	 case(OBJECT):{
            objinstptr rotateobj = SELTOOBJINST(selectobj);

            /* rotate the object and redisplay */
	    if (rotateobj->rotation < 0) {
	       flipflag = True;
	       rotateobj->rotation = -rotateobj->rotation;
	    }
            rotateobj->rotation += ldirec;
            if (rotateobj->rotation <= 0) rotateobj->rotation += RSTEPS;
            else if (rotateobj->rotation > RSTEPS) rotateobj->rotation -= RSTEPS;
	    if (flipflag == True) rotateobj->rotation = -rotateobj->rotation;
	    }break;

	 case(LABEL):{
            labelptr rotatetext = SELTOLABEL(selectobj);

            /* rotate the label and redisplay (note that text can't be flipped) */
            rotatetext->rotation += ldirec;
            if (rotatetext->rotation <= 0) rotatetext->rotation += RSTEPS;
            else if (rotatetext->rotation > RSTEPS)
	       rotatetext->rotation -= RSTEPS;
	    }break;

	 case(ARC):{
	    arcptr rotatearc = SELTOARC(selectobj);

	    rotatearc->angle1 += (float)(ROT_UNIT * ldirec);
	    rotatearc->angle2 += (float)(ROT_UNIT * ldirec);
            if (rotatearc->angle1 >= 360) {
               rotatearc->angle1 -= 360;
               rotatearc->angle2 -= 360;
            }
            else if (rotatearc->angle2 <= 0) {
               rotatearc->angle1 += 360;
               rotatearc->angle2 += 360;
            } 
	    calcarc(rotatearc);
	    }break;

	 case(SPLINE):{
	    splineptr rotatespline = SELTOSPLINE(selectobj);
	    XPoint newctrl[4], zeropt;

	    zeropt.x = 0; zeropt.y = 0;
	    ldirec++;
            if (ldirec <= 0) ldirec += RSTEPS;
	    UTransformPoints(rotatespline->ctrl, newctrl, 4, zeropt, 1.0, ldirec);
	    zeropt.x = rotatespline->ctrl[0].x - newctrl[0].x;
	    zeropt.y = rotatespline->ctrl[0].y - newctrl[0].y;
	    UTransformPoints(newctrl, rotatespline->ctrl, 4, zeropt, 1.0, 0);
	    calcspline(rotatespline);
	    }break;

	 case(POLYGON):{
	    polyptr rotatepoly = SELTOPOLY(selectobj);
	    XPoint *newpts, zeropt;

	    zeropt.x = 0; zeropt.y = 0;
	    ldirec++;
            if (ldirec <= 0) ldirec += RSTEPS;
	    newpts = (XPoint *)malloc(rotatepoly->number * sizeof(XPoint));
	    UTransformPoints(rotatepoly->points, newpts, rotatepoly->number,
		   zeropt, 1.0, ldirec);
	    zeropt.x = rotatepoly->points[0].x - newpts[0].x;
	    zeropt.y = rotatepoly->points[0].y - newpts[0].y;
	    UTransformPoints(newpts, rotatepoly->points, rotatepoly->number,
		   zeropt, 1.0, 0);
	    free(newpts);
	    }break;
      }
      if (eventmode != NORMAL_MODE) {
	 XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
	 easydraw(*selectobj, DOFORALL);
      }
   }
   if (eventmode == NORMAL_MODE) objectdeselect();
}

/*-------------------------------------------------*/
/* Edit an element in an element-dependent fashion */
/*-------------------------------------------------*/

void edit(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XButtonEvent *event;
{
   short *selectobj;

   if (areastruct.selects != 1) selectobj = objectselect(LABEL |
	POLYGON | SPLINE | ARC | PATH);
   else selectobj = areastruct.selectlist;
   if (areastruct.selects != 1) {
      if (areastruct.selects > 1) Wprintf("Select one only to edit");
      objectdeselect();
      return;
   }
   areastruct.editpart = *selectobj;

   XDefineCursor (dpy, win, EDCURSOR);
   switch(SELECTTYPE(selectobj)) {
       case LABEL: {
	 labelptr *lastlabel = (labelptr *)EDITPART;
	 short tmplength, tmpheight, curfont;
	 uchar *sptr;
	
	 objectdeselect();

	 /* save the old string */
	 labelbuf = (char *) malloc ((strlen((*lastlabel)->string) + 1) *
	    sizeof(char));
	 strcpy(labelbuf, (*lastlabel)->string);
	 
	 /* place text cursor line at end of text */

	 for (sptr = (*lastlabel)->string; *sptr != '\0'; sptr++)
            if (*sptr == TEXT_ESC) curfont = (int)(*(sptr + 1)) - FONT_START;
	 textpos = sptr - (*lastlabel)->string;

	 /* change menu buttons accordingly */

	 setfontmarks(curfont, (*lastlabel)->justify);

         tmplength = ULength((*lastlabel)->string, (*lastlabel)->scale, 0);
         tmpheight = (short)((*lastlabel)->scale * TEXTHEIGHT);
         areastruct.origin.x = (*lastlabel)->position.x + ((*lastlabel)->
	    justify & NOTLEFT ? ((*lastlabel)->justify & RIGHT ? 0 : tmplength
	    / 2) : tmplength);
         areastruct.origin.y = (*lastlabel)->position.y + ((*lastlabel)->
	    justify & NOTBOTTOM ? ((*lastlabel)->justify & TOP ? -tmpheight :
	    -tmpheight / 2) : 0);
#ifdef SCHEMA
	 if ((*lastlabel)->pin)
	    pinadjust((*lastlabel)->justify, &(areastruct.origin.x),
		&(areastruct.origin.y));
#endif
         UDrawTLine(*lastlabel);

	 if (eventmode == CATALOG_MODE) eventmode = CATTEXT_MODE;
	 else eventmode = TEXT3_MODE;
         XDefineCursor(dpy, win, TEXTPTR);

	 /* write the text at the bottom */

	 charreport(*lastlabel);
      } break;
      case PATH: {
	 /* choose editcycle type etc. from the type of the first */
	 /* part encountered (for now?).			  */         

	 pathedit((*((pathptr *)EDITPART))->plist + areastruct.editsubpart,
		PATH);

      } break;
      case POLYGON: case ARC: case SPLINE:
	 pathedit(EDITPART, 0);
	 break;
   }
}

/*------------------------------------------------------------------*/
/* edit() routine for path parts				    */
/*------------------------------------------------------------------*/

void pathedit(genericptr *editpart, short mode)
{
   switch((*editpart)->type) {
      case POLYGON: {
	 polyptr *lastpoly = (polyptr *)editpart;
	 XPoint editpt, *savept, *npt, *lpt;

	 objectdeselect();

	 /* determine which point of polygon is closest to cursor */
	 areastruct.editcycle = closepoint(*lastpoly, &areastruct.save);
         savept = (*lastpoly)->points + areastruct.editcycle;

	 /* Push onto the editstack */
	 polyeditpush(*lastpoly);

	 checkwarp(savept);

	 XcSetXORFg((*lastpoly)->color, BACKGROUND);
	 XcSetFunction(GXxor);

	 finddir(*lastpoly);

	 XtAddEventHandler(areastruct.area, PointerMotionMask, False,
	    (XtEventHandler)trackpoly, NULL);
	 eventmode = (mode == PATH) ? EPATH_MODE : EBOX_MODE;
	 Wprintf("d=delete i=insert x=break");
      } break;
      case SPLINE: {
	 splineptr *lastspline = (splineptr *)editpart;
	 XPoint *curpt, savept;

	 objectdeselect();

	 /* find which point is closest to the cursor */

         areastruct.editcycle =  (wirelength(&(*lastspline)->ctrl[0],
	      &areastruct.save) < wirelength(&(*lastspline)->ctrl[3],
	      &areastruct.save)) ? 0 : 3;
	 curpt = &(*lastspline)->ctrl[areastruct.editcycle];

	 /* Push onto the editstack */
	 splineeditpush(*lastspline);

         checkwarp(curpt);

	 UDrawXLine((*lastspline)->ctrl[0], (*lastspline)->ctrl[1]);
	 UDrawXLine((*lastspline)->ctrl[3], (*lastspline)->ctrl[2]);
         XcSetXORFg((*lastspline)->color, BACKGROUND);
         XcSetFunction(GXxor);
	 XtAddEventHandler(areastruct.area, PointerMotionMask, False,
               (XtEventHandler)trackspline, NULL);
	 eventmode = (mode == PATH) ? EPATH_MODE : ESPLINE_MODE;
      } break;
      case ARC: {
	 arcptr *lastarc = (arcptr *)editpart;
	 XPoint savept, curpt;
	 float tmpratio, tlen;

	 objectdeselect();

	 /* find a part of the arc close to the pointer */

	 tlen = (float)wirelength(&areastruct.save, &((*lastarc)->position));
	 tmpratio = (float)(abs((*lastarc)->radius)) / tlen;
	 curpt.x = (*lastarc)->position.x + tmpratio * (areastruct.save.x
	     - (*lastarc)->position.x);
	 tmpratio = (float)(*lastarc)->yaxis / tlen;
	 curpt.y = (*lastarc)->position.y + tmpratio * (areastruct.save.y
	     - (*lastarc)->position.y);
	 areastruct.editcycle = 0;
	 saveratio = (double)((*lastarc)->yaxis) / (double)(abs((*lastarc)->radius));
	 
	 /* Push onto the editstack */
	 arceditpush(*lastarc);

	 checkwarp(&curpt);

	 UDrawXLine(curpt, (*lastarc)->position);
	 areastruct.save.x = curpt.x;	/* for redrawing dotted edit line */
	 areastruct.save.y = curpt.y;
	 XcSetXORFg((*lastarc)->color, BACKGROUND);
	 XcSetFunction(GXxor);
	 XtAddEventHandler(areastruct.area, PointerMotionMask, False,
	    (XtEventHandler)trackarc, NULL);
	 eventmode = (mode == PATH) ? EPATH_MODE : EARC_MODE;
      } break;
   }
}

/*-------------------------------------------------*/
/* Raise/Lower an object			   */
/*-------------------------------------------------*/

void exchange()
{
   short *selectobj = areastruct.selectlist;
   genericptr *exchobj, *exchobj2, temp;

   if (areastruct.selects > 2 || areastruct.selects == 0) {
      Wprintf("Select 1 or 2 objects");
      return;
   }

   if (areastruct.selects == 1) {  /* lower if on top; raise otherwise */
      exchobj = objectdata->plist + *selectobj;
      temp = *exchobj;
      if (*selectobj == objectdata->parts - 1) {
	 for (exchobj2 = objectdata->plist + objectdata->parts - 2;
		exchobj2 >= objectdata->plist; exchobj2--)
	    *(exchobj2 + 1) = *exchobj2;
	 *(objectdata->plist) = temp;
	 *selectobj = 0;
      }
      else {
	 for (exchobj2 = objectdata->plist + *selectobj; exchobj2 <
		objectdata->plist + objectdata->parts - 1; exchobj2++)
	    *exchobj2 = *(exchobj2 + 1);
	 *(objectdata->plist + objectdata->parts - 1) = temp;
	 *selectobj = objectdata->parts - 1;
      }
   }
      
   else {  /* exchange the two objects */
      exchobj = objectdata->plist + *selectobj;
      exchobj2 = objectdata->plist + *(selectobj + 1);

      temp = *exchobj;
      *exchobj = *exchobj2;
      *exchobj2 = temp;
   }

   objectdeselect();
   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Flip an element horizontally (POLYGON, ARC, or SPLINE) */
/*--------------------------------------------------------*/

void elhflip(genobj)
  genericptr *genobj;
{
   switch((*genobj)->type) {
      case POLYGON:{
	 polyptr flippoly = TOPOLY(genobj);
	 pointlist ppoint;
         u2u_snap(&areastruct.save);
	 for (ppoint = flippoly->points; ppoint < flippoly->points +
	       flippoly->number; ppoint++)
	    ppoint->x = (areastruct.save.x << 1) - ppoint->x;	
	 }break;

      case ARC:{
	 arcptr fliparc = TOARC(genobj);
	 float tmpang = 180 - fliparc->angle1;
	 fliparc->angle1 = 180 - fliparc->angle2;
	 fliparc->angle2 = tmpang;
	 if (fliparc->angle2 < 0) {
	    fliparc->angle1 += 360;
	    fliparc->angle2 += 360;
	 }
	 fliparc->radius = -fliparc->radius;
	 fliparc->position.x = (areastruct.save.x << 1) - fliparc->position.x;
	 calcarc(fliparc);
	 }break;

      case SPLINE:{
	 splineptr flipspline = TOSPLINE(genobj);
	 int i;
	 u2u_snap(&areastruct.save);
	 for (i = 0; i < 4; i++)
	    flipspline->ctrl[i].x = (areastruct.save.x << 1) -
		  flipspline->ctrl[i].x;
	 calcspline(flipspline);
	 }break;
   }
}

/*--------------------------------------------------------*/
/* Flip an element vertically (POLYGON, ARC, or SPLINE)   */
/*--------------------------------------------------------*/

void elvflip(genobj)
  genericptr *genobj;
{
   switch((*genobj)->type) {

      case POLYGON:{
	 polyptr flippoly = TOPOLY(genobj);
	 pointlist ppoint;

         u2u_snap(&areastruct.save);
	 for (ppoint = flippoly->points; ppoint < flippoly->points +
	    	   flippoly->number; ppoint++)
	    ppoint->y = (areastruct.save.y << 1) - ppoint->y;	
	 }break;

      case ARC:{
	 arcptr fliparc = TOARC(genobj);
	 float tmpang = 360 - fliparc->angle1;
	 fliparc->angle1 = 360 - fliparc->angle2;
	 fliparc->angle2 = tmpang;
	 if (fliparc->angle1 >= 360) {
	    fliparc->angle1 -= 360;
	    fliparc->angle2 -= 360;
	 }
	 fliparc->radius = -fliparc->radius;
	 fliparc->position.y = (areastruct.save.y << 1) - fliparc->position.y;
	 calcarc(fliparc);
	 }break;

      case SPLINE:{
	 splineptr flipspline = TOSPLINE(genobj);
	 int i;
	 u2u_snap(&areastruct.save);
	 for (i = 0; i < 4; i++)
	    flipspline->ctrl[i].y = (areastruct.save.y << 1) -
		  flipspline->ctrl[i].y;
	 calcspline(flipspline);
	 }break;
   }
}

/*--------------------------------------------------------*/
/* Horizontally flip an object (any except LABEL)	  */
/*--------------------------------------------------------*/

void objectflip()
{
   short *selectobj;

   if (!checkselect(OBJECT|POLYGON|ARC|SPLINE|PATH)) return;
   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
	+ areastruct.selects; selectobj++) {

      /* erase the object */
      XSetFunction(dpy, areastruct.gc, GXcopy);
      XSetForeground(dpy, areastruct.gc, BACKGROUND);
      easydraw(*selectobj, DOFORALL);

      switch(SELECTTYPE(selectobj)) {

	 case OBJECT:{
            objinstptr flipobj = SELTOOBJINST(selectobj);
   	    flipobj->rotation = -flipobj->rotation;
	    }break;
	 case POLYGON: case ARC: case SPLINE:
	    elhflip(objectdata->plist + *selectobj);
	    break;
	 case PATH:{
	    genericptr *genpart;
	    pathptr flippath = SELTOPATH(selectobj);

	    for (genpart = flippath->plist; genpart < flippath->plist
		  + flippath->parts; genpart++)
	       elhflip(genpart);
	    }break;
      }

      if (eventmode != NORMAL_MODE) {
         XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
	 easydraw(*selectobj, DOFORALL);
      }
   }
   if (eventmode == NORMAL_MODE) objectdeselect();
}

/*------------------------------------------------*/
/* Vertically flip an object (any except LABEL)	  */
/*------------------------------------------------*/

void objectvflip()   /* flip and rotate by 180 degrees */
{
   short *selectobj;
   short fsign;

   if (!checkselect(OBJECT | POLYGON | ARC | SPLINE | PATH)) return;
   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
	+ areastruct.selects; selectobj++) {

      /* erase the object */
      XSetFunction(dpy, areastruct.gc, GXcopy);
      XSetForeground(dpy, areastruct.gc, BACKGROUND);
      easydraw(*selectobj, DOFORALL);

      switch(SELECTTYPE(selectobj)) {
	 case(OBJECT):{
            objinstptr flipobj = SELTOOBJINST(selectobj);

	    fsign = -sign(flipobj->rotation);
	    flipobj->rotation = abs(flipobj->rotation) + (RSTEPS >> 1);
	    if (flipobj->rotation >= RSTEPS) flipobj->rotation -= RSTEPS;
   	    flipobj->rotation = fsign * flipobj->rotation;
	    }break;
	 case POLYGON: case ARC: case SPLINE:
	    elvflip(objectdata->plist + *selectobj);
	    break;
	 case PATH:{
	    genericptr *genpart;
	    pathptr flippath = SELTOPATH(selectobj);

	    for (genpart = flippath->plist; genpart < flippath->plist
		  + flippath->parts; genpart++)
	       elvflip(genpart);
	    }break;
      }
      if (eventmode != NORMAL_MODE) {
         XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
	 easydraw(*selectobj, DOFORALL);
      }
   }
   if (eventmode == NORMAL_MODE) objectdeselect();
}

/*---------------------------------------*/
/* ButtonPress handler during center pan */
/*---------------------------------------*/

void panbutton(w, ptype, event)
  Widget w;
  void *ptype;
  XButtonEvent *event;
{
   Window pwin, nullwin;
   int  nullint, xpos, ypos, newllx, newlly, newurx, newury;
   XPoint savell;
   unsigned int   nullui;
   Dimension hwidth = areastruct.width >> 1, hheight = areastruct.height >> 1;

   savell.x = areastruct.lowerleft->x;
   savell.y = areastruct.lowerleft->y;

   if ((int) ptype == 1) {
      xpos = 0;
      ypos = hheight;
   }
   else if ((int) ptype == 2) {
      xpos = areastruct.width;
      ypos = hheight;
   }
   else if ((int) ptype == 3) {
      xpos = hwidth;
      ypos = 0;
   }
   else if ((int) ptype == 4) {
      xpos = hwidth;
      ypos = areastruct.height;
   }
   else if ((int) ptype == 5) {
      xpos = event->x;
      ypos = event->y;
   }
   else {
      XQueryPointer(dpy, win, &nullwin, &pwin, &nullint, &nullint, &xpos,
	  &ypos, &nullui);
      XWarpPointer(dpy, None, win, 0, 0, 0, 0, hwidth, hheight);
   }

   xpos -= hwidth;
   ypos = hheight - ypos;

   newllx = (int)areastruct.lowerleft->x + (int)((float)xpos / (*areastruct.vscale));
   newlly = (int)areastruct.lowerleft->y + (int)((float)ypos / (*areastruct.vscale));

   areastruct.lowerleft->x = (short) newllx;
   areastruct.lowerleft->y = (short) newlly;

   if ((newllx << 1) != (long)(areastruct.lowerleft->x << 1) || (newlly << 1)
	   != (long)(areastruct.lowerleft->y << 1) || checkbounds() == -1) {
      areastruct.lowerleft->x = savell.x;
      areastruct.lowerleft->x = savell.y;
      Wprintf("Reached bounds:  cannot pan further.");
      return;
   }
   else if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      drag(areastruct.area, NULL, event);

   Wprintf(" ");
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);

   drawarea(w, objectdata, NULL);
   drawhbar(areastruct.scrollbarh, NULL, Number(0));
   drawvbar(areastruct.scrollbarv, NULL, Number(1));
}

/*--------------------------*/
/* Register a "press" event */
/*--------------------------*/

XtTimerCallbackProc makepress(id, clientdata)
  XtIntervalId	id;
  caddr_t	clientdata;
{
   /* Button was pressed long enough to make a "press", not a "tap" */

   if (eventmode != NORMAL_MODE) return;
   eventmode = PRESS_MODE;
   if (areastruct.selects == 0) objectselect(ANY);
   u2u_snap(&areastruct.save);
   XDefineCursor(dpy, win, ARROW);
   XtAddEventHandler(areastruct.area, Button1MotionMask, False, 
	(XtEventHandler)drag, NULL);
}

/*---------------------------*/
/* Begin the rotate function */
/*---------------------------*/
   
void startrotate(w, number, clientdata)
  Widget w;
  caddr_t number;
  caddr_t clientdata; 
{
   if (eventmode == NORMAL_MODE) {
      eventmode = ROTATE_MODE;
      saverot = (int)number;
      XDefineCursor (dpy, win, ROTATECURSOR);
      if ((int)number == 64 || (int)number == 65)
         Wprintf("Click on element to flip.");
      else
         Wprintf("Click on element to rotate.");
   }
}     

/*-------------------------*/
/* Begin the copy function */
/*-------------------------*/

void startcopy(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   if (eventmode == NORMAL_MODE) {
      eventmode = COPY_MODE;
      XDefineCursor (dpy, win, COPYCURSOR);
      Wprintf("Click on element and drag.");
   }
}

/*-------------------------*/
/* Begin the edit function */
/*-------------------------*/

void startedit(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   if (eventmode == NORMAL_MODE) {
      eventmode = EDIT_MODE;
      XDefineCursor (dpy, win, EDCURSOR);
      Wprintf("Click on element to edit.");
   }
}

/*---------------------------*/
/* Begin the delete function */
/*---------------------------*/

void startdelete(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   if (eventmode == NORMAL_MODE) {
      eventmode = DELETE_MODE;
      XDefineCursor (dpy, win, SCISSORS);
   }
}

/*-----------------------------------------------------------*/
/* Select list numbering revision for routine objectdelete() */
/*-----------------------------------------------------------*/

void reviseselect(selectobj)
  short *selectobj;
{
   short *chkselect;

   for (chkselect = selectobj + 1; chkselect < areastruct.selectlist
	 + areastruct.selects; chkselect++)
      if (*chkselect > *selectobj) (*chkselect)--;
}

/*----------------------------------------*/
/* Process of object instance deletion	  */
/*----------------------------------------*/

void objectdelete(drawmode)
  short drawmode;
{
   short *selectobj;
   objectptr *delbuf, delobj;
   genericptr *genobj;

   if (!checkselect(ANY)) return;

   /* flush the delete buffer and realloc */

   if (areastruct.selects > 0) {
      if (areastruct.deletes == DELBUFSIZE) {
         reset(*(areastruct.delbuffer), DESTROY);
	 for (delbuf = areastruct.delbuffer + 1; delbuf < areastruct.
	      delbuffer + areastruct.deletes; delbuf++)
	    *(delbuf - 1) = *delbuf;
      }
      else areastruct.deletes++;
      delbuf = areastruct.delbuffer + areastruct.deletes - 1;
      *delbuf = (objectptr) malloc(sizeof(object));
      delobj = *delbuf; 
      initmem(delobj);
   }

   if (drawmode) {
      XSetFunction(dpy, areastruct.gc, GXcopy);
      XSetForeground(dpy, areastruct.gc, BACKGROUND);
   }

   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
	 + areastruct.selects; selectobj++) {
      genobj = objectdata->plist + *selectobj;
      if (drawmode) easydraw(*selectobj, DOFORALL);
      PLIST_INCR(delobj);	
      *(delobj->plist + delobj->parts) = *genobj;
      delobj->parts++;

#ifdef SCHEMA
      /* need to delete label in corresponding schematic/symbol */
      if (areastruct.schemon && (*genobj)->type == LABEL) {
	 if (TOLABEL(genobj)->pin) deletepinlabel(TOLABEL(genobj));
      }
#endif

      for (++genobj; genobj < objectdata->plist + objectdata->parts; genobj++)
	 *(genobj - 1) = *genobj;
      objectdata->parts--;
      reviseselect(selectobj);
   }

   if (drawmode) XSetForeground(dpy, areastruct.gc, FOREGROUND);
   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;
}
  
/*----------------------------------------*/
/* ButtonPress handler during delete mode */
/*----------------------------------------*/

void deletebutton(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XButtonEvent *event;
{
   if (eventmode == DELETE_MODE)
      window_to_user(event->x, event->y, &areastruct.save);

   if (event->button == Button1) objectdelete(ERASE);
   else if (event->button == Button2)
      objectselect(ANY);
   if (event->button == Button3) {
      objectdeselect();
      eventmode = NORMAL_MODE;
      XDefineCursor (dpy, win, CROSS); 
   }
}

/*-------------------------------*/
/* Undelete last deleted objects */
/*-------------------------------*/

short undelete(w, mode, event)
  Widget w;
  void *mode;
  XButtonEvent *event;
{
   objectptr delobj = *(areastruct.delbuffer + areastruct.deletes - 1);
   genericptr *regen;
   short      count = 0;

   if (areastruct.deletes == 0) return 0;

   if (Number(mode)) {
      XSetFunction(dpy, areastruct.gc, GXcopy);
   }

   for (regen = delobj->plist; regen < delobj->plist + delobj->parts; regen++) {
      PLIST_INCR(objectdata);
      objectdata->parts++; count++;
      *(ENDPART - 1) = *regen;
      if (Number(mode)) {
         XTopSetForeground((*regen)->color);
	 easydraw(objectdata->parts - 1, DEFAULTCOLOR);
      }
   }

   /* flush the delete buffer and realloc but don't delete the elements */
   reset(delobj, SAVE);
   free(delobj);
   areastruct.deletes--;
   return count;
}

/*----------------------------*/
/* select save object handler */
/*----------------------------*/

void printname(curobject)
  objectptr 	curobject;
{
   Dimension swidth, swidth2, sarea;
   char tmpname[256];
   char editstr[10], pagestr[10];
   char *sptr = tmpname;
   Arg	wargs[1];
   
   /* print full string to make message widget proper size */

   strcpy(editstr, (curobject == areastruct.libtop[LIBRARY] ||
	curobject == areastruct.libtop[USERLIB]) ? "" : "Editing: ");
   if (strstr(curobject->name, "Page") == NULL && curobject !=
	  areastruct.libtop[LIBRARY] && curobject != areastruct.libtop[USERLIB])
      sprintf(pagestr, " (p. %d)", areastruct.page + 1);
   else
      pagestr[0] = '\0';
   sprintf(_STR, "%s%s%s", editstr, curobject->name, pagestr); 
   W2printf(_STR);

   XtSetArg(wargs[0], XtNwidth, &sarea);
   XtGetValues(message2, wargs, 1); 
   
   /* in the remote case that the string is longer than message widget,    */
   /* truncate the string and denote the truncation with an ellipsis (...) */

   strcpy(tmpname, curobject->name);
   swidth2 = XTextWidth(appdata.xcfont, editstr, strlen(editstr));
   swidth = XTextWidth(appdata.xcfont, tmpname, strlen(tmpname));

   if ((swidth + swidth2) > sarea) {
      char *ip;
      while ((swidth + swidth2) > sarea) {
         sptr++;
         swidth = XTextWidth(appdata.xcfont, sptr, strlen(sptr));
      }
      for(ip = sptr; ip < sptr + 3 && *ip != '\0'; ip++) *ip = '.';

      sprintf(_STR, "Editing: %s", sptr); 
      W2printf(_STR);
   }
}

/* List of strings which appear in the profile and cannot be used
   as object names---these are the "most likely" offenders */

#define NUMSTRS 39
static char *psstrings[] = {"add", "and", "arc", "arcn", "array", "bitshift", "bop",
	"copy", "curveto", "dict", "elb", "ele", "ellipse", "eq", "exch", "exec",
	"fill", "if", "index", "label", "mul", "neg", "nellip", "not", "or",
	"pellip", "polyc", "polygon", "pop", "repeat", "roll", "save", "scale",
	"setgray", "spline", "sub", "transform", "scb", "sce", "xor"};

/*-----------------------------------------------------------*/
/* Make sure that name for new object does not conflict with */
/* existing object definitions or PostScript commands/defs   */
/* found in xcircps.pro					     */
/*-----------------------------------------------------------*/

void checkname(newobj)
  objectptr	newobj;
{
   int i;
   short dupl;  /* flag a duplicate string */
   objectptr *libobj;
   char *sptr;
   short digcount = 0;

   /* Check for illegal characters which have syntactical meaning in */
   /* PostScript, and the presence of nonprintable characters or     */
   /* whitespace.		     				     */

   for (i = 0; i < strlen(_STR); i++) {
      if (_STR[i] == '/' || _STR[i] == '}' || _STR[i] == '{' ||
	     _STR[i] == ']' || _STR[i] == '[' || _STR[i] == ')' ||
	     _STR[i] == '(') {
	 _STR[i] = '_';
	 Wprintf("Replaced illegal character in name with underscore");
      }
      else if (isdigit(_STR[i])) digcount++;
      else if ((!isprint(_STR[i])) || isspace(_STR[i])) {
	 _STR[i] = '_';
	 Wprintf("Replaced illegal whitespace in name with underscore");
      }
   }

   /* Name cannot be an integer number */

   if (digcount == strlen(_STR)) {
      _STR[i] = '_';
      _STR[i + 1] = '\0';
      Wprintf("Name cannot be an integer number:  appended an underscore");
   }

   do {
      dupl = 0;
      for (i = 0; i < NUMSTRS; i++)
         if (!strcmp(_STR, psstrings[i])) {
	    sprintf(_STR, "_%s", psstrings[i]);
	    dupl = 1;
	    Wprintf("Altered name to avoid conflict");
	    break;
         }
      for (libobj = areastruct.library; libobj != areastruct.userlib +
               areastruct.userobjs; libobj++) {
         if (libobj == areastruct.library + areastruct.builtins) {
            libobj = areastruct.userlib;
            if (areastruct.userobjs == 0) break;
         }
	 if (*libobj == newobj) continue;
         if (!strcmp(_STR, (*libobj)->name)) { 
	     sprintf(_STR, "_%s", (*libobj)->name);
	     dupl = 1;
	     Wprintf("Altered name to avoid conflict");
	 }
      }

   } while (dupl == 1);
   sscanf(_STR, "%s", newobj->name); 
}

/*--------------------------------------*/
/* Add value origin to all points	*/
/*--------------------------------------*/

movepoints(ssgen, deltax, deltay)
  genericptr *ssgen;
  short deltax, deltay;
{
   switch((*ssgen)->type) {
	 case(ARC):{
            fpointlist sspoints;
            TOARC(ssgen)->position.x += deltax;
            TOARC(ssgen)->position.y += deltay;
            for (sspoints = TOARC(ssgen)->points; sspoints < TOARC(ssgen)->points +
	          TOARC(ssgen)->number; sspoints++) {
	       sspoints->x += deltax;
	       sspoints->y += deltay;
            }
	    }break;

	 case(POLYGON):{
            pointlist sspoints;
            for (sspoints = TOPOLY(ssgen)->points; sspoints < TOPOLY(ssgen)->points +
	          TOPOLY(ssgen)->number; sspoints++) {
	       sspoints->x += deltax;
	       sspoints->y += deltay;
	    }
	    }break;

	 case(SPLINE):{
            fpointlist sspoints;
            short j;
            for (sspoints = TOSPLINE(ssgen)->points; sspoints <
		  TOSPLINE(ssgen)->points + INTSEGS; sspoints++) {
	       sspoints->x += deltax;
	       sspoints->y += deltay;
            }
            for (j = 0; j < 4; j++) {
               TOSPLINE(ssgen)->ctrl[j].x += deltax;
               TOSPLINE(ssgen)->ctrl[j].y += deltay;
            }
            }break;
   }
}

/*----------------------------------------------------------------*/
/* Set the name for a new user-defined object and make the object */
/*----------------------------------------------------------------*/

void domakeobject(w, nulldata)
  Widget w;
  caddr_t	nulldata;
{
   objectptr *libobj, *newobj;
   objinstptr *newinst;
   genericptr *ssgen;
   buttonsave *popdata = (buttonsave *)malloc(sizeof(buttonsave));
   XPoint origin;

   /* make room for new entry in library list */

   areastruct.userlib = (objectptr *) realloc(areastruct.userlib,
	 (areastruct.userobjs + 1) * sizeof(objectptr));

   newobj = areastruct.userlib + areastruct.userobjs;

   /* shove selections onto the delete stack, then grab the stack object */

   objectdelete(NORMAL);
   *newobj = *(areastruct.delbuffer + areastruct.deletes - 1);
   areastruct.deletes--;

   /* create a temporary instance for this object for boundingbox calc */
   calcbbox(*newobj); 
   areastruct.userobjs++;

   /* find closest snap point to bbox center and make this the obj. center */

   if (areastruct.center) {
      origin.x = (*newobj)->lowerleft.x + (*newobj)->width / 2;
      origin.y = (*newobj)->lowerleft.y + (*newobj)->height / 2;
   }
   else {
      origin.x = origin.y = 0;
   }
   u2u_snap(&origin);

   for (ssgen = (*newobj)->plist; ssgen < (*newobj)->plist + (*newobj)->parts;
	  ssgen++) {
      switch((*ssgen)->type) {

	 case(OBJECT):
            TOOBJINST(ssgen)->position.x -= origin.x;
            TOOBJINST(ssgen)->position.y -= origin.y;
	    break;

	 case(LABEL):
            TOLABEL(ssgen)->position.x -= origin.x;
            TOLABEL(ssgen)->position.y -= origin.y;
	    break;

	 case(PATH):{
	    genericptr *pathlist;
	    for (pathlist = TOPATH(ssgen)->plist;  pathlist < TOPATH(ssgen)->plist
		  + TOPATH(ssgen)->parts; pathlist++) {
	       movepoints(pathlist, -origin.x, -origin.y);
	    }
	    }break;

	 default:
	    movepoints(ssgen, -origin.x, -origin.y);
	    break;
      }
   }
   calcbbox(*newobj);

   /* put new object back into place */

   (*newobj)->hidden = False;
#ifdef SCHEMA
   (*newobj)->schemtype = SYMBOL;
#endif

   NEW_OBJINST(newinst, objectdata);
   objectdata->parts++;
   (*newinst)->position.x = origin.x;
   (*newinst)->position.y = origin.y;
   (*newinst)->color = areastruct.color;  /* should choose color of an element? */
   (*newinst)->rotation = 1;
   (*newinst)->scale = 1.0;
   (*newinst)->thisobject = *newobj;

   XSetFunction(dpy, areastruct.gc, GXcopy);
   XTopSetForeground((*newinst)->color);
   UDrawObject(*newinst, (*newinst)->thisobject, SINGLE, (*newinst)->color);

   /* copy name from popup prompt buffer and check */

   strcpy(_STR, _STR2);
   checkname(*newobj);

   /* recompile the user catalog */

   composelib(USERLIB);
}

/*-------------------------------------------*/
/* Make a user object from selected elements */
/*-------------------------------------------*/

void selectsave(w, clientdata, nulldata)
  Widget w;
  caddr_t	clientdata, nulldata;
{
   buttonsave *popdata = (buttonsave *)malloc(sizeof(buttonsave));

   if (areastruct.selects == 0) return;  /* nothing was selected */

   /* Get a name for the new object */

   eventmode = NORMAL_MODE;
   popdata->dataptr = NULL;
   popdata->button = NULL; /* indicates that no button is assc'd w/ the popup */
   popupprompt(w, "Enter name for new object:", "\0", domakeobject, popdata, False);
}

/*-----------------------------*/
/* Edit-stack support routines */
/*-----------------------------*/

arceditpush(lastarc)
  arcptr lastarc;
{
   arcptr *newarc;

   NEW_ARC(newarc, areastruct.editstack);
   areastruct.editstack->parts++;
   arccopy(*newarc, lastarc);
}

/*--------------------------------------*/

splineeditpush(lastspline)
  splineptr lastspline;
{
   splineptr *newspline;

   NEW_SPLINE(newspline, areastruct.editstack);
   areastruct.editstack->parts++;
   splinecopy(*newspline, lastspline);
}

/*--------------------------------------*/

polyeditpush(lastpoly)
   polyptr lastpoly;
{
   polyptr *newpoly;

   NEW_POLY(newpoly, areastruct.editstack);
   areastruct.editstack->parts++;
   polycopy(*newpoly, lastpoly);
}

/*-----------------------------*/
/* Copying support routines    */
/*-----------------------------*/

arccopy(newarc, copyarc)
  arcptr newarc, copyarc;
{
   newarc->style = copyarc->style;
   newarc->color = copyarc->color;
   newarc->position.x = copyarc->position.x;
   newarc->position.y = copyarc->position.y;
   newarc->radius = copyarc->radius;
   newarc->yaxis = copyarc->yaxis;
   newarc->angle1 = copyarc->angle1;
   newarc->angle2 = copyarc->angle2;
   newarc->width = copyarc->width;
   calcarc(newarc);
}

/*------------------------------------------*/

polycopy(newpoly, copypoly)
  polyptr newpoly, copypoly;
{
   pointlist copypoints, newpoints;

   newpoly->style = copypoly->style;
   newpoly->color = copypoly->color;
   newpoly->width = copypoly->width;
   newpoly->number = copypoly->number;
   newpoly->points = (pointlist) malloc(newpoly->number *
	  sizeof(XPoint));
   copypoints = copypoly->points;
   for (newpoints = newpoly->points; newpoints < newpoly->points +
		newpoly->number; newpoints++, copypoints++) {
      newpoints->x = copypoints->x;
      newpoints->y = copypoints->y;
   }
}

/*------------------------------------------*/

splinecopy(newspline, copyspline)
  splineptr	newspline, copyspline;
{
   short i;

   newspline->style = copyspline->style;
   newspline->color = copyspline->color;
   newspline->width = copyspline->width;
   for (i = 0; i < 4; i++) {
     newspline->ctrl[i].x = copyspline->ctrl[i].x;
     newspline->ctrl[i].y = copyspline->ctrl[i].y;
   }
   for (i = 0; i < INTSEGS; i++) {
      newspline->points[i].x = copyspline->points[i].x;
      newspline->points[i].y = copyspline->points[i].y;
   }
}

/*---------------------*/
/* copy handler	       */
/*---------------------*/

void copybutton(w, copydata, event)
  Widget w;
  objectptr	copydata;
  XButtonEvent *event;
{
   short *selectobj;

   window_to_user(event->x, event->y, &areastruct.save);

   if (event->button == Button1) {
      if (!checkselect(ANY)) return;
      u2u_snap(&areastruct.save);
      XSetFunction(dpy, areastruct.gc, GXxor);
      for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist
         + areastruct.selects; selectobj++) {
         XSetXORFg(SELTOCOLOR(selectobj), BACKGROUND);
         switch(SELECTTYPE(selectobj)) {
            case LABEL: { /* copy label */
	       labelptr copytext = SELTOLABEL(selectobj);
	       labelptr *newtext;
	
	       NEW_LABEL(newtext, copydata);
	       (*newtext)->string = (uchar *) malloc((strlen(copytext->string) 
		  + 1) * sizeof(uchar));
	       strcpy((*newtext)->string, copytext->string);
	       (*newtext)->position.x = copytext->position.x;
	       (*newtext)->position.y = copytext->position.y;
	       (*newtext)->rotation = copytext->rotation;
	       (*newtext)->scale = copytext->scale;
	       (*newtext)->justify = copytext->justify;
	       (*newtext)->color = copytext->color;
#ifdef SCHEMA
	       (*newtext)->pin = copytext->pin;

	       /* copy label both here and on corresponding schematic/symbol */
	       copypinlabel(*newtext);
#endif
            } break;
            case OBJECT: { /* copy object instance */
	       objinstptr copyobj = SELTOOBJINST(selectobj);
	       objinstptr *newobj;
	
	       NEW_OBJINST(newobj, copydata);
	       (*newobj)->position.x = copyobj->position.x;
	       (*newobj)->position.y = copyobj->position.y;
	       (*newobj)->rotation = copyobj->rotation;
	       (*newobj)->scale = copyobj->scale;
	       (*newobj)->thisobject = copyobj->thisobject;
	       (*newobj)->color = copyobj->color;
            } break;
	    case PATH: { /* copy path */
	       pathptr copypath = SELTOPATH(selectobj);
	       pathptr *newpath;
	       genericptr *genpart;
	       short cpart;
	       NEW_PATH(newpath, copydata);
	       (*newpath)->style = copypath->style;
	       (*newpath)->color = copypath->color;
	       (*newpath)->width = copypath->width;
	       (*newpath)->parts = 0;
	       (*newpath)->plist = (genericptr *)malloc(copypath->parts *
		     sizeof(genericptr));
	       for (genpart = copypath->plist; genpart < copypath->plist
		  + copypath->parts; genpart++) {
	          switch((*genpart)->type){
		     case ARC: {
	       		arcptr copyarc = TOARC(genpart);
	       		arcptr *newarc;
   	       		NEW_ARC(newarc, (*newpath));
	       		arccopy(*newarc, copyarc);
	       		(*newpath)->parts++;
			} break;
		     case POLYGON: {
               		polyptr copypoly = TOPOLY(genpart);
               		polyptr *newpoly;
   	       		NEW_POLY(newpoly, (*newpath));
	       		polycopy(*newpoly, copypoly);
	       		(*newpath)->parts++;
			} break;
		     case SPLINE: {
	       		splineptr copyspline = TOSPLINE(genpart);
	       		splineptr *newspline;
   	       		NEW_SPLINE(newspline, (*newpath));
	       		splinecopy(*newspline, copyspline);
	       		(*newpath)->parts++;
			} break;
		  }
	       }
	    } break;	
	    case ARC: { /* copy arc */
	       arcptr copyarc = SELTOARC(selectobj);
	       arcptr *newarc;
   	       NEW_ARC(newarc, copydata);
	       arccopy(*newarc, copyarc);
            } break;
            case POLYGON: { /* copy polygons */
               polyptr copypoly = SELTOPOLY(selectobj);
               polyptr *newpoly;
   	       NEW_POLY(newpoly, copydata);
	       polycopy(*newpoly, copypoly);
            } break;
	    case SPLINE: { /* copy spline */
	       splineptr copyspline = SELTOSPLINE(selectobj);
	       splineptr *newspline;
   	       NEW_SPLINE(newspline, copydata);
	       splinecopy(*newspline, copyspline);
	    } break;
         }

	 /* change selection from the old to the new object */

	 *selectobj = copydata->parts;

	 easydraw(copydata->parts++, DOFORALL);
      }
      if (areastruct.selects > 0) {
	 if (eventmode == NORMAL_MODE || eventmode == COPY_MODE) {
	    XDefineCursor(dpy, win, COPYCURSOR);
	    eventmode = COPY2_MODE;
            XtAddEventHandler(areastruct.area, PointerMotionMask, False, 
		(XtEventHandler)drag, NULL);
	 }
      }
   }
   else if (event->button == Button2)
      objectselect(ANY);
   if (event->button == Button3) {
      objectdeselect();
      eventmode = NORMAL_MODE;
      XDefineCursor (dpy, win, CROSS);
   }
}

/*---------------------*/
/* ButtonPress handler */
/*---------------------*/

void selectbutton(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XButtonEvent *event;
{
   XPoint ppos;

   /* ignore multiple buttons (using local button information) */

   if ((unsigned int)curbutton != 0) return;
   curbutton = (unsigned char)event->button;

   if (eventmode != EARC_MODE && eventmode != ARC_MODE)
      window_to_user(event->x, event->y, &areastruct.save);
   snap(event->x, event->y, &ppos);
   printpos(ppos.x, ppos.y);

/* printf("Press:  mode is %d\n", eventmode); */
   switch(eventmode) {
      case(NORMAL_MODE):
         if (event->button == Button1) /* setup timeout event */
            areastruct.time_id = XtAddTimeOut(PRESSTIME, 
		(XtTimerCallbackProc)makepress, objectdata);
         else if (event->button == Button2)
	    areastruct.time_id = XtAddTimeOut(PRESSTIME,
		(XtTimerCallbackProc)startselect, objectdata);
         else if (event->button == Button3)
            objectdeselect(); 
         break;
      case(PAN_MODE): case(CATPAN_MODE):
	 panbutton(w, objectdata, event);
         break; 
      case(BOX0_MODE):
	 if (event->button != Button3) boxbutton(w, objectdata, event);
	 else {
	    eventmode = NORMAL_MODE;
	    Wprintf("Cancelled box.");
	 }
	 break;
      case(SPLINE0_MODE):
         if (event->button != Button3) splinebutton(w, objectdata, event);
         else {
            eventmode = NORMAL_MODE;
            Wprintf("Cancelled spline.");
         }
         break;
      case(ARC0_MODE):
         if (event->button != Button3) arcbutton(w, objectdata, event);
         else {
            eventmode = NORMAL_MODE;
            Wprintf("Cancelled arc.");
         }
         break;
      case(TEXT0_MODE): case(TEXT1_MODE):
	 if (event->button != Button3)
	    textbutton(w, (eventmode == TEXT0_MODE) ? False : True, event);
	 else {
	    eventmode = NORMAL_MODE;
	    Wprintf("Cancelled text.");
	    XDefineCursor(dpy, win, CROSS);
	 }
	 break;
      case(SELAREA2_MODE):
	 areastruct.origin.x = areastruct.save.x;
	 areastruct.origin.y = areastruct.save.y;
         UDrawBox(areastruct.origin, areastruct.save);
         XtAddEventHandler(areastruct.area, ButtonMotionMask, False,
            (XtEventHandler)trackselarea, NULL);
	 break;
      case(DELETE_MODE):
	 deletebutton(w, objectdata, event);
	 break;
      case(PUSH_MODE):
	 pushobject(w, objectdata, event);
	 break;
      case(DESEL_MODE):
         deselect(-ANY);
	 break;
      case(COPY_MODE):
	 copybutton(w, objectdata, event);
	 break;
      case(ROTATE_MODE):
	 rotatebutton(w, objectdata, event);
	 break;
      case(EDIT_MODE):
	 edit(w, objectdata, event);
	 break;
      case(COPY2_MODE):
	 u2u_snap(&areastruct.save);
	 break;
      case(CATALOG_MODE):
	 catbutton(w, objectdata, event);
	 break;
      case(FONTCAT_MODE): case(FONTCAT2_MODE):
	 fontcatbutton(w, objectdata, event);
	 break;
      case(WIRE_MODE):
	 wirebutton(w, objectdata, event);
	 break;
   }
}

/*-----------------------*/
/* ButtonRelease handler */
/*-----------------------*/

void releasebutton(w, clientdata, event)
  Widget w;
  caddr_t  clientdata;
  XButtonEvent *event;
{
   /* ignore multiple buttons (using local button information) */

   if ((unsigned int)curbutton != event->button) return;
   curbutton = (unsigned char)0;

   /* remove timeout, if pending */

   if (areastruct.time_id != 0) {
      XtRemoveTimeOut(areastruct.time_id);
      areastruct.time_id = 0;
   }

/* printf("Release:  mode is %d\n", eventmode); */
   switch(eventmode) {
      case(EDIT_MODE): case(PUSH_MODE): case(DESEL_MODE):
      case(PAN_MODE): case(ROTATE_MODE):
	 eventmode = NORMAL_MODE;
	 break;
      case (FONTCAT_MODE):
	 eventmode = TEXT2_MODE;
	 XDefineCursor (dpy, win, TEXTPTR);
         drawarea(NULL, objectdata, NULL);
	 break;
      case (FONTCAT2_MODE):
	 eventmode = TEXT3_MODE;
	 XDefineCursor (dpy, win, TEXTPTR);
         drawarea(NULL, objectdata, NULL);
	 break;
      case (CATPAN_MODE):
	 eventmode = CATALOG_MODE;
         XDefineCursor(dpy, win, CROSS);
	 break;
      case(WIRE_MODE):
	 if(event->button == Button2) {
	    eventmode = NORMAL_MODE;
         } break;
      case(COPY_MODE):
	 if(event->button == Button1) {
            XtRemoveEventHandler(areastruct.area, PointerMotionMask, False, 
		(XtEventHandler)drag, NULL);
            objectdeselect();
	 } break;
      case(COPY2_MODE): {
	 eventmode = NORMAL_MODE;
	 attachto = 0;
	 Wprintf("");
         XtRemoveEventHandler(areastruct.area, PointerMotionMask, False, 
	    (XtEventHandler)drag, NULL);
	 XDefineCursor(dpy, win, CROSS);
	 if(event->button == Button3) {
	    objectdelete(NORMAL);

	    /* Redraw the screen so that an object directly under */
	    /* the delete object won't get painted black */

            drawarea(NULL, objectdata, NULL);
	 }
	 else if (event->button == Button2)
            objectdeselect();
	 else {
	    short *csel;
            XSetFunction(dpy, areastruct.gc, GXcopy);
            XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
	    for (csel = areastruct.selectlist; csel < areastruct.selectlist +
                  areastruct.selects; csel++)
	       easydraw(*csel, DOFORALL);
	    copybutton(w, objectdata, event);
	 }
      } break;
      case(TEXT0_MODE): {
	 eventmode = TEXT2_MODE;
      } break;
      case(TEXT2_MODE): {
	 labelptr curlabel = TOLABEL(EDITPART);
	 UDrawTLine(curlabel);
	 if (event->button == Button3) {
	     UDrawString(areastruct.topobject, curlabel);
	     free(curlabel->string); 
	     free(curlabel);
	 }
	 else objectdata->parts++;
	 eventmode = NORMAL_MODE;
	 setdefaultfontmarks();
      } break;
      case(TEXT3_MODE): case(CATTEXT_MODE): {
	 labelptr curlabel = TOLABEL(EDITPART);
	 UDrawTLine(curlabel);
	 if (event->button == Button3) {
	    undrawtext(curlabel);
	    strcpy(curlabel->string, labelbuf); /* restore original text */
	    redrawtext(curlabel);
	 }
	 free(labelbuf);
	 if (eventmode == CATTEXT_MODE) eventmode = CATALOG_MODE;
	 else eventmode = NORMAL_MODE;
	 Wprintf("");
	 setdefaultfontmarks();
      } break;
      case(PRESS_MODE): {
         eventmode = NORMAL_MODE;
         XtRemoveEventHandler(areastruct.area, Button1MotionMask, False, 
		(XtEventHandler)drag, NULL);
	 attachto = 0;
	 Wprintf("");
         objectdeselect();
      } break;
      case(SELAREA_MODE): {
	 eventmode = NORMAL_MODE;
	 XtRemoveEventHandler(areastruct.area, ButtonMotionMask, False,
	    (XtEventHandler)trackselarea, NULL);
	 UDrawBox(areastruct.origin, areastruct.save);
	 selectarea();
      } break;
      case(SELAREA2_MODE): {
	 eventmode = SELAREA_MODE;
	 zoomin(w, Number(1), event);
      } break;
      case(NORMAL_MODE): {
         if(event->button == Button1) {
	    u2u_snap(&areastruct.save);
            startwire(areastruct.save);
	    eventmode = WIRE_MODE;
         }
	 else if(event->button == Button2) objectselect(ANY);
      } break;
      case(EPATH_MODE):
	 pathbutton((*((pathptr *)EDITPART))->plist + areastruct.editsubpart,
		event);
	 break;
      case(BOX_MODE): case(EBOX_MODE): case (ARC_MODE):
	     case(EARC_MODE): case(SPLINE_MODE): case(ESPLINE_MODE):
	 pathbutton(EDITPART, event);
	 break;
   }
   if (eventmode == NORMAL_MODE) {
      XDefineCursor(dpy, win, CROSS);
#ifdef DOUBLEBUFFER
      drawarea(NULL, objectdata, NULL);
#endif
   }
}

/*--------------------------------------------------------------*/
/* Button release handler for path components			*/
/*--------------------------------------------------------------*/

void pathbutton(genericptr *editpart, XButtonEvent *event) {
   short etype = (*editpart)->type;
   switch(etype) {
      case(POLYGON): {
	 if (eventmode == BOX_MODE) {
            polyptr   newbox;
            pointlist pointptr;

            newbox = TOPOLY(editpart);
	    UDrawPolygon(areastruct.topobject, newbox);

            /* prevent length and/or width zero boxes */
            if (newbox->points->x != (newbox->points + 2)->x && (newbox->points 
	         + 1)->y != (newbox->points + 3)->y) {
	       if (event->button != Button3) {
                  XSetFunction(dpy, areastruct.gc, GXcopy);
                  XTopSetForeground(newbox->color);
	          objectdata->parts++;
                  UDrawPolygon(areastruct.topobject, newbox);
	       }
	       else free(newbox);
            }
	    else free(newbox);

            XtRemoveEventHandler(areastruct.area, PointerMotionMask, False,
               (XtEventHandler)trackbox, NULL);
            eventmode = NORMAL_MODE;
         }
	 else {   /* EBOX_MODE or EPATH_MODE */
            genericptr   *replace;
            polyptr      newpoly;
            short        dotrack = True;
  
            replace = editpart;
            newpoly = TOPOLY(replace);
            attachto = 0;
   
            if (event->button != Button1) {
	       dotrack = False;
               UDrawPolygon(areastruct.topobject, newpoly);
            }

            if (event->button == Button1) {
	       nextpolycycle(newpoly, 1);
	       polyeditpush(newpoly);
	    }
            else if (event->button == Button2) {
               XPoint *curpt;
               XSetFunction(dpy, areastruct.gc, GXcopy); 
               XTopSetForeground(newpoly->color);
               UDrawPolygon(areastruct.topobject, newpoly);
	    }
            else {
	       free(newpoly);
	       if (areastruct.editstack->parts > 0) {
	          newpoly = TOPOLY(areastruct.editstack->plist +
			areastruct.editstack->parts - 1);
	          areastruct.editstack->parts--;
	          *replace = (genericptr)newpoly;
	          if (areastruct.editstack->parts > 0) {
		     dotrack = True;
		     nextpolycycle(newpoly, -1);
	          }
	          else {
         	     XcSetFunction(GXcopy);
         	     XcSetForeground(newpoly->color);
	          }
                  UDrawPolygon(areastruct.topobject, newpoly);
	       }
	       else
	          objectdata->parts--;
	    }

            if (!dotrack) {
	       /* Free the editstack */
	       reset(areastruct.editstack, NORMAL);

               XtRemoveEventHandler(areastruct.area, PointerMotionMask, False,
                   (XtEventHandler)trackpoly, NULL);
               eventmode = NORMAL_MODE;
            }
 	 }
      } break;
      case(ARC): {
	 genericptr *replace;
         arcptr   newarc;
         short    dotrack = True;

	 replace = editpart;
	 newarc = TOARC(replace);

	 if (event->button != Button1) {
	    dotrack = False;
	    UDrawArc(areastruct.topobject, newarc);
	    UDrawXLine(areastruct.save, newarc->position);
         }

	 if (event->button == Button1) {
	    nextarccycle(newarc, 1);
	    arceditpush(newarc);
	 }

         /* prevent radius zero arcs */
         else if (newarc->radius != 0 && newarc->yaxis != 0 &&
		   (newarc->angle1 != newarc->angle2)) {
            if (event->button == Button2) {
               XSetFunction(dpy, areastruct.gc, GXcopy);
               XTopSetForeground(newarc->color);
               if (eventmode == ARC_MODE) objectdata->parts++;
               UDrawArc(areastruct.topobject, newarc);
	    }
	    else {	 /* restore previous arc from edit stack */
	       free(newarc);
	       if (areastruct.editstack->parts > 0) {
	          newarc = TOARC(areastruct.editstack->plist +
			areastruct.editstack->parts - 1);
	          areastruct.editstack->parts--;
		  *replace = (genericptr)newarc;
		  if (areastruct.editstack->parts > 0) {
		     dotrack = True;
	             nextarccycle(newarc, -1);
		     UDrawXLine(areastruct.save, newarc->position);
		  }
		  else {
                     XSetFunction(dpy, areastruct.gc, GXcopy);
                     XTopSetForeground(newarc->color);
		  }
                  UDrawArc(areastruct.topobject, newarc);
	       }
	    }
         }
	 else {
	    if (eventmode != ARC_MODE) objectdata->parts--;
	    free(newarc);
	 } 

	 if (!dotrack) {
	    /* Free the editstack */
	    reset(areastruct.editstack, NORMAL);

            XtRemoveEventHandler(areastruct.area, PointerMotionMask, False,
               (XtEventHandler)trackarc, NULL);
            eventmode = NORMAL_MODE;
	 }
      } break;
      case(SPLINE): {
	 genericptr	*replace;
	 splineptr	newspline;
	 short 		dotrack = True;

	 replace = editpart;
	 newspline = TOSPLINE(editpart);

	 if (event->button != Button1) {
	    UDrawEditSpline(areastruct.topobject, newspline);
	    dotrack = False;
	 }

	 if (event->button == Button1) {
	    nextsplinecycle(newspline, -1);
	    splineeditpush(newspline);
	 }

	 /* unlikely but possible to create zero-length splines */
	 else if (newspline->ctrl[0].x != newspline->ctrl[3].x ||
	      newspline->ctrl[0].x != newspline->ctrl[1].x ||
	      newspline->ctrl[0].x != newspline->ctrl[2].x ||
	      newspline->ctrl[0].y != newspline->ctrl[3].y ||
	      newspline->ctrl[0].y != newspline->ctrl[1].y ||
	      newspline->ctrl[0].y != newspline->ctrl[2].y) {

	    if (event->button == Button2) {
	       XSetFunction(dpy, areastruct.gc, GXcopy);
	       XTopSetForeground(newspline->color);
	       if (eventmode == SPLINE_MODE) objectdata->parts++;
	       UDrawSpline(areastruct.topobject, newspline);
	    }
	    else {	/* restore previous spline from edit stack */
	       free(newspline);
	       if (areastruct.editstack->parts > 0) {
	          newspline = TOSPLINE(areastruct.editstack->plist +
			areastruct.editstack->parts - 1);
	          areastruct.editstack->parts--;
		  *replace = (genericptr)newspline;
		  if (areastruct.editstack->parts > 0) {
		     dotrack = True;
	             nextsplinecycle(newspline, 1);
	             UDrawEditSpline(areastruct.topobject, newspline);
		  }
		  else {
	             XSetFunction(dpy, areastruct.gc, GXcopy);
	             XTopSetForeground(newspline->color);
	             UDrawSpline(areastruct.topobject, newspline);
		  }
	       }
	    }
	 }
	 else {
	    if (eventmode != SPLINE_MODE) objectdata->parts--;
	    free(newspline);
	 }

	 if (!dotrack) {
	    /* Free the editstack */
	    reset(areastruct.editstack, NORMAL);

	    XtRemoveEventHandler(areastruct.area, PointerMotionMask, False,
	         (XtEventHandler)trackspline, NULL);
	    eventmode = NORMAL_MODE;
	 }
      } break;
   }
}

/*-------------------------------------------------------*/
/* Recalculate values for a drawing-area widget resizing */
/*-------------------------------------------------------*/

XtCallbackProc resizearea(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   Arg	wargs[2];
   XEvent discard;
#ifdef DOUBLEBUFFER
   int savewidth = areastruct.width, saveheight = areastruct.height;
#endif

   XtSetArg(wargs[0], XtNwidth, &areastruct.width);
   XtSetArg(wargs[1], XtNheight, &areastruct.height);
   XtGetValues(areastruct.area, wargs, 2);

   if (XtIsRealized(areastruct.area)) {

#ifdef DOUBLEBUFFER
      if (areastruct.width != savewidth || areastruct.height != saveheight) {
         if (dbuf != (Pixmap)NULL) XFreePixmap(dpy, dbuf);
         dbuf = XCreatePixmap(dpy, win, areastruct.width, areastruct.height,
	       DefaultDepthOfScreen(XtScreen(w)));
      }
#endif

      /* Recompute values for boundary and recenter image in resized window */

      calcbbox(objectdata);
      zoomview(NULL, Number(0), NULL);

      /* Flush all expose events from the buffer */

      while (XCheckWindowEvent(dpy, win, ExposureMask, &discard) == True);
   }
}

/*------------------------------------------*/
/* Draw the primary graphics window, "area" */
/*------------------------------------------*/

XtCallbackProc drawarea(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   float x, y, spc, spc2, i, j, fpart;
   short maxgridpts;
   short *selectobj;
   XPoint originpt;
   XEvent discard;
#ifdef DOUBLEBUFFER
   Window tmpwin;
#endif

   XSetFunction(dpy, areastruct.gc, GXcopy);

#ifdef DOUBLEBUFFER
   XSetForeground(dpy, areastruct.gc, BACKGROUND);
   XFillRectangle(dpy, dbuf, areastruct.gc, 0, 0, areastruct.width,
	areastruct.height);
   tmpwin = win;
   win = (Window)dbuf;
#else
   XClearWindow(dpy, win);
#endif

   XSetLineAttributes(dpy, areastruct.gc, 0, LineSolid, CapRound, JoinBevel);

   /* draw GRIDCOLOR lines for grid; mark axes in AXESCOLOR */

   if (eventmode != CATALOG_MODE && eventmode != CATPAN_MODE &&
	eventmode != FONTCAT_MODE && eventmode != FONTCAT2_MODE) {

      spc = areastruct.gridspace[areastruct.page] * (*areastruct.vscale);
      if (areastruct.gridon && spc > 8) {
	 fpart = (float)(-areastruct.lowerleft->x)
			/ areastruct.gridspace[areastruct.page];
         x = areastruct.gridspace[areastruct.page] *
		(fpart - (float)((int)fpart)) * (*areastruct.vscale);
	 fpart = (float)(-areastruct.lowerleft->y)
			/ areastruct.gridspace[areastruct.page];
         y = areastruct.gridspace[areastruct.page] *
		(fpart - (float)((int)fpart)) * (*areastruct.vscale);

         XSetForeground(dpy, areastruct.gc, GRIDCOLOR);
         for (i = x; i < (float)areastruct.width; i += spc) 
	    XDrawLine (dpy, win, areastruct.gc, (int)(i + 0.5),
		0, (int)(i + 0.5), areastruct.height);
         for (j = (float)areastruct.height - y; j > 0; j -= spc)
            XDrawLine (dpy, win, areastruct.gc, 0, (int)(j - 0.5),
		areastruct.width, (int)(j - 0.5));
      };
      if (areastruct.axeson) {
         XPoint zeropt;
         zeropt.x = zeropt.y = 0;
         XSetForeground(dpy, areastruct.gc, AXESCOLOR);
         user_to_window(zeropt, &originpt);
         XDrawLine (dpy, win, areastruct.gc, originpt.x, 0, originpt.x, 
	    areastruct.height);
         XDrawLine (dpy, win, areastruct.gc, 0, originpt.y, areastruct.width,
	    originpt.y);
      }

      /* bounding box goes beneath everything except grid/axis lines */
      if (areastruct.bboxon)
         UDrawBBox(areastruct.topobject->thisobject);

      /* draw a little red dot at each snap-to point */

      spc2 = areastruct.snapspace[areastruct.page] * (*areastruct.vscale);
      if (areastruct.snapto && spc2 > 8) {
         float x2, y2;

	 fpart = (float)(-areastruct.lowerleft->x)
			/ areastruct.snapspace[areastruct.page];
         x2 = areastruct.snapspace[areastruct.page] *
		(fpart - (float)((int)fpart)) * (*areastruct.vscale);
	 fpart = (float)(-areastruct.lowerleft->y)
			/ areastruct.snapspace[areastruct.page];
         y2 = areastruct.snapspace[areastruct.page] *
		(fpart - (float)((int)fpart)) * (*areastruct.vscale);

         XSetForeground(dpy, areastruct.gc, SNAPCOLOR);
         for (i = x2; i < areastruct.width; i += spc2)
            for (j = areastruct.height - y2; j > 0; j -= spc2)
               XDrawPoint (dpy, win, areastruct.gc, (int)(i + 0.5),
			(int)(j - 0.5));
      };
      XSetBackground(dpy, areastruct.gc, BACKGROUND);
   }

   /* draw all of the elements on the screen */

   XSetForeground(dpy, areastruct.gc, FOREGROUND);
   UDrawObject(areastruct.topobject, objectdata, TOPLEVEL, FOREGROUND);

   /* draw selected elements in the SELECTION color */

   for (selectobj = areastruct.selectlist; selectobj < areastruct.selectlist +
	 areastruct.selects; selectobj++)
      drawselected(selectobj);

   /* fast copy of buffer */

#ifdef DOUBLEBUFFER
   XSetFunction(dpy, areastruct.gc, GXcopy);
   win = tmpwin;
   XCopyArea(dpy, dbuf, win, areastruct.gc, 0, 0, areastruct.width,
	     areastruct.height, 0, 0);
#endif

   /* draw pending elements, if any */

   if (eventmode != NORMAL_MODE) {
      XSetFunction(dpy, areastruct.gc, GXcopy);
      if (eventmode == TEXT2_MODE) {
         labelptr newlabel = TOLABEL(EDITPART);
         XTopSetForeground(newlabel->color);
         UDrawString(areastruct.topobject, newlabel);
	 UDrawTLine(newlabel);
      }
      else if (eventmode == TEXT3_MODE || eventmode == CATTEXT_MODE) {
	 labelptr newlabel = TOLABEL(EDITPART);
	 UDrawTLine(newlabel);
      }
      else if (eventmode == SELAREA_MODE) {
	 UDrawBox(areastruct.origin, areastruct.save);
      }
      else {
         XSetFunction(dpy, areastruct.gc, GXxor);
         if (eventmode == BOX_MODE || eventmode == WIRE_MODE) {
            polyptr newpoly = TOPOLY(EDITPART);
            XSetXORFg(newpoly->color, BACKGROUND);
            UDrawPolygon(areastruct.topobject, newpoly);
         }
         else if (eventmode == ARC_MODE) {
            arcptr newarc = TOARC(EDITPART);
            XSetXORFg(newarc->color, BACKGROUND);
            UDrawArc(areastruct.topobject, newarc);
	    UDrawXLine(areastruct.save, newarc->position);
         }
         else if (eventmode == SPLINE_MODE) {
            splineptr newspline = TOSPLINE(EDITPART);
            XSetXORFg(newspline->color, BACKGROUND);
            UDrawEditSpline(areastruct.topobject, newspline);
         }
      }
   }

   /* flush out multiple expose/resize events from the event queue */

   while (XCheckWindowEvent(dpy, win, ExposureMask, &discard) == True);

   /* end by restoring graphics state */

   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
}

/*-------------------------------------------------------------------------*/
