/*-----------------------------------------------------------------------*/
/* text.c --- text processing routines for xcircuit		 	 */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University        	 */
/*-----------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Display *dpy;
extern Clientdata areastruct;
extern short textpos;
extern short fontcount;
extern fontinfo *fonts;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c            */
/*------------------------------------------------------------------------*/

short ULength();
extern float UTopScale();
extern int *appcolors;

/*---------------------------------------------------------------------*/
/* draw a single character at 0, 0 using current transformation matrix */
/*---------------------------------------------------------------------*/

short UDrawChar(localdata, code, styles, ffont, groupheight)
  objinstptr 	localdata;
  short 	code, styles, ffont;
  int		groupheight;
{
   objectptr drawchar;
   XPoint alphapts[2], *curpoint;
   short  totalpts, i;
   short  localwidth;
   XPoint *pointptr;
   objinst charinst;  /* to be replaced? */

   alphapts[0].x = 0;
   alphapts[0].y = 0;
   charinst.type = OBJECT;
   charinst.color = DEFAULTCOLOR;
   charinst.rotation = 0;
   charinst.scale = fonts[ffont].scale;
   charinst.position = alphapts[0];
   
   /* get proper font and character */

   drawchar = fonts[ffont].encoding[code];
   charinst.thisobject = drawchar;

   localwidth = (drawchar->lowerleft.x + drawchar->width) * fonts[ffont].scale;

   if ((fonts[ffont].flags & 0x22) == 0x22) { /* font is derived and italic */
      USlantCTM(DCTM, 0.25);  		/* premultiply by slanting function */
   }

   if (!(styles & 64)) {
      
      UDrawObject(&charinst, drawchar, SINGLE, charinst.color);

      /* under- and overlines */
      if (styles & 8)
         alphapts[0].y = alphapts[1].y = -6;
      else if (styles & 16)
         alphapts[0].y = alphapts[1].y = groupheight + 4;
      if (styles & 24) {
         alphapts[0].x = 0; alphapts[1].x = localwidth;
         UDrawSimpleLine(localdata, &alphapts[0], &alphapts[1]);
      }
   }
   return localwidth;
}

/*----------------------------------------------------------------------*/

void UDrawString(localdata, drawlabel)
  objinstptr 	localdata;
  labelptr	drawlabel;
{
   uchar *strptr = drawlabel->string;
   short  tmplength, fstyle, ffont, tmpwidth, tmpjust;
   short  backct = 1;
   int    group = 0;
   short  oldy, oldfont, oldstyle;
   float  tmpscale = 1.0;
   float  tmpthick = areastruct.wirewidth[areastruct.page];
   XPoint newpoint, bboxin[2], bboxout[2];
   Boolean xm, ym;

   if (fontcount == 0) return;

   /* calculate the transformation matrix for this object */
   /* in natural units of the alphabet vectors		  */
   /* (conversion to window units)			  */

   UPushCTM();
   UPreMultCTM(DCTM, drawlabel->position, drawlabel->scale, drawlabel->rotation);

   /* check for flip invariance; recompute CTM and justification if necessary */

   tmpjust = flipadjust(drawlabel->justify);

   tmplength = ULength(drawlabel->string, 0.0, 0); /* "natural" (unscaled) length */

   newpoint.x = (tmpjust & NOTLEFT ?
       (tmpjust & RIGHT ? -tmplength : -tmplength >> 1) : 0);
   newpoint.y = (tmpjust & NOTBOTTOM ?
       (tmpjust & TOP ? -TEXTHEIGHT : -HALFHEIGHT) : 0);


#ifdef SCHEMA
   /* Pinlabels have an additional offset spacing to pad */
   /* them from the circuit point to which they attach.  */

   if (drawlabel->pin)
      pinadjust(tmpjust, &(newpoint.x), &(newpoint.y));

#endif

   oldy = newpoint.y;

   /* do quick calculation on bounding box; don't draw if off-screen */

   bboxin[0].x = newpoint.x;
   bboxin[0].y = newpoint.y - HALFHEIGHT;
   bboxin[1].x = newpoint.x + tmplength;
   bboxin[1].y = newpoint.y + TEXTHEIGHT + HALFHEIGHT;
   UTransformbyCTM(DCTM, bboxin, bboxout, 2);
   xm = (bboxout[0].x < bboxout[1].x) ? 0 : 1;
   ym = (bboxout[0].y < bboxout[1].y) ? 0 : 1;

   if (bboxout[xm].x < areastruct.width && bboxout[ym].y < areastruct.height &&
       bboxout[!xm].x > 0 && bboxout[!ym].y > 0) {

       for (strptr = drawlabel->string; *strptr != '\0'; strptr++) {

          /* make a character transformation matrix by translating to newpoint */

          UPushCTM();
          UPreMultCTM(DCTM, newpoint, tmpscale, 0);

          /* deal with in-line text format control escape sequences */

          if ((short)(*strptr) == TEXT_ESC) {
	     short control = (short)(*(++strptr));

             if (control >= FONT_START) {
	        ffont = (control - FONT_START);
		fstyle = 0;		   /* style reset by font change */
	        if (oldy == newpoint.y) {  /* set top-level font and style */
	           oldfont = ffont;
	           oldstyle = fstyle;
	        }
		
		/* simple boldface technique for derived fonts */

		areastruct.wirewidth[areastruct.page] =
   		   ((fonts[ffont].flags & 0x21) == 0x21) ?  4.0 : 2.0;
             }
             else if (control == SUBSCRIPT) {
	        tmpscale *= SUBSCALE; 
	        newpoint.y -= (short)((TEXTHEIGHT >> 1) * tmpscale);
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == SUPERSCRIPT) {
	        tmpscale *= SUBSCALE;
	        newpoint.y += (short)(TEXTHEIGHT * tmpscale);
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == NORMALSCRIPT) {
	        tmpscale = 1.0;
	        ffont = oldfont;	/* revert to top-level font and style */
	        fstyle = oldstyle;
	        newpoint.y = oldy;
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == UNDERLINE) {
	        fstyle &= 0xfc7;
	        fstyle |= 8;
	     }
             else if (control == OVERLINE) {
	        uchar *quickptr = strptr + 1;
		objectptr charptr;
		int tmpheight;

	        /* any control character stops the /overline */ 
		group = 0;
	        for (; (*quickptr != TEXT_ESC) && (*quickptr != '\0'); quickptr++) {
		   charptr = fonts[ffont].encoding[*quickptr];
		   tmpheight = (int)((float)charptr->height * fonts[ffont].scale);
		   if (group < tmpheight) group = tmpheight;
		}
	        fstyle &= 0xfc7;
	        fstyle |= 16;
             }
             else if (control == NOLINE) fstyle &= 0xfc7;
             else if (control == BACKSPACE) {
		objectptr charptr;
	        if (strptr + 1 - (backct * 3) >= drawlabel->string) {
	           newpoint.x -= UDrawChar(localdata, *(strptr + 1 - (backct * 3)), 
	                  fstyle | 64, ffont, group) * tmpscale;
	        }
	        backct = ((short)(*(strptr + 1)) != TEXT_ESC || (short)(*(strptr + 2))
	    	!= BACKSPACE) ? 1 : backct + 1;
             }
	     else if (control == HALFSPACE || control == QTRSPACE) {
		short addx = UDrawChar(localdata, 32, fstyle, ffont, group);
		newpoint.x += addx >> ((control == HALFSPACE) ? 1 : 2);
	     }
          }
          else
	     newpoint.x += UDrawChar(localdata, *strptr, fstyle, ffont, group)
	    		* tmpscale;
   
          /* pop the character transformation matrix */

          UPopCTM();
      }
   }

   /* pop the string transformation matrix */

   UPopCTM();

#ifdef SCHEMA
   if (drawlabel->pin) {
      UDrawX(drawlabel);
   }
#endif

   areastruct.wirewidth[areastruct.page] = tmpthick;
}

/*------------------------------------------------------------------------*/

short ULength(string, newscale, dostop)
  uchar *string;
  float	newscale;
  short dostop;
{
   short backct = 1;
   float oldscale, strscale, locscale = 1.0, xtotal = 0.5;
   uchar *strptr;
   objinstptr *scaleobj;
   objectptr *somebet = NULL, chptr;

   if (fontcount == 0) return;

   if (newscale > 0.0) strscale = UTopScale() * newscale;
   else strscale = 1.0;
     
   oldscale = strscale;

   for (strptr = string; *strptr != '\0'; strptr++) {
      if (dostop & ((short)(strptr - string) == textpos)) break;
      if ((short)(*strptr) == TEXT_ESC) {
	 short control = (short)(*(++strptr));	 

         if (control == SUPERSCRIPT || control == SUBSCRIPT) strscale *= SUBSCALE;
         else if (control == NORMALSCRIPT) strscale = oldscale;
	 else if (control == HALFSPACE) {
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale / 2;
	 }      
	 else if (control == QTRSPACE) {
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x) 
			* locscale * strscale / 4;
	 }
	 else if (control == BACKSPACE) {
	    if (strptr + 1 - (backct * 3) >= string) {
	       chptr = (*(somebet + *(strptr + 1 - (backct * 3))));
	       xtotal -= (float)(chptr->width + chptr->lowerleft.x)
			 * locscale * strscale;
	    }
	    backct = (*(strptr + 1) != TEXT_ESC || *(strptr + 2) != BACKSPACE)
		 ? 1 : backct + 1;
         }
         else if (control >= FONT_START) {
	    somebet = fonts[control - FONT_START].encoding;
	    locscale = fonts[control - FONT_START].scale;
	 }
      }
      else {
	 chptr = (*(somebet + *strptr));
	 xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale;
      }
   }
   return ((short)xtotal);
}

/*------------------------------------------------------------------------*/
/* simple routines for drawing and erasing labels */
/*------------------------------------------------------------------------*/

undrawtext(settext)
  labelptr settext;
{
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BACKGROUND);
   UDrawString(areastruct.topobject, settext);
}

redrawtext(settext)
  labelptr settext;
{
   XTopSetForeground(settext->color);
   UDrawString(areastruct.topobject, settext);
}

/*------------------------------------------------------------------------*/
