/*-------------------------------------------------------------------------*/
/* libraries.c --- xcircuit routines for the builtin and user libraries    */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"

#include <math.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

extern short   eventmode;	/* keep track of the mode the screen is in */
extern Display	*dpy;   /* Works well to make this globally accessible */
extern Window win;
extern int *appcolors;
extern Cursor	appcursors[NUM_CURSORS];
extern Globaldata xobjs;
extern Clientdata areastruct;
extern Pagedata **pagelist;
extern objectptr libtop[LIBS];
extern char _STR[150];
extern objectpair *pushlist;
extern short pushes;
extern short fontcount;
extern fontinfo *fonts;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to events.c	 	  */
/*------------------------------------------------------------------------*/

extern void drawarea(), drag(), newpage(), setpage();
extern short *allocselect();
void composelib(), composepagelib(), changecat();

#ifdef SCHEMA
#define _lowerleft lleft2
#define _width width2
#define _height height2
#else
#define _lowerleft lowerleft
#define _width width
#define _height height
#endif

/*---------------------------------------------------------*/
/* Find the Helvetica font for use in labeling the objects */
/*---------------------------------------------------------*/

short findhelvetica()
{
   short fval;

   for (fval = 0; fval < fontcount; fval++)
      if (!strcmp(fonts[fval].psname, "Helvetica"))
	 break; 

   /* If not there, use the first Helvetica font */

   if (fval == fontcount) {
      for (fval = 0; fval < fontcount; fval++)
         if (!strcmp(fonts[fval].family, "Helvetica"))
	    break; 
   }

   /* If still not there, use the first non-Symbol font */
   /* If this doesn't work, then the libraries are probably misplaced. . .*/

   if (fval == fontcount) {
      for (fval = 0; fval < fontcount; fval++)
         if (strcmp(fonts[fval].family, "Symbol"))
	    break; 
   }

   return fval;
}

/*-------------------------------------------*/
/* Return to drawing window from the library */
/*-------------------------------------------*/

void catreturn()
{
   /* pop the object being edited from the push stack */

   popobject(NULL, Number(1), NULL);
}

/*------------------------------------------------------*/
/* Find page number from cursor position		*/
/* Mode = 0:  Look for exact corresponding page number  */
/*   and return -1 if out-of-bounds			*/
/* Mode = 1:  Look for position between pages, return	*/
/*   page number of page to the right.  		*/
/*------------------------------------------------------*/

int pageposition(XButtonEvent *event, int mode)
{
   int xin, yin, bpage;
   int xdel, ydel;
   int gxsize, gysize;

   /* gxsize is the number of pages per line */

   gxsize = (int)sqrt((double)xobjs.pages) + 1;
   gysize = 1 + xobjs.pages / gxsize;

   /* 0.5 is the default vscale */

   xdel = areastruct.width / (0.5 * gxsize);
   ydel = areastruct.height / (0.5 * gysize);

   window_to_user(event->x, event->y, &areastruct.save);

   if (mode == 0) {	/* On-page */
      if (areastruct.save.x >= 0 && areastruct.save.y <= 0) {
         xin = areastruct.save.x / xdel;
         yin = areastruct.save.y / ydel; 
         if (xin < gxsize && yin > -gysize) {
            bpage = (xin % gxsize) - (yin * gxsize);
            if (bpage < xobjs.pages)
	       return bpage;
         }
      }
      return -1;
   }
   else {		/* Between-pages */
      xin = (areastruct.save.x + (xdel >> 1)) / xdel;
      if (xin > gxsize) xin = gxsize;
      if (xin < 0) xin = 0;
      yin = areastruct.save.y  / ydel; 
      if (yin > 0) yin = 0;
      if (yin < -gysize) yin = -gysize;
      bpage = (xin % (gxsize + 1)) + 1 - (yin * gxsize);
      if (bpage > xobjs.pages + 1) bpage = xobjs.pages + 1;
      return bpage;
   }
}

/*------------------------------------------------------*/
/* ButtonPress handler during page catalog viewing mode */
/*------------------------------------------------------*/

XtEventHandler pagecatbutton(w, mode, event)
  Widget w;
  caddr_t mode;
  XButtonEvent *event;
{
   int bpage;

   if (event->button == Button1 || event->button == Button2) {
      if ((bpage = pageposition(event, 0)) >= 0) {
	 if (event->button == Button1) {

	    /* like catreturn(), but don't actually go to the popped page */
	    objectdeselect();
      	    eventmode = NORMAL_MODE;
	    newpage(bpage);
      	    eventmode = PRESS_MODE;
	    return;
	 }
	 else {  /* Button2 */
	    objectselect(OBJECT);
	 }
      }
   }
   else {
      eventmode = NORMAL_MODE;
      catreturn();
   }
}

/*--------------------------------------*/
/* Draw the catalog of page ordering	*/
/*--------------------------------------*/

void composepagelib()
{
   objinstptr *drawinst;
   labelptr *drawname;
   objectptr libobj;
   objectptr libinst = libtop[PAGELIB];
   short visobjects, libobjects, i;
   polyptr *drawbox;
   pointlist pointptr;
   XPoint dpt;
   int xdel, ydel, gxsize, gysize, bbuf;
   float scalex, scaley;

   reset(libinst, NORMAL);

   /* Find the number of non-empty pages. */

   visobjects = 0;
   for(i = 0; i < xobjs.pages; i++)
      if (pagelist[i]->pageobj != NULL) visobjects++;

   /* add the bounding boxes */

   visobjects += xobjs.pages;

   /* generate the list of object instances */

   libinst->plist = (genericptr *) realloc(libinst->plist, visobjects
		* sizeof(genericptr));
   libinst->parts = 0;

   gxsize = (int)sqrt((double)xobjs.pages) + 1;
   gysize = 1 + xobjs.pages / gxsize;

   /* 0.5 is the default vscale;  g#size is the number of pages per line */
   xdel = areastruct.width / (0.5 * gxsize);
   ydel = areastruct.height / (0.5 * gysize);
   bbuf = xdel / 40;

   for (i = 0; i < xobjs.pages; i++) {

      if ((libobj = pagelist[i]->pageobj) == NULL) continue;
      
      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->params = NULL;
      (*drawinst)->num_params = (uchar)0;
      (*drawinst)->thisobject = libobj;

      (*drawinst)->position.x = (i % gxsize) * xdel;
      (*drawinst)->position.y = -(i / gxsize + 1) * ydel;

      /* center the object on its page bounding box */

      scalex = (0.9 * xdel) / libobj->width;
      scaley = (0.9 * ydel) / libobj->height;
      if (scalex > scaley) {
         (*drawinst)->scale = scaley;
	 (*drawinst)->position.x -= (libobj->lowerleft.x * scaley);
         (*drawinst)->position.x += (xdel - (libobj->width * scaley)) / 2;
         (*drawinst)->position.y += 0.05 * ydel - libobj->lowerleft.y
			* (*drawinst)->scale;
      }
      else {
         (*drawinst)->scale = scalex;
	 (*drawinst)->position.y -= (libobj->lowerleft.y * scalex);
         (*drawinst)->position.y += (ydel - (libobj->height * scalex)) / 2;
         (*drawinst)->position.x += 0.05 * xdel - libobj->lowerleft.x
			* (*drawinst)->scale;
      }
      libinst->parts++;
   }

   /* separate pages (including empty ones) with bounding boxes */

   for (i = 0; i < xobjs.pages; i++) {
      drawbox = (polyptr *)libinst->plist + libinst->parts;
      *drawbox = (polyptr) malloc(sizeof(polygon));
      (*drawbox)->type = POLYGON;
      (*drawbox)->color = SNAPCOLOR;   /* default red */
      (*drawbox)->style = NORMAL;      /* CLOSED */
      (*drawbox)->width = 1.0;
      (*drawbox)->number = 4;
      (*drawbox)->points = (pointlist) malloc(4 * sizeof(XPoint));
      pointptr = (*drawbox)->points;
      pointptr->x = (i % gxsize) * xdel + bbuf;
      pointptr->y = -(i / gxsize) * ydel - bbuf;
      pointptr = (*drawbox)->points + 1;
      pointptr->x = ((i % gxsize) + 1) * xdel - bbuf;
      pointptr->y = -(i / gxsize) * ydel - bbuf;
      pointptr = (*drawbox)->points + 2;
      pointptr->x = ((i % gxsize) + 1) * xdel - bbuf;
      pointptr->y = -((i / gxsize) + 1) * ydel + bbuf;
      pointptr = (*drawbox)->points + 3;
      pointptr->x = (i % gxsize) * xdel + bbuf;
      pointptr->y = -((i / gxsize) + 1) * ydel + bbuf;
      libinst->parts++;
   }

   /* calculate a bounding box for this display */
   /* and center it in its window */

   calcbbox(libinst);
   centerview(libinst);
}

/*----------------------*/
/* Rearrange pages	*/
/*----------------------*/

void pagecatmove(XKeyEvent *event)
{
   int bpage;
   short *newselect;
   objectptr exchobj;
   Pagedata *ipage, **testpage, **tpage2;

   if (areastruct.selects == 0) return;
   else if (areastruct.selects > 2) {
      Wprintf("Select maximum of two objects.");
      return;
   }

   /* Get the page corresponding to the first selected object */

   newselect = areastruct.selectlist;
   exchobj = SELTOOBJINST(newselect)->thisobject;
   for (testpage = pagelist; testpage < pagelist + xobjs.pages; testpage++)
      if (*testpage != NULL && (*testpage)->pageobj == exchobj) break;

   /* If two objects are selected, then exchange their order */

   if (areastruct.selects == 2) {
      newselect = areastruct.selectlist + 1;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      for (tpage2 = pagelist; tpage2 < pagelist + xobjs.pages; tpage2++)
	 if (*tpage2 != NULL && (*tpage2)->pageobj == exchobj) break;

      ipage = *testpage;
      *testpage = *tpage2;
      *tpage2 = ipage;
   }

   /* If one object selected; find place to put from cursor position */

   else if ((bpage = pageposition((XButtonEvent *)event, 1)) >= 0) {
      int k, epage;
      Pagedata *eptr;

      /* Find page number of the original page */

      epage = (int)(testpage - pagelist);
      eptr = *(pagelist + epage);

      /* move page (epage) to position between current pages */
      /* (bpage - 2) and (bpage - 1) by shifting pointers.   */

      if ((bpage - 1) < epage) {
	 for (k = epage - 1; k >= bpage - 1; k--) {
	    *(pagelist + k + 1) = *(pagelist + k);
	    renamepage(k + 1);
	 }
	 *(pagelist + bpage - 1) = eptr;
	 renamepage(bpage - 1);
      }
      else if ((bpage - 2) > epage) {
	 for (k = epage + 1; k <= bpage - 2; k++) {
	    *(pagelist + k - 1) = *(pagelist + k);
	    renamepage(k - 1);
	 }
	 *(pagelist + bpage - 2) = eptr;
	 renamepage(bpage - 2);
      }
   }

   objectdeselect();
   composepagelib();
   drawarea(NULL, objectdata, NULL);
}

/*-----------------------------------------*/
/* Draw the catalog of predefined elements */
/*-----------------------------------------*/

void composelib(mode)
   short mode;
{
   short visobjects, libobjects;
   objinstptr *drawinst;
   labelptr *drawname;
   objectptr *curlib, *libobj, libinst = libtop[mode];
   int xpos = 0, ypos = areastruct.height << 1;
   int nypos = 220, nxpos;
   short fval;

   reset(libinst, NORMAL);
   if (libobjects == 0) return;

   /* Create pointers to the current library and number of objects */

   libobjects = (mode == LIBRARY) ? xobjs.builtins : xobjs.userobjs;
   curlib = (mode == LIBRARY) ? xobjs.library : xobjs.userlib;

   /* Look for all "hidden" objects and subtract them from the total number */

   visobjects = libobjects;
   for(libobj = curlib; libobj < curlib + libobjects; libobj++)
      if ((*libobj)->hidden == True)
	 visobjects--;

   /* Find the Helvetica font for use in labeling the objects */

   fval = findhelvetica();

   /* generate the list of object instances and their labels */

   libinst->plist = (genericptr *) realloc(libinst->plist,
	((fval == fontcount) ? visobjects : (visobjects * 2)) * sizeof(genericptr));
   libinst->parts = 0;

   for(libobj = curlib; libobj < curlib + libobjects; libobj++){
      
      /* "Hidden" objects are not drawn */

      if ((*libobj)->hidden == True) continue;

      /* Otherwise, determine the area needed on the page to draw the object */

      nxpos = xpos + (((*libobj)->_width > 170) ? (*libobj)->_width + 30 : 200);
      if ((nxpos > (areastruct.width << 1)) && (xpos > 0)) {
	 nxpos -= xpos; 
	 xpos = 0;
	 ypos -= nypos;
	 nypos = 200;
      }
      /* extra space of 20 is to leave room for the label */

      if ((*libobj)->_height > (nypos - 50)) nypos = (*libobj)->_height + 50;

      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->scale = 1.0;
      (*drawinst)->params = NULL;	/* library object has all default values */
      (*drawinst)->num_params = (uchar)0;
      (*drawinst)->thisobject = (*libobj);

      (*drawinst)->position.x = xpos - (*libobj)->_lowerleft.x;
      if ((*libobj)->_width <= 170) (*drawinst)->position.x += 
	   ((170 - (*libobj)->_width) >> 1);

      (*drawinst)->position.y = ypos - ((*libobj)->_height + (*libobj)->_lowerleft.y);
      if ((*libobj)->_height <= 170) (*drawinst)->position.y -=
	   ((170 - (*libobj)->_height) >> 1);

      if (fval < fontcount) {
         drawname = (labelptr *)libinst->plist + (libinst->parts++) + visobjects;
         *drawname = (labelptr) malloc(sizeof(label));      
         (*drawname)->type = LABEL;
         (*drawname)->color = DEFAULTCOLOR;
         (*drawname)->rotation = 1;
         (*drawname)->scale = 0.75;
         (*drawname)->string = (uchar *) malloc((strlen((*libobj)->name) + 3) *
	      sizeof(uchar));
         sprintf((*drawname)->string, "%c%c%s", (char)TEXT_ESC,
		(char)(FONT_START + fval), (*libobj)->name);
         (*drawname)->justify = TOP | NOTBOTTOM | NOTLEFT;
#ifdef SCHEMA
         (*drawname)->pin = False;
#endif

         if ((*libobj)->_width > 170)
            (*drawname)->position.x = xpos + ((*libobj)->_width >> 1);
         else
            (*drawname)->position.x = xpos + 85;

         if ((*libobj)->_height > 170)
            (*drawname)->position.y = (*drawinst)->position.y +
			(*libobj)->_lowerleft.y - 10;
         else
            (*drawname)->position.y = ypos - 180;

      }
      else (libinst->parts++);
      xpos = nxpos;
   }
   libinst->parts = (fval == fontcount) ? visobjects : (visobjects * 2);

   /* Compute the bounding box of the library page */
   calcbbox(libinst);
   centerview(libinst);
}

/*----------------------------------------------------------------*/
/* Find any dependencies on an object.				  */
/*   Return values:  0 = no dependency, 1 = dependency on page,	  */
/*	2 = dependency in another library object.		  */
/*   Object/Page with dependency (if any) returned in "compobjp". */
/*----------------------------------------------------------------*/

short finddepend(libobj, compobjp)
  objinstptr libobj;
  objectptr **compobjp;
{
   genericptr *testobj;
   short page;
   objectptr *compobj;

   for (compobj = xobjs.library; compobj != xobjs.userlib +
          xobjs.userobjs; compobj++) {
      if (compobj == xobjs.library + xobjs.builtins) {
         compobj = xobjs.userlib;
         if (xobjs.userobjs == 0) break;
      }
      *compobjp = compobj;
		     
      for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	       + (*compobj)->parts; testobj++) {
	 if ((*testobj)->type == OBJECT) {
	    if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 2;
	 }
      }
   }

   /* also look in the pagelist */

   for (page = 0; page < xobjs.pages; page++) {
      compobj = &(pagelist[page]->pageobj);
      if (*compobj == NULL) continue;
      *compobjp = compobj;
      for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	       + (*compobj)->parts; testobj++) {
	 if ((*testobj)->type == OBJECT) {
	    if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 1;
	 }
      }
   }
   return 0;
}

/*----------------------------------------------------------------*/
/* "Hide" an object (must have a dependency or else it disappears)*/
/*----------------------------------------------------------------*/

cathide()
{
   short *newselect;
   objectptr *libpage, *compobj;
   objinstptr libobj;
   short *libpobjs;

   if (areastruct.selects == 0) return;

   if (objectdata == libtop[LIBRARY]) {
      libpage = xobjs.library;
      libpobjs = &xobjs.builtins;
   }
   else {
      libpage = xobjs.userlib;
      libpobjs = &xobjs.userobjs;
   }

   /* Can only hide objects which are instances in other objects; */
   /* Otherwise, object would be "lost".			  */

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      if (finddepend(libobj, &compobj) == 0) {
	 sprintf(_STR, "Cannot hide: no dependencies");
	 Wprintf(_STR);
      }
      else { 		/* All's clear to hide. */
	 libobj->thisobject->hidden = True;
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if (objectdata == libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*----------------------------------------------------------------*/
/* Delete an object from the library if there are no dependencies */
/*----------------------------------------------------------------*/

catdelete()
{
   short *newselect;
   genericptr *testobj, *tobj;
   objinstptr libobj;
   objectptr *libpage, *compobj, *tlib, *slib;
   short *libpobjs;

   if (areastruct.selects == 0) return;

   if (objectdata == libtop[LIBRARY]) {
      libpage = xobjs.library;
      libpobjs = &xobjs.builtins;
   }
   else {
      libpage = xobjs.userlib;
      libpobjs = &xobjs.userobjs;
   }

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      /* Cannot delete an object if another object uses an instance of it, */
      /* or if the object is used on a page.				   */

      if (finddepend(libobj, &compobj)) {
	 sprintf(_STR, "Cannot delete: dependency in \"%s\"", (*compobj)->name);
	 Wprintf(_STR);
      }
      else { 		/* All's clear to delete.		     	   */

         /* First remove any instances of this object in the delete buffer */

         for (compobj = xobjs.delbuffer; compobj != xobjs.delbuffer +
                xobjs.deletes; compobj++) {
	    for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	          + (*compobj)->parts; testobj++) {
	       if ((*testobj)->type == OBJECT) {
	          if (TOOBJINST(testobj)->thisobject == libobj->thisobject) {
		     free(TOOBJINST(testobj));
		     for (tobj = testobj; tobj < (*compobj)->plist
			   + (*compobj)->parts - 1; tobj++) {
		        (*tobj) = (*(tobj + 1));
		     }
		     (*compobj)->parts--;
		  }
	       }
	    }
         }

	 for (tlib = libpage; tlib < libpage + *libpobjs; tlib++)
	    if ((*tlib) == libobj->thisobject) {
	       for (slib = tlib; slib < libpage + *libpobjs - 1; slib++)
		  (*slib) = (*(slib + 1));
	       (*libpobjs)--;
	       break;
	    }
	 
	 reset(libobj->thisobject, DESTROY);
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if (objectdata == libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Rearrange objects in the library			  */
/*--------------------------------------------------------*/

void catmove(XKeyEvent *event)
{
   short *newselect;
   objectptr lobj, exchobj;
   objectptr *libpage, iobj, *sobj, *testobj, *tobj2;
   objinstptr *libobj, *saveobj;
   short *libpobjs, begflag;
   short ocentx, ocenty, rangey;

   if (areastruct.selects == 0) return;
   else if (areastruct.selects > 2) {
      Wprintf("Select maximum of two objects.");
      return;
   }

   if (objectdata == libtop[LIBRARY]) {
      libpage = xobjs.library;
      libpobjs = &xobjs.builtins;
   }
   else {
      libpage = xobjs.userlib;
      libpobjs = &xobjs.userobjs;
   }

   /* If two objects are selected, then exchange their order */

   if (areastruct.selects == 2) {
      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      newselect = areastruct.selectlist + 1;
      lobj = SELTOOBJINST(newselect)->thisobject;

      for (testobj = libpage; testobj < libpage + *libpobjs; testobj++)
	 if (*testobj == exchobj) break;

      for (tobj2 = libpage; tobj2 < libpage + *libpobjs; tobj2++)
	 if (*tobj2 == lobj) break;

      iobj = *testobj;
      *testobj = *tobj2;
      *tobj2 = iobj;
   }
   else {	/* one object selected; find place to put from cursor position */

      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      saveobj = NULL;
      begflag = 1;

      window_to_user(event->x, event->y, &areastruct.save);
	
      for (libobj = (objinstptr *)ENDPART - 1; libobj >=
		(objinstptr *)objectdata->plist; libobj--) {

	 if ((*libobj)->type == OBJECT) {
 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->_lowerleft.x
	        + ((*libobj)->thisobject->_width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->_lowerleft.y
	        + ((*libobj)->thisobject->_height >> 1);
	    rangey = ((*libobj)->thisobject->_height > 200) ? 
	        ((*libobj)->thisobject->_height >> 1) : 100;

	    if (areastruct.save.y < ocenty + rangey && areastruct.save.y 
	         > ocenty - rangey) {
	       if (areastruct.save.x > ocentx) break;
	       saveobj = libobj;
	    }
	 }
      }

      if (libobj < (objinstptr *)objectdata->plist) {

	 /* insert at beginning */
	 if (areastruct.save.y > ocenty - rangey) {
            begflag = 0;
	    libobj++;
	 }
	 /* insert at beginning of a line */
	 else if (saveobj != NULL) {
	    libobj = saveobj - 1;
	 }
	 /* relocate to the end */
	 else {
	    libobj = (objinstptr *)ENDPART;
	    while ((*(--libobj))->type != OBJECT);
	 }
      }

      for (testobj = libpage; testobj < libpage + *libpobjs; testobj++)
	 if (*testobj == exchobj) break;
      for (tobj2 = libpage; tobj2 < libpage + *libpobjs; tobj2++)
	 if (*tobj2 == (*libobj)->thisobject) break;

      iobj = *testobj;

      /* shift library list */

      if (tobj2 > testobj) {
	 for (sobj = testobj; sobj < tobj2; sobj++)
	    *sobj = *(sobj + 1);
         *(tobj2) = iobj;
      }
      else if (tobj2 != testobj) {
	 for (sobj = testobj; sobj > tobj2 + begflag; sobj--)
	    *sobj = *(sobj - 1);
         *(tobj2 + begflag) = iobj;
      }
   }

   objectdeselect();
   if (objectdata == libtop[LIBRARY])
      composelib(LIBRARY);
   else
      composelib(USERLIB);

   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Make a duplicate of an object, put in the User Library */
/*--------------------------------------------------------*/

void copycat()
{
   short *newselect;
   objectptr *newobj, *curlib, oldobj;
   objinstptr libobj;
   int i;

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {

      libobj = SELTOOBJINST(newselect);
      oldobj = libobj->thisobject;

      /* generate new entry in user library */

      curlib = (objectptr *) realloc(xobjs.userlib,
	 (xobjs.userobjs + 1) * sizeof(objectptr));
      xobjs.userlib = curlib;
      newobj = xobjs.userlib + xobjs.userobjs;
      *newobj = (objectptr) malloc(sizeof(object));
      xobjs.userobjs++;

      /* give the new object a unique name */

      sprintf(_STR, "_%s", oldobj->name);
      checkname(*newobj);

      /* copy other object properties */

      (*newobj)->width = oldobj->width;
      (*newobj)->height = oldobj->height;
      (*newobj)->lowerleft.x = oldobj->lowerleft.x;
      (*newobj)->lowerleft.y = oldobj->lowerleft.y;
#ifdef SCHEMA
      /* don't attach the same schematic. . . */
      (*newobj)->symschem = NULL;
      /* and don't copy the parameter structure (for now) */
      (*newobj)->params = NULL;
      (*newobj)->num_params = 0;

      (*newobj)->lleft2.x = oldobj->lleft2.x;
      (*newobj)->lleft2.y = oldobj->lleft2.y;
      (*newobj)->width2 = oldobj->width2;
      (*newobj)->height2 = oldobj->height2;
#endif
      (*newobj)->hidden = False;

      /* copy over all the elements of the original object */

      (*newobj)->parts = 0;
      (*newobj)->plist = (genericptr *)malloc(sizeof(genericptr));

      for (i = 0; i < oldobj->parts; i++) {
	 switch((*(oldobj->plist + i))->type) {
	    case(PATH): {
	       register pathptr *npath, cpath = TOPATH(oldobj->plist + i);
	       register genericptr *gpart;
	       NEW_PATH(npath, (*newobj));
	       (*npath)->style = cpath->style;
	       (*npath)->width = cpath->width;
	       (*npath)->parts = 0;
	       (*npath)->plist = (genericptr *)malloc(cpath->parts *
			sizeof(genericptr));
	       for (gpart = cpath->plist; gpart < cpath->plist + cpath->parts;
			gpart++) {
		  switch((*gpart)->type){
		     case ARC: {
			arcptr copyarc = TOARC(gpart);
			arcptr *newarc;
			NEW_ARC(newarc, (*npath));
			arccopy(*newarc, copyarc);
			(*npath)->parts++;
			} break;
		     case POLYGON: {
                        polyptr copypoly = TOPOLY(gpart);
                        polyptr *newpoly;
                        NEW_POLY(newpoly, (*npath));
                        polycopy(*newpoly, copypoly);
                        (*npath)->parts++;
                        } break;
                     case SPLINE: {
                        splineptr copyspline = TOSPLINE(gpart);
                        splineptr *newspline;
                        NEW_SPLINE(newspline, (*npath));
                        splinecopy(*newspline, copyspline);
                        (*npath)->parts++;
                        } break;
		  }
	       }	
	       } break;
	    case(ARC): {
	       register arcptr *narc, carc = TOARC(oldobj->plist + i);
	       NEW_ARC(narc, (*newobj));
	       arccopy(*narc, carc);
               } break;
      
	    case(POLYGON): {
	       register polyptr *npoly, cpoly;
	       register int j;

	       cpoly = TOPOLY(oldobj->plist + i);
	       NEW_POLY(npoly, (*newobj));
	       polycopy(*npoly, cpoly);
               } break;

	    case(SPLINE): {
	       register splineptr *nspl, cspl;
	       register int j;

	       cspl = TOSPLINE(oldobj->plist + i);
	       NEW_SPLINE(nspl, (*newobj));
	       splinecopy(*nspl, cspl);
      	       } break;

	    case(LABEL): {
	       register labelptr *nlabel, clabel;

	       clabel = TOLABEL(oldobj->plist + i);
	       NEW_LABEL(nlabel, (*newobj));
	       (*nlabel)->rotation = clabel->rotation;
	       (*nlabel)->color = clabel->color;
	       (*nlabel)->position.x = clabel->position.x;
	       (*nlabel)->position.y = clabel->position.y;
	       (*nlabel)->scale = clabel->scale;
	       (*nlabel)->justify = clabel->justify;
	       (*nlabel)->string = (char *)malloc((strlen(clabel->string) + 1) *
		      sizeof(char));
	       strcpy((*nlabel)->string, clabel->string);
#ifdef SCHEMA
	       (*nlabel)->pin = clabel->pin;
#endif
               } break;
      
	    case(OBJECT): {
	       register objinstptr *ninst, cinst;

	       cinst = TOOBJINST(oldobj->plist + i);
	       NEW_OBJINST(ninst, (*newobj));
	       (*ninst)->rotation = cinst->rotation;
	       (*ninst)->color = cinst->color;
	       (*ninst)->position.x = cinst->position.x;
	       (*ninst)->position.y = cinst->position.y;
	       (*ninst)->scale = cinst->scale;
	       (*ninst)->thisobject = cinst->thisobject;

	       /* Library instance's parameter list is always default (null) */
	       (*ninst)->params = NULL;
	       (*ninst)->num_params = (uchar)0;
	      
#ifdef SCHEMA
	       /* here---deal with attached schematics */
#endif
               } break;
         }
	 (*newobj)->parts++;
      }
   }

   composelib(USERLIB);
   objectdeselect();

   if (objectdata == libtop[USERLIB])
      drawarea(NULL, objectdata, NULL);
   else changecat();
}

/*--------------------------------------------------------*/
/* ButtonPress handler during normal catalog viewing mode */
/*--------------------------------------------------------*/

XtEventHandler catbutton(w, mode, event)
  Widget w;
  caddr_t mode;
  XButtonEvent *event;
{
   short *newselect;
   objinstptr *newobject, *libobj;
   objectptr libpage = objectdata;
   short ocentx, ocenty, rangex, rangey, xdiff, ydiff, flag = 0;
   XPoint oldpos;

   if (event->button == Button1 || event->button == Button2) {

      window_to_user(event->x, event->y, &areastruct.save);
	
      for (libobj = (objinstptr *)objectdata->plist; libobj <
		(objinstptr *)objectdata->plist + objectdata->parts; libobj++) {
	 if ((*libobj)->type == OBJECT) {

 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->_lowerleft.x
	        + ((*libobj)->thisobject->_width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->_lowerleft.y
	        + ((*libobj)->thisobject->_height >> 1);

	    rangex = ((*libobj)->thisobject->_width > 200) ? 
	        ((*libobj)->thisobject->_width >> 1) : 100;
	    rangey = ((*libobj)->thisobject->_height > 200) ? 
	        ((*libobj)->thisobject->_height >> 1) : 100;

	    if (areastruct.save.x > ocentx - rangex && areastruct.save.x <
		   ocentx + rangex && areastruct.save.y < ocenty + rangey
		   && areastruct.save.y > ocenty - rangey) {

	       /* setup to move object around and draw the selected object */

               if (event->button == Button1) {

	          /* revert to old page */

	          objectdata = (pushes == 0) ? pagelist[areastruct.page]->pageobj
			   : (*(pushlist + pushes - 1)).thisobject;

	          /* retrieve drawing window state and set position of object to
	             be correct in reference to that window */

	          snap(event->x, event->y, &oldpos);

   	          setpage();
	    
	          snap(event->x, event->y, &areastruct.save);
	          xdiff = areastruct.save.x - oldpos.x;
	          ydiff = areastruct.save.y - oldpos.y;

                  /* collect all of the selected items */

	          for (newselect = areastruct.selectlist; newselect <
		     areastruct.selectlist + areastruct.selects; newselect++) {
		     NEW_OBJINST(newobject, objectdata);
		     objectdata->parts++;
		     (*newobject)->color = areastruct.color;
		     (*newobject)->rotation = TOOBJINST(libpage->plist +
			    *newselect)->rotation;
	             (*newobject)->scale = TOOBJINST(libpage->plist +
		            *newselect)->scale;
	             (*newobject)->thisobject = TOOBJINST(libpage->plist +
		            *newselect)->thisobject;
		     (*newobject)->position.x = TOOBJINST(libpage->plist +
		            *newselect)->position.x + xdiff;
		     (*newobject)->position.y = TOOBJINST(libpage->plist +
		            *newselect)->position.y + ydiff;
		     (*newobject)->params = NULL;
		     (*newobject)->num_params = (uchar)0;
	             u2u_snap(&((*newobject)->position));
		     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);
		     if ((*newobject)->thisobject == (*libobj)->thisobject)
		        flag = 1;
	          }

                  /* add final object to the list of object instances */

	          if (!flag) {
		     NEW_OBJINST(newobject, objectdata);
                     objectdata->parts++;
	             (*newobject)->rotation = (*libobj)->rotation;
		     (*newobject)->color = areastruct.color;
	             (*newobject)->scale = (*libobj)->scale;
	             (*newobject)->thisobject = (*libobj)->thisobject;
                     (*newobject)->position.x = areastruct.save.x;
	             (*newobject)->position.y = areastruct.save.y; 
		     (*newobject)->params = NULL;
		     (*newobject)->num_params = (uchar)0;

	             /* add this object to the list of selected items */

	             newselect = allocselect();
                     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);

	          }
	          if ((void *)mode == (void *)(1)) {

		     /* key "c" pressed for "copy" */

                     XDefineCursor(dpy, win, COPYCURSOR);
                     eventmode = COPY2_MODE;
                     XtAddEventHandler(areastruct.area, PointerMotionMask, False, drag,
                          NULL);
	          }
	          else {
                     eventmode = PRESS_MODE;
                     XtAddEventHandler(areastruct.area, Button1MotionMask, False,
		          drag, NULL);
	          }
                  catreturn();
               }

               /* If button2 was pressed, select and stay in the catalog. */
	       else {
		  objectselect(OBJECT);
	       }
	       break;
	    }
	 }
      }
   }
   /* If button3 was pressed, return without a selection. */

   else {
      if (areastruct.selects > 0) {
	 free(areastruct.selectlist);
         areastruct.selects = 0;
      }
      eventmode = NORMAL_MODE;
      catreturn();
   }
}

/*--------------------------------------*/
/* Change from one catalog to the other */
/*--------------------------------------*/

void changecat()
{
   if (objectdata == libtop[LIBRARY]) {
      if ((libtop[USERLIB])->parts > 0) {
	 objectdata = libtop[USERLIB];
      }
      else {
	Wprintf("User library is empty");
	return;
      }
   }
   else if (objectdata == libtop[USERLIB]) {
      if ((libtop[LIBRARY])->parts > 0) {
	 objectdata = libtop[LIBRARY];
      }
      else {
	Wprintf("Built-in library is empty");
	return;
      }
   }
   else return;

   setpage();
   refresh(NULL, NULL, NULL);
}

/*--------------------------------------*/
/* Begin catalog viewing mode		*/
/*--------------------------------------*/

XtCallbackProc startcatalog(w, libmod, nulldata)
  Widget w;
  void *libmod;
  caddr_t nulldata;
{
   /* Can't look at a library with no objects in it */
   /* But first try to switch to a library which contains something useful. */

   if ((int)libmod != FONTLIB && (int)libmod != PAGELIB) {
      if (libtop[USERLIB]->parts == 0) libmod = (void *)LIBRARY;
      if (libtop[LIBRARY]->parts == 0) libmod = (void *)USERLIB;
      if (libtop[(int)libmod]->parts == 0) {
         Wprintf("Both libraries are empty.");
         return;
      }

      /* If we're already in catalog mode. . . */

      if (eventmode == CATALOG_MODE) {
         if (objectdata == libtop[(int)libmod]) {
	    if (libmod == USERLIB)
	       Wprintf("Already in User Library!");
	    else
	       Wprintf("Already in Library!");
         }
         else changecat();
         return;
      }
      else if (eventmode != NORMAL_MODE) return;
   }
   else if (libtop[(int)libmod]->parts == 0) return;

   if ((int)libmod == FONTLIB) {
      XDefineCursor (dpy, win, CROSS);
      if (eventmode == TEXT2_MODE)
         eventmode = FONTCAT_MODE;
      else
         eventmode = FONTCAT2_MODE;
   }
   else if ((int)libmod == PAGELIB) {
      XDefineCursor (dpy, win, CROSS);
      eventmode = PAGECAT_MODE;
   }
   else
      eventmode = CATALOG_MODE;

   /* push previously edited object on push stack */

   if (pushes == 0)
      pushlist = (objectpair *) malloc(sizeof(objectpair));
   else
      pushlist = (objectpair *) realloc(pushlist,
         (pushes + 1) * sizeof(objectpair));
   (*(pushlist + pushes)).thisobject = objectdata;
   (*(pushlist + pushes)).thisinst = NULL;
   pushes++;

   /* set library as new object */

   objectdata = libtop[(int)libmod];
   setpage();
   if ((int)libmod == FONTLIB)
      zoomview(NULL, NULL, NULL);

   /* unselect all current selects */

   if (areastruct.selects > 0) {
      free(areastruct.selectlist);
      areastruct.selects = 0;
   }

   /* draw the new screen */

   newmatrix();
   refresh(NULL, NULL, NULL);
}

#undef _lowerleft
#undef _width
#undef _height

/*-------------------------------------------------------------------------*/
