/*-------------------------------------------------------------------------*/
/* menucalls.c --- callback routines from the menu buttons		   */
/*		   and associated routines				   */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <errno.h>
#include <limits.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>

#include "Xw/Xw.h"
#include "Xw/MenuBtn.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"
#include "menudefs.h"
#include "menus.h"
#include "menudep.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

extern char	 _STR2[250];
extern char	 _STR[150];          /* Generic multipurpose string */
extern Widget	 top;
extern Display   *dpy;
extern Window    win;
extern short	 popups;	     /* total number of popup windows */
extern Widget    menuwidgets[];
extern Globaldata xobjs;
extern Clientdata areastruct;
extern Pagedata   **pagelist;
extern int	  *appcolors;
extern int	  number_colors;
extern colorindex *colorlist;
extern ApplicationData appdata;
extern Cursor	  appcursors[NUM_CURSORS];

extern fontinfo *fonts;
extern short fontcount;
extern short eventmode, textpos, textend;
unsigned short *fontnumbers;
uchar nfontnumbers;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to xcircuit.c		  */
/*------------------------------------------------------------------------*/

extern void Wprintf(), W1printf();
extern void redrawtext(), undrawtext(), makecursors();
extern void loadfontfile();

extern float getpsscale();
extern int xc_alloccolor();

void setpmode(), setorient(), getgeneric();
void setdscale(), setpagelabel();
void renamepage(), setpagesize();
void setfloat(), setscalex(), setscaley();
void setgrid(), setfilename(), setlabel();

XtCallbackProc setcolor();

/*----------------------------------------------------------------*/
/* Generic routine for use by all other data handling routines 	  */
/*----------------------------------------------------------------*/

void getgeneric(saveptr, button, getfunction, dataptr)
  buttonsave *saveptr;
  Widget button;
  void  (*getfunction)();
  void	*dataptr;
{
   char buffer[200];
   Arg  wargs[1];

   saveptr->button = button;
   saveptr->buttoncall = getfunction;
   saveptr->dataptr = dataptr;

   XtSetArg(wargs[0], XtNforeground, &saveptr->foreground);
   XtGetValues(button, wargs, 1);
   XtSetArg(wargs[0], XtNforeground, OFFBUTTONCOLOR);
   XtSetValues(button, wargs, 1);
   XtRemoveAllCallbacks(button, XtNselect);
}


/*----------------------------------------------------------------*/
/* setgrid, getgridspace are for grid and snap spacing sizes;	  */
/* include routines to parse fractions				  */
/*----------------------------------------------------------------*/

void setgrid(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   float oscale, iscale = (float)pagelist[areastruct.page]->drawingscale.y /
        (float)pagelist[areastruct.page]->drawingscale.x;
   float fval;

   /* For now, assume that the value is in the current display style. */
   /* Might be nice in the future to make it accept any input. . .    */

   switch (pagelist[areastruct.page]->coordstyle) {
      case CM:
         oscale = pagelist[areastruct.page]->outscale * CMSCALE;
	 if (sscanf(_STR2, "%f", &fval) == 0) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	 }
	 else *dataptr = fval * IN_CM_CONVERT / (iscale * oscale);
	 break;
      case DEC_INCH: case FRAC_INCH: {
	 short parts;
	 char *sptr;
	 int f2, f3;

         oscale = pagelist[areastruct.page]->outscale * INCHSCALE;
	 for (sptr = _STR2; *sptr != '\0'; sptr++)
	    if (*sptr == '/') *sptr = ' ';
	 parts = sscanf(_STR2, "%f %d %d", &fval, &f2, &f3);
	 if ((parts == 0) || (parts != 1 && (fval != (float)((int)fval)))) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	    break;
	 }
	 if (parts == 2) fval /= (float)f2;
	 else if (parts == 3) fval += ((float)f2 / (float)f3);
	 *dataptr = fval * 72.0 / (iscale * oscale);
	 } break;
   }
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* Write a measurement value into string "buffer" dependant on    */
/* the current default units of measure (centimeters or inches).  */
/*----------------------------------------------------------------*/

void measurestr(value, buffer)
  float         value;
  char          *buffer;
{  
   float oscale, iscale = (float)pagelist[areastruct.page]->drawingscale.y /
        (float)pagelist[areastruct.page]->drawingscale.x;

   switch (pagelist[areastruct.page]->coordstyle) {
      case CM:
         oscale = pagelist[areastruct.page]->outscale * CMSCALE;
         sprintf(buffer, "%5.3f cm", value * iscale * oscale / IN_CM_CONVERT);
         break; 
      case DEC_INCH:
         oscale = pagelist[areastruct.page]->outscale * INCHSCALE;
         sprintf(buffer, "%5.3f in", value * iscale * oscale / 72.0);
         break;
      case FRAC_INCH:
         oscale = pagelist[areastruct.page]->outscale * INCHSCALE;
         fraccalc(((value * iscale * oscale) / 72.0), buffer);
         strcat(buffer, " in");
         break;
   }
}        

/*--------------------------------------------------------------*/
/* Generate popup dialog for snap space value input		*/
/*--------------------------------------------------------------*/

XtCallbackProc getsnapspace(button, clientdata, calldata)
  Widget 	button;
  caddr_t 	clientdata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *floatptr = &pagelist[areastruct.page]->snapspace;

   getgeneric(savebutton, button, getgridspace, (void *)floatptr);
   measurestr(*floatptr, buffer);
   popupprompt(button, "Enter value:", buffer, setgrid, savebutton, False);
}

/*--------------------------------------------------------------*/
/* Generate popup dialog for grid space value input		*/
/*--------------------------------------------------------------*/

XtCallbackProc getgridspace(button, clientdata, calldata)
  Widget 	button;
  caddr_t 	clientdata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *floatptr = &pagelist[areastruct.page]->gridspace;

   getgeneric(savebutton, button, getgridspace, (void *)floatptr);
   measurestr(*floatptr, buffer);
   popupprompt(button, "Enter value:", buffer, setgrid, savebutton, False);
}

/*----------------------------------------------------------------*/
/* Generic routine for setting a floating-point value (through a  */
/* (float *) pointer passed as the second parameter)		  */
/*----------------------------------------------------------------*/

void setfloat(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* set the global default line width.  The unit used internally   */
/* is twice the value passed through pointer "dataptr".		  */
/*----------------------------------------------------------------*/

void setwidth(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   if (sscanf(_STR2, "%f", dataptr) == 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
      return;
   }
   (*dataptr) *= 2.0;
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* Set text scale.						  */
/*----------------------------------------------------------------*/

void settsize(w, settext)
  Widget w;
  labelptr settext;
{
   float tmpres;
   short *osel;
   labelptr nstext;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {  /* can't interpret value or bad value */
      Wprintf("Illegal value");
      return;
   }

   /* In edit mode, change scale of text being edited only. */

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      if (strlen(settext->string) > 0) {
	 undrawtext(settext);
         settext->scale = tmpres;
	 redrawtext(settext);
      }
      else (settext->scale = tmpres);
   }

   /* Change scale on all selected text objects */

   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
	 if (SELECTTYPE(osel) == LABEL) {
	    nstext = SELTOLABEL(osel);
	    undrawtext(nstext);
            nstext->scale = tmpres;
	    redrawtext(nstext);
	 }
      }
      objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set the denomenator value of the drawing scale 		  */ 
/*----------------------------------------------------------------*/

void setscaley(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   };
   *dataptr = (*dataptr * 72) / objectdata->height;
   *dataptr /= getpsscale(1.0, areastruct.page);
}

/*----------------------------------------------------------------*/
/* Set the numerator value of the drawing scale			  */
/*----------------------------------------------------------------*/

void setscalex(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   *dataptr = (*dataptr * 72) / objectdata->width;
   *dataptr /= getpsscale(1.0, areastruct.page);
}

/*----------------------------------------------------------------*/
/* Set the page orientation (either Landscape or Portrait)	  */
/*----------------------------------------------------------------*/

void setorient(w, dataptr)
  Widget w;
  short *dataptr;
{
   Arg wargs[1];

   if (*dataptr == 0) {
      *dataptr = 90;
      XtSetArg(wargs[0], XtNlabel, "Landscape");
   }
   else {
      *dataptr = 0;
      XtSetArg(wargs[0], XtNlabel, "Portrait");
   }
   XtSetValues(w, wargs, 1);
}

/*----------------------------------------------------------------*/
/* Set the output mode to "Full Page" (unencapsulated PostScript) */
/* or "Embedded" (encapsulated PostScript)			  */
/*----------------------------------------------------------------*/

void setpmode(w, dataptr)
  Widget w;
  short *dataptr;
{
   Arg wargs[1];
   Widget pwidg = XtParent(w);

   if (*dataptr == 0) {
      *dataptr = 1;
      XtSetArg(wargs[0], XtNlabel, "Full Page");

      XtManageChild(XtNameToWidget(pwidg, "fpedit"));
      XtManageChild(XtNameToWidget(pwidg, "fpokay"));
   }
   else {
      *dataptr = 0;
      XtSetArg(wargs[0], XtNlabel, "Embedded");

      XtUnmanageChild(XtNameToWidget(pwidg, "fpedit"));
      XtUnmanageChild(XtNameToWidget(pwidg, "fpokay"));
   }
   XtSetValues(w, wargs, 1);
}

/*----------------------------------------------------------------*/
/* Set the output page size, in the current unit of measure	  */
/*----------------------------------------------------------------*/

void setpagesize(w, dataptr)
  Widget w;
  XPoint *dataptr;
{
   float px, py;
   char units[10], *expos, fpedit[75];
   Arg wargs[1];

   strcpy(units, "in");

   if (sscanf(_STR2, "%f %*c %f %s", &px, &py, units) < 4) {
      if (sscanf(_STR2, "%f %*c %f", &px, &py) < 3) {
	 if ((expos = strchr(_STR2, 'x')) == NULL) {
            Wprintf("Illegal Form for page size.");
	    return;
	 }
	 else {
	    *expos = '\0';
	    if (sscanf(_STR2, "%f", &px) == 0 ||
		  sscanf(expos + 1, "%f %s", &py, units) == 0) {
               Wprintf("Illegal Form for page size.");
	       return;
	    }
	 }
      }
   }

   dataptr->x = (short)(px * 72.0);
   dataptr->y = (short)(py * 72.0);

   if (!strcmp(units, "cm")) {
      sprintf(fpedit, "%3.2f x %3.2f cm", (float)pagelist[areastruct.page]->pagesize.x
	  / 72.0, (float)pagelist[areastruct.page]->pagesize.y / 72.0);
      dataptr->x /= 2.54;
      dataptr->y /= 2.54;
   }
   else {
      sprintf(fpedit, "%3.2f x %3.2f in", (float)pagelist[areastruct.page]->pagesize.x
	  / 72.0, (float)pagelist[areastruct.page]->pagesize.y / 72.0);
   }
   XtSetArg(wargs[0], XtNstring, fpedit);
   XtSetValues(XtNameToWidget(XtParent(w), "fpedit"), wargs, 1);
}

/*----------------------------------------------------------------*/
/* Generate popup dialog to get the drawing scale, specified as a */
/* whole-number ratio X:Y					  */
/*----------------------------------------------------------------*/

XtCallbackProc getdscale(button, nulldata, calldata)
  Widget	button;
  caddr_t	nulldata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   XPoint *ptptr = &(pagelist[areastruct.page]->drawingscale);

   getgeneric(savebutton, button, getdscale, ptptr);
   sprintf(buffer, "%d:%d", ptptr->x, ptptr->y);
   popupprompt(button, "Enter Scale:", buffer, setdscale, savebutton, False);
}

/*----------------------------------------------------------------*/
/* Set the drawing scale (specified as ratio X:Y)		  */
/*----------------------------------------------------------------*/

void setdscale(w, dataptr)
  Widget w;
  XPoint *dataptr;
{
   char *sptr;

   if ((sptr = strchr(_STR2, ':')) == NULL)
      Wprintf("Use ratio X:Y");
   else {
      *sptr = '\0';
      sscanf(_STR2, "%hd", &(dataptr->x));
      sscanf(sptr + 1, "%hd", &(dataptr->y));
      sprintf(_STR2, "New scale is %hd:%hd", dataptr->x, dataptr->y);
      Wprintf(_STR2);
      W1printf(" ");
   }
}

/*----------------------------------------------------------------*/
/* Generate the popup dialog for getting text scale.		  */ 
/*----------------------------------------------------------------*/

XtCallbackProc gettsize(button, nulldata, calldata)
  Widget        button;
  caddr_t       nulldata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   labelptr settext = NULL;
   float      *floatptr = &areastruct.textscale;
   short *osel;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      floatptr = &(settext->scale);
   }
   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
		areastruct.selects; osel++)
	 if (SELECTTYPE(osel) == LABEL) {
            settext = SELTOLABEL(osel);
            floatptr = &(settext->scale);
	    break;
	 }
   }
   sprintf(buffer, "%4.2f", *floatptr);
   if (settext != NULL) {
      getgeneric(savebutton, button, gettsize, settext);
      popupprompt(button, "Enter label scale:", buffer, settsize, savebutton, False);
   }
   else {
      getgeneric(savebutton, button, gettsize, floatptr);
      popupprompt(button,
	    "Enter default text scale:", buffer, setfloat, savebutton, False);
   }
}

/*----------------------------------------------------------------*/
/* Set the scale of an object or group of selected objects	  */
/*----------------------------------------------------------------*/

void setosize(w, dataptr)
  Widget w;
  objinstptr dataptr;
{
   float tmpres;
   short *osel;
   objinstptr nsobj;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {
      Wprintf("Illegal value");
      return;
   }
   for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
      if (SELECTTYPE(osel) == OBJECT) {
	 nsobj = SELTOOBJINST(osel);
         nsobj->scale = tmpres;
      }
   }
   objectdeselect();
   drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* Generate popup dialog for getting object scale		  */
/*----------------------------------------------------------------*/

XtCallbackProc getosize(button, nullptr, calldata)
  Widget        button;
  void		*nullptr;
  caddr_t       calldata;
{
   char buffer[50];
   float flval;
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   short selects = 0;
   objinstptr setobj = NULL;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++)
      if (SELECTTYPE(osel) == OBJECT) {
	 setobj = SELTOOBJINST(osel);
	 selects++;
	 break;
      }
   if (setobj == NULL) {
      Wprintf("No objects were selected for scaling.");
      return;
   }
   flval = setobj->scale;
   getgeneric(savebutton, button, getosize, setobj);
   sprintf(buffer, "%4.2f", flval);
   popupprompt(button, "Enter object scale:", buffer, setosize, savebutton, False);
}

/*----------------------------------------------------------------*/
/* Generate popup prompt for getting global linewidth		  */
/*----------------------------------------------------------------*/

XtCallbackProc getwirewidth(button, clientdata, calldata)
  Widget	button;
  caddr_t	clientdata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *widthptr;

   widthptr = &(pagelist[areastruct.page]->wirewidth);
   getgeneric(savebutton, button, getwirewidth, widthptr);
   sprintf(buffer, "%4.2f", *widthptr / 2.0);
   popupprompt(button, "Enter new global linewidth:", buffer, setwidth,
	savebutton, False);
}

/*----------------------------------------------------------------*/
/* Set the linewidth of all selected arcs, polygons, splines, and */
/* paths.							  */
/*----------------------------------------------------------------*/

void setwwidth(w, dataptr)
  Widget w;
  void *dataptr;
{
   float     tmpres;
   short     *osel;
   arcptr    nsarc;
   polyptr   nspoly;
   splineptr nsspline;
   pathptr   nspath;

   if (sscanf(_STR2, "%f", &tmpres) == 0) {
      Wprintf("Illegal value");
      return;
   }
   for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
      if (SELECTTYPE(osel) == ARC) {
	 nsarc = SELTOARC(osel);
         nsarc->width = tmpres;
      }
      else if (SELECTTYPE(osel) == POLYGON) {
	 nspoly = SELTOPOLY(osel);
         nspoly->width = tmpres;
      }
      else if (SELECTTYPE(osel) == SPLINE) {
	 nsspline = SELTOSPLINE(osel);
         nsspline->width = tmpres;
      }
      else if (SELECTTYPE(osel) == PATH) {
	 nspath = SELTOPATH(osel);
         nspath->width = tmpres;
      }
   }
   objectdeselect();
   drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* Generate popup dialong for getting linewidths of elements	  */ 
/*----------------------------------------------------------------*/

XtCallbackProc getwwidth(button, nullptr, calldata)
  Widget        button;
  void		*nullptr;
  caddr_t       calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   genericptr setel;
   float flval;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++) {
      setel = *(objectdata->plist + (*osel));
      if (setel->type == ARC) {
	 flval = ((arcptr)setel)->width;
	 break;
      }
      else if (setel->type == POLYGON) {
	 flval = ((polyptr)setel)->width;
	 break;
      }
      else if (setel->type == SPLINE) {
	 flval = ((splineptr)setel)->width;
	 break;
      }
      else if (setel->type == PATH) {
	 flval = ((pathptr)setel)->width;
	 break;
      }
   }
   if (osel == areastruct.selectlist + areastruct.selects) {
      Wprintf("No arcs, polygons, splines, or paths were selected.");
      return;
   }
   getgeneric(savebutton, button, getwwidth, setel);
   sprintf(buffer, "%4.2f", flval);
   popupprompt(button, "Enter new line width:", buffer, setwwidth, savebutton, False);
}

/*----------------------------------------------------------------*/
/* Generic popup prompt for getting a floating-point value	  */
/*----------------------------------------------------------------*/

XtCallbackProc getfloat(button, floatptr, calldata)
  Widget	button;
  float		*floatptr;
  caddr_t	calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, button, getfloat, floatptr);
   sprintf(buffer, "%4.2f", *floatptr);
   popupprompt(button, "Enter value:", buffer, setfloat, savebutton, False);
}

/*----------------------------------------------------------------*/
/* Set the filename for the current page			  */
/*----------------------------------------------------------------*/

void setfilename(w, dataptr)
  Widget w;
  char	**dataptr;
{
   short cpage, depend = 0;
   objectptr checkpage;

   if (!strcmp(*dataptr, _STR2)) return;   /* no change in string */

   /* Check for dependencies on this string from other pages.        */
   /* If there are no other pointers to this string, we can free it. */

   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = pagelist[cpage]->pageobj;
      if ((checkpage != NULL) && (cpage != areastruct.page))  
	 if (pagelist[cpage]->filename == pagelist[areastruct.page]->filename)  
	    depend = 1;
   }

   /* check if the new filename is an existing filename */

   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = pagelist[cpage]->pageobj;
      if ((checkpage != NULL) && (cpage != areastruct.page)) {
	 if (!strcmp(_STR2, pagelist[cpage]->filename)) {
	    if (!depend) free(pagelist[areastruct.page]->filename);
	    pagelist[areastruct.page]->filename = pagelist[cpage]->filename;
	    break;
	 }
      }
   }
   if (cpage == xobjs.pages) {
      if (!depend) free(pagelist[areastruct.page]->filename);
      pagelist[areastruct.page]->filename = (char *) malloc((strlen(_STR2)
		+ 1) * sizeof(char));
      strcpy(pagelist[areastruct.page]->filename, _STR2);
   }
}

/*----------------------------------------------------------------*/
/* Set the page label for the current page			  */
/*----------------------------------------------------------------*/

void setpagelabel(w, dataptr)
  Widget w;
  char	*dataptr;
{
   short cpage, i;
   objectptr checkpage;

   /* whitespace and non-printing characters not allowed */

   for (i = 0; i < strlen(_STR2); i++) {
      if ((!isprint(_STR2[i])) || (isspace(_STR2[i]))) {
         _STR2[i] = '_';
         Wprintf("Replaced illegal whitespace in name with underscore");
      }
   }
   if (!strcmp(dataptr, _STR2)) return; /* no change in string */
   if (strlen(_STR2) == 0)
      sprintf(objectdata->name, "Page %d", areastruct.page + 1);
   else
      sprintf(objectdata->name, "%.79s", _STR2);
   printname(objectdata);
   renamepage(areastruct.page);
}

/*----------------------------------------------------------------*/
/* Set the string for all selected labels			  */
/*----------------------------------------------------------------*/

void setlabel(w, dataptr)
  Widget w;
  char  *dataptr;
{
   char oldvalue[150];
   labelptr *whatlabel;

   strcpy(oldvalue, dataptr);
   for (whatlabel = (labelptr *)objectdata->plist; whatlabel <
	(labelptr *)objectdata->plist + objectdata->parts; whatlabel++)
      if ((*whatlabel)->type == LABEL)
         if ((*whatlabel)->string == (uchar *)dataptr) break;
   dataptr = (char *) realloc(dataptr, (strlen(_STR2) + 1) * sizeof(char));
   (*whatlabel)->string = (uchar *)dataptr;
   strcpy(dataptr, _STR2);
   if (strcmp(oldvalue, dataptr)) drawarea(NULL, objectdata, dataptr);
}

/*----------------------------------------------------------------*/
/* Add a new font name to the list of known fonts		  */
/*----------------------------------------------------------------*/

void makenewfontbutton()
{
   Arg	wargs[1];
   int  n = 0;
   Widget newbutton, cascade;

   if (fontcount == 0) return;

   cascade = XtParent(FontAddNewFontButton);
   XtnSetArg(XtNfont, appdata.xcfont);
   newbutton = XtCreateWidget(fonts[fontcount - 1].family, XwmenubuttonWidgetClass,
	 cascade, wargs, n);

   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)setfont,
		Number(fontcount - 1));
   XtManageChild(newbutton);

   nfontnumbers++;
   if (nfontnumbers == 1)
      fontnumbers = (unsigned short *)malloc(sizeof(unsigned short)); 
   else
      fontnumbers = (unsigned short *)realloc(fontnumbers, nfontnumbers
		* sizeof(unsigned short));
   fontnumbers[nfontnumbers - 1] = fontcount - 1;
}

/*--------------------------------------------------------------*/
/* Make new encoding menu button				*/
/*--------------------------------------------------------------*/

void makenewencodingbutton(char *ename, char value)
{
   Arg	wargs[1];
   int  n = 0;
   Widget newbutton, cascade;

   cascade = XtParent(EncodingStandardButton);

   /* return if button has already been made */
   newbutton = XtNameToWidget(cascade, ename);
   if (newbutton != NULL) return;

   XtnSetArg(XtNfont, appdata.xcfont);
   newbutton = XtCreateWidget(ename, XwmenubuttonWidgetClass,
	 cascade, wargs, n);

   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)setfontencoding,
		Number(value));
   XtManageChild(newbutton);
}

/*--------------------------------------------------------------------*/
/* Toggle one of a set of menu items, only one of which can be active */
/*--------------------------------------------------------------------*/

void toggleexcl(widget, menu, menulength)
  Widget	widget;
  menuptr	menu;
  int		menulength;
{
   Arg          args[1];
   Widget       parent = XtParent(widget);
   Widget       sibling;
   menuptr	mitem, mtest;
   short	i;

   /* find the menu item which corresponds to the widget which was pushed */

   for (mtest = menu; mtest < menu + menulength; mtest++) {
      sibling = XtNameToWidget(parent, mtest->name);
      if (sibling == widget) break;
   }

   /* remove checkmark from other widgets in the list */

   if (menu == Fonts) {     		/* special action for font list */
      for (i = 0; i < fontcount; i++) {
	 sibling = XtNameToWidget(parent, fonts[i].family);
	 if (sibling != widget) {
	    XtSetArg(args[0], XtNsetMark, False);
	    XtSetValues(sibling, args, 1);
	 }
      }
      mtest = &Fonts[3];   /* so that mtest->func has correct value below */
   }
   else if (mtest == menu + menulength) return;  /* something went wrong? */

   for (mitem = menu; mitem < menu + menulength; mitem++) {
      sibling = XtNameToWidget(parent, mitem->name);
      if (mitem->func == mtest->func) {
         XtSetArg(args[0], XtNsetMark, False);
         XtSetValues(sibling, args, 1);
      }
   }

   /* Now set the currently pushed widget */

   XtSetArg(args[0], XtNsetMark, True);
   XtSetValues(widget, args, 1);
}

/*--------------------*/
/* Toggle a menu item */
/*--------------------*/

XtCallbackProc toggle(w, boolvalue, calldata)
  Widget w;
  Boolean	*boolvalue;
  caddr_t	calldata;
{
   Arg	wargs[1];

   *boolvalue = !(*boolvalue);
   XtSetArg(wargs[0], XtNsetMark, *boolvalue);
   XtSetValues(w, wargs, 1);
   drawarea(w, objectdata, boolvalue);
}

/*---------------------------------------------------------------*/
/* Some Xcircuit routines using toggle and toggleexcl, 		 */
/* put here because they reference the menu structures directly. */
/*---------------------------------------------------------------*/

void setcolorscheme(boolvalue)
  Boolean	boolvalue;
{
   if (boolvalue) {
      BBOXCOLOR = appdata.bboxpix;
      AUXCOLOR = appdata.auxpix;
      BARCOLOR = appdata.barpix;
      OFFBUTTONCOLOR = appdata.buttonpix;
      SELECTCOLOR = appdata.selectpix;
      QUERYCOLOR = appdata.querypix;
      GRIDCOLOR = appdata.gridpix;
      SNAPCOLOR = appdata.snappix;
      AXESCOLOR = appdata.axespix;
      BACKGROUND = appdata.bg;
      FOREGROUND = appdata.fg;
   }
   else {
      BBOXCOLOR = appdata.bboxpix2;
      AUXCOLOR = appdata.auxpix2;
      BARCOLOR = appdata.barpix2;
      OFFBUTTONCOLOR = appdata.buttonpix2;
      SELECTCOLOR = appdata.selectpix2;
      QUERYCOLOR = appdata.querypix2;
      GRIDCOLOR = appdata.gridpix2;
      SNAPCOLOR = appdata.snappix2;
      AXESCOLOR = appdata.axespix2;
      BACKGROUND = appdata.bg2;
      FOREGROUND = appdata.fg2;
   }

   makecursors();
}

/*----------------------------------------------------------------*/
/* Invert the color scheme used for the background/foreground	  */
/*----------------------------------------------------------------*/

XtCallbackProc inversecolor(w, boolvalue, calldata)
  Widget w;
  Boolean	*boolvalue;
  caddr_t	calldata;
{
   /* change color scheme */

   setcolorscheme(*boolvalue);

   /* toggle checkmark in menu on "Alt colors" */

   if (w == NULL) w = OptionsAltColorsButton;
   if (w != NULL) toggle(w, boolvalue, calldata);
   if (eventmode == NORMAL_MODE) XDefineCursor (dpy, win, CROSS);
}

/*----------------------------------------------------------------*/
/* Change menu selection for reported measurement units		  */
/*----------------------------------------------------------------*/

void togglegrid(type)
  unsigned short type;
{
   Widget button, bparent = XtParent(GridtypedisplayDecimalInchesButton);

   if (type == CM) button = XtNameToWidget(bparent, "Centimeters");
   else if (type == FRAC_INCH) button = XtNameToWidget(bparent, "Fractional Inches");
   else if (type == DEC_INCH) button = XtNameToWidget(bparent, "Decimal Inches");
   else button = XtNameToWidget(bparent, "Coordinates");
   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   W1printf(" ");
}

/*----------------------------------------------------------------*/
/* Set the default reported grid units to inches or centimeters   */
/*----------------------------------------------------------------*/

void setgridtype(string)
   char *string;
{
   Widget button, bparent = XtParent(GridtypedisplayDecimalInchesButton);

   if (!strcmp(string, "inchscale")) {
      button = XtNameToWidget(bparent, "Fractional Inches");
      getgridtype(button, FRAC_INCH, NULL);
   }
   else if (!strcmp(string, "cmscale")) {
      button = XtNameToWidget(bparent, "Centimeters");
      getgridtype(button, CM, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Called by setgridtype() to complete setting the reported 	  */
/* measurement units						  */
/*----------------------------------------------------------------*/

XtCallbackProc getgridtype(button, value, calldata)
  Widget button;
  unsigned short value;
  caddr_t calldata;
{
   short oldtype = pagelist[areastruct.page]->coordstyle;
   float scalefac = getpsscale(1.0, areastruct.page) / INCHSCALE;

   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   pagelist[areastruct.page]->coordstyle = (short)value;

   switch(value) {
      case FRAC_INCH: case DEC_INCH:
	 if (oldtype == CM) {
            pagelist[areastruct.page]->outscale *= scalefac;
	    pagelist[areastruct.page]->pagesize.x = 612;
	    pagelist[areastruct.page]->pagesize.y = 792; /* letter */
	 }
	 break;
      case CM:
	 if (oldtype != CM) {
            pagelist[areastruct.page]->outscale *= scalefac;
	    pagelist[areastruct.page]->pagesize.x = 595;
	    pagelist[areastruct.page]->pagesize.y = 842; /* A4 */
	 }
	 break;
   }
   if (oldtype != pagelist[areastruct.page]->coordstyle) {
      drawarea(NULL, objectdata, NULL);
      W1printf(" ");
   }
}

/*----------------------------------------------*/
/* Make new page; goto that page		*/
/* (wrapper for routine events.c:newpage())	*/
/*----------------------------------------------*/

XtCallbackProc newpagemenu(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   if (eventmode == CATALOG_MODE) {
      eventmode = NORMAL_MODE;
      catreturn();
   }
   newpage((short)value);
}

/*--------------------------------------------------------------*/
/* Routine called by newpage() if new button needs to be made 	*/
/* to add to the "Pages" cascaded menu.				*/
/*--------------------------------------------------------------*/

void makepagebutton()
{
   Widget pagemenu, newbutton;
   Arg wargs[1];
   char pagestring[10];

   /* make new entry in the menu */

   pagemenu = XtParent(GotoPageAddNewPageButton);
   XtSetArg(wargs[0], XtNfont, appdata.xcfont);
   sprintf(pagestring, "Page %d", xobjs.pages);
   newbutton = XtCreateWidget(pagestring, XwmenubuttonWidgetClass,
         pagemenu, wargs, 1);
   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)newpagemenu,
	Number(xobjs.pages - 1));
   XtManageChild(newbutton);
}

/*----------------------------------------------------------------*/
/* Find the Page menu button associated with the page number	  */
/* (passed parameter) and set the label of that button to the	  */
/* object name (= page label)					  */
/*----------------------------------------------------------------*/

void renamepage(pagenumber)
   short pagenumber;
{
   Arg wargs[1];
   Widget parent = XtParent(GotoPageAddNewPageButton);
   Widget button;
   char bname[10];

   sprintf(bname, "Page %d", pagenumber + 1);
   button = XtNameToWidget(parent, bname);

   if (button != NULL) {
      if (pagelist[pagenumber]->pageobj->name != NULL)
         XtSetArg(wargs[0], XtNlabel, pagelist[pagenumber]->pageobj->name);
      else
         XtSetArg(wargs[0], XtNlabel, bname);
      XtSetValues(button, wargs, 1);
   }
   else
      printf("Error:  No Button Widget named \"%s\"\n", bname);
}

/*----------------------------------------------*/
/* Set Poly and Arc line styles and fill styles */
/*----------------------------------------------*/

#define BORDERS  (NOBORDER | DOTTED | DASHED)
#define ALLFILLS (FILLSOLID | FILLED)

/*----------------------------------------------*/
/* Set the menu marks according to style value  */
/*----------------------------------------------*/

void setstylemarks(w, value, boolval)
  Widget w;
  short value;
  Boolean boolval;
{
   Arg	wargs[1];

   /* if widget is unknown, find button for each part */

   if (value & UNCLOSED) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & OPAQUE) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & BORDERS || value & ALLFILLS || value == 0) {
      toggleexcl(w, BorderStyles, XtNumber(BorderStyles));
      toggleexcl(w, Stipples, XtNumber(Stipples));
   }
}

/*----------------------------------------------------------------*/
/* Set the menu checkmarks on the color menu			  */
/*----------------------------------------------------------------*/

void setcolormark(colorval)
  short colorval;
{
   Widget w = NULL;
   short i;


   for (i = 0; i < number_colors; i++)
      if (colorlist[i].color.pixel == colorval) {
         w = colorlist[i].cbutton;
	 break;
      }

   if (colorval == DEFAULTCOLOR)
      w = ColorInheritColorButton;

   if (w != NULL) toggleexcl(w, Colors, XtNumber(Colors));
}

/*----------------------------------------------------------------*/
/* Set the checkmarks on the element styles menu		  */
/*----------------------------------------------------------------*/

void setallstylemarks(styleval)
  short styleval;
{
   Widget w;

   w = BorderClosedButton;
   setstylemarks(w, UNCLOSED, !(styleval & UNCLOSED));
   w = FillOpaqueButton;
   setstylemarks(w, OPAQUE, (styleval & OPAQUE) > 0);

   if (styleval & NOBORDER)
      w = BorderUnborderedButton;
   else if (styleval & DASHED)
      w = BorderDashedButton;
   else if (styleval & DOTTED)
      w = BorderDottedButton;
   else
      w = BorderSolidButton;
   setstylemarks(w, BORDERS, 0);

   if (!(styleval & FILLED))
      w = FillWhiteButton;
   else {
      styleval &= FILLSOLID;
      styleval /= STIP0;
      switch(styleval) {
	 case 0: w = FillGray87Button; break;
	 case 1: w = FillGray75Button; break;
	 case 2: w = FillGray62Button; break;
	 case 3: w = FillGray50Button; break;
	 case 4: w = FillGray37Button; break;
	 case 5: w = FillGray25Button; break;
	 case 6: w = FillGray12Button; break;
	 case 7: w = FillBlackButton;  break;
      }
   }
   setstylemarks(w, ALLFILLS, 0);
}

/*----------------------------------------------------------------*/
/* Get a value for element fill style				  */
/*----------------------------------------------------------------*/

XtCallbackProc getfill(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   Boolean boolval, selected = False;
   short *sstyle;

   if (areastruct.selects > 0) 
      for (sstyle = areastruct.selectlist; sstyle < areastruct.selectlist
	   + areastruct.selects; sstyle++) {
	 short stype = SELECTTYPE(sstyle);
	 if (stype == ARC || stype == POLYGON || stype == SPLINE || stype == PATH) {
	    short *estyle;
	    switch (stype) {
	       case ARC:
	          estyle = &((SELTOARC(sstyle))->style);
		  break;
	       case SPLINE:
	          estyle = &((SELTOSPLINE(sstyle))->style);
		  break;
	       case POLYGON:
	          estyle = &((SELTOPOLY(sstyle))->style);
		  break;
	       case PATH:
	          estyle = &((SELTOPATH(sstyle))->style);
		  break;
	    }

	    XSetFunction(dpy, areastruct.gc, GXcopy);
	    XSetForeground(dpy, areastruct.gc, BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    if (value & UNCLOSED) {
	       boolval = (*estyle & UNCLOSED) ? True : False;
	       *estyle &= ~UNCLOSED;
	       *estyle |= (boolval ? NORMAL : UNCLOSED);
	    }
	    else if (value & OPAQUE) {
	       boolval = (*estyle & OPAQUE) ? True : False;
	       *estyle &= ~OPAQUE;
	       *estyle |= (boolval ? NORMAL : OPAQUE);
	    }
	    else {
	       if (value & BORDERS || value == NORMAL) *estyle &= ~BORDERS;
	       else *estyle &= ~ALLFILLS;
	       if (value != FILLSOLID) *estyle |= value;
	       if ((*estyle & NOBORDER) && !(*estyle & FILLED)) {
		  Wprintf("Must have either a border or filler");
		  *estyle &= ~NOBORDER;
		  value &= ~NOBORDER;
	       }
	    }
	    XSetFunction(dpy, areastruct.gc, GXxor);
	    XSetForeground(dpy, areastruct.gc, SELECTCOLOR ^ BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    selected = True;
	 }
      }
   if (!selected) {
      if (value & UNCLOSED) {
         boolval = (areastruct.style & UNCLOSED) ? True : False;
	 areastruct.style &= ~UNCLOSED;
	 areastruct.style |= (boolval ? NORMAL : UNCLOSED);
      }
      else if (value & OPAQUE) {
         boolval = (areastruct.style & OPAQUE) ? True : False;
	 areastruct.style &= ~OPAQUE;
	 areastruct.style |= (boolval ? NORMAL : OPAQUE);
      }
      else {
	 short savestyle = areastruct.style;

         if (value & BORDERS || value == NORMAL)
	    areastruct.style &= ~BORDERS;
         else areastruct.style &= ~ALLFILLS;
         if (value != FILLSOLID) areastruct.style |= value;

	 if ((areastruct.style & NOBORDER) && !(areastruct.style & FILLED)) {
	    Wprintf("Must have either a border or filler");
	    areastruct.style = savestyle;
	    return;
	 }
      }
   }
   if (w)
      setstylemarks(w, (short)value, boolval);
   else
      setallstylemarks(areastruct.style);

   objectdeselect();
}

/*----------------------------------------------------------------*/
/* The following three routines are all wrappers for getfill(),	  */
/* used in menudefs.h to differentiate between sections, each of  */
/* which has settings independent of the others.		  */
/*----------------------------------------------------------------*/

XtCallbackProc getset(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}

/*----------------------------------------------------------------*/

XtCallbackProc getline(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}

/*----------------------------------------------------------------*/

XtCallbackProc getopaque(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}
   
/*-----------------------------------------------*/
/* Set the color value for all selected elements */
/*-----------------------------------------------*/

XtCallbackProc setcolor(w, value, nulldata)
  Widget        w; 
  unsigned short value;
  caddr_t       nulldata;
{
   short *scolor;
   int *ecolor, cindex;
   Arg wargs[1];
   Boolean selected = False;

   /* Get the color index value from the widget itself */

   if (value == 1)
      cindex = -1;
   else {
      XtSetArg(wargs[0], XtNrectColor, &cindex);
      XtGetValues(w, wargs, 1);
   }
   

   if (areastruct.selects > 0)
      for (scolor = areastruct.selectlist; scolor < areastruct.selectlist
	   + areastruct.selects; scolor++) {
	 ecolor = &(SELTOCOLOR(scolor));

	 *ecolor = cindex;
	 selected = True;
      }

   if (!selected) {
      areastruct.color = cindex;      
   }

   toggleexcl(w, Colors, XtNumber(Colors));
   objectdeselect();
}

/*----------------------------------------------------------------*/
/* Parse a new color entry and add it to the color list.	  */
/*----------------------------------------------------------------*/

void setnewcolor(w, nullptr)
  Widget  w;
  caddr_t *nullptr;
{
   int ccolor, red, green, blue;
   char *ppos;

   if ((ppos = strchr(_STR2, '#')) != 0) {
      if (strlen(ppos + 1) == 6) {
	 sscanf(ppos + 1, "%2x%2x%2x", &red, &green, &blue);
	 red *= 256;
	 green *= 256;
	 blue *= 256;
      }
      else if (sscanf(ppos + 1, "%4x%4x%4x", &red, &green, &blue) != 3) {
	 Wprintf("Bad color entry.  Use #rrggbb");
	 return;
      }
      ccolor = rgb_alloccolor(red, green, blue);
   }
   else
      ccolor = xc_alloccolor(_STR2);

   addnewcolorentry(ccolor);
}

/*----------------------------------------------------------------*/
/* Generate popup dialog for adding a new color name or RGB value */
/*----------------------------------------------------------------*/

XtCallbackProc addnewcolor(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, w, addnewcolor, NULL);
   popupprompt(w, "Enter color name or #rgb:", "\0", setnewcolor,
	savebutton, False);
}

/*------------------------------------*/
/* Set checkmarks on label style menu */
/*------------------------------------*/

void setfontmarks(fvalue, jvalue)
  short fvalue, jvalue;
{
   Widget w, fp;
   Arg wargs[1];

   if ((fvalue >= 0) && (fontcount > 0)) {
      switch(fonts[fvalue].flags & 0x03) {
         case 0: w = StyleNormalButton; break;
         case 1: w = StyleBoldButton; break;
         case 2: w = StyleItalicButton; break;
         case 3: w = StyleBoldItalicButton; break;
      }
      toggleexcl(w, FontStyles, XtNumber(FontStyles));

      switch((fonts[fvalue].flags & 0xf80) >> 7) {
         case 0: w = EncodingStandardButton; break;
         case 2: w = EncodingISOLatin1Button; break;
	 default: w = NULL;
      }
      if (w != NULL) toggleexcl(w, FontEncodings, XtNumber(FontEncodings));

      fp = XtParent(FontAddNewFontButton);
      w = XtNameToWidget(fp, fonts[fvalue].family);
      toggleexcl(w, Fonts, XtNumber(Fonts));
   }
   if (jvalue >= 0) {
      switch(jvalue & (NOTLEFT | RIGHT)) {
         case NORMAL: w = JustificationLeftJustifiedButton; break;
         case NOTLEFT: w = JustificationCenterJustifiedButton; break;
         case RIGHT|NOTLEFT: w = JustificationRightJustifiedButton; break;
      }
      toggleexcl(w, Justifs, XtNumber(Justifs));

      switch(jvalue & (NOTBOTTOM | TOP)) {
         case NORMAL: w = JustificationBottomJustifiedButton; break;
         case NOTBOTTOM: w = JustificationMiddleJustifiedButton; break;
         case TOP|NOTBOTTOM: w = JustificationTopJustifiedButton; break;
      }

      toggleexcl(w, Justifs, XtNumber(Justifs));

      w = JustificationFlipInvariantButton;
      if (jvalue & FLIPINV)
         XtSetArg(wargs[0], XtNsetMark, True);
      else
         XtSetArg(wargs[0], XtNsetMark, False);
      XtSetValues(w, wargs, 1);
   }
}

/*--------------------------------------------------------------*/
/* Parameterize a label string (wrapper for parameterize()).	*/
/*--------------------------------------------------------------*/

XtCallbackProc startparam(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   labelptr settext;
   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      makeparam(settext);
      objectdeselect();
   }
   else if (checkselect(LABEL)) parameterize();
   else
      if (eventmode == NORMAL_MODE) {
         eventmode = PARAM_MODE;
         Wprintf("Click on label to parameterize.");
      }      
}

/*--------------------------------------------------------------*/
/* Unparameterize a label string (wrapper for parameterize()).	*/
/*--------------------------------------------------------------*/

XtCallbackProc startunparam(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   if (checkselect(LABEL)) unparameterize();
   else Wprintf("Select string or part of substring to remove parameters.");
   objectdeselect();
}

/*----------------------------------------------------------------*/
/* Set checkmarks on font menu according to the global defaults	  */
/*----------------------------------------------------------------*/

void setdefaultfontmarks()
{
   setfontmarks(areastruct.psfont, areastruct.justify);
}

/*----------------------------------------------------------------*/
/* Change flip invariance on the label passed as 2nd parameter	  */
/*----------------------------------------------------------------*/

void doflipflip(w, settext)
  Widget        w;
  labelptr settext;
{
   short newjust;
   Bool boolval;
   unsigned short value;

   if (settext != NULL) {
      boolval = (settext->justify & FLIPINV) ? True : False;
      value = boolval ? 0 : FLIPINV;
      newjust = (settext->justify & (~FLIPINV)) | value;

      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
   }
   else {
      boolval = (areastruct.justify & FLIPINV) ? True : False;
      value = boolval ? 0 : FLIPINV;
      newjust = (areastruct.justify & (~FLIPINV)) | value;
      areastruct.justify = newjust;
   }
   if (w != NULL) toggle(w, &boolval, NULL);
}

/*----------------------------------------------------------------*/
/* Set the justification for the label passed as 3rd parameter	  */
/*----------------------------------------------------------------*/

void setjust(w, value, settext, mode)
  Widget        w;
  unsigned short value;
  labelptr settext;
  short mode;
{
   short newjust;

   if (settext != NULL) {

      if (mode == 1)
         newjust = (settext->justify & (FLIPINV | TOP | NOTBOTTOM)) | value;
      else
         newjust = (settext->justify & (FLIPINV | RIGHT | NOTLEFT)) | value;

      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
   }
   else {
      if (mode == 1)
         newjust = (areastruct.justify & (FLIPINV | TOP | NOTBOTTOM)) | value;
      else
         newjust = (areastruct.justify & (FLIPINV | RIGHT | NOTLEFT)) | value;
      areastruct.justify = newjust;
   }
   if (w != NULL) toggleexcl(w, Justifs, XtNumber(Justifs));
}

/*----------------------------------------------------------------*/
/* Set vertical justification (top, middle, bottom) on all	  */
/* selected labels						  */
/*----------------------------------------------------------------*/

XtCallbackProc setvjust(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setjust(w, value, settext, 2);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 2);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 2);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set horizontal justification (left, center, right) on all	  */
/* selected labels						  */
/*----------------------------------------------------------------*/

XtCallbackProc sethjust(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setjust(w, value, settext, 1);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 1);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 1);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set flip invariance on all selected labels			  */ 
/*----------------------------------------------------------------*/

XtCallbackProc setflipinv(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      doflipflip(w, settext);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            doflipflip(NULL, settext);
         }
      }
      if (labelcount == 0) doflipflip(w, NULL);
      else objectdeselect();
   }
}

/*---------------------------*/
/* Set polygon editing style */
/*---------------------------*/

XtCallbackProc boxedit(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   if (w == NULL) {
      switch (value) {
         case MANHATTAN: w = PolygonEditManhattanBoxEditButton; break;
         case RHOMBOIDX: w = PolygonEditRhomboidXButton; break;
	 case RHOMBOIDY: w = PolygonEditRhomboidYButton; break;
	 case RHOMBOIDA: w = PolygonEditRhomboidAButton; break;
	 case NORMAL: w = PolygonEditNormalButton; break;
      }
   }

   if (areastruct.boxedit == value) return;

   toggleexcl(w, BoxEditStyles, XtNumber(BoxEditStyles));
   areastruct.boxedit = value;
}

/*----------------------------------------------------------------*/
/* Pick up font name from _STR2 and pass it to loadfontfile()	  */
/*----------------------------------------------------------------*/

void locloadfont(w, value)
  Widget w;
  char *value;
{
   loadfontfile(_STR2);
   free(value);
}

/*----------------------------------------------------*/
/* Generate popup dialog for entering a new font name */
/*----------------------------------------------------*/

XtCallbackProc addnewfont(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   if (fontcount == (255 - FONT_START))
      Wprintf("Cannot add any more fonts.  Sorry.");
   else {
      char *tempstr = malloc(2 * sizeof(char));
      tempstr[0] = '\0';
      getgeneric(savebutton, w, addnewfont, tempstr);
      popupprompt(w, "Enter font name:", tempstr, locloadfont, savebutton, False);
   }
}

/*----------------------------------------------------------------*/
/* Find the best matching font given new font, style, or encoding */
/*----------------------------------------------------------------*/
/* newfont, style, or encoding = -1 when not applicable		  */
/* Return the number of the best matching font.			  */  
/*----------------------------------------------------------------*/

short findbestfont(short curfont, short newfont, short style, short encoding) {

   char *newfamily;
   short i, newstyle, newenc;

   if (fontcount == 0) return -1;

   if (newfont < 0)
      newfamily = fonts[curfont].family;
   else if (newfont >= fontcount) {	/* move to next font family */
      short newidx;
      newfont = 0;
      while (strcmp(fonts[fontnumbers[newfont]].family, fonts[curfont].family))
	 newfont++;
      newidx = (newfont + 1) % nfontnumbers;
      while (!strcmp(fonts[curfont].family, fonts[fontnumbers[newidx]].family) &&
		newfont != newidx)
	 newidx = (newidx + 1) % nfontnumbers;
      newfamily = fonts[fontnumbers[newidx]].family;
      newfont = fontnumbers[newidx];
   }
   else
      newfamily = fonts[newfont].family;

   if (style < 0) 
      newstyle = fonts[curfont].flags & 0x03;
   else
      newstyle = style & 0x03;
 
   if (encoding < 0)
      newenc = fonts[curfont].flags & 0xf80;
   else
      newenc = encoding << 7;

   /* Best position is a match on all conditions */

   for (i = 0; i < fontcount; i++)
      if ((!strcmp(fonts[i].family, newfamily)) &&
		((fonts[i].flags & 0x03) == newstyle) &&
	 	((fonts[i].flags & 0xf80) == newenc))
	 return i;

   /* Fallback position 1:  Match requested property and one other. 	*/
   /* order of preference:  					   	*/
   /*   Requested property	Priority 1	Priority 2		*/
   /*		Font		  Style		  Encoding		*/
   /*		Style		  Font		  (none)		*/
   /*		Encoding	  Font		  (none)		*/

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if ((!strcmp(fonts[i].family, newfamily)) &&
		(fonts[i].flags & 0x03) == newstyle) return i;
      }
      else if (style >= 0) {
	 if (((fonts[i].flags & 0x03) == newstyle) &&
	 	(!strcmp(fonts[i].family, newfamily))) return i;
      }
      else if (encoding >= 0) {
	 if ((fonts[i].flags & 0xf80 == newenc) &&
	 	(!strcmp(fonts[i].family, newfamily))) return i;
      }
   }

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if ((!strcmp(fonts[i].family, newfamily)) &&
	    ((fonts[i].flags & 0xf80) >> 7) == newenc) return i;
      }
   }

   /* Fallback position 2:  Match only the requested property.  */
   /* For font selection only:  Don't want to select a new font */
   /* just because a certain style or encoding wasn't available.*/

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if (!strcmp(fonts[i].family, newfamily)) return i;
      }
   }      

   /* Failure to find matching font property */

   if (style >= 0) {
      sprintf(_STR, "Font %s not available in this style", newfamily);
   }
   else {
      sprintf(_STR, "Font %s not available in this encoding", newfamily);
   }
   Wprintf(_STR);
   return (-1);
}

/*----------------------------------------------------------------*/
/* Set the font.  This depends on the system state as to whether  */
/* font is set at current position in label, for an entire label, */
/* or as the default font to begin new labels.			  */
/*----------------------------------------------------------------*/

void setfontval(w, value, settext)
  Widget w;
  unsigned short value;
  labelptr settext;
{
   short newfont;
   short i, tc, tflag;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < strlen(settext->string)) {
	 tflag = (textpos > 1) ? 1 : 0;
	 if ((short)*(settext->string + textpos - 1 - tflag) == TEXT_ESC &&
		 (tc = (short)*(settext->string + textpos - tflag) - FONT_START) >= 0) {
	    /* grab the current style and encoding and try to use it */

	    i = findbestfont(tc, (short)value, -1, -1);
	    if (i >= 0) {
	       undrawtext(settext);
	       *(settext->string + textpos - tflag) = i + FONT_START;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
   	          toggleexcl(w, Fonts, XtNumber(Fonts));
	       }
	    }
	    return;
         }
      }

      /* otherwise, look for the last style used in the string */
      for (i = textpos - 2; i >= 0; i--)
	 if ((short)*(settext->string + i - 1) == TEXT_ESC
		&& (tc = (short)*(settext->string + i) - FONT_START) >= 0)
	    break;
   }
   else tc = areastruct.psfont;

   /* Font change command will always find a value since at least one	*/
   /* font has to exist for the font menu button to exist.		*/

   if ((newfont = findbestfont(tc, (short)value, -1, -1)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      labeltext((unsigned short)newfont + FONT_START, 2);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }

   if (w != NULL) toggleexcl(w, Fonts, XtNumber(Fonts));

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontval()				  */
/*----------------------------------------------------------------*/

XtCallbackProc setfont(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE || eventmode == PAGECAT_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontval(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontval(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontval(w, value, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set the style (bold, italic, normal) for the font, similarly   */
/* to the above routine setfontval().				  */
/*----------------------------------------------------------------*/

void setfontstyle(w, value, settext)
  Widget w;
  unsigned short value;
  labelptr settext;
{
   short newfont;
   short i, tc, tflag;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < strlen(settext->string)) {
	 tflag = (textpos > 1) ? 1 : 0;
         if ((short)*(settext->string + textpos - 1 - tflag) == TEXT_ESC &&
                 (tc = (short)*(settext->string + textpos - tflag) - FONT_START) >= 0) {
	    /* find font which matches family and style, if available */

	    i = findbestfont(tc, -1, (short)value, -1);
	    if (i >= 0) {
	       undrawtext(settext);
               *(settext->string + textpos - tflag) = i + FONT_START;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
	          toggleexcl(w, FontStyles, XtNumber(FontStyles));
	       }
	    }
            return;
         }
      }

      /* otherwise, look for the last font used in the string */
      for (i = textpos - 2; i >= 0; i--)
	 if ((short)*(settext->string + i - 1) == TEXT_ESC
		&& (tc = (short)*(settext->string + i) - FONT_START) >= 0)
	    break;
   }
   else tc = areastruct.psfont;

   if ((newfont = findbestfont(tc, -1, (short)value, -1)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      labeltext((unsigned short)newfont + FONT_START, 2);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }
   if (w != NULL) toggleexcl(w, FontStyles, XtNumber(FontStyles));

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontstyle()				  */
/*----------------------------------------------------------------*/

XtCallbackProc fontstyle(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE || eventmode == PAGECAT_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontstyle(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontstyle(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontstyle(w, value, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set the encoding (standard, ISO-Latin1, special) for the font, */
/* similarly to the above routine setfontval().			  */
/*----------------------------------------------------------------*/

XtCallbackProc setfontencoding(w, value, settext)
  Widget w;
  unsigned short value;
  labelptr settext;
{
   short newfont;
   short i, tc, tflag;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < strlen(settext->string)) {
	 tflag = (textpos > 1) ? 1 : 0;
	 if ((short)*(settext->string + textpos - 1 - tflag) == TEXT_ESC &&
		 (tc = (short)*(settext->string + textpos - tflag) - FONT_START) >= 0) {

	    i = findbestfont(tc, -1, -1, (short)value);
	    if (i >= 0) {
	       undrawtext(settext);
	       *(settext->string + textpos - tflag) = i + FONT_START;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
   	          toggleexcl(w, FontEncodings, XtNumber(FontEncodings));
	       }
	    }
	    return;
	 }
      }

      /* otherwise, look for the last style used in the string */
      for (i = textpos - 2; i >= 0; i--)
	 if ((short)*(settext->string + i - 1) == TEXT_ESC
		&& (tc = (short)*(settext->string + i) - FONT_START) >= 0)
	    break;
   }
   else tc = areastruct.psfont;

   if ((newfont = findbestfont(tc, -1, -1, (short)value)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      labeltext((unsigned short)newfont + FONT_START, 2);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }

   if (w != NULL) toggleexcl(w, FontEncodings, XtNumber(FontEncodings));

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontencoding()			  */
/*----------------------------------------------------------------*/

XtCallbackProc fontencoding(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE || eventmode == PAGECAT_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontencoding(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontencoding(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontencoding(w, value, NULL);
      else objectdeselect();
   }
}

/*-------------------------------------------------*/
/* Wrapper for labeltext when called from the menu */
/*-------------------------------------------------*/

XtCallbackProc addtotext(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   if (eventmode != TEXT2_MODE && eventmode != TEXT3_MODE) return;
   if (value == (unsigned short)SPECIAL)
      labeltext((unsigned long)'\\', 0);
   else
      labeltext((unsigned long)value, 2);
}

/*-------------------------------------------------------------------------*/
