/*-----------------------------------------------------------------------*/
/* files.c --- file handling routines for xcircuit			 */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	 */
/*-----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <time.h>
#include <pwd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*------------------------------------------------------------------------*/
/* Useful (local) defines						  */
/*------------------------------------------------------------------------*/

#define OUTPUTWIDTH 80	/* maximum text width of output */

#define S_OBLIQUE 13	/* position of Symbol-Oblique in font array */

/* type checks */

#define IS_POLYGON(a) 	((*a)->type == POLYGON)
#define IS_LABEL(a)	((*a)->type == LABEL)
#define IS_OBJINST(a)	((*a)->type == OBJECT)
#define IS_ARC(a)	((*a)->type == ARC)
#define IS_SPLINE(a)	((*a)->type == SPLINE)
#define IS_PATH(a)	((*a)->type == PATH)


/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern char _STR2[250], _STR[150];
extern Globaldata xobjs;
extern Clientdata areastruct;
extern fontinfo *fonts;
extern short fontcount;
extern Display *dpy;
extern Window win;
extern short eventmode;
extern short beeper;
extern short pushes;
extern int *appcolors;
extern int number_colors;
extern colorindex *colorlist;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c            */
/*------------------------------------------------------------------------*/

#ifdef SCHEMA
extern int checkschem(objectptr, char *);
extern int checksym(objectptr, char *);
#endif

void savefile(), startloadfile(), importfile(), loadfile(), printobjects();
void loadlibrary(), savelibrary(), loadbackground(), execscript();
Boolean objectread();
void initmem(objectptr);
float getpsscale(float, short);

extern void calcbboxvalues(), setgridtype();
extern XtCallbackProc centerview(objectptr);
extern oparamptr parampos(objectptr, labelptr, char *, short *, short *);

/*------------------------------------------------------*/
/* Global variable definitions				*/
/*------------------------------------------------------*/

float version;

/*---------------------------------------------------------*/
/* Reset an object structure by freeing all alloc'd memory */
/*---------------------------------------------------------*/

void reset(localdata, mode)
  objectptr	localdata;
  short		mode;
{
   labelptr *resetlabel;
   polyptr  *resetpoly;

   if (localdata->parts > 0) {
      genericptr *genobj;
      if (mode != SAVE) {
	 for (genobj = localdata->plist; genobj < localdata->plist
	        + localdata->parts; genobj++) {
	    if (IS_POLYGON(genobj)) free(((polyptr)(*genobj))->points);
	    else if (IS_LABEL(genobj)) free(((labelptr)(*genobj))->string);
	    else if (IS_PATH(genobj)) free(((pathptr)(*genobj))->plist);
	    else if (IS_OBJINST(genobj)) {
	       int i;
	       for (i = 0; i < TOOBJINST(genobj)->num_params; i++) {
		  if (TOOBJINST(genobj)->params[i] != NULL)
		     free (TOOBJINST(genobj)->params[i]);
	       }
	       if (TOOBJINST(genobj)->params != NULL)
	          free(TOOBJINST(genobj)->params);
	    }
	    free(*genobj);
	 }
      }
      free(localdata->plist);
      
      removeparams(localdata);

      initmem(localdata);
      if (mode == DESTROY) free(localdata->plist);
   }
}

/*---------------------------------------------------------*/

void topreset()
{
   short rpage;
   objectptr resetpage;

   /* free alloc'd filename but */
   /* don't free it if another page points to it */

   if (xobjs.pagelist[areastruct.page]->filename != NULL) {
      for (rpage = 0; rpage < xobjs.pages; rpage++) {
         resetpage = xobjs.pagelist[rpage]->pageobj;
         if (resetpage != NULL) {
            if ((rpage != areastruct.page) && xobjs.pagelist[rpage]->filename ==
	        xobjs.pagelist[areastruct.page]->filename) break;
         }
      }
      if (rpage == xobjs.pages) free(xobjs.pagelist[areastruct.page]->filename);
   }
   if (xobjs.pagelist[areastruct.page]->background.name != NULL)
      free(xobjs.pagelist[areastruct.page]->background.name);

   rpage = areastruct.page;

   if (areastruct.selects > 0) {
      free(areastruct.selectlist);
      areastruct.selects = 0;
   }
   areastruct.textscale = 1.0;
   areastruct.linewidth = 1.0;

   xobjs.pagelist[rpage]->wirewidth = 2.0;
   xobjs.pagelist[rpage]->orient = 0;
   xobjs.pagelist[rpage]->pmode = 0;
   xobjs.pagelist[rpage]->outscale = 1.0;
   xobjs.pagelist[rpage]->filename = (char *)malloc
		((1 + strlen(xobjs.pagelist[rpage]->pageobj->name)) * sizeof(char));
   xobjs.pagelist[rpage]->background.name = (char *)NULL;

   strcpy(xobjs.pagelist[rpage]->filename, xobjs.pagelist[rpage]->pageobj->name);
   xobjs.pagelist[rpage]->drawingscale.x = xobjs.pagelist[rpage]->drawingscale.y = 1;
   xobjs.pagelist[rpage]->gridspace = 32;
   xobjs.pagelist[rpage]->snapspace = 16;

   /* use the coordstyle of page 0 as the default for new pages */

   if (xobjs.pagelist[0]->coordstyle == CM) {
      xobjs.pagelist[rpage]->coordstyle = CM;
      xobjs.pagelist[rpage]->pagesize.x = 595;
      xobjs.pagelist[rpage]->pagesize.y = 842;  /* A4 */
   }
   else {
      xobjs.pagelist[rpage]->coordstyle = FRAC_INCH;
      xobjs.pagelist[rpage]->pagesize.x = 612;	/* letter */
      xobjs.pagelist[rpage]->pagesize.y = 792;
   }
}

/*---------------------------------------------------------*/

void initmem(localdata)
  objectptr localdata;
{
   localdata->parts = 0;
   localdata->plist = (genericptr *)malloc(sizeof(genericptr));
   localdata->hidden = False;
   localdata->params = NULL;
   localdata->num_params = 0;

   localdata->viewscale = 0.5;
   localdata->pcorner.x = -areastruct.width;
   localdata->pcorner.y = -areastruct.height; 
   localdata->width = 0;
   localdata->height = 0;
   localdata->lowerleft.x = 0;
   localdata->lowerleft.y = 0;

#ifdef SCHEMA
   localdata->schemtype = SCHEMATIC;
   localdata->symschem = NULL;
   localdata->netlist = NULL;
   localdata->pinlist = NULL;
   localdata->calllist = NULL;
   localdata->traversed = 0;

   localdata->width2 = 0;
   localdata->height2 = 0;
   localdata->lleft2.x = 0;
   localdata->lleft2.y = 0;
#endif
}

/*--------------------------------------------------------------*/
/* Exhaustively compare the contents of two objects and return  */
/* true if equivalent, false if not.				*/
/*--------------------------------------------------------------*/

Boolean elemcompare(compgen, gchk)
   genericptr *compgen, *gchk;
{
   Boolean bres;
   switch((*compgen)->type) {
      case(ARC):
	 bres = (TOARC(compgen)->position.x == TOARC(gchk)->position.x &&
            TOARC(compgen)->position.y == TOARC(gchk)->position.y &&
	    TOARC(compgen)->style == TOARC(gchk)->style &&
	    TOARC(compgen)->width == TOARC(gchk)->width &&
	    abs(TOARC(compgen)->radius) == abs(TOARC(gchk)->radius) &&
	    TOARC(compgen)->yaxis  == TOARC(gchk)->yaxis &&
	    TOARC(compgen)->angle1 == TOARC(gchk)->angle1 &&
	    TOARC(compgen)->angle2 == TOARC(gchk)->angle2);
	 break;
      case(SPLINE):
	 bres = (TOSPLINE(compgen)->style == TOSPLINE(gchk)->style &&
	    TOSPLINE(compgen)->width == TOSPLINE(gchk)->width &&
	    TOSPLINE(compgen)->ctrl[0].x == TOSPLINE(gchk)->ctrl[0].x &&
	    TOSPLINE(compgen)->ctrl[0].y == TOSPLINE(gchk)->ctrl[0].y &&
	    TOSPLINE(compgen)->ctrl[1].x == TOSPLINE(gchk)->ctrl[1].x &&
	    TOSPLINE(compgen)->ctrl[1].y == TOSPLINE(gchk)->ctrl[1].y &&
	    TOSPLINE(compgen)->ctrl[2].x == TOSPLINE(gchk)->ctrl[2].x &&
	    TOSPLINE(compgen)->ctrl[2].y == TOSPLINE(gchk)->ctrl[2].y &&
	    TOSPLINE(compgen)->ctrl[3].x == TOSPLINE(gchk)->ctrl[3].x &&
	    TOSPLINE(compgen)->ctrl[3].y == TOSPLINE(gchk)->ctrl[3].y);
	 break;
      case(POLYGON): {
	 int i;
	 if (TOPOLY(compgen)->style == TOPOLY(gchk)->style &&
	       TOPOLY(compgen)->width == TOPOLY(gchk)->width &&
	       TOPOLY(compgen)->number == TOPOLY(gchk)->number) {
	    for (i = 0; i < TOPOLY(compgen)->number; i++) {
	       if (TOPOLY(compgen)->points[i].x != TOPOLY(gchk)->points[i].x
		     || TOPOLY(compgen)->points[i].y != TOPOLY(gchk)->points[i].y)
		  break;
	    }
	    bres = (i == TOPOLY(compgen)->number);
	 }
	 else bres = False;
	 }break;
   }
   return bres;
}

/*--------------------------------------------------------------------*/

short objcompare(obja, objb)
   objectptr obja, objb;
{
   genericptr *compgen, *glist, *gchk, *remg;
   short      csize;
   Boolean    bres;

   /* quick check on equivalence of number of objects */

   if (obja->parts != objb->parts) return False;

   /* check equivalence of parameters */

   if (obja->params == NULL && objb->params != NULL) return False;
   else if (obja->params != NULL && objb->params == NULL) return False;
   else if (obja->params != NULL || objb->params != NULL) {
      if (obja->num_params != objb->num_params) return False;
      else {
	 short i;
	 for (i = 0; i < obja->num_params; i++) {
	    oparamptr opsa = *(obja->params + i);
	    oparamptr opsb = *(objb->params + i);
	    if (opsa->type != opsb->type) return False;
	    if (opsa->type == XC_STRING && 
		  strcmp(opsa->pdefault, opsb->pdefault)) return False;
	    else if (opsa->type == XC_SHORT &&
		  *((short *)opsa->pdefault) != *((short *)opsb->pdefault))
	       return False;
	 }
      }
   }

   /* For the exhaustive check we must match component for component. */
   /* Best not to assume that elements are in same order for both.    */

   csize = obja->parts;

   glist = (genericptr *)malloc(csize * sizeof(genericptr));
   for (compgen = objb->plist; compgen < objb->plist + csize; compgen++)
      (*(glist + (int)(compgen - objb->plist))) = *compgen;
   for (compgen = obja->plist; compgen < obja->plist + obja->parts;
	 compgen++) {
      bres = False;
      for (gchk = glist; gchk < glist + csize; gchk++) {
	 if ((*gchk)->type == (*compgen)->type && (*compgen)->color == (*gchk)->color) {
	    switch((*compgen)->type) {
	       case(OBJECT):{
		  objinst *newobj = TOOBJINST(compgen);
		  objinst *oldobj = TOOBJINST(gchk);
	 	  bres = (newobj->position.x == oldobj->position.x &&
			newobj->color == oldobj->color &&
	    		newobj->position.y == oldobj->position.y &&
	     		newobj->rotation == oldobj->rotation &&
	     		newobj->scale == oldobj->scale &&
	     		newobj->thisobject == oldobj->thisobject &&
			newobj->num_params == oldobj->num_params);	/* more! */
		  } break;
	       case(LABEL):
	 	  bres = (TOLABEL(compgen)->position.x == TOLABEL(gchk)->position.x &&
			TOLABEL(compgen)->color == TOLABEL(gchk)->color &&
	     		TOLABEL(compgen)->position.y == TOLABEL(gchk)->position.y &&
	     		TOLABEL(compgen)->rotation == TOLABEL(gchk)->rotation &&
	     		TOLABEL(compgen)->scale == TOLABEL(gchk)->scale &&
	     		TOLABEL(compgen)->justify == TOLABEL(gchk)->justify &&
#ifdef SCHEMA
			((TOLABEL(compgen)->pin == TOLABEL(gchk)->pin) ||
				!areastruct.schemon) &&
#endif
	     		!strcmp(TOLABEL(compgen)->string, TOLABEL(gchk)->string));
		  break;
	       case(PATH): /* elements *must* be in same order for a path */
		  bres = (TOPATH(compgen)->parts == TOPATH(gchk)->parts &&
			TOPATH(compgen)->color == TOPATH(gchk)->color &&
			TOPATH(compgen)->style == TOPATH(gchk)->style &&
			TOPATH(compgen)->width == TOPATH(gchk)->width);
		  if (bres) {
		     genericptr *pathchk, *gpath;
		     for (pathchk = TOPATH(compgen)->plist, gpath =
			   TOPATH(gchk)->plist; pathchk < TOPATH(compgen)->plist
			   + TOPATH(compgen)->parts; pathchk++, gpath++) {
		        if (!elemcompare(pathchk, gpath)) bres = False;
		     }
		  }
		  break;
	       case(ARC): case(SPLINE): case(POLYGON):
		  bres = elemcompare(compgen, gchk);
		  break;
	    }
	 }
	 if (bres) {
	   csize--;
	   for (remg = gchk; remg < glist + csize; remg++)
               *remg = *(remg + 1);
           break;
	 }
      }
   }
   free(glist);
   if (csize != 0) return False;

   return(True);
}

/*------------------------*/
/* scale renormalization  */
/*------------------------*/

float getpsscale(float value, short page)
{
   if (xobjs.pagelist[page]->coordstyle == FRAC_INCH ||
 	 xobjs.pagelist[page]->coordstyle == DEC_INCH)
      return (value * INCHSCALE);
   else
      return (value * CMSCALE);
}

/*---------------------------------------------------------------*/
/* Keep track of columns of output and split lines when too long */
/*---------------------------------------------------------------*/

void dostcount(ps, count, addlength)
  FILE *ps;
  short *count, addlength;
{
   *count += addlength;
   if (*count > OUTPUTWIDTH) {
      *count = addlength;
      fprintf(ps, "\n");
   }
}

/*-----------------------------------------------*/
/* Check parameters for pointer to a given point */
/*-----------------------------------------------*/

#define xymat(z) \
    varcheck(ps, &(z.x), localdata, xaddin, &stcount); \
    varcheck(ps, &(z.y), localdata, yaddin, &stcount)
#define xypmat(z) \
    varcheck(ps, &(z->x), localdata, xaddin, &stcount); \
    varcheck(ps, &(z->y), localdata, yaddin, &stcount)
#define xyfmat(z) \
   fprintf(ps, "%hd %hd ", (int)z.x + xaddin, (int)z.y + yaddin);
  
/*---------------------------------------------------------------------*/

short varcheck(FILE *ps, short *value, objectptr localdata, int addin, short *stptr)
{
   int i, j;
   oparamptr ops;
   Boolean found = False;

   if (localdata->params != NULL) {
      for (i = 0; i < localdata->num_params; i++) {
	 ops = *(localdata->params + i);
	 if (ops->type == XC_SHORT) {
	    for (j = 0; j < ops->number; j++) {
	       if ((short *)(*(ops->position + j)) == value) {
		  sprintf(_STR, "v%d ", i + 1);		  
      		  dostcount (ps, stptr, strlen(_STR));
      		  fputs(_STR, ps);
		  found = True;
		  break;
	       }
	    }
	 }
	 if (found) break;
      }
   }
   if (!found) {
      sprintf(_STR, "%d ", (int)(*value) + addin);
      dostcount (ps, stptr, strlen(_STR));
      fputs(_STR, ps);
   }
}

/*------------------------*/
/* Load a PostScript file */
/*------------------------*/

#define LOAD_MODES 4

void getfile(button, mode, nulldata)
  Widget        button;
  void		*mode;
  caddr_t       nulldata;
{
   buttonsave *savebutton;
   static void ((*runprog[LOAD_MODES])()) =
	{startloadfile, importfile, loadbackground, execscript};
   static char *substr[LOAD_MODES] = 
	{"load", "import", "render", "execute"};
   char promptstr[25];

   if (pushes != 0) {
      Wprintf("Can only read file into top-level page!");
      return;
   }
   else if ((int)mode >= LOAD_MODES) {
      Wprintf("Unknown mode passed to routine getfile()\n");
      return;
   }
   sprintf(promptstr, "Select file to %s:", substr[(int)mode]);
   savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getfile, mode);
   popupprompt(button, promptstr, "\0", runprog[(int)mode],
		savebutton, True);
}

/*---------------------------------------------------------*/

Boolean nextfilename()	/* extract next filename from _STR2 into _STR */
{
   char *cptr, *slptr;

   sprintf(_STR, "%.149s", _STR2);
   if ((cptr = strrchr(_STR2, ',')) != NULL) {
      slptr = strrchr(_STR, '/');
      if (slptr == NULL || ((slptr - _STR) > (cptr - _STR2))) slptr = _STR - 1;
      sprintf(slptr + 1, "%s", cptr + 1); 
      *cptr = '\0';
      return True;
   }
   else return False;
}

/*---------------------------------------------------------*/

void loadfontlib()
{
   loadlibrary(FONTLIB);
}

/*------------------------------------------------------*/
/* Handle library loading and refresh current page if 	*/
/* it is a library page that just changed.		*/
/*------------------------------------------------------*/

void loadglib(Boolean lflag, short ilib, short tlib)
{
   while (nextfilename()) {
      if (lflag)
	 lflag = False;
      else
         ilib = createlibrary();
      loadlibrary(ilib);
      if (ilib == tlib) zoomview(NULL, NULL, NULL);
   }
   if (lflag)
      lflag = False;
   else
      ilib = createlibrary();
   loadlibrary(ilib);
   if (ilib == tlib) zoomview(NULL, NULL, NULL);
}

/*------------------------------------------------------*/
/* Load new library:  Create new library page and load	*/
/* to it.						*/
/*------------------------------------------------------*/

void loadulib()
{
   loadglib(False, (short)0, (short)is_library(objectdata) + LIBRARY);
}

/*-----------------------------------------------------------*/
/* Add to library:  If on library page, add to that library. */
/* Otherwise, create a new library page and load to it.	     */
/*-----------------------------------------------------------*/

void loadblib()
{
   short ilib, tlib;
   Boolean lflag = True;

   /* Flag whether current page is a library page or not */

   if ((tlib = is_library(objectdata)) < 0) {
      ilib = LIBRARY;
      lflag = False;
   }
   else
      ilib = tlib + LIBRARY;

   loadglib(lflag, ilib, tlib + LIBRARY);
}

/*---------------------------------------------------------*/

void getlib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getlib, NULL);
   popupprompt(button, "Enter library to load:", "\0", loadblib, savebutton,
	True);
}

/*---------------------------------------------------------*/

void getuserlib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, getuserlib, NULL);
   popupprompt(button, "Enter library to load:", "\0", loadulib, savebutton,
	True);
}

/*------------------------------------------------------*/
/* Load a library page (given in parameter "mode") and	*/
/* rename the library page to match the library name as */
/* found in the file header.				*/
/*------------------------------------------------------*/

void loadlibrary(mode)
  short mode;
{
   FILE *ps;
   char inname[150], temp[150], keyword[30], percentc;
   objectptr	*libobj;

   sscanf(_STR, "%s", inname);

   ps = fopen(inname, "r");
   if (ps == NULL) {
      char *inptr;

      /* try adding a .lps */
      
      if ((inptr = strrchr(inname, '/')) == NULL) inptr = inname;
      if (strchr(inptr, '.') == NULL) {
	 sprintf(inname, "%s.lps", _STR);
         ps = fopen(inname, "r");
      }
      if (ps == NULL) {

         /* if not found in cwd, look for environment variable  */
	 /* "XCIRCUIT_LIB_DIR" defined			 	*/
	 /* (Thanks to Ali Moini, U. Adelaide, S. Australia)	*/

	 char *tmp_s = getenv((const char *)"XCIRCUIT_LIB_DIR");

	 if (tmp_s != NULL) {
	    sprintf(inname, "%s/%s", tmp_s, _STR);
      	    ps = fopen(inname, "r");
      	    if (ps == NULL) {
	       sprintf(inname, "%s/%s.lps", tmp_s, _STR);
	       ps = fopen(inname, "r");
	    }
            if (ps == NULL && mode == FONTLIB) {
	       /* printf("looking in %s/fonts\n", tmp_s); */
	       sprintf(inname, "%s/fonts/%s", tmp_s, _STR);
	       ps = fopen(inname, "r");
	       if (ps == NULL) {
	          sprintf(inname, "%s/fonts/%s.lps", tmp_s, _STR);
	          ps = fopen(inname, "r");
	       }
	       if (ps == NULL) printf("not found there.\n");
            }
	 }

	 /* last resort:  hard-coded directory BUILTINS_DIR */

	 if (ps == NULL) {
	    sprintf(inname, "%s/%s", BUILTINS_DIR, _STR);
      	    ps = fopen(inname, "r");
      	    if (ps == NULL) {
	       sprintf(inname, "%s/%s.lps", BUILTINS_DIR, _STR);
	       ps = fopen(inname, "r");
	    }
            if (ps == NULL && mode == FONTLIB) {
	       /* printf("looking in %s/fonts\n", BUILTINS_DIR); */
	       sprintf(inname, "%s/fonts/%s", BUILTINS_DIR, _STR);
	       ps = fopen(inname, "r");
	       if (ps == NULL) {
	          sprintf(inname, "%s/fonts/%s.lps", BUILTINS_DIR, _STR);
	          ps = fopen(inname, "r");
	       }
	       if (ps == NULL) printf("not found there.\n");
	    }
	    if (ps == NULL) {
               Wprintf("No library file found.");
               return;
	    }
	 }
      }
   }

   version = VERSION;
   for(;;) {
      if (fgets(temp, 149, ps) == NULL) {
         Wprintf("Error in library.");
         return;
      }
      sscanf(temp, "%c %29s", &percentc, keyword);

      /* Commands in header are PostScript comments (%) */
      if (percentc == '%') {

         /* Rename the library page if the library name is found in the header 	*/
	 /* and this library is not a font description.				*/
         if ((mode != FONTLIB) && !strcmp(keyword, "Library")) {
	    char *cptr, *nptr;
	    cptr = strchr(temp, ':');
	    if (cptr != NULL) {
	       /* Don't write terminating newline to the object's name string */
	       if ((nptr = strchr(cptr + 2, '\n')) != NULL) *nptr = '\0';
	       if (xobjs.userlibs[mode - LIBRARY].number == 0) {
	          sprintf(xobjs.libtop[mode]->name, "Library: %.79s", cptr + 2);
		  renamelib(mode);
	       }
	    }
         }

         /* This comment gives the Xcircuit version number */
	 else if (!strcmp(keyword, "Version:")) {
	    float tmpv;
	    if (sscanf(temp, "%*c %*s %f", &tmpv) > 0) version = tmpv;
	 }

         /* This PostScript comment marks the end of the file header */
         else if (!strcmp(keyword, "XCircuitLib")) break;
      }
   }
   objectread(ps, objectdata, 0, 0, mode, temp, DEFAULTCOLOR);

   if (mode != FONTLIB) composelib(mode);

   /* printf("Loaded library %s\n", inname); */
   sprintf(_STR, "Loaded library %s", inname);
   Wprintf(_STR);
}

/*---------------------------------------------------------*/

void startloadfile()
{
   short firstpage = areastruct.page;

   while (nextfilename()) {
      loadfile(0);

      /* find next undefined page */

      while(areastruct.page < xobjs.pages &&
	   xobjs.pagelist[areastruct.page]->pageobj != NULL) areastruct.page++;
      changepage(areastruct.page);
   }
   loadfile(0);

   /* Display the first page loaded */

   newpage(firstpage);
   zoomview(NULL, NULL, NULL);

#ifdef SCHEMA
   setsymschem();
#endif
}

/*------------------------------------------------------*/
/* Import an xcircuit file onto the current page	*/
/*------------------------------------------------------*/

void importfile()
{
   while (nextfilename()) loadfile(1);
   loadfile(1);
   zoomview(NULL, NULL, NULL);
}

/*--------------------------------------------------------------*/
/* Load an xcircuit file into xcircuit				*/
/*--------------------------------------------------------------*/

void loadfile(mode)
  short mode;
{
   FILE *ps;
   char inname[250], temp[150], keyword[30], percentc, *pdchar;
   char teststr[50], teststr2[20], pagestr[100];
   short offx, offy, multipage, page, temppmode = 0, oldpage;
   objectptr	*libobj;
   float tmpfl;
   XPoint pagesize;

   /* First, if we're in catalog mode, return with error */

   if (eventmode == CATALOG_MODE) {
      Wprintf("Cannot load file from library window");
      return;
   }

#ifdef LGF
   /* quick check for .lgf or .lfo file */
   if ((pdchar = strchr(_STR, '.')) != NULL)
      if (!strcmp(pdchar + 1, "lgf") || !strcmp(pdchar + 1, "lfo")) {
	 loadlgf(mode);
         return;
      }
#endif

   sscanf(_STR, "%s", inname);
   ps = fopen(inname, "r");
   if (ps == NULL) {
      sprintf(inname, "%s.ps", _STR);
      ps = fopen(inname, "r");
      if (ps == NULL) {

	 /* Could possibly be a library file? */

         if ((pdchar = strchr(_STR, '.')) == NULL) {
	    sprintf(inname, "%s.lps", _STR);
	    if ((ps = fopen(inname, "r")) != NULL) {
	       fclose(ps);
	       loadlibrary(USERLIB);
	       return;
	    }

#ifdef LGF
	    sprintf(inname, "%s.lgf", _STR);
	    if ((ps = fopen(inname, "r")) != NULL) {
	       fclose(ps);
	       loadlgf(mode);
	       return;
	    }

	    sprintf(inname, "%s.lfo", _STR);
	    if ((ps = fopen(inname, "r")) != NULL) {
	       fclose(ps);
	       loadlgf(mode);
	       return;
	    }
#endif
	 }

	 /* What to do if no file was found:				  */
	 /* If extension was .lgf or .lfo, just flag an error.  	  */
         /* Otherwise, if we're on an empty page, go ahead and rename it. */

#ifdef LGF

         else {
	    if (!strcmp(pdchar + 1, "lgf") || !strcmp(pdchar + 1, "lfo")) {
	       sprintf(temp, "Error: Can't find lgf-format file %s", _STR);
               Wprintf(temp);
	       return;
	    }
	 }
#endif

         if (objectdata->parts == 0 && (mode == 0)) {

	    /* Check for file extension, and remove if "ps". */

            if ((pdchar = strchr(_STR, '.')) != NULL)
		if (!strcmp(pdchar + 1, "ps")) *pdchar = '\0';

            xobjs.pagelist[areastruct.page]->filename = (char *)realloc(
		 xobjs.pagelist[areastruct.page]->filename, (strlen(_STR) + 1)
		 * sizeof(char));
            sprintf(xobjs.pagelist[areastruct.page]->filename, "%s", _STR);

	    /* If the name has a path component, use only the root 	*/
	    /* for the object name, but the full path for the filename. */

	    if ((pdchar = strrchr(_STR, '/')) != NULL)
               sprintf(objectdata->name, "%s", pdchar + 1);
            else
	       sprintf(objectdata->name, "%s", _STR);

	    renamepage(areastruct.page);
	    printname(objectdata);
            Wprintf("Starting new drawing");
         }
         else {
	    sprintf(temp, "Can't find file %s, won't overwrite current page",
			_STR);
            Wprintf(temp);
         }
	 return;
      }
   }

   version = 1.0;
   multipage = 1;
   pagesize.x = 612;
   pagesize.y = 792;
   for(;;) {
      if (fgets(temp, 149, ps) == NULL) {
	 Wprintf("Error: EOF in or before prolog.");
	 return;
      }
      sscanf(temp, "%c%29s", &percentc, keyword);
      for (pdchar = keyword; isspace(*pdchar); pdchar++);
      if (percentc == '%') {
	 if (!strcmp(pdchar, "XCircuit")) break;
	 if (!strcmp(pdchar, "XCircuitLib")) {
	    /* currently doesn't have version control in libraries. . . */ 
	    version = VERSION;
	    break;
	 }
	 if (!strcmp(pdchar, "%Page:")) break;
	 if (strstr(pdchar, "PS-Adobe") != NULL)
	    temppmode = (strstr(temp, "EPSF") != NULL) ? 0 : 1;
         else if (!strcmp(pdchar, "Version:"))
	    sscanf(temp, "%*c%*s %f", &version);
	 else if (!strcmp(pdchar, "%Pages:")) {
	    pdchar = temp;
	    for (; !isspace(*pdchar); pdchar++);
	    for (; isspace(*pdchar); pdchar++);
	    multipage = atoi(pdchar);
	 }
	 else if (!strcmp(pdchar, "%BoundingBox:")) {
	    sscanf(temp, "%*s %hd %hd %hd %hd", &offx, &offy,
		&(pagesize.x), &(pagesize.y));
	    pagesize.x += offx;
	    pagesize.y += offy;
	 }
      }
#ifdef LGF
      else if (percentc == '-' && !strcmp(keyword, "5")) {
	 fclose(ps);
	 loadlgf(mode);
	 return;
      }
#endif
   }

   /* Look for old-style files (no %%Page; maximum one page in file) */

   if (!strcmp(pdchar, "XCircuit")) {
      int pch;

      do {
	 pch = getc(ps);
      } while (pch == '\n');
      ungetc(pch, ps);
      if (pch == '%') fgets(temp, 149, ps);
   }

   for (page = 0; page < multipage; page++) {
      sprintf(pagestr, "%d", page + 1);

      /* read out-of-page library definitions */

      if (strstr(temp, "%%Page:") == NULL && strstr(temp, "offsets") == NULL)
	 objectread(ps, objectdata, 0, 0, USERLIB, temp, DEFAULTCOLOR);

      if (strstr(temp, "%%Page:") != NULL)
	 sscanf(temp + 8, "%s", pagestr);

      /* go to new page if necessary */

      if (page > 0) {

	 /* find next undefined page */

	 while(areastruct.page < xobjs.pages &&
		xobjs.pagelist[areastruct.page]->pageobj != NULL) areastruct.page++;
	 changepage(areastruct.page);
      }

      /* If this file was a library file then there is no page to load */

      if (strstr(temp, "EndLib") != NULL) {
   	 composelib(USERLIB);
	 Wprintf("Loaded library.");
	 return;
      }

      /* good so far;  let's clear out the old data structure */
      /* and set the filename and pagename */

      if (mode == 0) {
	 char tpstr[6], *rootptr;

         reset(objectdata, NORMAL);
         topreset();
	 xobjs.pagelist[areastruct.page]->pmode = temppmode;
	 if (temppmode == 1) {
	    xobjs.pagelist[areastruct.page]->pagesize.x = pagesize.x;
	    xobjs.pagelist[areastruct.page]->pagesize.y = pagesize.y;
	 }

	 /* set filename and page title.				     */
         /* for multiple-page files, filename is not linked to the page name */
	 /* but should point to a single string containing the file name and */
	 /* malloc'd from the first page of the set.			     */
	 /* for single-page files, filename is normally linked to the page   */
	 /* name unless the page label has been made different.		     */

         /* (delete the filename suffix if it is ".ps") */

	 if (page == 0) {
	    if (multipage > 1) oldpage = areastruct.page;
            xobjs.pagelist[areastruct.page]->filename = (char *)realloc(
		 xobjs.pagelist[areastruct.page]->filename, (strlen(inname) + 1)
		 * sizeof(char));
            sprintf(xobjs.pagelist[areastruct.page]->filename, "%s", inname);
            if ((pdchar = strchr(xobjs.pagelist[areastruct.page]->filename, '.')) != NULL) 
	       if (!strcmp(pdchar + 1, "ps")) *pdchar = '\0';
	 }
	 else {
	    xobjs.pagelist[areastruct.page]->filename = xobjs.pagelist[oldpage]->filename;
	 }

	 rootptr = strrchr(xobjs.pagelist[areastruct.page]->filename, '/');
	 if (rootptr == NULL) rootptr = xobjs.pagelist[areastruct.page]->filename;
	 else rootptr++;

	 sprintf(tpstr, "%d", page + 1);
	 if (!strcmp(pagestr, tpstr)) {
	    if (multipage > 1)
	       sprintf(objectdata->name, "%.72s:%s", rootptr, tpstr);
	    else
	       sprintf(objectdata->name, "%.79s", rootptr);
	 }
	 else
	    sprintf(objectdata->name, "%.79s", pagestr);

         renamepage(areastruct.page);
      }

      /* read to the "scale" line, picking up inch/cm type, drawing */
      /* scale, and grid/snapspace along the way		    */

      for(;;) {
         if (strstr(temp, "offsets") != NULL) {
            sscanf(temp, "%c %hd %hd %*s", &percentc, &offx, &offy);
            if(percentc != '%') {
               Wprintf("Something wrong in offsets line.");
               offx = offy = 0;
            }
	 }

	 if (strstr(temp, "drawingscale") != NULL)
	    sscanf(temp, "%*c %hd:%hd %*s",
		&xobjs.pagelist[areastruct.page]->drawingscale.x,
		&xobjs.pagelist[areastruct.page]->drawingscale.y);
#ifdef SCHEMA
	 else if (strstr(temp, "is_symbol") != NULL) {
	    sscanf(temp, "%*c %49s", teststr);
	    checkschem(objectdata, teststr);
	 }
#endif
	 else if (strstr(temp, "gridspace"))
	    sscanf(temp, "%*c %f %f %*s", &xobjs.pagelist[areastruct.page]->gridspace,
		 &xobjs.pagelist[areastruct.page]->snapspace);
         else if (strstr(temp, "scale") != NULL || strstr(temp, "rotate") != NULL) {
            /* rotation (landscape mode) is optional; parse accordingly */

            sscanf(temp, "%f %s", &tmpfl, teststr);
            if (strstr(teststr, "scale") != NULL) {
	       setgridtype(teststr);
               xobjs.pagelist[areastruct.page]->outscale = tmpfl;
            }
            else if (!strcmp(teststr, "rotate")) {
               xobjs.pagelist[areastruct.page]->orient = (short)tmpfl;
               fgets(temp, 149, ps);
               sscanf(temp, "%f %s", &tmpfl, teststr2);
	       if (strstr(teststr2, "scale") != NULL) {
	          setgridtype(teststr2);
	          xobjs.pagelist[areastruct.page]->outscale = tmpfl;
	       }
	       else {
	          sscanf(temp, "%*f %*f %s", teststr2);
	          if (!strcmp(teststr2, "scale"))
	             xobjs.pagelist[areastruct.page]->outscale = tmpfl /
		          getpsscale(1.0, areastruct.page);
	          else {
	             Wprintf("Error in scale/rotate constructs.");
	             return;
	          }
	       }
            }
            else {     /* old style scale? */
               sscanf(temp, "%*f %*s %s", teststr2);
	       if ((teststr2 != NULL) && (!strcmp(teststr2, "scale")))
                  xobjs.pagelist[areastruct.page]->outscale = tmpfl /
		        getpsscale(1.0, areastruct.page);
	       else {
                  Wprintf("Error in scale/rotate constructs.");
                  return;
	       }
            }
	 }
         else if (strstr(temp, "setlinewidth") != NULL) {
            sscanf(temp, "%f %*s", &xobjs.pagelist[areastruct.page]->wirewidth);
            xobjs.pagelist[areastruct.page]->wirewidth /= 1.3;
	    break;
	 }
	 else if (strstr(temp, "insertion") != NULL) {
	    /* read in an included background image */
	    readbackground(ps);
	 }

	 if (fgets(temp, 149, ps) == NULL) {
            Wprintf("Error: Problems encountered in page header.");
            return;
         }
      }

      objectread(ps, objectdata, offx, offy, LIBRARY, temp, DEFAULTCOLOR);

      /* skip to next page boundary or file trailer */

      if (strstr(temp, "showpage") != NULL && multipage != 1) {
	 int pch;
         do {
	    pch = getc(ps);
         } while (pch == '\n');
	 ungetc(pch, ps);
         if (pch == '%') fgets(temp, 149, ps);
      }

      /* set object position to fit to window separately for each page */
      centerview(objectdata);
   }

   sprintf(_STR, "Loaded file: %s (%d page%s)", inname, multipage,
	(multipage > 1 ? "s" : ""));
   Wprintf(_STR);

   composelib(USERLIB);
   composelib(PAGELIB);

   if (version > VERSION) {
      sprintf(_STR, "WARNING: file %s is version %2.1f vs. executable version %2.1f",
		inname, version, VERSION);
      Wprintf(_STR);
   }

   version = VERSION;
}

/*----------------------------------------------------------------------*/
/* Read label segments							*/
/*----------------------------------------------------------------------*/

void readlabel(objectptr localdata, char **linehook, uchar **tmphook,
	short *tmpfontptr, float *scale, char *buffer)
{
   Boolean fline = False;
   char *sptr;
   short j;
   int paramno;

   char *lineptr = *linehook;
   uchar *tmpptr = *tmphook;

   if (lineptr == buffer) return;

   for (--lineptr; *lineptr == ' '; lineptr--);

   /* Embedded command is in braces: {} */

   if (*lineptr == '}') {	
      *lineptr-- = '\0';
      for (; *lineptr != '{'; lineptr--);

      if (!strcmp(lineptr + 1, "Ss")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = SUPERSCRIPT;
      }
      else if (!strcmp(lineptr + 1, "ss")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = SUBSCRIPT;
      }
      else if (!strcmp(lineptr + 1, "ns")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = NORMALSCRIPT;
      }
      else if (!strcmp(lineptr + 1, "hS")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = HALFSPACE;
      }
      else if (!strcmp(lineptr + 1, "qS")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = QTRSPACE;
      }
      else if (*(lineptr + 1) == '(')   /* "bs" for backspace assumed */
         for (sptr = lineptr + 2; *sptr != ')'; sptr++) {
	    *tmpptr++ = TEXT_ESC;
	    *tmpptr++ = BACKSPACE;
         }
      else if (!strcmp(lineptr + 1, "ol")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = OVERLINE;
         fline = True;
      }
      else if (!strcmp(lineptr + 1, "ul")) {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = UNDERLINE;
         fline = True;
      }
      else if (*(lineptr + 1) == '\0') {
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = NOLINE;
         fline = False;
      }
      else {      /* "cf" for changefont assumed */
         char *newptr = lineptr + 2;
         for (; *newptr != ' '; newptr++);  *newptr = '\0'; /* end-of-word */

         for (j = 0; j < fontcount; j++) {
            if (!strcmp(lineptr + 2, fonts[j].psname)) {
               if (tmpfontptr != NULL) *tmpfontptr = j; 
	       break;
	    }
         }

         if (j == fontcount) {		/* this is a non-loaded font */
            loadfontfile(lineptr + 2);
         }
         *tmpptr++ = TEXT_ESC;
         *tmpptr++ = (char)(j + FONT_START);

         for (++newptr; *newptr == ' '; newptr++);
         if (isdigit(*newptr)) { /* second form of "cf" command---includes scale */
	    float locscale;
	    sscanf(newptr, "%f", &locscale);
	    if ((scale != NULL) && (*scale == 0.0)) {
	       sscanf(newptr, "%f", scale);
	    }
	    else if ((scale == NULL) || (locscale != *scale)) {
	       if (scale != NULL)
	          printf("Font:  New scale change from %f to %f\n", *scale, locscale);
	       else
	          printf("Font:  Scale of %f is part of parameter\n", locscale);
	       *tmpptr++ = TEXT_ESC;
	       *tmpptr++ = FONT_SCALE;
	       strncpy(tmpptr, newptr, 4);
	       tmpptr += 4;
	       *tmpptr++ = TEXT_ESC;
	       *tmpptr++ = END_SCALE;
	    }
         }
      }
   }

   /* Text substring is in parentheses: () */

   else if (*lineptr == ')') {   /* parameterized substring */
      *lineptr-- = '\0';
      for (; *lineptr != '(' || (*lineptr == '(' && *(lineptr - 1) == '\\'); lineptr--) {
	 if (lineptr == buffer + 1) {
	    lineptr--;
	    break;
	 }
      }
      for (sptr = lineptr + 1; *sptr != '\0'; sptr++) {
         if (*sptr == '\\') {
            sptr++;
	    if (*sptr >= '0' && *sptr < '8') {
	       int tmpdig;
	       sscanf(sptr, "%3o", &tmpdig);
	       *tmpptr++ = (uchar)tmpdig;
	       sptr += 2;
	    }
            else *tmpptr++ = (uchar) *sptr;
	 }
         else *tmpptr++ = (uchar) *sptr;
      }
      if (fline == True) {
	 *tmpptr++ = TEXT_ESC;
	 *tmpptr++ = NOLINE;
	 fline = False;
      }
   }

   /* Parameterized substrings are denoted v1, v2, v3,... */

   else {
      oparamptr ops;
      char *defptr;

      for (--lineptr; *lineptr != ' ' && lineptr > buffer; lineptr--);
      sscanf(lineptr + ((lineptr == buffer) ? 1 : 2), "%d", &paramno);
      /* create hook to the parameter from the object's list */
      ops = localdata->params[paramno - 1];
      ops->number++;
      ops->position = (uchar **)realloc(ops->position, ops->number * sizeof(void *));
      ops->position[ops->number - 1] = (void *)tmpptr;
      /* insert default value */
      *tmpptr++ = TEXT_ESC;
      *tmpptr++ = PARAM_START;
      for (defptr = ops->pdefault; *defptr != '\0'; defptr++)
         *tmpptr++ = *defptr;
      *tmpptr++ = TEXT_ESC;
      *tmpptr++ = PARAM_END;

      /* printf("Parameter %d called from object %s\n", paramno, localdata->name); */
   }

   *linehook = lineptr;
   *tmphook = tmpptr;
}

/*------------------------------------------------------*/
/* Find matching parenthesis, bracket, or brace		*/
/* Don't count when backslash character '\' is in front */
/*------------------------------------------------------*/

char *find_match(char *fstring)
{
   int count = 1;
   char *search = fstring; 
   char source = *fstring;
   char target;

   switch(source) {
      case '(':  target = ')'; break;
      case '[':  target = ']'; break;
      case '{':  target = '}'; break;
      default:   target = source;
   }

   while (*++search != '\0') {
      if (*search == source && *(search - 1) != '\\') count++;
      else if (*search == target && *(search - 1) != '\\') count--;
      if (count == 0) break;
   }
   return search;
}

/*--------------------------------------------------------------*/
/* Read a parameter point					*/
/*--------------------------------------------------------------*/

varscan(objectptr localdata, char *lineptr, short *dpoint)
{
   oparamptr ops;
   char *defptr;
   int paramno;

   while (*lineptr == ' ') lineptr++;
   sscanf(lineptr + 1, "%d", &paramno);
   /* create hook to the parameter from the object's list */
   ops = localdata->params[paramno - 1];
   ops->number++;
   ops->position = (uchar **)realloc(ops->position, ops->number * sizeof(void *));
   ops->position[ops->number - 1] = (void *)dpoint;
}

/*--------------------------------------------------------------*/
/* Read an object (page) from a file into xcircuit		*/
/*--------------------------------------------------------------*/

Boolean objectread(ps, localdata, offx, offy, mode, retstr, ccolor)
  FILE *ps;
  objectptr	localdata;
  short		offx, offy, mode;
  char		*retstr;
  int		ccolor;
{
   char *temp, *buffer, keyword[80]; 
   short tmpfont = 0;
   int bufsize = 256;
   float tmpscale = 0.0;
   objectptr	*libobj;
   int curcolor = ccolor;
   int paramno, i, j, k;

   /* path-handling variables */
   pathptr *newpath;
   XPoint startpoint;

   keyword[0] = '\0';

   buffer = (char *)malloc(bufsize * sizeof(char));
   temp = buffer;

   for(;;) {
      char *lineptr, *keyptr;

      if (fgets(temp, 255, ps) == NULL) {
	 if (strcmp(keyword, "restore")) {
            Wprintf("Error: end of file.");
	    *retstr = '\0';
	 }
	 break;
      }
      temp = buffer;

      /* because PostScript is a stack language, we will scan from the end */
      for (lineptr = buffer; (*lineptr != '\n') && (*lineptr != '\0'); lineptr++);
      if (lineptr != buffer) {  /* ignore any blank lines */
         for (keyptr = lineptr - 1; isspace(*keyptr) && keyptr != buffer; keyptr--);
         for (; !isspace(*keyptr) && keyptr != buffer; keyptr--);
         sscanf(keyptr, "%79s", keyword);

         if (!strcmp(keyword, "showpage")) {
            strncpy(retstr, buffer, 150);
            retstr[149] = '\0';
	    free(buffer);
	    calcbbox(localdata);
	    return False;  /* end of page */
	 }

	 /* make a color change, adding the color if necessary */

	 else if (!strcmp(keyword, "scb")) {
	    float red, green, blue;
	    sscanf(buffer, "%f %f %f", &red, &green, &blue);
	    curcolor = rgb_alloccolor((int)(red * 65535), (int)(green * 65535),
		(int)(blue * 65535)); 
	    addnewcolorentry(curcolor);
	 }

	 /* end the color change, returning to default */

	 else if (!strcmp(keyword, "sce")) curcolor = ccolor;

	 /* begin a path constructor */

	 else if (!strcmp(keyword, "beginpath")) {
	    sscanf(buffer, "%hd %hd", &startpoint.x, &startpoint.y);
	    startpoint.x -= offx; startpoint.y -= offy;
	    NEW_PATH(newpath, localdata);
	    (*newpath)->plist = (genericptr *)malloc(sizeof(genericptr));
	    (*newpath)->parts = 0;
	    (*newpath)->color = curcolor;
	 }

	 /* end the path constructor */

	 else if (!strcmp(keyword, "endpath")) {
	    sscanf(buffer, "%hd %f", &(*newpath)->style, &(*newpath)->width);
	    if ((*newpath)->parts > 0) {
	       localdata->parts++;
	    }
	    else {	/* in case of an empty path */
	       free((*newpath)->plist);
	       free(*newpath);
	    }
	    newpath = NULL;
	 }

	 /* read path parts */

	 else if (!strcmp(keyword, "polyc")) {
	    polyptr *newpoly;
	    pointlist newpoints;
	    short tmpnum;

	    NEW_POLY(newpoly, (*newpath));
	    (*newpath)->parts++;

	    for (--keyptr; *keyptr == ' '; keyptr--);
	    for (; *keyptr != ' '; keyptr--);
	    sscanf(keyptr, "%hd", &tmpnum);
	    (*newpoly)->number = tmpnum + 1;
	    (*newpoly)->width = 1.0;
	    (*newpoly)->style = UNCLOSED;
	    (*newpoly)->color = curcolor;

            (*newpoly)->points = (pointlist) malloc((*newpoly)->number * 
		   sizeof(XPoint));

	    lineptr = buffer;
            for (newpoints = (*newpoly)->points + (*newpoly)->number - 1;
		   newpoints > (*newpoly)->points; newpoints--) {
	       sscanf(lineptr, "%hd", &newpoints->x); 
	       newpoints->x -= offx;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	       sscanf(lineptr, "%hd", &newpoints->y);
	       newpoints->y -= offy;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	    }
	    newpoints->x = startpoint.x;
	    newpoints->y = startpoint.y;
	    startpoint.x = (newpoints + (*newpoly)->number - 1)->x;
	    startpoint.y = (newpoints + (*newpoly)->number - 1)->y;
	 }

	 else if (!strcmp(keyword, "arc") || !strcmp(keyword, "arcn")) {
	    arcptr *newarc;
	    NEW_ARC(newarc, (*newpath));
	    (*newpath)->parts++;
	    sscanf(buffer, "%hd %hd %hd %f %f", &(*newarc)->position.x,
		  &(*newarc)->position.y, &(*newarc)->radius,
		  &(*newarc)->angle1, &(*newarc)->angle2);
	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->yaxis = (*newarc)->radius;
	    (*newarc)->width = 1.0;
	    (*newarc)->style = UNCLOSED;
	    (*newarc)->color = curcolor;
	    if (!strcmp(keyword, "arcn")) {
	       float tmpang = (*newarc)->angle1;
	       (*newarc)->radius = -((*newarc)->radius);
	       (*newarc)->angle1 = (*newarc)->angle2;
	       (*newarc)->angle2 = tmpang;
	    }
		
	    calcarc(*newarc);
	    startpoint.x = (short)(*newarc)->points[(*newarc)->number - 1].x;
	    startpoint.y = (short)(*newarc)->points[(*newarc)->number - 1].y;
	 }

	 else if (!strcmp(keyword, "pellip") || !strcmp(keyword, "nellip")) {
	    arcptr *newarc;
	    NEW_ARC(newarc, (*newpath));
	    (*newpath)->parts++;
	    sscanf(buffer, "%hd %hd %hd %hd %f %f", &(*newarc)->position.x,
		 &(*newarc)->position.y, &(*newarc)->radius, &(*newarc)->yaxis,
		 &(*newarc)->angle1, &(*newarc)->angle2);
	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->width = 1.0;
	    (*newarc)->style = UNCLOSED;
	    (*newarc)->color = curcolor;
	    if (!strcmp(keyword, "nellip")) {
	       float tmpang = (*newarc)->angle1;
	       (*newarc)->radius = -((*newarc)->radius);
	       (*newarc)->angle1 = (*newarc)->angle2;
	       (*newarc)->angle2 = tmpang;
		
	    }
	    calcarc(*newarc);
	    startpoint.x = (short)(*newarc)->points[(*newarc)->number - 1].x;
	    startpoint.y = (short)(*newarc)->points[(*newarc)->number - 1].y;
	 }

	 else if (!strcmp(keyword, "curveto")) {
	    splineptr *newspline;
	    NEW_SPLINE(newspline, (*newpath));
	    (*newpath)->parts++;
            sscanf(buffer, "%hd %hd %hd %hd %hd %hd", 
		  &(*newspline)->ctrl[1].x, &(*newspline)->ctrl[1].y,
		  &(*newspline)->ctrl[2].x, &(*newspline)->ctrl[2].y,
		  &(*newspline)->ctrl[3].x, &(*newspline)->ctrl[3].y);
	    (*newspline)->ctrl[0].x = startpoint.x;
	    (*newspline)->ctrl[0].y = startpoint.y;
	    (*newspline)->ctrl[1].x -= offx; (*newspline)->ctrl[2].x -= offx;
	    (*newspline)->ctrl[3].x -= offx;
	    (*newspline)->ctrl[1].y -= offy; (*newspline)->ctrl[2].y -= offy;
	    (*newspline)->ctrl[3].y -= offy;
	    (*newspline)->width = 1.0;
	    (*newspline)->style = UNCLOSED;
	    (*newspline)->color = curcolor;
	    calcspline(*newspline);
	    startpoint.x = (*newspline)->ctrl[3].x;
	    startpoint.y = (*newspline)->ctrl[3].y;
	 }

         /* read arcs */

         else if (!strcmp(keyword, "xcarc")) {
	    arcptr *newarc;
	 
	    NEW_ARC(newarc, localdata);

	    /* backward compatibility */
	    if (version < 1.5) 
	       sscanf(buffer, "%hd %hd %hd %f %f %f %hd", &(*newarc)->position.x,
	          &(*newarc)->position.y, &(*newarc)->radius, &(*newarc)->angle1,
	          &(*newarc)->angle2, &(*newarc)->width, &(*newarc)->style);
	    else
	       sscanf(buffer, "%hd %f %hd %hd %hd %f %f", &(*newarc)->style,
		  &(*newarc)->width, &(*newarc)->position.x, &(*newarc)->position.y,
		  &(*newarc)->radius, &(*newarc)->angle1, &(*newarc)->angle2);

	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->yaxis = (*newarc)->radius;
	    (*newarc)->color = curcolor;
	    calcarc(*newarc);
	    localdata->parts++;
         }

	 /* read ellipses */

         else if (!strcmp(keyword, "ellipse")) {
	    arcptr *newarc;
	 
	    NEW_ARC(newarc, localdata);

	    sscanf(buffer, "%hd %f %hd %hd %hd %hd %f %f", &(*newarc)->style,
		 &(*newarc)->width, &(*newarc)->position.x, &(*newarc)->position.y,
		 &(*newarc)->radius, &(*newarc)->yaxis, &(*newarc)->angle1,
		 &(*newarc)->angle2);

	    (*newarc)->position.x -= offx; (*newarc)->position.y -= offy;
	    (*newarc)->color = curcolor;
	    calcarc(*newarc);
	    localdata->parts++;
         }

         /* read polygons */
	 /* (and wires---backward compatibility for v1.5 and earlier) */

         else if (!strcmp(keyword, "polygon") || !strcmp(keyword, "wire")) {
	    polyptr *newpoly;
	    pointlist newpoints;

	    NEW_POLY(newpoly, localdata);
	    lineptr = buffer;

	    if (!strcmp(keyword, "wire")) {
	       (*newpoly)->number = 2;
	       (*newpoly)->width = 1.0;
	       (*newpoly)->style = UNCLOSED;
	    }
	    else {
	       /* backward compatibility */
	       if (version < 1.5) {
	          for (--keyptr; *keyptr == ' '; keyptr--);
	          for (; *keyptr != ' '; keyptr--);
	          sscanf(keyptr, "%hd", &(*newpoly)->style);
	          for (--keyptr; *keyptr == ' '; keyptr--);
	          for (; *keyptr != ' '; keyptr--);
	          sscanf(keyptr, "%f", &(*newpoly)->width);
	       }
	       for (--keyptr; *keyptr == ' '; keyptr--);
	       for (; *keyptr != ' '; keyptr--);
	       sscanf(keyptr, "%hd", &(*newpoly)->number);

	       if (version >= 1.5) {
	          sscanf(lineptr, "%hd", &(*newpoly)->style);
	          for (; *lineptr == ' '; lineptr++);
	          for (; *lineptr != ' '; lineptr++); 
	          sscanf(lineptr, "%f", &(*newpoly)->width);
	          for (; *lineptr == ' '; lineptr++);
	          for (; *lineptr != ' '; lineptr++); 
	       }
	    }

	    if ((*newpoly)->style & BBOX)
	       (*newpoly)->color = BBOXCOLOR;
	    else
	       (*newpoly)->color = curcolor;
            (*newpoly)->points = (pointlist) malloc((*newpoly)->number * 
		   sizeof(XPoint));

            for (newpoints = (*newpoly)->points; newpoints < (*newpoly)->points
		+ (*newpoly)->number; newpoints++) {
	       if (sscanf(lineptr, "%hd", &newpoints->x) == 0)
		  varscan(localdata, lineptr, &newpoints->x);
	       newpoints->x -= offx;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	       if (sscanf(lineptr, "%hd", &newpoints->y) == 0)
		  varscan(localdata, lineptr, &newpoints->y);
	       newpoints->y -= offy;
	       for (; *lineptr == ' '; lineptr++); 
	       for (; *lineptr != ' '; lineptr++); 
	    }
	    localdata->parts++;
         }

	 /* read spline curves */

         else if (!strcmp(keyword, "spline")) {
            splineptr *newspline;
	    short i;

	    NEW_SPLINE(newspline, localdata);

	    /* backward compatibility */
	    if (version < 1.5) {
               sscanf(buffer, "%f %hd %hd %hd %hd %hd %hd %hd %hd %hd", 
	          &(*newspline)->width, &(*newspline)->ctrl[1].x,
	          &(*newspline)->ctrl[1].y, &(*newspline)->ctrl[2].x,
	          &(*newspline)->ctrl[2].y, &(*newspline)->ctrl[3].x,
	          &(*newspline)->ctrl[3].y, &(*newspline)->ctrl[0].x,
	          &(*newspline)->ctrl[0].y, &(*newspline)->style);
	    }
	    else {
               sscanf(buffer, "%hd %f %hd %hd %hd %hd %hd %hd %hd %hd", 
		  &(*newspline)->style, &(*newspline)->width,
		  &(*newspline)->ctrl[1].x, &(*newspline)->ctrl[1].y,
		  &(*newspline)->ctrl[2].x, &(*newspline)->ctrl[2].y,
		  &(*newspline)->ctrl[3].x, &(*newspline)->ctrl[3].y,
		  &(*newspline)->ctrl[0].x, &(*newspline)->ctrl[0].y);
	    }

	    (*newspline)->ctrl[1].x -= offx; (*newspline)->ctrl[2].x -= offx;
	    (*newspline)->ctrl[0].x -= offx;
	    (*newspline)->ctrl[3].x -= offx;
	    (*newspline)->ctrl[1].y -= offy; (*newspline)->ctrl[2].y -= offy;
	    (*newspline)->ctrl[3].y -= offy;
	    (*newspline)->ctrl[0].y -= offy;
	    (*newspline)->color = curcolor;
            localdata->parts++;
	    calcspline(*newspline);
         }

         /* read labels */

         else if (!strcmp(keyword, "fontset")) { 	/* old style */
            char tmpstring[100];
            int i;
            sscanf(buffer, "%f %*c%s", &tmpscale, tmpstring);
            for (i = 0; i < fontcount; i++)
               if (!strcmp(tmpstring, fonts[i].psname)) {
		  tmpfont = i;
		  break;
	       }
	    if (i == fontcount) i = 0;	/* Why bother with anything fancy? */
         }

         else if (!strcmp(keyword, "label") || !strcmp(keyword, "pinlabel")
		|| !strcmp(keyword, "pinglobal") || !strcmp(keyword, "infolabel")) {

	    labelptr *newlabel;
	    uchar tmpstring[150], *tmpptr = tmpstring;
	    char *sptr;

	    NEW_LABEL(newlabel, localdata);

	    /* scan backwards to get the number of substrings */
	    lineptr = keyptr - 1;
	    for (i = 0; i < 5; i++) {
	       for (; *lineptr == ' '; lineptr--);
	       for (; *lineptr != ' '; lineptr--);
	    }
	    sscanf(lineptr, "%*hd %hd %hd %hd %hd", &(*newlabel)->justify,
		&(*newlabel)->rotation, &(*newlabel)->position.x,
		&(*newlabel)->position.y);
	    (*newlabel)->position.x -= offx; (*newlabel)->position.y -= offy;
	    if ((*newlabel)->rotation > 0)
	       (*newlabel)->rotation = 360 - (*newlabel)->rotation;
	    (*newlabel)->rotation /= ROT_UNIT;
	    (*newlabel)->rotation++;
	    (*newlabel)->scale = tmpscale;
	    (*newlabel)->color = curcolor;

#ifdef SCHEMA
	    (*newlabel)->pin = False;
	    if (strcmp(keyword, "label")) {	/* all the schematic types */
	       /* enable schematic capture if it is not already on. */
	       if (!areastruct.schemon) doxschema(NULL, NULL, NULL);
	       if (!strcmp(keyword, "pinlabel"))
		  (*newlabel)->pin = LOCAL;
	       else if (!strcmp(keyword, "pinglobal"))
		  (*newlabel)->pin = GLOBAL;
	       else if (!strcmp(keyword, "infolabel")) {
		  (*newlabel)->pin = INFO;
		  if (curcolor == DEFAULTCOLOR)
		     (*newlabel)->color = AUXCOLOR;
	       }
	    }
#endif

	    for (; *lineptr == ' '; lineptr--);

	    if (*lineptr == ')') {

	       /* This is for old-style labels only */

	       *tmpptr++ = TEXT_ESC;
	       *tmpptr++ = tmpfont + FONT_START;
	       *lineptr-- = '\0';
	       for (; *lineptr != '(' || (*lineptr == '(' && *(lineptr - 1) == 
		     '\\'); lineptr--)
	          if (lineptr == buffer + 1) {
		     lineptr--;
		     break;
	          }
	       for (sptr = lineptr + 1; *sptr != '\0'; sptr++) {
	          if (*sptr == '\\') {
		     sptr++;
		     if (*sptr >= '0' && *sptr < '8') {
		        int tmpdig;
		        sscanf(sptr, "%3o", &tmpdig);
		        *tmpptr++ = (uchar)tmpdig;
		        sptr += 2;
		     }
		     else *tmpptr++ = (uchar) *sptr;
	          }
	          else *tmpptr++ = (uchar) *sptr;
	       }
	    }
	    else lineptr++;

	    while (lineptr > buffer) {
	       readlabel(localdata, &lineptr, &tmpptr, &tmpfont,
			&((*newlabel)->scale), buffer);
	    }
	    *tmpptr = '\0';

	    (*newlabel)->string = (uchar *) malloc((strlen(tmpstring) + 1) *
               sizeof(uchar));
	    strcpy((*newlabel)->string, (char *)tmpstring);

	    /* reattach the parameter hooks to the allocated string */

	    rehook(localdata, (char *)tmpstring, (*newlabel)->string);

#ifdef SCHEMA

	    /* link string to the one that corresponds to this one */
	    /* in the symbol, if it exists.			   */

	    if (areastruct.schemon && localdata->symschem != NULL &&
		  (*newlabel)->pin && ((*newlabel)->pin != INFO)) {
	       genericptr *pgen;
	       for (pgen = localdata->symschem->plist; pgen < localdata->
			symschem->plist + localdata->symschem->parts; pgen++) {
		  if ((*pgen)->type == LABEL)
		     if (!strcmp(TOLABEL(pgen)->string, (*newlabel)->string)) {
/*			free((*newlabel)->string);			*/
/*			(*newlabel)->string = TOLABEL(pgen)->string;	*/
		 	break;
		     }
	       }
	       if (pgen == localdata->symschem->plist + localdata->symschem->parts)
		  Wprintf("Error:  Unattached pin");
	    }
#endif

	    localdata->parts++;
         }

	 /* read symbol-to-schematic connection */

#ifdef SCHEMA
	 else if (!strcmp(keyword, "is_schematic")) {
	    char tempstr[50];
	    for (lineptr = buffer; *lineptr == ' '; lineptr++);
	    sscanf(++lineptr, "%49s", tempstr);
	    checksym(localdata, tempstr);
	 }
#endif

         /* read bounding box */

         else if (!strcmp(keyword, "bbox")) {
	    for (lineptr = buffer; *lineptr == ' '; lineptr++);
            if (*lineptr != '%') {
	       Wprintf("Illegal bbox.");
	       free(buffer);
               *retstr = '\0';
	       return True;
	    }
#ifdef SCHEMA
	    sscanf(++lineptr, "%hd %hd %hd %hd", &localdata->lleft2.x, 
	       &localdata->lleft2.y, &localdata->width2, &localdata->height2);
	    if (mode == FONTLIB) {
	       localdata->lowerleft.x = localdata->lleft2.x;
	       localdata->lowerleft.y = localdata->lleft2.y;
	       localdata->width = localdata->width2;
	       localdata->height = localdata->height2;
	    }
#else
	    sscanf(++lineptr, "%hd %hd %hd %hd", &localdata->lowerleft.x, 
	       &localdata->lowerleft.y, &localdata->width, &localdata->height);
#endif
         }

	 /* read "hidden" attribute */

	 else if (!strcmp(keyword, "hidden")) {
	    localdata->hidden = True;
	 }

         /* read objects */

         else if (!strcmp(keyword, "{")) {  /* This is an object definition */
	    Boolean redef = False;
	    objectptr *newobject;
	    objectptr *curlib = (mode == FONTLIB) ?
		xobjs.fontlib.library : xobjs.userlibs[mode - LIBRARY].library;
	    short *libobjects = (mode == FONTLIB) ?
		&xobjs.fontlib.number : &xobjs.userlibs[mode - LIBRARY].number;

	    for (lineptr = buffer; *lineptr == ' '; lineptr++);
	    if (*lineptr++ != '/') {
	       Wprintf("Bad object definition.");
	       free(buffer);
               *retstr = '\0';
	       
	    }
	    sscanf(lineptr, "%79s", keyword);

	    curlib = (objectptr *) realloc(curlib, (*libobjects + 1)
		      * sizeof(objectptr));
	    if (mode == FONTLIB) xobjs.fontlib.library = curlib;
	    else xobjs.userlibs[mode - LIBRARY].library = curlib;
	    /* initial 1-pointer allocations */
            newobject = curlib + (*libobjects);
	    *newobject = (objectptr) malloc(sizeof(object));
	    initmem(*newobject);

	    /* check that this object is not already in list of objects */

	    if (mode == FONTLIB) {
	       for (libobj = xobjs.fontlib.library; libobj != xobjs.fontlib.library +
	            xobjs.fontlib.number; libobj++) {
	          if (!strcmp(keyword, (*libobj)->name)) {  /* Is a redefinition */
		     redef = True;
		     break;
	          }
	       }
	    }
	    else {
	       for (i = 0; i < xobjs.numlibs; i++) {
		  for (j = 0; j < xobjs.userlibs[i].number; j++) {
		     libobj = xobjs.userlibs[i].library + j;
	             if (!strcmp(keyword, (*libobj)->name)) {  /* Is a redefinition */
		        redef = True;
		        break;
		     }
	          }
		  if (redef) break;
	       }
	    }

            (*libobjects)++;
	    sprintf((*newobject)->name, "%s", keyword);

#ifdef SCHEMA
	    /* initmem() initialized schemtype to SCHEMATIC;  change it. */
	    (*newobject)->schemtype = SYMBOL;
#endif

	    if (objectread(ps, *newobject, 0, 0, mode, retstr, curcolor) == True) {
               strncpy(retstr, buffer, 150);
               retstr[149] = '\0';
	       free(buffer);
	       return True;
            }

	    /* do an exhaustive comparison between the objects. */
	    /* if they are the same, destroy the duplicate.  If */
	    /* different, rename the original one.		*/

	    else if (redef == True) {
	       if (objcompare(*newobject, *libobj) == True) {
		   reset(*newobject, DESTROY);
		   (*libobjects)--;
	       }
	       else {
		   strcpy(_STR, (*libobj)->name);
		   checkname(*libobj);
	       }
	    }
         }
         else if (!strcmp(keyword, "def")) {
            strncpy(retstr, buffer, 150);
            retstr[149] = '\0';
	    free (buffer);
	    if (mode != FONTLIB) {
	       calcbboxvalues(localdata, (uchar)1, (genericptr *)NULL);
	       centerview(localdata);
	    }
	    return False; /* end of object def or end of object library */
	 }

	 else if (!strcmp(keyword, "loadfontencoding")) {
	    for (lineptr = buffer; *lineptr != '%'; lineptr++);
	    sscanf (lineptr + 1, "%s", _STR);
	    if (*(lineptr + 1) != '%') loadfontfile(_STR);
	 }
	 /* loadlibrary now takes an optional second argument, the library page */
	 /* (starts at library page 1)						*/
	 else if (!strcmp(keyword, "loadlibrary")) {
	    int ilib, tlib;

	    for (lineptr = buffer; *lineptr != '%'; lineptr++);
	    sscanf (++lineptr, "%s", _STR);
	    while (isspace(*lineptr)) lineptr++;
	    while (!isspace(*++lineptr));
	    while (isspace(*++lineptr));
	    if (sscanf (lineptr, "%d", &ilib) > 0) {
	       while (ilib >= xobjs.numlibs) {
		  tlib = createlibrary();
		  if (tlib != xobjs.numlibs - 2 + LIBRARY) {
		     ilib = tlib;
		     break;
		  }
	       }
	       mode = ilib - 1 + LIBRARY;
	    }
	    loadlibrary(mode);
	 }
	 else if (!strcmp(keyword, "beginparm")) { /* parameterized object */
	    short tmpnum, i;
	    char *stringend;
	    for (--keyptr; *keyptr == ' '; keyptr--);
	    for (; *keyptr != ' '; keyptr--);
	    sscanf(keyptr, "%hd", &tmpnum);
	    lineptr = keyptr - 1;
	    for (; *lineptr == ' '; lineptr--);
	    if (tmpnum < 256) {		/* read parameter defaults in order */
	       localdata->num_params = (unsigned char) tmpnum;
	       localdata->params = (oparamptr *)malloc(tmpnum * sizeof(oparamptr));
	       for (i = tmpnum - 1; i >= 0; i--) {
		  oparamptr ops;
		  *(localdata->params + i) = (oparamptr)malloc(sizeof(oparam));
 		  ops = *(localdata->params + i);
		  if (*lineptr == ')' || *lineptr == '}') {  /* type is XC_STRING */
		     uchar tmpstring[256], *tmpptr = tmpstring;
		     char *linetmp = lineptr;
		     /* get simple substring or set of substrings and commands */
		     while (linetmp-- > buffer) {
			if (*linetmp == '{' || *linetmp == '(')
			   if (find_match(linetmp) == lineptr)
			      break;
		     }
		     if (*lineptr == ')') lineptr++;
			
		     while (lineptr > linetmp + (*linetmp == '{') ? 1 : 0) 
		        readlabel(localdata, &lineptr, &tmpptr, NULL, NULL, buffer);
		     *tmpptr = '\0';

		     ops->type = (uchar)XC_STRING;
		     ops->pdefault = (void *)malloc(strlen(tmpstring) + 1);
		     strcpy((char *)ops->pdefault, tmpstring);
		     /* printf("Parameter %d to object %s defaults to string \"%s\"\n",
				i + 1, localdata->name, (char *)ops->pdefault); */
		  }
		  else {	/* type is XC_SHORT */
		     short tmpshort;
		     ops->type = (uchar)XC_SHORT;
		     ops->pdefault = (void *)malloc(sizeof(short));
	             for (; *lineptr != ' ' && lineptr > buffer; lineptr--); 
	             sscanf(lineptr, "%hd", &tmpshort);
	             *((short *)ops->pdefault) = tmpshort;
		     /* printf("Parameter %d to object %s defaults to value %hd\n",
				i + 1, localdata->name,
				(short)(*((short *)ops->pdefault))); 	*/
		  }
		  ops->number = 0;
		  ops->position = (uchar **)malloc(sizeof(void *));
		  if (i > 0)
		     for (--lineptr; *lineptr == ' ' || *lineptr == '{'; lineptr--); 
	       }
	    }
	 }
#ifdef SCHEMA
	 else if (!strcmp(keyword, "fundamental")) {
	    localdata->schemtype = FUNDAMENTAL;
	    if (!areastruct.schemon) doxschema(NULL, NULL, NULL);
	 }
#endif
         else if (!strcmp(keyword, "begingate")) {
	    localdata->num_params = (unsigned char) 0;
	    localdata->params = (oparamptr *)NULL;
	 }
         else if (!strcmp(keyword, "EndLib")) break;
	 else if (!strcmp(keyword, "restore"));    /* handled at top */
	 else if (!strcmp(keyword, "grestore"));   /* ignore */
         else if (!strcmp(keyword, "endgate"));    /* also ignore */
	 else if (!strcmp(keyword, "xyarray"));	   /* ignore for now */
         else {
	    objinstptr *newinst;
	    char *tmpptr;
	    Boolean found = False;

	    /* first, make sure this is not a general comment line */
	    /* and return if we have a page boundary	   	   */

	    for (tmpptr = buffer; isspace(*tmpptr); tmpptr++);
	    if (*tmpptr == '%') {
	       if (strstr(buffer, "%%Page:") == tmpptr) {
                  strncpy(retstr, buffer, 150);
                  retstr[149] = '\0';
		  free (buffer);
		  return True;
	       }
	       continue;
	    }

	    /* (Assume that this line calls an object instance) */
	    /* Double loop through user libraries 		*/

            for (k = 0; k < ((mode == FONTLIB) ? 1 : xobjs.numlibs); k++) {
	       for (j = 0; j < ((mode == FONTLIB) ? xobjs.fontlib.number :
			xobjs.userlibs[k].number); j++) {
		  libobj = (mode == FONTLIB) ? xobjs.fontlib.library + j :
			xobjs.userlibs[k].library + j;

	          if (!strcmp(keyword, (*libobj)->name)) {
		     char *arrayptr, *endptr;
		     found = True;
		     NEW_OBJINST(newinst, localdata);
      	             localdata->parts++;
		     /* localdata->hidden = False; */

	             sscanf(buffer, "%f %hd %hd %hd", &(*newinst)->scale,
		        &(*newinst)->rotation, &(*newinst)->position.x,
		        &(*newinst)->position.y); 
		     (*newinst)->position.x -= offx;
		     (*newinst)->position.y -= offy;
                     if ((*newinst)->rotation > 0)
                        (*newinst)->rotation = (360 - (*newinst)->rotation) /
                           ROT_UNIT + 1;
                     else if ((*newinst)->rotation == 0) (*newinst)->rotation++;
                     else
                        (*newinst)->rotation = ((*newinst)->rotation + 1) /
                            ROT_UNIT - 1;
		     (*newinst)->thisobject = *libobj;
		     (*newinst)->color = curcolor;
		     (*newinst)->params = NULL;
		     (*newinst)->num_params = (uchar)0;

		     /* Does this instance contain parameters? */

		     if ((arrayptr = strchr(buffer, '[')) != NULL) {
		        (*newinst)->params = (void **)malloc((*libobj)->num_params
				* sizeof(void *));
		        (*newinst)->num_params = (*libobj)->num_params;
		        endptr = find_match(arrayptr);
		        /* printf("Object instance contains parameters\n"); */
		        arrayptr++;
		        for (i = 0; i < (*libobj)->num_params; i++) {
			   if (*arrayptr == '\0' || arrayptr == endptr)
			      (*newinst)->params[i] = NULL;
			   else {
			      if (*arrayptr == '(' || *arrayptr == '{') { 

			         /* type XC_STRING */

			         char *substrend, *lineptr;
			         uchar tmpstring[256], *tmpptr = tmpstring;

			         substrend = find_match(arrayptr);
			         lineptr = substrend;
			         if (*arrayptr == '(') lineptr++;
			         else arrayptr++;
			         while (lineptr > arrayptr) {
				    readlabel(*libobj, &lineptr, &tmpptr, NULL, NULL,
				        arrayptr);
			         }
			         *tmpptr = '\0';

			         (*newinst)->params[i] = (void *)malloc(
				      strlen(tmpstring) + 1);
			         strcpy((*newinst)->params[i], tmpstring);
			         arrayptr = substrend + 1;
			      /* printf("Object %s called with parameter %d of \"%s\"\n",
					(*newinst)->thisobject->name, i + 1,
					(char *)((*newinst)->params[i])); */
			      }
			      else {

			         /* type XC_SHORT */

			         short tmpshort;
			         (*newinst)->params[i] = (void *)malloc(sizeof(short));
			         sscanf(arrayptr, "%hd", &tmpshort);
			         *((short *)((*newinst)->params[i])) = tmpshort;
			         for (; *arrayptr != ' ' && *arrayptr != ']'; arrayptr++);
			      /* printf("Object %s called with parameter %d value %hd\n",
					(*newinst)->thisobject->name, i + 1,
					*((short *)((*newinst)->params[i]))); */
			      }
			      for (; *arrayptr == ' ' && *arrayptr != ']'; arrayptr++);
			   }
		        }
		     }
		     break;
	          } /* if !strcmp */
	       } /* for j loop */
	       if (found) break;
	    } /* for k loop */
	    if (!found) {
	       /* will assume that we have a continuation line */

               for (lineptr = buffer; (*lineptr != '\n') && (*lineptr != '\0');
		        lineptr++);
	       if (*lineptr == '\n') *lineptr = ' ';

	       bufsize = (int)(lineptr - buffer) + 256;
	       buffer = (char *)realloc(buffer, bufsize * sizeof(char));
	       temp = buffer + (bufsize - 256);
 	    }
         }
      }
   }
   strncpy(retstr, buffer, 150);
   retstr[149] = '\0';
   free(buffer);
   return True;
}

/*------------------------*/
/* Save a PostScript file */
/*------------------------*/

void setfile(button, fnamewidget, clientdata) 
  Widget	button, fnamewidget;	/* Widget containing filename info */
  caddr_t	clientdata;
{
   /* see if name has been changed in the buffer */

   sprintf(_STR2, "%.249s", (char *)XwTextCopyBuffer(fnamewidget));
   if (strcmp(xobjs.pagelist[areastruct.page]->filename, _STR2)) {
      Wprintf("Changing name of edit file.");
      xobjs.pagelist[areastruct.page]->filename = (char *) realloc (
	   xobjs.pagelist[areastruct.page]->filename, (strlen(_STR2) + 1) *
	   sizeof(char));
      strcpy(xobjs.pagelist[areastruct.page]->filename, _STR2);
   }
   if (strstr(xobjs.pagelist[areastruct.page]->filename, "Page ") != NULL) {
      Wprintf("Warning: Enter a new name.");
      if (beeper) XBell(dpy, 100);
   }
   else {

      Arg wargs[1];
      Widget db, di;

      savefile(objectdata, CURRENT_PAGE); 

      /* Change "close" button to read "done" */

      di = XtParent(button);
      db = XtNameToWidget(di, "Close");
      XtSetArg(wargs[0], XtNlabel, "  Done  ");
      XtSetValues(db, wargs, 1);
      if (beeper) XBell(dpy, 100);
   }
}

/*--------------------------------------------------------------*/
/* tempfile save						*/
/*--------------------------------------------------------------*/

XtTimerCallbackProc savetemp(Widget w, XtIntervalId *id)
{
   if (xobjs.tempfile == NULL)
      xobjs.tempfile = tempnam(NULL, "XC");        /* generate temporary file */

   savefile(objectdata, ALL_PAGES);
   xobjs.timeout_id = XtAddTimeOut((unsigned long)60000 *
	xobjs.save_interval, (XtTimerCallbackProc)savetemp, NULL);
}

/*--------------------------------------------------------------*/

void savelib(button, clientdata, nulldata)
  Widget        button;
  caddr_t       clientdata, nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   int ilib;

   if ((ilib = is_library(objectdata)) < 0) ilib = xobjs.numlibs - 1;

   if (xobjs.userlibs[ilib].number == 0) {
      Wprintf("No user objects to save.");
      return;
   }
   getgeneric(savebutton, button, savelib, ilib);
   popupprompt(button, "Enter name for library:", "\0", savelibrary,
	savebutton, False);
}

/*---------------------------------------------------------*/

void savelibrary(Widget w, int ilib)
{
   FILE *ps;
   char outname[150], *outptr;
   objectptr *wroteobjs, *libptr;
   short written = 0;
   char *uname = NULL;
   char *hostname = NULL;
   struct passwd *mypwentry = NULL;

   sscanf(_STR2, "%s", outname);
   if ((outptr = strrchr(outname, '/')) == NULL) outptr = outname;
   if (strchr(outptr, '.') == NULL) sprintf(outname, "%s.lps", _STR2);

   ps = fopen(outname, "w");
   if (ps == NULL) {
      Wprintf("Can't open PS file.");
      return;
   }

   fprintf(ps, "%%! PostScript set of library objects for XCircuit\n");
   fprintf(ps, "%%  Version: %2.1f\n", version);
   fprintf(ps, "%%  Library name is: %s\n", _STR2);
   uname = getenv((const char *)"USER");
   if (uname != NULL) mypwentry = getpwnam(uname);

   /* Check for both $HOST and $HOSTNAME environment variables.  Thanks */
   /* to frankie liu <frankliu@Stanford.EDU> for this fix.		*/
   
   if ((hostname = getenv((const char *)"HOSTNAME")) == NULL)
      if ((hostname = getenv((const char *)"HOST")) == NULL) {
	 if (gethostname(_STR, 149) != 0)
	    hostname = uname;
	 else
	    hostname = _STR;
      }

   if (mypwentry != NULL)
      fprintf(ps, "%%  Author: %s <%s@%s>\n", mypwentry->pw_gecos, uname,
		hostname);
   fprintf(ps, "%%\n\n%% XCircuitLib library objects\n");

   /* list of library objects already written */

   wroteobjs = (objectptr *) malloc (sizeof(objectptr));

   /* write all of the object definitions used, bottom up */

   for (libptr = xobjs.userlibs[ilib].library; libptr < xobjs.userlibs[ilib].library + 
        xobjs.userlibs[ilib].number; libptr++)
      printobjects(ps, *libptr, 0.0, 0.0, 0.0, 0, 0, &wroteobjs, &written, DEFAULTCOLOR);

   /* and the postlog */

   fprintf(ps, "\n%% EndLib\n");
   fclose(ps);
   sprintf(_STR, "Library %s saved.", outname);
   Wprintf(_STR);

   free(wroteobjs);
}

/*----------------------------------------------------------------------*/
/* Recursive routine to search the object hierarchy for fonts used	*/
/*----------------------------------------------------------------------*/

findfonts(objectptr writepage, short *fontsused) {
   genericptr *dfp;
   uchar *chp;
   int findex;

   for (dfp = writepage->plist; dfp < writepage->plist + writepage->parts; dfp++) {
      if (IS_LABEL(dfp)) {
         for (chp = TOLABEL(dfp)->string; *chp != '\0'; chp++) {
	    if ((short)(*chp) == TEXT_ESC && (findex = (short)(*(chp + 1)) -
		      FONT_START) >= 0) {
	       if (fontsused[findex] == 0) {
	          fontsused[findex] = 0x8000 | fonts[findex].flags;
	       }
	    }
	 }
      }
      else if (IS_OBJINST(dfp)) {
	 findfonts(TOOBJINST(dfp)->thisobject, fontsused);
      }
   }
}

/*----------------------------------------------------------------------*/
/* Main file saving routine						*/
/*----------------------------------------------------------------------*/

void savefile(localdata, mode) 
   objectptr	localdata;
   short	mode;
{
   FILE *ps, *pro;
   char outname[150], temp[150], prologue[150], *fname, *fptr;
   short written = 0, fontsused[256], i, page, multipage, savepage, stcount;
   objectptr *wroteobjs, writepage, *mpages;
   int *mpageno, findex;
   genericptr *dfp;
   float psscale, psnorm;
   float xmargin = 72.0, ymargin = 72.0;
   int bboxx, bboxy;
   time_t tdate;

   if (mode == CURRENT_PAGE)
      fname = xobjs.pagelist[areastruct.page]->filename;
   else {
      /* doubly-protected backup: protect against errors during file write */
      sprintf(outname, "%sB", xobjs.tempfile);
      rename(xobjs.tempfile, outname);
      fname = xobjs.tempfile;
   }

   if ((fptr = strrchr(fname, '/')) == NULL) fptr = fname;
   if ((mode == CURRENT_PAGE) && (strchr(fptr, '.') == NULL))
      sprintf(outname, "%s.ps", fname);
   else sprintf(outname, "%s", fname);

   ps = fopen(outname, "w");
   if (ps == NULL) {
      Wprintf("Can't open PS file.");
      return;
   }

   /* calculate the bounding box of the drawing */

   /* find margins */

   psscale = getpsscale(xobjs.pagelist[areastruct.page]->outscale, areastruct.page);
   if (xobjs.pagelist[areastruct.page]->pmode & 1) {
      if (xobjs.pagelist[areastruct.page]->orient == 90) {
	 xmargin = (xobjs.pagelist[areastruct.page]->pagesize.x -
		((float)localdata->height * psscale)) / 2;
	 ymargin = (xobjs.pagelist[areastruct.page]->pagesize.y -
		((float)localdata->width * psscale)) / 2;
      }
      else {
         xmargin = (xobjs.pagelist[areastruct.page]->pagesize.x -
		((float)localdata->width * psscale)) / 2;
         ymargin = (xobjs.pagelist[areastruct.page]->pagesize.y -
		((float)localdata->height * psscale)) / 2;
      }
   }

#ifdef SCHEMA
   /* find all the schematics which are subcircuits of the top-level-page */
   /* circuit and give them the same filename, so that they will be saved */
   /* together as a set.						  */

   if (areastruct.schemon) {
      Wprintf("Gathering all subcircuits. . .");
      findsubschems(areastruct.page, objectdata);
   }
#endif

   /* check for multiple-page output: get the number of pages */

   multipage = 0;
   mpages = (objectptr *) malloc (sizeof(objectptr));
   mpageno = (int *) malloc (sizeof(int));
   savepage = areastruct.page;

   for (page = 0; page < xobjs.pages; page++) {
      writepage = xobjs.pagelist[page]->pageobj;
      if (writepage != NULL)
	 if ((mode == ALL_PAGES) || (xobjs.pagelist[page]->filename
			== xobjs.pagelist[areastruct.page]->filename)) {
	    multipage++;
	    mpages = (objectptr *)realloc(mpages, multipage * sizeof(objectptr));
	    mpageno = (int *)realloc(mpageno, multipage * sizeof(int));
	    mpages[multipage - 1] = writepage;
	    mpageno[multipage - 1] = page;
	 }
   }
   if (multipage == 0) {
      Wprintf("Panic:  could not find this page in page list!");
      free (mpages);
      free (mpageno);
      fclose(ps);
      return;
   }

   /* Print the PostScript DSC Document Header */

   fprintf(ps, "%%!PS-Adobe-3.0");
   if (multipage == 1 && !(xobjs.pagelist[areastruct.page]->pmode & 1))
      fprintf(ps, " EPSF-3.0\n");
   else
      fprintf(ps, "\n");
   fprintf(ps, "%%%%Title: %s\n", xobjs.pagelist[areastruct.page]->filename);
   fprintf(ps, "%%%%Creator: Xcircuit v%2.1f\n", VERSION);
   tdate = time(NULL);
   fprintf(ps, "%%%%CreationDate: %s", asctime(localtime(&tdate)));
   fprintf(ps, "%%%%Pages: %d\n", multipage);
   if (xobjs.pagelist[areastruct.page]->orient == 90) {
      bboxx = (int)((float)localdata->height * psscale) + (int)xmargin + 4;
      bboxy = (int)((float)localdata->width * psscale) + (int)ymargin + 4;
   }
   else {
      bboxx = (int)((float)localdata->width * psscale) + (int)xmargin + 4;
      bboxy = (int)((float)localdata->height * psscale) + (int)ymargin + 4;
   }
   /* in order to reconstruct the page size, it is necessary that the   */
   /* bounding box extent + margin = page size.  This of course assumes */
   /* that the picture is centered on the page so margins are equal.    */

   if (xobjs.pagelist[areastruct.page]->pmode & 1) 
      fprintf(ps, "%%%%BoundingBox: %d %d %d %d\n",
	   xobjs.pagelist[areastruct.page]->pagesize.x - bboxx,
	   xobjs.pagelist[areastruct.page]->pagesize.y - bboxy, bboxx, bboxy);
   else
      fprintf(ps, "%%%%BoundingBox: %d %d %d %d\n", (int)xmargin - 4,
	   (int)ymargin - 4, bboxx, bboxy);
   for(i = 0; i <= fontcount; i++) fontsused[i] = 0;
   fprintf(ps, "%%%%DocumentNeededResources: ");
   stcount = 27;

   /* find all of the fonts used in this document */
   /* log all fonts which are native PostScript   */

   for (page = 0; page < multipage; page++) {
      writepage = mpages[page];
      findfonts(writepage, fontsused);
   }

   for(i = 0; i <= fontcount; i++) {
      if (fontsused[i] & 0x8000)
	 if ((fonts[i].flags & 0x8018) == 0x0) {
	    stcount += strlen(fonts[i].psname) + 6;
	    if (stcount > OUTPUTWIDTH) {
	       stcount = strlen(fonts[i].psname) + 6;
	       fprintf(ps, "\n%%%%+ ");
	    }
	    fprintf(ps, "font %s ", fonts[i].psname);
	 }
   }

   fprintf(ps, "\n%%%%EndComments\n");

   sprintf(prologue, "%s/%s", PROLOGUE_DIR, PROLOGUE_FILE);
   pro = fopen(prologue, "r");
   if (pro == NULL) {
      sprintf(prologue, "%s", PROLOGUE_FILE);
      pro = fopen(prologue, "r");
      if (pro == NULL) {
         Wprintf("Can't open prolog.");
	 free(mpages);
	 free(mpageno);
	 fclose(ps);
         return;
      }
   }

   /* write the prolog to the output */

   for(;;) {
      if (fgets(temp, 149, pro) == NULL) break;
      fputs(temp, ps);
   }
   fclose(pro);

   /* Special font handling */

   for (findex = 0; findex < fontcount; findex++) {

      /* Derived font slant */

      if ((fontsused[findex] & 0x032) == 0x032)
         fprintf(ps, "/%s /%s .167 fontslant\n\n",
		fonts[findex].psname, fonts[findex].family);

      /* Derived ISO-Latin1 encoding */

      if ((fontsused[findex] & 0xf80) == 0x100) {
	 char *fontorig = NULL;
	 short i;
	 /* find the original standard-encoded font (can be itself) */
	 for (i = 0; i < fontcount; i++) {
	    if (i == findex) continue;
	    if (!strcmp(fonts[i].family, fonts[findex].family) &&
		 ((fonts[i].flags & 0x03) == (fonts[findex].flags & 0x03)))
	       fontorig = fonts[i].psname;
	    if (fontorig == NULL) fontorig = fonts[findex].psname;
	 }
	 fprintf(ps, "/%s findfont dup length dict begin\n", fontorig);
	 fprintf(ps, "{1 index /FID ne {def} {pop pop} ifelse} forall\n");
	 fprintf(ps, "/Encoding ISOLatin1Encoding def currentdict end\n");
	 fprintf(ps, "/%s exch definefont pop\n\n", fonts[findex].psname);
      }

      /* To do:  ISO-Latin2 encoding */

      if ((fontsused[findex] & 0xf80) == 0x180) {
      }

      /* To do:  Special encoding */

      if ((fontsused[findex] & 0xf80) == 0x80) {
      }

      /* To do:  Vectored (drawn) font */

      if (fontsused[findex] & 0x8) {
      }
   }

   fprintf(ps, "%% XCircuit output starts here.\n\n");

   /* list of objects already written---maximum possible = # of builtins */

   wroteobjs = (objectptr *) malloc (sizeof(objectptr));

   for (page = 0; page < multipage; page++) {
      writepage = mpages[page];

      psnorm = xobjs.pagelist[mpageno[page]]->outscale;
      psscale = getpsscale(psnorm, mpageno[page]);
      if (xobjs.pagelist[mpageno[page]]->pmode & 1) {
         if (xobjs.pagelist[mpageno[page]]->orient == 90) {
	    xmargin = (xobjs.pagelist[areastruct.page]->pagesize.x -
		((float)writepage->height * psscale)) / 2;
	    ymargin = (xobjs.pagelist[areastruct.page]->pagesize.y -
		((float)writepage->width * psscale)) / 2;
         }
         else {
            xmargin = (xobjs.pagelist[areastruct.page]->pagesize.x -
		((float)writepage->width * psscale)) / 2;
            ymargin = (xobjs.pagelist[areastruct.page]->pagesize.y -
		((float)writepage->height * psscale)) / 2;
         }
      }

      /* write all of the object definitions used, bottom up */

      printobjects(ps, writepage, psnorm, xmargin, ymargin, page + 1,
	    mpageno[page], &wroteobjs, &written, DEFAULTCOLOR);
      fprintf(ps, "pgsave restore showpage\n\n");
      fflush(ps);
   }
   free (wroteobjs);
   free (mpages);

   /* Done! */

   fprintf(ps, "%%%%Trailer\n");
   fprintf(ps, "XCIRCsave restore\n");
   fprintf(ps, "%%%%EOF\n");
   fclose(ps);

   sprintf(_STR, "File %s saved (%d page%s).", fname, multipage,
		(multipage > 1 ? "s" : ""));
   Wprintf(_STR);

   /* remove the temporary redundant backup */
   if (mode == ALL_PAGES) {
      sprintf(outname, "%sB", xobjs.tempfile);
      unlink(outname);
   }
}

/*----------------------------------------------------*/
/* Write string to PostScript string, ignoring NO_OPs */
/*----------------------------------------------------*/

int nosprint(char *sptr)
{
   uchar *pptr, *qptr = _STR;
   int retval = 0;

   *qptr++ = '(';

   for (pptr = sptr; *pptr != '\0'; pptr++) {
      if (*pptr != NO_OP) *qptr++ = *pptr;
   }
   if (qptr == (uchar *)_STR + 1) {
      qptr--;
      retval = -1;
   }
   else {
      *qptr++ = ')';
      *qptr++ = ' ';
   }
   *qptr++ = '\0';

   return retval;
}

/*--------------------------------------*/
/* Write label segments to the output	*/
/*--------------------------------------*/

short writelabel(FILE *ps, uchar **chrhook, short *stcount, float scale,
	uchar *tmpstring)
{
   uchar *chrptr = *chrhook;
   short segs = 0;

   while (chrptr > tmpstring) {
      if (*(chrptr - 1) == TEXT_ESC) {
         short control = (short)*chrptr;

	 if (control == PARAM_END) {
	    sprintf(_STR, "v%d ", (int)(*(chrptr - 2)));
	    dostcount(ps, stcount, strlen(_STR));
	    chrptr -= 3;
	 }
         else if (control == SUBSCRIPT) {
	    sprintf(_STR, "{ss} ");
	    dostcount(ps, stcount, 5);
         }
         else if (control == SUPERSCRIPT) {
	    sprintf(_STR, "{Ss} ");
	    dostcount(ps, stcount, 5);
         }
         else if (control == NORMALSCRIPT) {
            sprintf(_STR, "{ns} ");
            dostcount(ps, stcount, 5);
         }
         else if (control == UNDERLINE) {
            sprintf(_STR, "{ul} ");
            dostcount (ps, stcount, 5);
         }
         else if (control == OVERLINE) {
            sprintf(_STR, "{ol} ");
            dostcount (ps, stcount, 5);
         }
	 else if (control == NOLINE) {
            sprintf(_STR, "{} ");
            dostcount (ps, stcount, 3);
	 }
         else if (control == HALFSPACE) {
            sprintf(_STR, "{hS} ");
            dostcount (ps, stcount, 5);
         }
         else if (control == QTRSPACE) {
	    sprintf(_STR, "{qS} ");
	    dostcount (ps, stcount, 5);
         }
         else if (control == BACKSPACE) {
	    short bsctr = 1;
		
	    for (chrptr -= 2; (short)(*chrptr) == BACKSPACE &&
		      (short)(*(chrptr - 1)) == TEXT_ESC; chrptr-=2) bsctr++;
	    *(++chrptr) = '\0';
	    sprintf(_STR, "{(%s) bs} ", chrptr - bsctr);
	    dostcount (ps, stcount, 6 + strlen(chrptr - bsctr));
	    chrptr++;
         }
         else if (control >= FONT_START) {
	    char *chtmp, *lastfont = fonts[control - FONT_START].psname;
	    for (chtmp = chrptr - 2; chtmp > (char *)tmpstring; chtmp--)
	       if ((short)(*chtmp - 1) == TEXT_ESC && (short)*chtmp >= FONT_START) break;
	    if (chtmp <= (char *)tmpstring + 1) {
	       sprintf(_STR, "{/%s %5.3f cf} ", lastfont, scale);
	       dostcount(ps, stcount, 13 + strlen(lastfont));
	    }
            else {
	       sprintf(_STR, "{/%s cf} ", lastfont);
	       dostcount(ps, stcount, 7 + strlen(lastfont));
            }
         }
         else if (control == END_SCALE) {
	    float locscale;
	    char *chtmp, *lastfont = fonts[*(chrptr - 6) - FONT_START].psname;
	    *(chrptr - 1) = '\0';
	    sscanf(chrptr - 5, "%f", &locscale);
	    sprintf(_STR, "{/%s %5.3f cf} ", lastfont, locscale);
	    dostcount(ps, stcount, 13 + strlen(lastfont));
	    chrptr -= 5;
	 }
         *(--chrptr) = '\0';
      }
      else {
	 while (chrptr > tmpstring && *(chrptr - 1) != TEXT_ESC) chrptr--;
	 segs += nosprint((chrptr == tmpstring) ? chrptr : chrptr + 1);
      }
      fputs(_STR, ps);
      segs++;
   }
   *chrhook = chrptr;

   return segs;
}

/*--------------------------------------------------------------*/
/* Routine to write all the label segments as stored in _STR	*/
/*--------------------------------------------------------------*/

writelabelsegs(FILE *ps, short *stcount, uchar *tmpptr)
{
   uchar *chrptr;
   short lsegs = 0;

   chrptr = tmpptr;
   while (*chrptr != '\0') {
      if (*chrptr++ == TEXT_ESC) lsegs = 1;
   }
   if (lsegs > 0) {
      fprintf(ps, "{");
      stcount++;
   }
   /* Need this in case of a null parameter string */
   if (*tmpptr == '\0') *(++chrptr) = '\0';
   writelabel(ps, &chrptr, stcount, 1.0, tmpptr);
   if (lsegs > 0) {
      fprintf(ps, "}");
      stcount++;
   }
}

/*--------------------------------------------------------------------------*/
/* recursive routine to print out the library objects used in this drawing, */
/* starting at the bottom of the object hierarchy so that each object is    */
/* defined before it is called.  A list of objects already written is 	    */
/* maintained so that no object is written twice. 			    */
/*--------------------------------------------------------------------------*/

void printobjects(ps, localdata, psnorm, xmargin, ymargin, page, mpage,
	wrotelist, written, ccolor)
  FILE *ps;
  objectptr	localdata, **wrotelist;
  float		psnorm, xmargin, ymargin;
  short		page, mpage, *written;
  int		ccolor;
{
   genericptr *gptr, *savegen;
   objectptr *optr;
   pointlist savept;
   Boolean already, cchangeflag = False;
   short stcount;
   int xaddin, yaddin;
   int curcolor = ccolor;
   float psscale = getpsscale(psnorm, mpage);

   /* first, get a total count of all objects and give warning if large */

   if (localdata->parts > 255) {
      sprintf(_STR, "Warning: \"%s\" may exceed printer's PS limit for definitions",
	    localdata->name);
      Wprintf(_STR);
   }
	   
#ifdef SCHEMA
#define height height2
#define _width width2
#else
#define _width width
#endif

   /* search for all object definitions needed */

   for (gptr = localdata->plist; gptr < localdata->plist + localdata->parts; gptr++)
      if (IS_OBJINST(gptr))
         printobjects(ps, TOOBJINST(gptr)->thisobject, 0.0, 0.0, 0.0, 0, 0,
		wrotelist, written, curcolor);

   if (psscale != 0) {  /* on topmost object of page */
      char *rootptr = strrchr(xobjs.pagelist[mpage]->filename, '/');
      if (rootptr == NULL) rootptr = xobjs.pagelist[mpage]->filename;
      else rootptr++;

      /* If the page label is just the root name of the file, or has been left   */
      /* as "Page n" or "Page_n", just do the normal page numbering.  Otherwise, */
      /* write out the page label explicitly.					 */

      if ((!strcmp(rootptr, localdata->name)) || (strchr(localdata->name, ' ')
		!= NULL) || (strstr(localdata->name, "Page_") != NULL))
         fprintf (ps, "%%%%Page: %d %d\n", page, page);
      else
         fprintf (ps, "%%%%Page: %s %d\n", localdata->name, page);

      if (xobjs.pagelist[mpage]->orient == 90)
         fprintf (ps, "%%%%PageOrientation: Landscape\n");
      else
         fprintf (ps, "%%%%PageOrientation: Portrait\n");
     
      fprintf (ps, "/pgsave save def bop\n");

#ifdef SCHEMA
      if (localdata->symschem != NULL) {
	 fprintf(ps, "%% %s is_symbol\n", localdata->symschem->name);
      }
#endif

      xaddin = (int)(xmargin / psscale) - localdata->lowerleft.x;
      yaddin = (int)(ymargin / psscale) - localdata->lowerleft.y;
      fprintf(ps, "%% %hd %hd offsets\n", xaddin, yaddin);

      if (xobjs.pagelist[mpage]->drawingscale.x != 1
		|| xobjs.pagelist[mpage]->drawingscale.y != 1)
         fprintf(ps, "%% %hd:%hd drawingscale\n", xobjs.pagelist[mpage]->drawingscale.x,
	 	xobjs.pagelist[mpage]->drawingscale.y);

      if (xobjs.pagelist[mpage]->gridspace != 32
		|| xobjs.pagelist[mpage]->snapspace != 16)
         fprintf(ps, "%% %4.2f %4.2f gridspace\n", xobjs.pagelist[mpage]->gridspace,
		xobjs.pagelist[mpage]->snapspace);

      if (xobjs.pagelist[mpage]->background.name != (char *)NULL) {
	 float iscale = (xobjs.pagelist[mpage]->coordstyle == CM) ? CMSCALE : INCHSCALE;
	 fprintf(ps, "%d %d insertion\n",
		(int)((float)xaddin * iscale), (int)((float)yaddin * iscale));
	 savebackground(ps, xobjs.pagelist[mpage]->background.name);
	 fprintf(ps, "end_insert\n");
      }

      if (xobjs.pagelist[mpage]->orient == 90)
         fprintf(ps, "90 rotate %d %d translate\n", (int)(ymargin - xmargin),
	     -((int)((float)localdata->height * psscale) + 
	     (int)(xmargin + ymargin)));

      fprintf(ps, "%5.4f ", psnorm);
      switch(xobjs.pagelist[mpage]->coordstyle) {
	 case FRAC_INCH: case DEC_INCH:
	    fprintf(ps, "inchscale\n");
	    break;
	 case CM:
	    fprintf(ps, "cmscale\n");
	    break;
      };
      fprintf(ps, "%5.4f setlinewidth\n\n", 1.3 *
      	     xobjs.pagelist[mpage]->wirewidth);
   }

   /* search among the list of objects already written to the output */

   already = False;
   for (optr = *wrotelist; optr < *wrotelist + *written; optr++)
      if (*optr == localdata) {
	  already = True;
	  break;
      }

   if (!already) {

      /* update the list of objects already written to the output */
      if (psscale == 0) {
	 *wrotelist = (objectptr *)realloc(*wrotelist, (*written + 1) * 
		sizeof(objectptr));
         *(*wrotelist + *written) = localdata;
         (*written)++;
      }

      if (psscale == 0) {
	 fprintf(ps, "/%s {\n", localdata->name);
#ifdef SCHEMA
	 sprintf(_STR, "%% %hd %hd %hd %hd bbox\n", localdata->lleft2.x,
	      localdata->lleft2.y, localdata->_width, localdata->height);
#else
	 sprintf(_STR, "%% %hd %hd %hd %hd bbox\n", localdata->lowerleft.x,
	      localdata->lowerleft.y, localdata->_width, localdata->height);
#endif
	 fputs(_STR, ps);
	 if (localdata->hidden == True) fprintf(ps, "%% hidden\n");

#ifdef SCHEMA
         if (localdata->symschem != NULL) {
	    fprintf(ps, "%% %s is_schematic\n", localdata->symschem->name);
         }
#endif

	 /* Check for parameters and default values */
         if (localdata->params != NULL) {
	    uchar tmpstring[250];
	    short i;
	    stcount = 0;
	    for (i = 0; i < localdata->num_params; i++) {
	       if (localdata->params[i]->type == XC_STRING) {
	          strcpy(tmpstring, localdata->params[i]->pdefault);
		  writelabelsegs(ps, &stcount, tmpstring);
		  fprintf(ps, " ");
	       }
	       else {	/* XC_SHORT */
		  fprintf(ps, "%hd ",
			(short)(*((short *)(localdata->params[i]->pdefault))));
	       }
	       stcount++;
	    }
	    sprintf(_STR, "%d beginparm\n", localdata->num_params);
	    dostcount (ps, &stcount, strlen(_STR));
	    fputs(_STR, ps);
	 }
	 else 
	    fprintf(ps, "begingate\n");
      }

      if (psscale == 0) {
	 xaddin = yaddin = 0;
      }


#ifdef SCHEMA
      /* specify whether this is a fundamental circuit schematic type */
      if (areastruct.schemon && localdata->schemtype == FUNDAMENTAL) {
	 fprintf(ps, "%% fundamental\n");
      }

#undef height
#endif
#undef _width

      /* write all the elements in order */

      for (savegen = localdata->plist; savegen < localdata->plist +
	   localdata->parts; savegen++) {

	 /* change current color if different */

	 if ((*savegen)->color != curcolor) {
	    short i;

	    if (cchangeflag == True) {
	       fprintf(ps, "sce\n");
	       cchangeflag = False;
	       curcolor = ccolor;
	    }
	    if ((*savegen)->color != DEFAULTCOLOR) {
	       for (i = 0; i < number_colors; i++) {
	          if (colorlist[i].color.pixel == (*savegen)->color) {
	             fprintf(ps, "%4.3f %4.3f %4.3f scb\n",
			   (float)colorlist[i].color.red   / 65535,
			   (float)colorlist[i].color.green / 65535,
			   (float)colorlist[i].color.blue  / 65535);
		     break;
	          }
	       }
	       if (i < number_colors) {
	          curcolor = (*savegen)->color;
	          cchangeflag = True;
	       }
	    }
	 }

	 stcount = 0;
	 switch((*savegen)->type) {

	    case(POLYGON):
               sprintf(_STR, "%hd %5.2f ", TOPOLY(savegen)->style,
		     TOPOLY(savegen)->width);
	       stcount = strlen(_STR);
	       fputs(_STR, ps);
               for (savept = TOPOLY(savegen)->points; savept < TOPOLY(savegen)->
		      points + TOPOLY(savegen)->number; savept++) {
		  xypmat(savept);
	       }
	       sprintf(_STR, "%hd polygon\n", TOPOLY(savegen)->number);
	       dostcount (ps, &stcount, strlen(_STR));
	       fputs(_STR, ps);
	       break;

	    case(PATH): {
	       XPoint *startpoint;
	       XfPoint *testpt;
	       genericptr *pgen;
	       pgen = TOPATH(savegen)->plist;
	       switch((*pgen)->type) {
		  case ARC:
		     xyfmat(TOARC(pgen)->points[0]);
		     break;
		  case POLYGON:
		     xymat(TOPOLY(pgen)->points[0]);
		     break;
		  case SPLINE:
		     xyfmat(TOSPLINE(pgen)->ctrl[0]);
		     break;
	       }
	       dostcount(ps, &stcount, 9);
	       fprintf(ps, "beginpath\n");
	       for (pgen = TOPATH(savegen)->plist; pgen < TOPATH(savegen)->plist
			+ TOPATH(savegen)->parts; pgen++) {
	          switch((*pgen)->type) {
		     case ARC:
			if (abs(TOARC(pgen)->radius) == TOARC(pgen)->yaxis) {
			   xymat(TOARC(pgen)->position);
			   fprintf(ps, "%hd ", abs(TOARC(pgen)->radius));
			   if (TOARC(pgen)->radius < 0)
			      fprintf(ps, "%3.2f %3.2f arcn\n",
			         TOARC(pgen)->angle2, TOARC(pgen)->angle1);
			   else
			      fprintf(ps, "%3.2f %3.2f arc\n",
			         TOARC(pgen)->angle1, TOARC(pgen)->angle2);
			}
		        else {
			   xymat(TOARC(pgen)->position);
			   fprintf(ps, "%hd %hd ",
			      abs(TOARC(pgen)->radius), TOARC(pgen)->yaxis);
			   if (TOARC(pgen)->radius < 0)
			      fprintf(ps, "%3.2f %3.2f nellip\n",
			         TOARC(pgen)->angle2, TOARC(pgen)->angle1);
			   else
			      fprintf(ps, "%3.2f %3.2f pellip\n",
			         TOARC(pgen)->angle1, TOARC(pgen)->angle2);
			}
			break;
		     case POLYGON:
               		for (savept = TOPOLY(pgen)->points + TOPOLY(pgen)->number
				- 1; savept > TOPOLY(pgen)->points; savept--) {
			   xypmat(savept);
	       		}
	       		sprintf(_STR, "%hd polyc\n", TOPOLY(pgen)->number - 1);
	       		dostcount (ps, &stcount, strlen(_STR));
	       		fputs(_STR, ps);
	       		break;
		     case SPLINE:
			xymat(TOSPLINE(pgen)->ctrl[1]);
			xymat(TOSPLINE(pgen)->ctrl[2]);
			xymat(TOSPLINE(pgen)->ctrl[3]);
			fprintf(ps, "curveto\n");
			break;
		  }
	       }
	       fprintf(ps, "%hd %5.2f endpath\n", TOPATH(savegen)->style,
			TOPATH(savegen)->width);
	       } break;
	    case(SPLINE):
	       fprintf(ps, "%hd %5.2f ",
	          TOSPLINE(savegen)->style, TOSPLINE(savegen)->width);
	       xymat(TOSPLINE(savegen)->ctrl[1]);
	       xymat(TOSPLINE(savegen)->ctrl[2]);
	       xymat(TOSPLINE(savegen)->ctrl[3]);
	       xymat(TOSPLINE(savegen)->ctrl[0]);
	       fprintf(ps, "spline\n");
	       break;

	    case(ARC):
	       if (abs(TOARC(savegen)->radius) == TOARC(savegen)->yaxis) {
                  fprintf(ps, "%hd %5.2f ",
			TOARC(savegen)->style, TOARC(savegen)->width);
		  xymat(TOARC(savegen)->position);
                  fprintf(ps, "%hd %3.2f %3.2f xcarc\n", abs(TOARC(savegen)->radius),
			TOARC(savegen)->angle1, TOARC(savegen)->angle2);
	       }
	       else {
                  fprintf(ps, "%hd %5.2f ",
			TOARC(savegen)->style, TOARC(savegen)->width);
		  xymat(TOARC(savegen)->position);
                  fprintf(ps, "%hd %hd %3.2f %3.2f ellipse\n",
			abs(TOARC(savegen)->radius), TOARC(savegen)->yaxis,
			TOARC(savegen)->angle1, TOARC(savegen)->angle2);
	       }
	       break;

	    case(OBJECT):
	       {objinstptr sobj = TOOBJINST(savegen);
	       short rotation = sobj->rotation;
	       if (rotation < 0) rotation = (rotation + 1) * ROT_UNIT - 1;
	       else rotation = 360 - ((rotation - 1) * ROT_UNIT);
	       if (rotation == 360) rotation = 0;
	       sprintf(_STR, "%3.2f %hd ", sobj->scale, rotation);
	       dostcount(ps, &stcount, strlen(_STR));
	       fputs(_STR, ps);
	       xymat(sobj->position);

	       /* check for parameters */
	       if (sobj->params != NULL) {
		  short i, iend;
		  uchar tmpstring[250];

		  for (iend = sobj->num_params; iend > 0; iend--)
		     if (sobj->params[iend - 1] != NULL) break;

		  if (iend) {
		     fprintf(ps, "[");
		     stcount++;
		  }
		  for (i = 0; i < iend; i++) {
		     if (sobj->thisobject->params[i]->type == XC_STRING) {
			if (sobj->params[i] == NULL)     /* use the default */
			   strcpy(tmpstring,
				  (char *)sobj->thisobject->params[i]->pdefault);
			else
	                   strcpy(tmpstring, (char *)sobj->params[i]);
		        writelabelsegs(ps, &stcount, tmpstring);
		     }
		     else {	/* XC_SHORT */
			if (sobj->params[i] == NULL) {   /* use the default */
			   sprintf(_STR, "%hd ",
				(short)(*((short *)(localdata->params[i]->pdefault))));
			}
			else {
			   sprintf(_STR, "%hd ", (short)(*((short *)sobj->params[i])));
			}
			dostcount(ps, &stcount, strlen(_STR));
	       		fputs(_STR, ps);
		     }
		  }
		  if (iend) {
		     fprintf(ps, "] ");
		     stcount += 2;
		  }
	       }
	       fprintf(ps, "%s\n", TOOBJINST(savegen)->thisobject->name);
	       }break;
            
	    case(LABEL):
	       {uchar tmpstring[250];
	       uchar *chrptr, *tmpptr = tmpstring;
	       short segs = 0, rotation;

               /* sprintf but with modifications for special characters */
	       /* (also count label segments while writing the string) */

	       chrptr = TOLABEL(savegen)->string;
	       for (; *chrptr != '\0'; chrptr++) {

		  /* Replace parameter by its parameter number */
	
		  if (*(chrptr - 1) == TEXT_ESC && *chrptr == PARAM_START) {
		     short value;
		     *tmpptr++ = *chrptr;
		     parampos(localdata, NULL, chrptr - 1, NULL, &value);
		     *tmpptr++ = (uchar)value + 1;
		     while(*chrptr != TEXT_ESC || *(chrptr + 1) != PARAM_END) {
			if (*chrptr == TEXT_ESC) {
			   segs++;
			   if (*(chrptr + 2) != TEXT_ESC) segs++;
			}
			chrptr++;
		     }
		     *tmpptr++ = *chrptr++;
		  }

	          /* Extended character set (non-ASCII) */

		  if (chrptr == TOLABEL(savegen)->string
			|| (*(chrptr - 1) != TEXT_ESC)) {
	             if (*chrptr == ')' || *chrptr == '(' || *chrptr == '\\'
		      	    || *chrptr > (char)126)
			*tmpptr++ = '\\';
	             if (*chrptr > (char)126) {
		        sprintf(tmpptr, "%3o", (int)(*chrptr));
	 	        tmpptr += 3; 
	             }
		  }
		  if (*chrptr <= (char)126) *tmpptr++ = *chrptr;
	       }
	       *tmpptr = '\0';

	       /* Search label string for instances of redundant font changes */
	       /* and delete them.  Also look for instances of no text.	*/

	       for (chrptr = tmpstring; chrptr < tmpstring + strlen(tmpstring);
		      chrptr++) {
	          if ((short)(*chrptr) == TEXT_ESC && (short)(*(chrptr + 1))
			  >= FONT_START) {
	             uchar *chtmp;
	             for (chtmp = chrptr + 2; (short)(*chtmp) == TEXT_ESC &&
			      *chtmp != '\0'; chtmp += 2) {
		        if ((short)(*chtmp) == TEXT_ESC && (short)(*(chtmp + 1))
			      >= FONT_START) {
		           uchar *chclose;
		           *(chrptr + 1) = *(chtmp + 1);
		           for (chclose = chtmp; chclose < tmpstring +
			        strlen(tmpstring); chclose++)
			      *chclose = *(chclose + 2);
		           chtmp -= 2;
		        }
	             }
	             /* no text after font change */
	             if (*chtmp == '\0') {
		        *chrptr = '\0'; 
		        break;
	             }
	             else chrptr = chtmp;
	          }
	       }

	       /* New style label always starts with a font and scale */

	       chrptr = tmpstring + strlen(tmpstring) - 1;
	       segs += writelabel(ps, &chrptr, &stcount, 
			TOLABEL(savegen)->scale, tmpstring);
	       rotation = 360 - (TOLABEL(savegen)->rotation - 1) * ROT_UNIT;
	       if (rotation == 360) rotation = 0; 
	       if (segs > 0) {
                  sprintf(_STR, "%hd %hd %hd ", segs,
		      TOLABEL(savegen)->justify, rotation);
		  dostcount(ps, &stcount, strlen(_STR));
	          fputs(_STR, ps);
		  xymat(TOLABEL(savegen)->position);

#ifdef SCHEMA
		  if (areastruct.schemon) {
		     switch(TOLABEL(savegen)->pin) {
		        case LOCAL:
			   strcpy(_STR, "pinlabel\n"); break;
			case GLOBAL:
			   strcpy(_STR, "pinglobal\n"); break;
			case INFO:
			   strcpy(_STR, "infolabel\n"); break;
			default:
		     	   strcpy(_STR, "label\n");
		     }
		  }
		  else strcpy(_STR, "label\n");
#else
		  strcpy(_STR, "label\n");
#endif
		  dostcount(ps, &stcount, strlen(_STR));
	          fputs(_STR, ps);
	       }
	       }break;
         }

      }

      /* if the color has been changed, end the current color context */

      if (cchangeflag == True) fprintf(ps, "sce\n");

      /* if this is not the top level, end the object definition */

      if (psscale == 0) fprintf(ps, "endgate\n} def\n\n");

   }
}

/*----------------------------------------------------------------------*/
