/*----------------------------------------------------------------------*/
/* parameter.c								*/
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	*/
/*----------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/26/99    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "xcircuit.h"

extern Globaldata xobjs;
extern Clientdata areastruct;
extern objectpair *pushlist;
extern short pushes;
extern short textpos, textend;

oparamptr parampos(objectptr, labelptr, char *, short *, short *);
oparamptr paramcross(objectptr, labelptr);

/*-------------------------------------------------------------------------*/

/* transformation types are enumerated as follows: */

enum transform {IDENT = 0, SUBST, INSERT, DELETE};

/* (recursive) structure for a transition between any two letters */

struct pathnode {
   char   lfrom[3], lto[3];     /* Letters involved in transform */
   int	  from, to;		/* Positions of transform	 */	  
   int    cost;                 /* Levenshtein distance          */
   struct pathnode *previous;   /* Previous path taken           */
   struct pathnode *next;       /* Next path to take             */
   int    transform;            /* Type of transformation made   */
};

/*--------------------------------------------------------------*/
/* Reattach parameter hooks from original string to new one	*/
/*--------------------------------------------------------------*/

void rehook(objectptr localdata, char *origstr, char *newstr)
{
   short paramno, i;
   oparamptr ops;
   char *tmpptr = origstr;

   if (localdata->params == NULL) return;   /* Any parameters? */

   while (*tmpptr != '\0') tmpptr++;  /* tmpptr points to end of string */

   for (paramno = 0; paramno < localdata->num_params; paramno++) {
      ops = localdata->params[paramno];
      for (i = 0; i < ops->number; i++) {
	 if ((char *)ops->position[i] < tmpptr &&
	    	(char *)ops->position[i] >= origstr)
	    ops->position[i] += (int)(newstr - origstr);
      }
   }
}

/*--------------------------------------------------------------*/
/* Adjust parameter hooks when string is altered by having	*/
/* shifted by "width" characters after position "ppos".   	*/
/*--------------------------------------------------------------*/

void hookexpand(objectptr localdata, char *ppos, short width)
{
   short paramno, i;
   oparamptr ops;
   char *tmpptr = ppos;

   if (localdata->params == NULL) return;   /* Any parameters? */

   while (*tmpptr++ != '\0');	/* tmpptr points to end of string */
   tmpptr -= width;		/* points to original end of string */

   for (paramno = 0; paramno < localdata->num_params; paramno++) {
      ops = localdata->params[paramno];
      for (i = 0; i < ops->number; i++)
	 if ((char *)ops->position[i] < tmpptr &&
	    	(char *)ops->position[i] >= ppos)
	    ops->position[i] += width;
   }
}

/*--------------------------------------------------------------*/
/* Parameterize a label string.					*/
/*--------------------------------------------------------------*/

void makeparam(labelptr thislabel)
{
   oparamptr ops;
   uchar *spos, *newstring, *npos;

   newstring = (uchar *)malloc(strlen(thislabel->string) + 5);
   spos = thislabel->string;
   npos = newstring;

   /* make sure this does not overlap another parameter */

   if (paramcross(objectdata, thislabel) != NULL) {
      Wprintf("Parameters cannot be nested!");
      textend = 0;
      return;
   }

   if (textend > 0 && textend < textpos) {
      short plen = textpos - textend;
      memmove(npos, spos, textend);
      npos += textend;
      spos += textend;
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_START;
      memmove(npos, spos, plen);
      npos += plen;
      spos += plen;
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_END;
      memmove(npos, spos, strlen(spos) + 1);
   }
   else {
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_START;
      strcpy(npos, spos);
      npos += strlen(spos);
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_END;
      *(npos++) = '\0';
   }
   
   rehook(objectdata, (char *)thislabel->string, (char *)newstring);
   hookexpand(objectdata, npos, 4);
   free(thislabel->string);
   thislabel->string = newstring;
   spos = thislabel->string + 2;

   /* insert temporary string end delimiter in place of TEXT_ESC */
   if (textend > 0 && textend < textpos) {
      spos = thislabel->string + textend + 2;
      *(spos - textend + textpos) = '\0';
   }

   if (objectdata->params == NULL) {
      objectdata->num_params = 1;
      objectdata->params = (oparamptr *)malloc(sizeof(oparamptr));
   }
   else {
      objectdata->num_params++;
      objectdata->params = (oparamptr *)realloc(objectdata->params,
		objectdata->num_params * sizeof(oparamptr));
   }
   *(objectdata->params + objectdata->num_params - 1) = (oparamptr)
	malloc(sizeof(oparam));

   ops = *(objectdata->params + objectdata->num_params - 1);
   ops->type = XC_STRING;
   ops->number = 1;

   ops->pdefault = (void *)malloc(strlen(spos) + 1);
   ops->position = (uchar **)malloc(sizeof(void *));
   *(ops->position) = spos - 2;
   strcpy(ops->pdefault, spos);
   if (textend > 0 && textend < textpos)
      *(thislabel->string + textpos + 2) = TEXT_ESC;
   /* printf("default string is %s\n", ops->pdefault); */

   textend = 0;
}

/*--------------------------------------------------------------*/
/* Search and destroy the selected parameter in all instances 	*/
/* of the specified object.					*/
/*--------------------------------------------------------------*/

void searchinst(objectptr topobj, objectptr refobj, short pnum)
{
   objinstptr tinst;
   genericptr *pgen;
   short k;

   if (topobj == NULL) return;

   for (pgen = topobj->plist; pgen < topobj->plist + topobj->parts; pgen++) {
      if ((*pgen)->type == OBJECT) {
	 tinst = TOOBJINST(pgen);
	 if (tinst->thisobject == refobj) {
	    for (k = pnum; k < tinst->num_params - 1; k++)
	       *(tinst->params + k) = *(tinst->params + k + 1);
	    tinst->num_params--;
	 }
      }
   }
}

/*--------------------------------------------------------------*/
/* Remove parameterization from a label string or substring.	*/
/*--------------------------------------------------------------*/

void unmakeparam(labelptr thislabel, short tptr)
{
   oparamptr ops;
   short i, j, k, tend;
   uchar *spos;

   /* make sure there is a parameter here */

   if ((ops = parampos(objectdata, NULL, (char *)thislabel->string + tptr,
		&j, &i)) == NULL) {
      Wprintf("There is no parameter here.");
      return;
   }

   tend = tptr;
   while (*(thislabel->string + tend) != TEXT_ESC || *(thislabel->string + tend + 1)
	!= PARAM_END) tend++;

   /* Step 1: Pad over parameter begin and end points with NO_OP characters */

   spos = thislabel->string + tptr;
   *(spos++) = NO_OP;
   *(spos++) = NO_OP;
   spos += tend - tptr - 2;
   *(spos++) = NO_OP;
   *(spos++) = NO_OP;
   
   /* Step 2: Remove object parameter */

   if (ops != *(objectdata->params + i)) {
      printf("Something wrong in return values from parampos()\n");
   }
   else {
      free(ops->pdefault);
      free(ops->position);
      free(ops);
      for (k = i; k < objectdata->num_params - 1; k++)
         *(objectdata->params + k) = *(objectdata->params + k + 1);

      objectdata->num_params--;
      if (objectdata->num_params == 0)
         free(objectdata->params);
   }

   /* Step 3: Search and destroy on all instances */
   /* This is obnoxious and complicated, but it has to be done. . . */

   for (k = 0; k < xobjs.pages; k++) {
      searchinst(xobjs.pagelist[k]->pageobj, objectdata, i);
   }
   for (j = 0; j < xobjs.numlibs; j++) {
      for (k = 0; k < xobjs.userlibs[j].number; k++) {
         searchinst(*(xobjs.userlibs[j].library + k), objectdata, i);
      }
   }
   for (k = 0; k < xobjs.delbuffer.number; k++) {
      searchinst(*(xobjs.delbuffer.library + k), objectdata, i);
   }
}

/*------------------------------------------------------*/
/* Wrapper for unmakeparam()				*/
/*------------------------------------------------------*/

void unparameterize()
{
   short *fselect, i, ts, te;
   labelptr settext;

   if (!checkselect(LABEL)) objectselect(LABEL);
   if (!checkselect(LABEL)) return;
   if (areastruct.selects == 1 && textend > 0 && textend < textpos) {
      settext = SELTOLABEL(areastruct.selectlist);
      for (i = textend; i >= 0; i--)
	 if (*(settext->string + i) == TEXT_ESC)
	    if (*(settext->string + i + 1) == PARAM_START)
	       unmakeparam(settext, i);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            settext = SELTOLABEL(fselect);
     	    for (i = 0; i < strlen((char *)settext->string); i++)
	       if (*(settext->string + i) == TEXT_ESC)
	          if (*(settext->string + i + 1) == PARAM_START)
	             unmakeparam(settext, i);
	 }
      }
   }
}

/*--------------------------------------------------------------*/
/* Wrapper for makeparam()					*/
/*--------------------------------------------------------------*/

void parameterize()
{
   short *fselect;
   labelptr settext;

   if (!checkselect(LABEL)) objectselect(LABEL);
   if (!checkselect(LABEL)) return;
   for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
      if (SELECTTYPE(fselect) == LABEL) {
         settext = SELTOLABEL(fselect);
         makeparam(settext);
      }
   }
   objectdeselect();
}

/*----------------------------------------------------------------------*/
/* Find the object to which a parameter points				*/
/*----------------------------------------------------------------------*/

genericptr findparam(objectptr locobj, void *posptr, uchar ptype)
{
   genericptr *pgen;
   short k;

   for (k = 0; k < locobj->parts; k++) {
      pgen = locobj->plist + k;
      if ((ptype == XC_STRING) && ((*pgen)->type == LABEL)) {
	 labelptr plab = TOLABEL(pgen);
	 uchar *tmpstr = plab->string;
	 int plen = strlen((char *)plab->string);
	 if ((uchar *)posptr >= plab->string && (uchar *)posptr
		< plab->string + strlen((char *)plab->string))
	    break;
      }
      else if (ptype == XC_SHORT) {
	 if ((*pgen)->type == POLYGON) {	/* to be completed */
	    polyptr ppoly = TOPOLY(pgen);
	 }
      }
   }

   if (k < locobj->parts) return *pgen;
   else return NULL;
}

/*----------------------------------------------------------------------*/
/* Find and return a pointer to the parameter (if any) pointing to the  */
/* specified label at given text position (or "textpos" if NULL).	*/
/*----------------------------------------------------------------------*/

oparamptr parampos(objectptr tobj, labelptr tlab, char *tptr, short *jret,
	short *iret)
{
   oparamptr ops;
   char *pptr, *sptr = (tptr != NULL) ? tptr : (char *)tlab->string + textpos;
   short i, j, plen;

   for (i = 0; i < tobj->num_params; i++) {
      ops = *(tobj->params + i);
      if (ops->type == XC_STRING) {
         plen = strlen((char *)ops->pdefault) + 4;
         for (j = 0; j < ops->number; j++) {
	    pptr = (char *)(*((char **)ops->position + j));
	    if (sptr >= pptr && sptr < (pptr + plen)) {
	       if (jret != NULL) *jret = j;
	       if (iret != NULL) *iret = i;
	       return ops;
	    }
	 }
      }
   }   
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Same routine as above, but looks for a parameter overlapping the	*/
/* textend <--> textpos space.						*/
/*----------------------------------------------------------------------*/

oparamptr paramcross(objectptr tobj, labelptr tlab)
{
   oparamptr ops;
   char *pptr, *qptr, *tptr = (char *)tlab->string + textpos;
   char *sptr = (char *)tlab->string + textend; 
   short i, j, plen;

   for (i = 0; i < tobj->num_params; i++) {
      ops = *(tobj->params + i);
      if (ops->type == XC_STRING) {
         plen = strlen((char *)ops->pdefault) + 4;
         for (j = 0; j < ops->number; j++) {
	    pptr = (char *)(*((char **)ops->position + j));
	    if (pptr < tptr && (pptr + plen) > sptr)
	       return ops;
	 }
      }
   }   
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Parameter substitution and default value restoration			*/
/*----------------------------------------------------------------------*/

void psubstitute(objinstptr localdata)
{
   uchar nops = localdata->thisobject->num_params;
   oparamptr ops, sops;
   int i, j, k;
   genericptr pgen;

   if (localdata->thisobject->params == NULL) return;	/* object has no parameters */

   for (i = 0; i < nops; i++) {
      ops = *(localdata->thisobject->params + i);
      for (j = 0; j < ops->number; j++) {
         if (ops->type != XC_STRING) {
	    short *parmptr;
	    if (localdata->num_params <= i)
	       parmptr = (short *)ops->pdefault;
	    else if (*(localdata->params + i) == NULL)
	       parmptr = (short *)ops->pdefault;
	    else
	       parmptr = (short *)(*(localdata->params + i));
	    *((short *)(ops->position[j])) = *parmptr;
	    /* printf("Substituting parameter value %hd\n", *parmptr); */
	 }
	 else {
	    /* This is a little tricky---if size(substitution) != size(default), */
	    /* have to either pad string with NO_OP characters or widen string   */
	    /* to fit the substitution and pad the default with NO_OP characters.*/
	    short slen;
	    short dlen;
	    char *parmptr;

	    if (localdata->num_params <= i) parmptr = (char *)ops->pdefault;
	    else if (*(localdata->params + i) == NULL) parmptr = (char *)ops->pdefault;
	    else parmptr = (char *)(*(localdata->params + i));

 	    slen = strlen(parmptr);
	    dlen = strlen((char *)ops->pdefault);
	    if (slen < dlen) {
	       for (k = slen; k < dlen; k++)
		  *((char *)(*(ops->position + j)) + k + 2) = NO_OP;
	    }	  
	    else if (slen > dlen) {
	       labelptr plab;
	       uchar *tmpstr;
	       int l, plen;

	       ops->pdefault = (void *)realloc(ops->pdefault, slen + 1);
	       for (k = dlen; k < slen; k++)
		   *((char *)ops->pdefault + k) = NO_OP;
	       *((char *)ops->pdefault + k) = '\0';

	       pgen = findparam(localdata->thisobject, *(ops->position + j), ops->type);
	       if (pgen != NULL) plab = TOLABEL(&pgen);
	       else {
		  printf("Warning:  parameter hooks to no label string!\n");
		  return;
	       }
	       tmpstr = plab->string;
	       plen = strlen((char *)tmpstr);
	       plab->string = (uchar *)realloc(plab->string, plen + slen - dlen + 1);
	       *(plab->string + plen + slen - dlen) = '\0';

	       /* Re-hook, both for new alloc'd position and for character shift */
	       for (k = 0; k < nops; k++) {
      		  sops = *(localdata->thisobject->params + k);
      		  for (l = 0; l < sops->number; l++) {
		     if (l == j && k == i) continue;
         	     if (sops->type == XC_STRING) {
			if ((uchar *)sops->position[l] > tmpstr
				   && sops->position[l] < ops->position[j])
	    		   sops->position[l] += (int)(plab->string - tmpstr);
			else if ((uchar *)sops->position[l] > tmpstr
				   && (int)sops->position[l] < (int)tmpstr + plen)
	    		   sops->position[l] += (int)(plab->string - tmpstr
					+ slen - dlen);
	             }
	          }
	       }
	       ops->position[j] += (int)(plab->string - tmpstr);

	       /* use memmove() to move overlapping string segments */
	       /* start after the TEXT_ESC START_PARAM (thus the + 2) */
	       memmove((char *)ops->position[j] + slen - dlen + 2,
			(char *)ops->position[j] + 2,
			strlen((char *)ops->position[j] + 2));
	    }
	    /* use memmove() here to avoid copying end-of-string NULL to destination */
	    memmove((char *)(*(ops->position + j)) + 2, parmptr, slen);
	 }
      }
   }
}


/*-----------------------------------------------------------------------*/
/* "Natural" string length---strlen() without counting NO_OP characters. */
/*-----------------------------------------------------------------------*/

int natstrlen(uchar *tstr)
{
   int slen = 0;
   uchar *pstr = tstr;
   while (*pstr != '\0') {
      if (*pstr == TEXT_ESC && *(pstr + 1) == PARAM_END) break;
      if (*pstr++ != NO_OP) slen++;
   }
   return slen;
}

/*----------------------------------------------------------------------*/
/*  Remove trailing NO_OPs from a string 				*/ 
/*----------------------------------------------------------------------*/

curtail(uchar *tstr)
{
   uchar *pstr = tstr;
   while (*pstr++ != '\0');
   if (pstr > tstr) {
      if (*(--pstr) == NO_OP) {
	 if (pstr > tstr) {
	    while (*(--pstr) == NO_OP);
	    pstr++;
	 }
      }
      *pstr = '\0';
   }
}

/*----------------------------------------------------------------------*/
/* Check whether this page object was entered via a library page	*/
/*----------------------------------------------------------------------*/

objectptr checklibtop()
{
   int k;
   objectptr pobj;

   for (k = pushes - 1; k >= 0; k--) {
      pobj = (*(pushlist + k)).thisobject;
      if (pobj == xobjs.libtop[USERLIB] || pobj == xobjs.libtop[LIBRARY])
	 return pobj;
   }
   return (objectptr)NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object	instance			*/
/* (Reverts all parameters to default value)				*/
/*----------------------------------------------------------------------*/

removeinst(objinstptr thisinst)
{
   free(thisinst->params);
   thisinst->params = NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object.				*/
/*----------------------------------------------------------------------*/

removeparams(objectptr thisobj)
{
   int i, j;
}

/*-----------------------------------------------------------------------*/
/* Determine how the old label relates to the new label, and re-hook the */
/* parameters accordingly.						 */
/*-----------------------------------------------------------------------*/

void resolveparams(objectptr localdata, uchar *origstr, labelptr curlabel)
{
   oparamptr ops;
   uchar *sptr, *pptr, *ssave;
   short offset, nval, ival, nlen, dlen, k;

   if (objectdata->params == NULL) return;  /* Any paramters at all?	*/

   /* Remove trailing NO_OPs from strings */
   curtail(origstr);
   curtail(curlabel->string);

   /*-------------------------------------------------------------------*/
   /* 1) Expand string with NO_OPs as necessary to hold default strings */
   /* 2) See if any part of this string is a parameter of the object	*/
   /* 3) Re-link parameters.						*/
   /* 4) Write back new parameter values 				*/
   /*-------------------------------------------------------------------*/

   /* subroutine labeltext() prevents PARAM boundary markers (PARAM_START */
   /* and PARAM_END) from being deleted, so there will be a 1:1 match	  */
   /* between the markers in the old and new strings.  Use these markers  */
   /* to determine new instance values or changed default values.	  */

   /* Pass 1: make sure that new parameters are at least as long as the	  */
   /* original (= length of default) values.  Pad if shorter.		  */

   sptr = curlabel->string;
   pptr = origstr;
   while (*(sptr + 1) != '\0') {
      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_START) && *(sptr + 1) != '\0')
         sptr++;
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_START) && *(pptr + 1) != '\0')
         pptr++;
      offset = 0;
      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_END) && *(sptr + 1) != '\0') {
	 offset--;
         sptr++;
      }
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_END) && *(pptr + 1) != '\0') {
	 offset++;
         pptr++;
      }
      if (offset > 0) {
	 short tmppos = (short)(sptr - curlabel->string);
   	 curlabel->string = (uchar *)realloc(curlabel->string,
		strlen((char *)curlabel->string) + offset + 1);
	 memmove(curlabel->string + tmppos + offset, curlabel->string + tmppos,
		strlen((char *)curlabel->string + tmppos) + 1); 
	
	 for (k = 0; k < offset; k++)
	    *(curlabel->string + tmppos + k) = NO_OP;

	 sptr = curlabel->string + tmppos + offset + 2;
      }
   }

   /* Pass 2: now that no more reallocation will occur, rehook the pointers */

   sptr = curlabel->string;
   pptr = origstr;
   while (*(sptr + 1) != '\0') {

      /* find the beginning and end of both original and new parameter substrings */

      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_START) && *(sptr + 1) != '\0')
         sptr++;
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_START) && *(pptr + 1) != '\0')
         pptr++;
      if (*(sptr + 1) == '\0') break;
      offset = 0;
      ssave = sptr;
      ops = parampos(objectdata, NULL, pptr, &nval, &ival);
      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_END) && *(sptr + 1) != '\0') {
	 offset--;
         sptr++;
      }
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_END) && *(pptr + 1) != '\0') {
	 offset++;
         pptr++;
      }

      *((char **)ops->position + nval) = ssave;		/* rehook */

      /* Get the length of the parameter (less the TEXT_ESC pair) */
      dlen = sptr - ssave - 2;

      /* If one of the libraries is in the hierarchy stack, */
      /* then we change the default.  Always, change the    */
      /* parameter list in the calling object instance	     */
      /* (top of hierarchy stack).			     */

      if ((checklibtop() != NULL) || (pushes == 0)) {
         /* printf("Came from library or top page:  changing default value\n"); */
         ops->pdefault = (uchar *)realloc(ops->pdefault, dlen + 1);
         strncpy((char *)ops->pdefault, *((char **)ops->position + nval) + 2, dlen);
         *((char *)ops->pdefault + dlen) = '\0';
      }
      else {
         objinstptr pinst = (*(pushlist + pushes - 1)).thisinst;
         void **pparm;
         short mlen;

	 /* printf("Came from page:  changing instance value\n"); */

	 /* If the new string == the default, then use the default */
	 mlen = natstrlen(ops->pdefault);
	 if ((mlen == dlen) && !strncmp(ops->pdefault, *(ops->position + nval), mlen)) {
	    /* printf("Reverting to default value\n"); */
	    if (pinst->num_params > ival) {
	       pparm = pinst->params + ival;
	       if (*pparm != NULL) {
		  free(*pparm);
		  *pparm = NULL;
	       }
	    }
	 }
	 else {
	    if (pinst->params == NULL) {
	       pinst->params = (void **)malloc(objectdata->num_params
			* sizeof(void *));
	       for (k = 0; k < objectdata->num_params; k++)
			*(pinst->params + k) = NULL;
	       pinst->num_params = objectdata->num_params;
	    }
	    else if (ival >= pinst->num_params) {
	       pinst->params = (void **)realloc(pinst->params,
			(ival + 1) * sizeof(void *));
	       for (k = pinst->num_params; k <= ival; k++)
			*(pinst->params + k) = NULL;
	       pinst->num_params = ival + 1;
	    }
	    pparm = pinst->params + ival;
	    if (*pparm == NULL)
	       *pparm = (void *)malloc((dlen + 1) * sizeof(uchar));
	    else
	       *pparm = (void *)realloc(*pparm, (dlen + 1) * sizeof(uchar));
	    strncpy((char *)(*pparm), (char *)(*(ops->position + nval)) + 2, dlen);
	    *((char *)(*pparm) + dlen) = '\0';

	    /* pad the default with NO_OP if shorter than the new string */

	    if (dlen > (nlen = strlen(ops->pdefault))) {
	       ops->pdefault = (void *)realloc(ops->pdefault, dlen + 1);
	       for (k = nlen; k < dlen; k++) 
	          *((char *)ops->pdefault + k) = NO_OP;
	       *((char *)ops->pdefault + k) = '\0';
	    }
         }
      }
   }
}

/*-----------------------------------------------------------------------*/
