/*-------------------------------------------------------------------------*/
/* schema.c --- xcircuit routines specific to the schematic capture system */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/13/97    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"
#include "Xw/MenuBtn.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"
#include "menudep.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Globaldata xobjs;
extern Clientdata areastruct;
extern short	  eventmode;
extern short	  pushes;
extern int	  *appcolors;
extern Widget     menuwidgets[];
extern Widget	  netbutton;

XtCallbackProc	  schembutton();
objectptr	  recurselect();
short		  *selectobject();

Boolean		  armed = False;

/*-------------------------------------------------------------------------*/
#ifdef SCHEMA

extern Widget wsymb, wschema;

/*--------------------------------------------------------*/
/* Menu calls (procedure wrappers)			  */
/*--------------------------------------------------------*/

XtCallbackProc callgennet(Widget w, unsigned int mode, caddr_t calldata)
{
   switch(mode) {
     case 0:
        gennet("spice", "spc");
	break;
     case 1:
        gennet("sim", "sim");
	break;
     case 2:
        gennet("pcb", "pcb");
	break;
   }
}

/*------------------------------------------------------------*/
/* Enable schematic capture (set schemon and realize widgets) */
/*------------------------------------------------------------*/

XtCallbackProc doxschema(Widget w, caddr_t clientdata, caddr_t calldata)
{
   short i;
   Arg wargs[1];

   if (areastruct.schemon) {
      areastruct.schemon = False;
      XtUnmanageChild(wschema);
      XtUnmanageChild(wsymb);
      XtUnmanageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Enable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }
   else {
      areastruct.schemon = True;
      XtManageChild(wschema);
      XtManageChild(wsymb);
      XtManageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Disable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }
}

/*----------------------------------*/
/* Find connectivity of a selection */
/*----------------------------------*/

void startconnect(button, clientdata, calldata)
  Widget       button;
  caddr_t      clientdata, calldata;
{
   if (eventmode == NORMAL_MODE) {
      Wprintf("Click button1 on wire to see connectivity.");
      eventmode = CONNECT_MODE;
   }
}

/*------------------------------------------------------------------------*/

XtCallbackProc connectivity(button, clientdata, calldata)
  Widget       button;
  caddr_t      clientdata, calldata;
{
   polyptr gpoly;
   int netid;

   /* Will have to refine recursive selection at a later date. . . */
   /* objectptr gobj;				*/
   /* gobj = recurselect(&gpoly, objectdata);	*/

   objectselect(POLYGON);
   if (areastruct.selects != 1) {
      Wprintf("Must select one polygon to find network connections.");
   }
   else {
      gpoly = SELTOPOLY(areastruct.selectlist);
      objectdeselect();
      createnets(True);
      netid = resolved(objectdata->netlist, gpoly);
      /* printf(">> Resolving connectivity for net %d\n", netid);  */
      if (netid != 0)
         highlightnet(objectdata, netid);
      else
         Wprintf("No network is associated with this polygon");
      destroynets();
   }
}

/*--------------------------------------------------------*/
/* Change a label into a pinlabel and vice versa	  */
/*--------------------------------------------------------*/

void pinconv(labelptr cpin)
{
   if (!cpin->pin) {
      cpin->pin = LOCAL;
      cpin->color = SNAPCOLOR;
      Wprintf("Changed label to type LOCAL pin");
   }
   else {
      cpin->pin = False;
      cpin->color = DEFAULTCOLOR;
      Wprintf("Changed pin to normal label");
   }
}

/*--------------------------------------------------------*/

XtCallbackProc pinconvert(Widget w, caddr_t clientdata, caddr_t calldata)
{
   short *gsel;

   if (areastruct.selects == 0) {
      Wprintf("Must first select a label to convert to a pin");
      return;
   }

   for (gsel = areastruct.selectlist; gsel < areastruct.selectlist +
		areastruct.selects; gsel++)
      if (SELECTTYPE(gsel) == LABEL)
	 pinconv(SELTOLABEL(gsel));

   objectdeselect();
   drawarea(NULL, objectdata, NULL);
}

/*--------------------------------------------------------*/
/* Change a pinlabel's type to GLOBAL, INFO, or LOCAL	  */
/*--------------------------------------------------------*/

XtCallbackProc dopintype(Widget w, caddr_t clientdata, caddr_t calldata)
{
   short *gsel;
   labelptr glab;

   if (areastruct.selects == 0) {
      Wprintf("Must first select a label to make global");
      return;
   }

   for (gsel = areastruct.selectlist; gsel < areastruct.selectlist +
		areastruct.selects; gsel++) {
      if (SELECTTYPE(gsel) == LABEL) {
	 glab = SELTOLABEL(gsel);
	 if (glab->pin == GLOBAL) {
	    glab->pin = INFO;
	    glab->color = AUXCOLOR;
	    Wprintf("Changed pin to type INFO");
	 }
	 else if (glab->pin == LOCAL) {
	    glab->pin = GLOBAL;
	    glab->color = BBOXCOLOR;
	    Wprintf("Changed pin to type GLOBAL");
	 }
	 else if (glab->pin == INFO) {
	    glab->pin = LOCAL;
	    glab->color = SNAPCOLOR;
	    Wprintf("Changed pin to type LOCAL");
	 }
	 else pinconvert(w, clientdata, calldata);
      }
   }
}

/*----------------------------------------------------------*/
/* Set colors on the symbol/schematic buttons appropriately */
/*----------------------------------------------------------*/

void setsymschem()
{
   Arg aargs[2], bargs[2];

   if (armed) return;	/* don't change colors until selection is made! */

   if (objectdata->schemtype == SCHEMATIC) {
      if (objectdata->symschem == NULL) {
         XtSetArg(aargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(aargs[1], XtNforeground, OFFBUTTONCOLOR);
      }
      else {
         XtSetArg(aargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(aargs[1], XtNforeground, FOREGROUND);
      }
      XtSetArg(bargs[0], XtNbackground, SNAPCOLOR);
      XtSetArg(bargs[1], XtNforeground, FOREGROUND);
   }
   else {
      if (objectdata->symschem == NULL) {
         XtSetArg(bargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(bargs[1], XtNforeground, OFFBUTTONCOLOR);
      }
      else {
         XtSetArg(bargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      }

      if (objectdata->schemtype == FUNDAMENTAL)
         XtSetArg(aargs[0], XtNbackground, AUXCOLOR);
      else if (objectdata->schemtype == SYMBOL)
         XtSetArg(aargs[0], XtNbackground, SNAPCOLOR);
      else
         XtSetArg(aargs[0], XtNbackground, BACKGROUND);
      XtSetArg(aargs[1], XtNforeground, FOREGROUND);
   }

   XtSetValues(wsymb, aargs, 2);
   XtSetValues(wschema, bargs, 2);
}

/*--------------------------------------------------------*/
/* Change an object's type to Fundamental or back again	  */
/*--------------------------------------------------------*/

XtCallbackProc makefund(Widget w, caddr_t clientdata, caddr_t calldata)
{
   short *fsel;
   objinstptr fobj;

   if (areastruct.selects == 0) {
      if (objectdata->schemtype == SYMBOL) {
	 objectdata->schemtype = FUNDAMENTAL;
	 Wprintf("Changed top-level object to type FUNDAMENTAL");
      }
      else if (objectdata->schemtype == FUNDAMENTAL) {
	 if (objectdata->symschem == NULL) {
	    objectdata->schemtype = SYMBOL;
	    Wprintf("Changed top-level object to type SYMBOL");
	 }
	 else {
	    if (objectdata->symschem->schemtype == SYMBOL) {
	       objectdata->schemtype = SCHEMATIC;
	       Wprintf("Changed top-level object to type SCHEMATIC");
	    }
	    else if (objectdata->symschem->schemtype == SCHEMATIC) {
	       objectdata->schemtype = SYMBOL;
	       Wprintf("Changed top-level object to type SYMBOL");
	    }
	 }
      }

      setsymschem();
      return;
   }

   for (fsel = areastruct.selectlist; fsel < areastruct.selectlist +
		areastruct.selects; fsel++) {
      if (SELECTTYPE(fsel) == OBJECT) {
	 fobj = SELTOOBJINST(fsel);
	 if (fobj->thisobject->schemtype == SYMBOL) {
	    fobj->thisobject->schemtype = FUNDAMENTAL;
	    Wprintf("Changed object to type FUNDAMENTAL");
	 }
	 else if (fobj->thisobject->schemtype == FUNDAMENTAL) {
	    if (fobj->thisobject->symschem != NULL) {
	       if (fobj->thisobject->symschem->schemtype == SYMBOL) {
		  fobj->thisobject->schemtype = SCHEMATIC;
	          Wprintf("Changed object to type SCHEMATIC");
	       }
	       else if (fobj->thisobject->symschem->schemtype == SCHEMATIC) {
		  fobj->thisobject->schemtype = SYMBOL;
	          Wprintf("Changed object to type SYMBOL");
	       }
	    }
	    else {
	       fobj->thisobject->schemtype = SYMBOL;
	       Wprintf("Changed object to type SYMBOL");
	    }
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Find the page number for an object			  */
/*--------------------------------------------------------*/

int findpageobj(pobj)
  objectptr pobj;
{
   int tpage;
   objectptr chkpage;

   for (tpage = 0; tpage < xobjs.pages; tpage++) {
      if (xobjs.pagelist[tpage]->pageobj == pobj) return tpage;
   }
   return -1;
}

/*--------------------------------------------------------*/
/* Recursively find all sub-circuits associated with the  */
/* top-level circuit and set their filenames to be the	  */
/* same.						  */
/*--------------------------------------------------------*/

findsubschems(toppage, cschem)
  int toppage;
  objectptr cschem;
{
   genericptr *cgen;

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {
	 objectptr cobj = TOOBJINST(cgen)->thisobject;

	 if (cobj->symschem != NULL) {
	    int pageno = findpageobj(cobj->symschem);
	    /* printf("Symbol %s has schematic at page %d\n", cobj->name, pageno); */
	    if (pageno < 0) {	/* This shouldn't happen (hopefully) */
	       Wprintf("Error: disconnected schematic!\n");
	    }
	    else {
	       xobjs.pagelist[pageno]->filename = xobjs.pagelist[toppage]->filename;
	    }

	    /* A symbol on its own schematic page is allowed for clarity */
	    /* of the schematic, but this cannot be a functional part of */
	    /* the schematic circuit!					 */
	    
	    if (cobj->symschem != cschem) {
	       /* printf("Searching for dependencies of symbol %s\n", cobj->name); */
	       findsubschems(toppage, cobj->symschem);
	    }
/*
	    else {
	       printf("Notice:  Symbol on its own schematic\n");
	    }
*/
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Copy label to corresponding schematic/symbol		  */
/*--------------------------------------------------------*/

copypinlabel(pinlab)
  labelptr pinlab;
{
   labelptr *newlabel, tlabel;
   genericptr *tgen;
   objectptr schemobj = objectdata->symschem;

   if (areastruct.schemon && schemobj != NULL && (pinlab->pin == LOCAL)) {

      /* If this pin already exists, don't copy it */

      for (tgen = schemobj->plist; tgen < schemobj->plist + schemobj->parts;
		tgen++) {
         if ((*tgen)->type == LABEL) {
	    tlabel = TOLABEL(tgen);
	    if (!strcmp(tlabel->string, pinlab->string)) return;
         }
      }

      NEW_LABEL(newlabel, schemobj);
      (*newlabel)->pin = pinlab->pin;
      (*newlabel)->rotation = pinlab->rotation;
      (*newlabel)->justify = pinlab->justify;
      (*newlabel)->color = pinlab->color;
      (*newlabel)->scale = pinlab->scale;
      (*newlabel)->string = (char *)malloc((1 + strlen(pinlab->string))
		* sizeof(char));
      noparmstrcpy((*newlabel)->string, pinlab->string);

      /* place label just outside bounding box, then recompute bbox */

      (*newlabel)->position.x = schemobj->lowerleft.x + (schemobj->width >> 1);
      (*newlabel)->position.y = schemobj->lowerleft.y - TEXTHEIGHT
	   * (*newlabel)->scale;
      schemobj->parts++;
      singlebbox(ENDPART);
   }
}

/*--------------------------------------------------------*/
/* Check if top-level page is the same name as a library  */
/* object; if so, connect it.				  */
/*--------------------------------------------------------*/

int checkschem(objectptr thisobj, char *cname)
{
   objectptr *tlib;
   short i, j;

   if (areastruct.schemon == False || thisobj->symschem != NULL) return 0;

   for (i = 0; i < xobjs.numlibs; i++) {
      for (j = 0; j < xobjs.userlibs[i].number; j++) {
	 tlib = xobjs.userlibs[i].library + j;

         if (!strcmp(cname, (*tlib)->name)) {
	    thisobj->symschem = (*tlib);
	    thisobj->schemtype = SCHEMATIC;
	    (*tlib)->symschem = thisobj;
	    (*tlib)->schemtype = SYMBOL;
	    return 1;
	 }
      }
   }
   return 0;
}

/*--------------------------------------------------------*/
/* Complement to the above routine:  If a library object  */
/* name is the same as a top-level page, connect them.	  */
/*--------------------------------------------------------*/

int checksym(objectptr symobj, char *cname)
{
   short cpage;
   objectptr checkpage;

   if (areastruct.schemon == False || symobj->symschem != NULL) return 0;

   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = xobjs.pagelist[cpage]->pageobj;
      if (checkpage != NULL) {
         if (!strcmp(checkpage->name, cname)) {
	    symobj->symschem = checkpage;
	    symobj->schemtype = SYMBOL;
	    checkpage->symschem = symobj;
	    checkpage->schemtype = SCHEMATIC;
	    return 1;
	 }
      }
   }
  return 0;
}

/*--------------------------------------------------------*/
/* Find location of corresponding pin in symbol/schematic */
/*--------------------------------------------------------*/

labelptr findotherpin(curlabel)
  labelptr curlabel;
{
   objectptr other = objectdata->symschem;
   genericptr *tgen;
   labelptr tlab;

   if (areastruct.schemon == False || other == NULL || curlabel->pin == False)
      return NULL;

   for (tgen = other->plist; tgen < other->plist + other->parts; tgen++) {
      if ((*tgen)->type == LABEL) {
	 tlab = TOLABEL(tgen);
	 if (!strcmp(tlab->string, curlabel->string)) return tlab;
      }
   }
   return NULL;
}

/*------------------------------------------*/
/* Swap object schematic and symbol pages   */
/*------------------------------------------*/

XtCallbackProc swapschem(Widget w, caddr_t clientdata, caddr_t calldata)
{
   objectptr savepage = objectdata;
   labelptr  *pinlab;
   genericptr *plab;
   Boolean lflag;

   if (areastruct.schemon == False) return;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);

   if (objectdata->symschem == NULL) {  /* create symbol or schematic */
      if (objectdata->schemtype == SYMBOL) {
	 int tpage;

	 /* create a new page for the new schematic */

	 for (tpage = 0; tpage < xobjs.pages; tpage++)
	    if (xobjs.pagelist[tpage]->pageobj == NULL) break;
         changepage(tpage);
      }
      else {
	 objectptr *newobject;
	 short libnum = USERLIB - LIBRARY;

	 /* create a new library object for the new symbol */

	 xobjs.userlibs[libnum].library = (objectptr *)
		realloc(xobjs.userlibs[libnum].library,
		++xobjs.userlibs[libnum].number * sizeof(objectptr));
	 newobject = xobjs.userlibs[libnum].library
		+ xobjs.userlibs[libnum].number - 1;
         *newobject = (objectptr) malloc(sizeof(object));
	 initmem(*newobject);
	 (*newobject)->schemtype = SYMBOL;
	 (*newobject)->hidden = False;
	 objectdata = *newobject;

	 /* Generate the default bounding box for a size-zero object */
	 calcbbox(*newobject);
      }

      /* set links between the two objects */

      savepage->symschem = objectdata;
      objectdata->symschem = savepage;

      /* make the name of the new object equal to that of the old */

      strcpy(objectdata->name, savepage->name);

      /* copy all pin labels into the new object */

      for (plab = savepage->plist; plab < savepage->plist + savepage->parts;
		plab++) {
	 if ((*plab)->type == LABEL) {
	    genericptr *tgen;
	    labelptr tlab, lpin = (labelptr)*plab;

	    if (lpin->pin == LOCAL) {

      	       /* Only make one copy of each pin name */

	       lflag = False;
               for (tgen = objectdata->plist; tgen <
			objectdata->plist + objectdata->parts; tgen++) {
		  if ((*tgen)->type == LABEL) {
		     tlab = TOLABEL(tgen);
	             if (!strcmp(tlab->string, lpin->string)) lflag = True;
		  }
      	       }
	       if (lflag == True) continue;

	       NEW_LABEL(pinlab, objectdata);
	       (*pinlab)->pin = lpin->pin;
	       (*pinlab)->color = lpin->color;
	       (*pinlab)->rotation = 1;
	       (*pinlab)->scale = 1.0;
	       (*pinlab)->justify = areastruct.justify; 
	       (*pinlab)->position.x = 0;
	       (*pinlab)->position.y = objectdata->parts * (TEXTHEIGHT + 10);
	       u2u_snap(&((*pinlab)->position));
	       (*pinlab)->string = (char *)malloc((1 + strlen(lpin->string))
			* sizeof(char));
	       strcpy((*pinlab)->string, lpin->string);
	       objectdata->parts++;
	    }
         }
      }

      /* Recreate the user library with the new symbol */
      if (savepage->schemtype != SYMBOL) composelib(USERLIB);
   }
   else {
      objectdata = objectdata->symschem;
   }

   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;

   setpage();
   transferselects();
   refresh(NULL, NULL, NULL);

   setsymschem();
}

/*------------------------------------------*/
/* Disassociate a symbol from its schematic */
/*------------------------------------------*/

XtCallbackProc schemdisassoc(Widget w, caddr_t clientdata, caddr_t calldata)
{
   if (eventmode != NORMAL) {
      Wprintf("Cannot disassociate schematics in this mode");
   }
   else if (objectdata->symschem == NULL) {
      if (objectdata->schemtype == SCHEMATIC) 
         Wprintf("Schematic not associated with any symbol.");
      else
         Wprintf("Symbol not associated with any schematic.");
   }
   else {
      objectdata->symschem->symschem = NULL;
      objectdata->symschem = NULL;
      setsymschem();
   }
}

/*------------------------------------------------------*/
/* Callback procedures on the schematic/symbol buttons	*/
/*------------------------------------------------------*/

XtCallbackProc schemassoc(Widget w, objectptr associate, caddr_t calldata)
{
   if (eventmode != NORMAL) {
      Wprintf("Cannot make association in this mode");
      return;
   }

   XtRemoveAllCallbacks (w, XtNselect);
   XtAddCallback (w, XtNselect, (XtCallbackProc)schembutton, NULL);
   armed = False;

   /* If one clicks twice on the same page, a new page is created for the */
   /* new schematic or symbol.						  */
   
   if (associate == objectdata)
      swapschem(w, NULL, NULL);
   else if (objectdata->symschem != NULL)
      Wprintf("This page already associated");
   else if (objectdata->schemtype == FUNDAMENTAL)
      Wprintf("Fundamental symbol cannot be associated with a schematic");
   else {
      /* Change name of schematic to match the symbol if necessary */
      if (associate->schemtype == SCHEMATIC) {
	 /* A top-level page cannot be a symbol. */
	 if (pushes == 0) {
	    Wprintf("Page must be an object to be a symbol");
	    setsymschem();
	    return;
	 }
	 objectdata->schemtype = SYMBOL;
/*
	 sprintf(associate->name, objectdata->name);
	 renamepage(areastruct.page);
*/
      }
      else {
	 objectdata->schemtype = SCHEMATIC;
/*
	 sprintf(objectdata->name, associate->name);
*/
      }
      associate->symschem = objectdata;
      objectdata->symschem = associate;
   }
   setsymschem();
}

/*-----------------------------------------------------------------------*/

XtCallbackProc schembutton(Widget w, caddr_t clientdata, caddr_t calldata)
{
   Arg wargs[2];

   if (eventmode != NORMAL) {
      Wprintf("Cannot switch schematics in this mode");
      return;
   }

   if (w == wsymb) {
      if (objectdata->schemtype != SCHEMATIC) {
	 if (objectdata->symschem == NULL) {	/* change type */
	    if (armed) {
               XtRemoveAllCallbacks (wschema, XtNselect);
               XtAddCallback (wschema, XtNselect, (XtCallbackProc)schembutton, NULL);
	       armed = False;
	       setsymschem();
            }
	    else
	       makefund(w, NULL, NULL);
	 }
      }
      else {
	 if (objectdata->symschem != NULL) {	/* swap */
	    swapschem(w, NULL, NULL);
	 }
	 else {		/* associate a page with the symbol */
	    Wprintf("Press again for new symbol or go to symbol page and press again");
	    armed = True;
      	    XtSetArg(wargs[0], XtNforeground, QUERYCOLOR);
      	    XtSetArg(wargs[1], XtNbackground, BACKGROUND);
   	    XtSetValues(wsymb, wargs, 2);
	    XtRemoveCallback (w, XtNselect, (XtCallbackProc)schembutton, NULL);
	    XtAddCallback (w, XtNselect, (XtCallbackProc)schemassoc, objectdata);
	 }
      }
   }
   else {  /* w == wschema */
      if (objectdata->schemtype == SCHEMATIC) {
	 if (objectdata->symschem == NULL) {	/* change type */
	    if (armed) {
               XtRemoveAllCallbacks (wsymb, XtNselect);
               XtAddCallback (wsymb, XtNselect, (XtCallbackProc)schembutton, NULL);
	       armed = False;
	       setsymschem();
            }
	    else
	       makefund(w, NULL, NULL);
	 }
      }
      else if (objectdata->schemtype == FUNDAMENTAL)
	 Wprintf("Cannot attach a schematic to fundamental symbol");
      else {
	 if (objectdata->symschem != NULL) { 	/* swap */
	    swapschem(w, NULL, NULL);
	 }
	 else {		/* associate a page with the schematic */
	    Wprintf(
		"Press again for new schematic or go to schematic page and press again");
	    armed = True;
      	    XtSetArg(wargs[0], XtNforeground, QUERYCOLOR);
      	    XtSetArg(wargs[1], XtNbackground, BACKGROUND);
   	    XtSetValues(wschema, wargs, 2);
	    XtRemoveCallback (w, XtNselect, (XtCallbackProc)schembutton, NULL);
	    XtAddCallback (w, XtNselect, (XtCallbackProc)schemassoc, objectdata);
	 }
      }
   }
}

#endif
/*-------------------------------------------------------------------------*/
