/*----------------------------------------------------------------------*/
/* netlist.c --- xcircuit routines specific to schematic capture and	*/
/*		 netlist generation					*/
/*----------------------------------------------------------------------*/
/*  Copyright (c) 2000 Tim Edwards, Johns Hopkins University		*/
/*  Written April 1998 to January 2000				   	*/
/*  Original version for Pcb netlisting 3/20/98 by Chow Seong Hwai,	*/
/*			Leeds University, U.K.				*/
/*----------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*----------------------------------------------------------------------*/
/* Local includes							*/
/*----------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"

extern Display  *dpy;
extern int *appcolors;

/*-------------------------------------------------------------------------*/
#ifdef SCHEMA

/*----------------------------------------------------------------------*/
/* Declaration of routines and their parameters				*/
/*----------------------------------------------------------------------*/

void gennet(char *, char *);
void gennetlist(objectptr);
void genpinlist(objectptr, Boolean);
int onsegment(XPoint *, XPoint *, XPoint *, XPoint *, objectptr);
int resolved(NetlistPtr, polyptr);
void changenet(NetlistPtr, int, int);
void linknet(NetlistPtr *, polyptr, int);
int whichnet(NetlistPtr, XPoint *);
void makepin(PinlistPtr *, labelptr, int);
void makecall(CalllistPtr *, objectptr, labelptr, objinstptr, int);
void writepcb(objectptr, CalllistPtr, char *);
void outputpcb(objectptr, FILE *);
void freepcb();
void writenet(objectptr, char *, char *);
void freenets(objectptr);
void freeglobals();
char *nettopin(int, objectptr, char *);
void searchconnect(XPoint *, int, int, objinstptr);

extern char _STR[150];
extern char _STR2[250];
extern Clientdata areastruct;

PinlistPtr globallist;
struct Ptab *ptable;

int netindex;	/* count for labeling nets in the flattened netlist */
int devindex;   /* similarly for labeling devices		    */
int subindex;   /* similarly for labeling spice subcircuits	    */

#define ONDIST 2  /* "slack" in algorithm for almost-touching lines */

/*-------------------------------------------------------------------------*/
/* gennet(): Generate netlist structures				   */
/*									   */
/* Result is the creation of three linked lists inside each object in the  */
/* circuit hierarchy:							   */
/*									   */
/*    1) the netlist:  assigns a net number to each polygon on the object  */
/*	 	       page.						   */
/*    2) the pinlist:  a list of every pin label in the object with their  */
/*		       net numbers.					   */
/*    3) the calllist: a list of calls made from the object to all sub-	   */
/*		       circuits.  Each link contains one parameter to one  */
/*		       call which can be used to link the net in the top   */
/*		       level to its corresponding net in the subcircuit	   */
/*		       (see structure definitions in xcircuit.h).	   */
/*-------------------------------------------------------------------------*/

void createnets(Boolean keepcalls)
{
   netindex = 1;
   gennetlist(objectdata);
   genpinlist(objectdata, keepcalls);
}

/*-------------------------------------------------------------------------*/

void destroynets()
{
   freenets(objectdata);
   freeglobals();
}

/*-------------------------------------------------------------------------*/

void gennet(char *mode, char *suffix)
{
   createnets(False);
   
   if (!strcmp(mode, "pcb")) {
      if (objectdata->pinlist != NULL) {
         writenet(objectdata, mode, suffix);
      }
   }
   else if (objectdata->netlist != NULL) {
      writenet(objectdata, mode, suffix);
   }
   else
      Wprintf("Error generating netlist: no file written");

   destroynets();
}

/*-------------------------------------------------------------------------*/
/* Resolve nets and subcircuit calls					   */
/*-------------------------------------------------------------------------*/

void gennetlist(objectptr cschem)
{
   genericptr *cgen, *tgen, *ogen, *pseek;
   labelptr olabel;
   polyptr cpoly, tpoly;
   objinstptr cobj, cinst;
   objectptr callobj, callsymb;

   XPoint xpos, *tpt, *tpt2, *endpt, *endpt2, *tmppts;
   Matrix locctm;
   int netid, orignet, curnet, nextnet = 1;
   NetlistPtr *nethead = &(cschem->netlist);
   CalllistPtr *callhead = &(cschem->calllist);
   ParamlistPtr paramlist;

   tmppts = (XPoint *)malloc(sizeof(XPoint));
	
   /* Turn polygons into nets */

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == POLYGON) {
 	 cpoly = TOPOLY(cgen);

	 if ((curnet = resolved(*nethead, cpoly)) == 0) {
	    curnet = nextnet;
	    linknet(nethead, cpoly, curnet);		
	    nextnet++;
	 }

	 /* Check for attachment of this segment to every other */
	 /* include the unlikely case that a net is made with a closed curve */

	 for (endpt = cpoly->points; endpt < cpoly->points
		+ cpoly->number; endpt++) {

	    endpt2 = endpt + 1;
	    if (endpt == cpoly->points + cpoly->number - 1) {
		if (!(cpoly->style & UNCLOSED))
	           endpt2 = cpoly->points;
	        else continue;
	    }

	    for (tgen = cgen + 1; tgen < cschem->plist + cschem->parts; tgen++) {
   	       if ((*tgen)->type == POLYGON) {
		  tpoly = TOPOLY(tgen);

	     	  /* check segments */

	     	  for (tpt = tpoly->points; tpt < tpoly->points
			+ tpoly->number; tpt++) {

		     tpt2 = tpt + 1;
	    	     if (tpt == tpoly->points + tpoly->number - 1) {
			if (!(tpoly->style & UNCLOSED)) 
	       	  	   tpt2 = tpoly->points;
	    	        else continue;
		     }

	     	     if (onsegment(endpt, endpt2, tpt, tpt2, cschem)) {
	     		if (orignet = resolved(*nethead, tpoly)) {
			   changenet(*nethead, orignet, curnet);
	     		}
	     		else {
			   linknet(nethead, tpoly, curnet);
	     		}
			/* found connected polygon; get the next one */
	     		break;
	     	     }
	     	  }
	       }
	    }
	 }
      }

      /* Pin survey---make connections where appropriate, not only where */
      /* pins are explicitly marked.					 */

      else if ((*cgen)->type == OBJECT) {
	 cinst = TOOBJINST(cgen);

	 /* Ignore connections into symbols having their own schematics. */
	 /* For now, this survey is only for symbols acting as their own schem. */

	 if (cinst->thisobject->symschem != NULL ||
		cinst->thisobject->schemtype == FUNDAMENTAL) {
	    /* printf("Ignoring potential connections between %s and %s\n",	*/
	    /*	cschem->name, cinst->thisobject->name);				*/
	    continue;
	 }

         /* printf("*** Complete pin survey of object %s:\n", cschem->name); */

	 for (pseek = cschem->plist; pseek < cschem->plist + cschem->parts; pseek++) {

	    /* Look for pins connecting to pins in the object */

	    if ((*pseek)->type == LABEL) {
	       olabel = TOLABEL(pseek);
	       if (olabel->pin && (olabel->pin != INFO))
	          searchconnect(&(olabel->position), 1, UNCLOSED, cinst);
	    }

	    /* Look for polygon ends connecting into the object */

	    else if ((*pseek)->type == POLYGON) {
	       tpoly = TOPOLY(pseek);
	       searchconnect(tpoly->points, tpoly->number, tpoly->style, cinst);
	    }

	    /* Also look for polygon ends in other objects connecting into object */
	    /* Again, survey is for symbols acting as their own schematics.	  */

	    else if ((*pseek)->type == OBJECT) {
	       genericptr *pseek2;
	       int j;
	       polyptr tpoly2;
	       objinstptr isib = TOOBJINST(pseek);
	       objectptr sibling = isib->thisobject;
	       if (isib == cinst) continue;
	       else if (isib->thisobject->symschem != NULL) {
	          /* printf("Ignoring potential connections between %s and %s\n", */
		  /*	cschem->name, isib->thisobject->name);			  */
		  continue;
	       }
	       /* printf("Searching for potential connections between %s and %s\n", */
	       /*	   cschem->name, sibling->name); */

	       for (j = 0; j < sibling->parts; j++) {
		  pseek2 = sibling->plist + j;
		  if ((*pseek2)->type == POLYGON) {
		     tpoly2 = TOPOLY(pseek2);
		     tmppts = (XPoint *)realloc(tmppts, tpoly2->number * sizeof(XPoint));
   		     UTransformPoints(tpoly2->points, tmppts, tpoly2->number,
				isib->position, isib->scale, isib->rotation);
		     searchconnect(tmppts, tpoly2->number, tpoly2->style, cinst);
		  }
	       }
	    }
         }
      }
   }
   free(tmppts);

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {

      /* When encountering object instances, call gennetlist() on the object if */
      /* it does not have a valid netlist, then recursively call gennetlist(). */
      /* Ignore "info" labels, which are not part of the network.		*/

 	 cobj = TOOBJINST(cgen);

	 /* Call gennetlist() recursively on subcircuits of the hierarchy.*/
	 /* ***However,*** an instance of the circuit's OWN symbol is      */
	 /* allowed on the schematic page for reference in the drawing;    */
	 /* don't be fooled into following this infinitely recursive path. */

	 if (cobj->thisobject->symschem != NULL)
	    callobj = cobj->thisobject->symschem;
	 else
	    callobj = cobj->thisobject;
	
	 /* object on its own schematic */
	 if (callobj == cschem) continue;

	 else {
	    if (callobj->netlist == NULL && callobj->schemtype != FUNDAMENTAL) {
	       gennetlist(callobj);
	    }
	 }

	 /* for each pin-label in the subcircuit or symbol, translate the */
	 /* position to the level above and search for & link in any      */
	 /* connecting nets.						  */

	 callsymb = cobj->thisobject;
	 UResetCTM(&locctm);
	 UPreMultCTM(&locctm, cobj->position, cobj->scale, cobj->rotation);

	 for (ogen = callsymb->plist; ogen < callsymb->plist + callsymb->parts;
		  ogen++) {
	    if ((*ogen)->type == LABEL) {
	       olabel = TOLABEL(ogen);
	       if (olabel->pin && (olabel->pin != INFO)) {
		
		  UTransformbyCTM(&locctm, &(olabel->position), &xpos, 1);  
	          netid = whichnet(*nethead, &xpos);
		  if (!netid) {
		     /* pin is not connected to a net.  However, it  */
		     /* may be connected directly to another object, */
		     /* so generate a "one-point" polygon and add it */
		     /* to the net list of the calling object.	     */
		     polyptr newpoly = (polyptr) malloc(sizeof(polygon));
		     newpoly->type = POLYGON;
		     newpoly->style = 0;	/* treat as CLOSED poly */
		     newpoly->number = 1;
		     newpoly->points = (XPoint *)malloc(sizeof(XPoint));
		     newpoly->points->x = xpos.x;
		     newpoly->points->y = xpos.y;
		     netid = ((*nethead) == NULL) ? 1 : (*nethead)->netid + 1;
	    	     linknet(nethead, newpoly, netid);		
		  }
	          makecall(callhead, callobj, olabel, cobj, netid);
	       }
	    }
	 }
      }
   }
}

/*-------------------------------------------------------------------------*/
/* Check whether given polygon is already resolved into netlist 	   */
/* Return the net id if already resolved, 0 otherwise	        	   */
/*-------------------------------------------------------------------------*/

int resolved(NetlistPtr netlist, polyptr poly)
{
   for (netlist; netlist != NULL; netlist = netlist->next) {
      if (netlist->poly == poly) {
 	 return (netlist->netid);
      }
   }
   return 0;
}

/*----------------------------------------------------------------------*/
/* Highlight all the polygons in a network (recursively, downward)	*/
/*----------------------------------------------------------------------*/

void highlightnet(objectptr cschem, int netid)
{
   NetlistPtr netlist;
   CalllistPtr calllist;
   ParamlistPtr paramlist;
   polyptr cpoly;
   objinstptr cinst;
   genericptr *curgen;

   XcSetFunction(GXcopy);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR);

   netlist = cschem->netlist;

   while (netlist != NULL) {
      if (netlist->netid == netid) {
         cpoly = netlist->poly;
	 if (cpoly->number > 1) {   /* don't draw temporary 1-point polygons */
	    /* printf(" >> Found polygon belonging to net %d at (%d, %d)\n", */
	    /*	netid, cpoly->points[0].x, cpoly->points[0].y);		     */
            UDrawPolygon((objinstptr)NULL, cpoly);
	 }
      }
      netlist = netlist->next;
   }

   calllist = cschem->calllist;

   while (calllist != NULL) {
      paramlist = calllist->params;
      while (paramlist != NULL) {
         if (paramlist->netid == netid) {
	    cinst = calllist->callno;
	    if (cinst->thisobject->symschem == NULL) {
	       UPushCTM();
	       UPreMultCTM(DCTM, cinst->position, cinst->scale, cinst->rotation);
	       /* printf(" > Calling object %s at (%d, %d)\n",			  */
	       /* calllist->callobj->name, cinst->position.x, cinst->position.y); */
	       highlightnet(calllist->callobj, pintonet(paramlist->pin,
				calllist->callobj));
	       UPopCTM();
	    }
	 }
	 paramlist = paramlist->next;
      }
      calllist = calllist->next;
   }
}

/*------------------------------------------------------------------*/
/* Check if value w is within ONDIST (tolerance of roundoff errors) */
/*------------------------------------------------------------------*/

Boolean neardist(long w)
{
   if (abs(w) < ONDIST) return True;
   else return False;
}

/*------------------------------------------------------------------*/
/* Check proximity of two points (within roundoff tolerance ONDIST) */ 
/*------------------------------------------------------------------*/

Boolean nearpoint(XPoint *point1, XPoint *point2)
{
   int dx, dy;

   dx = point1->x - point2->x;
   dy = point1->y - point2->y;

   if ((abs(dx) < ONDIST) && (abs(dy) < ONDIST)) return True;
   else return False;
}

/*-------------------------------------------------------------------------*/
/* Create a temporary I/O pin (special label type denoted by empty string) */
/*-------------------------------------------------------------------------*/

void make_tmp_pin(objinstptr cinst, XPoint *pinpt)
{
   labelptr *newlabel;
   char *tmpptr;

   NEW_LABEL(newlabel, cinst->thisobject);
   (*newlabel)->position.x = pinpt->x;
   (*newlabel)->position.y = pinpt->y;
   (*newlabel)->pin = LOCAL;
   (*newlabel)->scale = 1.0;
   (*newlabel)->rotation = 0;
   (*newlabel)->justify = 0;
   (*newlabel)->color = DEFAULTCOLOR;
   (*newlabel)->string = (char *)malloc(12);
   tmpptr = (*newlabel)->string;
   *tmpptr++ = 0;	/* Denotes special (temporary) label */
   *tmpptr++ = 0;
   sprintf(tmpptr, "ext%d", netindex++);
   cinst->thisobject->parts++;
}

/*--------------------------------------------------------------*/
/* Search for connections into a non-symbol subcircuit, based	*/
/* on various combinations of polygon and pin label overlaps.	*/
/*--------------------------------------------------------------*/
/* Eventually, this should be called recursively so that the	*/
/* polygon in question is translated down the hierarchy of	*/
/* schematics without symbols (where they exist).		*/
/*--------------------------------------------------------------*/

void searchconnect(XPoint *points, int number, int style, objinstptr cinst)
{
   genericptr *pseek, *pseek2;
   XPoint *tmppts, *tpt, *tpt2, *endpt, *endpt2, *pinpt, opinpt;
   objinstptr tinst;
   polyptr tpoly;
   labelptr tlab;
   Boolean labeled;
   long w;

   /* Generate temporary polygon in the coordinate system of	*/
   /* the object instance in question			 	*/

   tmppts = (XPoint *)malloc(number * sizeof(XPoint));
   InvTransformPoints(points, tmppts, number,
		cinst->position, cinst->scale, cinst->rotation);
   /* printf("Info: translated polygon w.r.t. object %s\n",	*/
   /* 		cinst->thisobject->name);			*/

   for (endpt = tmppts; endpt < tmppts + number; endpt++) {
      endpt2 = endpt + 1;
      if (endpt == tmppts + number - 1) {
         if (!(style & UNCLOSED))
  	    endpt2 = tmppts;
	 else if (number == 1)
	    endpt2 = endpt;
	 else continue;
      }

      labeled = False;
      for (pseek = cinst->thisobject->plist; pseek <
		cinst->thisobject->plist + cinst->thisobject->parts; pseek++) {
	 if ((*pseek)->type == OBJECT) {

	    /* Search for connections from polygons to object instance pin labels */

	    tinst = TOOBJINST(pseek);
	    for (pseek2 = tinst->thisobject->plist; pseek2 < tinst->thisobject->plist
			+ tinst->thisobject->parts; pseek2++) {
	       if ((*pseek2)->type == LABEL) {
	    	  tlab = TOLABEL(pseek2);
	          if (tlab->pin && (tlab->pin != INFO)) {
		     UTransformPoints(&(tlab->position), &opinpt, 1, tinst->position,
				tinst->scale, tinst->rotation);
	             if (neardist(finddist(endpt2, endpt, &opinpt))) {
		        /* printf("%s connects to pin %s of %s in %s\n",    	*/
			/*	((number > 1) ? "Polygon" : "Pin"),		*/
			/*	tlab->string + 2, tinst->thisobject->name,	*/
			/* 	cinst->thisobject->name);			*/
		        make_tmp_pin(cinst, &opinpt);
   			labeled = True;
		        break;
		     }
		  }
	       }
	    }
	 }
	 else if ((*pseek)->type == POLYGON) {

	    /* Search for connections from polygons to polygons */

	    tpoly = TOPOLY(pseek);
	    for (tpt = tpoly->points; tpt < tpoly->points + tpoly->number; tpt++) {
	       tpt2 = tpt + 1;
	       if (tpt == tpoly->points + tpoly->number - 1) {
		  if (!(tpoly->style & UNCLOSED))
		     tpt2 = tpoly->points;
		  else continue;
	       }

  	       pinpt = NULL;
	       if (neardist(finddist(endpt2, endpt, tpt))) pinpt = tpt;
	       if (neardist(finddist(endpt2, endpt, tpt2))) pinpt = tpt2;
	       if (neardist(finddist(tpt, tpt2, endpt))) pinpt = endpt;
	       if (neardist(finddist(tpt, tpt2, endpt2))) pinpt = endpt2;

	       if (pinpt != NULL) {	 /* Create new pinlabel */

      		  /* If there is already a pinlabel at this point, then ignore */

      		  for (pseek2 = cinst->thisobject->plist; pseek2 <
		         cinst->thisobject->plist + cinst->thisobject->parts; pseek2++) {
	 	     if ((*pseek2)->type == LABEL) {
	    	        tlab = TOLABEL(pseek2);
	    		if (tlab->pin && nearpoint(&(tlab->position), pinpt)) {
	 	           /* printf(						     */
			   /*	"Info: Pinlabel %s found at point (%d, %d)\n",	     */
		  	   /*   tlab->string+2, tlab->position.x, tlab->position.y); */

	       		   break;
		        }
	    	     }
	 	  }
		  if (pseek2 == cinst->thisobject->plist + cinst->thisobject->parts) {
		     make_tmp_pin(cinst, pinpt);
   		     labeled = True;
		  }
	       }
	    }
	 }
	 if (labeled) break;
      }
   }
   free(tmppts);
}

/*-------------------------------------------------------------------------*/
/* Insert given polygon at top of netlist 				   */
/* Associate polygon with given net id    				   */ 
/*-------------------------------------------------------------------------*/

void linknet(NetlistPtr *nethead, polyptr poly, int netid)
{
   NetlistPtr netptr = NULL;
	
   netptr = (NetlistPtr) malloc(sizeof(Netlist));
   if (netptr != NULL) {
      netptr->poly = poly;
      netptr->netid = netid;

      netptr->next = *nethead;
      *nethead = netptr;
   }
   else {
      fprintf(stderr, "Not enough memory\n");  /* debug printing */
   }
}

/*-------------------------------------------------------------------------*/

long zsign(long a, long b)
{
   if (a > b) return 1;
   else if (a < b) return -1;
   else return 0; 
}

/*----------------------------------------------------------------------*/
/* Basic geometry routine for line crossing detection algorithm		*/
/*----------------------------------------------------------------------*/

int linescross(XPoint *af, XPoint *at, XPoint *bf, XPoint *bt)
{
   char v;

   v = zsign((at->x - bf->x)*(bt->y - bf->y), (at->y - bf->y)*(bt->x - bf->x));
   if ((zsign((bt->x - at->x)*(af->y - at->y),
		(bt->y - at->y)*(af->x - at->x)) != v) ||
         (zsign((af->x - bt->x)*(bf->y - bt->y),
		(af->y - bt->y)*(bf->x - bt->x)) != v) ||
   	 (zsign((bf->x - af->x)*(at->y - af->y),
		(bf->y - af->y)*(at->x - af->x)) != v)) return 0; 
   return 1;
}

/*-------------------------------------------------------------------------*/
/* Check whether two line segments attach to or cross each other	   */
/* Return 1 if attached, 0 otherwise					   */ 
/*-------------------------------------------------------------------------*/

int onsegment(XPoint *af, XPoint *at, XPoint *bf, XPoint *bt, objectptr cschem)
{
   long w;
   genericptr *cgen;
   objinstptr cobj;
   long ddist;

   /* check if any endpoint connects to the other segment */

   w = finddist(af, at, bf);
   if (abs(w) <= ONDIST) return 1;
   w = finddist(af, at, bt);
   if (abs(w) <= ONDIST) return 1;
   w = finddist(bf, bt, af);
   if (abs(w) <= ONDIST) return 1;
   w = finddist(bf, bt, at);
   if (abs(w) <= ONDIST) return 1;

   if (linescross(af, at, bf, bt)) { 

      /* lines cross:  find crossing point and look for "dot" at that point; */
      /* look for any objects with name *dot* (* for wildcard) attaching     */
      /* to both line segments (using finddist())			     */

      for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
         if ((*cgen)->type == OBJECT) {
	    cobj = TOOBJINST(cgen);
	    if (strstr(cobj->thisobject->name, "dot") != NULL) {
	       ddist = finddist(af, at, &(cobj->position)) +
			  finddist(bf, bt, &(cobj->position));
	       if (abs(ddist) <= ONDIST) {
		  /* lines are joined with a dot */
		  return 1;
	       }
            }
         }
      }
   }
	
   return 0;
}

/*--------------------------------------------------------------------------*/
/* Traverse along netlist and change nets of a given id to another given id */
/* Parameters: netlist - pointer to top of netlist                          */
/*	       orignet - original id to be changed			    */
/*	       newnet - new id to be changed to				    */	
/*--------------------------------------------------------------------------*/

void changenet(NetlistPtr netlist, int orignet, int newnet) 
{
   for (; netlist != NULL; netlist = netlist->next) {
      if (netlist->netid == orignet) {
	 netlist->netid = newnet;
      }
   }
}

/*--------------------------------------------------------------------------*/
/* Same as above, for the call list					    */
/* (changes net for both netlist and calllist)				    */
/*--------------------------------------------------------------------------*/

void changecall(objectptr cschem, int orignet, int newnet) 
{
   CalllistPtr calllist = cschem->calllist;
   ParamlistPtr paramlist;
   NetlistPtr netlist = cschem->netlist;

   for (; calllist != NULL; calllist = calllist->next) {
      paramlist = calllist->params;
      while (paramlist != NULL) {
         if (paramlist->netid == orignet) {
	    paramlist->netid = newnet;
	 }
	 paramlist = paramlist->next;
      }
   }

   changenet(netlist, orignet, newnet);
}

/*--------------------------------------------------------------------------*/
/* Same as above, for the pin list					    */
/* (changes net for all of netlist, calllist, and pinlist		    */
/*--------------------------------------------------------------------------*/

void changepin(objectptr cschem, int orignet, int newnet) 
{
   PinlistPtr pinlist = cschem->pinlist;
   NetlistPtr netlist = cschem->netlist;

   for (; pinlist != NULL; pinlist = pinlist->next) {
      if (pinlist->netid == orignet) {
	 pinlist->netid = newnet;
      }
   }
   changecall(cschem, orignet, newnet);
}

/*-------------------------------------------------------------------------*/
/* Remove a call from the call list					   */
/*-------------------------------------------------------------------------*/

void removecall(objectptr cschem, CalllistPtr *callstart)
{
   CalllistPtr *replace, seeklist;
   ParamlistPtr paramlist = (*callstart)->params, savelist;

   /* find the instance call before this one and link it to the one following */
   seeklist = cschem->calllist;
   if (seeklist == *callstart) {
      replace = &(cschem->calllist);
      seeklist = cschem->calllist;
   }
   else {
      while (seeklist->next != *callstart) seeklist = seeklist->next;
      replace = &(seeklist->next);
   }

   while (paramlist != NULL) {
      savelist = paramlist;
      paramlist = paramlist->next;
      free (savelist);
   }

   *replace = (*callstart)->next;
   free(*callstart);
   *callstart = seeklist;
}

/*-------------------------------------------------------------------------*/
/* Resolve pins recursively						   */
/*-------------------------------------------------------------------------*/

void genpinlist(objectptr cschem, Boolean keepcalls)
{
   genericptr *pgen;
   objinstptr pobj, callid;
   objectptr callobj;
   labelptr plabel;
   int netid, globals, globnet, net1, net2;
   PinlistPtr *pinhead = &(cschem->pinlist), ptst, gtst;
   NetlistPtr netlist = cschem->netlist;
   CalllistPtr calllist = cschem->calllist;
   ParamlistPtr paramlist, seeklist, seeklist2;
	
   if (netlist == NULL) return;

   /* recursively execute genpinlist() on all calls in the calllist */

   while (calllist != NULL) {
      if (calllist->callobj->pinlist == NULL)
	 genpinlist(calllist->callobj, keepcalls);

      paramlist = calllist->params;
      while (paramlist != NULL) {

	 /* 1) resolve connecting nets in "jumper"-type objects.  */
	 /*    Ignore unconnected pins (which may appear in 	  */
	 /*    "fundamental" low-level objects.			  */

	 net1 = pintonet(paramlist->pin, calllist->callobj);
         if (net1 != 0) {
	    seeklist = paramlist->next;
	    while (seeklist != NULL) {
	       net2 = pintonet(seeklist->pin, calllist->callobj);
	       if (net2 == net1) {
	          /* printf("Combined nets %d and %d in %s call to %s\n",	*/
		  /*     seeklist->netid, calllist->netid, cschem->name,	*/
		  /*     calllist->callobj->name);				*/
		  /* printf("More info:  net names are %s and %s\n",		*/
		  /*	nettopin(seeklist->netid, cschem, NULL),		*/
		  /*	nettopin(calllist->netid, cschem, NULL));		*/

	          /* combine the two nets on the level above. 	  */
	          /* make the lowest-numbered net the remaining one. */

	          if (paramlist->netid < seeklist->netid)
	             changecall(cschem, seeklist->netid, paramlist->netid);
	          else
	             changecall(cschem, paramlist->netid, seeklist->netid);

		  /* The calls are redundant.  Remove the most recent one */
		  /* found (does this obviate the need for the call to	  */
		  /* changecall()?)					  */
		  
		  seeklist2 = paramlist;
		  while (seeklist2->next != seeklist) seeklist2 = seeklist2->next;
		  /* printf("last pin is %s;  deleting redundant pin %s\n",	*/
		  /*	seeklist2->pin->string + 2, seeklist->pin->string + 2); */
		  seeklist2->next = seeklist->next;
		  free(seeklist);
		  seeklist = seeklist2->next;
		  
	       }
	       else seeklist = seeklist->next;
	    }
	 }
	 paramlist = paramlist->next;
      }

      paramlist = calllist->params;
      while (paramlist != NULL) {

         /* 2) resolve global nets upward from the hierchy bottom */

         if ((globnet = pintonet(paramlist->pin, calllist->callobj)) < 0) {
	    /* printf("Changed net %d to global %d in %s\n",	*/
	    /*	 calllist->netid, globnet, cschem->name);	*/

	    /* change all references in local netlist and calllist */
	    changecall(cschem, paramlist->netid, globnet);
	    
	    paramlist->netid = globnet;
	 }
	 paramlist = paramlist->next;
      }

      /* if the called object has no calls itself (contains only nets), */
      /* then remove the call now that the nets have been resolved.	*/

      if (calllist->callobj->calllist == NULL) {
	 if (calllist->callobj->schemtype != FUNDAMENTAL) {
	    /* printf("Removing call to %s in %s\n", 		*/
	    /*       calllist->callobj->name, cschem->name);	*/
            if (!keepcalls) removecall(cschem, &calllist);
	 }
      }
      calllist = calllist->next;
   }

   /* Now for each pin-label in the object, update the pinlist. */

   for (pgen = cschem->plist; pgen < cschem->plist + cschem->parts; pgen++) {
      if ((*pgen)->type == LABEL) {
	 plabel = TOLABEL(pgen);
	 if (plabel->pin) {
	    netid = whichnet(netlist, &(plabel->position));

	    /* if a pin is declared global, compare it to the names in the */
	    /* global net list, adding to the list if it is not there.	   */
	    /* Change its own net to the global net number (designated by  */
	    /* negative numbers).					   */

	    if (plabel->pin == GLOBAL) {
	       /* printf("Global node %s found in %s\n", plabel->string + 2,	*/
	       /*		cschem->name);					*/
	       gtst = globallist;
	       while (gtst != NULL) {
		  if (!strcmp(gtst->pin->string + 2, plabel->string + 2)) {
	             changepin(cschem, netid, -(gtst->netid));
		     netid = -(gtst->netid);
		     break;
		  }
		  gtst = gtst->next;
	       }
	       if (gtst == NULL) {
		  if (globallist == NULL)
		     globals = 1;
	          else
		     globals = globallist->netid + 1;
	          changepin(cschem, netid, -globals);
	          netid = -globals;
	          makepin(&globallist, plabel, -netid);
	       }
	    }

	    /* if 2 pins share the same name in the object, connect their */
	    /* nets together. 					     */

	    for (ptst = *pinhead; ptst != NULL; ptst = ptst->next) {
	       if (!strcmp(ptst->pin->string + 2, plabel->string + 2)) {
		  /* printf("In %s linking nets %d and %d\n", cschem->name, */
		  /*	netid, ptst->netid); 				    */
		  changepin(cschem, netid, ptst->netid);
		  netid = ptst->netid;
		  break;
	       }
	    }

	    /* add the pin to the pin list */

	    if (netid)
	       makepin(pinhead, plabel, netid);
	 }
      }
   }
}
				
/*-------------------------------------------------------------------------*/
/* Traverse netlist and return netid of polygon on which pin is located    */
/* If pin is not on any polygon, 0 returned                                */
/*-------------------------------------------------------------------------*/

int whichnet (NetlistPtr netlist, XPoint *testpoint) {
   XPoint *tpt, *tpt2;
	
   for (netlist; netlist != NULL; netlist = netlist->next) {
      for (tpt = netlist->poly->points; tpt < netlist->poly->points
		+ netlist->poly->number; tpt++) {

	 tpt2 = tpt + 1;
	 if (tpt == netlist->poly->points + netlist->poly->number - 1) {
	    if (!(netlist->poly->style & UNCLOSED))
	       tpt2 = netlist->poly->points;
	    else continue;
	 }

	 if (finddist(tpt, tpt2, testpoint) <= 1) {
	    return (netlist->netid);
	 }
      }
   }
   return 0;
}

/*-------------------------------------------------------------------------*/
/* Allocate memory for the new pin list element             		   */
/* Define the values for the new pin list element structure 		   */
/* Insert new pin into pin list and sort according to net id value	   */
/* Pin with greatest net id value goes to head of list              	   */
/*-------------------------------------------------------------------------*/

void makepin(PinlistPtr *pinhead, labelptr pin, int netid)
{
   PinlistPtr pinptr, pinpointer = *pinhead;
	
   pinptr = (PinlistPtr) malloc(sizeof(Pinlist));
	
   if (pinptr != NULL) {
      pinptr->pin = pin;
      pinptr->pinx = NULL;
      pinptr->netid = netid;
      pinptr->next = NULL;
   }
   else {
      fprintf(stderr, "Not enough memory\n"); /* Debug printing */
      return;
   }

   if ((*pinhead == NULL)) {
      pinptr->next = *pinhead;
      *pinhead = pinptr;
   }
   else if ((*pinhead)->netid <= pinptr->netid) {
      pinptr->next = *pinhead;
      *pinhead = pinptr;
   }
   else {
      while (pinpointer->next != NULL) {
	 if (pinpointer->next->netid <= pinptr->netid) {
 	    break;
	 }
	 pinpointer = pinpointer->next;
      }
      pinptr->next = pinpointer->next;
      pinpointer->next = pinptr;
   }
}
		
/*-------------------------------------------------------------------------*/
/* Allocate memory for the new call list element             		   */
/* Define the values for the new call list element		   	   */
/* Insert new call into call list					   */
/* Preferable to keep the list in order of parameters calls		   */ 
/*-------------------------------------------------------------------------*/

void makecall(CalllistPtr *callhead, objectptr callobj, labelptr callpin,
	objinstptr callid, int netid)
{
   CalllistPtr tmpcall = NULL, callpointer = *callhead;
   ParamlistPtr tmpparam;
	
   tmpparam = (ParamlistPtr)malloc(sizeof(Paramlist));
   if (tmpparam != NULL) {
      tmpparam->pin = callpin;
      tmpparam->netid = netid;
   }
   else {
      fprintf(stderr, "Not enough memory\n"); /* Debug printing */
   }

   if ((*callhead != NULL) && ((*callhead)->callno == callid)) {
      tmpcall = *callhead;
      tmpparam->next = tmpcall->params;
   }
   else {
      tmpcall = (CalllistPtr) malloc(sizeof(Calllist));
      if (tmpcall != NULL) {
         tmpcall->callobj = callobj;
         tmpcall->callno = callid;
	 tmpparam->next = NULL;
         tmpcall->next = *callhead;
         *callhead = tmpcall;
      }
      else {
         fprintf(stderr, "Not enough memory\n"); /* Debug printing */
      }
   }
   tmpcall->params = tmpparam;
}
		
/*------------------------------------------------------------------------*/
/* pinx keeps track of temporary pin labels when flattening the hierarchy */
/* without destroying the original pin names.				  */
/*------------------------------------------------------------------------*/

void logpinx(char *pinxfrom, labelptr labto, objectptr cschem)
{
   PinlistPtr pinlist = cschem->pinlist;
   int netid = pintonet(labto, cschem);

   while (pinlist != NULL) {
      if (netid == pinlist->netid) {
	 pinlist->pinx = pinxfrom;
	 break;
      }
      pinlist = pinlist->next;
   }
}

/*----------------------------------------------------------------------*/
/* Write a character to the output, avoiding text escapes and NO_OPs	*/
/*----------------------------------------------------------------------*/

void nfprintch(FILE *fp, uchar **strt)
{
   uchar *sptr = *strt;
   uchar tchar = *sptr;

   switch (tchar) {
      case TEXT_ESC:
	 (*strt)++;
	 break;
      case NO_OP:
	 break;
      default:
	 fprintf(fp, "%c", tchar);
	 break;
   }
}

/*-------------------------------------------------------------------------*/
/* Write a low-level device						   */
/*-------------------------------------------------------------------------*/

void writedevice(FILE *fp, char *mode, objectptr cfrom, CalllistPtr clist,
	char *prefix)
{
   genericptr *pgen;
   labelptr plabel;
   objinstptr callid;
   char *pinstr;
   uchar *strt, *fnsh;
   ParamlistPtr paramlist;
   objectptr cschem = clist->callobj;
   int i;
   oparamptr ops;

   if (clist == NULL) {
      fprintf(fp, "error: null device\n");
      return;
   }

   psubstitute(clist->callno);	/* Make parameter substitutions */

   /* Look for information labels in the object parts list. */
   /* (This is not very robust to errors---needs work! */

   for (pgen = cschem->plist; pgen < cschem->plist + cschem->parts; pgen++) {
      if ((*pgen)->type == LABEL) {
	 plabel = TOLABEL(pgen);
         if ((plabel->pin == INFO) && !strncmp(plabel->string + 2, mode, strlen(mode))) {
	    if ((strt = strchr(plabel->string + 2, ':')) != NULL) {
	       for (++strt; *strt != '\0'; strt++) {
		  if (*strt == '%') {
		     strt++;
		     switch(*strt) {
			case 'i':
			   fprintf(fp, "%d", devindex++);
			   break;

			case 'p':
			   /* Pin name either has no spaces or is in quotes */
			   strt++;
			   if (*strt == '"') strt++;
			   if (*strt == '"' || *strt == '\0') break;
			   fnsh = strt + 1;
			   while (*fnsh != '\0' && !isspace(*fnsh) && 
					*fnsh != '"') fnsh++; 
			   strncpy(_STR, strt, (int)(fnsh - strt));
			   _STR[(int)(fnsh - strt)] = '\0';
			   strt = fnsh;
			   if (isspace(*fnsh) || *fnsh == '\0') strt--;

			   /* Compare this name against the calllist */
			   paramlist = clist->params;
			   while (paramlist != NULL) {
			      if (!strcmp(paramlist->pin->string + 2, _STR)) {
	            		 fprintf(fp, "%s", nettopin(paramlist->netid,	
					    cfrom, prefix));
			         break;
			      }
			      paramlist = paramlist->next;
			   }
			   if (paramlist == NULL) {
			      sprintf(_STR2, "No pin named %s in device %s",
				  _STR, cschem->name);
			      Wprintf(_STR2);
			   }
			   break;

			case 'v':
			   /* Parameter name either has no spaces or is in quotes */
			   strt++;
			   if (*strt == '"') strt++;
			   if (*strt == '"' || *strt == '\0') break;
			   fnsh = strt + 1;
			   while (*fnsh != '\0' && !isspace(*fnsh) && 
					*fnsh != '"') fnsh++; 
			   strncpy(_STR, strt, (int)(fnsh - strt));
			   _STR[(int)(fnsh - strt)] = '\0';
			   strt = fnsh;
			   if (isspace(*fnsh) || *fnsh == '\0') strt--;

			   /* Compare this name against the parameter defaults */
			   for (i = 0; i < cschem->num_params; i++) {
			      ops = *(cschem->params + i);
			      if (ops->type == XC_STRING) {
			         if (!strncmp((char *)ops->pdefault, _STR,
					strlen(_STR))) {
				    /* substitute the parameter or default */
				    char *optr;
				    short slen;
				    if ((clist->callno->num_params > i) &&
					   *(clist->callno->params + i) != NULL)
				       optr = (char *)(*(clist->callno->params + i));
				    else
				       optr = (char *)ops->pdefault;
				    if ((slen = natstrlen(optr)) > 0) {
				       strncpy(_STR, optr, slen);
				       _STR[slen] = '\0';
				       fputs(_STR, fp);
				    }
			            break;
				 }
			      }
			   }
			   if (i == cschem->num_params) {
			      sprintf(_STR, "No parameter named %s in device %s",
				  _STR, cschem->name);
			      Wprintf(_STR);
			   }
			   break;

			default:
			   nfprintch(fp, &strt);
		     }
		  }
	          else nfprintch(fp, &strt);
	       }
	       fprintf(fp, "\n");
	       break;
	    }
	 }
      }
   }

   if (pgen == cschem->plist + cschem->parts) {

      /* No info about this netlist mode.  Flag a warning and print a default */
      /* type netlist.							   */

      sprintf(_STR, "No info for %s netlist in device %s", mode, cschem->name);
      Wprintf(_STR);

      paramlist = clist->params;
      pinstr = nettopin(paramlist->netid, cfrom, prefix);
      fprintf(fp, "   %s (%s", cschem->name, pinstr);
      paramlist = paramlist->next;

      while (paramlist != NULL) {
         pinstr = nettopin(paramlist->netid, cfrom, prefix);
         fprintf(fp, ", %s", pinstr);
         paramlist = paramlist->next;
      }
      fprintf(fp, ")\n");
   }
}

/*-------------------------------------------------------------------------*/
/* Get rid of locally-defined pin names	 				   */ 
/*-------------------------------------------------------------------------*/

void clearpins(objectptr cschem)
{
   PinlistPtr pinlist = cschem->pinlist;
  
   while (pinlist != NULL) {
      /* if (pinlist->pinx != NULL)	*/
      /*    free(pinlist->pinx);	*/
      pinlist->pinx = NULL;
      pinlist = pinlist->next;
   }
}

/*-------------------------------------------------------------------------*/
/* Save netlist into a flattened sim file				   */
/*-------------------------------------------------------------------------*/

void writesimflat(objectptr cschem, CalllistPtr cfrom, char *prefix, FILE *fp)
{
   CalllistPtr calllist = cschem->calllist;
   ParamlistPtr paramlist;
   int i, j, item = 1, netidx;
   objinstptr callid;
   char *newprefix = (char *)malloc(sizeof(char));

   netindex = 1;

   /* write all the subcircuits */

   while (calllist != NULL) {

      if (calllist->callobj->schemtype == FUNDAMENTAL) {
         writedevice(fp, "sim", cschem, calllist, prefix);
      }
      else {
         paramlist = calllist->params;
         while (paramlist != NULL) {
	    logpinx(nettopin(paramlist->netid, cschem, prefix),
		paramlist->pin, calllist->callobj);
	    paramlist = paramlist->next;
         }
	 sprintf(_STR, "%s%d", calllist->callobj->name, item++);
	 newprefix = (char *)realloc(newprefix, sizeof(char) * (strlen(prefix)
		   + strlen(_STR) + 2));
	 sprintf(newprefix, "%s%s/", prefix, _STR);
	 netidx = netindex;
         writesimflat(calllist->callobj, calllist, newprefix, fp);
	 netindex = netidx;
      }
      calllist = calllist->next;
   }
   clearpins(cschem);
   free(newprefix);
}

/*-------------------------------------------------------------------------*/
/* Write out the list of global nets and their pin names		   */
/*-------------------------------------------------------------------------*/

void writeglobals(objectptr cschem, FILE *fp)
{
   PinlistPtr pinptr;

   pinptr = globallist;
   while (pinptr != NULL) {
      fprintf(fp, "%s = %d\n", pinptr->pin->string + 2, -pinptr->netid);
      pinptr = pinptr->next;
   }
   fprintf(fp, "\n");
}

/*-------------------------------------------------------------------------*/
/* Save netlist into a hierarchical file				   */
/*-------------------------------------------------------------------------*/

void writehierarchy(objectptr cschem, CalllistPtr cfrom, FILE *fp)
{
   CalllistPtr calllist = cschem->calllist;
   ParamlistPtr paramlist;
   objinstptr curcall;
   int netid;
   char *stsave;

   /* make sure that all the subcircuits have been written first */

   for (; calllist != NULL; calllist = calllist->next) {
      if (calllist->callobj->traversed == False) {
         calllist->callobj->traversed = True;
         writehierarchy(calllist->callobj, calllist, fp);
      }
   }

   /* write own subcircuit netlist */

   paramlist = (cfrom == NULL) ? NULL : cfrom->params;

   if (cschem->schemtype == FUNDAMENTAL)
      return;

   else if (paramlist != NULL) {
      fprintf(fp, ".subckt %s", cschem->name);

      /* List of parameters to subcircuit. 			  */
      /* Each parameter connects to a net which may have multiple */
      /* names, so find the net associated with the parameter     */
      /* and convert it back into a pin name in the usual manner  */
      /* using nettopin().					  */

      while (paramlist != NULL) {
	 netid = pintonet(paramlist->pin, cschem);
         fprintf(fp, " %s", nettopin(netid, cschem, NULL));
	 paramlist = paramlist->next;
      }
      fprintf(fp, "\n");
   }

   calllist = cschem->calllist;
   while (calllist != NULL) {

      /* The call is to a fundamental device. . . */

      if (calllist->callobj->schemtype == FUNDAMENTAL) {
         writedevice(fp, "spice", cschem, calllist, NULL);
      }

      /* . . . or else is a call to a subcircuit */

      else {
         fprintf(fp, "X%d", subindex++);
	 stsave = calllist->callobj->name;
	 paramlist = calllist->params;
	 while (paramlist != NULL) {
	    fprintf(fp, " %s", nettopin(paramlist->netid, cschem, NULL));
	    paramlist = paramlist->next;
         }
         fprintf(fp, " %s\n", stsave);
      }
      calllist = calllist->next;
   }
   if (cfrom == NULL)
      fprintf(fp, ".end\n");
   else
      fprintf(fp, ".ends\n\n");
}

/*-------------------------------------------------------------------------*/

void writenet(objectptr cschem, char *mode, char *suffix)
{
   char filename[100], *str;
   char *prefix, *cpos;
   FILE *fp;

   netindex = devindex = subindex = 1;
 
   if ((cpos = strchr(cschem->name, ':')) != NULL) *cpos = '\0';
   sprintf(filename, "%s.%s", cschem->name, suffix);
   if (cpos != NULL) *cpos = ':';

   if ((fp = fopen(filename, "w")) != NULL) {
      if (!strcmp(mode, "spice")) {
         /* writeglobals(cschem, fp); */
         fprintf(fp, "Spice circuit %s\n\n", cschem->name);
         writehierarchy(cschem, NULL, fp);
      }
      else {
         prefix = (char *)malloc(sizeof(char));
	 *prefix = '\0';
         if (!strcmp(mode, "sim"))
	    writesimflat(cschem, NULL, prefix, fp);
         else if (!strcmp(mode, "pcb")) {
	    ptable = (struct Ptab *)NULL;
	    writepcb(cschem, NULL, "");
	    outputpcb(cschem, fp);
	    freepcb();
	 }
         free(prefix);
      }
      fclose(fp);
      sprintf(_STR, "%s netlist saved as %s", mode, filename);
   }
   else {
      sprintf(_STR, "Could not open file %s for writing.", filename);
   }
   Wprintf(_STR);
}

/*-------------------------------------------------------------------------*/
/* Find a pin name for the given net number				   */
/* Either take the last pin name with the net, or generate a new pin,	   */
/* assigning it a net number for a string.				   */
/* prefix = NULL indicates spice netlist				   */
/*-------------------------------------------------------------------------*/

char *nettopin(int netid, objectptr cschem, char *prefix)
{
   PinlistPtr pinlist = cschem->pinlist, psave;
   char *newstring;
   labelptr newlabel;

   if (prefix == NULL) {   	/* spice netlist */
      if (netid < 0) {
         pinlist = globallist;
         netid = -netid;
      }

      psave = (PinlistPtr)NULL;
      while (pinlist != NULL) {
         if ((pinlist->netid == netid) && (*(pinlist->pin->string) != 0))
	    return pinlist->pin->string + 2;
	 else if (pinlist->netid == netid) psave = pinlist;
         pinlist = pinlist->next;
      }
      if (psave != NULL) return psave->pin->string + 2;

      /* Pinlist is NULL: make a new pin and assign a net ID number */

      newlabel = (labelptr) malloc(sizeof(label));
      newlabel->type = LABEL;
      newlabel->pin = False;
      sprintf(_STR, "net.%d", netindex++);
      newlabel->string = (char *)malloc((2 + sizeof(_STR)) *
	     sizeof(char));
      sprintf(newlabel->string, "%c%c%s", TEXT_ESC, FONT_START, _STR);
      makepin(&(cschem->pinlist), newlabel, netid);

      return (newlabel->string + 2);
   }

   /* sim netlist */

   if (netid < 0) {
      pinlist = globallist;
      netid = -netid;
   }
   while (pinlist != NULL) {
      if (pinlist->netid == netid) {
	 if (pinlist->pinx != NULL)
	    return pinlist->pinx;
	 else if (pinlist->pin && (pinlist->pin->pin == GLOBAL))
	    return (pinlist->pin->string + 2);
	 else 
	    break;
      }
      pinlist = pinlist->next;
   }

   /* Generate the string for the local instantiation of this pin	*/
   /* If this node does not have a pin, create one using the current	*/
   /* hierarchy prefix as the string.					*/

   if (pinlist != NULL)
      strcpy(_STR, pinlist->pin->string + 2);
   else
      sprintf(_STR, "net%d", netindex++);  /* for now. . . */

   newstring = (char *)malloc((strlen(_STR) + strlen(prefix) + 3)
		* sizeof(char));
   sprintf(newstring, "%s%s", prefix, _STR);

   /* If this is the first time at this point in the schematic,   */
   /* then the pinlist is NULL and a new pin needs to be created. */
   /* Use pin = False so that memory freeing routine knows to     */
   /* destroy the label after the netlist is written.		  */

   if (pinlist == NULL) {
      newlabel = (labelptr) malloc(sizeof(label));
      newlabel->type = LABEL;
      newlabel->pin = False;
      newlabel->string = (char *)malloc((2 + sizeof(_STR)) *
	  sizeof(char));
      sprintf(newlabel->string, "%c%c%s", TEXT_ESC, FONT_START, _STR);
      makepin(&(cschem->pinlist), newlabel, netid);
   }
   
   return newstring;
}

/*-------------------------------------------------------------------------*/
/* Find the net which connects to the given pin label			   */
/*-------------------------------------------------------------------------*/

int pintonet(labelptr testpin, objectptr cschem)
{
   PinlistPtr pinlist = cschem->pinlist;

   while(pinlist != NULL) {
      if (pinlist->pin != NULL)
         if (pinlist->pin->string == testpin->string)
	    return pinlist->netid;
      pinlist = pinlist->next;
   }

   /* If there was no net found, try equating the strings themselves,   */
   /* i.e., otherwise unconnected I/O pin has same name as another pin. */

   pinlist = cschem->pinlist;
   while(pinlist != NULL) {
      if (!strcmp(pinlist->pin->string + 2, testpin->string + 2))
	  return pinlist->netid;
      pinlist = pinlist->next;
   }

   /* otherwise return 0 */

   return 0;
}

/*-------------------------------------------------------------------------*/
/* Flatten netlist and save into a hash table of pcb-style nets		   */
/*-------------------------------------------------------------------------*/

void writepcb(objectptr cschem, CalllistPtr cfrom, char *prefix)
{
   NetlistPtr netlist = cschem->netlist;
   CalllistPtr calllist = cschem->calllist;
   ParamlistPtr paramlist;
   int i, j, item = 1, testnet, netidx, tmplen;
   objinstptr callid;
   char *newprefix = (char *)malloc(sizeof(char));
   struct Ptab *hidx, *htmp;
   struct Pstr *tmpstr;
   struct Pnet *tmpnet;

   netindex = 1;

   /* Step 1:  Go through the netlist of this object and compile a table of nets. */

   while (netlist != NULL) {
      testnet = netlist->netid;
      hidx = ptable;
      while (hidx != NULL) {
         if (hidx->nets != NULL) {
	    if (hidx->nets->netidx == testnet) break;
	 }
	 hidx = hidx->next;
      }
      if (hidx == NULL) {	/* make new entry for net in hash table */
	 htmp = (struct Ptab *)malloc(sizeof(struct Ptab));
	 tmpnet = (struct Pnet *)malloc(sizeof(struct Pnet));
	 tmpnet->netidx = testnet;
	 tmpnet->next = NULL;
	 htmp->nets = tmpnet;
	 htmp->pins = NULL;
	 htmp->next = ptable;
	 ptable = hidx = htmp;
	 /* printf("Added new index entry for net %d\n", testnet); */
      }
      netlist = netlist->next;
   }

   /* Step 2:  Go through the list of calls to search for endpoints */

   while (calllist != NULL) {

      /* Step 3:  If call is to a bottom-most schematic, output the device connections */

      sprintf(_STR, "%s%d", calllist->callobj->name, item++);
      newprefix = (char *)realloc(newprefix, sizeof(char) * (strlen(prefix)
		+ strlen(_STR) + 2));
      sprintf(newprefix, "%s%s/", prefix, _STR);


      if (calllist->callobj->calllist == NULL) {

         /* printf("Reached lowest-level schematic:  Finding connections\n"); */
         paramlist = calllist->params;
         while (paramlist != NULL) {

	    hidx = ptable;
	    while (hidx != NULL) {
	       if (hidx->nets != NULL)
	          if (hidx->nets->netidx == paramlist->netid)
		     break;
	       hidx = hidx->next;
	    }
	    if (hidx == NULL) {
	       printf("Warning:  Unconnected pin %s%s\n", newprefix,
			paramlist->pin->string + 2);
	    }
	    else {
	       tmpstr = (struct Pstr *)malloc(sizeof(struct Pstr));
	       tmpstr->string = (char *)malloc(strlen(newprefix)
			+ strlen(paramlist->pin->string + 2) + 1);
	       strcpy(tmpstr->string, newprefix);
	       /* Replace slash '/' with dash '-' at bottommost level */
	       if ((tmplen = strlen(newprefix)) > 0)
	          tmpstr->string[tmplen - 1] = '-';
	       strcat(tmpstr->string, paramlist->pin->string + 2);
	       tmpstr->next = hidx->pins;
	       hidx->pins = tmpstr;

	       /* printf("Logged net %d pin %s\n", hidx->nets->netidx,	*/
	       /*	tmpstr->string);				*/
            }
	    paramlist = paramlist->next;
         }
      }
      else {

         /* Step 3a: Push current net translations */
	 /* (Don't push or pop global net numbers:  no translation needed!) */

         hidx = ptable;
         while (hidx != NULL) {
            if ((hidx->nets != NULL) && (hidx->nets->netidx >= 0)) {
               tmpnet = (struct Pnet *)malloc(sizeof(struct Pnet));
               tmpnet->netidx = 0;
               tmpnet->next = hidx->nets;
               hidx->nets = tmpnet;
            }
            hidx = hidx->next;
         }

	 /* Step 3b: Generate net translation table for each subcircuit */

         paramlist = calllist->params;
         while (paramlist != NULL) {
            hidx = ptable; 		/* Generate new net translations */
            while (hidx != NULL) {
               if (hidx->nets != NULL)
	          if (hidx->nets->next != NULL)
		     if (hidx->nets->next->netidx == paramlist->netid) break;
               hidx = hidx->next;
            }
	    if (hidx != NULL) {
	       hidx->nets->netidx = pintonet(paramlist->pin, calllist->callobj);
	       /* printf("Translation: net %d in object %s is net %d in object %s\n", */
	       /*    paramlist->netid, cschem->name, hidx->nets->netidx,	      */
	       /*    calllist->callobj->name);					      */
	    }
            paramlist = paramlist->next;
         }

         /* Step 3c: Run routine recursively on the subcircuit */

         netidx = netindex;
	 /* printf("Recursive call of writepcb() to %s\n", calllist->callobj->name); */
         writepcb(calllist->callobj, calllist, newprefix);
         netindex = netidx;

         /* Step 3d: Pop the translation table */
	 /* (Don't pop global nets (designated by negative net number)) */

         hidx = ptable;
         while (hidx != NULL) {
            if ((hidx->nets != NULL) && (hidx->nets->netidx >= 0)) {
	       tmpnet = hidx->nets->next;
	       free(hidx->nets);
	       hidx->nets = tmpnet;
            }
            hidx = hidx->next;
         }
      }
      calllist = calllist->next;
   }

   /* Step 4: Cleanup */

   clearpins(cschem);
   free(newprefix);
}

/*-------------------------------------------------------------------------*/
/* Save PCB hash table into pcb-style file				   */
/*-------------------------------------------------------------------------*/

void outputpcb(objectptr cschem, FILE *fp)
{
   int netidx = 1, ccol;
   struct Ptab *pseek;
   struct Pstr *sseek;

   for (pseek = ptable; pseek != NULL; pseek = pseek->next) {
      if (pseek->pins != NULL) {
	 if (pseek->nets != NULL)
             sprintf(_STR, "%s", nettopin(pseek->nets->netidx, cschem, ""));
	 else
             sprintf(_STR, "NET%d ", netidx++);
         fprintf(fp, "%-11s ", _STR);
         ccol = 12;
         for (sseek = pseek->pins; sseek != NULL; sseek = sseek->next) {
	    ccol += strlen(sseek->string) + 3;
	    if (ccol > 78) {
	       fprintf(fp, "\\\n              ");
      	       ccol = 18 + strlen(sseek->string);
	    }
	    fprintf(fp, "%s   ", sseek->string);
         }
         fprintf(fp, "\n");
      }
/*    else fprintf(fp, "NET%d	*UNCONNECTED*\n", netidx++);	*/
   }
}

/*----------------------------------------------*/
/* free memory allocated to PCB net hash tables */
/*----------------------------------------------*/

void freepcb()
{
   struct Ptab *pseek, *pseek2;
   struct Pstr *sseek, *sseek2;
   struct Pnet *nseek, *nseek2;

   pseek = ptable;
   pseek2 = pseek;

   while (pseek2 != NULL) {
      pseek = pseek->next;

      sseek = pseek2->pins;
      sseek2 = sseek;
      while (sseek2 != NULL) {
	 sseek = sseek->next;
	 free(sseek2->string);
	 free(sseek2);
	 sseek2 = sseek;
      }

      nseek = pseek2->nets;
      nseek2 = nseek;
      while (nseek2 != NULL) {
	 nseek = nseek->next;
	 free(nseek2);
	 nseek2 = nseek;
      }

      free(pseek2);
      pseek2 = pseek;
   }
}

/*-------------------------------------------------------------------------*/
/* Free memory allocated for netlist 					   */
/*-------------------------------------------------------------------------*/

void freenets(objectptr cschem)
{
   NetlistPtr netlist = cschem->netlist, nptr = netlist;
   PinlistPtr pinlist = cschem->pinlist, pptr = pinlist;
   CalllistPtr calllist = cschem->calllist, cptr = calllist;
   ParamlistPtr paramlist, parptr;
   genericptr *cgen;
   objinstptr cobj;
   objectptr callobj;
	
   /* Recursively call freenets() on all subobjects, a la gennetlist() */

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {
 	 cobj = TOOBJINST(cgen);

	 if (cobj->thisobject->symschem != NULL)
	    callobj = cobj->thisobject->symschem;
	 else
	    callobj = cobj->thisobject;
	
	 /* Don't infinitely recurse if object is on its own schematic */

	 if (callobj != cschem) freenets(callobj);
      }

      /* Free any temporary labels which have been created	 */
      /* (All elements after the 1st temp label are temp labels) */

      else if ((*cgen)->type == LABEL) {
	 labelptr clab = TOLABEL(cgen);
	 int tmpval = (int)(cgen - cschem->plist);
	 if (*(clab->string) == 0) {
	    while (cgen < cschem->plist + cschem->parts) {
	       clab = TOLABEL(cgen);
	       free(clab->string);
	       free(clab);
	       cgen++;
	    }
	    cschem->parts = tmpval;
	    break;
	 }
      }
   }

   /* Free the allocated structures for this object */

   while (cptr != NULL) {
      calllist = calllist->next;
      paramlist = cptr->params;
      parptr = paramlist; 
      while (parptr != NULL) {
	 paramlist = paramlist->next;
	 parptr->next= NULL;
	 free(parptr);
	 parptr = paramlist;
      }
      cptr->next = NULL;
      free(cptr);
      cptr = calllist;
   }

   while (nptr != NULL) {
      netlist = netlist->next;
      if (nptr->poly->number == 1) {   /* free single-point polys */
	 free (nptr->poly->points);
	 free (nptr->poly);
      }
      nptr->next = NULL;
      free (nptr);
      nptr = netlist;
   }

   while (pptr != NULL) {
      pinlist = pinlist->next;
      if (pptr->pin)
         if (pptr->pin->pin == False) {
	    if (pptr->pin->string != NULL) {
               free(pptr->pin->string);
               pptr->pin->string = NULL;
	    }
	    free(pptr->pin);
            pptr->pin = NULL;
         }
      pptr->next = NULL;
      free(pptr);
      pptr = pinlist;
   }

   cschem->netlist = NULL;
   cschem->pinlist = NULL;
   cschem->calllist = NULL;
   cschem->traversed = False;
}

/*-------------------------------------------------------------------------*/
/* Free the global pin list 					   	   */
/*-------------------------------------------------------------------------*/

void freeglobals()
{
   PinlistPtr pinlist = globallist, pptr = pinlist;

   while (pptr != NULL) {
      pinlist = pinlist->next;
      pptr->next = NULL;
      free(pptr);
      pptr = pinlist;
   }
   globallist = NULL;
}

#endif
/*-------------------------------------------------------------------------*/
