/*----------------------------------------------------------------------*/
/* parameter.c								*/
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	*/
/*----------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/26/99    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "xcircuit.h"

extern Globaldata xobjs;
extern Clientdata areastruct;
extern objectpair *pushlist;
extern short pushes;
extern short textpos, textend;
extern char _STR[150];

oparamptr parampos(objectptr, labelptr, char *, short *, short *);
oparamptr paramcross(objectptr, labelptr);
short paramlen(uchar *);

/*-------------------------------------------------------------------------*/

/* transformation types are enumerated as follows: */

enum transform {IDENT = 0, SUBST, INSERT, DELETE};

/* (recursive) structure for a transition between any two letters */

struct pathnode {
   char   lfrom[3], lto[3];     /* Letters involved in transform */
   int	  from, to;		/* Positions of transform	 */	  
   int    cost;                 /* Levenshtein distance          */
   struct pathnode *previous;   /* Previous path taken           */
   struct pathnode *next;       /* Next path to take             */
   int    transform;            /* Type of transformation made   */
};

/*--------------------------------------------------------------*/
/* Reattach parameter hooks from original string to new one	*/
/*--------------------------------------------------------------*/

void rehook(objectptr localdata, char *origstr, char *newstr)
{
   short paramno, i;
   oparamptr ops;
   char *tmpptr = newstr;
   int displace = (int)(origstr - newstr);

   if (localdata->params == NULL) return;   /* Any parameters? */

   while (*tmpptr != '\0') tmpptr++;  /* tmpptr points to end of string */
   tmpptr += displace;

   /* (origstr,tmpstr) bounds what used to be the string before its	*/
   /* memory was reallocated.						*/

   for (paramno = 0; paramno < localdata->num_params; paramno++) {
      ops = localdata->params[paramno];
      for (i = 0; i < ops->number; i++) {
	 if ((char *)ops->position[i] < tmpptr &&
	    	(char *)ops->position[i] >= origstr)
	    ops->position[i] -= displace;
      }
   }
}

/*--------------------------------------------------------------*/
/* Adjust parameter hooks when string is altered by having	*/
/* shifted by "width" characters after position "ppos".   	*/
/*--------------------------------------------------------------*/

void hookexpand(objectptr localdata, char *ppos, short width)
{
   short paramno, i;
   oparamptr ops;
   char *tmpptr = ppos;

   if (localdata->params == NULL) return;   /* Any parameters? */

   while (*tmpptr++ != '\0');	/* tmpptr points to end of string */
   tmpptr -= width;		/* points to original end of string */
   ppos -= width;

   for (paramno = 0; paramno < localdata->num_params; paramno++) {
      ops = localdata->params[paramno];
      for (i = 0; i < ops->number; i++)
	 if ((char *)ops->position[i] < tmpptr &&
	    	(char *)ops->position[i] >= ppos)
	    ops->position[i] += width;
   }
}

/*--------------------------------------------------------------*/
/* make a short integer parameter				*/
/*--------------------------------------------------------------*/

void makeshortp(genericptr *gelem)
{
   oparamptr ops;

   /* cannot form a parameter on a top-level page */
   if (is_page(objectdata) != -1) {
      Wprintf("Cannot form a parameter in a top-level page!");
      return;
   }

   if (objectdata->params == NULL) {
      objectdata->num_params = 1;
      objectdata->params = (oparamptr *)malloc(sizeof(oparamptr));
   }
   else {
      objectdata->num_params++;
      objectdata->params = (oparamptr *)realloc(objectdata->params,
		objectdata->num_params * sizeof(oparamptr));
   }
   *(objectdata->params + objectdata->num_params - 1) = (oparamptr)
	malloc(sizeof(oparam));

   ops = *(objectdata->params + objectdata->num_params - 1);
   ops->type = XC_SHORT;
   ops->number = 1;
   ops->pdefault = (short *)malloc(sizeof(short));
   ops->position = (uchar **)malloc(sizeof(void *));

   switch((*gelem)->type) {	/* to be completed */
      case LABEL:
	 *(ops->pdefault) = TOLABEL(gelem)->position.x;
	 *(ops->position) = (uchar *)(&(TOLABEL(gelem)->position.x));
	 break;
      case ARC:
	 *(ops->pdefault) = TOARC(gelem)->position.x;
	 *(ops->position) = (uchar *)(&(TOARC(gelem)->position.x));
	 break;
      case OBJECT:
	 *(ops->pdefault) = TOOBJINST(gelem)->position.x;
	 *(ops->position) = (uchar *)(&(TOOBJINST(gelem)->position.x));
	 break;
   }
}

/*--------------------------------------------------------------*/
/* remove a short integer parameter				*/
/*--------------------------------------------------------------*/

void unmakeshortp(genericptr *gelem)
{
}

/*--------------------------------------------------------------------*/
/* Do strcpy() on a string but don't copy parameter boundary markings */
/*--------------------------------------------------------------------*/

void noparmstrcpy(uchar *newstring, uchar *oldstring)
{  
   uchar *sptr = oldstring;
   uchar *pptr = newstring;
   
   do {
      if (*sptr == TEXT_ESC && (*(sptr + 1) == PARAM_START ||
		*(sptr + 1) == PARAM_END))
	 sptr++;
      else if (*sptr != NO_OP) *pptr++ = *sptr;
   } while (*sptr++ != '\0');
}
         
/*--------------------------------------------------------------*/
/* Insert an existing parameter into a string.			*/
/* (For now, just use the first string parameter found.	 To be  */
/* enhanced to prompt for which one, if more than one exists)	*/
/*--------------------------------------------------------------*/

void insertparam()
{
   labelptr tlab;
   oparamptr ops;
   short i, j, plen, tmppos, nparms = 0, selparm = -1;
   char *sptr;

   /* First pass is to check how many string parameters there are:	*/
   /* If only one, then automatically use it.  Otherwise, prompt	*/
   /* for which parameter to use.					*/

   for (i = 0; i < objectdata->num_params; i++) {
      ops = *(objectdata->params + i);
      if (ops->type == XC_STRING) {
	 nparms++;
	 selparm = i;
      }
   }
   if (nparms > 1) {
      nparms = 0;
      sptr = _STR;
      strcpy(sptr, "Choose: ");
      sptr += strlen(sptr);
      for (i = 0; i < objectdata->num_params; i++) {
	 ops = *(objectdata->params + i);
	 if (ops->type == XC_STRING) {
	    nparms++;
	    if (nparms != 1) {
	       strcat(sptr, ", ");
	       sptr += 2;
	    }
	    sprintf(sptr, "%d = \"%s\"", nparms, (char *)ops->pdefault);
            sptr += strlen(sptr);
	 }
      }
      Wprintf(_STR);

      /* This is a bit quirky;  but works for now.  Later, would prefer */
      /* a GUI-oriented selection mechanism.				*/
      selparm = getkeynum();
   }

   if ((selparm >= 0) && (selparm < objectdata->num_params)) {
      ops = *(objectdata->params + selparm);
      tmppos = textpos;
      labeltext(PARAM_START, 2);
      plen = strlen((char *)ops->pdefault);
      for (j = 0; j < plen; j++)
	 labeltext(*((char *)ops->pdefault + j), 0);
      labeltext(PARAM_END, 2);
      tlab = TOLABEL(EDITPART);
      ops->number++;
      ops->position = (uchar **)realloc(ops->position,
		ops->number * sizeof(void *));
      *(ops->position + ops->number - 1) = (void *)(tlab->string + tmppos);
   }
   else
      Wprintf("Parameter number nonexistant.");
}

/*--------------------------------------------------------------*/
/* Parameterize a label string.					*/
/*--------------------------------------------------------------*/

void makeparam(labelptr thislabel)
{
   oparamptr ops;
   uchar *spos, *newstring, *npos;
   int llen = strlen(thislabel->string);

   /* cannot form a parameter on a top-level page */
   if (is_page(objectdata) != -1) {
      Wprintf("Cannot form a parameter in a top-level page!");
      return;
   }

   newstring = (uchar *)malloc(strlen(thislabel->string) + 5);
   spos = thislabel->string;
   npos = newstring;

   /* make sure this does not overlap another parameter */

   if (paramcross(objectdata, thislabel) != NULL) {
      Wprintf("Parameters cannot be nested!");
      textend = 0;
      return;
   }

   if (textend > 0 && textend < textpos) {
      short plen = textpos - textend;
      memmove(npos, spos, textend);
      npos += textend;
      spos += textend;
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_START;
      memmove(npos, spos, plen);
      npos += plen;
      spos += plen;
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_END;
      memmove(npos, spos, strlen(spos) + 1);
   }
   else {
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_START;
      strcpy(npos, spos);
      npos += strlen(spos);
      *(npos++) = TEXT_ESC;
      *(npos++) = PARAM_END;
      *(npos++) = '\0';
   }
   
   rehook(objectdata, (char *)thislabel->string, (char *)newstring);
   hookexpand(objectdata, npos, 4);
   free(thislabel->string);
   thislabel->string = newstring;
   spos = thislabel->string + 2;

   /* insert temporary string end delimiter in place of TEXT_ESC */
   if (textend > 0 && textend < textpos) {
      spos = thislabel->string + textend + 2;
      *(spos - textend + textpos) = '\0';
   }

   if (objectdata->params == NULL) {
      objectdata->num_params = 1;
      objectdata->params = (oparamptr *)malloc(sizeof(oparamptr));
   }
   else {
      objectdata->num_params++;
      objectdata->params = (oparamptr *)realloc(objectdata->params,
		objectdata->num_params * sizeof(oparamptr));
   }
   *(objectdata->params + objectdata->num_params - 1) = (oparamptr)
	malloc(sizeof(oparam));

   ops = *(objectdata->params + objectdata->num_params - 1);
   ops->type = XC_STRING;
   ops->number = 1;

   ops->pdefault = (short *)malloc(strlen(spos) + 1);
   ops->position = (uchar **)malloc(sizeof(void *));
   *(ops->position) = spos - 2;
   strcpy((char *)ops->pdefault, spos);
   if (textend > 0 && textend < textpos)
      *(thislabel->string + textpos + 2) = TEXT_ESC;
   /* printf("default string is %s\n", (char *)ops->pdefault); */

   textend = 0;
}

/*--------------------------------------------------------------*/
/* Search and destroy the selected parameter in all instances 	*/
/* of the specified object.					*/
/*--------------------------------------------------------------*/

void searchinst(objectptr topobj, objectptr refobj, short pnum)
{
   objinstptr tinst;
   genericptr *pgen;
   short k;

   if (topobj == NULL) return;

   for (pgen = topobj->plist; pgen < topobj->plist + topobj->parts; pgen++) {
      if ((*pgen)->type == OBJECT) {
	 tinst = TOOBJINST(pgen);
	 if (tinst->thisobject == refobj) {
	    if (tinst->num_params > 0) {
	       for (k = pnum; k < tinst->num_params - 1; k++)
	          *(tinst->params + k) = *(tinst->params + k + 1);
	       tinst->num_params--;
	    }
	 }
      }
   }
}

/*--------------------------------------------------------------*/
/* Check if this string contains a parameter			*/
/*--------------------------------------------------------------*/

char *searchparam(char *tstr)
{
   char *rval = tstr;
   while (*rval != TEXT_ESC || *(rval + 1) != PARAM_END) {
      if (*(rval + 1) == '\0') {
	 rval = NULL;
	 break;
      }
      rval++;
   }
   return rval;
}

/*--------------------------------------------------------------*/
/* Remove parameterization from a label string or substring.	*/
/*--------------------------------------------------------------*/

void unmakeparam(labelptr thislabel, short tptr)
{
   oparamptr ops;
   short i, j, k, tend;
   uchar *spos;

   /* make sure there is a parameter here */

   if ((ops = parampos(objectdata, NULL, (char *)thislabel->string + tptr,
		&j, &i)) == NULL) {
      Wprintf("There is no parameter here.");
      return;
   }

   tend = tptr;
   while (*(thislabel->string + tend) != TEXT_ESC || *(thislabel->string + tend + 1)
	!= PARAM_END) tend++;

   /* Step 1: Pad over parameter begin and end points with NO_OP characters */

   spos = thislabel->string + tptr;
   *(spos++) = NO_OP;
   *(spos++) = NO_OP;
   spos += tend - tptr - 2;
   *(spos++) = NO_OP;
   *(spos++) = NO_OP;
   
   /* Step 2: Remove object parameter */

   if (ops != *(objectdata->params + i)) {
      printf("Something wrong in return values from parampos()\n");
      return;
   }

   for (j = 0; j < ops->number; j++) {
      if (*((char **)ops->position + j) == (char *)(thislabel->string + tptr)) {
	 for (k = j; k < ops->number - 1; k++)
	    *((char **)ops->position + k) = *((char **)ops->position + k + 1);
	 ops->number--;
	 break;
      }
   } 
   if (ops->number == 0) {
      free(ops->pdefault);
      ops->pdefault = NULL;
      free(ops->position);
      ops->position = NULL;
      free(ops);
      for (k = i; k < objectdata->num_params - 1; k++)
         *(objectdata->params + k) = *(objectdata->params + k + 1);

      objectdata->num_params--;
      if (objectdata->num_params == 0) {
         free(objectdata->params);
	 objectdata->params = NULL;
      }

      /* Step 3: Search and destroy on all instances */
      /* This is obnoxious and complicated, but it has to be done. . . */

      for (k = 0; k < xobjs.pages; k++) {
         searchinst(xobjs.pagelist[k]->pageobj, objectdata, i);
      }
      for (j = 0; j < xobjs.numlibs; j++) {
         for (k = 0; k < xobjs.userlibs[j].number; k++) {
            searchinst(*(xobjs.userlibs[j].library + k), objectdata, i);
         }
      }
      for (k = 0; k < xobjs.delbuffer.number; k++) {
         searchinst(*(xobjs.delbuffer.library + k), objectdata, i);
      }
   }
}

/*------------------------------------------------------*/
/* Wrapper for unmakeparam()				*/
/*------------------------------------------------------*/

void unparameterize(void *mode)
{
   short *fselect, i, ts, te, ptype = (short)((int)mode);
   labelptr settext;

   if (!checkselect(ptype)) objectselect(ptype);
   if (!checkselect(ptype)) return;

   if ((areastruct.selects == 1) && (ptype & LABEL) && textend > 0
		&& textend < textpos) {
      settext = SELTOLABEL(areastruct.selectlist);
      for (i = textend; i >= 0; i--)
	 if (*(settext->string + i) == TEXT_ESC)
	    if (*(settext->string + i + 1) == PARAM_START)
	       unmakeparam(settext, i);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if ((ptype & LABEL) && SELECTTYPE(fselect) == LABEL) {
            settext = SELTOLABEL(fselect);
     	    for (i = 0; i < strlen((char *)settext->string); i++)
	       if (*(settext->string + i) == TEXT_ESC)
	          if (*(settext->string + i + 1) == PARAM_START)
	             unmakeparam(settext, i);
	 }
	 else
	    unmakeshortp(objectdata->plist + (*fselect));
      }
   }
}

/*--------------------------------------------------------------*/
/* Wrapper for makeparam()					*/
/*--------------------------------------------------------------*/

void parameterize(void *mode)
{
   short *fselect, ptype = (short)((int)mode);
   labelptr settext;

   if (!checkselect(ptype)) objectselect(ptype);
   if (!checkselect(ptype)) return;
   for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
      if ((ptype & LABEL) && areastruct.selects == 1 &&
		SELECTTYPE(fselect) == LABEL) {
         settext = SELTOLABEL(fselect);
         makeparam(settext);
      }
      else
	 makeshortp(objectdata->plist + (*fselect));
   }
   objectdeselect();
}

/*----------------------------------------------------------------------*/
/* Find the object to which a parameter points				*/
/*----------------------------------------------------------------------*/

genericptr findparam(objectptr locobj, void *posptr, uchar ptype)
{
   genericptr *pgen;
   short k;

   for (k = 0; k < locobj->parts; k++) {
      pgen = locobj->plist + k;
      if ((ptype == XC_STRING) && ((*pgen)->type == LABEL)) {
	 labelptr plab = TOLABEL(pgen);
	 uchar *tmpstr = plab->string;
	 int plen = strlen((char *)plab->string);
	 if ((uchar *)posptr >= plab->string && (uchar *)posptr
		< plab->string + strlen((char *)plab->string))
	    break;
      }
      else if (ptype == XC_SHORT) {	/* to be completed */
	 if ((*pgen)->type == LABEL) {
	    labelptr plab = TOLABEL(pgen);
	 }
      }
   }

   if (k < locobj->parts) return *pgen;
   else return NULL;
}

/*----------------------------------------------------------------------*/
/* Find and return a pointer to the parameter (if any) pointing to the  */
/* specified label at given text position (or "textpos" if NULL).	*/
/*----------------------------------------------------------------------*/

oparamptr parampos(objectptr tobj, labelptr tlab, char *tptr, short *jret,
	short *iret)
{
   oparamptr ops;
   char *pptr, *sptr = (tptr != NULL) ? tptr : (char *)tlab->string + textpos;
   short i, j, plen;

   for (i = 0; i < tobj->num_params; i++) {
      ops = *(tobj->params + i);
      if (ops->type == XC_STRING) {
         plen = strlen((char *)ops->pdefault) + 4;
         for (j = 0; j < ops->number; j++) {
	    pptr = (char *)(*((char **)ops->position + j));
	    if (sptr >= pptr && sptr < (pptr + plen)) {
	       if (jret != NULL) *jret = j;
	       if (iret != NULL) *iret = i;
	       return ops;
	    }
	 }
      }
   }   
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Same routine as above, but looks for a parameter overlapping the	*/
/* textend <--> textpos space.						*/
/*----------------------------------------------------------------------*/

oparamptr paramcross(objectptr tobj, labelptr tlab)
{
   oparamptr ops;
   char *pptr, *qptr, *tptr = (char *)tlab->string + textpos;
   char *sptr = (char *)tlab->string + textend; 
   short i, j, plen;

   if (sptr == tptr)
      sptr += strlen((char *)tlab->string);

   for (i = 0; i < tobj->num_params; i++) {
      ops = *(tobj->params + i);
      if (ops->type == XC_STRING) {
         plen = strlen((char *)ops->pdefault) + 4;
         for (j = 0; j < ops->number; j++) {
	    pptr = (char *)(*((char **)ops->position + j));
	    if ((pptr <= sptr) && ((pptr + plen) >= tptr))
	       return ops;
	 }
      }
   }   
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Parameter substitution and default value restoration			*/
/*----------------------------------------------------------------------*/

void psubstitute(objinstptr localdata)
{
   uchar nops = localdata->thisobject->num_params;
   oparamptr ops, sops;
   int i, j, k;
   genericptr pgen;
   Boolean do_update = False;

   if (localdata->thisobject->params == NULL) return;	/* object has no parameters */

   for (i = 0; i < nops; i++) {
      ops = *(localdata->thisobject->params + i);
      for (j = 0; j < ops->number; j++) {
         if (ops->type != XC_STRING) {
	    short *parmptr;
	    if (localdata->num_params <= i)
	       parmptr = ops->pdefault;
	    else if (*(localdata->params + i) == NULL)
	       parmptr = ops->pdefault;
	    else
	       parmptr = (short *)(*(localdata->params + i));
	    *((short *)(ops->position[j])) = *parmptr;
	    /* printf("Substituting parameter value %hd\n", *parmptr); */
	 }
	 else {
	    /* This is a little tricky---if size(substitution) != size(target), */
	    /* have to either pad string with NO_OP characters or widen string   */
	    /* to fit the substitution and pad the target with NO_OP characters.*/
	    short slen;
	    short tlen;
	    char *parmptr;

	    if (localdata->num_params <= i) parmptr = (char *)ops->pdefault;
	    else if (*(localdata->params + i) == NULL) parmptr = (char *)ops->pdefault;
	    else parmptr = (char *)(*(localdata->params + i));

 	    slen = strlen(parmptr);
	    tlen = paramlen((uchar *)(*(ops->position + j)));
	    if (slen < tlen) {
	       for (k = slen; k < tlen; k++)
		  *((char *)(*(ops->position + j)) + k + 2) = NO_OP;
	    }	  
	    else if (slen > tlen) {
	       labelptr plab;
	       uchar *tmpstr;
	       int l, plen;

	       pgen = findparam(localdata->thisobject, *(ops->position + j), ops->type);
	       if (pgen != NULL) plab = TOLABEL(&pgen);
	       else {
		  printf("Error:  parameter hooks to no label string!\n");
		  return;
	       }
	       tmpstr = plab->string;
	       plen = strlen((char *)tmpstr);
	       plab->string = (uchar *)realloc(plab->string, plen + slen - tlen + 1);
	       *(plab->string + plen + slen - tlen) = '\0';

	       /* Re-hook, both for new alloc'd position and for character shift */
	       for (k = 0; k < nops; k++) {
      		  sops = *(localdata->thisobject->params + k);
      		  for (l = 0; l < sops->number; l++) {
		     if (l == j && k == i) continue;
         	     if (sops->type == XC_STRING) {
			if ((uchar *)sops->position[l] > tmpstr
				   && sops->position[l] < ops->position[j])
	    		   sops->position[l] += (int)(plab->string - tmpstr);
			else if ((uchar *)sops->position[l] > tmpstr
				   && (int)sops->position[l] < (int)tmpstr + plen)
	    		   sops->position[l] += (int)(plab->string - tmpstr
					+ slen - tlen);
	             }
	          }
	       }
	       ops->position[j] += (int)(plab->string - tmpstr);

	       /* use memmove() to move overlapping string segments */
	       /* start after the TEXT_ESC START_PARAM (thus the + 2) */
	       memmove((char *)ops->position[j] + slen - tlen + 2,
			(char *)ops->position[j] + 2,
			strlen((char *)ops->position[j] + 2));

	       /* Recompute the bounding box on this object if the label */
	       /* has been expanded.			       		 */

	       do_update = True;
	    }

	    /* use memmove() here to avoid copying end-of-string NULL to destination */
	    memmove((char *)(*(ops->position + j)) + 2, parmptr, slen);

	 }
      }
   }
   if (do_update)
      calcbboxvalues(localdata->thisobject, (uchar)0, &pgen);
}

/*----------------------------------------------------------------------*/
/* Get the length of a string between the parameter endpoint markers	*/
/*----------------------------------------------------------------------*/

short paramlen(uchar *tstr)
{
   uchar *pstr = tstr;
   if (*pstr != TEXT_ESC && *(pstr + 1) != PARAM_START) return -1;
   pstr += 2;
   while ((!(*pstr == PARAM_END && *(pstr - 1) == TEXT_ESC))
		&& *pstr != '\0')
      pstr++;
   return ((short)(pstr - tstr - 3));
}

/*-----------------------------------------------------------------------*/
/* "Natural" string length---strlen() without counting NO_OP characters. */
/*-----------------------------------------------------------------------*/

int natstrlen(uchar *tstr)
{
   int slen = 0;
   uchar *pstr = tstr;
   while (*pstr != '\0') {
      if (*pstr == TEXT_ESC && *(pstr + 1) == PARAM_END) break;
      if (*pstr++ != NO_OP) slen++;
   }
   return slen;
}

/*--------------------------------------------------------------------*/
/* "Natural" string compare---strcmp() disregarding NO_OP characters. */
/* (except only returns 0 if matching, -1 if not matching)	      */
/*--------------------------------------------------------------------*/

int natstrcmp(uchar *sstr, uchar *tstr)
{
   uchar *pstr = sstr;
   uchar *qstr = tstr;

   for(;;) {
      while (*pstr == NO_OP) pstr++;
      while (*qstr == NO_OP) qstr++;
      if (*pstr != *qstr++) return -1;
      if (*pstr++ == '\0') return 0;
   }
}

/*----------------------------------------------------------------------*/
/*  Remove trailing NO_OPs from a string 				*/ 
/*----------------------------------------------------------------------*/

curtail(uchar *tstr)
{
   uchar *pstr = tstr;
   while (*pstr++ != '\0');
   if (pstr > tstr) {
      if (*(--pstr) == NO_OP) {
	 if (pstr > tstr) {
	    while (*(--pstr) == NO_OP);
	    pstr++;
	 }
      }
      *pstr = '\0';
   }
}

/*----------------------------------------------------------------------*/
/* Check whether this page object was entered via a library page	*/
/*----------------------------------------------------------------------*/

objectptr checklibtop()
{
   int k;
   objectptr pobj;

   for (k = pushes - 1; k >= 0; k--) {
      pobj = (*(pushlist + k)).thisobject;
      if (is_library(pobj) >= 0) return pobj;
   }
   return (objectptr)NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object	instance			*/
/* (Reverts all parameters to default value)				*/
/*----------------------------------------------------------------------*/

removeinst(objinstptr thisinst)
{
   free(thisinst->params);
   thisinst->params = NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object.				*/
/*----------------------------------------------------------------------*/

removeparams(objectptr thisobj)
{
   int i, j;
}

/*-----------------------------------------------------------------------*/
/* Determine how the old label relates to the new label, and re-hook the */
/* parameters accordingly.						 */
/* Return values: -1 -> no parameters to resolve			 */
/*		   0 -> parameter is a new local (instance) substitution */
/*		  +1 -> parameter is a new global (default) value	 */
/*-----------------------------------------------------------------------*/

int resolveparams(objectptr localdata, uchar *origstr, labelptr curlabel)
{
   oparamptr ops;
   uchar *sptr, *pptr, *ssave;
   short offset, nval, ival, nlen, dlen, k;
   int is_default = 0;

   if (objectdata->params == NULL) return -1;  /* Any parameters at all? */

   /* Remove trailing NO_OPs from strings */
   curtail(origstr);
   curtail(curlabel->string);

   /*-------------------------------------------------------------------*/
   /* 1) Expand string with NO_OPs as necessary to hold default strings */
   /* 2) See if any part of this string is a parameter of the object	*/
   /* 3) Re-link parameters.						*/
   /* 4) Write back new parameter values 				*/
   /*-------------------------------------------------------------------*/

   /* subroutine labeltext() prevents PARAM boundary markers (PARAM_START */
   /* and PARAM_END) from being deleted, so there will be a 1:1 match	  */
   /* between the markers in the old and new strings.  Use these markers  */
   /* to determine new instance values or changed default values.	  */

   /* Pass 1: make sure that new parameters are at least as long as the	  */
   /* original (= length of default) values.  Pad if shorter.		  */

   sptr = curlabel->string;
   pptr = origstr;
   while (*(sptr + 1) != '\0') {
      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_START) && *(sptr + 1) != '\0')
         sptr++;
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_START) && *(pptr + 1) != '\0')
         pptr++;
      offset = 0;
      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_END) && *(sptr + 1) != '\0') {
	 offset--;
         sptr++;
      }
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_END) && *(pptr + 1) != '\0') {
	 offset++;
         pptr++;
      }
      if (offset > 0) {
	 short tmppos = (short)(sptr - curlabel->string);
   	 curlabel->string = (uchar *)realloc(curlabel->string,
		strlen((char *)curlabel->string) + offset + 1);
	 memmove(curlabel->string + tmppos + offset, curlabel->string + tmppos,
		strlen((char *)curlabel->string + tmppos) + 1); 
	
	 for (k = 0; k < offset; k++)
	    *(curlabel->string + tmppos + k) = NO_OP;

	 sptr = curlabel->string + tmppos + offset + 2;
      }
   }

   /* Check if this parameter change affects the default value or if it */
   /* is a local instance substitution (depends on whether this object	*/
   /* was pushed into from the library or from a page).			*/

   if ((checklibtop() != NULL) || (pushes == 0)) {
      /* printf("Came from library or top page:  changing default value\n"); */
      is_default = True;
   }

   /* Pass 2: now that no more reallocation will occur, rehook the pointers */

   sptr = curlabel->string;
   pptr = origstr;
   while (*(sptr + 1) != '\0') {

      /* find the beginning and end of both original and new parameter substrings */

      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_START) && *(sptr + 1) != '\0')
         sptr++;
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_START) && *(pptr + 1) != '\0')
         pptr++;
      if (*(sptr + 1) == '\0') break;
      offset = 0;
      ssave = sptr;

      /* ops == NULL indicates that a parameter was inserted, and does not exist    */
      /* in the original string.  Therefore all hooks are new and none need fixing. */
      if ((ops = parampos(objectdata, NULL, pptr, &nval, &ival)) == NULL)
	 return is_default;

      while ((*sptr != TEXT_ESC || *(sptr + 1) != PARAM_END) && *(sptr + 1) != '\0') {
	 offset--;
         sptr++;
      }
      while ((*pptr != TEXT_ESC || *(pptr + 1) != PARAM_END) && *(pptr + 1) != '\0') {
	 offset++;
         pptr++;
      }

      *((char **)ops->position + nval) = ssave;		/* rehook */

      /* Get the length of the parameter (less the TEXT_ESC pair) */
      dlen = sptr - ssave - 2;

      /* If one of the libraries is in the hierarchy stack, */
      /* then we change the default.  Always, change the    */
      /* parameter list in the calling object instance	     */
      /* (top of hierarchy stack).			     */

      if (is_default) {
         ops->pdefault = (short *)realloc(ops->pdefault, dlen + 1);
         strncpy((char *)ops->pdefault, *((char **)ops->position + nval) + 2, dlen);
         *((char *)ops->pdefault + dlen) = '\0';
      }
      else {
         objinstptr pinst = (*(pushlist + pushes - 1)).thisinst;
         void **pparm;
         short mlen;

	 /* printf("Came from page:  changing instance value\n"); */

	 /* If the new string == the default, then use the default */
	 mlen = natstrlen((char *)ops->pdefault);
	 if ((mlen == dlen) && !strncmp((char *)ops->pdefault,
			*(ops->position + nval), mlen)) {
	    /* printf("Reverting to default value\n"); */
	    if (pinst->num_params > ival) {
	       pparm = pinst->params + ival;
	       if (*pparm != NULL) {
		  free(*pparm);
		  *pparm = NULL;
	       }
	    }
	 }
	 else {
	    if (pinst->params == NULL) {
	       pinst->params = (void **)malloc(objectdata->num_params
			* sizeof(void *));
	       for (k = 0; k < objectdata->num_params; k++)
			*(pinst->params + k) = NULL;
	       pinst->num_params = objectdata->num_params;
	    }
	    else if (ival >= pinst->num_params) {
	       pinst->params = (void **)realloc(pinst->params,
			(ival + 1) * sizeof(void *));
	       for (k = pinst->num_params; k <= ival; k++)
			*(pinst->params + k) = NULL;
	       pinst->num_params = ival + 1;
	    }
	    pparm = pinst->params + ival;
	    if (*pparm == NULL)
	       *pparm = (void *)malloc((dlen + 1) * sizeof(uchar));
	    else
	       *pparm = (void *)realloc(*pparm, (dlen + 1) * sizeof(uchar));
	    strncpy((char *)(*pparm), (char *)(*(ops->position + nval)) + 2, dlen);
	    *((char *)(*pparm) + dlen) = '\0';

	    /* pad the default with NO_OP if shorter than the new string */

	    if (dlen > (nlen = strlen((char *)ops->pdefault))) {
	       ops->pdefault = (short *)realloc(ops->pdefault, dlen + 1);
	       for (k = nlen; k < dlen; k++) 
	          *((char *)ops->pdefault + k) = NO_OP;
	       *((char *)ops->pdefault + k) = '\0';
	    }
         }
      }
   }
   return is_default;
}

/*-----------------------------------------------------------------------*/
