/*-------------------------------------------------------------------------*/
/* schema.c --- xcircuit routines specific to the schematic capture system */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/13/97    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"
#include "Xw/MenuBtn.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"
#include "menudep.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Globaldata xobjs;
extern Clientdata areastruct;
extern short	  eventmode;
extern short	  pushes;
extern int	  *appcolors;
extern Widget     menuwidgets[];
extern Widget	  netbutton;

XtCallbackProc	  schembutton();
objectptr	  recurselect();
short		  *selectobject();

/*-------------------------------------------------------------------------*/
#ifdef SCHEMA

extern Widget wsymb, wschema;

/*--------------------------------------------------------*/
/* Menu calls (procedure wrappers)			  */
/*--------------------------------------------------------*/

XtCallbackProc callgennet(Widget w, unsigned int mode, caddr_t calldata)
{
   switch(mode) {
     case 0:
        gennet("spice", "spc");
	break;
     case 1:
        gennet("sim", "sim");
	break;
     case 2:
        gennet("pcb", "pcb");
	break;
   }
}

/*------------------------------------------------------------*/
/* Enable schematic capture (set schemon and realize widgets) */
/*------------------------------------------------------------*/

XtCallbackProc doxschema(Widget w, caddr_t clientdata, caddr_t calldata)
{
   short i;
   Arg wargs[1];

   if (areastruct.schemon) {
      areastruct.schemon = False;
      XtUnmanageChild(wschema);
      XtUnmanageChild(wsymb);
      XtUnmanageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Enable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }
   else {
      areastruct.schemon = True;
      XtManageChild(wschema);
      XtManageChild(wsymb);
      XtManageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Disable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }

   /* Redraw the screen, since this action may effect the presence of	*/
   /* pins and schematic informational labels.				*/

   drawarea(NULL, objectdata, NULL);
}

/*----------------------------------*/
/* Find connectivity of a selection */
/*----------------------------------*/

void startconnect(button, clientdata, calldata)
  Widget       button;
  caddr_t      clientdata, calldata;
{
   if (eventmode == NORMAL_MODE) {
      Wprintf("Click button1 on wire to see connectivity.");
      eventmode = CONNECT_MODE;
   }
}

/*------------------------------------------------------------------------*/

XtCallbackProc connectivity(button, clientdata, calldata)
  Widget       button;
  caddr_t      clientdata, calldata;
{
   polyptr gpoly;
   int netid;

   /* Will have to refine recursive selection at a later date. . . */
   /* objectptr gobj;				*/
   /* gobj = recurselect(&gpoly, objectdata);	*/

   objectselect(POLYGON);
   if (areastruct.selects != 1) {
      Wprintf("Must select one polygon to find network connections.");
   }
   else {
      gpoly = SELTOPOLY(areastruct.selectlist);
      objectdeselect();
      createnets(True);
      netid = resolved(objectdata->netlist, gpoly);
      /* printf(">> Resolving connectivity for net %d\n", netid);  */
      if (netid != 0)
         highlightnet(objectdata, netid);
      else
         Wprintf("No network is associated with this polygon");
      destroynets();
   }
}

/*-----------------------------------------------------------------------*/
/* Set object type to FUNDAMENTAL if it contains one or more info labels */
/*-----------------------------------------------------------------------*/

void setobjecttype(objectptr cschem)
{
   genericptr *cgen;
   labelptr clab;

   if (cschem->schemtype == SCHEMATIC
	|| cschem->symschem != NULL) return;  /* does not apply to these types */

   if (cschem->schemtype == FUNDAMENTAL) cschem->schemtype = SYMBOL;
   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == LABEL) {
         clab = TOLABEL(cgen);
	 if (clab->pin == INFO) {
	    cschem->schemtype = FUNDAMENTAL;
	    break;
	 }
      }
   }
}

/*------------------------------------------------------*/
/* Pin conversion subroutine for dopintype()		*/
/*------------------------------------------------------*/

void pinconvert(labelptr thislab, unsigned int mode)
{
   thislab->pin = mode;
   switch (mode) {
      case NORMAL:
	 thislab->color = DEFAULTCOLOR;		/* nominally black */
	 break;
      case GLOBAL:
	 thislab->color = BBOXCOLOR;		/* orange */
         break;
      case LOCAL:
	 thislab->color = SNAPCOLOR;		/* red */
         break;
      case INFO:
	 thislab->color = AUXCOLOR;		/* green */
         break;
   }
}

/*---------------------------------------------------------*/
/* Change a label's type to NORMAL, GLOBAL, INFO, or LOCAL */
/*---------------------------------------------------------*/

XtCallbackProc dopintype(Widget w, unsigned int mode, caddr_t calldata)
{
   short *gsel;
   char typestr[40];
   short savetype = -1;

   if (areastruct.selects == 0) {
      Wprintf("Must first select a label to change type");
      return;
   }

   strcpy(typestr, "Changed label to ");
   switch(mode) {
      case NORMAL:
	 strcat(typestr, "normal label");
         break;
      case GLOBAL:
	 strcat(typestr, "global pin");
         break;
      case LOCAL:
	 strcat(typestr, "local pin");
         break;
      case INFO:
	 strcat(typestr, "info-label");
         break;
   }

   for (gsel = areastruct.selectlist; gsel < areastruct.selectlist +
		areastruct.selects; gsel++)
      if (SELECTTYPE(gsel) == LABEL) {
	 labelptr glab = SELTOLABEL(gsel);
	 savetype = glab->pin;
	 pinconvert(glab, mode);
	 setobjecttype(objectdata);
      }

   if (savetype >= 0) {
      objectdeselect();
      drawarea(NULL, objectdata, NULL);
      Wprintf(typestr);
   }
   else {
      Wprintf("No labels selected.");
   }
}

/*----------------------------------------------------------*/
/* Set colors on the symbol/schematic buttons appropriately */
/*----------------------------------------------------------*/

void setsymschem()
{
   Arg aargs[2], bargs[2];

   /* Set menu items appropriately for this object */

   if (objectdata->symschem != NULL) {
      if (objectdata->schemtype == SCHEMATIC) {
         XtSetArg(aargs[0], XtNlabel, "Go To Symbol");
         XtSetArg(bargs[0], XtNlabel, "Disassociate Symbol");
      }
      else {
         XtSetArg(aargs[0], XtNlabel, "Go To Schematic");
         XtSetArg(bargs[0], XtNlabel, "Disassociate Schematic");
      }
   }
   else {
      if (objectdata->schemtype == SCHEMATIC) {
         XtSetArg(aargs[0], XtNlabel, "Make Matching Symbol");
         XtSetArg(bargs[0], XtNlabel, "Associate with Symbol");
      }
      else {
         XtSetArg(aargs[0], XtNlabel, "Make Matching Schematic");
         XtSetArg(bargs[0], XtNlabel, "Associate with Schematic");
      }
   }
   XtSetValues(NetlistMakeMatchingSymbolButton, aargs, 1);
   XtSetValues(NetlistAssociatewithSymbolButton, bargs, 1);

   /* Set colors on the symbol and schematic buttons */

   if (objectdata->schemtype == SCHEMATIC) {
      if (objectdata->symschem == NULL) {
         XtSetArg(aargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(aargs[1], XtNforeground, OFFBUTTONCOLOR);
      }
      else {
         XtSetArg(aargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(aargs[1], XtNforeground, FOREGROUND);
      }

      XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      XtSetArg(bargs[0], XtNbackground, SNAPCOLOR);
   }
   else {
      if (objectdata->symschem != NULL) {
         XtSetArg(bargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      }
      else {
         XtSetArg(bargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(bargs[1], XtNforeground, OFFBUTTONCOLOR);
      }

      XtSetArg(aargs[1], XtNforeground, FOREGROUND);
      if (objectdata->schemtype == FUNDAMENTAL)
         XtSetArg(aargs[0], XtNbackground, AUXCOLOR);
      else if (objectdata->schemtype == TRIVIAL || objectdata->symschem != NULL)
         XtSetArg(aargs[0], XtNbackground, SNAPCOLOR);
      else {
         XtSetArg(aargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(aargs[1], XtNforeground, OFFBUTTONCOLOR);
         XtSetArg(bargs[0], XtNbackground, BBOXCOLOR);
         XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      }
   }

   XtSetValues(wsymb, aargs, 2);
   XtSetValues(wschema, bargs, 2);
}

/*--------------------------------------------------------*/
/* Find the page number for an object			  */
/*--------------------------------------------------------*/

int findpageobj(pobj)
  objectptr pobj;
{
   int tpage;
   objectptr chkpage;

   for (tpage = 0; tpage < xobjs.pages; tpage++) {
      if (xobjs.pagelist[tpage]->pageobj == pobj) return tpage;
   }
   return -1;
}

/*--------------------------------------------------------*/
/* Recursively find all sub-circuits associated with the  */
/* top-level circuit and set their filenames to be the	  */
/* same.						  */
/*--------------------------------------------------------*/

findsubschems(toppage, cschem)
  int toppage;
  objectptr cschem;
{
   genericptr *cgen;

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {
	 objectptr cobj = TOOBJINST(cgen)->thisobject;

	 if (cobj->symschem != NULL) {
	    int pageno = findpageobj(cobj->symschem);
	    /* printf("Symbol %s has schematic at page %d\n", cobj->name, pageno); */
	    if (pageno < 0) {	/* This shouldn't happen (hopefully) */
	       Wprintf("Error: disconnected schematic!\n");
	    }
	    else {
	       xobjs.pagelist[pageno]->filename = xobjs.pagelist[toppage]->filename;
	    }

	    /* A symbol on its own schematic page is allowed for clarity */
	    /* of the schematic, but this cannot be a functional part of */
	    /* the schematic circuit!					 */
	    
	    if (cobj->symschem != cschem) {
	       /* printf("Searching for dependencies of symbol %s\n", cobj->name); */
	       findsubschems(toppage, cobj->symschem);
	    }
/*
	    else {
	       printf("Notice:  Symbol on its own schematic\n");
	    }
*/
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Copy label to corresponding schematic/symbol		  */
/*--------------------------------------------------------*/

copypinlabel(pinlab)
  labelptr pinlab;
{
   labelptr *newlabel, tlabel;
   genericptr *tgen;
   objectptr schemobj = objectdata->symschem;

   if (areastruct.schemon && schemobj != NULL && (pinlab->pin == LOCAL)) {

      /* If this pin already exists, don't copy it */

      for (tgen = schemobj->plist; tgen < schemobj->plist + schemobj->parts;
		tgen++) {
         if ((*tgen)->type == LABEL) {
	    tlabel = TOLABEL(tgen);
	    if (!strcmp(tlabel->string, pinlab->string)) return;
         }
      }

      NEW_LABEL(newlabel, schemobj);
      (*newlabel)->pin = pinlab->pin;
      (*newlabel)->rotation = pinlab->rotation;
      (*newlabel)->justify = pinlab->justify;
      (*newlabel)->color = pinlab->color;
      (*newlabel)->scale = pinlab->scale;
      (*newlabel)->string = (char *)malloc((1 + strlen(pinlab->string))
		* sizeof(char));
      noparmstrcpy((*newlabel)->string, pinlab->string);

      /* place label just outside bounding box, then recompute bbox */

      (*newlabel)->position.x = schemobj->lowerleft.x + (schemobj->width >> 1);
      (*newlabel)->position.y = schemobj->lowerleft.y - TEXTHEIGHT
	   * (*newlabel)->scale;
      schemobj->parts++;
      singlebbox(objectdata, ENDPART);
   }
}

/*-------------------------------------------------------*/
/* Check if top-level page is the same name as a library */
/* object; if so, connect it.				 */
/* Note that it is not an error not to find the matching */
/* symbol/schematic.  "is_schematic" and "is_symbol"	 */
/* comments in the .ps file are complementary, so the	 */
/* first one encountered will always fail, and the other */
/* will succeed.					 */
/*-------------------------------------------------------*/

int checkschem(objectptr thisobj, char *cname)
{
   objectptr *tlib;
   short i, j;

   if (areastruct.schemon == False || thisobj->symschem != NULL) return 0;

   for (i = 0; i < xobjs.numlibs; i++) {
      for (j = 0; j < xobjs.userlibs[i].number; j++) {
	 tlib = xobjs.userlibs[i].library + j;

         if (!strcmp(cname, (*tlib)->name)) {
	    thisobj->symschem = (*tlib);
	    thisobj->schemtype = SCHEMATIC;
	    (*tlib)->symschem = thisobj;
	    (*tlib)->schemtype = SYMBOL;
	    return 1;
	 }
      }
   }
   return 0;
}

/*--------------------------------------------------------*/
/* Complement to the above routine:  If a library object  */
/* name is the same as a top-level page, connect them.	  */
/*--------------------------------------------------------*/

int checksym(objectptr symobj, char *cname)
{
   short cpage;
   objectptr checkpage;

   if (areastruct.schemon == False || symobj->symschem != NULL) return 0;

   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = xobjs.pagelist[cpage]->pageobj;
      if (checkpage != NULL) {
         if (!strcmp(checkpage->name, cname)) {
	    symobj->symschem = checkpage;
	    symobj->schemtype = SYMBOL;
	    checkpage->symschem = symobj;
	    checkpage->schemtype = SCHEMATIC;
	    return 1;
	 }
      }
   }
   return 0;
}

/*--------------------------------------------------------*/
/* Find location of corresponding pin in symbol/schematic */
/*--------------------------------------------------------*/

labelptr findotherpin(curlabel)
  labelptr curlabel;
{
   objectptr other = objectdata->symschem;
   genericptr *tgen;
   labelptr tlab;

   if (areastruct.schemon == False || other == NULL || curlabel->pin == False)
      return NULL;

   for (tgen = other->plist; tgen < other->plist + other->parts; tgen++) {
      if ((*tgen)->type == LABEL) {
	 tlab = TOLABEL(tgen);
	 if (!strcmp(tlab->string, curlabel->string)) return tlab;
      }
   }
   return NULL;
}

/*----------------------------------------------------------------------*/
/* Swap object schematic and symbol pages.   				*/
/*  mode = 0 disallows creation of a new schematic or symbol; i.e., if	*/
/*  there is no corresponding schematic/symbol, nothing happens.	*/
/*----------------------------------------------------------------------*/

XtCallbackProc swapschem(Widget w, caddr_t mode, caddr_t calldata)
{
   objectptr savepage = objectdata;
   labelptr  *pinlab;
   genericptr *plab;
   Boolean lflag;

   if (areastruct.schemon == False) return;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);

   /* Create symbol or schematic, if allowed by mode */

   if ((objectdata->symschem == NULL) && (mode != (caddr_t)0)) {
      if (objectdata->schemtype == SYMBOL) {
	 int tpage;

	 /* create a new page for the new schematic */

	 for (tpage = 0; tpage < xobjs.pages; tpage++)
	    if (xobjs.pagelist[tpage]->pageobj == NULL) break;
         changepage(tpage);
      }
      else {
	 objectptr *newobject;
	 short libnum = USERLIB - LIBRARY;

	 /* create a new library object for the new symbol */

	 xobjs.userlibs[libnum].library = (objectptr *)
		realloc(xobjs.userlibs[libnum].library,
		++xobjs.userlibs[libnum].number * sizeof(objectptr));
	 newobject = xobjs.userlibs[libnum].library
		+ xobjs.userlibs[libnum].number - 1;
         *newobject = (objectptr) malloc(sizeof(object));
	 initmem(*newobject);
	 (*newobject)->schemtype = SYMBOL;
	 (*newobject)->hidden = False;
	 objectdata = *newobject;

	 /* Generate the default bounding box for a size-zero object */
	 calcbbox(*newobject);
      }

      /* set links between the two objects */

      savepage->symschem = objectdata;
      objectdata->symschem = savepage;

      /* make the name of the new object equal to that of the old */

      strcpy(objectdata->name, savepage->name);

      /* copy all pin labels into the new object */

      for (plab = savepage->plist; plab < savepage->plist + savepage->parts;
		plab++) {
	 if ((*plab)->type == LABEL) {
	    genericptr *tgen;
	    labelptr tlab, lpin = (labelptr)*plab;

	    if (lpin->pin == LOCAL) {

      	       /* Only make one copy of each pin name */

	       lflag = False;
               for (tgen = objectdata->plist; tgen <
			objectdata->plist + objectdata->parts; tgen++) {
		  if ((*tgen)->type == LABEL) {
		     tlab = TOLABEL(tgen);
	             if (!strcmp(tlab->string, lpin->string)) lflag = True;
		  }
      	       }
	       if (lflag == True) continue;

	       NEW_LABEL(pinlab, objectdata);
	       (*pinlab)->pin = lpin->pin;
	       (*pinlab)->color = lpin->color;
	       (*pinlab)->rotation = 1;
	       (*pinlab)->scale = 1.0;
	       (*pinlab)->justify = areastruct.justify; 
	       (*pinlab)->position.x = 0;
	       (*pinlab)->position.y = objectdata->parts * (TEXTHEIGHT + 10);
	       u2u_snap(&((*pinlab)->position));
	       (*pinlab)->string = (char *)malloc((1 + strlen(lpin->string))
			* sizeof(char));
	       strcpy((*pinlab)->string, lpin->string);
	       objectdata->parts++;
	    }
         }
	 calcbbox(objectdata);
      }

      /* Recreate the user library with the new symbol */
      if (savepage->schemtype != SYMBOL) composelib(USERLIB);
   }
   else if (objectdata->symschem != NULL) {
      objectdata = objectdata->symschem;
   }

   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;

   setpage();
   transferselects();
   refresh(NULL, NULL, NULL);

   setsymschem();
}

/*------------------------------------------*/
/* Disassociate a symbol from its schematic */
/*------------------------------------------*/

void schemdisassoc()
{
   if (eventmode != NORMAL) {
      Wprintf("Cannot disassociate schematics in this mode");
   }
   else {
      objectdata->symschem->symschem = NULL;
      objectdata->symschem = NULL;
      setsymschem();
      Wprintf("Schematic and symbol are now unlinked.");
   }
}

/*--------------------------------------------------------------*/
/* Schematic<-->symbol association.  Determine action from 	*/
/* context (associate/disassociate, symbol or schematic)	*/
/*--------------------------------------------------------------*/

XtCallbackProc startschemassoc(Widget w, caddr_t clientdata, caddr_t calldata)
{
   if (objectdata->symschem != NULL)
      schemdisassoc();
   else {
      eventmode = ASSOC_MODE;
      if (objectdata->schemtype == SCHEMATIC) {
	 /* Find a symbol to associate */
	 startcatalog(w, Number(LIBLIB), NULL);
	 Wprintf("Click on library page, then symbol to associate.");
      }
      else {
	 /* Find a schematic (page) to associate */
	 startcatalog(w, Number(PAGELIB), NULL);
	 Wprintf("Click on schematic page to associate.");
      }
   }
}

/*--------------------------------------------------------------*/
/* Callback procedures on the schematic/symbol buttons.		*/
/*--------------------------------------------------------------*/

void schemassoc(objectptr schemobj, objectptr symbolobj)
{
   if (eventmode != ASSOC_MODE) {
      Wprintf("Cannot make (dis)association in this mode");
   }
   else if (schemobj->symschem != NULL || symbolobj->symschem != NULL) {
      Wprintf("Both objects must be disassociated first.");
   }
   else {
      schemobj->symschem = symbolobj;
      symbolobj->symschem = schemobj;
      if (symbolobj->schemtype == TRIVIAL)
	 symbolobj->schemtype = SYMBOL;

      /* Schematic takes the name of its associated symbol, by default */
      strcpy(schemobj->name, symbolobj->name);
   
      setsymschem();	/* Set buttons and menu items appropriately */
   }
}

#endif
/*-------------------------------------------------------------------------*/
