/*-----------------------------------------------------------------------*/
/* text.c --- text processing routines for xcircuit		 	 */
/* Copyright (c) 2000  Tim Edwards, Johns Hopkins University        	 */
/*-----------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Display *dpy;
extern Clientdata areastruct;
extern Globaldata xobjs;
extern short textpos, textend;
extern short fontcount;
extern fontinfo *fonts;
extern short eventmode;
extern int *appcolors;
extern colorindex *colorlist;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c            */
/*------------------------------------------------------------------------*/

TextExtents ULength();
extern float UTopScale();

/*---------------------------------------------------------------------*/
/* draw a single character at 0, 0 using current transformation matrix */
/*---------------------------------------------------------------------*/

short UDrawChar(localdata, code, styles, ffont, groupheight, passcolor)
  objinstptr 	localdata;
  uchar		code;
  short 	styles, ffont;
  int		groupheight;
  int		passcolor;
{
   objectptr drawchar;
   XPoint alphapts[2], *curpoint;
   short  totalpts, i;
   short  localwidth;
   XPoint *pointptr;
   objinst charinst;  /* to be replaced? */

   alphapts[0].x = 0;
   alphapts[0].y = 0;
   charinst.type = OBJECT;
   charinst.color = DEFAULTCOLOR;
   charinst.rotation = 0;
   charinst.scale = fonts[ffont].scale;
   charinst.position = alphapts[0];
   
   /* get proper font and character */

   drawchar = fonts[ffont].encoding[(uchar)code];
   charinst.thisobject = drawchar;

   localwidth = (drawchar->lowerleft.x + drawchar->width) * fonts[ffont].scale;

   if ((fonts[ffont].flags & 0x22) == 0x22) { /* font is derived and italic */
      USlantCTM(DCTM, 0.25);  		/* premultiply by slanting function */
   }

   if (!(styles & 64)) {
      
      UDrawObject(&charinst, drawchar, SINGLE, passcolor);

      /* under- and overlines */
      if (styles & 8)
         alphapts[0].y = alphapts[1].y = -6;
      else if (styles & 16)
         alphapts[0].y = alphapts[1].y = groupheight + 4;
      if (styles & 24) {
         alphapts[0].x = 0; alphapts[1].x = localwidth;
         UDrawSimpleLine(localdata, &alphapts[0], &alphapts[1]);
      }
   }
   return localwidth;
}

/*----------------------------------------------------------------------*/

void UDrawString(localdata, drawlabel, passcolor)
  objinstptr 	localdata;
  labelptr	drawlabel;
  int		passcolor;
{
   uchar *strptr = drawlabel->string;
   short  fstyle, ffont, tmpwidth, tmpjust, baseline;
   short  backct = 1;
   int    group = 0;
   int    defaultcolor, curcolor;
   short  oldx, oldy, oldfont, oldstyle;
   float  tmpscale = 1.0;
   float  tmpthick = xobjs.pagelist[areastruct.page]->wirewidth;
   XPoint newpoint, bboxin[2], bboxout[2];
   Boolean xm, ym;
   TextExtents tmpext;

   if (fontcount == 0) return;

   if (passcolor == DOSUBSTRING)
      defaultcolor = curcolor = drawlabel->color;
   else
      defaultcolor = curcolor = passcolor;

   if (defaultcolor != DOFORALL) {
      if (drawlabel->color != DEFAULTCOLOR)
	 curcolor = drawlabel->color;
      else
	 curcolor = defaultcolor;
      XTopSetForeground(curcolor);
   }

   /* calculate the transformation matrix for this object */
   /* in natural units of the alphabet vectors		  */
   /* (conversion to window units)			  */

   UPushCTM();
   UPreMultCTM(DCTM, drawlabel->position, drawlabel->scale, drawlabel->rotation);

   /* check for flip invariance; recompute CTM and justification if necessary */

   tmpjust = flipadjust(drawlabel->justify);

   /* "natural" (unscaled) length */
   tmpext = ULength(drawlabel->string, 0.0, 0, NULL);

   newpoint.x = (tmpjust & NOTLEFT ?
       (tmpjust & RIGHT ? -tmpext.width : -tmpext.width >> 1) : 0);
   newpoint.y = (tmpjust & NOTBOTTOM ?
       (tmpjust & TOP ? -tmpext.ascent : -(tmpext.ascent + tmpext.base) >> 1)
		: -tmpext.base);

#ifdef SCHEMA
   /* Pinlabels have an additional offset spacing to pad */
   /* them from the circuit point to which they attach.  */

   if (drawlabel->pin) {
      if (!areastruct.schemon)
	 goto enddraw;	/* Don't draw pins when schematic capture is off */
      else {
	 pinadjust(tmpjust, &(newpoint.x), &(newpoint.y), 1);
         UDrawX(drawlabel);
      }
   }
#endif

   oldx = newpoint.x;
   oldy = newpoint.y;
   baseline = newpoint.y;

   /* do quick calculation on bounding box; don't draw if off-screen */

   bboxin[0].x = newpoint.x;
   bboxin[0].y = newpoint.y + tmpext.descent;
   bboxin[1].x = newpoint.x + tmpext.width;
   bboxin[1].y = newpoint.y + tmpext.ascent;
   UTransformbyCTM(DCTM, bboxin, bboxout, 2);
   xm = (bboxout[0].x < bboxout[1].x) ? 0 : 1;
   ym = (bboxout[0].y < bboxout[1].y) ? 0 : 1;

   if (bboxout[xm].x < areastruct.width && bboxout[ym].y < areastruct.height &&
       bboxout[!xm].x > 0 && bboxout[!ym].y > 0) {

       for (strptr = drawlabel->string; *strptr != '\0'; strptr++) {

          /* make a character transformation matrix by translating to newpoint */

          UPushCTM();
          UPreMultCTM(DCTM, newpoint, tmpscale, 0);

          /* deal with in-line text format control escape sequences */

          if ((short)(*strptr) == TEXT_ESC) {
	     short control = (short)(*(++strptr));

             if (control >= FONT_START) {
	        ffont = (control - FONT_START);
		fstyle = 0;		   /* style reset by font change */
	        if (oldy == newpoint.y) {  /* set top-level font and style */
	           oldfont = ffont;
	           oldstyle = fstyle;
	        }
		
		/* simple boldface technique for derived fonts */

		xobjs.pagelist[areastruct.page]->wirewidth =
   		   ((fonts[ffont].flags & 0x21) == 0x21) ?  4.0 : 2.0;
             }
	     else if (control == FONT_SCALE) {
	        *(strptr + 5) = '\0';
	        sscanf(++strptr, "%f", &tmpscale);
	        *(strptr + 4) = TEXT_ESC;
	        strptr += 5;
	     }
	     else if (control == KERN) {
	        newpoint.x += (short)(*(strptr + 1)) - 128;
	        newpoint.y += (short)(*(strptr + 2)) - 128;
	        strptr += 4;
	     }
	     else if (control == FONT_COLOR) {
		if (defaultcolor != DOFORALL) {
		   if (*(strptr + 1) > 1)
		      curcolor = colorlist[(int)(*(strptr + 1)) - 2].color.pixel;
		   else {
		      if (curcolor != DEFAULTCOLOR)
			 XTopSetForeground(defaultcolor);
		      curcolor = DEFAULTCOLOR;
		   }
		}
	        strptr += 3;
	     }
	     else if (control == RETURN) {
		baseline -= BASELINE * tmpscale;
	        newpoint.y = baseline;
		newpoint.x = oldx;
		fstyle &= 0xfc7;
	     }
             else if (control == SUBSCRIPT) {
	        tmpscale *= SUBSCALE; 
	        newpoint.y -= (short)((TEXTHEIGHT >> 1) * tmpscale);
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == SUPERSCRIPT) {
	        tmpscale *= SUBSCALE;
	        newpoint.y += (short)(TEXTHEIGHT * tmpscale);
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == NORMALSCRIPT) {
	        tmpscale = 1.0;
	        ffont = oldfont;	/* revert to top-level font and style */
	        fstyle = oldstyle;
	        newpoint.y = oldy;
	        fstyle &= 0xfc7;	/* cancel under/overline */
             }
             else if (control == UNDERLINE) {
	        fstyle &= 0xfc7;
	        fstyle |= 8;
	     }
             else if (control == OVERLINE) {
	        uchar *quickptr = strptr + 1;
		objectptr charptr;
		int tmpheight;

	        /* any control character stops the /overline */ 
		group = 0;
	        for (; (*quickptr != TEXT_ESC) && (*quickptr != '\0'); quickptr++) {
		   charptr = fonts[ffont].encoding[*quickptr];
		   tmpheight = (int)((float)charptr->height * fonts[ffont].scale);
		   if (group < tmpheight) group = tmpheight;
		}
	        fstyle &= 0xfc7;
	        fstyle |= 16;
             }
             else if (control == NOLINE) fstyle &= 0xfc7;
             else if (control == BACKSPACE) {
		objectptr charptr;
	        if (strptr + 1 - (backct * 3) >= drawlabel->string) {
	           newpoint.x -= UDrawChar(localdata, *(strptr + 1 - (backct * 3)), 
	                  fstyle | 64, ffont, group, curcolor) * tmpscale;
	        }
	        backct = ((short)(*(strptr + 1)) != TEXT_ESC || (short)(*(strptr + 2))
	    	!= BACKSPACE) ? 1 : backct + 1;
             }
	     else if (control == HALFSPACE || control == QTRSPACE) {
		short addx = UDrawChar(localdata, (uchar)32, fstyle, ffont, group,
			curcolor);
		newpoint.x += addx >> ((control == HALFSPACE) ? 1 : 2);
	     }
          }
          else if ((short)(*strptr) != NO_OP) {
	     /* Special case of selection:  only substring is drawn in the	*/
	     /* selection color.						*/
	     int scolor = curcolor;
	     if (passcolor == DOSUBSTRING) {
	        short pos = (short)(strptr - drawlabel->string);
	        if (pos < textpos && pos >= textend)
		   scolor = SELECTCOLOR;
		else if (pos == textpos)
      		   XTopSetForeground(curcolor);
	     }
	     newpoint.x += UDrawChar(localdata, *strptr, fstyle, ffont, group,
			scolor) * tmpscale;
	  }
   
          /* pop the character transformation matrix */

          UPopCTM();
      }
   }

enddraw:

   /* pop the string transformation matrix */

   UPopCTM();

   if ((defaultcolor != DOFORALL) && (passcolor != curcolor))
      XTopSetForeground(passcolor);

   xobjs.pagelist[areastruct.page]->wirewidth = tmpthick;
}

/*------------------------------------------------------------------------*/

TextExtents ULength(string, newscale, dostop, tbreak)
  uchar *string;
  float	newscale;
  short dostop;
  XPoint *tbreak;
{
   short backct = 1;
   float oldscale, strscale, locscale = 1.0, xtotal = 0.5;
   float lasttotal = xtotal;
   uchar *strptr, *lastptr;
   objinstptr *scaleobj;
   objectptr *somebet = NULL, chptr;
   short locpos;
   float ykern;
   TextExtents retext;

   retext.width = retext.ascent = retext.descent = retext.base = 0;
   if (fontcount == 0) return retext;

   if (newscale > 0.0) strscale = UTopScale() * newscale;
   else strscale = 1.0;
     
   oldscale = strscale;
   lastptr = string;
   ykern = 0.0;

   for (strptr = string; *strptr != '\0'; strptr++) {

      if (dostop && ((short)(strptr - string) >= dostop)) break;

      if ((short)(*strptr) == TEXT_ESC) {
	 short control = (short)(*(++strptr));	 

         if (control == SUPERSCRIPT) {
	    strscale *= SUBSCALE;
	    ykern += TEXTHEIGHT * strscale / 2.0;
	 }
         else if (control == SUBSCRIPT) {
	    strscale *= SUBSCALE;
	    ykern -= TEXTHEIGHT * strscale / 2.0;
	 }
         else if (control == NORMALSCRIPT) {
	    strscale = oldscale;
	    ykern = 0.0;
	 }
	 else if (control == RETURN) {
	    retext.base -= BASELINE * strscale;
	    if (!dostop)
	       retext.width = max(retext.width, xtotal);
	    xtotal = 0.5;
	 }
	 else if (control == HALFSPACE) {
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale / 2;
	 }      
	 else if (control == QTRSPACE) {
	    chptr = (*(somebet + 32));
	    xtotal += (float)(chptr->width + chptr->lowerleft.x) 
			* locscale * strscale / 4;
	 }
	 else if (control == BACKSPACE) {
	    if (strptr + 1 - (backct * 3) >= string) {
	       chptr = (*(somebet + *(strptr + 1 - (backct * 3))));
	       xtotal -= (float)(chptr->width + chptr->lowerleft.x)
			 * locscale * strscale;
	    }
	    backct = (*(strptr + 1) != TEXT_ESC || *(strptr + 2) != BACKSPACE)
		 ? 1 : backct + 1;
         }
         else if (control == FONT_SCALE) {
	    *(strptr + 5) = '\0';
	    sscanf(++strptr, "%f", &strscale);
   	    if (newscale > 0.0)
	       strscale *= UTopScale();
	    *(strptr + 4) = TEXT_ESC;
	    strptr += 5;
	 }
	 else if (control == KERN) {
	    xtotal += (short)(*(strptr + 1)) - 128;
	    ykern += (short)(*(strptr + 2)) - 128;
	    strptr += 4;
	 }
	 else if (control == FONT_COLOR) {
	    strptr += 3;
	 }
         else if (control >= FONT_START) {
	    somebet = fonts[control - FONT_START].encoding;
	    locscale = fonts[control - FONT_START].scale;
	 }
      }
      else if ((short)(*strptr) != NO_OP) {
	 chptr = (*(somebet + *strptr));
	 xtotal += (float)(chptr->width + chptr->lowerleft.x)
			* locscale * strscale;
	 retext.ascent = max(retext.ascent, (short)(retext.base + ykern +
			(float)((chptr->height + chptr->lowerleft.y)
			* locscale * strscale)));
	 retext.descent = min(retext.descent, (short)(retext.base + ykern +
			(float)(chptr->lowerleft.y * locscale * strscale)));
      }

      if (tbreak != NULL) 
	 if ((xtotal > tbreak->x) && (retext.base <= tbreak->y)) break;
      lasttotal = xtotal;
      lastptr = strptr;
   }

   /* special case: return character position in retext.width */
   if (tbreak != NULL) {
      if ((tbreak->x - lasttotal) < (xtotal - tbreak->x))
	 locpos = (short)(lastptr - string + 1);
      else
         locpos = (short)(strptr - string + 1);
      if (locpos < 2) locpos = 2;
      if (*(string + locpos - 1) == TEXT_ESC) {
	 locpos++;
	 switch (*(string + locpos)) {
	    case FONT_SCALE:
	       locpos += 2;
	    case KERN:
	       locpos++;
	    case FONT_COLOR:
	       locpos += 4;
	 }
      }
      if (locpos > strlen(string)) {
	 locpos = strlen(string);
         if (*(string + locpos - 1) == TEXT_ESC)
	    locpos--;
      }
      retext.width = locpos;
      return retext;
   }
   retext.width = max(retext.width, xtotal);
   return retext;
}

/*------------------------------------------------------------------------*/
/* simple routines for drawing and erasing labels */
/*------------------------------------------------------------------------*/

undrawtext(settext)
  labelptr settext;
{
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XTopSetForeground(BACKGROUND);
   UDrawString(areastruct.topobject, settext, DOFORALL);
}

redrawtext(settext)
  labelptr settext;
{
   UDrawString(areastruct.topobject, settext, settext->color);
}

/* Global value of distance between characters in the font catalog */
short del;

/*-------------------------------------*/
/* Draw the catalog of font characters */
/*-------------------------------------*/

void composefontlib(short cfont)
{
   objinstptr *drawinst;
   labelptr *drawname;
   objectptr *curlib, libobj, saveobj, nullobj;
   objectptr libinst = xobjs.libtop[FONTLIB];
   short visobjects, libobjects, i, qdel;
   polyptr *drawbox;
   pointlist pointptr;
   XPoint dpt;

   reset(libinst, NORMAL);

   /* Create a pointer to the font library */

   curlib = xobjs.fontlib.library;

   /* Find the number of non-null characters.  Do this by assuming */
   /* that all fonts encode nullchar at position zero.		   */

   visobjects = 0;
   nullobj = fonts[cfont].encoding[0];
   for(i = 1; i < 256; i++)
      if (fonts[cfont].encoding[i] != nullobj) visobjects++;

   /* add the box and gridlines */

   visobjects += 34;

   /* generate the list of object instances */

   libinst->plist = (genericptr *) realloc(libinst->plist, visobjects
		* sizeof(genericptr));
   libinst->parts = 0;

   /* 0.5 is the default vscale;  16 is no. characters per line */
   del = min(areastruct.width, areastruct.height) / (0.5 * 16);
   qdel = del >> 2;

   for (i = 0; i < 256; i++) {

      if ((libobj = fonts[cfont].encoding[i]) == nullobj) continue;
      
      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->scale = 1.0;
      (*drawinst)->thisobject = libobj;
      (*drawinst)->params = NULL;
      (*drawinst)->num_params = (uchar)0;

      (*drawinst)->position.x = (i % 16) * del + qdel;
      (*drawinst)->position.y = -(i / 16) * del + qdel;

      libinst->parts++;
   }

   /* separate characters with gridlines (17 vert., 17 horiz.) */

   for (i = 0; i < 34; i++) {
      drawbox = (polyptr *)libinst->plist + libinst->parts;
      *drawbox = (polyptr) malloc(sizeof(polygon));
      (*drawbox)->type = POLYGON;
      (*drawbox)->color = SNAPCOLOR;   /* default red */
      (*drawbox)->style = UNCLOSED;
      (*drawbox)->width = 1.0;
      (*drawbox)->number = 2;
      (*drawbox)->points = (pointlist) malloc(2 * sizeof(XPoint));
      if (i < 17) {
         pointptr = (*drawbox)->points;
         pointptr->x = i * del;
         pointptr->y = 0;
         pointptr = (*drawbox)->points + 1;
         pointptr->x = i * del;
         pointptr->y = -16 * del;
      }
      else {
         pointptr = (*drawbox)->points;
         pointptr->x = 0;
         pointptr->y = (17 - i) * del;
         pointptr = (*drawbox)->points + 1;
         pointptr->x = 16 * del;
         pointptr->y = (17 - i) * del;
      }
      libinst->parts++;
   }

   /* calculate a bounding box for this display */

   saveobj = objectdata;
   objectdata = libinst;
   calcbbox(objectdata);
   objectdata = saveobj;
}

/*------------------------------------------------------*/
/* ButtonPress handler during font catalog viewing mode */
/*------------------------------------------------------*/

XtEventHandler fontcatbutton(w, mode, event)
  Widget w;
  caddr_t mode;
  XButtonEvent *event;
{
   short chx, chy;
   unsigned long rch = 0;

   if (event->button == Button1 || event->button == Button2) {

      window_to_user(event->x, event->y, &areastruct.save);
 
      chy = -areastruct.save.y / del + 1;
      chx = areastruct.save.x / del;

      chx = min(15, chx);
      chy = min(15, chy);
    
      rch = (unsigned long)(chy * 16 + chx);
   }

   catreturn();

   if (rch == (unsigned long)'\\')
      labeltext(rch, 256);
   else if (rch != 0)
      labeltext(rch, 0);
}

/*-------------------------------------------------------------------------*/
