/*-------------------------------------------------------------------------*/
/* libraries.c --- xcircuit routines for the builtin and user libraries    */
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(DARWIN)
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"

#include <math.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

extern short   eventmode;	/* keep track of the mode the screen is in */
extern Display	*dpy;   /* Works well to make this globally accessible */
extern Window win;
extern int *appcolors;
extern Cursor	appcursors[NUM_CURSORS];
extern Globaldata xobjs;
extern Clientdata areastruct;
extern char _STR[150];
extern objectpair *pushlist;
extern short fontcount;
extern fontinfo *fonts;

/*---------------------------------------------------------*/

#ifdef SCHEMA
#define _lowerleft lleft2
#define _width width2
#define _height height2
#else
#define _lowerleft lowerleft
#define _width width
#define _height height
#endif

/*---------------------------------------------------------*/
/* Find the Helvetica font for use in labeling the objects */
/*---------------------------------------------------------*/

short findhelvetica()
{
   short fval;

   for (fval = 0; fval < fontcount; fval++)
      if (!strcmp(fonts[fval].psname, "Helvetica"))
	 break; 

   /* If not there, use the first Helvetica font */

   if (fval == fontcount) {
      for (fval = 0; fval < fontcount; fval++)
         if (!strcmp(fonts[fval].family, "Helvetica"))
	    break; 
   }

   /* If still not there, use the first non-Symbol font */
   /* If this doesn't work, then the libraries are probably misplaced. . .*/

   if (fval == fontcount) {
      for (fval = 0; fval < fontcount; fval++)
         if (strcmp(fonts[fval].family, "Symbol"))
	    break; 
   }

   return fval;
}

/*-------------------------------------------*/
/* Return to drawing window from the library */
/*-------------------------------------------*/

void catreturn()
{
   /* Pop the object being edited from the push stack. */

   popobject(NULL, Number(1), NULL);
}

/*------------------------------------------------------*/
/* Find page number from cursor position		*/
/* Mode = 0:  Look for exact corresponding page number  */
/*   and return -1 if out-of-bounds			*/
/* Mode = 1:  Look for position between pages, return	*/
/*   page number of page to the right.  		*/
/*------------------------------------------------------*/

int pageposition(short libmode, XButtonEvent *event, int mode)
{
   int xin, yin, bpage, pages;
   int gxsize, gysize, xdel, ydel;

   pages = (libmode == PAGELIB) ? xobjs.pages : xobjs.numlibs;
   computespacing(libmode, &gxsize, &gysize, &xdel, &ydel);
   window_to_user(event->x, event->y, &areastruct.save);

   if (mode == 0) {	/* On-page */
      if (areastruct.save.x >= 0 && areastruct.save.y <= 0) {
         xin = areastruct.save.x / xdel;
         yin = areastruct.save.y / ydel; 
         if (xin < gxsize && yin > -gysize) {
            bpage = (xin % gxsize) - (yin * gxsize);
            if (bpage < pages)
	       return bpage;
         }
      }
      return -1;
   }
   else {		/* Between-pages */
      xin = (areastruct.save.x + (xdel >> 1)) / xdel;
      if (xin > gxsize) xin = gxsize;
      if (xin < 0) xin = 0;
      yin = areastruct.save.y  / ydel; 
      if (yin > 0) yin = 0;
      if (yin < -gysize) yin = -gysize;
      bpage = (xin % (gxsize + 1)) + 1 - (yin * gxsize);
      if (bpage > pages + 1) bpage = pages + 1;
      return bpage;
   }
}

/*------------------------------------------------------*/
/* Test whether an object is a page, and return the	*/
/* page number if it is.  Otherwise, return -1.		*/
/*------------------------------------------------------*/

int is_page(objectptr thisobj)
{
   int i;
   
   for (i = 0; i < xobjs.pages; i++)
      if (xobjs.pagelist[i]->pageobj == thisobj) return i;

   return -1;
}

/*------------------------------------------------------*/
/* Test whether an object is a library, and return the	*/
/* library number if it is.  Otherwise, return -1.	*/
/*------------------------------------------------------*/

int is_library(objectptr thisobj)
{
   int i;
   
   for (i = 0; i < xobjs.numlibs; i++)
      if (xobjs.libtop[i + LIBRARY] == thisobj) return i;

   return -1;
}

/*------------------------------------------------------*/
/* ButtonPress handler during page catalog viewing mode */
/*------------------------------------------------------*/

void pagecatbutton(XButtonEvent *event)
{
   int bpage;
   short mode;

   for (mode = 0; mode < LIBRARY; mode++) {
      if (objectdata == xobjs.libtop[mode]) break;
   }
   if (mode == LIBRARY) return;  /* Something went wrong if this happens */

   if (event->button == Button1 || event->button == Button2) {
      if ((bpage = pageposition(mode, event, 0)) >= 0) {
		  
#ifdef SCHEMA
	 if (eventmode == ASSOC_MODE) {
	    if (mode == PAGELIB) {
	       /* using changepage() allows use of new page for schematic */
	       changepage(bpage);
	       /* associate the new schematic */
	       schemassoc(objectdata, pushlist->thisobject);
	       /* pop back to calling (symbol) page */
	       catreturn();
	       eventmode = PRESS_MODE;
	    }
	    else {
	       startcatalog(NULL, LIBRARY + bpage, NULL);
	    }
	    return;
         }
	 else
#endif
	 if (event->button == Button1) {

	    /* like catreturn(), but don't actually go to the popped page */
	    objectdeselect();
	    eventmode = NORMAL_MODE;
	    if (mode == PAGELIB) {
	       newpage(bpage);
      	       eventmode = PRESS_MODE;
	    }
	    else {
	       startcatalog(NULL, LIBRARY + bpage, NULL);
	    }
	    return;
	 }
	 else {  /* Button2 */
	    if (mode == PAGELIB)    /* No such method for LIBLIB is defined. */
	       objectselect(OBJECT);
	 }
      }
   }
   else {
      eventmode = NORMAL_MODE;
      catreturn();
   }
}

/*------------------------------------------------------------------------------*/
/* Subroutine to find the correct scale and position of the object instance	*/
/* representing an entire page in the page directory.				*/
/*------------------------------------------------------------------------------*/

void pageinstpos(short mode, short tpage, objinstptr drawinst, int gxsize,
	int gysize, int xdel, int ydel)
{
   objectptr libobj = drawinst->thisobject;
   float scalex, scaley;

   drawinst->position.x = (tpage % gxsize) * xdel;
   drawinst->position.y = -(tpage / gxsize + 1) * ydel;

   /* center the object on its page bounding box */

   if (libobj->width == 0 || libobj->height == 0) {
      drawinst->scale = 0.45 * libobj->viewscale;
      drawinst->position.x += 0.05 * xdel - libobj->pcorner.x * drawinst->scale;
      drawinst->position.y += 0.05 * ydel - libobj->pcorner.y * drawinst->scale;
   }
   else {
      scalex = (0.9 * xdel) / libobj->width;
      scaley = (0.9 * ydel) / libobj->height;
      if (scalex > scaley) {
         drawinst->scale = scaley;
	 drawinst->position.x -= (libobj->lowerleft.x * scaley);
         drawinst->position.x += (xdel - (libobj->width * scaley)) / 2;
         drawinst->position.y += 0.05 * ydel - libobj->lowerleft.y
			* drawinst->scale;
      }
      else {
         drawinst->scale = scalex;
	 drawinst->position.y -= (libobj->lowerleft.y * scalex);
         drawinst->position.y += (ydel - (libobj->height * scalex)) / 2;
         drawinst->position.x += 0.05 * xdel - libobj->lowerleft.x
			* drawinst->scale;
      }
   }
}

/*-----------------------------------------------------------*/
/* Make a new instance for inserting into the page directory */
/*-----------------------------------------------------------*/

objinstptr makeinstofpage(objectptr libinst, objectptr libobj)
{
   objinstptr *drawinst;

   NEW_OBJINST(drawinst, libinst);
   (*drawinst)->color = DEFAULTCOLOR;
   (*drawinst)->rotation = 1;
   (*drawinst)->params = NULL;		/* values are the defaults */
   (*drawinst)->thisobject = libobj;
   libinst->parts++;
   return *drawinst;
}

/*-----------------------------------------------------------*/
/* Find spacing of objects for pages in the page directories */
/*-----------------------------------------------------------*/

void computespacing(short mode, int *gxsize, int *gysize, int *xdel, int *ydel)
{
   int pages = (mode == PAGELIB) ? xobjs.pages : xobjs.numlibs;

   *gxsize = (int)sqrt((double)pages) + 1;
   *gysize = 1 + pages / (*gxsize);

   /* 0.5 is the default vscale;  g#size is the number of pages per line */

   *xdel = areastruct.width / (0.5 * (*gxsize));
   *ydel = areastruct.height / (0.5 * (*gysize));
}

/*--------------------------------------*/
/* Draw the catalog of page ordering	*/
/*--------------------------------------*/

void composepagelib(short mode)
{
   objinstptr drawinst;
   objectptr libobj, libinst = xobjs.libtop[mode];
   short i;
   polyptr *drawbox;
   labelptr *pagelabel;
   stringpart *strptr;
   pointlist pointptr;
   int bbuf, xdel, ydel, gxsize, gysize;
   int pages = (mode == PAGELIB) ? xobjs.pages : xobjs.numlibs;
   short fval = findhelvetica();

   reset(libinst, NORMAL);

   /* generate the list of object instances */

   libinst->plist = (genericptr *)malloc(sizeof(genericptr));
   libinst->parts = 0;

   computespacing(mode, &gxsize, &gysize, &xdel, &ydel);
   bbuf = xdel / 40;

   for (i = 0; i < pages; i++) {
      libobj = (mode == PAGELIB) ? xobjs.pagelist[i]->pageobj
		: xobjs.libtop[i + LIBRARY];
      if (libobj != NULL) {
         drawinst = makeinstofpage(libinst, libobj);
         pageinstpos(mode, i, drawinst, gxsize, gysize, xdel, ydel);
      }

      /* separate pages (including empty ones) with bounding boxes */

      NEW_POLY(drawbox, libinst);
      (*drawbox)->color = SNAPCOLOR;   /* default red */
      (*drawbox)->style = NORMAL;      /* CLOSED */
      (*drawbox)->width = 1.0;
      (*drawbox)->number = 4;
      (*drawbox)->points = (pointlist) malloc(4 * sizeof(XPoint));
      pointptr = (*drawbox)->points;
      pointptr->x = (i % gxsize) * xdel + bbuf;
      pointptr->y = -(i / gxsize) * ydel - bbuf;
      pointptr = (*drawbox)->points + 1;
      pointptr->x = ((i % gxsize) + 1) * xdel - bbuf;
      pointptr->y = -(i / gxsize) * ydel - bbuf;
      pointptr = (*drawbox)->points + 2;
      pointptr->x = ((i % gxsize) + 1) * xdel - bbuf;
      pointptr->y = -((i / gxsize) + 1) * ydel + bbuf;
      pointptr = (*drawbox)->points + 3;
      pointptr->x = (i % gxsize) * xdel + bbuf;
      pointptr->y = -((i / gxsize) + 1) * ydel + bbuf;
      libinst->parts++;

      /* each page gets its name printed at the bottom */

      if (libobj != NULL) {
	 NEW_LABEL(pagelabel, libinst);
	 labeldefaults(*pagelabel, False, (pointptr->x + (pointptr-1)->x) / 2,
		pointptr->y - 5);
	 (*pagelabel)->color = DEFAULTCOLOR;
	 (*pagelabel)->scale = 0.75;
	 (*pagelabel)->string->data.font = fval;
	 strptr = makesegment(&((*pagelabel)->string), NULL);
	 strptr->type = TEXT_STRING;
	 strptr->data.string = (char *) malloc(1 + strlen(libobj->name));
	 strcpy(strptr->data.string, libobj->name);
	 (*pagelabel)->justify = TOP | NOTBOTTOM | NOTLEFT;
	 libinst->parts++;
      }
   }

   /* calculate a bounding box for this display */
   /* and center it in its window */

   calcbbox(libinst);
   centerview(libinst);
}

/*------------------------------------------------------------*/
/* Update the page or library directory based on new bounding */
/* box information for the page or library passed in "tpage". */
/*------------------------------------------------------------*/

void updatepagelib(short mode, short tpage)
{
   objectptr compobj, libinst = xobjs.libtop[mode];
   objinstptr pinst;
   genericptr *gelem;
   int i, xdel, ydel, gxsize, gysize, lpage;

   /* lpage is the number of the page as found on the directory page */
   lpage = (mode == PAGELIB) ? tpage : tpage - LIBRARY;
   compobj = (mode == PAGELIB) ? xobjs.pagelist[tpage]->pageobj
		: xobjs.libtop[tpage];

   computespacing(mode, &gxsize, &gysize, &xdel, &ydel);

   for (i = 0; i < libinst->parts; i++) {
      gelem = libinst->plist + i;
      if ((*gelem)->type == OBJECT) {
	 pinst = TOOBJINST(gelem);
         if (pinst->thisobject == compobj) {
	    /* recalculate scale and position of the object instance */
            pageinstpos(mode, lpage, pinst, gxsize, gysize, xdel, ydel);
	    break;
	 }
      }
   }

   /* if there is no instance for this page, then recompose the whole library */

   if (i == libinst->parts) composelib(mode);
}

/*----------------------*/
/* Rearrange pages	*/
/*----------------------*/

void pagecatmove(XKeyEvent *event)
{
   int bpage;
   short *newselect;
   objectptr exchobj;
   Pagedata *ipage, **testpage, **tpage2;

   if (areastruct.selects == 0) return;
   else if (areastruct.selects > 2) {
      Wprintf("Select maximum of two objects.");
      return;
   }

   /* Get the page corresponding to the first selected object */

   newselect = areastruct.selectlist;
   exchobj = SELTOOBJINST(newselect)->thisobject;
   for (testpage = xobjs.pagelist; testpage < xobjs.pagelist + xobjs.pages; testpage++)
      if (*testpage != NULL && (*testpage)->pageobj == exchobj) break;

   /* If two objects are selected, then exchange their order */

   if (areastruct.selects == 2) {
      newselect = areastruct.selectlist + 1;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      for (tpage2 = xobjs.pagelist; tpage2 < xobjs.pagelist + xobjs.pages; tpage2++)
	 if (*tpage2 != NULL && (*tpage2)->pageobj == exchobj) break;

      ipage = *testpage;
      *testpage = *tpage2;
      *tpage2 = ipage;
   }

   /* If one object selected; find place to put from cursor position */

   else if ((bpage = pageposition(PAGELIB, (XButtonEvent *)event, 1)) >= 0) {
      int k, epage;
      Pagedata *eptr;

      /* Find page number of the original page */

      epage = (int)(testpage - xobjs.pagelist);
      eptr = *(xobjs.pagelist + epage);

      /* move page (epage) to position between current pages */
      /* (bpage - 2) and (bpage - 1) by shifting pointers.   */

      if ((bpage - 1) < epage) {
	 for (k = epage - 1; k >= bpage - 1; k--) {
	    *(xobjs.pagelist + k + 1) = *(xobjs.pagelist + k);
	    renamepage(k + 1);
	 }
	 *(xobjs.pagelist + bpage - 1) = eptr;
	 renamepage(bpage - 1);
      }
      else if ((bpage - 2) > epage) {
	 for (k = epage + 1; k <= bpage - 2; k++) {
	    *(xobjs.pagelist + k - 1) = *(xobjs.pagelist + k);
	    renamepage(k - 1);
	 }
	 *(xobjs.pagelist + bpage - 2) = eptr;
	 renamepage(bpage - 2);
      }
   }

   objectdeselect();
   composelib(PAGELIB);
   drawarea(NULL, NULL, NULL);
}

/*-----------------------------------------*/
/* Draw the catalog of predefined elements */
/*-----------------------------------------*/

void composelib(short mode)
{
   short visobjects, libobjects;
   objinstptr *drawinst;
   labelptr *drawname;
   objectptr *curlib, *libobj, libinst = xobjs.libtop[mode];
   int xpos = 0, ypos = areastruct.height << 1;
   int nypos = 220, nxpos;
   short fval;

   /* Also make composelib() a wrapper for composepagelib() */
   if ((mode > FONTLIB) && (mode < LIBRARY)) {
      composepagelib(mode);
      return;
   }

   reset(libinst, NORMAL);

   /* Create pointers to the current library and number of objects */

   curlib = xobjs.userlibs[mode - LIBRARY].library;
   libobjects = xobjs.userlibs[mode - LIBRARY].number;
   if (libobjects == 0) return;

   /* Look for all "hidden" objects and subtract them from the total number */

   visobjects = libobjects;
   for(libobj = curlib; libobj < curlib + libobjects; libobj++)
      if ((*libobj)->hidden == True)
	 visobjects--;

   /* Find the Helvetica font for use in labeling the objects */

   fval = findhelvetica();

   /* generate the list of object instances and their labels */

   libinst->plist = (genericptr *) realloc(libinst->plist,
	((fval == fontcount) ? visobjects : (visobjects * 2)) * sizeof(genericptr));
   libinst->parts = 0;

   for(libobj = curlib; libobj < curlib + libobjects; libobj++){
      
      /* "Hidden" objects are not drawn */

      if ((*libobj)->hidden == True) continue;

      /* Otherwise, determine the area needed on the page to draw the object */

      nxpos = xpos + (((*libobj)->_width > 170) ? (*libobj)->_width + 30 : 200);
      if ((nxpos > (areastruct.width << 1)) && (xpos > 0)) {
	 nxpos -= xpos; 
	 xpos = 0;
	 ypos -= nypos;
	 nypos = 200;
      }
      /* extra space of 20 is to leave room for the label */

      if ((*libobj)->_height > (nypos - 50)) nypos = (*libobj)->_height + 50;

      drawinst = (objinstptr *)libinst->plist + libinst->parts;
      *drawinst = (objinstptr) malloc(sizeof(objinst));
      (*drawinst)->type = OBJECT;
      (*drawinst)->color = DEFAULTCOLOR;
      (*drawinst)->rotation = 1;
      (*drawinst)->scale = 1.0;
      (*drawinst)->params = NULL;	/* library object has all default values */
      (*drawinst)->thisobject = (*libobj);

      (*drawinst)->position.x = xpos - (*libobj)->_lowerleft.x;
      if ((*libobj)->_width <= 170) (*drawinst)->position.x += 
	   ((170 - (*libobj)->_width) >> 1);

      (*drawinst)->position.y = ypos - ((*libobj)->_height + (*libobj)->_lowerleft.y);
      if ((*libobj)->_height <= 170) (*drawinst)->position.y -=
	   ((170 - (*libobj)->_height) >> 1);

      if (fval < fontcount) {
	 stringpart *strptr;

         drawname = (labelptr *)libinst->plist + (libinst->parts++) + visobjects;
         *drawname = (labelptr) malloc(sizeof(label));      
         (*drawname)->type = LABEL;
	 labeldefaults(*drawname, False, 0, 0);

         (*drawname)->color = DEFAULTCOLOR;
         (*drawname)->scale = 0.75;
	 (*drawname)->string->data.font = fval;
	 strptr = makesegment(&((*drawname)->string), NULL);
	 strptr->type = TEXT_STRING;
	 strptr->data.string = (char *) malloc(1 + strlen((*libobj)->name));
         strcpy(strptr->data.string, (*libobj)->name);
         (*drawname)->justify = TOP | NOTBOTTOM | NOTLEFT;
#ifdef SCHEMA
         (*drawname)->pin = False;
#endif

         if ((*libobj)->_width > 170)
            (*drawname)->position.x = xpos + ((*libobj)->_width >> 1);
         else
            (*drawname)->position.x = xpos + 85;

         if ((*libobj)->_height > 170)
            (*drawname)->position.y = (*drawinst)->position.y +
			(*libobj)->_lowerleft.y - 10;
         else
            (*drawname)->position.y = ypos - 180;

      }
      else (libinst->parts++);
      xpos = nxpos;
   }
   libinst->parts = (fval == fontcount) ? visobjects : (visobjects * 2);

   /* Compute the bounding box of the library page */
   calcbbox(libinst);
   centerview(libinst);

   /* Update the library directory */
   updatepagelib(LIBLIB, mode);
}

/*----------------------------------------------------------------*/
/* Find any dependencies on an object.				  */
/*   Return values:  0 = no dependency, 1 = dependency on page,	  */
/*	2 = dependency in another library object.		  */
/*   Object/Page with dependency (if any) returned in "compobjp". */
/*----------------------------------------------------------------*/

short finddepend(objinstptr libobj, objectptr **compobjp)
{
   genericptr *testobj;
   short page, i, j;
   objectptr *compobj;
  
   for (i = 0; i < xobjs.numlibs; i++) {
      for (j = 0; j < xobjs.userlibs[i].number; j++) {
	 compobj = xobjs.userlibs[i].library + j;
         *compobjp = compobj;
		     
         for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	          + (*compobj)->parts; testobj++) {
	    if ((*testobj)->type == OBJECT) {
	       if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 2;
	    }
	 }
      }
   }

   /* also look in the xobjs.pagelist */

   for (page = 0; page < xobjs.pages; page++) {
      compobj = &(xobjs.pagelist[page]->pageobj);
      if (*compobj == NULL) continue;
      *compobjp = compobj;
      for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	       + (*compobj)->parts; testobj++) {
	 if ((*testobj)->type == OBJECT) {
	    if (TOOBJINST(testobj)->thisobject == libobj->thisobject) return 1;
	 }
      }
   }
   return 0;
}

/*----------------------------------------------------------------*/
/* "Hide" an object (must have a dependency or else it disappears)*/
/*----------------------------------------------------------------*/

void cathide()
{
   short i, *newselect;
   objectptr *compobj;
   objinstptr libobj;

   if (areastruct.selects == 0) return;

   /* Can only hide objects which are instances in other objects; */
   /* Otherwise, object would be "lost".			  */

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      if (finddepend(libobj, &compobj) == 0) {
	 sprintf(_STR, "Cannot hide: no dependencies");
	 Wprintf(_STR);
      }
      else { 		/* All's clear to hide. */
	 libobj->thisobject->hidden = True;
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if ((i = is_library(objectdata)) >= 0) composelib(LIBRARY + i);

   drawarea(NULL, NULL, NULL);
}

/*----------------------------------------------------------------*/
/* Delete an object from the library if there are no dependencies */
/*----------------------------------------------------------------*/

void catdelete()
{
   short *newselect;
   genericptr *testobj, *tobj;
   objinstptr libobj;
   objectptr *libpage, *compobj, *tlib, *slib;
   short i, *libpobjs;

   if (areastruct.selects == 0) return;

   if ((i = is_library(objectdata)) >= 0) {
      libpage = xobjs.userlibs[i].library;
      libpobjs = &xobjs.userlibs[i].number;
   }

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {
      libobj = SELTOOBJINST(newselect);

      /* Cannot delete an object if another object uses an instance of it, */
      /* or if the object is used on a page.				   */

      if (finddepend(libobj, &compobj)) {
	 sprintf(_STR, "Cannot delete: dependency in \"%s\"", (*compobj)->name);
	 Wprintf(_STR);
      }
      else { 		/* All's clear to delete.		     	   */

         /* First remove any instances of this object in the delete buffer */

         for (compobj = xobjs.delbuffer.library; compobj != xobjs.delbuffer.library +
                xobjs.delbuffer.number; compobj++) {
	    for (testobj = (*compobj)->plist; testobj < (*compobj)->plist
	          + (*compobj)->parts; testobj++) {
	       if ((*testobj)->type == OBJECT) {
	          if (TOOBJINST(testobj)->thisobject == libobj->thisobject) {
		     free(TOOBJINST(testobj));
		     for (tobj = testobj; tobj < (*compobj)->plist
			   + (*compobj)->parts - 1; tobj++) {
		        (*tobj) = (*(tobj + 1));
		     }
		     (*compobj)->parts--;
		  }
	       }
	    }
         }

	 for (tlib = libpage; tlib < libpage + *libpobjs; tlib++)
	    if ((*tlib) == libobj->thisobject) {
	       for (slib = tlib; slib < libpage + *libpobjs - 1; slib++)
		  (*slib) = (*(slib + 1));
	       (*libpobjs)--;
	       break;
	    }
	 
	 reset(libobj->thisobject, DESTROY);
      }
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   if ((i = is_library(objectdata)) >= 0) composelib(LIBRARY + i);

   drawarea(NULL, NULL, NULL);
}

/*--------------------------------------------------------*/
/* Rearrange objects in the library			  */
/*--------------------------------------------------------*/

void catmove(XKeyEvent *event)
{
   short *newselect;
   objectptr lobj, exchobj;
   objectptr *libpage, iobj, *sobj, *testobj, *tobj2;
   objinstptr *libobj, *saveobj;
   short libpobjs, begflag;
   short i, ocentx, ocenty, rangey;

   /* make catmove() a wrapper for pagecatmove() */

   if (is_library(objectdata) < 0) {
      pagecatmove(event);
      return;
   }

   if (areastruct.selects == 0) return;
   else if (areastruct.selects > 2) {
      Wprintf("Select maximum of two objects.");
      return;
   }

   if ((i = is_library(objectdata)) >= 0) {
      libpage = xobjs.userlibs[i].library;
      libpobjs = xobjs.userlibs[i].number;
   }

   /* If two objects are selected, then exchange their order */

   if (areastruct.selects == 2) {
      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      newselect = areastruct.selectlist + 1;
      lobj = SELTOOBJINST(newselect)->thisobject;

      for (testobj = libpage; testobj < libpage + libpobjs; testobj++)
	 if (*testobj == exchobj) break;

      for (tobj2 = libpage; tobj2 < libpage + libpobjs; tobj2++)
	 if (*tobj2 == lobj) break;

      iobj = *testobj;
      *testobj = *tobj2;
      *tobj2 = iobj;
   }
   else {	/* one object selected; find place to put from cursor position */

      newselect = areastruct.selectlist;
      exchobj = SELTOOBJINST(newselect)->thisobject;
      saveobj = NULL;
      begflag = 1;

      window_to_user(event->x, event->y, &areastruct.save);
	
      for (libobj = (objinstptr *)ENDPART - 1; libobj >=
		(objinstptr *)objectdata->plist; libobj--) {

	 if ((*libobj)->type == OBJECT) {
 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->_lowerleft.x
	        + ((*libobj)->thisobject->_width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->_lowerleft.y
	        + ((*libobj)->thisobject->_height >> 1);
	    rangey = ((*libobj)->thisobject->_height > 200) ? 
	        ((*libobj)->thisobject->_height >> 1) : 100;

	    if (areastruct.save.y < ocenty + rangey && areastruct.save.y 
	         > ocenty - rangey) {
	       if (areastruct.save.x > ocentx) break;
	       saveobj = libobj;
	    }
	 }
      }

      if (libobj < (objinstptr *)objectdata->plist) {

	 /* insert at beginning */
	 if (areastruct.save.y > ocenty - rangey) {
            begflag = 0;
	    libobj++;
	 }
	 /* insert at beginning of a line */
	 else if (saveobj != NULL) {
	    libobj = saveobj - 1;
	 }
	 /* relocate to the end */
	 else {
	    libobj = (objinstptr *)ENDPART;
	    while ((*(--libobj))->type != OBJECT);
	 }
      }

      for (testobj = libpage; testobj < libpage + libpobjs; testobj++)
	 if (*testobj == exchobj) break;
      for (tobj2 = libpage; tobj2 < libpage + libpobjs; tobj2++)
	 if (*tobj2 == (*libobj)->thisobject) break;

      iobj = *testobj;

      /* shift library list */

      if (tobj2 > testobj) {
	 for (sobj = testobj; sobj < tobj2; sobj++)
	    *sobj = *(sobj + 1);
         *(tobj2) = iobj;
      }
      else if (tobj2 != testobj) {
	 for (sobj = testobj; sobj > tobj2 + begflag; sobj--)
	    *sobj = *(sobj - 1);
         *(tobj2 + begflag) = iobj;
      }
   }

   objectdeselect();
   if ((i = is_library(objectdata)) >= 0) composelib(LIBRARY + i);

   drawarea(NULL, NULL, NULL);
}

/*--------------------------------------------------------*/
/* Make a duplicate of an object, put in the User Library */
/*--------------------------------------------------------*/

void copycat()
{
   short *newselect;
   objectptr *newobj, *curlib, oldobj;
   objinstptr libobj;
   int i, libnum = USERLIB - LIBRARY ;

   for (newselect = areastruct.selectlist; newselect < areastruct.selectlist
	  + areastruct.selects; newselect++) {

      libobj = SELTOOBJINST(newselect);
      oldobj = libobj->thisobject;

      /* generate new entry in user library */

      curlib = (objectptr *) realloc(xobjs.userlibs[libnum].library,
	 (xobjs.userlibs[libnum].number + 1) * sizeof(objectptr));
      xobjs.userlibs[libnum].library = curlib;
      newobj = xobjs.userlibs[libnum].library + xobjs.userlibs[libnum].number;
      *newobj = (objectptr) malloc(sizeof(object));
      xobjs.userlibs[libnum].number++;

      /* give the new object a unique name */

      sprintf(_STR, "_%s", oldobj->name);
      checkname(*newobj);

      /* copy other object properties */

      (*newobj)->width = oldobj->width;
      (*newobj)->height = oldobj->height;
      (*newobj)->lowerleft.x = oldobj->lowerleft.x;
      (*newobj)->lowerleft.y = oldobj->lowerleft.y;
      (*newobj)->pcorner.x = oldobj->pcorner.x;
      (*newobj)->pcorner.y = oldobj->pcorner.y;
      (*newobj)->viewscale = oldobj->viewscale;
#ifdef SCHEMA
      /* don't attach the same schematic. . . */
      (*newobj)->symschem = NULL;

      /* Copy the parameter structure */
      (*newobj)->num_params = oldobj->num_params;
      (*newobj)->params = (oparamptr *)malloc(oldobj->num_params *
		sizeof(oparamptr));
      for (i = 0; i < oldobj->num_params; i++) {
	 (*newobj)->params[i] = (oparamptr) malloc(sizeof(oparam));
	 (*newobj)->params[i]->type = oldobj->params[i]->type;
	 switch (oldobj->params[i]->type) {
	    case XC_SHORT:
	       (*newobj)->params[i]->parameter.svalue =
			oldobj->params[i]->parameter.svalue;
	       break;
	    case XC_STRING:
	       (*newobj)->params[i]->parameter.string =
			stringcopy(oldobj->params[i]->parameter.string);
	       break;
	 }
      }

      (*newobj)->lleft2.x = oldobj->lleft2.x;
      (*newobj)->lleft2.y = oldobj->lleft2.y;
      (*newobj)->width2 = oldobj->width2;
      (*newobj)->height2 = oldobj->height2;

      (*newobj)->schemtype = oldobj->schemtype;
      (*newobj)->netlist = NULL;
      (*newobj)->portlist = NULL;
      (*newobj)->calllist = NULL;
      (*newobj)->traversed = 0;
      (*newobj)->devname = NULL;
#endif
      (*newobj)->hidden = False;

      /* copy over all the elements of the original object */

      (*newobj)->parts = 0;
      (*newobj)->plist = (genericptr *)malloc(sizeof(genericptr));

      for (i = 0; i < oldobj->parts; i++) {
	 switch((*(oldobj->plist + i))->type) {
	    case(PATH): {
	       register pathptr *npath, cpath = TOPATH(oldobj->plist + i);
	       register genericptr *gpart;
	       NEW_PATH(npath, (*newobj));
	       (*npath)->style = cpath->style;
	       (*npath)->width = cpath->width;
	       (*npath)->parts = 0;
	       (*npath)->plist = (genericptr *)malloc(cpath->parts *
			sizeof(genericptr));
	       for (gpart = cpath->plist; gpart < cpath->plist + cpath->parts;
			gpart++) {
		  switch((*gpart)->type){
		     case ARC: {
			arcptr copyarc = TOARC(gpart);
			arcptr *newarc;
			NEW_ARC(newarc, (*npath));
			arccopy(*newarc, copyarc);
			(*npath)->parts++;
			} break;
		     case POLYGON: {
                        polyptr copypoly = TOPOLY(gpart);
                        polyptr *newpoly;
                        NEW_POLY(newpoly, (*npath));
                        polycopy(*newpoly, copypoly);
                        (*npath)->parts++;
                        } break;
                     case SPLINE: {
                        splineptr copyspline = TOSPLINE(gpart);
                        splineptr *newspline;
                        NEW_SPLINE(newspline, (*npath));
                        splinecopy(*newspline, copyspline);
                        (*npath)->parts++;
                        } break;
		  }
	       }	
	       } break;
	    case(ARC): {
	       register arcptr *narc, carc = TOARC(oldobj->plist + i);

	       NEW_ARC(narc, (*newobj));
	       arccopy(*narc, carc);
               } break;
      
	    case(POLYGON): {
	       register polyptr *npoly, cpoly;

	       cpoly = TOPOLY(oldobj->plist + i);
	       NEW_POLY(npoly, (*newobj));
	       polycopy(*npoly, cpoly);
               } break;

	    case(SPLINE): {
	       register splineptr *nspl, cspl;

	       cspl = TOSPLINE(oldobj->plist + i);
	       NEW_SPLINE(nspl, (*newobj));
	       splinecopy(*nspl, cspl);
      	       } break;

	    case(LABEL): {
	       register labelptr *nlabel, clabel;

	       clabel = TOLABEL(oldobj->plist + i);
	       NEW_LABEL(nlabel, (*newobj));
	       (*nlabel)->rotation = clabel->rotation;
	       (*nlabel)->color = clabel->color;
	       (*nlabel)->position.x = clabel->position.x;
	       (*nlabel)->position.y = clabel->position.y;
	       (*nlabel)->scale = clabel->scale;
	       (*nlabel)->justify = clabel->justify;
	       (*nlabel)->string = stringcopy(clabel->string);
#ifdef SCHEMA
	       (*nlabel)->pin = clabel->pin;
#endif
               } break;
      
	    case(OBJECT): {
	       register objinstptr *ninst, cinst;

	       cinst = TOOBJINST(oldobj->plist + i);
	       NEW_OBJINST(ninst, (*newobj));
	       (*ninst)->rotation = cinst->rotation;
	       (*ninst)->color = cinst->color;
	       (*ninst)->position.x = cinst->position.x;
	       (*ninst)->position.y = cinst->position.y;
	       (*ninst)->scale = cinst->scale;
	       (*ninst)->thisobject = cinst->thisobject;

	       /* Library instance's parameter list is always default (null) */
	       (*ninst)->params = NULL;

#ifdef SCHEMA
	       /* here---deal with attached schematics */
#endif
               } break;
         }
	 (*newobj)->parts++;
      }
   }

   composelib(USERLIB);
   objectdeselect();

   if (objectdata == xobjs.libtop[USERLIB])
      drawarea(NULL, NULL, NULL);
   else startcatalog(NULL, USERLIB, NULL);
}

/*--------------------------------------------------------*/
/* ButtonPress handler during normal catalog viewing mode */
/*--------------------------------------------------------*/

void catbutton(u_int mode, XButtonEvent *event)
{
   short *newselect;
   objinstptr *newobject, *libobj;
   objectptr libpage = objectdata;
   short ocentx, ocenty, rangex, rangey, xdiff, ydiff, flag = 0;
   XPoint oldpos;

   /* Make catbutton() a wrapper for pagecatbutton() */

   if (is_library(objectdata) < 0) {
      pagecatbutton(event);
      return;
   }

   if (event->button == Button1 || event->button == Button2) {

      window_to_user(event->x, event->y, &areastruct.save);
	
      for (libobj = (objinstptr *)objectdata->plist; libobj <
		(objinstptr *)objectdata->plist + objectdata->parts; libobj++) {
	 if ((*libobj)->type == OBJECT) {

 	    ocentx = (*libobj)->position.x + (*libobj)->thisobject->_lowerleft.x
	        + ((*libobj)->thisobject->_width >> 1);
 	    ocenty = (*libobj)->position.y + (*libobj)->thisobject->_lowerleft.y
	        + ((*libobj)->thisobject->_height >> 1);

	    rangex = ((*libobj)->thisobject->_width > 200) ? 
	        ((*libobj)->thisobject->_width >> 1) : 100;
	    rangey = ((*libobj)->thisobject->_height > 200) ? 
	        ((*libobj)->thisobject->_height >> 1) : 100;

	    if (areastruct.save.x > ocentx - rangex && areastruct.save.x <
		   ocentx + rangex && areastruct.save.y < ocenty + rangey
		   && areastruct.save.y > ocenty - rangey) {

	       /* setup to move object around and draw the selected object */

#ifdef SCHEMA
	       if (eventmode == ASSOC_MODE) {

	          /* revert to old page */
	          objectdata = (pushlist == NULL) ?
			   xobjs.pagelist[areastruct.page]->pageobj
			   : pushlist->thisobject;
		  /* associate the new object */
		  schemassoc(objectdata, (*libobj)->thisobject); 
   	          setpage();
		  catreturn();
		  eventmode = PRESS_MODE;
	       }
	       else
#endif
               if (event->button == Button1) {

	          /* revert to old page */

	          objectdata = (pushlist == NULL) ?
			   xobjs.pagelist[areastruct.page]->pageobj
			   : pushlist->thisobject;

	          /* retrieve drawing window state and set position of object to
	             be correct in reference to that window */

	          snap(event->x, event->y, &oldpos);

   	          setpage();
	    
	          snap(event->x, event->y, &areastruct.save);
	          xdiff = areastruct.save.x - oldpos.x;
	          ydiff = areastruct.save.y - oldpos.y;

                  /* collect all of the selected items */

	          for (newselect = areastruct.selectlist; newselect <
		     areastruct.selectlist + areastruct.selects; newselect++) {
		     NEW_OBJINST(newobject, objectdata);
		     objectdata->parts++;
		     (*newobject)->color = areastruct.color;
		     (*newobject)->rotation = TOOBJINST(libpage->plist +
			    *newselect)->rotation;
	             (*newobject)->scale = TOOBJINST(libpage->plist +
		            *newselect)->scale;
	             (*newobject)->thisobject = TOOBJINST(libpage->plist +
		            *newselect)->thisobject;
		     (*newobject)->position.x = TOOBJINST(libpage->plist +
		            *newselect)->position.x + xdiff;
		     (*newobject)->position.y = TOOBJINST(libpage->plist +
		            *newselect)->position.y + ydiff;
		     (*newobject)->params = NULL;
	             u2u_snap(&((*newobject)->position));
		     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);
		     if ((*newobject)->thisobject == (*libobj)->thisobject)
		        flag = 1;
	          }

                  /* add final object to the list of object instances */

	          if (!flag) {
		     NEW_OBJINST(newobject, objectdata);
                     objectdata->parts++;
	             (*newobject)->rotation = (*libobj)->rotation;
		     (*newobject)->color = areastruct.color;
	             (*newobject)->scale = (*libobj)->scale;
	             (*newobject)->thisobject = (*libobj)->thisobject;
                     (*newobject)->position.x = areastruct.save.x;
	             (*newobject)->position.y = areastruct.save.y; 
		     (*newobject)->params = NULL;

	             /* add this object to the list of selected items */

	             newselect = allocselect();
                     *newselect = (short)(newobject - (objinstptr *)objectdata->plist);

	          }
	          if ((void *)mode == (void *)(1)) {

		     /* key "c" pressed for "copy" */

                     XDefineCursor(dpy, win, COPYCURSOR);
                     eventmode = COPY2_MODE;
                     XtAddEventHandler(areastruct.area, PointerMotionMask, False,
			  (XtEventHandler)drag, NULL);
	          }
	          else {
                     eventmode = PRESS_MODE;
                     XtAddEventHandler(areastruct.area, Button1MotionMask, False,
		          (XtEventHandler)drag, NULL);
	          }
                  catreturn();
               }

               /* If button2 was pressed, select and stay in the catalog. */
	       /* Could just do "objectselect" here, but would not cover   */
	       /* the entire area in the directory surrounding the object. */

	       else {
		  short newinst = (short)(libobj - (objinstptr *)objectdata->plist);
		  /* (ignore this object if it is already in the list of selects) */
		  for (newselect = areastruct.selectlist; newselect <
			areastruct.selectlist + areastruct.selects; newselect++)
		     if (*newselect == newinst) break;
		  if (newselect == areastruct.selectlist + areastruct.selects) {
		     newselect = allocselect();
		     *newselect = newinst;
		     XcSetFunction(GXcopy);
		     XcSetForeground(SELECTCOLOR);
		     UDrawObject(*libobj, (*libobj)->thisobject, SINGLE, SELECTCOLOR);
		  }
	       }
	       break;
	    }
	 }
      }
   }
   /* If button3 was pressed, return without a selection. */

   else {
      if (areastruct.selects > 0) {
	 free(areastruct.selectlist);
         areastruct.selects = 0;
      }
      eventmode = NORMAL_MODE;
      catreturn();
   }
}

/*------------------------------*/
/* Switch to the next catalog	*/
/*------------------------------*/

void changecat()
{
   int i, j;

   if ((i = is_library(objectdata)) < 0) {
      j = 0;	/* default: go to first library page */
      if (xobjs.userlibs[j].number == 0)
	 return;  /* unless it's not there */
      else
	 eventmode = CATALOG_MODE;
   }
   else {
      j = (i + 1) % xobjs.numlibs;
      if (j == i) {
	 Wprintf("This is the only library.");
	 return;
      }
   }
   startcatalog(NULL, j + LIBRARY, NULL);
}

/*--------------------------------------*/
/* Begin catalog viewing mode		*/
/*--------------------------------------*/

void startcatalog(Widget w, u_int libmod, caddr_t nulldata)
{
   if ((xobjs.libtop[libmod] == NULL) || (objectdata == xobjs.libtop[libmod])) return;

   if (libmod == FONTLIB) {
      XDefineCursor (dpy, win, CROSS);
      if (eventmode == TEXT2_MODE)
         eventmode = FONTCAT_MODE;
      else
         eventmode = FONTCAT2_MODE;
   }
#ifdef SCHEMA
   else if (eventmode == ASSOC_MODE) {
      XDefineCursor (dpy, win, CROSS);
   }
#endif
   else if (libmod == PAGELIB || libmod == LIBLIB) {
      XDefineCursor (dpy, win, CROSS);
      eventmode = CATALOG_MODE;
   }
   else
      eventmode = CATALOG_MODE;

   /* Push the current page onto the push stack, unless	we're going */
   /* to a library from the library directory or vice versa, or	    */
   /* library to library.					    */


   if (!(((is_library(objectdata) >= 0) || (objectdata == xobjs.libtop[LIBLIB])
		|| (objectdata == xobjs.libtop[PAGELIB])) && libmod >= PAGELIB)) {
      objectpair *newpush;
      newpush = (objectpair *) malloc(sizeof(objectpair));
      newpush->nextpair = pushlist;
      pushlist = newpush;
      newpush->thisobject = objectdata;
      newpush->thisinst = NULL;
   }

   /* set library as new object */

   objectdata = xobjs.libtop[libmod];
   setpage();
   if (libmod == FONTLIB)
      zoomview(NULL, NULL, NULL);

   /* unselect all current selects */

   if (areastruct.selects > 0) {
      free(areastruct.selectlist);
      areastruct.selects = 0;
   }

   /* draw the new screen */

   newmatrix();
   refresh(NULL, NULL, NULL);
}

#undef _lowerleft
#undef _width
#undef _height

/*-------------------------------------------------------------------------*/
