/*-------------------------------------------------------------------------*/
/* schema.c --- xcircuit routines specific to the schematic capture system */
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/13/97    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(DARWIN)
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"
#include "Xw/MenuBtn.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "xcircuit.h"
#include "colordefs.h"
#include "menudep.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Globaldata xobjs;
extern Clientdata areastruct;
extern short	  eventmode;
extern int	  *appcolors;
extern Widget     menuwidgets[];
extern Widget	  netbutton;
extern objectpair *pushlist;
extern char	  _STR[150];
extern char	  _STR2[250];

/*-------------------------------------------------------------------------*/
#ifdef SCHEMA

extern Widget wsymb, wschema;

/*--------------------------------------------------------*/
/* Menu calls (procedure wrappers)			  */
/*--------------------------------------------------------*/

void callgennet(Widget w, u_int mode, caddr_t calldata)
{
   switch(mode) {
     case 0:
        gennet("spice", "spc");
	break;
     case 1:
        gennet("sim", "sim");
	break;
     case 2:
        gennet("pcb", "pcb");
	break;
     case 3:
        gennet("flatspice", "fspc");
	break;
     case 4:
	gennet("idxpcb", "");	/* Auto-numbering:  no output generated */
	break;
   }
}

/*------------------------------------------------------------*/
/* Enable schematic capture (set schemon and realize widgets) */
/*------------------------------------------------------------*/

void doxschema(Widget w, caddr_t clientdata, caddr_t calldata)
{
   Arg wargs[1];

   if (areastruct.schemon) {
      areastruct.schemon = False;
      XtUnmanageChild(wschema);
      XtUnmanageChild(wsymb);
      XtUnmanageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Enable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }
   else {
      areastruct.schemon = True;
      XtManageChild(wschema);
      XtManageChild(wsymb);
      XtManageChild(netbutton);

      XtSetArg(wargs[0], XtNlabel, "Disable XSchema");
      XtSetValues(OptionsEnableXSchemaButton, wargs, 1);
   }

   /* Redraw the screen, since this action may effect the presence of	*/
   /* pins and schematic informational labels.				*/

   drawarea(NULL, NULL, NULL);
}

/*--------------------------------------*/
/* Find connectivity of a selection	*/
/*--------------------------------------*/

void startconnect(Widget button, caddr_t clientdata, caddr_t calldata)
{
   if (eventmode == NORMAL_MODE) {
      Wprintf("Click button1 on wire to see connectivity.");
      eventmode = CONNECT_MODE;
   }
}

/*----------------------------------------------------------------------*/
/* rconnectivity():  Find electrical connections into a node.		*/
/* (does recursive search on object heirarchy)				*/
/*----------------------------------------------------------------------*/

void connectivity(Widget button, caddr_t clientdata, caddr_t calldata)
{
   genericptr ggen;
   int depth, netid;
   objectpair *seltop, *nextptr;
   objectptr nettop;
   char *snew;
   stringpart *ppin;

   seltop = (objectpair *)malloc(sizeof(objectpair));
   seltop->thisobject = objectdata;
   seltop->thisinst = NULL;
   seltop->nextpair = NULL;

   /* Determine the netid and the topmost object in which that	*/
   /* net appears.  Then build the transformation matrix down	*/
   /* to that object.  highlightnet() should end by popping the */
   /* entire matrix stack.					*/

   if ((ggen = recurselect(&seltop)) != NULL) {
      createnets(True, NULL);
      if ((netid = is_resolved(&ggen, seltop, &nettop)) != 0) {
         /* printf("Net ID is %d in object %s\n", netid, nettop->name); */
         depth = pushnetwork(seltop, nettop);
         /* printf(">> Pushed network %d levels deep\n", i); */
         nextptr = seltop;
         while (nextptr->thisobject != nettop)
	    nextptr = nextptr->nextpair;
         highlightnet(nettop, nextptr->thisinst, netid);

         /* pop the matrix stack */
         while (depth-- > 0) 
	    UPopCTM();

	 /* print the net name to the message window */

	 ppin = nettopin(netid, nettop, NULL);
         snew = textprint(ppin, NORMINST);
	 sprintf(_STR, "Network is \"%s\" in %s", snew, nettop->name);
	 Wprintf(_STR);
         free(snew);
      }
      else
	 Wprintf("Selected element is not part of a valid network.");
      destroynets();
   }
   else 
      Wprintf("No networks found near the cursor position");

   /* free up linked list */

   while (seltop != NULL) {
      nextptr = seltop->nextpair;
      free(seltop);
      seltop = nextptr;
   }
}

/*----------------------------------------------------------------------*/
/* connectivity():  Find electrical connections into a node.		*/
/* (simple search only on topmost object)				*/
/*----------------------------------------------------------------------*/

void tconnectivity(Widget button, caddr_t clientdata, caddr_t calldata)
{
   genericptr *ggen;
   int netid;
   short *csel;
   objectpair *seltop;
   objectptr retobj;

   seltop = (objectpair *)malloc(sizeof(objectpair));
   seltop->thisobject = objectdata;
   seltop->thisinst = NULL;
   seltop->nextpair = NULL;

   objectselect(POLYGON | LABEL);
   drawselects();
   if (areastruct.selects == 0) {
      Wprintf("Must select at least one polygon or label to find network connections.");
   }
   else {
      createnets(True, NULL);
      for (csel = areastruct.selectlist; csel < areastruct.selectlist +
		areastruct.selects; csel++) {
         ggen = objectdata->plist + (*csel);
         netid = is_resolved(ggen, seltop, &retobj);
         printf(">> Resolving connectivity for net %d\n", netid);
	 if (retobj != objectdata)
	    printf(">> Error:  Resolved object is not topmost object!\n");
         if (netid != 0)
            highlightnet(retobj, areastruct.topobject, netid);
         else
            Wprintf("No network is associated with this polygon");
      }
      destroynets();
   }
   clearselects();

   free(seltop);
}

/*--------------------------------------------------------------*/
/* Set object type to FUNDAMENTAL if it contains one or more	*/
/* info labels for the given netlist mode.			*/
/*								*/
/* Return TRUE if the object has eletrically relevant networks,	*/
/* FALSE if not.						*/
/*--------------------------------------------------------------*/

Boolean setobjecttype(objectptr cschem, char *mode)
{
   genericptr *cgen;
   labelptr clab;

   /* Apply only to schematic objects */
   if (cschem->schemtype != SCHEMATIC && cschem->symschem == NULL) {

      if (cschem->schemtype == FUNDAMENTAL) cschem->schemtype = SYMBOL;
      for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
         if ((*cgen)->type == LABEL) {
            clab = TOLABEL(cgen);
	    if (clab->pin == INFO) {
	       if ((mode == NULL) || !textncomp(clab->string, mode, NULL)) {
	          cschem->schemtype = FUNDAMENTAL;
	          break;
	       }
	    }
         }
      }
   }

   if ((cschem->symschem != NULL) && (cschem->schemtype == SYMBOL))
      return False;
   else if ((cschem->schemtype == TRIVIAL) || (cschem->schemtype == FUNDAMENTAL))
      return False;

   return True;
}

/*------------------------------------------------------*/
/* Pin conversion subroutine for dopintype()		*/
/*------------------------------------------------------*/

void pinconvert(labelptr thislab, u_int mode)
{
   thislab->pin = mode;
   switch (mode) {
      case NORMAL:
	 thislab->color = DEFAULTCOLOR;		/* nominally black */
	 break;
      case GLOBAL:
	 thislab->color = BBOXCOLOR;		/* orange */
         break;
      case LOCAL:
	 thislab->color = SNAPCOLOR;		/* red */
         break;
      case INFO:
	 thislab->color = AUXCOLOR;		/* green */
         break;
   }
}

/*---------------------------------------------------------*/
/* Change a label's type to NORMAL, GLOBAL, INFO, or LOCAL */
/*---------------------------------------------------------*/

void dopintype(Widget w, u_int mode, caddr_t calldata)
{
   short *gsel;
   char typestr[40];
   short savetype = -1;

   if (areastruct.selects == 0) {
      Wprintf("Must first select a label to change type");
      return;
   }

   strcpy(typestr, "Changed label to ");
   switch(mode) {
      case NORMAL:
	 strcat(typestr, "normal label");
         break;
      case GLOBAL:
	 strcat(typestr, "global pin");
         break;
      case LOCAL:
	 strcat(typestr, "local pin");
         break;
      case INFO:
	 strcat(typestr, "info-label");
         break;
   }

   for (gsel = areastruct.selectlist; gsel < areastruct.selectlist +
		areastruct.selects; gsel++)
      if (SELECTTYPE(gsel) == LABEL) {
	 labelptr glab = SELTOLABEL(gsel);
	 savetype = glab->pin;
	 pinconvert(glab, mode);
	 setobjecttype(objectdata, NULL);
      }

   if (savetype >= 0) {
      objectdeselect();
      drawarea(NULL, NULL, NULL);
      Wprintf(typestr);
   }
   else {
      Wprintf("No labels selected.");
   }
}

/*----------------------------------------------------------*/
/* Set colors on the symbol/schematic buttons appropriately */
/*----------------------------------------------------------*/

void setsymschem()
{
   Arg aargs[2], bargs[2];

   /* Set menu items appropriately for this object */

   if (objectdata->symschem != NULL) {
      if (objectdata->schemtype == SCHEMATIC) {
         XtSetArg(aargs[0], XtNlabel, "Go To Symbol");
         XtSetArg(bargs[0], XtNlabel, "Disassociate Symbol");
      }
      else {
         XtSetArg(aargs[0], XtNlabel, "Go To Schematic");
         XtSetArg(bargs[0], XtNlabel, "Disassociate Schematic");
      }
   }
   else {
      if (objectdata->schemtype == SCHEMATIC) {
         XtSetArg(aargs[0], XtNlabel, "Make Matching Symbol");
         XtSetArg(bargs[0], XtNlabel, "Associate with Symbol");
      }
      else {
         XtSetArg(aargs[0], XtNlabel, "Make Matching Schematic");
         XtSetArg(bargs[0], XtNlabel, "Associate with Schematic");
      }
   }
   XtSetValues(NetlistMakeMatchingSymbolButton, aargs, 1);
   XtSetValues(NetlistAssociatewithSymbolButton, bargs, 1);

   /* Set colors on the symbol and schematic buttons */

   if (objectdata->schemtype == SCHEMATIC) {
      if (objectdata->symschem == NULL) {
         XtSetArg(aargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(aargs[1], XtNforeground, OFFBUTTONCOLOR);
      }
      else {
         XtSetArg(aargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(aargs[1], XtNforeground, FOREGROUND);
      }

      XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      XtSetArg(bargs[0], XtNbackground, SNAPCOLOR);
   }
   else {
      if (objectdata->symschem != NULL) {
         XtSetArg(bargs[0], XtNbackground, BACKGROUND);
	 XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      }
      else {
         XtSetArg(bargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(bargs[1], XtNforeground, OFFBUTTONCOLOR);
      }

      XtSetArg(aargs[1], XtNforeground, FOREGROUND);
      if (objectdata->schemtype == FUNDAMENTAL)
         XtSetArg(aargs[0], XtNbackground, AUXCOLOR);
      else if (objectdata->schemtype == TRIVIAL || objectdata->symschem != NULL)
         XtSetArg(aargs[0], XtNbackground, SNAPCOLOR);
      else {
         XtSetArg(aargs[0], XtNbackground, OFFBUTTONCOLOR);
	 XtSetArg(aargs[1], XtNforeground, OFFBUTTONCOLOR);
         XtSetArg(bargs[0], XtNbackground, BBOXCOLOR);
         XtSetArg(bargs[1], XtNforeground, FOREGROUND);
      }
   }

   XtSetValues(wsymb, aargs, 2);
   XtSetValues(wschema, bargs, 2);
}

/*--------------------------------------------------------*/
/* Find the page number for an object			  */
/*--------------------------------------------------------*/

int findpageobj(objectptr pobj)
{
   int tpage;

   for (tpage = 0; tpage < xobjs.pages; tpage++) {
      if (xobjs.pagelist[tpage]->pageobj == pobj) return tpage;
   }
   return -1;
}

/*--------------------------------------------------------*/
/* Recursively find all sub-circuits associated with the  */
/* top-level circuit and set their filenames to be the	  */
/* same.						  */
/*--------------------------------------------------------*/

void findsubschems(int toppage, objectptr cschem)
{
   genericptr *cgen;

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {
	 objectptr cobj = TOOBJINST(cgen)->thisobject;

	 if (cobj->symschem != NULL) {
	    int pageno = findpageobj(cobj->symschem);
	    /* printf("Symbol %s has schematic at page %d\n", cobj->name, pageno); */
	    if (pageno < 0) {	/* This shouldn't happen (hopefully) */
	       Wprintf("Error: disconnected schematic!\n");
	    }
	    else {
	       xobjs.pagelist[pageno]->filename = xobjs.pagelist[toppage]->filename;
	    }

	    /* A symbol on its own schematic page is allowed for clarity */
	    /* of the schematic, but this cannot be a functional part of */
	    /* the schematic circuit!					 */
	    
	    if (cobj->symschem != cschem) {
	       /* printf("Searching for dependencies of symbol %s\n", cobj->name); */
	       findsubschems(toppage, cobj->symschem);
	    }
	 }
	 else if (cobj->schemtype != FUNDAMENTAL && cobj->schemtype != TRIVIAL) {
	    /* Check symbols acting as their own schematics */
	    findsubschems(toppage, cobj);
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Copy label to corresponding schematic/symbol		  */
/*--------------------------------------------------------*/

void copypinlabel(labelptr pinlab)
{
   labelptr *newlabel, tlabel;
   genericptr *tgen;
   objectptr schemobj = objectdata->symschem;

   if (areastruct.schemon && schemobj != NULL && (pinlab->pin == LOCAL)) {

      /* If this pin already exists, don't copy it */

      for (tgen = schemobj->plist; tgen < schemobj->plist + schemobj->parts;
		tgen++) {
         if ((*tgen)->type == LABEL) {
	    tlabel = TOLABEL(tgen);
	    if (!stringcomp(tlabel->string, pinlab->string)) return;
         }
      }

      NEW_LABEL(newlabel, schemobj);
      (*newlabel)->pin = pinlab->pin;
      (*newlabel)->rotation = pinlab->rotation;
      (*newlabel)->justify = pinlab->justify;
      (*newlabel)->color = pinlab->color;
      (*newlabel)->scale = pinlab->scale;
      (*newlabel)->string = stringcopy(pinlab->string);

      /* place label just outside bounding box, then recompute bbox */

      (*newlabel)->position.x = schemobj->lowerleft.x + (schemobj->width >> 1);
      (*newlabel)->position.y = schemobj->lowerleft.y - TEXTHEIGHT
	   * (*newlabel)->scale;
      schemobj->parts++;
      singlebbox(objectdata, ENDPART);
   }
}

/*-------------------------------------------------------*/
/* Check if top-level page is the same name as a library */
/* object; if so, connect it.				 */
/* Note that it is not an error not to find the matching */
/* symbol/schematic.  "is_schematic" and "is_symbol"	 */
/* comments in the .ps file are complementary, so the	 */
/* first one encountered will always fail, and the other */
/* will succeed.					 */
/*-------------------------------------------------------*/

int checkschem(objectptr thisobj, char *cname)
{
   objectptr *tlib;
   short i, j;

   if (areastruct.schemon == False || thisobj->symschem != NULL) return 0;

   for (i = 0; i < xobjs.numlibs; i++) {
      for (j = 0; j < xobjs.userlibs[i].number; j++) {
	 tlib = xobjs.userlibs[i].library + j;

         if (!strcmp(cname, (*tlib)->name)) {
	    thisobj->symschem = (*tlib);
	    thisobj->schemtype = SCHEMATIC;
	    (*tlib)->symschem = thisobj;
	    (*tlib)->schemtype = SYMBOL;
	    return 1;
	 }
      }
   }
   return 0;
}

/*--------------------------------------------------------*/
/* Complement to the above routine:  If a library object  */
/* name is the same as a top-level page, connect them.	  */
/*--------------------------------------------------------*/

int checksym(objectptr symobj, char *cname)
{
   short cpage;
   objectptr checkpage;

   if (areastruct.schemon == False || symobj->symschem != NULL) return 0;

   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = xobjs.pagelist[cpage]->pageobj;
      if (checkpage != NULL) {
         if (!strcmp(checkpage->name, cname)) {
	    symobj->symschem = checkpage;
	    symobj->schemtype = SYMBOL;
	    checkpage->symschem = symobj;
	    checkpage->schemtype = SCHEMATIC;
	    return 1;
	 }
      }
   }
   return 0;
}

/*----------------------------------------------------------------------*/
/* Find location of corresponding pin in symbol/schematic and change it */
/* to the text of the indicated label.  If "newlabel" is NULL, then	*/
/* change the other label from pin to normal text.			*/
/*----------------------------------------------------------------------*/

void changeotherpins(labelptr newlabel, stringpart *oldstring)
{
   objectptr other = objectdata->symschem;
   genericptr *tgen;
   labelptr tlab;

   if (other == NULL) return;

   for (tgen = other->plist; tgen < other->plist + other->parts; tgen++) {
      if ((*tgen)->type == LABEL) {
	 tlab = TOLABEL(tgen);
	 if (!stringcomp(tlab->string, oldstring)) {
	    if (newlabel != NULL) {
	       free(tlab->string);
	       tlab->string = stringcopy(newlabel->string);
	    }
	    else
	       tlab->pin = NORMAL;
	 }
      }
   }
}

/*----------------------------------------------------------------------*/
/* Swap object schematic and symbol pages.   				*/
/*  mode = 0 disallows creation of a new schematic or symbol; i.e., if	*/
/*  there is no corresponding schematic/symbol, nothing happens.	*/
/*----------------------------------------------------------------------*/

void swapschem(Widget w, u_int mode, caddr_t calldata)
{
   objectptr savepage = objectdata;
   labelptr  *pinlab;
   genericptr *plab;
   Boolean lflag;

   if (areastruct.schemon == False) return;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);

   /* Create symbol or schematic, if allowed by mode */

   if ((objectdata->symschem == NULL) && (mode != 0)) {
      if (objectdata->schemtype == SYMBOL) {
	 int tpage;

	 /* create a new page for the new schematic */

	 for (tpage = 0; tpage < xobjs.pages; tpage++)
	    if (xobjs.pagelist[tpage]->pageobj == NULL) break;
         changepage(tpage);
      }
      else {
	 objectptr *newobject;
	 short libnum = USERLIB - LIBRARY;

	 /* create a new library object for the new symbol */

	 xobjs.userlibs[libnum].library = (objectptr *)
		realloc(xobjs.userlibs[libnum].library,
		++xobjs.userlibs[libnum].number * sizeof(objectptr));
	 newobject = xobjs.userlibs[libnum].library
		+ xobjs.userlibs[libnum].number - 1;
         *newobject = (objectptr) malloc(sizeof(object));
	 initmem(*newobject);
	 (*newobject)->schemtype = SYMBOL;
	 (*newobject)->hidden = False;
	 (*newobject)->devname = NULL;
	 objectdata = *newobject;

	 /* Generate the default bounding box for a size-zero object */
	 calcbbox(*newobject);
      }

      /* set links between the two objects */

      savepage->symschem = objectdata;
      objectdata->symschem = savepage;

      /* make the name of the new object equal to that of the old */

      strcpy(objectdata->name, savepage->name);

      /* copy all pin labels into the new object */

      for (plab = savepage->plist; plab < savepage->plist + savepage->parts;
		plab++) {
	 if ((*plab)->type == LABEL) {
	    genericptr *tgen;
	    labelptr tlab, lpin = (labelptr)*plab;

	    if (lpin->pin == LOCAL) {

      	       /* Only make one copy of each pin name */

	       lflag = False;
               for (tgen = objectdata->plist; tgen <
			objectdata->plist + objectdata->parts; tgen++) {
		  if ((*tgen)->type == LABEL) {
		     tlab = TOLABEL(tgen);
	             if (!stringcomp(tlab->string, lpin->string)) lflag = True;
		  }
      	       }
	       if (lflag == True) continue;

	       NEW_LABEL(pinlab, objectdata);
	       (*pinlab)->pin = lpin->pin;
	       (*pinlab)->color = lpin->color;
	       (*pinlab)->rotation = 1;
	       (*pinlab)->scale = 1.0;
	       (*pinlab)->justify = areastruct.justify; 
	       (*pinlab)->position.x = 0;
	       (*pinlab)->position.y = objectdata->parts * (TEXTHEIGHT + 10);
	       u2u_snap(&((*pinlab)->position));
	       (*pinlab)->string = stringcopy(lpin->string);
	       objectdata->parts++;
	    }
         }
	 calcbbox(objectdata);
      }

      /* Recreate the user library with the new symbol */
      if (savepage->schemtype != SYMBOL) composelib(USERLIB);
   }
   else if (objectdata->symschem != NULL) {
      objectdata = objectdata->symschem;
   }

   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;

   setpage();
   transferselects();
   refresh(NULL, NULL, NULL);

   setsymschem();
}

/*----------------------------------------------------------------------*/
/* Wrapper for swapschem() when generating a new symbol.		*/
/*----------------------------------------------------------------------*/

void makesymbol(Widget w, caddr_t calldata)
{
   /* copy name from popup prompt buffer and check */

   strcpy(_STR, _STR2);
   checkname(objectdata);
   swapschem(w, (u_int)1, calldata);
}

/*----------------------------------------------------------------------*/
/* Check name before doing a swap:  If name begins with "Page", prompt	*/
/* for the object name, as you would when doing selectsave().		*/ 
/*----------------------------------------------------------------------*/

void dobeforeswap(Widget w, caddr_t clientdata, caddr_t calldata)
{
   buttonsave *popdata = (buttonsave *)malloc(sizeof(buttonsave));

   /* Check if this function is valid */

   if (areastruct.schemon == False) return;

   /* Check for requirement to change the name before creating the symbol */

   if ((objectdata->symschem == NULL) && (objectdata->schemtype == SCHEMATIC)
		&& (strstr(objectdata->name, "Page ") != NULL)) {

      /* Get a name for the new object */

      eventmode = NORMAL_MODE;
      popdata->dataptr = NULL;
      popdata->button = NULL; /* indicates that no button is assc'd w/ the popup */
      popupprompt(w, "Enter name for new object:", "\0", makesymbol, popdata, False);
   }
   else
      swapschem(w, (u_int)1, calldata);
}

/*------------------------------------------*/
/* Disassociate a symbol from its schematic */
/*------------------------------------------*/

void schemdisassoc()
{
   if (eventmode != NORMAL) {
      Wprintf("Cannot disassociate schematics in this mode");
   }
   else {
      objectdata->symschem->symschem = NULL;
      objectdata->symschem = NULL;
      setsymschem();
      Wprintf("Schematic and symbol are now unlinked.");
   }
}

/*--------------------------------------------------------------*/
/* Schematic<-->symbol association.  Determine action from 	*/
/* context (associate/disassociate, symbol or schematic)	*/
/*--------------------------------------------------------------*/

void startschemassoc(Widget w, caddr_t clientdata, caddr_t calldata)
{
   if (objectdata->symschem != NULL)
      schemdisassoc();
   else {
      eventmode = ASSOC_MODE;
      if (objectdata->schemtype == SCHEMATIC) {
	 /* Find a symbol to associate */
	 startcatalog(w, LIBLIB, NULL);
	 Wprintf("Click on library page, then symbol to associate.");
      }
      else {
	 /* Find a schematic (page) to associate */
	 startcatalog(w, PAGELIB, NULL);
	 Wprintf("Click on schematic page to associate.");
      }
   }
}

/*--------------------------------------------------------------*/
/* Callback procedures on the schematic/symbol buttons.		*/
/*--------------------------------------------------------------*/

void schemassoc(objectptr schemobj, objectptr symbolobj)
{
   if (eventmode != ASSOC_MODE) {
      Wprintf("Cannot make (dis)association in this mode");
   }
   else if (schemobj->symschem != NULL || symbolobj->symschem != NULL) {
      Wprintf("Both objects must be disassociated first.");
   }
   else {
      schemobj->symschem = symbolobj;
      symbolobj->symschem = schemobj;
      if (symbolobj->schemtype == TRIVIAL)
	 symbolobj->schemtype = SYMBOL;

      /* Schematic takes the name of its associated symbol, by default */
      strcpy(schemobj->name, symbolobj->name);
   
      setsymschem();	/* Set buttons and menu items appropriately */
   }
}

#endif
/*-------------------------------------------------------------------------*/
