/*----------------------------------------------------------------------*/
/* parameter.c								*/
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	*/
/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/26/99    				*/
/*	revised for segmented strings, 3/8/01				*/
/*----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if !defined(__DARWIN__)
#include <malloc.h>
#endif

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*----------------------------------------------------------------------*/
/* Local includes							*/
/*----------------------------------------------------------------------*/

#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*----------------------------------------------------------------------*/
/* Externally declared global variables					*/
/*----------------------------------------------------------------------*/

extern Globaldata xobjs;
extern Clientdata areastruct;
extern objectpair *pushlist;
extern short textpos, textend;
extern char _STR[150];

/*------------------------------------------------------*/
/* If the instance comes from the library, replace the	*/
/* default value with the instance value.		*/
/*------------------------------------------------------*/

void replaceparams(objinstptr thisinst)
{
   objectptr thisobj;
   oparamptr ops, ips;
   int i, nullparms = 0;

   thisobj = thisinst->thisobject;

   for (i = 0; i < thisobj->num_params; i++) {
      ops = thisobj->params[i];
      ips = thisinst->params[i];
      if (ips == NULL) {
	 nullparms++;
	 continue;  /* this parameter is already default */
      }

      switch(ops->type) {
	 case XC_STRING:
	    if (stringcomp(ops->parameter.string, ips->parameter.string)) {
	       freelabel(ops->parameter.string);
	       ops->parameter.string = ips->parameter.string;
	       free(ips);
	       thisinst->params[i] = NULL;
	       nullparms++;
	    }
	    break;
	 case XC_SHORT:
	    if (ops->parameter.svalue != ips->parameter.svalue) {
	       free(ops);
	       ops = ips;
	       thisinst->params[i] = NULL;
	       nullparms++;
	    }
	    break;
      }
   }

   /* Object instance takes all default values---remove the params list */

   if (nullparms == thisobj->num_params) {
      free(thisinst->params);
      thisinst->params = NULL;
   }
}

/*------------------------------------------------------*/
/* Resolve differences between the object instance	*/
/* parameters and the default parameters.  If they	*/
/* are the same for any parameter, delete that instance	*/
/* such that the instance reverts to the default value.	*/
/*------------------------------------------------------*/

void resolveparams(objinstptr thisinst)
{
   objectptr thisobj;
   oparamptr ops, ips;
   int i, nullparms = 0;

   /* If the instance has no parameters itself, ignore it. */
   if (thisinst == NULL || thisinst->params == NULL) return;

   if (checklibtop() != NULL) {
      /* printf("Came from library or top page:  changing default value\n"); */
      replaceparams(thisinst);
      return;
   }

   thisobj = thisinst->thisobject;

   for (i = 0; i < thisobj->num_params; i++) {
      ops = thisobj->params[i];
      ips = thisinst->params[i];
      if (ips == NULL) {
	 nullparms++;
	 continue;  /* this parameter is already default */
      }

      switch(ops->type) {
	 case XC_STRING:
	    if (!stringcomp(ops->parameter.string, ips->parameter.string)) {
	       freelabel(ips->parameter.string);
	       free(ips);
	       thisinst->params[i] = NULL;
	       nullparms++;
	    }
	    break;
	 case XC_SHORT:
	    if (ops->parameter.svalue == ips->parameter.svalue) {
	       free(ips);
	       thisinst->params[i] = NULL;
	       nullparms++;
	    }
	    break;
      }
   }

   /* Object instance takes all default values---remove the params list */

   if (nullparms == thisobj->num_params) {
      free(thisinst->params);
      thisinst->params = NULL;
   }
}

/*------------------------------------------------------*/
/* Fill any NULL instance parameters with the values	*/
/* from the calling instance, or from the instance	*/
/* object's defaults if newinst = callinst.		*/ 
/*------------------------------------------------------*/

void copyparams(objinstptr newinst, objinstptr callinst)
{
   objectptr callobj;
   oparamptr *cparams;
   int i;

   if (callinst == NULL) return;

   callobj = callinst->thisobject;
   cparams = callinst->params;

   if (callobj->num_params == 0) return;

   if (newinst->params == NULL) {
      newinst->params = (oparamptr *) malloc(callobj->num_params * sizeof(oparamptr));
      for (i = 0; i < callobj->num_params; i++)
	 newinst->params[i] = NULL;
   }

   for (i = 0; i < callobj->num_params; i++) {
      if (newinst->params[i] == NULL) {
         newinst->params[i] = (oparamptr) malloc(sizeof(oparam));
         newinst->params[i]->type = callobj->params[i]->type;
         switch(callobj->params[i]->type) {
	    case XC_STRING:
	       if (newinst == callinst || callinst->params == NULL ||
			callinst->params[i] == NULL)
	          newinst->params[i]->parameter.string =
	    	     	   stringcopy(callobj->params[i]->parameter.string);
	       else
	          newinst->params[i]->parameter.string =
			   stringcopy(callinst->params[i]->parameter.string);
	       break;
	    case XC_SHORT:
	       if (newinst == callinst || callinst->params == NULL ||
			callinst->params[i] == NULL)
	          newinst->params[i]->parameter.svalue =
			   callobj->params[i]->parameter.svalue;
	       else
	          newinst->params[i]->parameter.svalue =
			   callinst->params[i]->parameter.svalue;
	       break;
	    default:
	       printf("Error:  bad parameter\n");
	       break;
	 }
      }
   }
}

/*--------------------------------------------------------------*/
/* make a short integer parameter				*/
/*--------------------------------------------------------------*/

void makeshortp(genericptr *gelem)
{
   oparamptr ops;

   /* cannot form a parameter on a top-level page */
   if (is_page(objectdata) != -1) {
      Wprintf("Cannot form a parameter in a top-level page!");
      return;
   }

   if (objectdata->params == NULL) {
      objectdata->num_params = 1;
      objectdata->params = (oparamptr *)malloc(sizeof(oparamptr));
   }
   else {
      objectdata->num_params++;
      objectdata->params = (oparamptr *)realloc(objectdata->params,
		objectdata->num_params * sizeof(oparamptr));
   }
   *(objectdata->params + objectdata->num_params - 1) = (oparamptr)
	malloc(sizeof(oparam));

   ops = *(objectdata->params + objectdata->num_params - 1);
   ops->type = XC_SHORT;

   switch((*gelem)->type) {	/* to be completed */
      case LABEL:
	 ops->parameter.svalue = TOLABEL(gelem)->position.x;
	 break;
      case ARC:
	 ops->parameter.svalue = TOARC(gelem)->position.x;
	 break;
      case OBJECT:
	 ops->parameter.svalue = TOOBJINST(gelem)->position.x;
	 break;
   }
}

/*--------------------------------------------------------------*/
/* remove a short integer parameter				*/
/*--------------------------------------------------------------*/

void unmakeshortp(genericptr *gelem)
{
}

/*--------------------------------------------------------------*/
/* Insert an existing parameter into a string.			*/
/*--------------------------------------------------------------*/

void insertparam()
{
   labelptr tlab;
   oparamptr ops;
   short i, j, plen, nparms = 0;
   int selparm = -1;

   /* Don't allow nested parameters */

   tlab = TOLABEL(EDITPART);
   if (paramcross(objectdata, tlab)) {
      Wprintf("Parameters cannot be nested!");
      return;
   }

   /* First pass is to check how many string parameters there are:	*/
   /* If only one, then automatically use it.  Otherwise, prompt	*/
   /* for which parameter to use.					*/

   for (i = 0; i < objectdata->num_params; i++) {
      ops = *(objectdata->params + i);
      if (ops->type == XC_STRING) {
	 nparms++;
	 selparm = i;
      }
   }
   if (nparms > 1) {
      char *newstr, *sptr;
      char *sstart = (char *)malloc(250);
      nparms = 0;
      strcpy(sstart, "Choose: ");
      sptr = sstart + 8;
      for (i = 0; i < objectdata->num_params; i++) {
	 ops = *(objectdata->params + i);
	 if (ops->type == XC_STRING) {
	    nparms++;
	    if (nparms != 1) {
	       strcat(sptr, ", ");
	       sptr += 2;
	    }
	    sprintf(sptr, "%d = <", nparms);
	    newstr = stringprint(ops->parameter.string, NULL);
	    strcat(sptr, newstr);
	    free(newstr);
	    newstr = NULL;
            sptr += strlen(sptr);
	 }
      }
      Wprintf(sstart);
      free(sstart);

      /* This is a bit quirky;  but works for now.  Later, would prefer */
      /* a GUI-oriented selection mechanism.				*/
      selparm = getkeynum();
   }

   if ((selparm >= 0) && (selparm < objectdata->num_params))
      labeltext(PARAM_START, (char *)&selparm);
   else
      Wprintf("Parameter number nonexistant.");
}

/*--------------------------------------------------------------*/
/* Parameterize a label string.					*/
/*--------------------------------------------------------------*/

void makeparam(labelptr thislabel)
{
   oparamptr ops;
   stringpart *begpart, *endpart, *strptr;

   /* cannot form a parameter on a top-level page */

   if (is_page(objectdata) != -1) {
      Wprintf("Cannot form a parameter in a top-level page!");
      return;
   }

   /* make sure this does not overlap another parameter */

   if (paramcross(objectdata, thislabel)) {
      Wprintf("Parameters cannot be nested!");
      textend = 0;
      return;
   }

   /* First, place PARAM_START and PARAM_END structures at the	*/
   /* intended parameter boundaries				*/

   if (textend > 0 && textend < textpos) {  	/* partial string */
      begpart = splitstring(textend, &thislabel->string, NULL);
      endpart = splitstring(textpos, &thislabel->string, NULL);

      begpart = makesegment(&thislabel->string, begpart->nextpart);
      endpart = makesegment(&thislabel->string, endpart->nextpart);
   }
   else {				/* full string */
      makesegment(&thislabel->string, thislabel->string);
      begpart = thislabel->string;
      endpart = makesegment(&thislabel->string, NULL);
   }
   begpart->type = PARAM_START;
   begpart->data.paramno = (objectdata->params == NULL) ? 0 :
		objectdata->num_params;
   endpart->type = PARAM_END;

   /* Now move the sections of string to the object parameter */

   if (objectdata->params == NULL) {
      objectdata->num_params = 1;
      objectdata->params = (oparamptr *)malloc(sizeof(oparamptr));
   }
   else {
      objectdata->num_params++;
      objectdata->params = (oparamptr *)realloc(objectdata->params,
		objectdata->num_params * sizeof(oparamptr));
   }
   *(objectdata->params + objectdata->num_params - 1) = (oparamptr)
	malloc(sizeof(oparam));

   ops = *(objectdata->params + objectdata->num_params - 1);
   ops->type = XC_STRING;
   ops->parameter.string = begpart->nextpart;
   begpart->nextpart = endpart->nextpart;
   endpart->nextpart = NULL;

   textend = 0;
}

/*--------------------------------------------------------------*/
/* Search and destroy the selected parameter in all instances 	*/
/* of the specified object.					*/
/*--------------------------------------------------------------*/

void searchinst(objectptr topobj, objectptr refobj, short pnum)
{
   oparamptr ops;
   objinstptr tinst;
   genericptr *pgen, *lgen;
   short k;

   if (topobj == NULL) return;

   for (pgen = topobj->plist; pgen < topobj->plist + topobj->parts; pgen++) {
      if ((*pgen)->type == OBJECT) {
	 tinst = TOOBJINST(pgen);
	 if (tinst->thisobject == refobj) {
	    if (tinst->thisobject->num_params > 0) {
	       if (tinst->params != NULL) {
	          ops = *(tinst->params + pnum);
	          if (ops != NULL)
		     freelabel(ops->parameter.string);
		  free(ops);
		  ops = NULL;
		  if (pnum == 0) {
		     free(tinst->params);
		     tinst->params = NULL;
		  }
		  else {
	             /* Re-order the remaining parameters */
	             for (k = pnum; k < tinst->thisobject->num_params - 1; k++)
	                *(tinst->params + k) = *(tinst->params + k + 1);
		  }
	       }
	    }
	 }
      }
   }
}

/*--------------------------------------------------------------*/
/* Check if this string contains a parameter			*/
/*--------------------------------------------------------------*/

stringpart *searchparam(stringpart *tstr)
{
   stringpart *rval = tstr;
   for (rval = tstr; rval != NULL; rval = rval->nextpart)
      if (rval->type == PARAM_START)
	 break;
   return rval;
}

/*--------------------------------------------------------------*/
/* Remove parameterization from a label string or substring.	*/
/*--------------------------------------------------------------*/

void unmakeparam(labelptr thislabel, stringpart *thispart)
{
   genericptr *pgen;
   labelptr plab;
   oparamptr ops;
   stringpart *strptr, *lastpart, *nextpart, *endpart;
   stringpart *subs, *newstr;
   int p, j, k;
   
   /* make sure there is a parameter here */

   if (thispart->type != PARAM_START) {
      Wprintf("There is no parameter here.");
      return;
   }
   p = thispart->data.paramno;

   /* Find all instances of this object and remove any parameter */
   /* substitutions which may have been made.			 */

   for (k = 0; k < xobjs.pages; k++) {
      searchinst(xobjs.pagelist[k]->pageobj, objectdata, p);
   }
   for (j = 0; j < xobjs.numlibs; j++) {
      for (k = 0; k < xobjs.userlibs[j].number; k++) {
         searchinst(*(xobjs.userlibs[j].library + k), objectdata, p);
      }
   }
   for (k = 0; k < xobjs.delbuffer.number; k++) {
      searchinst(*(xobjs.delbuffer.library + k), objectdata, p);
   }

   /* Separate the parameter string from the parameter list */
   
   ops = *(objectdata->params + p);
   subs = ops->parameter.string;
   for (lastpart = subs; lastpart->nextpart->type != PARAM_END;
		lastpart = lastpart->nextpart);
   free(lastpart->nextpart);
   lastpart->nextpart = NULL;
   free(ops);
   
   /* Re-order the remaining parameters */

   for (k = p; k < objectdata->num_params - 1; k++)
      *(objectdata->params + k) = *(objectdata->params + k + 1);

   /* Decrement the number of parameters */

   if (--objectdata->num_params == 0) {
      free(objectdata->params);
      objectdata->params = NULL;
   }

   /* Copy the default parameter wherever the parameter is called, and   */
   /* decrement the parameter ID number of all parameters in all strings */
   /* of this object whose parameter IDs are greater.			 */

   for (pgen = objectdata->plist; pgen < objectdata->plist + objectdata->parts;
		pgen++) {
      if ((*pgen)->type == LABEL) {
	 plab = TOLABEL(pgen);
	 lastpart = NULL;
	 strptr = plab->string;
	 while (strptr != NULL) {
	    nextpart = strptr->nextpart;
	    if (strptr->type == PARAM_START) {
	       if (strptr->data.paramno > p)
		  strptr->data.paramno--;
	       else if (strptr->data.paramno == p) {
		  newstr = stringcopy(subs);
		  for (endpart = newstr; endpart->nextpart != NULL;
				endpart = endpart->nextpart);
		  endpart->nextpart = nextpart;
		  if (lastpart == NULL)
		     plab->string = newstr;
		  else
		     lastpart->nextpart = newstr;
		  free(strptr);

      		  /* Merge strings at boundaries, if possible. */
		  mergestring(lastpart);
		  strptr = lastpart;
		  mergestring(endpart);
		  nextpart = strptr->nextpart;
	       }
	    }
	    lastpart = strptr;
	    strptr = nextpart;
	 }
      }
   }
   freelabel(subs);
}

/*------------------------------------------------------*/
/* Wrapper for unmakeparam()				*/
/*------------------------------------------------------*/

void unparameterize(u_int mode)
{
   short *fselect, i, ptype = (short)mode;
   int locpos;
   stringpart *strptr, *tmpptr;
   labelptr settext;

   if (!checkselect(ptype)) objectselect(ptype);
   if (!checkselect(ptype)) return;

   if ((areastruct.selects == 1) && (ptype & LABEL) && textend > 0
		&& textend < textpos) {
      if (SELECTTYPE(areastruct.selectlist) != LABEL) return;	 /* Not a label */
      settext = SELTOLABEL(areastruct.selectlist);
      strptr = findstringpart(textend, &locpos, settext->string, NORMINST);
      while (strptr != NULL && strptr->type != PARAM_END)
	 strptr = strptr->nextpart;
      if (strptr == NULL) return;	/* No parameters */
      tmpptr = settext->string;
      while (tmpptr != NULL && tmpptr->type != PARAM_START &&
	 	tmpptr->nextpart != strptr->nextpart)
	 tmpptr = tmpptr->nextpart;

      if (tmpptr != NULL) unmakeparam(settext, tmpptr);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if ((ptype & LABEL) && SELECTTYPE(fselect) == LABEL) {
            settext = SELTOLABEL(fselect);
            strptr = settext->string;
            while (strptr != NULL && strptr->type != PARAM_START)
	       strptr = strptr->nextpart;
	    if (strptr != NULL) unmakeparam(settext, strptr);
	 }
	 else
	    unmakeshortp(objectdata->plist + (*fselect));
      }
   }
}

/*--------------------------------------------------------------*/
/* Wrapper for makeparam()					*/
/*--------------------------------------------------------------*/

void parameterize(u_int mode)
{
   short *fselect, ptype = (short)mode;
   labelptr settext;

   if (!checkselect(ptype)) objectselect(ptype);
   if (!checkselect(ptype)) return;
   for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
      if ((ptype & LABEL) && areastruct.selects == 1 &&
		SELECTTYPE(fselect) == LABEL) {
         settext = SELTOLABEL(fselect);
         makeparam(settext);
      }
      else
	 makeshortp(objectdata->plist + (*fselect));
   }
   objectdeselect();
}

/*----------------------------------------------------------------------*/
/* Looks for a parameter overlapping the textend <--> textpos space.	*/
/* Returns True if there is a parameter in this space.			*/
/*----------------------------------------------------------------------*/

Boolean paramcross(objectptr tobj, labelptr tlab)
{
   oparamptr ops;
   stringpart *firstptr, *lastptr;
   int locpos;

   lastptr = findstringpart(textpos, &locpos, tlab->string, NORMINST);

   /* This text position can't be inside another parameter */
   for (firstptr = lastptr; firstptr != NULL; firstptr = firstptr->nextpart)
      if (firstptr->type == PARAM_END) return True;

   /* The area between textend and textpos cannot contain a parameter */
   if (textend > 0)
      for (firstptr = findstringpart(textend, &locpos, tlab->string, NORMINST);
	    firstptr != lastptr; firstptr = firstptr->nextpart)
         if (firstptr->type == PARAM_START) return True;

   return False;
}

/*----------------------------------------------------------------------*/
/* Check whether this page object was entered via a library page	*/
/*----------------------------------------------------------------------*/

objectptr checklibtop()
{
   objectptr pobj;
   objectpair *thispush = pushlist;

   while (thispush != NULL) {
      pobj = thispush->thisobject;
      if (is_library(pobj) >= 0) return pobj;
      thispush = thispush->nextpair;
   }
   return (objectptr)NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object	instance			*/
/* (Reverts all parameters to default value)				*/
/*----------------------------------------------------------------------*/

void removeinst(objinstptr thisinst)
{
   free(thisinst->params);
   thisinst->params = NULL;
}

/*----------------------------------------------------------------------*/
/* Remove all parameters from an object.				*/
/*----------------------------------------------------------------------*/

void removeparams(objectptr thisobj)
{
   int i, j;
}

/*-----------------------------------------------------------------------*/
