/*--------------------------------------------------------------*/
/* python.c --- Embedded python interpreter for xcircuit	*/
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University    */
/*--------------------------------------------------------------*/
/* NOTE: These routines work only if python-2.0 is installed.   */
/*	Requires library file libpython2.0.a.  Hopefully	*/
/*	one day there will be a standard libpython${VERSION}.so	*/
/*	so that the whole thing doesn't have to be linked	*/
/*	into the xcircuit executable. . .			*/
/*								*/
/*	Modeled after demo.c in Python2.0 source Demo/embed	*/
/*--------------------------------------------------------------*/

#ifdef HAVE_PYTHON

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if !defined(__DARWIN__)
#include <malloc.h>
#endif
   
#include <Python.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"
      
/*----------------------------------------------------------------------*/
/* Local includes                                                       */
/*----------------------------------------------------------------------*/
      
#include "xcircuit.h"
#include "menudep.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/

#include "prototypes.h"

static PyObject *xc_new(PyObject *, PyObject *);
static PyObject *xc_set(PyObject *, PyObject *);
static PyObject *xc_override(PyObject *, PyObject *);
static PyObject *xc_library(PyObject *, PyObject *);
static PyObject *xc_font(PyObject *, PyObject *);
static PyObject *xc_color(PyObject *, PyObject *);
static PyObject *xc_pause(PyObject *, PyObject *);
static PyObject *xc_refresh(PyObject *, PyObject *);
static PyObject *xc_bind(PyObject *, PyObject *);
static PyObject *xc_unbind(PyObject *, PyObject *);
void init_interpreter();
void exit_interpreter();

/*----------------------------------------------------------------------*/
/* External variable declarations                                       */
/*----------------------------------------------------------------------*/
      
extern char _STR2[250], _STR[150];
extern fontinfo *fonts;
extern short fontcount;
extern Clientdata areastruct;
extern Globaldata xobjs;
extern short beeper; 
extern Widget menuwidgets[];
extern Display *dpy;
extern Window win;
extern keybinding *keylist;
extern colorindex *colorlist;

/*----------------------------------------------------------------------*/

short flags;

#define LIBOVERRIDE     1
#define LIBLOADED       2
#define COLOROVERRIDE   4
#define FONTOVERRIDE    8
#define KEYOVERRIDE	16

/*--------------------------------------------------------------*/
/* Define all of the xcircuit functions available to Python	*/
/* Currently this does the bare minimum of having Python do the	*/
/* argument parsing; ideally, we would like to be doing more	*/
/* fundamental processing here.					*/
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
/* Create objects and return a handle to that object in the	*/
/* form of a Python integer Object.				*/
/*--------------------------------------------------------------*/

static PyObject *xc_new(PyObject *self, PyObject *args)
{
   genericptr *newgen;
   const char *etype;
   int type;

   if (!PyArg_ParseTuple(args, "s:newelement", &etype))
      return NULL;
   
   switch(type = string_to_type(etype)) {
      case ARC:
         NEW_ARC((arcptr *)newgen, objectdata);
	 arcdefaults(TOARC(newgen), 0, 0);
	 break;
      case SPLINE:
         NEW_SPLINE((splineptr *)newgen, objectdata);
	 splinedefaults(TOSPLINE(newgen), 0, 0);
	 break;
      case POLYGON:
         NEW_POLY((polyptr *)newgen, objectdata);
	 polydefaults(TOPOLY(newgen), 4, 0, 0);
	 break;
      case LABEL:
         NEW_LABEL((labelptr *)newgen, objectdata);
	 labeldefaults(TOLABEL(newgen), 0, 0, 0);
	 break;
      case PATH:
         NEW_PATH((pathptr *)newgen, objectdata);
	 pathdefaults(TOPATH(newgen), 0, 0);
	 break;
      case OBJECT:
         NEW_OBJINST((objinstptr *)newgen, objectdata);
	 objectdefaults(TOOBJINST(newgen), NULL, 0, 0);
	 break;
      default:
         PyErr_SetString(PyExc_TypeError,
		"newelement() 2nd argument must be a valid element type");
         return NULL;
   } 
   objectdata->parts++;
   return PyInt_FromLong((long)(*newgen));
}

/*--------------------------------------------------------------*/
/* Convert object type to a string				*/
/*--------------------------------------------------------------*/

char *type_to_string(int type)
{
   char *retstr = NULL;

   switch(type) {
      case LABEL:
	 retstr = malloc(6);
	 strcpy(retstr, "Label");
	 break;
      case POLYGON:
	 retstr = malloc(8);
	 strcpy(retstr, "Polygon");
	 break;
      case SPLINE:
	 retstr = malloc(12);
	 strcpy(retstr, "Bezier Curve");
	 break;
      case OBJECT:
	 retstr = malloc(16);
	 strcpy(retstr, "Object Instance");
	 break;
      case PATH:
	 retstr = malloc(5);
	 strcpy(retstr, "Path");
	 break;
      case ARC:
	 retstr = malloc(4);
	 strcpy(retstr, "Arc");
	 break;
   }
   return retstr;	 
}

/*--------------------------------------------------------------*/
/* Convert a string to an element type				*/
/*--------------------------------------------------------------*/

int string_to_type(const char *etype)
{
   if (!strcmp(etype, "Arc"))
      return ARC;
   else if (!strcmp(etype, "Bezier Curve"))
      return SPLINE;
   else if (!strcmp(etype, "Polygon"))
      return POLYGON;
   else if (!strcmp(etype, "Label"))
      return LABEL;
   else if (!strcmp(etype, "Path"))
      return PATH;
   else if (!strcmp(etype, "Object Instance"))
      return OBJECT;
   else
      return -1;
}

/*--------------------------------------------------------------*/
/* Check if the handle (integer) is an existing element		*/
/*--------------------------------------------------------------*/

genericptr *CheckHandle(PyObject *ehandle)
{
   genericptr *gelem;
   int eaddr;

   eaddr = (int)PyInt_AsLong(ehandle);
   for (gelem = objectdata->plist; gelem < objectdata->plist +
	objectdata->parts; gelem++)
      if ((int)(*gelem) == eaddr) goto exists;
   return NULL;

exists:
   return gelem;
}

/*--------------------------------------------------------------*/
/* Check if the handle (integer) is an existing page		*/
/*--------------------------------------------------------------*/

objectptr CheckPageHandle(PyObject *ehandle)
{
   genericptr *gelem;
   int eaddr, pageno;

   eaddr = (int)PyInt_AsLong(ehandle);
   pageno = is_page((objectptr)eaddr);
   if (pageno < 0) return NULL;
   return (objectptr)eaddr;
}

/*--------------------------------------------------------------*/
/* Get the properties of a page (returned as a dictionary)	*/
/*--------------------------------------------------------------*/

static PyObject *xc_getpage(PyObject *self, PyObject *args)
{
   objectptr pageobj;
   PyObject *rdict, *dtup, *dlist;
   int pageno = -1, i, eaddr;

   if (!PyArg_ParseTuple(args, "|d:getpage", &pageno))
      return NULL;

   if (pageno < 0) pageno = areastruct.page;
   else pageno--;
   if (pageno >= xobjs.pages) return NULL;
   pageobj = xobjs.pagelist[pageno]->pageobj;

   /* return all the elements properties as a dictionary	*/

   rdict = PyDict_New();

   PyDict_SetItem(rdict, PyString_FromString("name"),
	PyString_FromString(pageobj->name));

   dlist = PyList_New(pageobj->parts);
   for (i = 0; i < pageobj->parts; i++)
      PyList_SetItem(dlist, i, PyInt_FromLong((long)(*(pageobj->plist + i))));
   PyDict_SetItem(rdict, PyString_FromString("parts"), dlist);

   return rdict;
}

/*--------------------------------------------------------------*/
/* Get the properties of an element (returned as a dictionary)	*/
/*--------------------------------------------------------------*/

static PyObject *xc_getattr(PyObject *self, PyObject *args)
{
   genericptr *gelem;
   PyObject *ehandle, *rdict, *dtup, *dlist, *lstr, *sdict, *stup;
   int i, eaddr, llen;
   char *tstr;
   stringpart *strptr;
   

   if (!PyArg_ParseTuple(args, "O:getattr", &ehandle))
      return NULL;

   /* Check to make sure that handle exists! */
   if ((gelem = CheckHandle(ehandle)) == NULL) {
      PyErr_SetString(PyExc_TypeError,
		"Argument must be a valid handle to an element.");
      return NULL;
   }

   /* return the element's properties as a dictionary	*/

   rdict = PyDict_New();
   tstr = type_to_string((*gelem)->type);
   if (tstr == NULL) {
      PyErr_SetString(PyExc_TypeError,
		"Element type is unknown.");
      return NULL;
   }
   PyDict_SetItem(rdict, PyString_FromString("type"),
        PyString_FromString(tstr));
   free(tstr);
   PyDict_SetItem(rdict, PyString_FromString("color"), 
	PyInt_FromLong((long)((*gelem)->color)));

   switch((*gelem)->type) {
      case LABEL:
	 dtup = PyTuple_New(2);
	 PyTuple_SetItem(dtup, 0,
		PyInt_FromLong((long)TOLABEL(gelem)->position.x));
	 PyTuple_SetItem(dtup, 1,
		PyInt_FromLong((long)TOLABEL(gelem)->position.y));
	 PyDict_SetItem(rdict, PyString_FromString("position"), dtup);
	 PyDict_SetItem(rdict, PyString_FromString("rotation"),
		PyInt_FromLong((long)TOLABEL(gelem)->rotation));
	 PyDict_SetItem(rdict, PyString_FromString("scale"),
		PyFloat_FromDouble((double)TOLABEL(gelem)->scale));
	 PyDict_SetItem(rdict, PyString_FromString("justify"),
		PyInt_FromLong((long)TOLABEL(gelem)->justify));
#ifdef SCHEMA
	 PyDict_SetItem(rdict, PyString_FromString("pin"),
		PyInt_FromLong((long)TOLABEL(gelem)->pin));
#endif
	 llen = stringparts(TOLABEL(gelem)->string);
	 lstr = PyList_New(llen);
	 for (strptr = TOLABEL(gelem)->string, i = 0; strptr != NULL;
		strptr = strptr->nextpart, i++) {
	    switch(strptr->type) {
	       case TEXT_STRING:
		  PyList_SetItem(lstr, i, PyString_FromString(strptr->data.string));
		  break;
	       case PARAM_START:
		  sdict = PyDict_New();
		  PyDict_SetItem(sdict, PyString_FromString("Parameter"),
		  	PyInt_FromLong((long)strptr->data.paramno));
		  PyList_SetItem(lstr, i, sdict);
		  break;
	       case FONT_NAME:
		  sdict = PyDict_New();
		  PyDict_SetItem(sdict, PyString_FromString("Font"),
		  	PyString_FromString(fonts[strptr->data.paramno].psname));
		  PyList_SetItem(lstr, i, sdict);
		  break;
	       case KERN:
		  sdict = PyDict_New();
		  stup = PyTuple_New(2);
		  PyTuple_SetItem(stup, 0, PyInt_FromLong((long)strptr->data.kern[0]));
		  PyTuple_SetItem(stup, 1, PyInt_FromLong((long)strptr->data.kern[1]));
		  PyDict_SetItem(sdict, PyString_FromString("Kern"), stup);
		  PyList_SetItem(lstr, i, sdict);
		  break;
	       case FONT_COLOR:
		  sdict = PyDict_New();
		  stup = PyTuple_New(3);
		  PyTuple_SetItem(stup, 0, PyInt_FromLong(
			(long)colorlist[strptr->data.color].color.red));
		  PyTuple_SetItem(stup, 1, PyInt_FromLong(
			(long)colorlist[strptr->data.color].color.green));
		  PyTuple_SetItem(stup, 2, PyInt_FromLong(
			(long)colorlist[strptr->data.color].color.blue));
		  PyDict_SetItem(sdict, PyString_FromString("Color"), stup); 
		  PyList_SetItem(lstr, i, sdict);
		  break;
	       case TABSTOP:
		  PyList_SetItem(lstr, i, PyString_FromString("Tab Stop"));
		  break;
	       case TABFORWARD:
		  PyList_SetItem(lstr, i, PyString_FromString("Tab Forward"));
		  break;
	       case TABBACKWARD:
		  PyList_SetItem(lstr, i, PyString_FromString("Tab Backward"));
		  break;
	       case RETURN:
		  PyList_SetItem(lstr, i, PyString_FromString("Return"));
		  break;
	       case SUBSCRIPT:
		  PyList_SetItem(lstr, i, PyString_FromString("Subscript"));
		  break;
	       case SUPERSCRIPT:
		  PyList_SetItem(lstr, i, PyString_FromString("Superscript"));
		  break;
	       case NORMALSCRIPT:
		  PyList_SetItem(lstr, i, PyString_FromString("Normalscript"));
		  break;
	       case UNDERLINE:
		  PyList_SetItem(lstr, i, PyString_FromString("Underline"));
		  break;
	       case OVERLINE:
		  PyList_SetItem(lstr, i, PyString_FromString("Overline"));
		  break;
	       case NOLINE:
		  PyList_SetItem(lstr, i, PyString_FromString("No Line"));
		  break;
	       case HALFSPACE:
		  PyList_SetItem(lstr, i, PyString_FromString("Half Space"));
		  break;
	       case QTRSPACE:
		  PyList_SetItem(lstr, i, PyString_FromString("Quarter Space"));
		  break;
	    }
	 }
	 PyDict_SetItem(rdict, PyString_FromString("string"), lstr);
	 break;
      case POLYGON:
	 PyDict_SetItem(rdict, PyString_FromString("style"), 
		PyInt_FromLong((long)TOPOLY(gelem)->style));
	 PyDict_SetItem(rdict, PyString_FromString("linewidth"),
		PyFloat_FromDouble((double)TOPOLY(gelem)->width));
	 dlist = PyList_New(TOPOLY(gelem)->number);
	 for (i = 0; i < TOPOLY(gelem)->number; i++) {
	    dtup = PyTuple_New(2);
	    PyTuple_SetItem(dtup, 0,
		   PyInt_FromLong((long)TOPOLY(gelem)->points[i].x));
	    PyTuple_SetItem(dtup, 1,
		   PyInt_FromLong((long)TOPOLY(gelem)->points[i].y));
	    PyList_SetItem(dlist, i, dtup);
	 }
	 PyDict_SetItem(rdict, PyString_FromString("points"), dlist);
	 break;
      case ARC:
	 PyDict_SetItem(rdict, PyString_FromString("style"), 
		PyInt_FromLong((long)TOARC(gelem)->style));
	 PyDict_SetItem(rdict, PyString_FromString("linewidth"),
		PyFloat_FromDouble((double)TOARC(gelem)->width));
	 PyDict_SetItem(rdict, PyString_FromString("radius"), 
		PyInt_FromLong((long)TOARC(gelem)->radius));
	 PyDict_SetItem(rdict, PyString_FromString("minor axis"), 
		PyInt_FromLong((long)TOARC(gelem)->yaxis));
	 PyDict_SetItem(rdict, PyString_FromString("start angle"),
		PyFloat_FromDouble((double)TOARC(gelem)->angle1));
	 PyDict_SetItem(rdict, PyString_FromString("end angle"),
		PyFloat_FromDouble((double)TOARC(gelem)->angle2));
	 dtup = PyTuple_New(2);
	 PyTuple_SetItem(dtup, 0,
		PyInt_FromLong((long)TOARC(gelem)->position.x));
	 PyTuple_SetItem(dtup, 1,
		PyInt_FromLong((long)TOARC(gelem)->position.y));
	 PyDict_SetItem(rdict, PyString_FromString("position"), dtup);
	 break;
      case SPLINE:
	 PyDict_SetItem(rdict, PyString_FromString("style"), 
		PyInt_FromLong((long)TOSPLINE(gelem)->style));
	 PyDict_SetItem(rdict, PyString_FromString("linewidth"),
		PyFloat_FromDouble((double)TOSPLINE(gelem)->width));
	 dlist = PyList_New(4);
	 for (i = 0; i < 4; i++) {
	    dtup = PyTuple_New(2);
	    PyTuple_SetItem(dtup, 0,
		   PyInt_FromLong((long)TOSPLINE(gelem)->ctrl[i].x));
	    PyTuple_SetItem(dtup, 1,
		   PyInt_FromLong((long)TOSPLINE(gelem)->ctrl[i].y));
	    PyList_SetItem(dlist, i, dtup);
	 }
	 PyDict_SetItem(rdict, PyString_FromString("control points"), dlist);
	 break;
      case PATH:
	 PyDict_SetItem(rdict, PyString_FromString("style"), 
		PyInt_FromLong((long)TOPATH(gelem)->style));
	 PyDict_SetItem(rdict, PyString_FromString("linewidth"),
		PyFloat_FromDouble((double)TOPATH(gelem)->width));
	 dlist = PyList_New(TOPATH(gelem)->parts);
	 for (i = 0; i < TOPATH(gelem)->parts; i++) {
	    PyList_SetItem(dlist, i,
		PyInt_FromLong((long)(*(TOPATH(gelem)->plist + i))));
	 }
	 PyDict_SetItem(rdict, PyString_FromString("parts"), dlist);
	 break;
      case OBJECT:
	 dtup = PyTuple_New(2);
	 PyTuple_SetItem(dtup, 0,
		PyInt_FromLong((long)TOOBJINST(gelem)->position.x));
	 PyTuple_SetItem(dtup, 1,
		PyInt_FromLong((long)TOOBJINST(gelem)->position.y));
	 PyDict_SetItem(rdict, PyString_FromString("rotation"),
		PyInt_FromLong((long)TOLABEL(gelem)->rotation));
	 PyDict_SetItem(rdict, PyString_FromString("scale"),
		PyFloat_FromDouble((double)TOLABEL(gelem)->scale));
	 break;
   }
   return rdict;
}

/*--------------------------------------------------------------*/
/* Set properties of an element (supplied as a dictionary)	*/
/*--------------------------------------------------------------*/

static PyObject *xc_setattr(PyObject *self, PyObject *args)
{
   genericptr *gelem;
   PyObject *ehandle, *attrdict, *dval, *pval, *qval;
   int i, eaddr;

   if (!PyArg_ParseTuple(args, "OO:setattr", &ehandle, &attrdict))
      return NULL;

   /* Check to make sure that handle exists! */
   if ((gelem = CheckHandle(ehandle)) == NULL) return NULL;

   /* Is the argument a dictionary? */
   if (!PyDict_Check(attrdict)) { 
      PyErr_SetString(PyExc_TypeError,
		"setatrr() 2nd argument must be a dictionary");
      return NULL;
   }

   /* First, make sure no attempt is made to change the object type. */

   if ((dval = PyDict_GetItemString(attrdict, "type")) != NULL) {
      int dtype = string_to_type(PyString_AsString(dval));
      if (dtype < 0) return NULL;
      if (dtype != (*gelem)->type) {
         PyErr_SetString(PyExc_TypeError,
		"Attempt to change the type of an object.");
         return NULL;
      }
   }

   /* Next, look for dictionary strings containing values which apply */
   /* to a number of different elements (position, color, etc.)	      */

   if ((dval = PyDict_GetItemString(attrdict, "color")) != NULL) {
      (*gelem)->color = (short)PyInt_AsLong(dval);
   }

   if ((dval = PyDict_GetItemString(attrdict, "position")) != NULL) {
      if (PyTuple_Check(dval) && PyTuple_Size(dval) == 2) {
         if ((pval = PyTuple_GetItem(dval, 0)) != NULL) {
            short xpos = (short)PyInt_AsLong(pval);
            switch((*gelem)->type) {
	       case ARC:
	          TOARC(gelem)->position.x = xpos;
	          calcarc(TOARC(gelem));
	          break;
	       case LABEL:
	          TOLABEL(gelem)->position.x = xpos;
	          break;
	       case OBJECT:
	          TOOBJINST(gelem)->position.x = xpos;
	          break;
               default:
	          PyErr_SetString(PyExc_TypeError,
		      "attempt to set position on Spline, Polygon, or Path");
	    }
         }

         if ((pval = PyTuple_GetItem(dval, 1)) != NULL) {
            short ypos = (short)PyInt_AsLong(pval);
            switch((*gelem)->type) {
	       case ARC:
	          TOARC(gelem)->position.y = ypos;
	          calcarc(TOARC(gelem));
	          break;
	       case LABEL:
	          TOLABEL(gelem)->position.y = ypos;
	          break;
	       case OBJECT:
	          TOOBJINST(gelem)->position.y = ypos;
	          break;
               default:
	          PyErr_SetString(PyExc_TypeError,
		      "attempt to set position on Spline, Polygon, or Path");
	    }
         }
      }
      else {
         PyErr_SetString(PyExc_TypeError,
		"position must be a tuple containing two integer values");
      }
   }

   if ((dval = PyDict_GetItemString(attrdict, "style")) != NULL) {
      short dstyle = (short)PyInt_AsLong(dval);
      switch((*gelem)->type) {
         case POLYGON:
	    TOPOLY(gelem)->style = dstyle;
	    break;
         case PATH:
	    TOPATH(gelem)->style = dstyle;
	    break;
         case ARC:
	    TOARC(gelem)->style = dstyle;
	    break;
         case SPLINE:
	    TOSPLINE(gelem)->style = dstyle;
	    break;
         default:
            PyErr_SetString(PyExc_TypeError,
		"attempt to set style on an Object Instance or Label");
      }
   }

   if ((dval = PyDict_GetItemString(attrdict, "linewidth")) != NULL) {
      float dwidth = (float)PyFloat_AsDouble(dval);
      switch((*gelem)->type) {
         case POLYGON:
	    TOPOLY(gelem)->width = dwidth;
	    break;
         case PATH:
	    TOPATH(gelem)->width = dwidth;
	    break;
         case ARC:
	    TOARC(gelem)->width = dwidth;
	    break;
         case SPLINE:
	    TOSPLINE(gelem)->width = dwidth;
	    break;
         default:
            PyErr_SetString(PyExc_TypeError,
		"attempt to set linewidth on an Object Instance or Label");
      }
   }

   if ((dval = PyDict_GetItemString(attrdict, "scale")) != NULL) {
      float dscale = (float)PyFloat_AsDouble(dval);
      switch((*gelem)->type) {
         case LABEL:
	    TOLABEL(gelem)->scale = dscale;
	    break;
         case OBJECT:
	    TOOBJINST(gelem)->scale = dscale;
	    break;
         default:
            PyErr_SetString(PyExc_TypeError,
		"attempt to set scale on something not a Label or Object Instance");
      }
   }

   if ((dval = PyDict_GetItemString(attrdict, "rotation")) != NULL) {
      float drot = (float)PyFloat_AsDouble(dval);
      switch((*gelem)->type) {
         case LABEL:
	    TOLABEL(gelem)->rotation = drot;
	    break;
         case OBJECT:
	    TOOBJINST(gelem)->rotation = drot;
	    break;
         default:
            PyErr_SetString(PyExc_TypeError,
		"attempt to set rotation on something not a Label or Object Instance");
      }
   }

   /* Dictionary entries specific to certain xcircuit types */

   switch((*gelem)->type) {
      case LABEL:
	 if ((dval = PyDict_GetItemString(attrdict, "justify")) != NULL) {
	    TOLABEL(gelem)->justify = PyInt_AsLong(dval);
	 }
	 if ((dval = PyDict_GetItemString(attrdict, "string")) != NULL) {
	    /* To be dealt with. . . requires a list of string parts */
	 }
#ifdef SCHEMA
	 if ((dval = PyDict_GetItemString(attrdict, "pin")) != NULL) {
	    TOLABEL(gelem)->pin = PyInt_AsLong(dval);
	 }
#endif
	 break;
      case ARC:
	 if ((dval = PyDict_GetItemString(attrdict, "start angle")) != NULL) {
	    TOARC(gelem)->angle1 = (float)PyFloat_AsDouble(dval);
	 }
	 if ((dval = PyDict_GetItemString(attrdict, "end angle")) != NULL) {
	    TOARC(gelem)->angle2 = (float)PyFloat_AsDouble(dval);
	 }
	 if ((dval = PyDict_GetItemString(attrdict, "radius")) != NULL) {
	    TOARC(gelem)->radius = PyInt_AsLong(dval);
	 }
	 if ((dval = PyDict_GetItemString(attrdict, "minor axis")) != NULL) {
	    TOARC(gelem)->yaxis = PyInt_AsLong(dval);
	 }
	 break;
      case SPLINE:
	 if ((dval = PyDict_GetItemString(attrdict, "control points")) != NULL) {
            if (PyList_Check(dval) && PyList_Size(dval) == 4) {
	       for (i = 0; i < 4; i++) {
		  pval = PyList_GetItem(dval, i);
		  if (PyTuple_Check(pval) && PyTuple_Size(pval) == 2) {
		     qval = PyTuple_GetItem(pval, 0);
		     TOSPLINE(gelem)->ctrl[i].x = (short)PyInt_AsLong(qval);
		     qval = PyTuple_GetItem(pval, 1);
		     TOSPLINE(gelem)->ctrl[i].y = (short)PyInt_AsLong(qval);
		  }
		  else {
                     PyErr_SetString(PyExc_TypeError,
				"must have a tuple of 2 values per point");
	             break;
		  }
	       }
	    }
	    else {
               PyErr_SetString(PyExc_TypeError,
			"must have 4 control points in a list");
	       break;
	    }
	 }
	 break;
      case POLYGON:
	 if ((dval = PyDict_GetItemString(attrdict, "points")) != NULL) {
            if (PyList_Check(dval)) {
	       int number = PyList_Size(dval);
	       if (TOPOLY(gelem)->number != number) {
		  TOPOLY(gelem)->points = (pointlist)realloc(
			TOPOLY(gelem)->points, number * sizeof(XPoint));
		  TOPOLY(gelem)->number = number;
	       }
	       for (i = 0; i < number; i++) {
		  pval = PyList_GetItem(dval, i);
		  if (PyTuple_Check(pval) && PyTuple_Size(pval) == 2) {
		     qval = PyTuple_GetItem(pval, 0);
		     TOPOLY(gelem)->points[i].x = (short)PyInt_AsLong(qval);
		     qval = PyTuple_GetItem(pval, 1);
		     TOPOLY(gelem)->points[i].y = (short)PyInt_AsLong(qval);
		  }
		  else {
                     PyErr_SetString(PyExc_TypeError,
				"must have a tuple of 2 values per point");
	             break;
		  }
	       }
	    }
	    else {
               PyErr_SetString(PyExc_TypeError,
			"points must be in a list of tuples");
	       break;
	    }
	 }
	 break;
   }

   return PyInt_FromLong((long)(*gelem));
}
/*--------------------------------------------------------------*/
/* Set various options through the "set" command.		*/
/* "set <option> on|off" supercedes "enable|disable <option>"	*/
/*--------------------------------------------------------------*/

static PyObject *xc_set(PyObject *self, PyObject *args)
{
   const char *sarg1, *sarg2;
   int i;
   Boolean a = True; 
   short cpage = areastruct.page;

   if (!PyArg_ParseTuple(args, "ss:set", &sarg1, &sarg2))
      return NULL;
   
   if (!strcmp(sarg2, "On") || !strcmp(sarg2, "on") || !strcmp(sarg2, "True")
	|| !strcmp(sarg2, "true"))
      a = False;     /* has to be backwards; toggle() inverts the value! */

   if (!strcmp(sarg1, "font")) {
      for (i = 0; i < fontcount; i++)
	 if (!strcmp(fonts[i].psname, sarg2)) break;

      if (i == fontcount)
	 loadfontfile((char *)sarg2);

      areastruct.psfont = i;
   }
   else if (!strcmp(sarg1, "fontscale")) {
      sscanf(sarg2, "%f", &areastruct.textscale);
   }
   else if (!strcmp(sarg1, "axis") || !strcmp(sarg1, "axes")) {
      areastruct.axeson = a;
      toggle(GridAxesButton, &areastruct.axeson, NULL);
   }
   else if (!strcmp(sarg1, "grid")) {
      areastruct.gridon = a;
      toggle(GridGridButton, &areastruct.gridon, NULL);
   }  
   else if (!strcmp(sarg1, "snap") || !strcmp(sarg1, "snap-to")) {
      areastruct.snapto = a;
      toggle(SnaptoSnaptoButton, &areastruct.snapto, NULL);
   }
   else if (!strcmp(sarg1, "gridspace")) {
      sscanf(sarg2, "%f", &xobjs.pagelist[cpage]->gridspace);
   }
   else if (!strcmp(sarg1, "snapspace")) {
      sscanf(sarg2, "%f", &xobjs.pagelist[cpage]->snapspace);
   }
   else if (!strcmp(sarg1, "pagestyle")) {
      if (!strcmp(sarg2, "encapsulated") || !strcmp(sarg2, "eps"))
	 xobjs.pagelist[cpage]->pmode = 0;
      else
	 xobjs.pagelist[cpage]->pmode = 1;
   }
   else if (!strcmp(sarg1, "boxedit")) {
      if (!strcmp(sarg2, "rhomboid-x")) boxedit(NULL, RHOMBOIDX, NULL);
      else if (!strcmp(sarg2, "rhomboid-y")) boxedit(NULL, RHOMBOIDY, NULL);
      else if (!strcmp(sarg2, "rhomboid-a")) boxedit(NULL, RHOMBOIDA, NULL);
      else if (!strcmp(sarg2, "manhattan")) boxedit(NULL, MANHATTAN, NULL);
      else if (!strcmp(sarg2, "normal")) boxedit(NULL, NORMAL, NULL);
   }
   else if (!strcmp(sarg1, "linewidth")) {
      sscanf(sarg2, "%f", &areastruct.linewidth);
   }
   else if (!strcmp(sarg1, "colorscheme")) {
      if (!strcmp(sarg2, "inverse"))
         areastruct.invert = False;
      inversecolor(NULL, &areastruct.invert, NULL);
   }
   else if (!strcmp(sarg1, "coordstyle")) {
      if (!strcmp(sarg2, "cm") || !strcmp(sarg2, "centimeters")) {
         xobjs.pagelist[cpage]->coordstyle = CM;
         xobjs.pagelist[cpage]->pagesize.x = 595;  /* A4 size */
         xobjs.pagelist[cpage]->pagesize.y = 842;
         togglegrid((u_short)xobjs.pagelist[cpage]->coordstyle);
      }
   }
   else if (!strcmp(sarg1, "orient")) {   /* "orient" or "orientation" */
      if (!strcmp(sarg2, "landscape"))
         xobjs.pagelist[cpage]->orient = 90; /* Landscape */
      else
         xobjs.pagelist[cpage]->orient = 0;  /* Portrait */
   }

#ifdef SCHEMA
   else if (!strcmp(sarg1, "xschema") || !strcmp(sarg1, "schema")) {
      areastruct.schemon = a;
      doxschema(OptionsEnableXSchemaButton, NULL, NULL);
   }
#endif
#ifdef HAVE_XPM
   else if (!strcmp(sarg1, "toolbar")) {
      areastruct.toolbar_on = a;
      dotoolbar(OptionsDisableToolbarButton, NULL, NULL);
    }
#endif

   return PyString_FromString(sarg2);
}

/*--------------------------------------------------------------*/

static PyObject *xc_override(PyObject *self, PyObject *args)
{
   const char *sarg1;

   if (!PyArg_ParseTuple(args, "s:override", &sarg1))
      return NULL;

   if (!strcmp(sarg1, "library") || !strcmp(sarg1, "libraries"))
      flags |= LIBOVERRIDE;
   else if (!strcmp(sarg1, "color") || !strcmp(sarg1, "colors"))
      flags |= COLOROVERRIDE;
   else if (!strcmp(sarg1, "font") || !strcmp(sarg1, "fonts"))
      flags |= FONTOVERRIDE;
   if (!strcmp(sarg1, "key") || !strcmp(sarg1, "keybindings"))
      flags |= KEYOVERRIDE;
   
   return PyInt_FromLong(0L);
}

/*--------------------------------------------------------------*/

static PyObject *xc_library(PyObject *self, PyObject *args)
{
   const char *libname;
   int libnum = 1;

   if (!PyArg_ParseTuple(args, "s|i:library", &libname, &libnum))
      return NULL;

   /* if loading of default libraries is not overridden, load them first */

   if (!(flags & (LIBOVERRIDE | LIBLOADED))) {
      defaultscript();
      flags |= LIBLOADED;   /* Pass through a Python variable? */
   }

   if (libnum >= xobjs.numlibs)
      libnum = createlibrary();
   else
      libnum += LIBRARY - 1;

   strcpy(_STR, libname);
   loadlibrary(libnum);
   return PyInt_FromLong((long)libnum);
}

/*--------------------------------------------------------------*/

static PyObject *xc_font(PyObject *self, PyObject *args)
{
   const char *fontname;

   if (!PyArg_ParseTuple(args, "s:font", &fontname))
      return NULL;

   if (!(flags & FONTOVERRIDE)) {
      loadfontfile("Helvetica");
      flags |= FONTOVERRIDE;
   }
   loadfontfile((char *)fontname);
   return PyString_FromString(fontname);
}

/*--------------------------------------------------------------*/

static PyObject *xc_color(PyObject *self, PyObject *args)
{
   const char *colorname;

   if (!PyArg_ParseTuple(args, "s:color", &colorname))
      return NULL;

   addnewcolorentry(xc_alloccolor((char *)colorname));
   return PyString_FromString(colorname);
}

/*--------------------------------------------------------------*/

static PyObject *xc_bind(PyObject *self, PyObject *args)
{
   const char *keyname = NULL;
   const char *function = NULL;
   PyObject *retobj;

   if (!PyArg_ParseTuple(args, "|ss:bind", &keyname, &function))
      return NULL;

   /* No arguments?  Return a dictionary of bound pairs */
   if (!keyname) {
      PyObject *key, *value;
      char *keyname, *funcname;
      keybinding *ksearch;

      retobj = PyDict_New();
      for (ksearch = keylist; ksearch != NULL; ksearch = ksearch->nextbinding) {
	 keyname = key_to_string(ksearch->keywstate);
	 if (ksearch->value >= 0) {
	    funcname = malloc(strlen(func_to_string(ksearch->function)) + 5);
	    sprintf(funcname, "%s %d", func_to_string(ksearch->function),
		ksearch->value);
	 }
	 else
	    funcname = func_to_string(ksearch->function);
	 key = PyString_FromString(keyname);
	 value = PyString_FromString(funcname);
	 PyDict_SetItem(retobj, key, value);
	 free(keyname);
	 if (ksearch->value >= 0) free(funcname);
      }
   }
   /* One argument?  Argument is a key or a function */
   else if (!function) {
      char *binding;
      int func = -1, keywstate;
      keywstate = string_to_key(keyname);
      if (keywstate == 0) { /* first argument (keyname) is function? */
	 keywstate = -1;
	 func = string_to_func(keyname, NULL);
      }
      binding = binding_to_string(keywstate, func);
      retobj = PyString_FromString(binding);
      free(binding);
   }
   else {
      add_keybinding(keyname, function);
      retobj = PyString_FromString(keyname);
   }
   return retobj;
}

/*--------------------------------------------------------------*/

static PyObject *xc_unbind(PyObject *self, PyObject *args)
{
   const char *keyname;
   const char *function;

   if (!PyArg_ParseTuple(args, "ss:unbind", &keyname, &function))
      return NULL;

   remove_keybinding(keyname, function);
   return PyString_FromString(keyname);
}

/*--------------------------------------------------------------*/
/* active delay 						*/
/*--------------------------------------------------------------*/
   
static PyObject *xc_pause(PyObject *self, PyObject *args)
{
   float delay;

   if (!PyArg_ParseTuple(args, "f:pause", &delay))
      return NULL;

   usleep((int)(1e6 * delay));
   return PyInt_FromLong((int)(1e6 * delay));
}

/*--------------------------------------------------------------*/
/* active refresh						*/
/*--------------------------------------------------------------*/
         
static PyObject *xc_refresh(PyObject *self, PyObject *args)
{
   XEvent event;

   if (!PyArg_ParseTuple(args, ":refresh"))
      return NULL;
      
   refresh(NULL, NULL, NULL);

   while(XCheckWindowEvent(dpy, win, ~NoEventMask, &event))
         XtDispatchEvent(&event);

   return PyInt_FromLong(0L);
}

/*--------------------------------------------------------------*/
/* Declaration of the xcircuit Python functions			*/
/*--------------------------------------------------------------*/

static PyMethodDef xc_methods[] = {
   {"set",		xc_set,		1},
   {"override",		xc_override,	1},
   {"library",		xc_library,	1},
   {"font",		xc_font,	1},
   {"color",		xc_color,	1},
   {"newelement",	xc_new,		1},
   {"getpage",		xc_getpage,	1},
   {"getattr",		xc_getattr,	1},
   {"setattr",		xc_setattr,	1},
   {"refresh",		xc_refresh,	1},
   {"pause",		xc_pause,	1},
   {"bind",		xc_bind,	1},
   {"unbind",		xc_unbind,	1},
   {NULL,	NULL}			/* sentinel */
};

/*--------------------------------------------------------------*/
/* Initialize Python interpreter and load all xcircuit methods 	*/
/*--------------------------------------------------------------*/
void init_interpreter()
{
   Py_SetProgramName("XCircuit");
   Py_Initialize();
   PyImport_AddModule("xc");
   Py_InitModule("xc", xc_methods);
   PyRun_SimpleString("from xc import *\n");
}

/*--------------------------------------------------------------*/
/* Exit the Python interpreter					*/
/*--------------------------------------------------------------*/

void exit_interpreter()
{
   Py_Exit(0);
}

/*--------------------------------------------------------------*/
/* Replace the functions of the simple rcfile.c	interpreter.	*/
/*--------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Execute a single command from a script or from the command line      */
/*----------------------------------------------------------------------*/

short execcommand(short pflags, char *cmdptr)
{
   flags = pflags;
   PyRun_SimpleString(cmdptr);
   refresh(NULL, NULL, NULL);
   return flags;
}

/*----------------------------------------------------------------------*/
/* Load the default script (like execscript() but don't allow recursive */
/* loading of the startup script)                                       */
/*----------------------------------------------------------------------*/

void defaultscript()
{
   FILE *fd;

   flags = LIBOVERRIDE | LIBLOADED | FONTOVERRIDE;

   sprintf(_STR2, "%s/%s", BUILTINS_DIR, STARTUP_FILE);

   if ((fd = fopen(_STR2, "r")) != NULL)
      PyRun_SimpleFile(fd, _STR2);
   else {
      sprintf(_STR, "Failed to open startup script \"%s\"\n", STARTUP_FILE);
      Wprintf(_STR);
   }
}

/*----------------------------------------------------------------------*/
/* Execute a script                                                     */
/*----------------------------------------------------------------------*/

void execscript()
{
   FILE *fd;

   flags = 0;

   if ((fd = fopen(_STR2, "r")) != NULL) {
      PyRun_SimpleFile(fd, _STR2);
      refresh(NULL, NULL, NULL);
   }
   else {
      sprintf(_STR, "Failed to open script file \"%s\"\n", _STR2);
      Wprintf(_STR);
   }  
}

/*----------------------------------------------------------------------*/
/* Execute the .xcircuitrc startup script                               */
/*----------------------------------------------------------------------*/

void loadrcfile()
{
   char *userdir = getenv((const char *)"HOME");
   FILE *fd;
   short i;

   /* Initialize flags */

   flags = 0;

   sprintf(_STR2, "%s", USER_RC_FILE);     /* Name imported from Makefile */
      
   /* try first in current directory, then look in user's home directory */
      
   if ((fd = fopen(_STR2, "r")) == NULL) {
      sprintf(_STR2, "%s/%s", userdir, USER_RC_FILE);
      fd = fopen(_STR2, "r");
   }  
   if (fd != NULL)
      PyRun_SimpleFile(fd, _STR2);

   /* Add the default font if not loaded already */

   if (!(flags & FONTOVERRIDE)) {
      loadfontfile("Helvetica");
      for (i = 0; i < fontcount; i++)
	 if (!strcmp(fonts[i].psname, "Helvetica")) areastruct.psfont = i;
   }
   setdefaultfontmarks();
      
   /* arrange the loaded libraries */
      
   if (!(flags & (LIBOVERRIDE | LIBLOADED)))
      defaultscript();
      
   /* Add the default colors */
      
   if (!(flags & COLOROVERRIDE)) {
      addnewcolorentry(xc_alloccolor("Gray40"));
      addnewcolorentry(xc_alloccolor("Gray60"));
      addnewcolorentry(xc_alloccolor("Gray80"));
      addnewcolorentry(xc_alloccolor("Gray90"));
      addnewcolorentry(xc_alloccolor("Red"));
      addnewcolorentry(xc_alloccolor("Blue"));
      addnewcolorentry(xc_alloccolor("Green"));
      addnewcolorentry(xc_alloccolor("Yellow"));
      addnewcolorentry(xc_alloccolor("Purple"));
      addnewcolorentry(xc_alloccolor("Cyan"));
      addnewcolorentry(xc_alloccolor("Magenta"));
      addnewcolorentry(xc_alloccolor("Orange"));
      addnewcolorentry(xc_alloccolor("Tan"));
      addnewcolorentry(xc_alloccolor("Brown"));
   }  

   if (!(flags & KEYOVERRIDE))
      default_keybindings();
}

#endif
/* #endif HAVE_PYTHON */
/*--------------------------------------------------------------*/
