/*----------------------------------------------------------------------*/
/* menucalls.c --- callback routines from the menu buttons		*/
/*		   and associated routines				*/
/* Copyright (c) 2001  Tim Edwards, Johns Hopkins University        	*/
/*----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if defined(DARWIN)
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#include <sys/types.h>
#include <errno.h>
#include <limits.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>

#include "Xw/Xw.h"
#include "Xw/MenuBtn.h"
#include "Xw/PopupMgr.h"

/*----------------------------------------------------------------------*/
/* Local includes							*/
/*----------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*----------------------------------------------------------------------*/
/* External Variable definitions					*/
/*----------------------------------------------------------------------*/

extern char	 _STR2[250];
extern char	 _STR[150];          /* Generic multipurpose string */
extern Widget	 top;
extern Display   *dpy;
extern Window    win;
extern short	 popups;	     /* total number of popup windows */
extern Widget    menuwidgets[];
extern Widget	 toolbar;
extern Globaldata xobjs;
extern Clientdata areastruct;
extern int	  *appcolors;
extern int	  number_colors;
extern colorindex *colorlist;
extern ApplicationData appdata;
extern Cursor	  appcursors[NUM_CURSORS];
extern fontinfo *fonts;
extern short fontcount;
extern short eventmode, textpos, textend;
extern objectpair *pushlist;

/*----------------------------------------------------------------------*/
/* The rest of the local includes depend on the prototype declarations	*/
/* and some of the external global variable declarations.		*/
/*----------------------------------------------------------------------*/
#include "menus.h"
#include "menudep.h"

/*----------------------------------------------------------------------*/
/* Local Variable definitions						*/
/*----------------------------------------------------------------------*/

u_short *fontnumbers;
u_char nfontnumbers;

/*--------------------------------------------------------------*/
/* Enable or Disable the toolbar				*/
/*--------------------------------------------------------------*/

#ifdef HAVE_XPM
void dotoolbar(Widget w, caddr_t clientdata, caddr_t calldata)   
{
   Arg wargs[1];

   if (areastruct.toolbar_on) {
      areastruct.toolbar_on = False;
      XtUnmanageChild(toolbar);
      XtSetArg(wargs[0], XtNlabel, "Enable Toolbar");
      XtSetValues(OptionsDisableToolbarButton, wargs, 1);
      XtRemoveCallback(areastruct.area, XtNresize, (XtCallbackProc)resizetoolbar, NULL);
   }
   else {
      areastruct.toolbar_on = True;
      XtManageChild(toolbar);
      XtSetArg(wargs[0], XtNlabel, "Disable Toolbar");
      XtSetValues(OptionsDisableToolbarButton, wargs, 1);
      XtAddCallback(areastruct.area, XtNresize, (XtCallbackProc)resizetoolbar, NULL);
   }
}  

/*--------------------------------------------------------------*/
/* Overwrite the toolbar pixmap for color or stipple entries	*/
/*--------------------------------------------------------------*/

void overdrawpixmap(Widget button)
{
   Widget pbutton;
   Arg args[3];
   int ltype, color;
   Pixmap stippix;

   if (button == NULL) return;

   XtSetArg(args[0], XtNlabelType, &ltype);
   XtGetValues(button, args, 1);

   if (ltype != XwRECT && button != ColorInheritColorButton) return;

   XtSetArg(args[0], XtNrectColor, &color);
   XtSetArg(args[1], XtNrectStipple, &stippix);
   XtGetValues(button, args, 2);

   if (stippix == (Pixmap)NULL && button != FillBlackButton)
      pbutton = ColorsToolButton;
   else
      pbutton = FillsToolButton;

   if (button == ColorInheritColorButton) {
      XtSetArg(args[0], XtNlabelType, XwIMAGE);
      XtSetValues(pbutton, args, 1);
   }
   else if (button == FillBlackButton) {
      XtSetArg(args[0], XtNlabelType, XwRECT);
      XtSetArg(args[1], XtNrectColor, color);
      XtSetArg(args[2], XtNrectStipple, (Pixmap)NULL);
      XtSetValues(pbutton, args, 3);
   }
   else if (button != FillOpaqueButton) {
      XtSetArg(args[0], XtNlabelType, XwRECT);
      if (stippix == (Pixmap)NULL)
         XtSetArg(args[1], XtNrectColor, color);
      else
         XtSetArg(args[1], XtNrectStipple, stippix);

      XtSetValues(pbutton, args, 2);
   }
}
#else
#define overdrawpixmap(a)
#endif

/*--------------------------------------------------------------*/
/* Generic routine for use by all other data handling routines 	*/
/*--------------------------------------------------------------*/

void getgeneric(buttonsave *saveptr, Widget button, void (*getfunction)(),
	void *dataptr)
{
   Arg  wargs[1];

   saveptr->button = button;
   saveptr->buttoncall = getfunction;
   saveptr->dataptr = dataptr;

   if (button != NULL) {
      XtSetArg(wargs[0], XtNforeground, &saveptr->foreground);
      XtGetValues(button, wargs, 1);
      XtSetArg(wargs[0], XtNforeground, OFFBUTTONCOLOR);
      XtSetValues(button, wargs, 1);
      XtRemoveAllCallbacks(button, XtNselect);
   }
}


/*----------------------------------------------------------------*/
/* setgrid, getgridspace are for grid and snap spacing sizes;	  */
/* include routines to parse fractions				  */
/*----------------------------------------------------------------*/

void setgrid(Widget w, float *dataptr)
{
   float oldvalue = *dataptr;
   float oscale, iscale = (float)xobjs.pagelist[areastruct.page]->drawingscale.y /
        (float)xobjs.pagelist[areastruct.page]->drawingscale.x;
   float fval;

   /* For now, assume that the value is in the current display style. */
   /* Might be nice in the future to make it accept any input. . .    */

   switch (xobjs.pagelist[areastruct.page]->coordstyle) {
      case CM:
         oscale = xobjs.pagelist[areastruct.page]->outscale * CMSCALE;
	 if (sscanf(_STR2, "%f", &fval) == 0) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	 }
	 else *dataptr = fval * IN_CM_CONVERT / (iscale * oscale);
	 break;
      case DEC_INCH: case FRAC_INCH: {
	 short parts;
	 char *sptr;
	 int f2, f3;

         oscale = xobjs.pagelist[areastruct.page]->outscale * INCHSCALE;
	 for (sptr = _STR2; *sptr != '\0'; sptr++)
	    if (*sptr == '/') *sptr = ' ';
	 parts = sscanf(_STR2, "%f %d %d", &fval, &f2, &f3);
	 if ((parts == 0) || (parts != 1 && (fval != (float)((int)fval)))) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	    break;
	 }
	 if (parts == 2) fval /= (float)f2;
	 else if (parts == 3) fval += ((float)f2 / (float)f3);
	 *dataptr = fval * 72.0 / (iscale * oscale);
	 } break;
   }
   if (oldvalue != *dataptr) drawarea(NULL, NULL, NULL);
}

/*----------------------------------------------------------------*/
/* Write a measurement value into string "buffer" dependant on    */
/* the current default units of measure (centimeters or inches).  */
/*----------------------------------------------------------------*/

void measurestr(float value, char *buffer)
{  
   float oscale, iscale;
   iscale = (float)(xobjs.pagelist[areastruct.page]->drawingscale.y) /
        (float)(xobjs.pagelist[areastruct.page]->drawingscale.x);

   switch (xobjs.pagelist[areastruct.page]->coordstyle) {
      case CM:
         oscale = xobjs.pagelist[areastruct.page]->outscale * CMSCALE;
         sprintf(buffer, "%5.3f cm", value * iscale * oscale / IN_CM_CONVERT);
         break; 
      case DEC_INCH:
         oscale = xobjs.pagelist[areastruct.page]->outscale * INCHSCALE;
         sprintf(buffer, "%5.3f in", value * iscale * oscale / 72.0);
         break;
      case FRAC_INCH:
         oscale = xobjs.pagelist[areastruct.page]->outscale * INCHSCALE;
         fraccalc(((value * iscale * oscale) / 72.0), buffer);
         strcat(buffer, " in");
         break;
   }
}        

/*--------------------------------------------------------------*/
/* Generate popup dialog for snap space value input		*/
/*--------------------------------------------------------------*/

void getsnapspace(Widget button, caddr_t clientdata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *floatptr = &xobjs.pagelist[areastruct.page]->snapspace;

   getgeneric(savebutton, button, getsnapspace, (void *)floatptr);
   measurestr(*floatptr, buffer);
   popupprompt(button, "Enter value:", buffer, setgrid, savebutton, NULL);
}

/*--------------------------------------------------------------*/
/* Generate popup dialog for grid space value input		*/
/*--------------------------------------------------------------*/

void getgridspace(Widget button, caddr_t clientdata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *floatptr = &xobjs.pagelist[areastruct.page]->gridspace;

   getgeneric(savebutton, button, getgridspace, (void *)floatptr);
   measurestr(*floatptr, buffer);
   popupprompt(button, "Enter value:", buffer, setgrid, savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Generic routine for setting a floating-point value (through a  */
/* (float *) pointer passed as the second parameter)		  */
/*----------------------------------------------------------------*/

void setfloat(Widget w, float *dataptr)
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   if (oldvalue != *dataptr) drawarea(NULL, NULL, NULL);
}

/*----------------------------------------------------------------*/
/* set the global default line width.  The unit used internally   */
/* is twice the value passed through pointer "dataptr".		  */
/*----------------------------------------------------------------*/

void setwidth(Widget w, float *dataptr)
{
   float oldvalue = *dataptr;
   if (sscanf(_STR2, "%f", dataptr) == 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
      return;
   }
   (*dataptr) *= 2.0;
   if (oldvalue != *dataptr) drawarea(NULL, NULL, NULL);
}

/*----------------------------------------------------------------*/
/* Set text scale.						  */
/*----------------------------------------------------------------*/

void settsize(Widget w, labelptr settext)
{
   float tmpres;
   short *osel;
   labelptr nstext;
   stringpart *strptr, *nextptr;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {  /* can't interpret value or bad value */
      Wprintf("Illegal value");
      return;
   }

   /* In edit mode, add font scale change. */

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      if (textpos > 0 || textpos < stringlength(settext->string, True, NORMINST)) {
	 undrawtext(settext);
	 strptr = findstringpart(textpos - 1, NULL, settext->string, NORMINST);
	 nextptr = findstringpart(textpos, NULL, settext->string, NORMINST);
	 if (strptr->type == FONT_SCALE)
	    strptr->data.scale = tmpres;
	 else if (nextptr && nextptr->type == FONT_SCALE)
	    nextptr->data.scale = tmpres;
	 else
	    labeltext(FONT_SCALE, (char *)&tmpres);
	 redrawtext(settext);
      }
      else if (stringlength(settext->string, True, NORMINST) > 0)
	 labeltext(FONT_SCALE, (char *)&tmpres);
      else (settext->scale = tmpres);
   }

   /* Change scale on all selected text objects */

   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
	 if (SELECTTYPE(osel) == LABEL) {
	    nstext = SELTOLABEL(osel);
	    undrawtext(nstext);
            nstext->scale = tmpres;
	    redrawtext(nstext);
	 }
      }
      objectdeselect();
   }
}

/*--------------------------------------------------------------*/
/* Auto-set:  Enable automatic scaling				*/
/*--------------------------------------------------------------*/

void autoset(Widget w, WidgetList entertext, caddr_t nulldata)
{
   xobjs.pagelist[areastruct.page]->pmode |= 2;
   updatetext(w, entertext, nulldata);
}

/*--------------------------------------------------------------*/

void autostop(Widget w, caddr_t clientdata, caddr_t nulldata)
{
   xobjs.pagelist[areastruct.page]->pmode &= 1;
}

/*--------------------------------------------------------------*/
/* Auto-scale							*/
/* (for now, with fixed 1" margins)				*/
/*--------------------------------------------------------------*/

void autoscale()
{
   float newxscale, newyscale;
   float scalefudge = (xobjs.pagelist[areastruct.page]->coordstyle
	== CM) ? CMSCALE : INCHSCALE;
   float margin = 144.0;

   if (!(xobjs.pagelist[areastruct.page]->pmode & 2)) return;
   else if (objectdata->width == 0 || objectdata->height == 0) {
      Wprintf("Cannot auto-fit empty page");
      return;
   }

   newxscale = (xobjs.pagelist[areastruct.page]->pagesize.x - margin)
	/ scalefudge;
   newyscale = (xobjs.pagelist[areastruct.page]->pagesize.y - margin)
	/ scalefudge;

   if (xobjs.pagelist[areastruct.page]->orient == 0) {	/* Portrait */
      newxscale /= objectdata->width;
      newyscale /= objectdata->height;
   }
   else {
      newxscale /= objectdata->height;
      newyscale /= objectdata->width;
   }

   xobjs.pagelist[areastruct.page]->outscale = min(newxscale, newyscale);
}

/*--------------------------------------------------------------*/
/* Set the denomenator value of the drawing scale 		*/ 
/*--------------------------------------------------------------*/

void setscaley(Widget w, float *dataptr)
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0 || objectdata->height == 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   else {
      *dataptr = (*dataptr * 72) / objectdata->height;
      *dataptr /= getpsscale(1.0, areastruct.page);
   }
}

/*----------------------------------------------------------------*/
/* Set the numerator value of the drawing scale			  */
/*----------------------------------------------------------------*/

void setscalex(Widget w, float *dataptr)
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0 || objectdata->width == 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   else {
      *dataptr = (*dataptr * 72) / objectdata->width;
      *dataptr /= getpsscale(1.0, areastruct.page);
   }
}

/*----------------------------------------------------------------*/
/* Set the page orientation (either Landscape or Portrait)	  */
/*----------------------------------------------------------------*/

void setorient(Widget w, short *dataptr)
{
   Arg wargs[1];

   if (*dataptr == 0) {
      *dataptr = 90;
      XtSetArg(wargs[0], XtNlabel, "Landscape");
   }
   else {
      *dataptr = 0;
      XtSetArg(wargs[0], XtNlabel, "Portrait");
   }
   XtSetValues(w, wargs, 1);
}

/*----------------------------------------------------------------*/
/* Set the output mode to "Full Page" (unencapsulated PostScript) */
/* or "Embedded" (encapsulated PostScript)			  */
/*----------------------------------------------------------------*/

void setpmode(Widget w, short *dataptr)
{
   Arg wargs[1];
   Widget pwidg = XtParent(w);
   Widget autowidg = XtNameToWidget(pwidg, "Auto-fit");

   if (!(*dataptr & 1)) {
      *dataptr = 1;
      XtSetArg(wargs[0], XtNlabel, "Full Page");

      XtManageChild(XtNameToWidget(pwidg, "fpedit"));
      XtManageChild(XtNameToWidget(pwidg, "fpokay"));
      XtManageChild(autowidg);
   }
   else {
      *dataptr = 0;	/* This also turns off auto-fit */
      XtSetArg(wargs[0], XtNset, False);
      XtSetValues(autowidg, wargs, 1);
      XtSetArg(wargs[0], XtNlabel, "Embedded");

      XtUnmanageChild(XtNameToWidget(pwidg, "fpedit"));
      XtUnmanageChild(XtNameToWidget(pwidg, "fpokay"));
      XtUnmanageChild(autowidg);
   }
   XtSetValues(w, wargs, 1);
}

/*----------------------------------------------------------------*/
/* Set the output page size, in the current unit of measure	  */
/*----------------------------------------------------------------*/

void setpagesize(Widget w, XPoint *dataptr)
{
   float px, py;
   char units[10], *expos, fpedit[75];
   Arg wargs[1];

   strcpy(units, "in");

   if (sscanf(_STR2, "%f %*c %f %9s", &px, &py, units) < 4) {
      if (sscanf(_STR2, "%f %*c %f", &px, &py) < 3) {
	 if ((expos = strchr(_STR2, 'x')) == NULL) {
            Wprintf("Illegal Form for page size.");
	    return;
	 }
	 else {
	    *expos = '\0';
	    if (sscanf(_STR2, "%f", &px) == 0 ||
		  sscanf(expos + 1, "%f %9s", &py, units) == 0) {
               Wprintf("Illegal Form for page size.");
	       return;
	    }
	 }
      }
   }

   /* Don't reduce page to less than the margins (1") or negative	*/
   /* scales result.							*/

   if ((px <= 2.0) || (py <= 2.0)) {
      Wprintf("Page size too small for margins.");
      return;
   }

   dataptr->x = (short)(px * 72.0);
   dataptr->y = (short)(py * 72.0);

   if (!strcmp(units, "cm")) {
      sprintf(fpedit, "%3.2f x %3.2f cm",
	  (float)xobjs.pagelist[areastruct.page]->pagesize.x
	  / 72.0, (float)xobjs.pagelist[areastruct.page]->pagesize.y / 72.0);
      dataptr->x /= 2.54;
      dataptr->y /= 2.54;
   }
   else {
      sprintf(fpedit, "%3.2f x %3.2f in",
	  (float)xobjs.pagelist[areastruct.page]->pagesize.x
	  / 72.0, (float)xobjs.pagelist[areastruct.page]->pagesize.y / 72.0);
   }
   XtSetArg(wargs[0], XtNstring, fpedit);
   XtSetValues(XtNameToWidget(XtParent(w), "fpedit"), wargs, 1);
}

/*--------------------------------------------------------------*/
/* Generate popup dialog to get a character kern value, which	*/
/* is two integer numbers in the range -128 to +127 (size char)	*/
/*--------------------------------------------------------------*/

void getkern(Widget button, caddr_t nulldata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   int kx, ky;
   stringpart *strptr, *nextptr;

   strcpy(buffer, "0,0");

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      labelptr curlabel = TOLABEL(EDITPART);
      strptr = findstringpart(textpos - 1, NULL, curlabel->string, NORMINST);
      nextptr = findstringpart(textpos, NULL, curlabel->string, NORMINST);
      if (strptr->type == KERN) {
	 kx = strptr->data.kern[0];
	 ky = strptr->data.kern[1];
         sprintf(buffer, "%d,%d", kx, ky);
      }
      else if (nextptr && nextptr->type == KERN) {
	 strptr = nextptr;
	 kx = strptr->data.kern[0];
	 ky = strptr->data.kern[1];
         sprintf(buffer, "%d,%d", kx, ky);
      }
      else strptr = NULL;
   }

   getgeneric(savebutton, button, getkern, strptr);
   popupprompt(button, "Enter Kern X,Y:", buffer, setkern, savebutton, NULL);
}

/*--------------------------------------------------------------*/
/* Set a character kern value					*/ 
/*--------------------------------------------------------------*/

void setkern(Widget w, stringpart *kpart)
{
   char *sptr;
   short kd[2];

   kd[0] = kd[1] = 0;

   if ((sptr = strchr(_STR2, ',')) == NULL)
      Wprintf("Use notation X,Y");
   else {
      *sptr = '\0';
      sscanf(_STR2, "%hd", &kd[0]);
      sscanf(sptr + 1, "%hd", &kd[1]);
      if (kpart == NULL)
         labeltext(KERN, (char *)kd);
      else {
         labelptr curlabel = TOLABEL(EDITPART);
	 undrawtext(curlabel);
	 kpart->data.kern[0] = kd[0];
	 kpart->data.kern[1] = kd[1];
	 redrawtext(curlabel);
      }
   }
}

/*----------------------------------------------------------------*/
/* Generate popup dialog to get the drawing scale, specified as a */
/* whole-number ratio X:Y					  */
/*----------------------------------------------------------------*/

void getdscale(Widget button, caddr_t nulldata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   XPoint *ptptr = &(xobjs.pagelist[areastruct.page]->drawingscale);

   getgeneric(savebutton, button, getdscale, ptptr);
   sprintf(buffer, "%d:%d", ptptr->x, ptptr->y);
   popupprompt(button, "Enter Scale:", buffer, setdscale, savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Set the drawing scale (specified as ratio X:Y)		  */
/*----------------------------------------------------------------*/

void setdscale(Widget w, XPoint *dataptr)
{
   char *sptr;

   if ((sptr = strchr(_STR2, ':')) == NULL)
      Wprintf("Use ratio X:Y");
   else {
      *sptr = '\0';
      sscanf(_STR2, "%hd", &(dataptr->x));
      sscanf(sptr + 1, "%hd", &(dataptr->y));
      sprintf(_STR2, "New scale is %hd:%hd", dataptr->x, dataptr->y);
      Wprintf(_STR2);
      W1printf(" ");
   }
}

/*--------------------------------------------------------------*/
/* Generate the popup dialog for getting text scale.		*/ 
/*--------------------------------------------------------------*/

void gettsize(Widget button, caddr_t nulldata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   labelptr settext = NULL;
   float      *floatptr = &areastruct.textscale;
   short *osel;
   stringpart *strptr, *nextptr;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      if (textpos > 0 || textpos < stringlength(settext->string, True, NORMINST)) {
         settext = *((labelptr *)EDITPART);
	 strptr = findstringpart(textpos - 1, NULL, settext->string, NORMINST);
	 nextptr = findstringpart(textpos, NULL, settext->string, NORMINST);
	 if (strptr->type == FONT_SCALE) {
	    floatptr = &strptr->data.scale;
	    sprintf(buffer, "%5.3f", strptr->data.scale);
	 }
	 else if (nextptr && nextptr->type == FONT_SCALE) {
	    floatptr = &nextptr->data.scale;
	    sprintf(buffer, "%5.3f", nextptr->data.scale);
	 }
	 else {
	    floatptr = (float *)NULL;
	    strcpy(buffer, "1.00");
	 }
      }
      else {
         settext = *((labelptr *)EDITPART);
         floatptr = &(settext->scale);
         sprintf(buffer, "%4.2f", *floatptr);
      }
   }
   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
		areastruct.selects; osel++)
	 if (SELECTTYPE(osel) == LABEL) {
            settext = SELTOLABEL(osel);
            floatptr = &(settext->scale);
	    break;
	 }
      sprintf(buffer, "%4.2f", *floatptr);
   }
   if (settext != NULL) {
      getgeneric(savebutton, button, gettsize, settext);
      popupprompt(button, "Enter text scale:", buffer, settsize, savebutton, NULL);
   }
   else {
      sprintf(buffer, "%4.2f", *floatptr);
      getgeneric(savebutton, button, gettsize, floatptr);
      popupprompt(button,
	    "Enter default text scale:", buffer, setfloat, savebutton, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Set the scale of an object or group of selected objects	  */
/*----------------------------------------------------------------*/

void setosize(Widget w, objinstptr dataptr)
{
   float tmpres;
   short *osel;
   objinstptr nsobj;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {
      Wprintf("Illegal value");
      return;
   }
   for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
      if (SELECTTYPE(osel) == OBJECT) {
	 nsobj = SELTOOBJINST(osel);
         nsobj->scale = tmpres;
      }
   }
   objectdeselect();
   drawarea(NULL, NULL, NULL);
}

/*----------------------------------------------------------------*/
/* Generate popup dialog for getting object scale		  */
/*----------------------------------------------------------------*/

void getosize(Widget button, caddr_t clientdata, caddr_t calldata)
{
   char buffer[50];
   float flval;
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   short selects = 0;
   objinstptr setobj = NULL;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++)
      if (SELECTTYPE(osel) == OBJECT) {
	 setobj = SELTOOBJINST(osel);
	 selects++;
	 break;
      }
   if (setobj == NULL) {
      Wprintf("No objects were selected for scaling.");
      return;
   }
   flval = setobj->scale;
   getgeneric(savebutton, button, getosize, setobj);
   sprintf(buffer, "%4.2f", flval);
   popupprompt(button, "Enter object scale:", buffer, setosize, savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Generate popup prompt for getting global linewidth		  */
/*----------------------------------------------------------------*/

void getwirewidth(Widget button, caddr_t clientdata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   float *widthptr;

   widthptr = &(xobjs.pagelist[areastruct.page]->wirewidth);
   getgeneric(savebutton, button, getwirewidth, widthptr);
   sprintf(buffer, "%4.2f", *widthptr / 2.0);
   popupprompt(button, "Enter new global linewidth:", buffer, setwidth,
	savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Set the linewidth of all selected arcs, polygons, splines, and */
/* paths.							  */
/*----------------------------------------------------------------*/

void setwwidth(Widget w, void *dataptr)
{
   float     tmpres;
   short     *osel;
   arcptr    nsarc;
   polyptr   nspoly;
   splineptr nsspline;
   pathptr   nspath;

   if (sscanf(_STR2, "%f", &tmpres) == 0) {
      Wprintf("Illegal value");
      return;
   }
   else if (areastruct.selects == 0) {
      areastruct.linewidth = tmpres;
   }
   else {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
         if (SELECTTYPE(osel) == ARC) {
	    nsarc = SELTOARC(osel);
            nsarc->width = tmpres;
         }
         else if (SELECTTYPE(osel) == POLYGON) {
	    nspoly = SELTOPOLY(osel);
            nspoly->width = tmpres;
         }
         else if (SELECTTYPE(osel) == SPLINE) {
	    nsspline = SELTOSPLINE(osel);
            nsspline->width = tmpres;
         }
         else if (SELECTTYPE(osel) == PATH) {
	    nspath = SELTOPATH(osel);
            nspath->width = tmpres;
         }
      }
      objectdeselect();
      drawarea(NULL, NULL, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Generate popup dialong for getting linewidths of elements	  */ 
/*----------------------------------------------------------------*/

void getwwidth(Widget button, caddr_t clientdata, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   genericptr setel;
   float flval;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++) {
      setel = *(objectdata->plist + (*osel));
      if (setel->type == ARC) {
	 flval = ((arcptr)setel)->width;
	 break;
      }
      else if (setel->type == POLYGON) {
	 flval = ((polyptr)setel)->width;
	 break;
      }
      else if (setel->type == SPLINE) {
	 flval = ((splineptr)setel)->width;
	 break;
      }
      else if (setel->type == PATH) {
	 flval = ((pathptr)setel)->width;
	 break;
      }
   }
   getgeneric(savebutton, button, getwwidth, setel);
   if (osel == areastruct.selectlist + areastruct.selects) {
      sprintf(buffer, "%4.2f", areastruct.linewidth);
      popupprompt(button, "Enter new default line width:", buffer, setwwidth,
		savebutton, NULL);
   }
   else {
      sprintf(buffer, "%4.2f", flval);
      popupprompt(button, "Enter new line width:", buffer, setwwidth,
		savebutton, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Generic popup prompt for getting a floating-point value	  */
/*----------------------------------------------------------------*/

void getfloat(Widget button, float *floatptr, caddr_t calldata)
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, button, getfloat, floatptr);
   sprintf(buffer, "%4.2f", *floatptr);
   popupprompt(button, "Enter value:", buffer, setfloat, savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Set the filename for the current page			  */
/*----------------------------------------------------------------*/

void setfilename(Widget w, char **dataptr)
{
   short cpage, depend = 0;
   objectptr checkpage;
   char *oldstr = xobjs.pagelist[areastruct.page]->filename;

   if (!strcmp(*dataptr, _STR2)) return;   /* no change in string */

   /* Make the change to the current page */
   xobjs.pagelist[areastruct.page]->filename = strdup(_STR2);

   /* All existing filenames which match the old string should also be changed */
   for (cpage = 0; cpage < xobjs.pages; cpage++) {
      checkpage = xobjs.pagelist[cpage]->pageobj;
      if ((checkpage != NULL) && (cpage != areastruct.page)) {
	 if (!strcmp(xobjs.pagelist[cpage]->filename, oldstr)) {
	    free(xobjs.pagelist[cpage]->filename);
	    xobjs.pagelist[cpage]->filename = strdup(_STR2);
	 }
      }
   }
   free(oldstr);
}

/*----------------------------------------------------------------*/
/* Set the page label for the current page			  */
/*----------------------------------------------------------------*/

void setpagelabel(Widget w, char *dataptr)
{
   short i;

   /* Whitespace and non-printing characters not allowed */

   for (i = 0; i < strlen(_STR2); i++) {
      if ((!isprint(_STR2[i])) || (isspace(_STR2[i]))) {
         _STR2[i] = '_';
         Wprintf("Replaced illegal whitespace in name with underscore");
      }
   }

   if (!strcmp(dataptr, _STR2)) return; /* no change in string */
   if (strlen(_STR2) == 0)
      sprintf(objectdata->name, "Page %d", areastruct.page + 1);
   else
      sprintf(objectdata->name, "%.79s", _STR2);

#ifdef SCHEMA
   /* For schematics, all pages with associations to symbols must have	*/
   /* unique names.							*/
   if (objectdata->symschem != NULL) checkpagename(objectdata);
#endif

   printname(objectdata);
   renamepage(areastruct.page);
}

/*--------------------------------------------------------------*/
/* Add a new font name to the list of known fonts		*/
/*--------------------------------------------------------------*/

void makenewfontbutton()
{
   Arg	wargs[1];
   int  n = 0;
   Widget newbutton, cascade;

   if (fontcount == 0) return;

   cascade = XtParent(FontAddNewFontButton);
   XtnSetArg(XtNfont, appdata.xcfont);
   newbutton = XtCreateWidget(fonts[fontcount - 1].family, XwmenubuttonWidgetClass,
	 cascade, wargs, n);

   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)setfont,
		Number(fontcount - 1));
   XtManageChild(newbutton);

   nfontnumbers++;
   if (nfontnumbers == 1)
      fontnumbers = (u_short *)malloc(sizeof(u_short)); 
   else
      fontnumbers = (u_short *)realloc(fontnumbers, nfontnumbers
		* sizeof(u_short));
   fontnumbers[nfontnumbers - 1] = fontcount - 1;
}

/*--------------------------------------------------------------*/
/* Make new encoding menu button				*/
/*--------------------------------------------------------------*/

void makenewencodingbutton(char *ename, char value)
{
   Arg	wargs[1];
   int  n = 0;
   Widget newbutton, cascade;

   cascade = XtParent(EncodingStandardButton);

   /* return if button has already been made */
   newbutton = XtNameToWidget(cascade, ename);
   if (newbutton != NULL) return;

   XtnSetArg(XtNfont, appdata.xcfont);
   newbutton = XtCreateWidget(ename, XwmenubuttonWidgetClass,
	 cascade, wargs, n);

   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)setfontencoding,
		Number(value));
   XtManageChild(newbutton);
}

/*--------------------------------------------------------------*/
/* Set the menu checkmarks on the font menu			*/
/*--------------------------------------------------------------*/

void togglefontmark(int fontval)
{
   Arg args[1];
   Widget widget, cascade, sibling;
   short i;

   cascade = XtParent(FontAddNewFontButton);
   widget = XtNameToWidget(cascade, fonts[fontval].family);

   /* Remove checkmark from all widgets in the list */

   XtSetArg(args[0], XtNsetMark, False);
   for (i = 0; i < fontcount; i++) {
      if (i != fontval) {
         sibling = XtNameToWidget(cascade, fonts[i].family);
         XtSetValues(sibling, args, 1);
      }
   }

   /* Add checkmark to designated font */

   XtSetArg(args[0], XtNsetMark, True);
   XtSetValues(widget, args, 1);
}

/*--------------------------------------------------------------------*/
/* Toggle one of a set of menu items, only one of which can be active */
/*--------------------------------------------------------------------*/

void toggleexcl(Widget widget, menuptr menu, int menulength)
{
   Arg          args[1];
   Widget       parent = XtParent(widget);
   Widget       sibling;
   menuptr	mitem, mtest;
   short	i;

   /* find the menu item which corresponds to the widget which was pushed */

   for (mtest = menu; mtest < menu + menulength; mtest++) {
      sibling = XtNameToWidget(parent, mtest->name);
      if (sibling == widget) break;
   }

   /* remove checkmark from other widgets in the list */

   XtSetArg(args[0], XtNsetMark, False);
   if (menu == Fonts) {     		/* special action for font list */
      for (i = 0; i < fontcount; i++) {
	 sibling = XtNameToWidget(parent, fonts[i].family);
	 if (sibling != widget)
	    XtSetValues(sibling, args, 1);
      }
      mtest = &Fonts[3];   /* so that mtest->func has correct value below */
   }
   else if (mtest == menu + menulength) return;  /* something went wrong? */

   for (mitem = menu; mitem < menu + menulength; mitem++) {
      sibling = XtNameToWidget(parent, mitem->name);
      if (mitem->func == mtest->func)
         XtSetValues(sibling, args, 1);
   }

   /* Now set the currently pushed widget */

   XtSetArg(args[0], XtNsetMark, True);
   XtSetValues(widget, args, 1);
}

/*--------------------*/
/* Toggle a menu item */
/*--------------------*/

void toggle(Widget w, Boolean *boolvalue, caddr_t calldata)
{
   Arg	wargs[1];

   *boolvalue = !(*boolvalue);
   XtSetArg(wargs[0], XtNsetMark, *boolvalue);
   XtSetValues(w, wargs, 1);
   drawarea(w, NULL, NULL);
}

/*---------------------------------------------------------------*/
/* Some Xcircuit routines using toggle and toggleexcl, 		 */
/* put here because they reference the menu structures directly. */
/*---------------------------------------------------------------*/

void setcolorscheme(Boolean boolvalue)
{
   if (boolvalue) {
      BBOXCOLOR = appdata.bboxpix;
      AUXCOLOR = appdata.auxpix;
      BARCOLOR = appdata.barpix;
      OFFBUTTONCOLOR = appdata.buttonpix;
      SELECTCOLOR = appdata.selectpix;
      QUERYCOLOR = appdata.querypix;
      FILTERCOLOR = appdata.filterpix;
      GRIDCOLOR = appdata.gridpix;
      SNAPCOLOR = appdata.snappix;
      AXESCOLOR = appdata.axespix;
      BACKGROUND = appdata.bg;
      FOREGROUND = appdata.fg;
   }
   else {
      BBOXCOLOR = appdata.bboxpix2;
      AUXCOLOR = appdata.auxpix2;
      BARCOLOR = appdata.barpix2;
      OFFBUTTONCOLOR = appdata.buttonpix2;
      SELECTCOLOR = appdata.selectpix2;
      QUERYCOLOR = appdata.querypix2;
      FILTERCOLOR = appdata.filterpix2;
      GRIDCOLOR = appdata.gridpix2;
      SNAPCOLOR = appdata.snappix2;
      AXESCOLOR = appdata.axespix2;
      BACKGROUND = appdata.bg2;
      FOREGROUND = appdata.fg2;
   }

   makecursors();
}

/*----------------------------------------------------------------*/
/* Invert the color scheme used for the background/foreground	  */
/*----------------------------------------------------------------*/

void inversecolor(Widget w, Boolean *boolvalue, caddr_t calldata)
{
   /* change color scheme */

   setcolorscheme(*boolvalue);

   /* toggle checkmark in menu on "Alt colors" */

   if (w == NULL) w = OptionsAltColorsButton;
   if (w != NULL) toggle(w, boolvalue, calldata);
   if (eventmode == NORMAL_MODE) XDefineCursor (dpy, win, CROSS);
}

/*----------------------------------------------------------------*/
/* Change menu selection for reported measurement units		  */
/*----------------------------------------------------------------*/

void togglegrid(u_short type)
{
   Widget button, bparent = XtParent(GridtypedisplayDecimalInchesButton);

   if (type == CM) button = XtNameToWidget(bparent, "Centimeters");
   else if (type == FRAC_INCH) button = XtNameToWidget(bparent, "Fractional Inches");
   else if (type == DEC_INCH) button = XtNameToWidget(bparent, "Decimal Inches");
   else button = XtNameToWidget(bparent, "Coordinates");
   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   W1printf(" ");
}

/*----------------------------------------------------------------*/
/* Set the default reported grid units to inches or centimeters   */
/*----------------------------------------------------------------*/

void setgridtype(char *string)
{
   Widget button, bparent = XtParent(GridtypedisplayDecimalInchesButton);

   if (!strcmp(string, "inchscale")) {
      button = XtNameToWidget(bparent, "Fractional Inches");
      getgridtype(button, FRAC_INCH, NULL);
   }
   else if (!strcmp(string, "cmscale")) {
      button = XtNameToWidget(bparent, "Centimeters");
      getgridtype(button, CM, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Called by setgridtype() to complete setting the reported 	  */
/* measurement units						  */
/*----------------------------------------------------------------*/

void getgridtype(Widget button, u_int value, caddr_t calldata)
{
   short oldtype = xobjs.pagelist[areastruct.page]->coordstyle;
   float scalefac = getpsscale(1.0, areastruct.page) / INCHSCALE;

   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   xobjs.pagelist[areastruct.page]->coordstyle = (short)value;

   switch(value) {
      case FRAC_INCH: case DEC_INCH:
	 if (oldtype == CM) {
            xobjs.pagelist[areastruct.page]->outscale *= scalefac;
	    xobjs.pagelist[areastruct.page]->pagesize.x = 612;
	    xobjs.pagelist[areastruct.page]->pagesize.y = 792; /* letter */
	 }
	 break;
      case CM:
	 if (oldtype != CM) {
            xobjs.pagelist[areastruct.page]->outscale *= scalefac;
	    xobjs.pagelist[areastruct.page]->pagesize.x = 595;
	    xobjs.pagelist[areastruct.page]->pagesize.y = 842; /* A4 */
	 }
	 break;
   }
   if (oldtype != xobjs.pagelist[areastruct.page]->coordstyle) {
      drawarea(NULL, NULL, NULL);
      W1printf(" ");
   }
}

/*------------------------------------------------------*/
/* Make new library, add new button to the "Libraries"	*/
/* cascaded menu, compose the library page, update the  */
/* library directory, and go to that library page.	*/
/*------------------------------------------------------*/

void newlibrary(Widget w, caddr_t clientdata, caddr_t calldata)
{
   int libnum = createlibrary();
   startcatalog(w, libnum, NULL);
}

/*----------------------------------------------*/
/* Find an empty library, and return its page	*/
/* number if it exists.  Otherwise, return -1.	*/
/* This search does not include the so-called	*/
/* "User Library" (last library in list).	*/
/*----------------------------------------------*/

int findemptylib()
{
  int i;

  for (i = 0; i < xobjs.numlibs - 1; i++) {
     if (xobjs.userlibs[i].number == 0)
	return i;
  }
  return -1;
}

/*----------------------------------------------*/
/* Make new library and add a new button to the */
/* "Libraries" cascaded menu.			*/
/*----------------------------------------------*/

int createlibrary()
{
   Widget libmenu, newbutton, oldbutton;
   Arg wargs[2];
   char libstring[20];
   int libnum;

   /* If there's an empty library, return its number */
   if ((libnum = findemptylib()) >= 0) return (libnum + LIBRARY);
   libnum = (xobjs.numlibs++) + LIBRARY;
   xobjs.libtop = (objectptr *)realloc(xobjs.libtop, (libnum + 1)
		* sizeof(objectptr));
   xobjs.libtop[libnum] = xobjs.libtop[libnum - 1];
   libnum--;
   xobjs.libtop[libnum] = (objectptr) malloc(sizeof(object));
   initmem(xobjs.libtop[libnum]);
   sprintf(xobjs.libtop[libnum]->name, "Library %d", libnum - LIBRARY + 1);

   /* Create the library */

   xobjs.userlibs = (Library *) realloc(xobjs.userlibs, xobjs.numlibs
	* sizeof(Library));
   xobjs.userlibs[libnum + 1 - LIBRARY] = xobjs.userlibs[libnum - LIBRARY];
   xobjs.userlibs[libnum - LIBRARY].library = (objectptr *) malloc(sizeof(objectptr));
   xobjs.userlibs[libnum - LIBRARY].number = 0;

   /* Previously last button becomes new library pointer */

   oldbutton = GotoLibraryLibrary2Button;
   XtRemoveAllCallbacks (oldbutton, XtNselect);
   XtAddCallback (oldbutton, XtNselect, (XtCallbackProc)startcatalog,
	Number(libnum));
   XtSetArg(wargs[0], XtNlabel, xobjs.libtop[libnum]->name);
   XtSetValues(oldbutton, wargs, 1);

   /* Make new entry in the menu to replace the User Library button */
   /* Widget name is unique so button can be found later.  Label is */
   /* always set to "User Library"				    */

   sprintf(libstring, "Library %d", libnum - LIBRARY + 2);
   libmenu = XtParent(GotoLibraryAddNewLibraryButton);
   XtSetArg(wargs[0], XtNfont, appdata.xcfont);
   XtSetArg(wargs[1], XtNlabel, "User Library");
   newbutton = XtCreateWidget(libstring, XwmenubuttonWidgetClass,
         libmenu, wargs, 2);
   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)startcatalog,
	Number(libnum + 1));
   XtManageChild(newbutton);
   GotoLibraryLibrary2Button = newbutton;

   /* Update the library directory to include the new page */

   composelib(LIBLIB);

   return libnum;
}

/*----------------------------------------------*/
/* Make new page; goto that page		*/
/* (wrapper for routine events.c:newpage())	*/
/*----------------------------------------------*/

void newpagemenu(Widget w, u_int value, caddr_t nulldata)
{
   newpage((short)value);
}

/*--------------------------------------------------------------*/
/* Routine called by newpage() if new button needs to be made 	*/
/* to add to the "Pages" cascaded menu.				*/
/*--------------------------------------------------------------*/

void makepagebutton()
{
   Widget pagemenu, newbutton;
   Arg wargs[1];
   char pagestring[10];

   /* make new entry in the menu */

   pagemenu = XtParent(GotoPageAddNewPageButton);
   XtSetArg(wargs[0], XtNfont, appdata.xcfont);
   sprintf(pagestring, "Page %d", xobjs.pages);
   newbutton = XtCreateWidget(pagestring, XwmenubuttonWidgetClass,
         pagemenu, wargs, 1);
   XtAddCallback (newbutton, XtNselect, (XtCallbackProc)newpagemenu,
	Number(xobjs.pages - 1));
   XtManageChild(newbutton);

   /* Update the page directory */

   composelib(PAGELIB);
}

/*----------------------------------------------------------------*/
/* Find the Page menu button associated with the page number	  */
/* (passed parameter) and set the label of that button to the	  */
/* object name (= page label)					  */
/*----------------------------------------------------------------*/

void renamepage(short pagenumber)
{
   Arg wargs[1];
   Widget parent = XtParent(GotoPageAddNewPageButton);
   Widget button;
   char bname[10];

   sprintf(bname, "Page %d", pagenumber + 1);
   button = XtNameToWidget(parent, bname);

   if (button != NULL) {
      if (xobjs.pagelist[pagenumber]->pageobj->name != NULL)
         XtSetArg(wargs[0], XtNlabel, xobjs.pagelist[pagenumber]->pageobj->name);
      else
         XtSetArg(wargs[0], XtNlabel, bname);
      XtSetValues(button, wargs, 1);
   }
   else
      printf("Error:  No Button Widget named \"%9s\"\n", bname);
}

/*--------------------------------------------------------------*/
/* Same routine as above, for Library page menu buttons		*/
/*--------------------------------------------------------------*/

void renamelib(short libnumber)
{
   Arg wargs[1];
   Widget parent = XtParent(GotoLibraryAddNewLibraryButton);
   Widget button;
   char bname[13];

   sprintf(bname, "Library %d", libnumber - LIBRARY + 1);
   button = XtNameToWidget(parent, bname);

   if (button != NULL) {
      if (xobjs.libtop[libnumber]->name != NULL)
         XtSetArg(wargs[0], XtNlabel, xobjs.libtop[libnumber]->name);
      else
         XtSetArg(wargs[0], XtNlabel, bname);
      XtSetValues(button, wargs, 1);
   }
   else
      printf("Error:  No Button Widget named \"%12s\"\n", bname);
}

/*----------------------------------------------*/
/* Set Poly and Arc line styles and fill styles */
/*----------------------------------------------*/

#define BORDERS  (NOBORDER | DOTTED | DASHED)
#define ALLFILLS (FILLSOLID | FILLED)

/*----------------------------------------------*/
/* Set the menu marks according to style value  */
/*----------------------------------------------*/

void setstylemarks(Widget w, u_int value, Boolean boolval)
{
   Arg	wargs[1];

   /* if widget is unknown, find button for each part */

   if (value & UNCLOSED) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & OPAQUE) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & BBOX) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & BORDERS || value & ALLFILLS || value == 0) {
      toggleexcl(w, BorderStyles, XtNumber(BorderStyles));
      toggleexcl(w, Stipples, XtNumber(Stipples));
   }
}

/*--------------------------------------------------------------*/
/* Set the menu checkmarks on the color menu			*/
/*--------------------------------------------------------------*/

void setcolormark(int colorval)
{
   Arg args[1];
   Widget w = NULL;
   short i;

   if (colorval == DEFAULTCOLOR)
      w = ColorInheritColorButton;
   else {
      for (i = 0; i < number_colors; i++)
         if (colorlist[i].color.pixel == colorval) {
            w = colorlist[i].cbutton;
	    break;
         }
   }

   /* Remove mark from all menu items */

   XtSetArg(args[0], XtNsetMark, False);
   for (i = 0; i < number_colors; i++)
      XtSetValues(colorlist[i].cbutton, args, 1);
   XtSetValues(ColorInheritColorButton, args, 1);

   /* Add mark to the menu button for the chosen color */

   if (w != (Widget)NULL) {
      overdrawpixmap(w);
      XtSetArg(args[0], XtNsetMark, True);
      XtSetValues(w, args, 1);
   }
}

/*----------------------------------------------------------------*/
/* Set the checkmarks on the element styles menu		  */
/*----------------------------------------------------------------*/

void setallstylemarks(short styleval)
{
   Widget w;

   w = BorderClosedButton;
   setstylemarks(w, UNCLOSED, !(styleval & UNCLOSED));
   w = FillOpaqueButton;
   setstylemarks(w, OPAQUE, (styleval & OPAQUE) > 0);
   w = BorderBoundingBoxButton;
   setstylemarks(w, BBOX, (styleval & BBOX) > 0);

   if (styleval & NOBORDER)
      w = BorderUnborderedButton;
   else if (styleval & DASHED)
      w = BorderDashedButton;
   else if (styleval & DOTTED)
      w = BorderDottedButton;
   else
      w = BorderSolidButton;
   setstylemarks(w, BORDERS, 0);

   if (!(styleval & FILLED))
      w = FillWhiteButton;
   else {
      styleval &= FILLSOLID;
      styleval /= STIP0;
      switch(styleval) {
	 case 0: w = FillGray87Button; break;
	 case 1: w = FillGray75Button; break;
	 case 2: w = FillGray62Button; break;
	 case 3: w = FillGray50Button; break;
	 case 4: w = FillGray37Button; break;
	 case 5: w = FillGray25Button; break;
	 case 6: w = FillGray12Button; break;
	 case 7: w = FillBlackButton;  break;
      }
   }
   setstylemarks(w, ALLFILLS, 0);
}

/*--------------------------------------------------------------*/
/* Check for a bounding box polygon				*/
/*--------------------------------------------------------------*/

polyptr checkforbbox(objectptr localdata)
{
   polyptr *cbbox;

   for (cbbox = (polyptr *)localdata->plist; cbbox <
	(polyptr *)localdata->plist + localdata->parts; cbbox++)
      if ((*cbbox)->type == POLYGON)
         if ((*cbbox)->style & BBOX) return *cbbox;

   return NULL;
}

/*--------------------------------------------------------------*/
/* Get a value for element fill style				*/
/*--------------------------------------------------------------*/

void getfill(Widget w, u_int value, caddr_t nulldata)
{
   Boolean boolval, selected = False;
   short *sstyle;

   if (areastruct.selects > 0) {
      if (value & BBOX) {
	 polyptr ckp;
	 if (areastruct.selects != 1) {
	    Wprintf("Choose only one polygon to be the bounding box");
	    return;
	 }
	 else if (SELECTTYPE(areastruct.selectlist) != POLYGON) {
	    Wprintf("Bounding box can only be a polygon");
	    return;
	 }
	 else if (((ckp = checkforbbox(objectdata)) != NULL) &&
		(ckp != SELTOPOLY(areastruct.selectlist))) {
	    Wprintf("Only one bounding box allowed per page");
	    return;
	 }
      }

      for (sstyle = areastruct.selectlist; sstyle < areastruct.selectlist
	   + areastruct.selects; sstyle++) {
	 short stype = SELECTTYPE(sstyle);
	 if (stype == ARC || stype == POLYGON || stype == SPLINE || stype == PATH) {
	    short *estyle;
	    switch (stype) {
	       case ARC:
	          estyle = &((SELTOARC(sstyle))->style);
		  break;
	       case SPLINE:
	          estyle = &((SELTOSPLINE(sstyle))->style);
		  break;
	       case POLYGON:
	          estyle = &((SELTOPOLY(sstyle))->style);
		  break;
	       case PATH:
	          estyle = &((SELTOPATH(sstyle))->style);
		  break;
	    }

	    XSetFunction(dpy, areastruct.gc, GXcopy);
	    XSetForeground(dpy, areastruct.gc, BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    if (value & UNCLOSED) {
	       boolval = (*estyle & UNCLOSED) ? True : False;
	       *estyle &= ~UNCLOSED;
	       *estyle |= (boolval ? NORMAL : UNCLOSED);
	    }
	    else if (value & OPAQUE) {
	       boolval = (*estyle & OPAQUE) ? False : True;
	       *estyle &= ~OPAQUE;
	       *estyle |= (boolval ? OPAQUE : NORMAL);
	    }
	    else if (value & BBOX) {
	       boolval = (*estyle & BBOX) ? False : True;
	       *estyle &= ~BBOX;
	       *estyle |= (boolval ? BBOX : NORMAL);
	       (SELTOPOLY(sstyle))->color = boolval ? BBOXCOLOR : DEFAULTCOLOR;
	    }
	    else {
	       if (value & BORDERS || value == NORMAL) *estyle &= ~BORDERS;
	       else *estyle &= ~ALLFILLS;
	       if (value != FILLSOLID) *estyle |= value;
	       if ((*estyle & NOBORDER) && !(*estyle & FILLED)) {
		  Wprintf("Must have either a border or filler");
		  *estyle &= ~NOBORDER;
		  value &= ~NOBORDER;
	       }
	    }
	    XSetFunction(dpy, areastruct.gc, GXxor);
	    XSetForeground(dpy, areastruct.gc, SELECTCOLOR ^ BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    selected = True;
	 }
      }
   }
   if (!selected) {
      if (value & BBOX) {
	 Wprintf("Cannot set default style to Bounding Box");
	 areastruct.style &= ~BBOX;
	 return;
      }
      else if (value & UNCLOSED) {
         boolval = (areastruct.style & UNCLOSED) ? True : False;
	 areastruct.style &= ~UNCLOSED;
	 areastruct.style |= (boolval ? NORMAL : UNCLOSED);
      }
      else if (value & OPAQUE) {
         boolval = (areastruct.style & OPAQUE) ? False : True;
	 areastruct.style &= ~OPAQUE;
	 areastruct.style |= (boolval ? OPAQUE : NORMAL);
      }
      else {
	 short savestyle = areastruct.style;

         if (value & BORDERS || value == NORMAL)
	    areastruct.style &= ~BORDERS;
         else areastruct.style &= ~ALLFILLS;
         if (value != FILLSOLID) areastruct.style |= value;

	 if ((areastruct.style & NOBORDER) && !(areastruct.style & FILLED)) {
	    Wprintf("Must have either a border or filler");
	    areastruct.style = savestyle;
	    return;
	 }
      }
      overdrawpixmap(w);
   }
   if (w)
      setstylemarks(w, value, boolval);
   else
      setallstylemarks(areastruct.style);

   objectdeselect();
}

/*----------------------------------------------------------------*/
/* The following four routines are all wrappers for getfill(),	  */
/* used in menudefs to differentiate between sections, each of    */
/* which has settings independent of the others.		  */
/*----------------------------------------------------------------*/

void makebbox(Widget w, u_int value, caddr_t calldata)
{
   getfill(w, value, NULL);
}

/*----------------------------------------------------------------*/

void getset(Widget w, u_int value, caddr_t calldata)
{
   getfill(w, value, NULL);
}

/*----------------------------------------------------------------*/

void getline(Widget w, u_int value, caddr_t calldata)
{
   getfill(w, value, NULL);
}

/*----------------------------------------------------------------*/

void getopaque(Widget w, u_int value, caddr_t calldata)
{
   getfill(w, value, NULL);
}
   
/*-----------------------------------------------*/
/* Set the color value for all selected elements */
/*-----------------------------------------------*/

void setcolor(Widget w, u_int value, caddr_t calldata)
{
   short *scolor;
   int *ecolor, cindex, cval;
   Arg wargs[1];
   Boolean selected = False;
   stringpart *strptr, *nextptr;

   /* Get the color index value from the menu button widget itself */

   if (value == 1)
      cindex = cval = -1;
   else {
      XtSetArg(wargs[0], XtNrectColor, &cval);
      XtGetValues(w, wargs, 1);
      for (cindex = 0; cindex < number_colors; cindex++)
         if (colorlist[cindex].color.pixel == cval)
	    break;
      if (cindex == number_colors) {
	 Wprintf("Error: No such color!");
	 return;
      }
   }
   
   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      labelptr curlabel = TOLABEL(EDITPART);
      strptr = findstringpart(textpos - 1, NULL, curlabel->string, NORMINST);
      nextptr = findstringpart(textpos, NULL, curlabel->string, NORMINST);
      if (strptr->type == FONT_COLOR) {
	 undrawtext(curlabel);
	 strptr->data.color = cindex;
	 redrawtext(curlabel);
      }
      else if (nextptr && nextptr->type == FONT_COLOR) {
	    undrawtext(curlabel);
	    nextptr->data.color = cindex;
	    redrawtext(curlabel);
      }
      else {
	 sprintf(_STR2, "%d", cindex);
	 labeltext(FONT_COLOR, (char *)&cindex);
      }
   }

   else if (areastruct.selects > 0) {
      for (scolor = areastruct.selectlist; scolor < areastruct.selectlist
	   + areastruct.selects; scolor++) {
	 ecolor = &(SELTOCOLOR(scolor));

	 *ecolor = cval;
	 selected = True;
      }
   }

   setcolormark(cindex);
   if (!selected) {
      if (eventmode != TEXT2_MODE && eventmode != TEXT3_MODE)
         areastruct.color = cval;      
      overdrawpixmap(w);
   }

   objectdeselect();
}

/*----------------------------------------------------------------*/
/* Parse a new color entry and add it to the color list.	  */
/*----------------------------------------------------------------*/

void setnewcolor(Widget w, caddr_t nullptr)
{
   int ccolor, red, green, blue;
   char *ppos, *cpos;

   ppos = strchr(_STR2, '#');
   cpos = strchr(_STR2, ',');

   if (cpos != NULL || ppos != NULL) {
      if (cpos != NULL || strlen(ppos + 1) == 6) {
	 if (cpos != NULL)
            sscanf(_STR2, "%d, %d, %d", &red, &green, &blue);
	 else
            sscanf(ppos + 1, "%2x%2x%2x", &red, &green, &blue);
	 red *= 256;
	 green *= 256;
	 blue *= 256;
      }
      else if (sscanf(ppos + 1, "%4x%4x%4x", &red, &green, &blue) != 3) {
	 Wprintf("Bad color entry.  Use #rrggbb");
	 return;
      }
      ccolor = rgb_alloccolor(red, green, blue);
   }
   else
      ccolor = xc_alloccolor(_STR2);

   addnewcolorentry(ccolor);
}

/*----------------------------------------------------------------*/
/* Generate popup dialog for adding a new color name or RGB value */
/*----------------------------------------------------------------*/

void addnewcolor(Widget w, caddr_t clientdata, caddr_t calldata)
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, w, addnewcolor, NULL);
   popupprompt(w, "Enter color name or #rgb or r,g,b:", "\0", setnewcolor,
	savebutton, NULL);
}

/*------------------------------------------------------*/
/* Set checkmarks on label style menu 			*/
/* fvalue is for font, jvalue is for justification	*/
/*------------------------------------------------------*/

void setfontmarks(short fvalue, short jvalue)
{
   Widget w;
   Arg wargs[1];

   if ((fvalue >= 0) && (fontcount > 0)) {
      switch(fonts[fvalue].flags & 0x03) {
         case 0: w = StyleNormalButton; break;
         case 1: w = StyleBoldButton; break;
         case 2: w = StyleItalicButton; break;
         case 3: w = StyleBoldItalicButton; break;
      }
      toggleexcl(w, FontStyles, XtNumber(FontStyles));

      switch((fonts[fvalue].flags & 0xf80) >> 7) {
         case 0: w = EncodingStandardButton; break;
         case 2: w = EncodingISOLatin1Button; break;
	 default: w = NULL;
      }
      if (w != NULL) toggleexcl(w, FontEncodings, XtNumber(FontEncodings));

      togglefontmark(fvalue);
   }
   if (jvalue >= 0) {
      switch(jvalue & (NOTLEFT | RIGHT)) {
         case NORMAL: w = JustificationLeftJustifiedButton; break;
         case NOTLEFT: w = JustificationCenterJustifiedButton; break;
         case RIGHT|NOTLEFT: w = JustificationRightJustifiedButton; break;
      }
      toggleexcl(w, Justifs, XtNumber(Justifs));

      switch(jvalue & (NOTBOTTOM | TOP)) {
         case NORMAL: w = JustificationBottomJustifiedButton; break;
         case NOTBOTTOM: w = JustificationMiddleJustifiedButton; break;
         case TOP|NOTBOTTOM: w = JustificationTopJustifiedButton; break;
      }

      toggleexcl(w, Justifs, XtNumber(Justifs));

      /* Flip Invariance property */
      w = JustificationFlipInvariantButton;
      if (jvalue & FLIPINV)
         XtSetArg(wargs[0], XtNsetMark, True);
      else
         XtSetArg(wargs[0], XtNsetMark, False);
      XtSetValues(w, wargs, 1);

#ifdef SCHEMA
      /* Pin visibility property */
      w = NetlistPinVisibilityButton;
      if (jvalue & PINVISIBLE)
         XtSetArg(wargs[0], XtNsetMark, True);
      else
         XtSetArg(wargs[0], XtNsetMark, False);
      XtSetValues(w, wargs, 1);
#endif
   }
}

/*--------------------------------------------------------------*/
/* Parameterize a label string (wrapper for parameterize()).	*/
/*--------------------------------------------------------------*/

void startparam(Widget w, u_int value, caddr_t calldata)
{
   genericptr *settext;

   if (value == LABEL) {
      if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
         settext = (genericptr *)EDITPART;
         makeparam(TOLABEL(settext));
         objectdeselect();
      }
      else if (checkselect(LABEL)) parameterize(value);
      else
         eventmode = LPARAM_MODE;
   }
   else {
      if ((eventmode == NORMAL_MODE) && (areastruct.selects == 0)) { 
         eventmode = IPARAM_MODE;
         Wprintf("Click on object to parameterize.");
      }      
      else parameterize(value);
   }
}

/*--------------------------------------------------------------*/
/* Unparameterize a label string (wrapper for parameterize()).	*/
/*--------------------------------------------------------------*/

void startunparam(Widget w, u_int value, caddr_t calldata)
{
   if (areastruct.selects == 0) {
      if (value == LABEL)
         eventmode = ULPARAM_MODE;
      else
         eventmode = UIPARAM_MODE;
      Wprintf("Select element or substring to remove parameters.");
   }
   else
      unparameterize(value);
   objectdeselect();
}

/*----------------------------------------------------------------*/
/* Set checkmarks on font menu according to the global defaults	  */
/*----------------------------------------------------------------*/

void setdefaultfontmarks()
{
   setfontmarks(areastruct.psfont, areastruct.justify);
}

/*----------------------------------------------------------------*/
/* Change flip invariance on the label passed as 2nd parameter	  */
/*----------------------------------------------------------------*/

void doflipflip(Widget w, labelptr settext)
{
   short newjust;
   Boolean boolval;
   u_short value;

   if (settext != NULL) {
      boolval = (settext->justify & FLIPINV) ? True : False;
      value = boolval ? 0 : FLIPINV;
      newjust = (settext->justify & (~FLIPINV)) | value;

      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
   }
   else {
      boolval = (areastruct.justify & FLIPINV) ? True : False;
      value = boolval ? 0 : FLIPINV;
      newjust = (areastruct.justify & (~FLIPINV)) | value;
      areastruct.justify = newjust;
   }
   if (w != NULL) toggle(w, &boolval, NULL);
}

/*----------------------------------------------------------------*/
/* Change pin visibility on the label passed as 2nd parameter	  */
/*----------------------------------------------------------------*/

void dopinvisible(Widget w, labelptr settext)
{
   short newjust;
   Boolean boolval;
   u_short value;

   if (settext != NULL && settext->pin) {
      boolval = (settext->justify & PINVISIBLE) ? True : False;
      value = boolval ? 0 : PINVISIBLE;
      newjust = (settext->justify & (~PINVISIBLE)) | value;

      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
      if (w != NULL) toggle(w, &boolval, NULL);
   }
}

/*----------------------------------------------------------------*/
/* Set the justification for the label passed as 3rd parameter	  */
/*----------------------------------------------------------------*/

void setjust(Widget w, u_int value, labelptr settext, short mode)
{
   short newjust;

   if (settext != NULL) {

      if (mode == 1)
         newjust = (settext->justify & (PINVISIBLE | FLIPINV | TOP | NOTBOTTOM))
			| value;
      else
         newjust = (settext->justify & (PINVISIBLE | FLIPINV | RIGHT | NOTLEFT))
			| value;

      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
   }
   else {
      if (mode == 1)
         newjust = (areastruct.justify & (PINVISIBLE | FLIPINV | TOP | NOTBOTTOM))
			| value;
      else
         newjust = (areastruct.justify & (PINVISIBLE | FLIPINV | RIGHT | NOTLEFT))
			| value;
      areastruct.justify = newjust;
   }
   if (w != NULL) toggleexcl(w, Justifs, XtNumber(Justifs));
}

/*----------------------------------------------------------------*/
/* Set vertical justification (top, middle, bottom) on all	  */
/* selected labels						  */
/*----------------------------------------------------------------*/

void setvjust(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setjust(w, value, settext, 2);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 2);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 2);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set horizontal justification (left, center, right) on all	  */
/* selected labels						  */
/*----------------------------------------------------------------*/

void sethjust(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setjust(w, value, settext, 1);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 1);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 1);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set flip invariance on all selected labels			  */ 
/*----------------------------------------------------------------*/

void setflipinv(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      doflipflip(w, settext);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            doflipflip(NULL, settext);
         }
      }
      if (labelcount == 0) doflipflip(w, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set pin visibility on all selected pins			  */ 
/*----------------------------------------------------------------*/

void setpinvisible(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      if (settext->pin)
	 dopinvisible(w, settext);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
	    if (settext->pin)
               dopinvisible(NULL, settext);
         }
      }
      objectdeselect();
   }
}

/*---------------------------*/
/* Set polygon editing style */
/*---------------------------*/

void boxedit(Widget w, u_int value, caddr_t nulldata)
{
   if (w == NULL) {
      switch (value) {
         case MANHATTAN: w = PolygonEditManhattanBoxEditButton; break;
         case RHOMBOIDX: w = PolygonEditRhomboidXButton; break;
	 case RHOMBOIDY: w = PolygonEditRhomboidYButton; break;
	 case RHOMBOIDA: w = PolygonEditRhomboidAButton; break;
	 case NORMAL: w = PolygonEditNormalButton; break;
      }
   }

   if (areastruct.boxedit == value) return;

   toggleexcl(w, BoxEditStyles, XtNumber(BoxEditStyles));
   areastruct.boxedit = value;
}

/*----------------------------------------------------------------*/
/* Pick up font name from _STR2 and pass it to loadfontfile()	  */
/*----------------------------------------------------------------*/

void locloadfont(Widget w, char *value)
{
   loadfontfile(_STR2);
   free(value);
}

/*----------------------------------------------------*/
/* Generate popup dialog for entering a new font name */
/*----------------------------------------------------*/

void addnewfont(Widget w, caddr_t clientdata, caddr_t calldata)
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   char *tempstr = malloc(2 * sizeof(char));
   tempstr[0] = '\0';
   getgeneric(savebutton, w, addnewfont, tempstr);
   popupprompt(w, "Enter font name:", tempstr, locloadfont, savebutton, NULL);
}

/*----------------------------------------------------------------*/
/* Find the best matching font given new font, style, or encoding */
/*----------------------------------------------------------------*/
/* newfont, style, or encoding = -1 when not applicable		  */
/* Return the number of the best matching font.			  */  
/*----------------------------------------------------------------*/

short findbestfont(short curfont, short newfont, short style, short encoding) {

   char *newfamily;
   short i, newstyle, newenc;

   if (fontcount == 0) return -1;

   if (newfont < 0)
      newfamily = fonts[curfont].family;
   else if (newfont >= fontcount) {	/* move to next font family */
      short newidx;
      newfont = 0;
      while (strcmp(fonts[fontnumbers[newfont]].family, fonts[curfont].family))
	 newfont++;
      newidx = (newfont + 1) % nfontnumbers;
      while (!strcmp(fonts[curfont].family, fonts[fontnumbers[newidx]].family) &&
		newfont != newidx)
	 newidx = (newidx + 1) % nfontnumbers;
      newfamily = fonts[fontnumbers[newidx]].family;
      newfont = fontnumbers[newidx];
   }
   else
      newfamily = fonts[newfont].family;

   if (style < 0) 
      newstyle = fonts[curfont].flags & 0x03;
   else
      newstyle = style & 0x03;
 
   if (encoding < 0)
      newenc = fonts[curfont].flags & 0xf80;
   else
      newenc = encoding << 7;

   /* Best position is a match on all conditions */

   for (i = 0; i < fontcount; i++)
      if ((!strcmp(fonts[i].family, newfamily)) &&
		((fonts[i].flags & 0x03) == newstyle) &&
	 	((fonts[i].flags & 0xf80) == newenc))
	 return i;

   /* Fallback position 1:  Match requested property and one other. 	*/
   /* order of preference:  					   	*/
   /*   Requested property	Priority 1	Priority 2		*/
   /*		Font		  Style		  Encoding		*/
   /*		Style		  Font		  (none)		*/
   /*		Encoding	  Font		  (none)		*/

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if ((!strcmp(fonts[i].family, newfamily)) &&
		(fonts[i].flags & 0x03) == newstyle) return i;
      }
      else if (style >= 0) {
	 if (((fonts[i].flags & 0x03) == newstyle) &&
	 	(!strcmp(fonts[i].family, newfamily))) return i;
      }
      else if (encoding >= 0) {
	 if (((fonts[i].flags & 0xf80) == newenc) &&
	 	(!strcmp(fonts[i].family, newfamily))) return i;
      }
   }

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if ((!strcmp(fonts[i].family, newfamily)) &&
	    ((fonts[i].flags & 0xf80) >> 7) == newenc) return i;
      }
   }

   /* Fallback position 2:  Match only the requested property.  */
   /* For font selection only:  Don't want to select a new font */
   /* just because a certain style or encoding wasn't available.*/

   for (i = 0; i < fontcount; i++) {
      if (newfont >= 0) {
	 if (!strcmp(fonts[i].family, newfamily)) return i;
      }
   }      

   /* Failure to find matching font property */

   if (style >= 0) {
      sprintf(_STR, "Font %s not available in this style", newfamily);
   }
   else {
      sprintf(_STR, "Font %s not available in this encoding", newfamily);
   }
   Wprintf(_STR);
   return (-1);
}

/*----------------------------------------------------------------*/
/* Set the font.  This depends on the system state as to whether  */
/* font is set at current position in label, for an entire label, */
/* or as the default font to begin new labels.			  */
/*----------------------------------------------------------------*/

void setfontval(Widget w, u_int value, labelptr settext)
{
   int newfont;
   short i, tc;
   stringpart *strptr;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < stringlength(settext->string, True, NORMINST)) {
	 strptr = findstringpart(textpos - 1, NULL, settext->string, NORMINST);
	 if (strptr->type == FONT_NAME) {
	    tc = strptr->data.font;
	    i = findbestfont(tc, (short)value, -1, -1);
	    if (i >= 0) {
	       undrawtext(settext);
	       strptr->data.font = i;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
      		  togglefontmark(i);
	       }
	    }
	    return;
         }
      }

      /* otherwise, look for the last style used in the string */
      tc = findcurfont(textpos, settext->string, NORMINST);
   }
   else tc = areastruct.psfont;

   /* Font change command will always find a value since at least one	*/
   /* font has to exist for the font menu button to exist.		*/

   if ((newfont = (int)findbestfont(tc, (short)value, -1, -1)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      sprintf(_STR2, "%d", newfont);
      labeltext(FONT_NAME, (char *)&newfont);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }

   if (w != NULL) togglefontmark(newfont);

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontval()				  */
/*----------------------------------------------------------------*/

void setfont(Widget w, u_int value, caddr_t calldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontval(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontval(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontval(w, value, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set the style (bold, italic, normal) for the font, similarly   */
/* to the above routine setfontval().				  */
/*----------------------------------------------------------------*/

void setfontstyle(Widget w, u_int value, labelptr settext)
{
   int newfont;
   short i, tc;
   stringpart *strptr;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < stringlength(settext->string, True, NORMINST)) {
	 strptr = findstringpart(textpos - 1, NULL, settext->string, NORMINST);
	 if (strptr->type == FONT_NAME) {
	    tc = strptr->data.font;

	    /* find font which matches family and style, if available */

	    i = findbestfont(tc, -1, (short)value, -1);
	    if (i >= 0) {
	       undrawtext(settext);
	       strptr->data.font = i;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
	          toggleexcl(w, FontStyles, XtNumber(FontStyles));
	       }
	    }
            return;
         }
      }

      /* otherwise, look for the last font used in the string */
      tc = findcurfont(textpos - 2, settext->string, NORMINST);
   }
   else tc = areastruct.psfont;

   if ((newfont = (int)findbestfont(tc, -1, (short)value, -1)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      sprintf(_STR2, "%d", newfont);
      labeltext(FONT_NAME, (char *)&newfont);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }
   if (w != NULL) toggleexcl(w, FontStyles, XtNumber(FontStyles));

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontstyle()				  */
/*----------------------------------------------------------------*/

void fontstyle(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontstyle(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontstyle(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontstyle(w, value, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------------------------*/
/* Set the encoding (standard, ISO-Latin1, special) for the font, */
/* similarly to the above routine setfontval().			  */
/*----------------------------------------------------------------*/

void setfontencoding(Widget w, u_int value, labelptr settext)
{
   int newfont;
   short i, tc;
   stringpart *strptr;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0 || textpos < stringlength(settext->string, True, NORMINST)) {
	 strptr = findstringpart(textpos - 1, NULL, settext->string, NORMINST);
	 if (strptr->type == FONT_NAME) {
	    tc = strptr->data.font;

	    i = findbestfont(tc, -1, -1, (short)value);
	    if (i >= 0) {
	       undrawtext(settext);
	       strptr->data.font = i;
	       redrawtext(settext);
	       if (w != NULL) {
	          charreport(settext);
   	          toggleexcl(w, FontEncodings, XtNumber(FontEncodings));
	       }
	    }
	    return;
	 }
      }

      /* otherwise, look for the last style used in the string */
      tc = findcurfont(textpos - 2, settext->string, NORMINST);
   }
   else tc = areastruct.psfont;

   if ((newfont = (int)findbestfont(tc, -1, -1, (short)value)) < 0) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont].psname);
      sprintf(_STR2, "%d", newfont);
      labeltext(FONT_NAME, (char *)&newfont);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont].psname);
      areastruct.psfont = newfont;
   }

   if (w != NULL) toggleexcl(w, FontEncodings, XtNumber(FontEncodings));

   Wprintf(_STR);
}

/*----------------------------------------------------------------*/
/* Wrapper for routine setfontencoding()			  */
/*----------------------------------------------------------------*/

void fontencoding(Widget w, u_int value, caddr_t nulldata)
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == CATALOG_MODE || eventmode == FONTCAT_MODE ||
		eventmode == FONTCAT2_MODE) return;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)EDITPART);
      setfontencoding(w, value, settext);
      charreport(settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontencoding(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontencoding(w, value, NULL);
      else objectdeselect();
   }
}

/*----------------------------------------------*/
/* Generate the table of special characters	*/
/*----------------------------------------------*/

void dospecial()
{
   labelptr curlabel;
   int cfont;

   curlabel = TOLABEL(EDITPART);
   cfont = findcurfont(textpos, curlabel->string, NORMINST);
   composefontlib(cfont);
   startcatalog(NULL, FONTLIB, NULL);
}

/*-------------------------------------------------*/
/* Wrapper for labeltext when called from the menu */
/*-------------------------------------------------*/

void addtotext(Widget w, u_int value, caddr_t nulldata)
{
   if (eventmode != TEXT2_MODE && eventmode != TEXT3_MODE) return;
   if (value == (u_int)SPECIAL)
      dospecial();
   else
      labeltext((int)value, (char *)1);
}

/*----------------------------------------------------------*/
/* Position a popup menu directly beside the toolbar button */
/*----------------------------------------------------------*/

void position_popup(Widget toolbutton, Widget menubutton)
{
   int n = 0;
   Arg wargs[2];
   Position pz, pw, ph;
   int dx, dy;

   Widget cascade = XtParent(menubutton);
   Widget pshell = XtParent(XtParent(cascade));

   XtnSetArg(XtNheight, &pz);
   XtGetValues(toolbutton, wargs, n); n = 0;

   XtnSetArg(XtNwidth, &pw);
   XtnSetArg(XtNheight, &ph);
   XtGetValues(cascade, wargs, n); n = 0;

   dx = -pw - 6;
   dy = (pz - ph) >> 1;

   XwPostPopup(pshell, cascade, toolbutton, dx, dy);
}

/*------------------------------------------------------*/
/* Functions which pop up a menu cascade in sticky mode */
/*------------------------------------------------------*/

void border_popup(Widget w, caddr_t clientdata, caddr_t calldata)
{ 
   position_popup(w, BorderLinewidthButton);
}

/*-------------------------------------------------------------------------*/

void color_popup(Widget w, caddr_t clientdata, caddr_t calldata)
{ 
   position_popup(w, ColorAddNewColorButton);
}

/*-------------------------------------------------------------------------*/

void fill_popup(Widget w, caddr_t clientdata, caddr_t calldata)
{ 
   position_popup(w, FillOpaqueButton);
}

/*-------------------------------------------------------------------------*/
