#-------------------------------------------------------------------------
# Start of xcircuit GUI configuration file.
# This file is sourced by "xcircuit.tcl" (".wishrc")
#-------------------------------------------------------------------------

# This script sets up all the xcircuit windows and callback functions.
# The callback routines are in the shared object file xcwrap.so

#-------------------------------
# Resource setup
#-------------------------------

option add *xcircuit*activeBackground brown4
option add *xcircuit*foreground brown4
option add *xcircuit*activeForeground beige
option add *xcircuit*borderwidth 0
option add *xcircuit*font -*-times-bold-r-normal--14-*

option add *xcircuit.menubar.filebutton*background beige
option add *xcircuit.menubar.editbutton*background beige
option add *xcircuit.menubar.textbutton*background beige
option add *xcircuit.menubar.optionsbutton*background beige
option add *xcircuit.menubar.windowbutton*background beige
option add *xcircuit.menubar.netlistbutton*background beige

option add *filelist*activeBackground brown4
option add *filelist*foreground brown4
option add *filelist*activeForeground beige
option add *filelist*borderwidth 0
option add *filelist*font -*-times-bold-r-normal--14-*
option add *filelist*txt*font -*-courier-bold-r-normal--14-*

option add *output*activeBackground brown4
option add *output*foreground brown4
option add *output*activeForeground beige
option add *output*borderwidth 0
option add *output*font -*-times-bold-r-normal--14-*
option add *output*txt*font -*-courier-bold-r-normal--14-*

option add *output.textent.butp.psmenu*background beige
option add *output.textent.txtf.sizb.sizemenu*background beige
option add *output.textent.buto.orientmenu*background beige

option add *help*activeBackground brown4
option add *help*foreground brown4
option add *help*activeForeground beige
option add *help*borderwidth 0
option add *help*font -*-times-bold-r-normal--14-*
option add *help*txt*font -*-times-courier-r-normal--14-*

option add *dialog*activeBackground brown4
option add *dialog*foreground brown4
option add *dialog*activeForeground beige
option add *dialog*borderwidth 0
option add *dialog*font -*-times-bold-r-normal--14-*
option add *dialog*txt*font -*-courier-bold-r-normal--14-*

#-------------------------------
# Main xcircuit drawing window
#-------------------------------

toplevel .xcircuit -width 1000 -height 800 -background red
wm title .xcircuit XCircuit

# This gets rid of the original "wish", in favor of our own window

if {[string range [wm title .] 0 3] == "wish"} {
   wm withdraw .
}

# All the internal frames

frame .xcircuit.menubar -background red
frame .xcircuit.infobar -background red
frame .xcircuit.mainframe -background red

pack propagate .xcircuit false

pack .xcircuit.menubar -side top -fill x -pady 1
pack .xcircuit.mainframe -side top -expand true -fill both
pack .xcircuit.infobar -side bottom -fill x -pady 1

frame .xcircuit.mainframe.mainarea -background red
frame .xcircuit.mainframe.toolbar -bg beige

pack .xcircuit.mainframe.toolbar -side right -fill y -padx 2
pack .xcircuit.mainframe.mainarea -expand true -fill both

simple .xcircuit.mainframe.mainarea.sbleft -width 13 -bg beige
simple .xcircuit.mainframe.mainarea.sbbottom -height 13 -bg beige
simple .xcircuit.mainframe.mainarea.drawing -bg white
simple .xcircuit.mainframe.mainarea.corner -width 13 -height 13 -bg black

# The drawing area and its scrollbars

grid .xcircuit.mainframe.mainarea.sbleft -row 0 -column 0 -sticky ns -padx 1
grid .xcircuit.mainframe.mainarea.sbbottom -row 1 -column 1 -sticky ew -pady 1
grid .xcircuit.mainframe.mainarea.drawing -row 0 -column 1 -sticky news
grid .xcircuit.mainframe.mainarea.corner -row 1 -column 0 -sticky news

grid rowconfigure .xcircuit.mainframe.mainarea 0 -weight 1
grid columnconfigure .xcircuit.mainframe.mainarea 1 -weight 1

# The top menu and message bar

menubutton .xcircuit.menubar.filebutton -text File \
	-menu .xcircuit.menubar.filebutton.filemenu
menubutton .xcircuit.menubar.editbutton -text Edit \
	-menu .xcircuit.menubar.editbutton.editmenu
menubutton .xcircuit.menubar.textbutton -text Text \
	-menu .xcircuit.menubar.textbutton.textmenu
menubutton .xcircuit.menubar.optionsbutton -text Options \
	-menu .xcircuit.menubar.optionsbutton.optionsmenu
menubutton .xcircuit.menubar.windowbutton -text Window \
	-menu .xcircuit.menubar.windowbutton.windowmenu
menubutton .xcircuit.menubar.netlistbutton -text Netlist \
	-menu .xcircuit.menubar.netlistbutton.netlistmenu 

pack .xcircuit.menubar.filebutton .xcircuit.menubar.editbutton \
	.xcircuit.menubar.textbutton .xcircuit.menubar.optionsbutton \
	.xcircuit.menubar.windowbutton .xcircuit.menubar.netlistbutton \
	-side left -padx 2 -ipadx 10 -pady 2

# The bottom message bar

label .xcircuit.menubar.message -text "Welcome to Xcircuit v3.1" \
	-background beige -justify left -anchor w
pack .xcircuit.menubar.message -side top -fill x -padx 2 -ipady 3 -pady 2

button .xcircuit.infobar.symb -text "Symbol" -bg gray30 -fg white
button .xcircuit.infobar.schem -text "Schematic" -bg red -fg white
label .xcircuit.infobar.message1 -text "Editing: Page 1" \
	-background beige
label .xcircuit.infobar.message2 -text "Grid 1/6 in : Snap 1/12 in" \
	-background beige -justify left -anchor w
pack .xcircuit.infobar.symb .xcircuit.infobar.schem \
	.xcircuit.infobar.message1 -side left -padx 2 -ipadx 6 -fill y
pack .xcircuit.infobar.message2 -padx 2 -ipadx 6 -expand true -fill both

#----------------------------------------------------------------------
# Support procedures for tag callbacks
#----------------------------------------------------------------------

# This procedure configures the output properties window according to
# the page mode (full or encapsulated)

proc xcircuit::setpstype {mode} {
   global pstype
   switch -- $mode {
      {eps} { .output.textent.butp configure -text "Embedded (EPS)"
	 grid remove .output.textent.but7
	 grid remove .output.textent.butf
	 grid remove .output.textent.txtf
      }
      {full} {.output.textent.butp configure -text "Full Page"
	 grid .output.textent.but7 -row 6 -column 3 -pady 5 -ipadx 10
	 grid .output.textent.butf -row 5 -column 2 -padx 10
	 grid .output.textent.txtf -row 6 -column 2 -sticky ew -padx 10
      }
   }
   ::set pstype $mode
   xcircuit::page encapsulation $mode
}

proc xcircuit::dolinks {} {
   global imultiple dmultiple
   set ilinks [xcircuit::page links independent]
   if {$ilinks > 1} {
      set imultiple 1
   } else {
      set imultiple 0
   }
   if {$ilinks == 1} { set plural ""} else { set plural "s"}
   .output.title.imulti configure -text "$ilinks schematic$plural"

   if {$dmultiple == 1} {
      set dlinks [xcircuit::page links dependent]
   } else {
      set dlinks 0
   }
   if {$dlinks == 1} { set plural ""} else { set plural "s"}
   .output.title.dmulti configure -text "$dlinks subcircuit$plural"
}

proc xcircuit::pageupdate { {subcommand "none"} } {
   global tcl_precision autofit orient imultiple dmultiple
   if {[info level] <= 1} {
     switch -- $subcommand {
	save {
	   .output.bbar.okay configure -text "Done"
	   .output.bbar.okay configure -command {wm withdraw .output}
	}
	make {
	   xcircuit::newpagebutton [xcircuit::page label]
	}
	default {
	  .output.title.field configure -text \
		"PostScript output properties (Page [xcircuit::page])"
	  ::set fname [xcircuit::page filename]
	  .output.textent.but1 configure -text Apply
	  .output.textent.but2 configure -text Apply
	  .output.textent.but3 configure -text Apply
	  .output.textent.but4 configure -text Apply
	  .output.textent.but5 configure -text Apply
	  .output.textent.but7 configure -text Apply
	  .output.textent.txt1 delete 0 end
	  .output.textent.txt1 insert 0 $fname
	  .output.textent.txt2 delete 0 end
	  .output.textent.txt2 insert 0 [xcircuit::page label]
	  .output.textent.txt3 delete 0 end
	  ::set oldp $tcl_precision
	  ::set tcl_precision 5
	  .output.textent.txt3 insert 0 [xcircuit::page scale]
	  .output.textent.txt4 delete 0 end
	  ::set tcl_precision 3
	  .output.textent.txt4 insert 0 [xcircuit::page width]
	  .output.textent.txt4 insert end " "
	  .output.textent.txt4 insert end [xcircuit::coordstyle get]
	  .output.textent.txt5 delete 0 end
	  .output.textent.txt5 insert 0 [xcircuit::page height]
	  .output.textent.txt5 insert end " "
	  .output.textent.txt5 insert end [xcircuit::coordstyle get]
	  .output.textent.txtf.txtp delete 0 end
	  .output.textent.txtf.txtp insert 0 [xcircuit::page size]
	  ::set tcl_precision $oldp
	  xcircuit::setpstype [xcircuit::page encapsulation]
	  ::set orient [xcircuit::page orientation]
	  if {$orient == 0} {
	     .output.textent.buto configure -text Portrait
	  } else {
	     .output.textent.buto configure -text Landscape
	  }
	  xcircuit::dolinks

	  ::set autofit [xcircuit::page fit]
	  if {[string match *.* $fname] == 0} {append fname .ps}
	  if {[glob -nocomplain ${fname}] == {}} {
	    .output.bbar.okay configure -text "Write File"
	  } else {
	    .output.bbar.okay configure -text "Overwrite File"
	  }
	  .output.bbar.okay configure -command \
		{xcircuit::page filename [.output.textent.txt1 get]; \
		 if {$dmultiple == 1} {xcircuit::page save} else { \
		 xcircuit::page saveonly }; wm withdraw .output}
        }
     }
  }
}

#----------------------------------------------------------------------
# Update the GUI based on the schematic class of the current page
# This is called internally from the xcircuit code and the function
# must be defined, even if it is a null proc.
#----------------------------------------------------------------------

proc xcircuit::setsymschem {} {
   if {[info level] <= 1} {
      set schemtype [xcircuit::schematic type]
      set symschem [xcircuit::schematic get]
      set m .xcircuit.menubar.netlistbutton.netlistmenu
      switch -- $schemtype {
	 schematic {
	    .xcircuit.infobar.schem configure -background red -foreground white
	    if {$symschem == {}} {
		.xcircuit.infobar.symb configure -background gray70 -foreground gray40
		$m entryconfigure 5 -label "Make Matching Symbol" \
			-command {if {![catch {xcircuit::symbol make}]} \
			{xcircuit::promptmakesymbol}}
		$m entryconfigure 6 -label "Associate With Symbol" \
			-command {xcircuit::symbol associate}
	    } else {
		.xcircuit.infobar.symb configure -background white -foreground black
		$m entryconfigure 5 -label "Go To Symbol" \
			-command {xcircuit::symbol go}
		$m entryconfigure 6 -label "Disassociate Symbol" \
			-command {xcircuit::symbol disassociate}
	    }
	 }
	 symbol -
	 fundamental -
	 trivial {
	    .xcircuit.infobar.symb configure -foreground white
	    if {$symschem == {}} {
		.xcircuit.infobar.schem configure -background gray70 -foreground gray40
		$m entryconfigure 5 -label "Make Matching Schematic" \
			-command {xcircuit::schematic make}
		$m entryconfigure 6 -label "Associate With Schematic" \
			-command {xcircuit::schematic associate}
	    } else {
		.xcircuit.infobar.schem configure -background white -foreground black
		$m entryconfigure 5 -label "Go To Schematic" \
			-command {xcircuit::schematic go}
		$m entryconfigure 6 -label "Disassociate Schematic" \
			-command {xcircuit::schematic disassociate}
	    }
	 }
      }
      switch -- $schemtype {
	 trivial {
	    .xcircuit.infobar.symb configure -background red
	 }
	 fundamental {
	    .xcircuit.infobar.symb configure -background green4 ;# bboxcolor
	 }
	 symbol {
	    .xcircuit.infobar.symb configure -background blue2 
	 }
      }
   }
}
   
#----------------------------------------------------------------------
# Set the coordinate style to inches from cm and vice versa.
# This routine avoids switching from fractional to decimal inches
# and vice versa if we are already in one of the two inches modes.
#
# with no argument, or argument "get", returns the "short" name
# ("cm" or "in") of the style.
#----------------------------------------------------------------------

proc xcircuit::coordstyle { { mode get } } {
   set curstyle [xcircuit::config coordstyle]
   switch -- $mode {
      inches {
	 switch -- $curstyle {
	    centimeters {
	       xcircuit::config coordstyle "decimal inches"
	       xcircuit::pageupdate
	    }
	 }
      }
      centimeters -
      cm {
	 switch -- $curstyle {
	    centimeters {
	    }
	    default {
	       xcircuit::config coordstyle "centimeters"
	       xcircuit::pageupdate
	    }
	 }
      }
      get {
	 switch -- $curstyle {
	    centimeters {
	       return "cm"
	    }
	    default {
	       return "in"
	    }
	 }
      }
   }
}

proc xcircuit::raiseconsole {} {
   xcircuit::consoleup
   .xcircuit.menubar.filebutton.filemenu entryconfigure \
	 4 -label "No Console" -command {xcircuit::lowerconsole}
}

proc xcircuit::lowerconsole {} {
   xcircuit::consoledown
   .xcircuit.menubar.filebutton.filemenu entryconfigure \
	 4 -label "Tcl Console" -command {xcircuit::raiseconsole}
}

#----------------------------------------------------------------------
# Command tags---these let the command-line entry functions update the
# Tk windows, so that the Tk window structure doesn't need to be hard-
# coded into the source.
#----------------------------------------------------------------------

xcircuit::tag page {xcircuit::pageupdate %1}
xcircuit::tag promptchecksavelib {xcircuit::promptsavelib %r}
xcircuit::tag promptsavepage {xcircuit::pageupdate ; wm deiconify .output ; \
	raise .output}
xcircuit::tag loadfont {xcircuit::newfontbutton %r}
xcircuit::tag color {set colorval %1; set iname img_co; \
	if {"%1" != "inherit"} {append iname l%1} ; \
	.xcircuit.mainframe.toolbar.bco configure -image $iname}
xcircuit::tag fill { switch -- "%1" { opaque { set opaque true } \
	   transparent { set opaque false } 0 - unfilled {set fillamount 0; \
	      .xcircuit.mainframe.toolbar.bfi configure -image img_fi} \
	   solid {set fillamount 100; \
	      .xcircuit.mainframe.toolbar.bfi configure -image img_stip100}
	   default {set fillamount %1; \
	      .xcircuit.mainframe.toolbar.bfi configure -image img_stip%1} } }

#------------------------------
# Create the file-list window
#------------------------------

# First, set the variables associated with toggle and radio buttons
set filter 1

toplevel .filelist -bg beige
wm withdraw .filelist

frame .filelist.listwin
frame .filelist.textent -bg beige
frame .filelist.bbar	-bg beige

pack .filelist.listwin -side top -padx 20 -pady 7 -expand true -fill both
pack .filelist.textent -side top -padx 20 -pady 7 -fill x
pack .filelist.bbar -side top -padx 20 -pady 7 -fill x

simple .filelist.listwin.win -bg white
simple .filelist.listwin.sb -width 13 -bg beige

grid .filelist.listwin.win -row 0 -column 0 -sticky news -padx 1 -pady 1
grid .filelist.listwin.sb -row 0 -column 1 -sticky ns -padx 1 -pady 1

grid columnconfigure .filelist.listwin 0 -weight 1 -minsize 100
grid rowconfigure .filelist.listwin 0 -weight 1 -minsize 100

frame .filelist.textent.title -bg beige
pack .filelist.textent.title -side top -fill x

label .filelist.textent.title.field -text "Select file to load:" -bg beige
label .filelist.textent.title.chklabel -text "Filter" -bg beige
checkbutton .filelist.textent.title.filter -bg beige -variable filter \
   -command {event generate .filelist.listwin.win <ButtonPress> -button 3 ; \
	event generate .filelist.listwin.win <ButtonRelease> -button 3}

entry .filelist.textent.txt -bg white -relief sunken -width 50

pack .filelist.textent.title.filter -side right
pack .filelist.textent.title.chklabel -side right
pack .filelist.textent.title.field -side left
pack .filelist.textent.txt -side bottom -fill x -expand true

button .filelist.bbar.okay -text Okay -bg beige
button .filelist.bbar.cancel -text Cancel -bg beige -command {wm withdraw .filelist}

pack .filelist.bbar.okay -side left -ipadx 10
pack .filelist.bbar.cancel -side right -ipadx 10

# Allow <return> to update or accept entry
bind .filelist.textent.txt <Return> \
	{event generate .filelist.listwin.win <ButtonPress> -button 2 ; \
	event generate .filelist.listwin.win <ButtonRelease> -button 2}

#--------------------------------------
# Create the output generating window
#--------------------------------------

# First, set the variables associated with toggle and radio buttons
set autofit 0
set imultiple 0
set dmultiple 1

toplevel .output -bg beige
wm withdraw .output

frame .output.title -bg beige
frame .output.textent -bg beige
frame .output.bbar -bg beige

pack .output.title -side top -padx 20 -pady 7 -fill x
pack .output.textent -side top -padx 20 -pady 7 -fill x
pack .output.bbar -side top -padx 20 -pady 7 -fill x

label .output.title.field -text "PostScript output properties (Page 1):" -bg tan
checkbutton .output.title.imulti -text "1 schematic" -bg beige -variable imultiple \
	-command {xcircuit::dolinks ; if {$imultiple == 1} {.output.textent.txt1 \
	delete 0 end; .output.textent.but1 configure -text Apply; xcircuit::page \
	filename {}; focus .output.textent.txt1 ; xcircuit::dolinks }}
checkbutton .output.title.dmulti -text "0 subcircuits" -bg beige -variable dmultiple \
	-command {xcircuit::dolinks ; .output.textent.txt1 \
	delete 0 end; .output.textent.but1 configure -text Apply; xcircuit::page \
	filename {}; focus .output.textent.txt1 ; xcircuit::dolinks }
pack .output.title.dmulti -side right -padx 5
pack .output.title.imulti -side right -padx 5

pack .output.title.field -side left

label .output.textent.lab1 -text "Filename:" -bg beige
label .output.textent.lab2 -text "Page label:" -bg beige
label .output.textent.lab3 -text "Scale:" -bg beige
label .output.textent.lab4 -text "Width:" -bg beige
label .output.textent.lab5 -text "Height:" -bg beige
label .output.textent.lab6 -text "Orientation:" -bg beige
label .output.textent.lab7 -text "Mode:" -bg beige

entry .output.textent.txt1 -bg white -relief sunken -width 20
entry .output.textent.txt2 -bg white -relief sunken -width 20
entry .output.textent.txt3 -bg white -relief sunken -width 20
entry .output.textent.txt4 -bg white -relief sunken -width 20
entry .output.textent.txt5 -bg white -relief sunken -width 20

menubutton .output.textent.buto -text Portrait -bg beige \
	-menu .output.textent.buto.orientmenu
menubutton .output.textent.butp -text "Embedded (EPS)" -bg beige \
	-menu .output.textent.butp.psmenu

checkbutton .output.textent.butf -text "Auto-fit" -bg beige -variable autofit \
	-onvalue true -offvalue false -command {xcircuit::page fit $autofit}
frame .output.textent.txtf -bg beige
menubutton .output.textent.txtf.sizb -text "Sizes" -bg beige \
	-menu .output.textent.txtf.sizb.sizemenu
entry .output.textent.txtf.txtp -bg white -relief sunken -width 14

pack .output.textent.txtf.txtp -side left -fill y
pack .output.textent.txtf.sizb -side left

button .output.textent.but1 -text Apply -bg beige \
	-command {xcircuit::page filename [.output.textent.txt1 get];\
	.output.textent.but1 configure -text Okay}
button .output.textent.but2 -text Apply -bg beige \
	-command {xcircuit::page label [.output.textent.txt2 get];\
	.output.textent.but2 configure -text Okay}
button .output.textent.but3 -text Apply -bg beige \
	-command {xcircuit::page scale [.output.textent.txt3 get];\
	.output.textent.but3 configure -text Okay}
button .output.textent.but4 -text Apply -bg beige \
	-command {xcircuit::page width [.output.textent.txt4 get];\
	.output.textent.but4 configure -text Okay}
button .output.textent.but5 -text Apply -bg beige \
	-command {xcircuit::page height [.output.textent.txt5 get];\
	.output.textent.but5 configure -text Okay}
button .output.textent.but7 -text Apply -bg beige \
	-command {xcircuit::page size [.output.textent.txtf.txtp get];\
	.output.textent.but7 configure -text Okay}

bind .output.textent.txt1 <Return> {.output.textent.but1 invoke}
bind .output.textent.txt2 <Return> {.output.textent.but2 invoke}
bind .output.textent.txt3 <Return> {.output.textent.but3 invoke}
bind .output.textent.txt4 <Return> {.output.textent.but4 invoke}
bind .output.textent.txt5 <Return> {.output.textent.but5 invoke}

grid .output.textent.lab1 -row 0 -column 0 -sticky w
grid .output.textent.lab2 -row 1 -column 0 -sticky w
grid .output.textent.lab3 -row 2 -column 0 -sticky w
grid .output.textent.lab4 -row 3 -column 0 -sticky w
grid .output.textent.lab5 -row 4 -column 0 -sticky w
grid .output.textent.lab6 -row 5 -column 0 -sticky w
grid .output.textent.lab7 -row 6 -column 0 -sticky w

grid .output.textent.txt1 -row 0 -column 1 -columnspan 2 -sticky ew -padx 10
grid .output.textent.txt2 -row 1 -column 1 -columnspan 2 -sticky ew -padx 10
grid .output.textent.txt3 -row 2 -column 1 -columnspan 2 -sticky ew -padx 10
grid .output.textent.txt4 -row 3 -column 1 -columnspan 2 -sticky ew -padx 10
grid .output.textent.txt5 -row 4 -column 1 -columnspan 2 -sticky ew -padx 10
grid .output.textent.buto -row 5 -column 1 -sticky w -padx 10
grid .output.textent.butp -row 6 -column 1 -sticky w -padx 10

grid .output.textent.but1 -row 0 -column 3 -pady 5 -ipadx 10
grid .output.textent.but2 -row 1 -column 3 -pady 5 -ipadx 10
grid .output.textent.but3 -row 2 -column 3 -pady 5 -ipadx 10
grid .output.textent.but4 -row 3 -column 3 -pady 5 -ipadx 10
grid .output.textent.but5 -row 4 -column 3 -pady 5 -ipadx 10

grid columnconfigure .output.textent 2 -weight 1

button .output.bbar.okay -text Okay -bg beige -command {xcircuit::page save; \
  wm withdraw .output}
button .output.bbar.cancel -text Cancel -bg beige -command {wm withdraw .output}

# Setup simple choice menus for page type and orientation
# First, set the variables associated with the radio buttons. . . 
set orient 0
set pstype eps

set m [menu .output.textent.buto.orientmenu -tearoff 0]
$m add radio -label "Portrait" -variable orient -value 0 -command \
	{.output.textent.buto configure -text Portrait ; \
	xcircuit::page orientation $orient}
$m add radio -label "Landscape" -variable orient -value 90 -command \
	{.output.textent.buto configure -text Landscape ; \
	xcircuit::page orientation $orient}
set m [menu .output.textent.butp.psmenu -tearoff 0]
$m add radio -label "Embedded (EPS)" -variable pstype -value eps -command \
	{xcircuit::setpstype eps}
$m add radio -label "Full Page" -variable pstype -value full -command \
	{xcircuit::setpstype full}
pack .output.bbar.okay -side left -ipadx 10
pack .output.bbar.cancel -side right -ipadx 10

set m [menu .output.textent.txtf.sizb.sizemenu -tearoff 0]
$m add radio -label "Letter (ANSI A)" -variable sheetsize -value letter -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "8.5 x 11.0 in"}
$m add radio -label "Legal" -variable sheetsize -value legal -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "8.5 x 14.0 in"}
$m add radio -label "Statement" -variable sheetsize -value statement -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "5.5 x 8.5 in"}
$m add radio -label "Tabloid (ANSI B)" -variable sheetsize -value tabloid -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "11.0 x 17.0 in"}
$m add radio -label "Ledger" -variable sheetsize -value ledger -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "17.0 x 11.0 in"}
$m add radio -label "Folio" -variable sheetsize -value folio -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "8.5 x 13.0 in"}
$m add radio -label "Quarto" -variable sheetsize -value quarto -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "8.472 x 10.833 in"}
$m add radio -label "10x14" -variable sheetsize -value tenfourteen -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "10.0 x 14.0 in"}
$m add radio -label "Executive" -variable sheetsize -value executive -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "7.5 x 10.0 in"}
$m add radio -label "ANSI C" -variable sheetsize -value ansic -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "17.0 x 22.0 in"}
$m add radio -label "ANSI D" -variable sheetsize -value ansid -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "22.0 x 34.0 in"}
$m add radio -label "ANSI E" -variable sheetsize -value ansie -command \
	{ xcircuit::coordstyle inches; xcircuit::page size "34.0 x 44.0 in"}
$m add radio -label "A3" -variable sheetsize -value a3 -command \
	{ xcircuit::coordstyle centimeters; xcircuit::page size "29.7 x 42.0 cm"}
$m add radio -label "A4" -variable sheetsize -value a4 -command \
	{ xcircuit::coordstyle centimeters; xcircuit::page size "21.0 x 29.7 cm"}
$m add radio -label "A5" -variable sheetsize -value a5 -command \
	{ xcircuit::coordstyle centimeters; xcircuit::page size "14.82 x 18.43 cm"}
$m add radio -label "B4" -variable sheetsize -value b4 -command \
	{ xcircuit::coordstyle centimeters; xcircuit::page size "25.7 x 36.4 cm"}
$m add radio -label "B5" -variable sheetsize -value b5 -command \
	{ xcircuit::coordstyle centimeters; xcircuit::page size "18.2 x 25.7 cm"}

#--------------------------------------
# Create the popup help window
#--------------------------------------

toplevel .help -bg beige
wm withdraw .help

frame .help.title -bg beige
frame .help.listwin

pack .help.title -side top -fill x
pack .help.listwin -side top -fill both -expand true

label .help.title.field -text "XCircuit Help" -bg beige
button .help.title.dbut -text "Dismiss" -bg beige -command {wm withdraw .help}
pack .help.title.field -side left -padx 10
pack .help.title.dbut -side right -ipadx 10

simple .help.listwin.win -bg darkslategray
simple .help.listwin.sb -width 13 -bg beige

grid .help.listwin.win -row 0 -column 0 -sticky news -padx 1 -pady 1
grid .help.listwin.sb -row 0 -column 1 -sticky ns -padx 1 -pady 1

grid columnconfigure .help.listwin 0 -weight 1 -minsize 100
grid rowconfigure .help.listwin 0 -weight 1 -minsize 100

#--------------------------------------
# Create a selection listbox
#--------------------------------------

toplevel .select -bg beige
wm withdraw .select

listbox .select.sellist -bg white
pack .select.sellist -expand true -fill both
bind .select.sellist <ButtonRelease-1> {set idx [.select.sellist nearest %y]}
bind .select.sellist <ButtonRelease-3> {set idx -1}

#--------------------------------------
# Create the simple popup prompt window
#--------------------------------------

toplevel .dialog -bg beige
wm withdraw .dialog

frame .dialog.textent -bg beige
frame .dialog.bbar	-bg beige

pack .dialog.textent -side top -padx 20 -pady 7 -fill x
pack .dialog.bbar -side top -padx 20 -pady 7 -fill x

frame .dialog.textent.title -bg beige
entry .dialog.textent.txt -bg white -relief sunken -width 50

pack .dialog.textent.title -side top -fill x
pack .dialog.textent.txt -side bottom -fill x -expand true

label .dialog.textent.title.field -text "Select file to load:" -bg beige
pack .dialog.textent.title.field -side left

button .dialog.bbar.okay -text Okay -bg beige
button .dialog.bbar.cancel -text Cancel -bg beige -command {wm withdraw .dialog}

bind .dialog.textent.txt <Return> {.dialog.bbar.okay invoke}

pack .dialog.bbar.okay -side left -ipadx 10
pack .dialog.bbar.cancel -side right -ipadx 10

#--------------------------------------
# Create the query prompt window
#--------------------------------------

toplevel .query -bg beige
wm withdraw .query

frame .query.title -bg beige
frame .query.bbar	-bg beige

pack .query.title -side top -padx 20 -pady 7 -fill x
pack .query.bbar -side top -padx 20 -pady 7 -fill x

label .query.title.field -text "Select:" -bg beige
pack .query.title.field -side left

button .query.bbar.okay -text Okay -bg beige
button .query.bbar.cancel -text Cancel -bg beige -command {wm withdraw .query}

pack .query.bbar.okay -side left -ipadx 10
pack .query.bbar.cancel -side right -ipadx 10

#--------------------------------------------------------
# Generate all of the menu cascades
# Most commands reference XCircuit internal routines
#--------------------------------------------------------

# Supporting procedures

proc xcircuit::printstring {stringlist} {
   set p ""
   foreach i $stringlist {
      switch -- [lindex $i 0] {
	 Text {append p [lindex $i 1]}
	 Half -
	 Quarter {append p " "} 
      }
   }
   return $p
}

proc xcircuit::promptselectparam {} {
   global idx
   ::set dlist [xcircuit::parameter default substring]
   ::set ilist {}
   ::set idx -1
   catch { ::set ilist [xcircuit::parameter get substring] }
   if {[llength $dlist] > 0} {
      if {[llength $dlist] == 1} {
	 ::set idx 0
      } else {
         foreach i $dlist {
            .select.sellist insert end [xcircuit::printstring $i]
         }
	 wm deiconify .select
	 raise .select
         tkwait variable idx
	 .select.sellist delete 0 end
	 wm withdraw .select
      }
   }
   return $idx
}

proc xcircuit::promptmakesymbol {} {
  .dialog.bbar.okay configure -command \
	  {xcircuit::symbol make [.dialog.textent.txt get]; \
	  wm withdraw .dialog}
  .dialog.textent.title.field configure -text "Enter name for new object:"
  .dialog.textent.txt delete 0 end
  wm deiconify .dialog
  focus .dialog.textent.txt
}

proc xcircuit::promptobjectsize {} {
   if {![catch {::set cscale [xcircuit::element [select get] scale]}]} {
      .dialog.bbar.okay configure -command \
	  {xcircuit::instance scale [.dialog.textent.txt get]; \
	  wm withdraw .dialog}
      .dialog.textent.title.field configure -text "Enter object size:"
      .dialog.textent.txt delete 0 end
      .dialog.textent.txt insert 0 $cscale
      wm deiconify .dialog
      focus .dialog.textent.txt
   }
}

proc xcircuit::promptborderwidth {} {
   global tcl_precision
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   if {[catch {::set elist [xcircuit::select get]}]} {
      .dialog.bbar.okay configure -command \
	   {xcircuit::config linewidth [.dialog.textent.txt get]; \
	   wm withdraw .dialog}
      .dialog.textent.title.field configure -text "Enter linewidth scale for page:"
      .dialog.textent.txt insert 0 [xcircuit::config linewidth]
   } else {
      .dialog.bbar.okay configure -command \
	   {xcircuit::element border set [.dialog.textent.txt get]; \
	   wm withdraw .dialog}
      .dialog.textent.title.field configure -text "Enter element linewidth:"
      .dialog.textent.txt insert 0 [xcircuit::element $elist border get]
   }
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptlinewidth {} {
   global tcl_precision
   .dialog.bbar.okay configure -command \
	{xcircuit::config linewidth [.dialog.textent.txt get]; \
	 wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter global linewidth scale:"
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   .dialog.textent.txt insert 0 [xcircuit::config linewidth]
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptdrawingscale {} {
   global tcl_precision
   .dialog.bbar.okay configure -command \
	{xcircuit::config drawingscale [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter drawing scale:"
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   .dialog.textent.txt insert 0 [xcircuit::config drawingscale]
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptgridspace {} {
   global tcl_precision
   .dialog.bbar.okay configure -command \
	{xcircuit::config grid spacing [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter grid spacing:"
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   .dialog.textent.txt insert 0 [xcircuit::config grid space]
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptsnapspace {} {
   global tcl_precision
   .dialog.bbar.okay configure -command \
	{xcircuit::config snap spacing [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter snap spacing:"
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   .dialog.textent.txt insert 0 [xcircuit::config snap space]
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptmakeobject {} {
   if {[xcircuit::select] > 0} {
      .dialog.bbar.okay configure -command \
	 {xcircuit::object make [.dialog.textent.txt get]; wm withdraw .dialog}
      .dialog.textent.title.field configure -text "Enter name for new object"
      .dialog.textent.txt delete 0 end
      wm deiconify .dialog
      focus .dialog.textent.txt
   }
}

proc xcircuit::promptloadlibrary {} {
   .filelist.bbar.okay configure -command \
	{xcircuit::library load [.filelist.textent.txt get]; wm withdraw .filelist}
   .filelist.listwin.win configure -data "lps"
   .filelist.textent.title.field configure -text "Select library file to load:"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::promptsavelib { libno } {
   .dialog.bbar.okay configure -command \
	[subst {xcircuit::library $libno save \[.dialog.textent.txt get\];\
	 wm withdraw .dialog}]
   .dialog.textent.title.field configure -text "Enter name for library:"
   .dialog.textent.txt delete 0 end
   set lname [xcircuit::library $libno]
   if {[string first Library: $lname] == 0} \
      {set lname [string range $lname 8 end]}
   set lname [string trimleft $lname]
   .dialog.textent.txt insert 0 [string map {" " _} $lname]
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptaddlibrary {} {
   .filelist.bbar.okay configure -command \
	{xcircuit::library [xcircuit::library] load [.filelist.textent.txt get];\
	 wm withdraw .filelist}
   .filelist.listwin.win configure -data "lps"
   .filelist.textent.title.field configure -text "Select library file to add:"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::promptloadfile {} {
   .filelist.bbar.okay configure -command \
	{xcircuit::page load [.filelist.textent.txt get]; wm withdraw .filelist}
   .filelist.listwin.win configure -data "ps"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::promptimportfile {} {
   .filelist.bbar.okay configure -command \
	{xcircuit::page import [.filelist.textent.txt get]; wm withdraw .filelist}
   .filelist.listwin.win configure -data "ps"
   .filelist.textent.title.field configure -text "Select file to import:"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::promptimportbackground {} {
   .filelist.bbar.okay configure -command \
	{xcircuit::page background [.filelist.textent.txt get]; wm withdraw .filelist}
   .filelist.listwin.win configure -data "ps"
   .filelist.textent.title.field configure -text "Select file to use as background:"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::promptexecscript {} {
   .filelist.bbar.okay configure -command \
	{source [.filelist.textent.txt get]; wm withdraw .filelist}
   .filelist.listwin.win configure -data "tcl"
   .filelist.textent.title.field configure -text "Select script to execute:"
   wm deiconify .filelist
   focus .filelist.textent.txt
}

proc xcircuit::prompttextsize {} {
   global tcl_precision
   .dialog.bbar.okay configure -command \
	{xcircuit::label scale [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter text scale:"
   .dialog.textent.txt delete 0 end
   ::set savep $tcl_precision
   ::set tcl_precision 3
   .dialog.textent.txt insert 0 [xcircuit::label scale]
   ::set tcl_precision $savep
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::newcolorbutton {r g b idx} {
   global XCIRCUIT_LIB_DIR
   ::set colorrgb [format "#%04X%04X%04X" $r $g $b]
   image create bitmap img_col$idx -foreground $colorrgb -file \
		${XCIRCUIT_LIB_DIR}/pixmaps/solid.xbm
   .xcircuit.menubar.optionsbutton.optionsmenu.elementsmenu.colormenu \
	add radio -image img_col$idx -activebackground $colorrgb \
	-variable colorval -value $idx -command "xcircuit::color $idx"
}

proc xcircuit::newencodingbutton {encodingname} {
   if {[catch {.xcircuit.menubar.textbutton.textmenu.encodingmenu \
		index $encodingname} result]} {
      .xcircuit.menubar.textbutton.textmenu.encodingmenu add radio -label \
	  	$encodingname -command "xcircuit::label encoding $encodingname" \
		-variable fontencoding -value $encodingname
   }
}

proc xcircuit::newfontbutton {familyname} {
   if {[catch {.xcircuit.menubar.textbutton.textmenu.fontmenu \
		index $familyname} result]} {
      .xcircuit.menubar.textbutton.textmenu.fontmenu add radio -label \
	  	$familyname -command "xcircuit::label family $familyname" \
		-variable fontfamily -value $familyname
   }
}

proc xcircuit::newlibrarybutton {libname} {
   if {[catch {.xcircuit.menubar.windowbutton.windowmenu.librarymenu \
		index $libname} result]} {
      ::set libidx [.xcircuit.menubar.windowbutton.windowmenu.librarymenu \
		index end]
      if {$libidx <= 1} {::set libidx [expr $libidx + 1]}
      .xcircuit.menubar.windowbutton.windowmenu.librarymenu insert $libidx \
		command -label $libname -command "xcircuit::library \"$libname\" goto"
   }
}

proc xcircuit::newpagebutton {pagename {pageno 0}} {
   if {[catch {.xcircuit.menubar.windowbutton.windowmenu.pagemenu \
		index $pagename} result]} {
      ::set target $pagename
      if {$pageno > 0} { ::set target $pageno }
      .xcircuit.menubar.windowbutton.windowmenu.pagemenu add command -label \
	  	$pagename -command "xcircuit::page \"$target\" goto"
   }
}

proc xcircuit::renamelib {libno libname} {
   ::set target [expr $libno + 1]
   .xcircuit.menubar.windowbutton.windowmenu.librarymenu entryconfigure \
		$target -label $libname -command "xcircuit::library \"$libname\" goto"
}

proc xcircuit::renamepage {pageno pagename} {
   ::set target [expr $pageno + 1]
   .xcircuit.menubar.windowbutton.windowmenu.pagemenu entryconfigure \
		$target -label $pagename -command "xcircuit::page \"$pagename\" goto"
}

proc xcircuit::promptnewfont {} {
   .dialog.bbar.okay configure -command \
	{xcircuit::loadfont [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter font name:"
   .dialog.textent.txt delete 0 end
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::promptkern {} {
   .dialog.bbar.okay configure -command \
	{xcircuit::label insert kern [.dialog.textent.txt get]; wm withdraw .dialog}
   .dialog.textent.title.field configure -text "Enter kern amount:"
   .dialog.textent.txt delete 0 end
   wm deiconify .dialog
   focus .dialog.textent.txt
}

proc xcircuit::xschema {value} {
   switch -- $value {
      true -
      enable {
	 pack forget .xcircuit.menubar.message
	 pack .xcircuit.menubar.netlistbutton -side left -padx 2 -ipadx 10 -pady 2
	 pack .xcircuit.menubar.message -side top -fill x -padx 2 -ipady 3 -pady 2
	 pack forget .xcircuit.infobar.message1
	 pack forget .xcircuit.infobar.message2
	 pack .xcircuit.infobar.symb .xcircuit.infobar.schem \
        	.xcircuit.infobar.message1 -side left -padx 2 -ipadx 6
	 pack .xcircuit.infobar.message2 -side bottom -padx 2 -ipadx 6 -fill x
	 ::set midx [.xcircuit.menubar.optionsbutton.optionsmenu \
		index "Enable XSchema"]
	 .xcircuit.menubar.optionsbutton.optionsmenu entryconfigure $midx \
		-command {xcircuit::xschema disable} -label \
		"Disable XSchema"
      }
      false -
      disable {
	 pack forget .xcircuit.menubar.netlistbutton
	 pack forget .xcircuit.infobar.symb
	 pack forget .xcircuit.infobar.schem
	 ::set midx [.xcircuit.menubar.optionsbutton.optionsmenu \
		index "Disable XSchema"]
	 .xcircuit.menubar.optionsbutton.optionsmenu entryconfigure $midx \
		-command {xcircuit::xschema enable} -label \
		"Enable XSchema"
      }
   }
}

set tools [list b a s t cp e d2 cw ccw fx fy pu2 po2 mk pz uj co bd fi \
	pm li yp pl z4 z5 i]

proc xcircuit::createtoolbar {} {
   global XCIRCUIT_LIB_DIR
   global tools
   set tooltips [list "draw box" "draw arc" "draw spline" "enter text" \
	"copy element" "edit element" "delete element" \
	"rotate 15 degrees clockwise" "rotate 15 degrees counterclockwise" \
	"flip horizontal" "flip vertical" "push (edit object)" \
	"pop (return from object edit)" "make an object from selection" \
	"join elements into polygon or path" "separate path into elements" \
	"set color" "set border and line properties" "set fill properties" \
	"parameterize properties" "go to next library" "go to library directory" \
	"go to page directory" "zoom in" "zoom out" "pop up help window"]	
   set toolactions [list {xcircuit::polygon make box} {xcircuit::arc make} \
	{xcircuit::spline make} {xcircuit::label make} {xcircuit::copy selected} \
	{xcircuit::edit selected} {xcircuit::delete selected} {xcircuit::rotate 15} \
	{xcircuit::rotate -15} {xcircuit::flip horizontal} \
	{xcircuit::flip vertical} {xcircuit::push selected} {xcircuit::pop} \
	{xcircuit::promptmakeobject} {xcircuit::path join selected} \
	{xcircuit::path unjoin selected} \
	{tk_popup .colormenu [expr {[winfo rootx \
		.xcircuit.mainframe.toolbar.bco] 
		- [winfo width .colormenu]}] \
		[expr {[winfo rooty .xcircuit.mainframe.toolbar.bco] \
		- [winfo height .colormenu] / 2}] } \
	{tk_popup .bordermenu [expr {[winfo rootx \
		.xcircuit.mainframe.toolbar.bbd] \
		- [winfo width .bordermenu]}] \
		[expr {[winfo rooty .xcircuit.mainframe.toolbar.bbd] \
		- [winfo height .bordermenu] / 2}] } \
	{tk_popup .fillmenu [expr {[winfo rootx \
		.xcircuit.mainframe.toolbar.bfi] \
		- [winfo width .fillmenu]}] \
		[expr {[winfo rooty .xcircuit.mainframe.toolbar.bfi] \
		- [winfo height .fillmenu] / 2}] } \
	{tk_popup .parammenu [expr {[winfo rootx \
		.xcircuit.mainframe.toolbar.bpm] \
		- [winfo width .parammenu]}] \
		[expr {[winfo rooty .xcircuit.mainframe.toolbar.bpm] \
		- [winfo height .parammenu] / 2}] } \
	{xcircuit::library next} \
	{xcircuit::library directory} {xcircuit::page directory} \
	{xcircuit::zoom 1.5; refresh} {xcircuit::zoom [expr {1 / 1.5}]; refresh} \
	{wm deiconify .help} ]

   for {set i 0} {$i < [llength $tools]} {incr i 1} {
      set bname [lindex $tools $i]
      set btip [lindex $tooltips $i]
      image create xpm img_${bname} -file ${XCIRCUIT_LIB_DIR}/pixmaps/${bname}.xpm
      button .xcircuit.mainframe.toolbar.b${bname} -image img_${bname} \
		-command [lindex $toolactions $i]
      bind .xcircuit.mainframe.toolbar.b${bname} <Enter> \
		[subst {.xcircuit.infobar.message2 configure -text "$btip"}]
      bind .xcircuit.mainframe.toolbar.b${bname} <Leave> \
		[subst {.xcircuit.infobar.message2 configure -text ""}]
   }
 
   # pack the first button so we can query its height for arrangement.
   # this assumes that the height of each button is the same!
   set bname [lindex $tools 0]
   place .xcircuit.mainframe.toolbar.b${bname} -x 0 -y 0
   update idletasks
}

proc xcircuit::arrangetoolbar {} {
   global tools

   set numtools [llength $tools]
   for {set i 0} {$i < $numtools} {incr i 1} {
      set bname [lindex $tools $i]
      place forget .xcircuit.mainframe.toolbar.b${bname}
   }
   set bname [lindex $tools 0]
   set bheight [winfo height .xcircuit.mainframe.toolbar.b${bname}]
   set bwidth [winfo width .xcircuit.mainframe.toolbar.b${bname}]
   set wheight [winfo height .xcircuit.mainframe]
   set nrows [expr {$wheight / $bheight}]
   .xcircuit.mainframe.toolbar configure -width [expr {$bwidth}]
   set j 0
   set k 0
   for {set i 0} {$i < [llength $tools]} {incr i; incr j} {
      if {$j == $nrows} {
	 set j 0
	 incr k
	 .xcircuit.mainframe.toolbar configure -width [expr {($k + 1) * $bwidth}]
      }
      set bname [lindex $tools $i]
      place .xcircuit.mainframe.toolbar.b${bname} \
		-x [expr {$k * $bwidth}] \
		-y [expr {$j * $bheight}]
   }
}

proc xcircuit::toolbar {value} {
   switch -- $value {
      true -
      enable {
	 pack forget .xcircuit.mainframe.mainarea
	 pack .xcircuit.mainframe.toolbar -side right -fill y -padx 2
	 pack .xcircuit.mainframe.mainarea -expand true -fill both
	 ::set midx [.xcircuit.menubar.optionsbutton.optionsmenu \
		index "Enable Toolbar"]
	 .xcircuit.menubar.optionsbutton.optionsmenu entryconfigure $midx \
		-command {xcircuit::toolbar disable} -label \
		"Disable Toolbar"
      }
      false -
      disable {
	 pack forget .xcircuit.mainframe.toolbar
	 ::set midx [.xcircuit.menubar.optionsbutton.optionsmenu \
		index "Disable Toolbar"]
	 .xcircuit.menubar.optionsbutton.optionsmenu entryconfigure $midx \
		-command {xcircuit::toolbar enable} -label \
		"Enable Toolbar"
      }
   }
}

# First, set the variables associated with toggle and radio buttons
set colorval inherit
set jhoriz left
set jvert bottom
set linestyle solid
set fillamount 0
set opaque 0
set polyedittype manhattan
set showgrid true
set showsnap true
set showaxes true
set showbbox false
set fontfamily Helvetica
set fontstyle normal
set fontencoding Standard
set fontlining normal
set fontscript normal
set gridstyle "fractional inches"
set flipinvariant true
set pinvisible false
set colorscheme normal
set bbox invisible
set editinplace true
set pinpositions invisible
set centerobject true
set manhattandraw false
set polyclosed true
set bboxtype false
set substringparam false
set positionparam false
set styleparam false
set justparam false
set startparam false
set endparam false
set radiusparam false
set minorparam false
set rotationparam false
set scaleparam false
set linewidthparam false
set sel_lab true
set sel_obj true
set sel_poly true
set sel_arc true
set sel_spline true
set sel_path true
set sheetsize letter

set m [menu .xcircuit.menubar.filebutton.filemenu -tearoff 0]
$m add command -label "Read XCircuit File" -command {xcircuit::promptloadfile}
$m add command -label "Import XCircuit PS" -command {xcircuit::promptimportfile}
$m add command -label "Import background PS" -command {xcircuit::promptimportbackground}
$m add command -label "Execute Script" -command {xcircuit::promptexecscript}
$m add command -label "Tcl Console" -command {xcircuit::raiseconsole}
$m add command -label "Write XCircuit PS" -command {xcircuit::promptsavepage}
$m add separator
$m add command -label "Add To Library" -command {xcircuit::promptaddlibrary}
$m add command -label "Load New Library" -command {xcircuit::promptloadlibrary}
$m add command -label "Save Library" -command {if {[catch xcircuit::library]} \
	{xcircuit::promptsavelib [xcircuit::library "User Library"]} \
	{xcircuit::promptsavelib [xcircuit::library]}}
$m add separator
$m add command -label "Clear Page" -command {xcircuit::page reset}
$m add separator
$m add command -label "Quit" -command {quit}

set m [menu .xcircuit.menubar.editbutton.editmenu -tearoff 0]
$m add command -label "Delete" -command {delete selected}
$m add command -label "Undelete" -command {undelete}
$m add command -label "Copy" -command {copy selected}
$m add command -label "Edit" -command {edit selected}
$m add cascade -label "Rotate/Flip" -menu $m.rotmenu
$m add command -label "Deselect" -command {deselect}
$m add cascade -label "Select Filter" -menu $m.selmenu
$m add command -label "Push Selected" -command {push selected}
$m add command -label "Pop Heirarchy" -command {pop}
$m add separator
$m add command -label "Make User Object" -command {xcircuit::promptmakeobject}
$m add command -label "Make Arc" -command {xcircuit::arc make}
$m add command -label "Make Box" -command {xcircuit::polygon make box}
$m add command -label "Make Spline" -command {xcircuit::spline make}
$m add command -label "Join" -command {xcircuit::path make}

set m2 [menu $m.rotmenu -tearoff 0]
$m2 add command -label "Flip Horizontal" -command {flip horizontal}
$m2 add command -label "Flip Vertical" -command {flip vertical}
$m2 add separator
$m2 add command -label "Rotate CW 90" -command {rotate 90}
$m2 add command -label "Rotate CW 45" -command {rotate 45}
$m2 add command -label "Rotate CW 30" -command {rotate 30}
$m2 add command -label "Rotate CW 15" -command {rotate 15}
$m2 add command -label "Rotate CW 5" -command {rotate 5}
$m2 add command -label "Rotate CW 1" -command {rotate 1}
$m2 add separator
$m2 add command -label "Rotate CCW 90" -command {rotate -90}
$m2 add command -label "Rotate CCW 45" -command {rotate -45}
$m2 add command -label "Rotate CCW 30" -command {rotate -30}
$m2 add command -label "Rotate CCW 15" -command {rotate -15}
$m2 add command -label "Rotate CCW 5" -command {rotate -5}
$m2 add command -label "Rotate CCW 1" -command {rotate -1}

set m2 [menu $m.selmenu -tearoff 0]
$m2 add check -label "Labels" -variable sel_lab -onvalue true -offvalue false \
	-command {xcircuit::config filter label $sel_lab}
$m2 add check -label "Objects" -variable sel_obj -onvalue true -offvalue false \
	-command {xcircuit::config filter instance $sel_obj}
$m2 add check -label "Polygons" -variable sel_poly -onvalue true -offvalue false \
	-command {xcircuit::config filter polygon $sel_poly}
$m2 add check -label "Arcs" -variable sel_arc -onvalue true -offvalue false \
	-command {xcircuit::config filter arc $sel_arc}
$m2 add check -label "Splines" -variable sel_spline -onvalue true -offvalue false \
	-command {xcircuit::config filter spline $sel_spline}
$m2 add check -label "Paths" -variable sel_path -onvalue true -offvalue false \
	-command {xcircuit::config filter path $sel_path}

set m [menu .xcircuit.menubar.textbutton.textmenu -tearoff 0]
$m add command -label "Text Size" -command {xcircuit::prompttextsize}
$m add cascade -label "Font" -menu $m.fontmenu
$m add cascade -label "Style" -menu $m.stylemenu
$m add cascade -label "Encoding" -menu $m.encodingmenu
$m add cascade -label "Insert" -menu $m.insertmenu
$m add cascade -label "Justification" -menu $m.justifymenu
$m add command -label "Parameterize" \
	-command {xcircuit::parameter make substring}
$m add command -label "Unparameterize" \
	-command {xcircuit::parameter forget substring}
$m add separator
$m add command -label "Make Label" -command {xcircuit::label make}

set m2 [menu $m.fontmenu -tearoff 0]
$m2 add command -label "Add New Font" -command {xcircuit::promptnewfont}
$m2 add separator

set m2 [menu $m.stylemenu -tearoff 0]
$m2 add radio -label "Normal" -variable fontstyle -value normal \
	-command "xcircuit::label style normal"
$m2 add radio -label "Bold" -variable fontstyle -value bold \
	-command "xcircuit::label style bold"
$m2 add radio -label "Italic" -variable fontstyle -value italic \
	-command "xcircuit::label style italic"
$m2 add radio -label "BoldItalic" -variable fontstyle -value bolditalic \
	-command "xcircuit::label style bolditalic"
$m2 add separator
$m2 add radio -label "Subscript" -variable fontscript -value subscript \
	-command "xcircuit::label insert subscript"
$m2 add radio -label "Superscript" -variable fontscript -value superscript \
	-command "xcircuit::label insert superscript"
$m2 add radio -label "Normalscript" -variable fontscript -value normal \
	-command "xcircuit::label insert normalscript"
$m2 add separator
$m2 add radio -label "Underline" -variable fontlining -value underline \
	-command "xcircuit::label insert underline"
$m2 add radio -label "Overline" -variable fontlining -value overline \
	-command "xcircuit::label insert overline"
$m2 add radio -label "No Line" -variable fontlining -value normal \
	-command "xcircuit::label insert noline"

set m2 [menu $m.encodingmenu -tearoff 0]
$m2 add radio -label "Standard" -variable fontencoding -value Standard \
	-command "xcircuit::label encoding Standard"
$m2 add radio -label "ISOLatin1" -variable fontencoding -value ISOLatin1 \
	-command "xcircuit::label encoding ISOLatin1"

set m2 [menu $m.insertmenu -tearoff 0]
$m2 add command -label "Tab stop" -command "xcircuit::label insert stop"
$m2 add command -label "Tab forward" -command "xcircuit::label insert forward"
$m2 add command -label "Tab backward" -command "xcircuit::label insert backward"
$m2 add command -label "Carriage Return" -command "xcircuit::label insert return"
$m2 add command -label "1/2 space" -command "xcircuit::label insert halfspace"
$m2 add command -label "1/4 space" -command "xcircuit::label insert quarterspace"
$m2 add command -label "Kern" -command "xcircuit::promptkern"
$m2 add command -label "Character" -command "xcircuit::label insert special"

set m2 [menu $m.justifymenu -tearoff 0]
$m2 add radio -label "Left Justified" -variable jhoriz -value left \
	-command "xcircuit::label justify left"
$m2 add radio -label "Center Justified" -variable jhoriz -value center \
	-command "xcircuit::label justify center"
$m2 add radio -label "Right Justified" -variable jhoriz -value right \
	-command "xcircuit::label justify right"
$m2 add separator
$m2 add radio -label "Top Justified" -variable jvert -value top \
	-command "xcircuit::label justify top"
$m2 add radio -label "Middle Justified" -variable jvert -value middle \
	-command "xcircuit::label justify middle"
$m2 add radio -label "Bottom Justified" -variable jvert -value bottom \
	-command "xcircuit::label justify bottom"
$m2 add separator
$m2 add check -label "Flip Invariant" -variable flipinvariant -onvalue true \
	-offvalue false -command "xcircuit::label flipinvariant"

set m [menu .xcircuit.menubar.optionsbutton.optionsmenu -tearoff 0]
$m add check -label "Alt Colors" -variable colorscheme -onvalue alternate \
	-offvalue normal -command {xcircuit::config colorscheme $colorscheme}
$m add check -label "Show Bounding Box" -variable showbbox -onvalue visible \
	-offvalue invisible -command {xcircuit::config bbox $showbbox}
$m add check -label "Edit In Place" -variable editinplace -onvalue true \
	-offvalue false -command {xcircuit::config editinplace $editinplace}
$m add check -label "Show Pin Positions" -variable pinpositions -onvalue visible \
	-offvalue invisible -command {xcircuit::config pinpositions $pinpositions}
$m add command -label "Disable XSchema" -command {xcircuit::xschema disable}
$m add command -label "Disable Toolbar" -command {xcircuit::toolbar disable}
$m add cascade -label "Grid" -menu $m.gridmenu
$m add cascade -label "Snap-to" -menu $m.snapmenu
$m add command -label "Global Linewidth" -command {xcircuit::promptlinewidth}
$m add cascade -label "Elements" -menu $m.elementsmenu
$m add separator
$m add command -label "Help!" -command {wm deiconify .help}

set m2 [menu $m.gridmenu -tearoff 0]
$m2 add check -label "Grid" -variable showgrid -onvalue true -offvalue false \
	-command {xcircuit::config grid $showgrid; refresh}
$m2 add check -label "Axes" -variable showaxes -onvalue true -offvalue false \
	-command {xcircuit::config axes $showaxes; refresh}
$m2 add command -label "Grid Spacing" -command {xcircuit::promptgridspace}
$m2 add cascade -label "Grid type/display" -menu $m2.gridsubmenu

set m3 [menu $m2.gridsubmenu -tearoff 0]
$m3 add radio -label "Decimal Inches" -variable gridstyle -value "decimal inches" \
	-command {xcircuit::config coordstyle $gridstyle}
$m3 add radio -label "Fractional Inches" -variable gridstyle -value "fractional inches" \
	-command {xcircuit::config coordstyle $gridstyle}
$m3 add radio -label "Centimeters" -variable gridstyle -value "centimeters" \
	-command {xcircuit::config coordstyle $gridstyle}
$m3 add separator
$m3 add command -label "Drawing Scale" -command {xcircuit::promptdrawingscale}

set m2 [menu $m.snapmenu -tearoff 0]
$m2 add check -label "Snap-to" -variable showsnap -onvalue true -offvalue false \
	-command {xcircuit::config snap $showsnap; refresh}
$m2 add command -label "Snap Spacing" -command {xcircuit::promptsnapspace}

set m2 [menu $m.elementsmenu -tearoff 0]
$m2 add cascade -label "Border" -menu $m2.bordermenu
$m2 add cascade -label "Fill" -menu $m2.fillmenu
$m2 add cascade -label "Color" -menu $m2.colormenu
$m2 add separator
$m2 add command -label "Object Size" -command {xcircuit::promptobjectsize}
$m2 add cascade -label "Parameters" -menu $m2.parammenu
$m2 add check -label "Center Object" -variable centerobject -onvalue true \
	-offvalue false -command {xcircuit::config centering $centerobject}
$m2 add check -label "Manhattan Draw" -variable manhattandraw -onvalue true \
	-offvalue false -command {xcircuit::config manhattan $manhattandraw}
$m2 add cascade -label "Polygon Edit" -menu $m2.polyeditmenu

set m3 [menu $m2.bordermenu -tearoff 0]
$m3 add command -label "Linewidth" -command {xcircuit::promptborderwidth}
$m3 add separator
$m3 add radio -label "Solid" -variable linestyle -value solid \
	-command {xcircuit::border $linestyle}
$m3 add radio -label "Dashed" -variable linestyle -value dashed \
	-command {xcircuit::border $linestyle}
$m3 add radio -label "Dotted" -variable linestyle -value dotted \
	-command {xcircuit::border $linestyle}
$m3 add radio -label "Unbordered" -variable linestyle -value unbordered \
	-command {xcircuit::border $linestyle}
$m3 add separator
$m3 add check -label "Closed" -variable polyclosed -onvalue closed \
	-offvalue unclosed -command {xcircuit::border  $polyclosed}
$m3 add check -label "Bounding Box" -variable bboxtype -onvalue true \
	-offvalue false -command {xcircuit::border bbox $bboxtype}

# stipple images
image create bitmap img_stip0 -foreground white -background black -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/solid.xbm
image create bitmap img_stip12 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip12.xbm
image create bitmap img_stip25 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip25.xbm
image create bitmap img_stip38 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip38.xbm
image create bitmap img_stip50 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip50.xbm
image create bitmap img_stip62 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip62.xbm
image create bitmap img_stip75 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip75.xbm
image create bitmap img_stip88 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/stip88.xbm
image create bitmap img_stip100 -foreground black -background white -file \
	${XCIRCUIT_LIB_DIR}/pixmaps/solid.xbm

set m3 [menu $m2.fillmenu -tearoff 0]
$m3 add radio -image img_stip100 -variable fillamount -value 100 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip88 -variable fillamount -value 88 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip75 -variable fillamount -value 75 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip62 -variable fillamount -value 62 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip50 -variable fillamount -value 50 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip38 -variable fillamount -value 38 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip25 -variable fillamount -value 25 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip12 -variable fillamount -value 12 \
	-command {xcircuit::fill $fillamount; xcircuit::fill opaque}
$m3 add radio -image img_stip0 -variable fillamount -value 0 \
	-command {xcircuit::fill $fillamount; xcircuit::fill transparent}
$m3 add separator
$m3 add radio -label "Opaque" -variable opaque -value true \
	-command {xcircuit::fill opaque}
$m3 add radio -label "Transparent" -variable opaque -value false \
	-command {xcircuit::fill transparent}

set m3 [menu $m2.colormenu -tearoff 0]
$m3 add command -label "Add New Color"
$m3 add separator
$m3 add radio -label "Inherit Color" -variable colorval -value inherit \
  -command {color $colorval}

set m3 [menu $m2.parammenu -tearoff 0]
$m3 add check -label "Substring" -variable substringparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make substring}
$m3 add check -label "Position" -variable positionparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make position}
$m3 add check -label "Style" -variable styleparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make style}
$m3 add check -label "Justification" -variable justparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make justification}
$m3 add check -label "Start Angle" -variable startparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make "start angle"}
$m3 add check -label "End Angle" -variable endparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make "end angle"}
$m3 add check -label "Radius" -variable radiusparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make radius}
$m3 add check -label "Minor Axis" -variable minorparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make "minor axis"}
$m3 add check -label "Rotation" -variable rotationparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make rotation}
$m3 add check -label "Scale" -variable scaleparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make scale}
$m3 add check -label "Linewidth" -variable linewidthparam -onvalue true \
	-offvalue false -command {xcircuit::parameter make linewidth}

set m3 [menu $m2.polyeditmenu -tearoff 0]
$m3 add radio -label "Manhattan Box Edit" -variable polyedittype \
    -value manhattan -command {xcircuit::config boxedit $polyedittype}
$m3 add radio -label "Rhomboid X" -variable polyedittype \
    -value rhomboidx -command {xcircuit::config boxedit $polyedittype}
$m3 add radio -label "Rhomboid Y" -variable polyedittype \
    -value rhomboidy -command {xcircuit::config boxedit $polyedittype}
$m3 add radio -label "Rhomboid A" -variable polyedittype \
    -value rhomboida -command {xcircuit::config boxedit $polyedittype}
$m3 add radio -label "Normal" -variable polyedittype \
    -value normal -command {xcircuit::config boxedit $polyedittype}

set m [menu .xcircuit.menubar.windowbutton.windowmenu -tearoff 0]
$m add command -label "Zoom In" -command {zoom 1.5; refresh}
$m add command -label "Zoom Out" -command {zoom [expr {1 / 1.5}]; refresh}
$m add command -label "Zoom Box" -command {zoom box}
$m add command -label "Full View" -command {zoom view; refresh}
$m add command -label "Center Pan" -command {pan}
$m add command -label "Refresh" -command {refresh}
$m add separator
$m add command -label "Library Directory" -command {xcircuit::library directory}
$m add cascade -label "Goto Library" -menu $m.librarymenu
$m add separator
$m add command -label "Page Directory" -command {xcircuit::page directory}
$m add cascade -label "Goto Page" -menu $m.pagemenu

set m2 [menu $m.librarymenu -tearoff 0]
$m2 add command -label "Add New Library" -command {xcircuit::library make}
$m2 add separator

set m2 [menu $m.pagemenu -tearoff 0]
$m2 add command -label "Add New Page" -command {xcircuit::page make}
$m2 add separator

set m [menu .xcircuit.menubar.netlistbutton.netlistmenu -tearoff 0]
$m add command -label "Make Pin" -command {xcircuit::label make pin}
$m add command -label "Make Info Pin" -command {xcircuit::label make info}
$m add command -label "Make Global Pin" -command {xcircuit::label make global}
$m add cascade -label "Convert Label To..." -menu $m.pinmenu
$m add check -label "Pin Visibility" -variable pinvisible -onvalue true \
	-offvalue false -command {xcircuit::label visible $pinvisible}
$m add command -label "Make Matching Symbol" -command {if {![catch \
	{xcircuit::symbol make}]} {xcircuit::promptmakesymbol}}
$m add command -label "Associate With Symbol" -command {xcircuit::symbol associate}
$m add command -label "Highlight Connectivity" -command {xcircuit::netlist highlight}
$m add command -label "Auto-number Components" -command {xcircuit::netlist autonumber}
$m add separator
$m add command -label "Write SPICE" -command {xcircuit::netlist write spice}
$m add command -label "Write flattened SPICE" -command \
	{xcircuit::netlist write spiceflat}
$m add command -label "Write sim" -command {xcircuit::netlist write sim}
$m add command -label "Write pcb" -command {xcircuit::netlist write pcb}

set m2 [menu $m.pinmenu -tearoff 0]
$m2 add command -label "Normal label" -command {xcircuit::label type normal}
$m2 add command -label "Local Pin" -command {xcircuit::label type pin}
$m2 add command -label "Global Pin" -command {xcircuit::label type global}
$m2 add command -label "Info label" -command {xcircuit::label type info}

#-------------------------------------------------
# Create the cloned menu links used by the toolbar
#-------------------------------------------------

.xcircuit.menubar.optionsbutton.optionsmenu.elementsmenu.parammenu clone .parammenu
.xcircuit.menubar.optionsbutton.optionsmenu.elementsmenu.colormenu clone .colormenu
.xcircuit.menubar.optionsbutton.optionsmenu.elementsmenu.bordermenu clone .bordermenu
.xcircuit.menubar.optionsbutton.optionsmenu.elementsmenu.fillmenu clone .fillmenu

#-----------------------------------------
# Create the toolbar and associated tools
#-----------------------------------------

xcircuit::createtoolbar
xcircuit::arrangetoolbar

#-----------------------------------------------------------------
# Final setup stuff before exiting back to interpreter
#-----------------------------------------------------------------

# Add buttons for the pre-allocated pages

xcircuit::newpagebutton "Page 1" 1
xcircuit::newpagebutton "Page 2" 2
xcircuit::newpagebutton "Page 3" 3
xcircuit::newpagebutton "Page 4" 4
xcircuit::newpagebutton "Page 5" 5
xcircuit::newpagebutton "Page 6" 6
xcircuit::newpagebutton "Page 7" 7
xcircuit::newpagebutton "Page 8" 8
xcircuit::newpagebutton "Page 9" 9
xcircuit::newpagebutton "Page 10" 10

# Add buttons for the pre-allocated libraries

xcircuit::newlibrarybutton "User Library"

# Automatic key focus on main drawing window

bind .xcircuit.mainframe.mainarea.drawing <Enter> {focus %W}

# New key bindings should pass through this function so that key
# strokes are captured correctly for label editing.

proc xcircuit::keyaction {keycode {keystate 0}} {
   switch -- [eventmode] {
      text2 -
      text3 -
      cattext {
	 standardaction $keycode key $keystate
	 return true
      }
   }
   return false
}

# Add key and button bindings for XCircuit commands (standard actions)
# These can be overridden by binding to specific keys and/or buttons.

bind .xcircuit.mainframe.mainarea.drawing <ButtonPress> {standardaction %b down %s}
bind .xcircuit.mainframe.mainarea.drawing <ButtonRelease> {standardaction %b up %s}
bind .xcircuit.mainframe.mainarea.drawing <KeyPress> {standardaction %k key %s}

# Wait for the drawing area to become visible, and set the focus on it.

tkwait visibility .xcircuit.mainframe.mainarea.drawing
focus .xcircuit.mainframe.mainarea.drawing

# This pops down the interpreter window, if the "console.tcl" script was run.

catch xcircuit::consoledown

# End of GUI configuration file.  Xcircuit continues to load the xcircuit
# startup configuration files.
