/*----------------------------------------------------------------------*/
/* undo.c 								*/
/*									*/
/* The comprehensive "undo" and "redo" command handler			*/
/*									*/
/* Copyright (c) 2004  Tim Edwards, Open Circuit Design, Inc., and	*/
/* MultiGiG, Inc.							*/
/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*      written by Tim Edwards, 1/29/04    				*/
/*----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*----------------------------------------------------------------------*/
/* Local includes							*/
/*----------------------------------------------------------------------*/

#ifdef TCL_WRAPPER
#include <tk.h>
#endif

#include "xcircuit.h"

/*----------------------------------------------------------------------*/
/* Function prototype declarations                                      */
/*----------------------------------------------------------------------*/
#include "prototypes.h"

/*----------------------------------------------------------------------*/
/* Externally declared variables					*/
/*----------------------------------------------------------------------*/

extern Globaldata xobjs;

/*----------------------------------------------------------------------*/
/* register_for_undo ---						*/
/*									*/
/*   Register an event with the undo handler.  This creates a record	*/
/*   based on the mode, which is one of the XCF_* bindings (see		*/
/*   xcircuit.h for the list).  This is a variable-argument routine,	*/
/*   with the arguments dependent on the command.			*/
/*									*/
/*   thisinst is the instance of the object in which the event occurred	*/
/*   and is registered for every event type.				*/
/*----------------------------------------------------------------------*/

void register_for_undo(u_int mode, objinstptr thisinst, ...)
{
   va_list args;
   int usize, drawmode;
   objectptr delobj;
   Undoptr newrecord;
 
   /* This action invalidates everything in the "redo" stack, so flush it */
   flush_redo_stack();

   /* Create the new record and push it onto the stack */
   newrecord = (Undoptr)malloc(sizeof(Undostack));
   newrecord->next = xobjs.undostack;
   newrecord->last = NULL;
   newrecord->type = mode;
   newrecord->thisinst =  thisinst;
   newrecord->undodata = (char *)NULL;

   if (xobjs.undostack)
      xobjs.undostack->last = newrecord;
   xobjs.undostack = newrecord;

   va_start(args, thisinst);

   switch(mode) {
      case XCF_Delete:
	 /* args (2): delobj, drawmode					*/
	 /*    delobj = pointer to object containing deleted entries	*/
	 /*    drawmode = true if elements should be erased		*/ 
	 delobj = va_arg(args, objectptr);
	 drawmode = va_arg(args, int);
	 newrecord->undodata = (char *)delobj;
	 newrecord->idata = drawmode;
	 break;
   }

   va_end(args);
}

/*----------------------------------------------------------------------*/
/* undo_action ---							*/
/*	Play undo record back one in the stack.				*/
/*----------------------------------------------------------------------*/

void undo_action()
{
   Undoptr thisrecord;
   objectptr thisobj;
   short *slist;
   int snum;

   /* Undo the recorded action and shift the undo record pointer.	*/

   thisrecord = xobjs.undostack;
   if (thisrecord == NULL) {
      Fprintf(stderr, "Nothing to undo!\n");
      return;
   }
   xobjs.undostack = thisrecord->next;
   xobjs.redostack = thisrecord;

   /* type-dependent part */

   switch(thisrecord->type) {
      case XCF_Delete:
	 thisobj = (objectptr)thisrecord->undodata;
	 snum = thisobj->parts;
	 slist = xc_undelete(thisrecord->thisinst,
			(objectptr)thisrecord->undodata,
			(short)thisrecord->idata);
	 thisrecord->idata = snum;
	 thisrecord->undodata = (char *)slist;
	 break;
      default:
	 Fprintf(stderr, "Undo not implemented for this action!\n");
	 break;
   }
}

/*----------------------------------------------------------------------*/
/* redo_action ---							*/
/*	Play undo record forward one in the stack.			*/
/*----------------------------------------------------------------------*/

void redo_action()
{
   Undoptr thisrecord;
   objectptr thisobj;
   short *slist;

   /* Undo the recorded action and shift the undo record pointer.	*/

   thisrecord = xobjs.redostack;
   if (thisrecord == NULL) {
      Fprintf(stderr, "Nothing to redo!\n");
      return;
   }
   xobjs.undostack = thisrecord;
   xobjs.redostack = thisrecord->last;

   /* type-dependent part */

   switch(thisrecord->type) {
      case XCF_Delete:
	 slist = (short *)thisrecord->undodata;
	 thisobj = delete_element(thisrecord->thisinst, slist,
		thisrecord->idata, DRAW);
	 thisrecord->undodata = (char *)thisobj;
	 thisrecord->idata = (int)DRAW;
	 break;
      default:
	 Fprintf(stderr, "Undo not implemented for this action!\n");
	 break;
   }
}

/*----------------------------------------------------------------------*/
/* flush_redo_stack ---							*/
/*	Free all memory allocated to the redo stack due to the		*/
/* 	insertion of a new undo record.					*/
/*----------------------------------------------------------------------*/

void flush_redo_stack()
{
   Undoptr thisrecord, nextrecord;

   if (xobjs.redostack == NULL) return;	/* no redo stack */

   thisrecord = xobjs.redostack;

   while (thisrecord != NULL) {
      nextrecord = thisrecord->last;
      free_redo_record(thisrecord);
      thisrecord = nextrecord;
   }
   xobjs.redostack = NULL;

   if (xobjs.undostack)
      xobjs.undostack->last = NULL;
}

/*----------------------------------------------------------------------*/
/* flush_undo_stack ---							*/
/*	Free all memory allocated to the undo and redo stacks.		*/
/*----------------------------------------------------------------------*/

void flush_undo_stack()
{
   Undoptr thisrecord, nextrecord;

   flush_redo_stack();

   thisrecord = xobjs.undostack;

   while (thisrecord != NULL) {
      nextrecord = thisrecord->next;
      free_undo_record(thisrecord);
      thisrecord = nextrecord;
   }
   xobjs.undostack = NULL;
}

/*----------------------------------------------------------------------*/
/* free_undo_data ---							*/
/*	Free memory allocated to the "undodata" part of the undo	*/
/*	record, based on the record type.				*/
/*----------------------------------------------------------------------*/

void free_undo_data(Undoptr thisrecord)
{
   u_int mode;
   objectptr uobj;

   mode = thisrecord->type;
   switch (mode) {
      case XCF_Delete:
	 uobj = (objectptr)thisrecord->undodata;
	 reset(uobj, DESTROY);
	 break;

      default:
	 if (thisrecord->undodata != NULL)
	    free(thisrecord->undodata);
	 break;
   }
   thisrecord->undodata = NULL;
}

/*----------------------------------------------------------------------*/
/* free_redo_data ---							*/
/*	Free memory allocated to the "undodata" part of the redo	*/
/*	record, based on the record type.  Note that the action taken	*/
/*	for a specific record is *NOT* the same for a record in the	*/
/*	undo stack as it is for a record in the redo stack, because	*/
/*	the data types are changed when moving from one record to the	*/
/*	next.
/*----------------------------------------------------------------------*/

void free_redo_data(Undoptr thisrecord)
{
   u_int mode;
   int snum;
   objectptr uobj;

   mode = thisrecord->type;
   switch (mode) {
      case XCF_Delete:
	 free(thisrecord->undodata);
	 break;

      default:
	 if (thisrecord->undodata != NULL)
	    free(thisrecord->undodata);
	 break;
   }
   thisrecord->undodata = NULL;
}

/*----------------------------------------------------------------------*/
/* free_undo_record ---							*/
/*	Free allocated memory for one record in the undo stack.		*/
/*----------------------------------------------------------------------*/

void free_undo_record(Undoptr thisrecord)
{
   Undoptr nextrecord, lastrecord;

   /* Reset the master list pointers */

   if (xobjs.undostack == thisrecord)
      xobjs.undostack = thisrecord->next;

   /* Relink the stack pointers */

   if (thisrecord->last)
      thisrecord->last->next = thisrecord->next;

   if (thisrecord->next)
      thisrecord->next->last = thisrecord->last;

   /* Free memory allocated to the record */

   free_undo_data(thisrecord);
   free(thisrecord);
}

/*----------------------------------------------------------------------*/
/* free_redo_record ---							*/
/*	Free allocated memory for one record in the redo stack.		*/
/*----------------------------------------------------------------------*/

void free_redo_record(Undoptr thisrecord)
{
   Undoptr nextrecord, lastrecord;

   /* Reset the master list pointers */

   if (xobjs.redostack == thisrecord)
      xobjs.redostack = thisrecord->last;

   /* Relink the stack pointers */

   if (thisrecord->next)
      thisrecord->next->last = thisrecord->last;

   if (thisrecord->last)
      thisrecord->last->next = thisrecord->next;

   /* Free memory allocated to the record */

   free_redo_data(thisrecord);
   free(thisrecord);
}
