#---------------------------------------------------------------------------
# symbol.tcl ---
#
#	xcircuit Tcl script for generating
#	a canonical circuit symbol
#
#	Tim Edwards 12/1/05 for MultiGiG
#---------------------------------------------------------------------------

proc xcircuit::makesymbol {} {
   global XCOps

   set symbolname [.dialog.textent.txt get]
   xcircuit::symbol make $symbolname $XCOps(library);

   set pinspace 64
   set halfspace [expr $pinspace / 2]
   set qtrspace [expr $pinspace / 4]

   # remove the old pin labels

   set oldpinlabels [object parts]
   foreach j $oldpinlabels {
      delete $j
   }
   set leftpins [.dialog.pinself.left.list index end]
   set toppins [.dialog.pinself.top.list index end]
   set rightpins [.dialog.pinself.right.list index end]
   set botpins [.dialog.pinself.bottom.list index end]

   set hpins $leftpins
   if {$rightpins > $leftpins} {set hpins $rightpins}
   set vpins $toppins
   if {$botpins > $toppins} {set vpins $botpins}

   set boxwidth [expr ($hpins + 1) * $pinspace]
   set boxheight [expr ($vpins + 1) * $pinspace]

   set hwidth [expr $boxwidth / 2]
   if {$hwidth < 256} {set hwidth 256}
   set hheight [expr $boxheight / 2]
   if {$hheight < 256} {set hheight 256}

   set sbox [polygon make box "-$hwidth -$hheight" "$hwidth $hheight"]
   set pinlabels {}

   set x [expr -$hwidth - $qtrspace]
   set y [expr -($leftpins - 1) * $halfspace]
   for {set j 0} {$j < $leftpins} {incr j} {
      set tabx [expr $x + $qtrspace]
      polygon make 2 "$x $y" "$tabx $y"
      set pintext [.dialog.pinself.left.list get $j]
      lappend pinlabels $pintext
      set tlab [label make pin "$pintext" "$x $y"]
      label $tlab justify right
      label $tlab justify middle
      set nlab [element $tlab copy relative "$halfspace 0"]
      label $nlab type normal
      label $nlab justify left
      incr y $pinspace
   }
   set y [expr $hheight + $qtrspace]
   set x [expr -($toppins - 1) * $halfspace]
   for {set j 0} {$j < $toppins} {incr j} {
      set taby [expr $y - $qtrspace]
      polygon make 2 "$x $y" "$x $taby"
      set pintext [.dialog.pinself.top.list get $j]
      lappend pinlabels $pintext
      set tlab [label make pin "$pintext" "$x $y"]
      rotate $tlab 90
      label $tlab justify right
      label $tlab justify middle
      set nlab [element $tlab copy relative "0 -$halfspace"]
      label $nlab type normal
      label $nlab justify left
      incr x $pinspace
   }
   set x [expr $hwidth + $qtrspace]
   set y [expr -($rightpins - 1) * $halfspace]
   for {set j 0} {$j < $rightpins} {incr j} {
      set tabx [expr $x - $qtrspace]
      polygon make 2 "$x $y" "$tabx $y"
      set pintext [.dialog.pinself.right.list get $j]
      lappend pinlabels $pintext
      set tlab [label make pin "$pintext" "$x $y"]
      label $tlab justify left
      label $tlab justify middle
      set nlab [element $tlab copy relative "-$halfspace 0"]
      label $nlab type normal
      label $nlab justify right
      incr y $pinspace
   }
   set y [expr -$hheight -$qtrspace]
   set x [expr -($botpins - 1) * $halfspace]
   for {set j 0} {$j < $botpins} {incr j} {
      set taby [expr $y + $qtrspace]
      polygon make 2 "$x $y" "$x $taby"
      set pintext [.dialog.pinself.bottom.list get $j]
      lappend pinlabels $pintext
      set tlab [label make pin "$pintext" "$x $y"]
      rotate $tlab 270
      label $tlab justify right
      label $tlab justify middle
      set nlab [element $tlab copy relative "0 $halfspace"]
      label $nlab type normal
      label $nlab justify left
      incr x $pinspace
   }

   deselect
   set nlab [label make "$symbolname" {0 0}]
   label $nlab justify middle
   label $nlab justify center
   element $nlab color set blue

   deselect
   parameter make substring idx "?"
   parameter make substring class "X"
   parameter make substring link "%n"

   set nlab [label make "{Parameter class} {Parameter idx}" "0 -$pinspace"]
   label $nlab justify center
   element $nlab color set blue

   deselect
   set itext [list {Text "spice:"} {Parameter class} {Parameter idx} {Text ""}]
   foreach j $pinlabels {
      lappend itext "%p$j"
   }
   lappend itext " %n"
   set y [expr -$hheight - 3 * $pinspace]

   deselect
   set nlab [label make info "$itext" "0 $y"] 
   label $nlab justify center
   
   deselect
   zoom view
}

#---------------------------------------------------------------------------
# This is a replacement for the procedure promptmakesymbol in wrapper.tcl.
# We make a sub-widget to list, change, and reorder the symbol pins,
# similar to the "addliblist" procedure in wrapper.tcl
#---------------------------------------------------------------------------

proc xcircuit::getpinlist {} {
   deselect
   set pinlist {}
   set objlist [object parts]
   foreach j $objlist {
      set etype [element $j type]
      if {$etype == "Label"} {
	 set ltype [label $j type]
	 if {$ltype == "local" || $ltype == "global"} {
	    lappend pinlist [label $j text]
	 }
      }
   }
   set pinlist [lsort -unique -dictionary $pinlist]
   return $pinlist
}

#---------------------------------------------------------------------------
#---------------------------------------------------------------------------

proc xcircuit::getselectedpinwidget {} {
   set w .dialog.pinself.left.list
   set result [$w curselection]
   if {$result != {}} {return $w}

   set w .dialog.pinself.top.list
   set result [$w curselection]
   if {$result != {}} {return $w}

   set w .dialog.pinself.right.list
   set result [$w curselection]
   if {$result != {}} {return $w}

   set w .dialog.pinself.bottom.list
   set result [$w curselection]
   if {$result != {}} {return $w}
}

proc xcircuit::removeselectedpin {} {
   set w [xcircuit::getselectedpinwidget]
   if {$w != {}} {
      set idx [$w curselection]
      $w delete $idx
   }
}

proc xcircuit::moveselectedpin {} {
   set w [xcircuit::getselectedpinwidget]
   if {$w != {}} {
      set idx [$w curselection]
      set pinname [$w get $idx]
      $w delete $idx
      switch -glob $w {
	 {.dialog.pinself.left.list} {
		.dialog.pinself.top.list insert end $pinname
		.dialog.pinself.top.list selection set end
	 }
	 {.dialog.pinself.top.list} {
		.dialog.pinself.right.list insert end $pinname
		.dialog.pinself.right.list selection set end
	 }
	 {.dialog.pinself.right.list} {
		.dialog.pinself.bottom.list insert end $pinname
		.dialog.pinself.bottom.list selection set end
	 }
	 {.dialog.pinself.bottom.list} {
		.dialog.pinself.left.list insert end $pinname
		.dialog.pinself.left.list selection set end
	 }
      }
   }
}

#---------------------------------------------------------------------------
# Create the pin arranger widget and add it to the dialog box.
#---------------------------------------------------------------------------

proc xcircuit::addpinarranger {w} {

   xcircuit::removepinarranger ${w}

   frame ${w}.pinself
   frame ${w}.pinself.left
   frame ${w}.pinself.top
   frame ${w}.pinself.right
   frame ${w}.pinself.bottom

   label ${w}.pinself.left.title -text "Left Pins"
   label ${w}.pinself.top.title -text "Top Pins"
   label ${w}.pinself.right.title -text "Right Pins"
   label ${w}.pinself.bottom.title -text "Bottom Pins"

   listbox ${w}.pinself.left.list
   listbox ${w}.pinself.top.list
   listbox ${w}.pinself.right.list
   listbox ${w}.pinself.bottom.list

   pack ${w}.pinself.left.title -side top
   pack ${w}.pinself.left.list -side top
   pack ${w}.pinself.top.title -side top
   pack ${w}.pinself.top.list -side top
   pack ${w}.pinself.right.title -side top
   pack ${w}.pinself.right.list -side top
   pack ${w}.pinself.bottom.title -side top
   pack ${w}.pinself.bottom.list -side top

   grid ${w}.pinself.left -row 0 -column 0 -sticky news -padx 1 -pady 1
   grid ${w}.pinself.top -row 0 -column 1 -sticky news -padx 1 -pady 1
   grid ${w}.pinself.right -row 0 -column 2 -sticky news -padx 1 -pady 1
   grid ${w}.pinself.bottom -row 0 -column 3 -sticky news -padx 1 -pady 1

   grid columnconfigure ${w}.pinself 0 -weight 1 -minsize 50
   grid columnconfigure ${w}.pinself 1 -weight 1 -minsize 50
   grid columnconfigure ${w}.pinself 2 -weight 1 -minsize 50
   grid columnconfigure ${w}.pinself 3 -weight 1 -minsize 50

   set pinlist [xcircuit::getpinlist]

   # Break the pinlist up into 4 parts

   set leftpins [expr [llength $pinlist] / 2]
   set rightpins  [expr [llength $pinlist] - $leftpins]
   set toppins [expr $leftpins / 2]
   set leftpins [expr $leftpins - $toppins]
   set botpins [expr $rightpins / 2]
   set rightpins [expr $rightpins - $botpins]
   incr toppins $leftpins
   incr rightpins $toppins
   incr botpins $rightpins

   for {set k 0} {$k < $leftpins} {incr k} {
      ${w}.pinself.left.list insert end [lindex $pinlist $k]
   }
   for {} {$k < $toppins} {incr k} {
      ${w}.pinself.top.list insert end [lindex $pinlist $k]
   }
   for {} {$k < $rightpins} {incr k} {
      ${w}.pinself.right.list insert end [lindex $pinlist $k]
   }
   for {} {$k < $botpins} {incr k} {
      ${w}.pinself.bottom.list insert end [lindex $pinlist $k]
   }

   pack ${w}.pinself -side top -anchor w -padx 20 -pady 5

   button ${w}.bbar.remove -text "Remove Pin" -bg beige -command \
	{xcircuit::removeselectedpin}
   button ${w}.bbar.move -text "Move Pin" -bg beige -command \
	{xcircuit::moveselectedpin}

   pack ${w}.bbar.remove -side left -ipadx 10
   pack ${w}.bbar.move -side left -ipadx 10
}

#---------------------------------------------------------------------------
# Remove the pin arranger widget from the dialog box.
#---------------------------------------------------------------------------

proc xcircuit::removepinarranger {w} {
   catch {
      pack forget ${w}.pinself
      destroy ${w}.pinself
   }
} 

#---------------------------------------------------------------------------
# Redefine popupdialog to remove the pin arranger any time it is invoked.
#---------------------------------------------------------------------------

proc xcircuit::popupdialog {} {
   xcircuit::removepinarranger .dialog
   xcircuit::removeliblist .dialog
   wm deiconify .dialog
   focus .dialog.textent.txt
}

#---------------------------------------------------------------------------
# Redefine the procedure for the "Make Matching Symbol" menu button.
#---------------------------------------------------------------------------

proc xcircuit::promptmakesymbol {{name ""}} {
  global XCOps
  .dialog.bbar.okay configure -command \
          {if {[string first "Page " [page label]] >= 0} { \
          page label [.dialog.textent.txt get]}; \
          xcircuit::makesymbol; \
          wm withdraw .dialog}
  .dialog.textent.title.field configure -text "Confirm symbol name:"
  .dialog.textent.txt delete 0 end
  .dialog.textent.txt insert 0 $name
  xcircuit::popupdialog
  xcircuit::addliblist .dialog "Place in: "
  xcircuit::addpinarranger .dialog
}        

